# arch-tag: cb40743b-1a3d-4c56-9647-512721976ad2
# Copyright (C) 2004 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Command-line back-end glue."""

import sys
from pybaz import compat

### Argument Checking ###

def _check_expected_param(expected):
    msg_format = ("'expected' param must be a sequence of positive"
                  " integers but was: %r")
    try:
        iterator = iter(expected)
    except TypeError:
        raise TypeError, msg_format % expected
    for status in iterator:
        if not isinstance(status, int):
            raise TypeError, msg_format % expected
        if status < 0:
            raise ValueError, msg_format % expected


### Default Backend ###

def default_backend():
    """Default command-line backend.

    :rtype: `CommandLineBackend`
    """
    import baz
    import logger
    backend =  CommandLineBackend(
        command = 'baz',
        module = baz,
        spawning = DelayedGuessedSpawningStrategy,
        logger = logger.Logger())
    return backend


class CommandLineBackend(object):

    """Facade for command-line back-end options."""

    def __init__(self, command, module, spawning, logger):
        self._command = command
        self._module = module
        self._spawning = spawning
        self._logger = logger
        self._spawner = None
        self._version = None

    def _get_command(self):
        return self._command

    def _set_command(self, command):
        assert command
        self._command = command
        self._spawner = None
        self._version = None

    command = property(
        _get_command, _set_command, None,
        """Name of the command line program.

        :type: str""")

    def _get_spawning(self):
        return self._spawning

    def _set_spawning(self, spawning):
        assert spawning
        self._spawning = spawning
        self._spawner = None

    spawning_strategy = property(
        _get_spawning, _set_spawning, None,
        """`SpawningStrategy` factory used to create the spawner.

        :type: factory of `SpawningStrategy`""")

    def _get_logger(self):
        return self._logger

    def _set_logger(self, logger):
        assert logger
        self._logger = logger
        self._spawner = None

    logger = property(
        _get_logger, _set_logger, None,
        """Command line logger.

        :type: `logger.Logger`""")

    def _get_spawner(self):
        if self._spawner is None:
            assert self._spawning
            assert self._command
            assert self._logger
            self._spawner = self._spawning(self._command, self._logger)
        return self._spawner

    def null_cmd(self, args, chdir=None):
        status = self._get_spawner().status_cmd(args, chdir, (0,))
        assert status == 0

    def one_cmd(self, args, chdir=None):
        status, line = self.status_one_cmd(args, (0,), chdir)
        assert status == 0
        return line

    def sequence_cmd(self, args, expected=(0,), chdir=None, stderr_too=False):
        # XXX stderr_too=True will fail with the TwistedSpawningStrategy. That
        # XXX will be fixed when the process handling subsystem is replaced by
        # XXX Gnarly. -- David Allouche 2005-05-27
        _check_expected_param(expected)
        sequence_cmd = self._get_spawner().sequence_cmd
        if stderr_too:
            return sequence_cmd(args, chdir, expected, stderr_too=stderr_too)
        else:
            return sequence_cmd(args, chdir, expected)

    def text_cmd(self, args, chdir=None):
        status, text = self._get_spawner().status_text_cmd(args, chdir, (0,))
        assert status == 0
        return text

    def status_cmd(self, args, expected, chdir=None):
        _check_expected_param(expected)
        return self._get_spawner().status_cmd(args, chdir, expected)

    def status_one_cmd(self, args, expected, chdir=None):
        _check_expected_param(expected)
        seq = self.sequence_cmd(args, expected, chdir)
        try:
            out = seq.next()
        except StopIteration:
            return seq.status, None
        tail = list(seq)
        if not tail:
            return seq.status, out
        raise AssertionError, (
            'one liner actually produced multiline output:\n%s'
            % '\n'.join([out] + tail))

    def status_text_cmd(self, args, expected):
        _check_expected_param(expected)
        return self._get_spawner().status_text_cmd(args, None, expected)

    def _get_version(self):
        if self._version is None:
            line = list(self.sequence_cmd(('--version',)))[0]
            self._version = compat.BazaarCommandVersion(line)
        return self._version

    version = property(_get_version, doc="""
    The current command version.

    Instance of pybaz.compat.BazaarCommandVersion.""")

    __pybaz_version = compat.BazaarCommandVersion(
        'baz Bazaar version 1.4~200504281027')

    def _get_compat(self):
        return compat._get_status(self.__pybaz_version, self.version)

    compatibility = property(_get_compat)


class SpawningStrategy(object):

    """Abstract class for process spawning strategies."""

    def status_cmd(self, args, chdir, expected):
        raise NotImplementedError

    def status_text_cmd(self, args, chdir, expected):
        raise NotImplementedError

    def sequence_cmd(self, args, chdir=None, expected=(0,)):
        raise NotImplementedError


class DelayedGuessedSpawningStrategy(SpawningStrategy):
    """SpawningStrategy that uses Twisted if it is present in ``sys.modules``.

    This SpawningStrategy tries to do the right thing my looking at
    the contents of ``sys.modules``: if "twisted" or a module of that
    package is loaded when the first spawning method is run, it will
    use the Twisted spawning strategy. Otherwise, it will use a simple
    fork/exec spawning strategy which does not depend on Twisted.
    """

    __pychecker__ = 'no-override'

    def __init__(self, *args, **kwargs):
        self._init_args = args
        self._init_kwargs = kwargs
        self._cached_guess = None

    def _guess(self):
        if self._cached_guess is not None:
            return self._cached_guess
        strategy = None
        for name in sys.modules:
            if name == 'twisted' or name.startswith('twisted.'):
                import knotted
                strategy = knotted.TwistedSpawningStrategy
                break
        else:
            import forkexec
            strategy = forkexec.PyArchSpawningStrategy
        self._cached_guess = strategy(*self._init_args, **self._init_kwargs)
        return self._cached_guess

    def sequence_cmd(self, *args, **kwargs):
        return self._guess().sequence_cmd(*args, **kwargs)

    def status_cmd(self, *args, **kwargs):
        return self._guess().status_cmd(*args, **kwargs)

    def status_text_cmd(self, *args, **kwargs):
        return self._guess().status_text_cmd(*args, **kwargs)
