/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Configuration back-end.
 *
 * Copyright (C) 2003 Isak Savo
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include "globals.h"
#include "cfg.h"
//#include "helper.h"

cfg_t cfg;

gboolean cfg_write_settings ()
{
   FILE *f;
   gchar filename[FILEPATH_MAX];
   
   g_snprintf (filename, FILEPATH_MAX, "%s/" RCFILE, g_get_home_dir ());
   if (!(f = fopen (filename, "w")))
   {
      perror ("Unable to open config-file for writing");
      return FALSE;
   }
    
  fputs ("# ~/.pureadminrc - Configuration file for pureadmin\n"
	 "# All variables are in the form VARNAME=VALUE where VALUE should be an UN-quoted string\n"
	 "# Comment lines BEGINS with a '#'. Otherwise the # is considered a part of the value/variablename\n\n", f);

  fputs ("# Default UID and GID to assign to new virtual users. These should point to an un-priveliged system account.\n", f);
  if (cfg.default_uid > 0)
    fprintf (f, "defaultuid=%d\n", cfg.default_uid);
  if (cfg.default_gid > 0)
    fprintf (f, "defaultgid=%d\n\n", cfg.default_gid);
  
  fputs ("# Default home is the directory which will be assigned to new users by default.\n", f);
  fprintf (f, "defaulthome=%s\n\n", cfg.default_home);
  
  fputs ("# Name and optional full path to the pure-pw command. Path is only needed if it is not installed in a standard location (i.e. Not in $PATH)\n", f);
  fprintf (f, "cmd_purepw=%s\n\n", cfg.cmd_purepw);
  
  fputs ("# Name and location of the password files where pureftpd reads user-information. The pdb-file is a binary database file,\n", f);
  fprintf (f, "pwfile=%s\n", cfg.pwfile);
  fprintf (f, "pdbfile=%s\n\n", cfg.pdbfile);
  
  fputs ("# Location of the logfile where pureftpd writes logging info\n", f);
  fprintf (f, "logfile=%s\n\n", cfg.logfile);
  fputs ("# Logging method, 0 means syslog and 1 means custom logfile\n", f);
  fprintf (f, "logmethod=%d\n", cfg.logmethod);
  fputs ("# Number of days history to show in logfile\n", f);
  fprintf (f, "logdays=%d\n\n", cfg.logdays);

  fputs ("# If this is enabled, then DNS name lookups will be performed resulting in IP-addresses being\n"
	 "# converted to human readable names\n"
	 "#  Setting this to 0 will make PureAdmin seem faster and more responsive.\n", f);
  fprintf (f, "resolve_hostnames=%d\n\n", cfg.resolve_hostnames);

  fputs ("# Indicates wether the first-time user-dialog messagebox have been seen by the user\n", f);
  fprintf (f, "seen_usrdlg_welcome=%d\n\n", cfg.seen_usrdlg_welcome);

  fputs ("# Indicates wether the FAM Error dialog should be shown or not\n", f);
  fprintf (f, "show_fam_errmsg=%d\n\n", cfg.show_fam_errmsg);

  fclose (f);
  return TRUE;
}


gboolean cfg_read_settings ()
{
   FILE *f;
   gchar filename[FILEPATH_MAX];
   gchar buf[LINE_MAX], **arr;

   
   g_snprintf (filename, FILEPATH_MAX, "%s/" RCFILE, g_get_home_dir ());
   if (!(f = fopen (filename, "r")))
   {
      pur_log_wrn ("Unable to open config-file for reading: %s", strerror (errno));
      pur_log_nfo ("Using default configuration\n");
      return FALSE;
   }
  
   while (fgets (buf, LINE_MAX, f))
   {
      g_strstrip (buf);
      if (*buf == '#' || *buf == '\0')
	continue;
      
      arr = g_strsplit (buf, "=", 2);
      if (!arr || !arr[0] || !arr[1])
	continue;
      g_strstrip (arr[0]);
      g_strstrip (arr[1]);
      
      if (!*arr[1]) /* Ignore values that is empty (i.e. "") */
	continue;
      if (strncmp (arr[0], "defaultuid", 10) == 0)
	cfg.default_uid = atoi (arr[1]);
      else if (strncmp (arr[0], "defaultgid", 10) == 0)
	cfg.default_gid = atoi (arr[1]);
      else if (strncmp (arr[0], "defaulthome", 11) == 0)
      {
	 g_free (cfg.default_home);
	 cfg.default_home = g_strdup (arr[1]);
      }
      else if (strncmp (arr[0], "cmd_purepw", 10) == 0)
      {
	 g_free (cfg.cmd_purepw);
	 cfg.cmd_purepw = g_strdup (arr[1]);
      }
      else if (strncmp (arr[0], "pwfile", 6) == 0)
      {
	 g_free (cfg.pwfile);
	 cfg.pwfile = g_strdup (arr[1]);
      }
      else if (strncmp (arr[0], "pdbfile", 7) == 0)
      {
	 g_free (cfg.pdbfile);
	 cfg.pdbfile = g_strdup (arr[1]);
      }
      else if (strncmp (arr[0], "logfile", 7) == 0)
      {
	 g_free (cfg.logfile);
	 cfg.logfile = g_strdup (arr[1]);
      }
      else if (strncmp (arr[0], "logmethod", 9) == 0)
      {
	 cfg.logmethod = atoi (arr[1]);
	 if (cfg.logmethod < 0 || cfg.logmethod >= NUM_LOGMETHODS)
	   cfg.logmethod = 0;
      }
      else if (strncmp (arr[0], "logdays", 7) == 0)
      {
	 cfg.logdays = atoi (arr[1]);
	 if (cfg.logdays < 0 || cfg.logdays > 100)
	   cfg.logdays = 0;
      }
      else if (strncmp (arr[0], "resolve_hostnames", 17) == 0)
	cfg.resolve_hostnames = atoi (arr[1]);
      else if (strncmp (arr[0], "seen_usrdlg_welcome", 19) == 0)
	cfg.seen_usrdlg_welcome = atoi (arr[1]);
      else if (strncmp (arr[0], "show_fam_errmsg", 19) == 0)
      	cfg.show_fam_errmsg = atoi (arr[1]);
      g_strfreev (arr);
   }
   fclose (f);
   
   return TRUE;
}

/* This should only be called ONCE and before any attempt to access or modify the cfg struct! */
void cfg_set_defaults (void)
{
   memset (&cfg, 0, sizeof (cfg_t));

   cfg.default_uid = srv_vars.id_ftpuid;
   cfg.default_gid = srv_vars.id_ftpgid;
   if (srv_vars.cmd_pure_pw)
     cfg.cmd_purepw = g_strdup (srv_vars.cmd_pure_pw);
   else
     cfg.cmd_purepw = (gchar *) g_malloc0 (1);

   if (srv_vars.dir_usrhome)
     cfg.default_home = g_strdup (srv_vars.dir_usrhome);
   else
     cfg.default_home = (gchar *) g_malloc0 (1);

   if (srv_vars.fl_pwfile)
     cfg.pwfile = g_strdup (srv_vars.fl_pwfile);
   else
     cfg.pwfile = (gchar *) g_malloc0 (1);

   if (srv_vars.fl_pdbfile)
     cfg.pdbfile = g_strdup (srv_vars.fl_pdbfile);
   else
     cfg.pdbfile = (gchar *) g_malloc0 (1);
   
   cfg.logfile = g_strdup ("/var/log/messages");
   cfg.logmethod = LOG_SYSLOG;
   cfg.show_fam_errmsg = TRUE;
}
