#!/usr/bin/env ruby

$:.unshift("../../lib") if __FILE__ =~ /\.rb$/

require 'puppettest'
require 'mocha'

class TestMongrelServer < PuppetTest::TestCase
    confine "Missing mongrel" => Puppet.features.mongrel?

    include PuppetTest::ServerTest

    def mkserver(handlers = nil)
        handlers ||= { :Status => nil }
        mongrel = Puppet::Network::Server::Mongrel.new(handlers)
    end

    # Make sure client info is correctly extracted.
    def test_client_info
        obj = Object.new
        obj.metaclass.send(:attr_accessor, :params)
        params = {}
        obj.params = params

        mongrel = mkserver

        ip = Facter.value(:ipaddress)
        params["REMOTE_ADDR"] = ip
        params[Puppet[:ssl_client_header]] = ""
        params[Puppet[:ssl_client_verify_header]] = "failure"
        info = nil
        Resolv.expects(:getname).with(ip).returns("host.domain.com").times(3)
        assert_nothing_raised("Could not call client_info") do
            info = mongrel.send(:client_info, obj)
        end
        assert(! info.authenticated?, "Client info object was marked valid even though headers were missing")
        assert_equal(ip, info.ip, "Did not copy over ip correctly")

        assert_equal("host.domain.com", info.name, "Did not copy over hostname correctly")

        # Now add a valid auth header.
        params[Puppet[:ssl_client_header]] = "/CN=host.domain.com"
        assert_nothing_raised("Could not call client_info") do
            info = mongrel.send(:client_info, obj)
        end
        assert(! info.authenticated?, "Client info object was marked valid even though the verify header was fals")
        assert_equal(ip, info.ip, "Did not copy over ip correctly")
        assert_equal("host.domain.com", info.name, "Did not copy over hostname correctly")

        # Now change the verify header to be true
        params[Puppet[:ssl_client_verify_header]] = "SUCCESS"
        assert_nothing_raised("Could not call client_info") do
            info = mongrel.send(:client_info, obj)
        end

        assert(info.authenticated?, "Client info object was not marked valid even though all headers were correct")
        assert_equal(ip, info.ip, "Did not copy over ip correctly")
        assert_equal("host.domain.com", info.name, "Did not copy over hostname correctly")

        # Now try it with a different header name
        params.delete(Puppet[:ssl_client_header])
        Puppet[:ssl_client_header] = "header_testing"
        params["header_testing"] = "/CN=other.domain.com"
        info = nil
        assert_nothing_raised("Could not call client_info with other header") do
            info = mongrel.send(:client_info, obj)
        end

        assert(info.authenticated?, "Client info object was not marked valid even though the header was present")
        assert_equal(ip, info.ip, "Did not copy over ip correctly")
        assert_equal("other.domain.com", info.name, "Did not copy over hostname correctly")

        # Now make sure it's considered invalid without that header
        params.delete("header_testing")
        info = nil
        assert_nothing_raised("Could not call client_info with no header") do
            info = mongrel.send(:client_info, obj)
        end

        assert(! info.authenticated?, "Client info object was marked valid without header")
        assert_equal(ip, info.ip, "Did not copy over ip correctly")
        assert_equal(Resolv.getname(ip), info.name, "Did not look up hostname correctly")
    end

    def test_daemonize
        mongrel = mkserver

        assert(mongrel.respond_to?(:daemonize), "Mongrel server does not respond to daemonize")
    end
end

# $Id: mongrel_test.rb 2752 2007-08-06 20:05:28Z luke $
