package AtomicData::USCurrency;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: USCurrency.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::USCurrency - data encapsulator for US dollar amounts

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for US dollar amounts.

=cut

use AtomicData::Decimal;
@AtomicData::USCurrency::ISA = qw (AtomicData::Decimal);

=head1 METHODS

For further methods, see root class.

=head2 parameters

For base documentation, see root class.

For additional parameters, see parent call.

Will also 'whole_dollars'/bool which will constrain amounts to whole
dollar amounts.

=head2 set_format

for base documentation, see root class.

Will accept the parameters: 

'commas'/bool which will add a comma every third digit right from the
decimal point or end of value. The defualt is false.

'whole_dollars'/bool will trim off all but whole dollars, defualt is
false

'dollar'/bool will prepend a dollar sign, defualt is false

'english_words'/will return the amount in word format. so 1.00 will be returned as 'one dollar and 00/100'
this format will override all other formats.  so, you cannot add a dollar sign, commas, etc to this format.

=head2 _canonicalize

 instance/private
 (float $value) _canonicalize (float $raw_value)

DESCRIPTION:

This is a private fucntion that will attempt to canonicalize the
encapsulated data. Will return the canonical form if successful and
set C<$this->{_can_failed}> if there was a problem and return
undef.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  # perform text canonicalization.
  $val = $this->AtomicData::Text::_canonicalize($val);

  # blank is already canonical.
  $val eq "" and return $val;

  #remove leading dollar sign if there
  $val =~ s/^\$//;

  #now we have something that Decimal can deal with
  $val = $this->SUPER::_canonicalize($val);
  defined $val or return;

  if (! ($val =~ /^([-\+]?)(\d*)\.?(\d*)?(e(\d+))?$/)) {
    $this->{_can_failed} = ["'$val' is a decimal but doesn't match??"];
    return undef;
  }
  my ($sign,$int,$frac,$exp) = ($1,$2,$3,$5);

  # round if we have too many digits.
  if (length($frac) > 2) {
    if (substr($frac,2,3) >= 5) {
      substr($frac,1,2) = substr($frac,1,2) + 1;
      $this->{_too_many_digits}++;
    }
  } elsif (length($frac) < 2) {
    $frac .= '0' x (2 - length($frac));
  }

  my $out = $sign.$int.".".$frac;
  $exp and $out .= "e$exp";

  return $out;
}

=head2 _verify

 instance/private
 (bool $verified, string \@errors) _verify (variable $value)

DESCRIPTION:

Verifies that the data falls within the set parameters noted
above. See C<verify()> in root class for behavior.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{whole_dollars} and $value !~ /\.00/) {
    return (0, ['The value must be in whole dollar amounts.']);
  }
  return (1, []);
}

=head2 _format

 instance/private
 (string $value) _format (float $canonical_value)

DESCRIPTION:

Will accept the canonical value and format it according to the values
of the format hash, as noted above in C<set_format>.

=cut

sub _format {
  my ($this, $value) = @_;

  my ($before, $throw_away, $after) = $value =~ /(-?\d*)(\.(\d+))?/;

  if ($this->{_format}->{english_words}) {
    return $this->currency_spelled_out($value);
  }

  if ($this->{_format}->{commas}) {
    $before = AtomicData::Decimal::commify($before);
  }

  if ($this->{_format}->{whole_dollars}) {
    undef $after;
  }

  if ($this->{_format}->{dollar}) {
    $before = '$'.$before; #'
  }

  ($value = "$before".'.'."$after") if length($after);

  return "$value";
}

sub currency_spelled_out {
  my ($this,$number) = @_;

  #Take the newly formatted number and get the parts before the
  #decimal and after.
  my ($first_half, $second_half) = split /\./, $number; 
  $first_half = AtomicData::Decimal::commify($first_half);

  #This holds the string we are building for a return
  my $return_string = "and";

  #Now, parse out each part that resides inside each comma group.
  #Example: This will break 1,002 into 1 and 002.
  my (@comma_parts) = split /\,/ , $first_half;

  #Count how many comma groups we have.
  my $num_of_parts = scalar(@comma_parts);

  #Declare our loop iterators to process each comma separated part.
  my $count=0;
  my $i;

  #We process from right to left.
  for ($i=$num_of_parts-1; $i>-1; $i--) {

    unless ($comma_parts[$i] == 0) {
      #Get the 3 digit name for this number
      $return_string = $this->digit3_to_name($comma_parts[$i])." ".
	#Tag on the 'thousands,millions...'name.
	$this->get_name_for_place($count).
	  #Add in the previous part.  
	  $return_string;
    }
    $count++;
  }

  #Return the string we built.
  if ($first_half!=0) {
    $return_string.=" ".$second_half."/100";
  }
  else {
    $return_string = "Zero and ".$second_half."/100";
  }
  return $return_string." Dollars";
}

=head2 get_name_for_place
 
 (String place_name) one_word_rep(integer $number)

 DESCRIPTION:
  Takes a place integer and returns its one word description. 
  Except for zero which is a special case. 

=cut


sub get_name_for_place {
  my ($this,$place) = @_;

  if ($place == 1) { return 'Thousand '; }
  if ($place == 2) { return 'Million '; }
  if ($place == 3) { return 'Billion '; }
  if ($place == 4) { return 'Trillion '; }
  if ($place == 5) { return 'Quadrillion '; }
  return '';
} 

=head2 one_word_rep

 (String one_word_representation) one_word_rep(integer $number)

 DESCRIPTION:
  Takes an integer and returns its one word description. Except for zero
  which is not used in text converting of numbers. 

=cut

sub one_word_rep {
  my ($this,$number) = @_;
 
  $number += 0;
  
  my %num_names = (
		   0=>'', 
		   1=>'One',
		   2=>'Two',
		   3=>'Three',
		   4=>'Four',
		   5=>'Five',
		   6=>'Six',
		   7=>'Seven',
		   8=>'Eight',
		   9=>'Nine',
		  ); 

  return $num_names{$number};

}

=head2 digit3_to_name
 
 (String number_full_name) digit3_to_name(integer $number)

 DESCRIPTION:
  Takes a 3 digit number and converts it to its english representation.

=cut

sub digit3_to_name {
  my ($this,$number) = @_;

  #Declare our string to build in.
  my $return_string = "";

  #Declare the place holder for the tens and ones spots.
  my ($tens,$ones);

  #Parse out the 10s and ones places.
  if ($number < 10) {
    return $this->one_word_rep($number);
  }
  elsif ($number < 100) {
    $ones = $number % 10;
    $tens = ($number - $ones) / 10;
  }
  else {
    $return_string .= $this->one_word_rep(substr($number,0,1))." Hundred";
    $tens = substr($number,1,1)+0;
    $ones = substr($number,2,1)+0;
    if (($tens!=0) || ($ones != 0)) {
      $return_string .= " ";
    }
  }
  
  my %num_names = (
		   10=>'Ten', 
		   11=>'Eleven',
		   12=>'Twelve',
		   13=>'Thirteen',
		   14=>'Fourteen',
		   15=>'Fifteen',
		   16=>'Sixteen',
		   17=>'Seventeen',
		   18=>'Eighteen',
		   19=>'Nineteen',
		  ); 

  if ($tens==1) {
    $return_string .= $num_names{($tens*10)+$ones};
  }
  elsif ($tens==2) {$return_string.=  'Twenty'}
  elsif ($tens==3) {$return_string.=  'Thirty'}
  elsif ($tens==4) {$return_string.=  'Fourty'}
  elsif ($tens==5) {$return_string.=  'Fifty'}
  elsif ($tens==6) {$return_string.=  'Sixty'}
  elsif ($tens==7) {$return_string.=  'Seventy'}
  elsif ($tens==8) {$return_string.=  'Eighty'}
  elsif ($tens==9) {$return_string.=  'Ninety'}

  if (($ones != 0) && ($tens>1)) {
    $return_string .= "-".$this->one_word_rep($ones);
  }
  
  return $return_string;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
