// for finding memory leaks in debug mode with Visual Studio 
#if defined _DEBUG && defined _MSC_VER
#include <crtdbg.h>
#endif

#include <stdint.h>
#ifndef _WIN32
#include <unistd.h> // usleep()
#endif
#include "pt_header.h"
#include "pt_helpers.h"
#include "pt_visuals.h"
#include "pt_scopes.h"
#include "pt_sampler.h"
#include "pt_palette.h"
#include "pt_tables.h"

// for monoscope
const int16_t mixScaleTable[AMIGA_VOICES] = { 388, 570, 595, 585 };

static volatile uint8_t scopesReading;
static uint64_t timeNext64;
static SDL_Thread *scopeThread;

// globals
scopeChannel_t scope[4];
volatile uint8_t emptyMonoscopeDrawn = false;
// -------

extern int8_t forceMixerOff;  // pt_audio.c
extern uint32_t *pixelBuffer; // pt_main.c

void updateScopes(void)
{
    uint8_t i;
    int32_t tmp32;
    scopeChannel_t *sc;
    moduleSample_t *s;

    if (editor.isWAVRendering)
        return;

    s = &modEntry->samples[editor.currSample];
    hideSprite(SPRITE_SAMPLING_POS_LINE);

    for (i = 0; i < AMIGA_VOICES; i++)
    {
        sc = &scope[i];
        if (sc->retriggered)
        {
            sc->retriggered = false;

            sc->frac_f = 0.0f;
            sc->phase  = 0;

            // data/length is already set from replayer thread (important)
            sc->loopFlag  = sc->newLoopFlag;
            sc->loopStart = sc->newLoopStart;

            sc->didSwapData = false;
            sc->active = true;
        }
        else if (sc->active)
        {
            sc->frac_f += sc->delta_f;
            if (sc->frac_f >= 1.0f)
            {
                tmp32 = (int32_t)(sc->frac_f); // truncate fraction to integer

                sc->frac_f -= tmp32;
                sc->phase  += tmp32;

                if (sc->phase >= sc->length)
                {
                    // sample reached end, simulate Paula register update

                    // wrap phase around one time with current length, then set new length
                    // and wrap around it (handles one-shot loops and sample swapping)
                    sc->phase -= sc->length;
                    sc->length = sc->newLength;

                    // store this one and test against 0 for safety
                    tmp32 = sc->length;
                    if (tmp32 > 0)
                        sc->phase %= tmp32;

                    // set new data pointer and update loop stuff
                    sc->data      = sc->newData;
                    sc->loopFlag  = sc->newLoopFlag;
                    sc->loopStart = sc->newLoopStart;

                    sc->didSwapData = true;
                }
            }
        }


        // move sample read position sprite (line)
        if (editor.ui.samplerScreenShown && !editor.muted[i] && (modEntry->channels[i].n_samplenum == editor.currSample) && !editor.ui.terminalShown)
        {
            if (sc->active && (sc->phase >= 2) && !editor.ui.samplerVolBoxShown && !editor.ui.samplerFiltersBoxShown)
            {
                // get real sampling position regardless of where the scope data points to
                tmp32 = (int32_t)(&sc->data[sc->phase] - &modEntry->sampleData[s->offset]);
                if ((tmp32 >= 0) && (tmp32 < s->length))
                {
                    tmp32 = 3 + smpPos2Scr(tmp32);
                    if ((tmp32 >= 3) && (tmp32 <= 316))
                        setSpritePos(SPRITE_SAMPLING_POS_LINE, tmp32, 138);
                }
            }
        }
    }
}

void drawScopes(void)
{
    uint8_t i, x, y, totalVoicesActive;
    int16_t scopeData, scaleValue, scopeTemp, monoScopeBuffer[MONOSCOPE_WIDTH];
    int32_t readPos;
    uint32_t *dstPtr, *scopePtr, scopePixel;
    scopeChannel_t cachedScope, *sc;

    scopesReading = true;
    if (editor.ui.visualizerMode == VISUAL_QUADRASCOPE)
    {
        // --- QUADRASCOPE ---

        scopePtr = &pixelBuffer[(71 * SCREEN_W) + 128];
        for (i = 0; i < AMIGA_VOICES; ++i)
        {
            // cache scope channel to lower thread race condition issues
            memcpy(&cachedScope, &scope[i], sizeof (scopeChannel_t));
            sc = &cachedScope;

            // render scope
            if (sc->active && ((sc->length > 2) || sc->loopFlag) && (sc->volume != 0) && !editor.muted[i] && (sc->data != NULL))
            {
                // scope is active

                scope[i].emptyScopeDrawn = false;

                // draw scope background

                dstPtr = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
                scopePixel = palette[PAL_BACKGRD]; // this palette can change

                for (y = 0; y < SCOPE_HEIGHT; ++y)
                {
                    for (x = 0; x < SCOPE_WIDTH; ++x)
                        dstPtr[x] = scopePixel;

                    dstPtr += SCREEN_W;
                }

                // render scope data

                scopePixel = palette[PAL_QADSCP];

                readPos = sc->phase;
                if (sc->loopFlag)
                {
                    // loop enabled

                    for (x = 0; x < SCOPE_WIDTH; ++x)
                    {
                        if (sc->didSwapData)
                            readPos %= sc->length; // sc->data = loopStartPtr, wrap readPos to 0
                        else if (readPos >= sc->length)
                            readPos = sc->loopStart; // sc->data = sampleStartPtr, wrap readPos to loop start

                        scopeData = sc->data[readPos++] * sc->volume;
                        scopeData = SAR16(scopeData, 8);

                        scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;

                    }
                }
                else
                {
                    // no loop

                    for (x = 0; x < SCOPE_WIDTH; ++x)
                    {
                        scopeData = 0;
                        if (readPos < sc->length)
                        {
                            scopeData = sc->data[readPos++] * sc->volume;
                            scopeData = SAR16(scopeData, 8);
                        }

                        scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;
                    }
                }
            }
            else
            {
                // scope is inactive, draw empty scope once until it gets active again

                sc = &scope[i];
                if (!sc->emptyScopeDrawn)
                {
                    // draw scope background

                    dstPtr     = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
                    scopePixel = palette[PAL_BACKGRD];

                    for (y = 0; y < SCOPE_HEIGHT; ++y)
                    {
                        for (x = 0; x < SCOPE_WIDTH; ++x)
                            dstPtr[x] = scopePixel;

                        dstPtr += SCREEN_W;
                    }

                    // draw line

                    scopePixel = palette[PAL_QADSCP];
                    for (x = 0; x < SCOPE_WIDTH; ++x) 
                        scopePtr[x] = scopePixel;

                    sc->emptyScopeDrawn = true;
                }
            }

            scopePtr += (SCOPE_WIDTH + 8);
        }
    }
    else
    {
        // --- MONOSCOPE ---

        // mix channels

        memset(monoScopeBuffer, 0, sizeof (monoScopeBuffer));

        totalVoicesActive = 0;
        for (i = 0; i < AMIGA_VOICES; ++i)
        {
            // cache scope channel to lower thread race condition issues
            memcpy(&cachedScope, &scope[i], sizeof (scopeChannel_t));
            sc = &cachedScope;

            if (sc->active && ((sc->length > 2) || sc->loopFlag) && (sc->volume != 0) && !editor.muted[i] && (sc->data != NULL))
            {
                // voice is active

                readPos = sc->phase;
                if (sc->loopFlag)
                {
                    // loop enabled

                    for (x = 0; x < MONOSCOPE_WIDTH; ++x)
                    {
                        if (sc->didSwapData)
                            readPos %= sc->length; // sc->data = loopStartPtr, wrap readPos to 0
                        else if (readPos >= sc->length)
                            readPos = sc->loopStart; // sc->data = sampleStartPtr, wrap readPos to loop start

                        monoScopeBuffer[x] += (sc->data[readPos++] * sc->volume);
                    }
                }
                else
                {
                    // no loop

                    for (x = 0; x < MONOSCOPE_WIDTH; ++x)
                    {
                        if (readPos < sc->length)
                            monoScopeBuffer[x] += (sc->data[readPos++] * sc->volume);
                    }
                }

                totalVoicesActive++;
            }
        }

        if (totalVoicesActive == 0)
        {
            if (!emptyMonoscopeDrawn)
            {
                // monoscope is inactive, draw empty monoscope once until it gets active again

                // draw monoscope background

                dstPtr = &pixelBuffer[(55 * SCREEN_W) + 120];
                scopePixel = palette[PAL_BACKGRD];

                for (y = 0; y < MONOSCOPE_HEIGHT; ++y)
                {
                    for (x = 0; x < MONOSCOPE_WIDTH; ++x)
                        dstPtr[x] = scopePixel;

                    dstPtr += SCREEN_W;
                }

                // draw line

                dstPtr = &pixelBuffer[(76 * SCREEN_W) + 120];
                scopePixel = palette[PAL_QADSCP];

                for (x = 0; x < MONOSCOPE_WIDTH; ++x)
                    dstPtr[x] = scopePixel;

                emptyMonoscopeDrawn = true;
            }
        }
        else
        {
            // monoscope is active

            emptyMonoscopeDrawn = false;

            // draw monoscope background

            dstPtr = &pixelBuffer[(55 * SCREEN_W) + 120];
            scopePixel = palette[PAL_BACKGRD];

            for (y = 0; y < MONOSCOPE_HEIGHT; ++y)
            {
                for (x = 0; x < MONOSCOPE_WIDTH; ++x)
                    dstPtr[x] = scopePixel;

                dstPtr += SCREEN_W;
            }

            // render monoscope data

            dstPtr = &pixelBuffer[(76 * SCREEN_W) + 120];
            scopePixel = palette[PAL_QADSCP];

            scaleValue = mixScaleTable[totalVoicesActive - 1];
            for (x = 0; x < MONOSCOPE_WIDTH; ++x)
            {
                scopeTemp = monoScopeBuffer[x] / scaleValue;
                scopeTemp = CLAMP(scopeTemp, -21, 22);

                dstPtr[(scopeTemp * SCREEN_W) + x] = scopePixel;
            }
        }
    }
    scopesReading = false;
}

int32_t SDLCALL scopeThreadFunc(void *ptr)
{
    int32_t time32;
    uint32_t diff32;
    uint64_t time64;

#ifdef _WIN32
    // this is needed for the scopes to stutter slightly less (confirmed)
    SDL_SetThreadPriority(SDL_THREAD_PRIORITY_HIGH);
#endif

    // set next frame time
    timeNext64 = SDL_GetPerformanceCounter() + editor.vblankTimeLen;

    while (editor.programRunning)
    {
        updateScopes();

        // sync scopes to 60Hz (ish)

        time64 = SDL_GetPerformanceCounter();
        if (time64 < timeNext64)
        {
            MY_ASSERT((timeNext64 - time64) <= 0xFFFFFFFFULL);
            diff32 = (uint32_t)(timeNext64 - time64);

#ifdef _WIN32
            // convert to milliseconds
            time32 = (int32_t)((diff32 * editor.perfFreqMulMilli_f) + 0.5);

            // delay until we have reached next tick
            if (time32 > 0)
                SDL_Delay(time32);
#else
            // convert to microseconds
            time32 = (int32_t)((diff32 * editor.perfFreqMulMicro_f) + 0.5);

            // delay until we have reached next tick
            if (time32 > 0)
                usleep(time32);
#endif
        }

        // update next tick time
        timeNext64 += editor.vblankTimeLen;
    }

    (void)(ptr); // make compiler happy

    return (true);
}

uint8_t initScopes(void)
{
    scopeThread = SDL_CreateThread(scopeThreadFunc, "PT Clone Scope Thread", NULL);
    if (scopeThread == NULL)
        return (false);

    // don't let thread wait for this thread, let it clean up on its own when done
    SDL_DetachThread(scopeThread);

    return (true);
}

void waitOnScopes(void)
{
    while (scopesReading);
}

void clearScopes(void)
{
    uint8_t i;

    waitOnScopes();
    memset(scope, 0, sizeof (scope));

    for (i = 0; i < AMIGA_VOICES; ++i)
        scope[i].length = scope[i].newLength = 2;
}
