/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 * + some more network related stuff.
 *
 * Module:  netinit.c
 * Purpose: Initialize the network module
 * Entries: init_net, if_config
 *
 **************************************************************************
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <general.h>
#include <net.h>
#include <romlib.h>
#include <arpa.h>
#include <system.h>		/* LOADBASE */
#include "./netpriv.h"
#include "./udp.h"
#include "./ip.h"
#include "./arp.h"



/*
 **************************************************************************
 * 
 * Global variables for the network library
 *
 */
unsigned char     myhwaddr[ETH_ALEN];		/* my own hardware addr	*/
         t_ipaddr myipaddr;			/* my own IP address	*/
         t_ipaddr mynetmask;			/* my own netmask	*/
         char    *net_module_name;		/* name of init module	*/
         t_ipaddr servaddr;			/* IP of RARP&TFTP server */

/* Broadcast hardware address */
unsigned char bcasthw[ETH_ALEN] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };



/*
 **************************************************************************
 * 
 * Interface configuration: sets all IP addresses which are related to
 * a specific interface.
 */
void if_config(myip, netmask)
t_ipaddr myip;
t_ipaddr netmask;
{
  /* Adjust netmask if none given */
  if (netmask == IP_ANY) {
	if (IN_CLASS_A(myip))
		netmask = IP_CLASS_A;
	else if (IN_CLASS_B(myip))
		netmask = IP_CLASS_B;
	else if (IN_CLASS_C(myip))
		netmask = IP_CLASS_C;
  }

  /* Now set the interface values */
  myipaddr  = myip;
  mynetmask = netmask;
}



/*
 **************************************************************************
 * 
 * Get hardware address
 */
void get_hw_addr(hwaddr)
unsigned char *hwaddr;
{
  memcpy(hwaddr, myhwaddr, ETH_ALEN);
}



/*
 **************************************************************************
 * 
 * Network intialization:
 */
void init_net()
{
  /* Initialize the different network layer modules */
  init_packet();
  if (!init_arp() || !init_udp()) {
	printf("\nERROR: init_%s\n", net_module_name);
	fatal();
  }
}


/* return ((n >> 8) & 0xFF) | ((n & 0xFF) << 8); */
/* It's nice to be on a big-endian machine. */
unsigned short _htons(unsigned short n) {  return n; }
unsigned long _htonl(unsigned long n) {  return n; }

/*
 * This routine implements the most stupid way to compute an Internet Checksum.
 * This routine has one advantage though: it's GPLed :)
 */
int
ipchksum(unsigned char *data, int leng)
{
	long sum;

	sum = 0;
	while (leng >= 2) {
		leng -= 2;
		sum += (data[0] << 8) | data[1];
		sum += (sum>>16) & 1;
		sum &= 0xFFFF;
		data += 2;
	}
	if (leng) {
		sum += *data << 8;
		sum += (sum>>16) & 1;
		sum &= 0xFFFF;
	}

	return (~sum) & 0xFFFF;
}

/*
 * Load boot image from server
 */
int load(unsigned int saddr, char *fname)
{
  int len;
  char *inbuf;
  char *loadptr;
  int csum = 0;

  loadptr = (char *)LOADBASE;

  /* Read the first block with the file header */
  if ((inbuf = tftp_open(ntohl(saddr), fname, strlen(fname))) == NULL)
	return(-1);
  if ((len = tftp_get()) < 0)
	return(-1);
  if (ld4(inbuf) == 0x01030108) {	/* Our private magic */
	csum = ld4(inbuf + 0x10) | 0x10000;
  } else if (ld4(inbuf) != AOUT_MAGIC) {
	printf("load: No a.out magic in the image\n");
	return(-1);
  }
  if (len < 32) {
	printf("load: Image too short\n");
	return(-1);
  }

  bcopy(inbuf+32, loadptr, len-32);
  loadptr += len-32;

  /* Read all blocks of image file */
  while ((len = tftp_get()) > 0) {
	bcopy(inbuf, loadptr, len);
	loadptr += len;
  }
  if (len < 0) {
	printf("load: failure after %d\n", loadptr-(char *)LOADBASE);
	return(-1);
  }

  len = loadptr-(char *)LOADBASE;
  printf("load: done %d(0x%x) bytes\n", len, len);
  if (csum) {
	if (ipchksum((char *)LOADBASE, len) == (csum & 0xFFFF)) {
		printf("load: checksum ok\n");
	} else {
		printf("load: checksum mismatch\n");
	}
  }

#if 0
  /* Discard any leftover stuff. This will also close the TFTP connection */
  while (len == SEGSIZE)
	len = tftp_get();
#endif

  return(0);
}

/*
 * Load a short file from the TFTP server.
 */
int load_cfg(unsigned int saddr, char *fname, char *buf, int bsize)
{
	char *inbuf;
	int total;
	int len;

	total = 0;

	/* Read the first block with the file header */
	if ((inbuf = tftp_open(ntohl(saddr), fname, strlen(fname))) == NULL)
		return -1;

	while (total < bsize && (len = tftp_get()) > 0) {
		if (len > bsize-total)
			len = bsize-total;
		bcopy(inbuf, buf, len);
		buf += len;
		total += len;
	}

#if 0
  /* Discard any leftover stuff. This will also close the TFTP connection */
  while (len == SEGSIZE)
	len = tftp_get();
#endif
	return total;
}
