/***************************************************************************
                          tagger.cpp  -  description
                             -------------------
    begin                : Wed May 9 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "tagger.h"

#if HAVE_ID3
#include "tagger_mp3.h"
#endif


#if HAVE_OGG
#include "tagger_oggvorbis.h"
#endif

#define MAXGENRE   147

char *genretable [MAXGENRE+2] = {
  "Blues","Classic Rock","Country","Dance","Disco","Funk","Grunge", "Hip-Hop","Jazz","Metal",
  "New Age","Oldies","Other","Pop","R&B", "Rap","Reggae","Rock","Techno","Industrial",
  "Alternative","Ska","Death Metal","Pranks","Soundtrack","Euro-Techno","Ambient","Trip-Hop","Vocal","Jazz+Funk",
  "Fusion","Trance","Classical","Instrumental","Acid","House","Game","Sound Clip","Gospel","Noise",
  "Alt. Rock","Bass","Soul","Punk","Space","Meditative","Instrumental Pop","Instrumental Rock","Ethnic","Gothic",
  "Darkwave","Techno-Industrial","Electronic","Pop-Folk","Eurodance","Dream","Southern Rock","Comedy","Cult","Gangsta Rap",
  "Top 40","Christian Rap","Pop/Funk","Jungle","Native American","Cabaret","New Wave","Psychedelic","Rave","Showtunes",
  "Trailer","Lo-Fi","Tribal","Acid Punk","Acid Jazz","Polka","Retro","Musical","Rock & Roll","Hard Rock",
  "Folk","Folk/Rock","National Folk","Swing","Fast-Fusion","Bebob","Latin","Revival","Celtic","Bluegrass",
  "Avantgarde","Gothic Rock","Progressive Rock","Psychedelic Rock","Symphonic Rock","Slow Rock","Big Band","Chorus","Easy Listening","Acoustic",
  "Humour","Speech","Chanson","Opera","Chamber Music","Sonata","Symphony","Booty Bass","Primus","Porn Groove",
  "Satire","Slow Jam","Club","Tango","Samba","Folklore","Ballad","Power Ballad","Rhythmic Soul","Freestyle",
  "Duet","Punk Rock","Drum Solo","A Cappella","Euro-House","Dance Hall","Goa","Drum & Bass","Club-House","Hardcore",
  "Terror","Indie","BritPop","Negerpunk","Polsk Punk","Beat","Christian Gangsta Rap","Heavy Metal","Black Metal","Crossover",
  "Contemporary Christian","Christian Rock","Merengue","Salsa","Thrash Metal","Anime","JPop","Synthpop",""
};

int alphagenreindex[MAXGENRE+2] = {
  148,123,74,73,34,99,40,20,26,145,
  90,116,41,135,85,96,138,89,0,107,
  132,65,88,104,102,97,136,61,141,1,
  32,128,112,57,140,2,139,58,125,3,
  50,22,4,55,127,122,120,98,52,48,
  124,25,54,84,81,115,80,119,5,30,
  36,59,126,38,91,49,6,79,129,137,
  7,35,100,131,19,46,47,33,146,29,
  8,63,86,71,45,142,9,77,82,64,
  133,10,66,39,11,103,12,75,134,53,
  62,13,109,117,23,108,92,93,67,121,
  43,14,15,68,16,76,87,118,78,17,
  143,114,110,69,21,111,95,105,42,37,
  24,56,44,101,83,94,106,147,113,51,
  18,130,144,60,70,31,72,27,28
};

int genrealphaindex[MAXGENRE+2] = {
  18,29,35,39,42,58,66,70,80,86,
  91,94,96,101,111,112,114,119,140,74,
  7,124,41,104,130,51,8,147,148,79,
  59,145,30,77,4,71,60,129,63,93,
  6,12,128,110,132,84,75,76,49,65,
  40,139,48,99,52,43,131,33,37,61,
  143,27,100,81,89,21,92,108,113,123,
  144,83,146,3,2,97,115,87,118,67,
  56,54,88,134,53,14,82,116,22,17,
  10,64,106,107,135,126,15,25,47,5,
  72,133,24,95,23,147,136,19,105,102,
  122,125,32,138,121,55,11,103,117,57,
  46,109,45,1,50,38,62,44,31,68,
  141,73,20,90,98,13,26,69,16,36,
  34,28,85,120,142,9,78,137,0
};

Tagger::Tagger(){}

Tagger::~Tagger(){}
	
bool Tagger::isKnownFile(QFileInfo *fi)
{
#if HAVE_ID3
  if(fi->extension(false).upper() == "MP3") return true;
#endif
#if HAVE_OGG
  if(fi->extension(false).upper() == "OGG") return true;
#endif
  return false;
}

bool Tagger::isExtensionValid(TRACK *track)
{
  QString ext = QFileInfo(track->filename).extension(false).upper();
#if HAVE_ID3
  if(ext == "MP3" && track->mimetype == MIMETYPE_MP3) return true;
#endif
#if HAVE_OGG
  if(ext == "OGG" && track->mimetype == MIMETYPE_OGG) return true;
#endif
  return false;
}

bool Tagger::getInfo(TRACK *track, bool lazyRead)
{
  QString ext = QFileInfo(track->filename).extension(false).upper();
#if HAVE_ID3
  if(ext == "MP3") return Tagger_MP3::getInfo(track, lazyRead);
#endif
#if HAVE_OGG
  if(ext == "OGG") return Tagger_OggVorbis::getInfo(track, lazyRead);
#endif
  return false;	// should never happen
}

bool Tagger::getTagOnly(TRACK *track)
{
  QString ext = QFileInfo(track->filename).extension(false).upper();
  
#if HAVE_ID3
  if(ext == "MP3") return Tagger_MP3::getTagOnly(track);
#endif
#if HAVE_OGG
  if(ext == "OGG") return Tagger_OggVorbis::getTagOnly(track);
#endif
  return false;	// should never happen
}

bool Tagger::writeInfo(TRACK *track)
{
  QString ext = QFileInfo(track->filename).extension(false).upper();
  
#if HAVE_ID3
  if(ext == "MP3") return Tagger_MP3::writeInfo(track);
#endif
#if HAVE_OGG
  if(ext == "OGG") return Tagger_OggVorbis::writeInfo(track);
#endif
  return false;	// should never happen
}

QString Tagger::getGenre(int genreNum)
{
  if(genreNum < 0 || genreNum > MAXGENRE+1) return "";
  else return genretable[genreNum];
}

QString Tagger::getGenreAlpha(int alphaNum)
{
  if(alphaNum < 0 || alphaNum > MAXGENRE+1) return "";
  else return genretable[alphagenreindex[alphaNum]];
}

int Tagger::genreToAlphaIndex(int genre)
{
  if(genre < 0 || genre > MAXGENRE+1) return 0;
  else return genrealphaindex[genre]; 
}

int Tagger::alphaToGenreIndex(int alpha)
{
  if(alpha < 0 || alpha > MAXGENRE+1) return 255;
  else return alphagenreindex[alpha]; 
}

QString Tagger::getTypeAsString(TRACK *track)
{
  if(track->mimetype == MIMETYPE_MP3) {
    QString xtype = "MPEG 1";
    if(track->version == 1) xtype = "MPEG 2.5";
    if(track->version == 2) xtype = "MPEG 2";
    QString xlayer = " Layer III";
    if(track->layer == 2) xlayer = " Layer II";
    if(track->layer == 1) xlayer = " Layer I";
    return xtype + xlayer;
  }
  if(track->mimetype == MIMETYPE_OGG) {
    return "Ogg/Vorbis";
  }
  return _("Unknown"); // should never happen
}

QString Tagger::getModeAsString(TRACK *track)
{
  QString xmodus = "stereo";
  if(track->mode == 1) xmodus = "jstereo";
  if(track->mode == 2) xmodus = "2channel";
  if(track->mode == 3) xmodus = "1channel";
  QString xmode;
  if(track->bitrate < 0) xmode.sprintf("%dkb VBR, %dHz, %s", -track->bitrate, track->samplerate, xmodus.latin1());
  else xmode.sprintf("%dkb, %dHz, %s", track->bitrate, track->samplerate, xmodus.latin1());
  return xmode;
}
  
QString Tagger::getSizeAsString(TRACK *track)
{
  QString xfsize;
  xfsize.sprintf("%dkb (%d:%02d) ", track->size/1024, track->length/60, track->length%60);
  return xfsize;
}
