/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include <sys/wait.h>
#include <sys/time.h>

#include "event_management.h"
#include "main_loop.h"
#include "brightness.h"
#include "clientConnection.h"
#include "cpufreq_management.h"
#include "globals.h"

using namespace Powersave::Globals;

/* apm workaround variables for sleep modes ******************/
PSD_SLEEP_STATE_REQUESTS PM_Interface::_requested_sleep_state = PSD_NO_SLEEP_REQUEST;
bool PM_Interface::_sleep_triggered = false;
/* apm workaround variables for sleep modes ******************/

int PM_Interface::rereadConfig()
{
	int ret = config_obj->readConfFiles();

	/* reset/reread everything that could have been changed in
	   configs ... */
	if (cpufreq->isSupported()) {
		cpufreq->setConfigs();
	}
	activateSettings();
	return ret;
}


PM_Interface::PM_Interface()
{
	pDebug(DBG_DEBUG, "Constructor PM_Interface");

	// initialise everything we have in the constructor !
	// acpi/apm specifc things are initialised in the
	// appropriate ACPI/APM_Interface constructor

	// **** I N I T   S E C T I O N *************************/

	// this should never happen, hopefully the term signal cleans up memory and stops app */
	if (!config_obj || !config_obj->current_scheme || !config_obj->AC_scheme || !config_obj->Battery_scheme) {
		pDebug(DBG_ERR, "Current, AC or Battery scheme is not set. Exit.");
		kill(getpid(), SIGTERM);
	}

	_supported_sleeping_states = getSupportedSleepStates();

	_cooling_mode_supported = true;
	_thermal_trip_points_supported = true;

	// allocate memory for event management object
	_eM = new EventManagement();

	// create a new client connection object
	_cC = new ClientConnection();

	_brightness = Brightness::Probe();
	_brightness->Init();
}

PM_Interface::~PM_Interface()
{

	pDebug(DBG_DEBUG, "PM Destructor: delete allocated Objects");

	_eM->executeEvent("daemon.terminate");
	delete(_cC);
	delete(_eM);
	delete(_brightness);
}

int PM_Interface::start()
{
	// ********  trigger start event ****************/
	_eM->executeEvent("daemon.start");

	_devices.init();	

	if (_battery.updateBatteryState(_cur_state.AC_STATE, _eM) < 0) {
		pDebug(DBG_INFO, "No battery available.");
	} else {
		pDebug(DBG_INFO, "There is at least one battery available.");
		// reset battery state, so the first checkBatteryChanges throws a battery
		// event if necessary
		_battery.resetState();
	}

	// Warning: when we start, the scheme should change from "nothing" to "active
	// scheme", a daemon.scheme.change event should be thrown and activateSettings
	// should be called. In reality, it changes from "Performance" to "active scheme",
	// which on AC is often "Performance" => no change => no event => no activate.
	// Not fatal, but we should fix it. In the meantime, we work around it here:
	activateSettings();
	_eM->executeEvent("daemon.scheme.change");
	
	/* this also sets current scheme, do not change order */
	updateACState();
	activateScheme(_cur_state.AC_STATE);
	
	// if the battery is already low, throw an appropriate event.
	checkBatteryStateChanges();

	return 1;
}

void PM_Interface::checkBatteryStateChanges()
{

	pDebug(DBG_DEBUG, "Check battery state");
	// only switch if a lower battery level is reached
	if (_battery.updateBatteryState(_cur_state.AC_STATE, _eM) == 1) {
		switch (_battery.batteryState()) {
		case BAT_NORM_STATE:
			_eM->executeEvent("battery.normal");
			pDebug(DBG_INFO, "Battery state changed to normal");
			break;
		case BAT_WARN_STATE:
			_eM->executeEvent("battery.warning");
			pDebug(DBG_INFO, "Battery state changed to warning");
			break;
		case BAT_LOW_STATE:
			_eM->executeEvent("battery.low");
			pDebug(DBG_INFO, "Battery state changed to low");
			break;
		case BAT_CRIT_STATE:
			_eM->executeEvent("battery.critical");
			pDebug(DBG_INFO, "Battery state changed to critical");
			break;
		case BAT_NONE_STATE:
			pDebug(DBG_INFO, "Battery disappeared.");
			break;
		default:
			pDebug(DBG_ERR, "Undefined battery state ..., that's bad");
		}
	} else {
		pDebug(DBG_INFO, "Battery state unchanged");
	}
}

int PM_Interface::checkACStateChanges()
{
	AC_ADAPTER_STATE ac_state;
	if (updateACState()) {
		pDebug(DBG_DIAG, "AC state changed");
		ac_state = _cur_state.AC_STATE;

		if (ac_state == AC_OFFLINE) {
			// set battery state
			_eM->executeEvent("acadapter.offline");
			pDebug(DBG_INFO, "ac adapter offline event occured");
			return 1;
		} else if (ac_state == AC_ONLINE) {
			_eM->executeEvent("acadapter.online");
			pDebug(DBG_INFO, "ac adapter online event occured");
			return 1;
		} else if (ac_state == AC_UNKNOWN) {
			// assume online for now...
			pDebug(DBG_DIAG, "AC state changed to unknown, assume online");
			ac_state = AC_ONLINE;
			return 1;
		} else {
			pDebug(DBG_ERR, "Could not read out AC adapter state, FATAL ERROR");
			return -1;
		}
	} else
		return 0;
}

void PM_Interface::checkTemperatureStateChanges()
{
	int ret = 0;
	pDebug(DBG_DEBUG, "Check temperature State");
	ret = updateTemperatureState();
	/* it is getting hotter */
	if (ret == 1) {
		switch (_cur_state.TEMP_STATE) {
		case CRITICAL:
			_eM->executeEvent("temperature.critical");
			pDebug(DBG_WARN, "Temperature state changed to critical.");
			break;
		case HOT:
			_eM->executeEvent("temperature.hot");
			pDebug(DBG_WARN, "Temperature state changed to hot.");
			break;
		case PASSIVE:
			_eM->executeEvent("temperature.passive");
			pDebug(DBG_INFO, "Temperature state changed to passive.");
			break;
			/* I do not differ between ACTIVE (fans on) and
			   OK mode (no cooling) at the moment */
		case ACTIVE:
			_eM->executeEvent("temperature.active");
			pDebug(DBG_INFO, "Temperature state changed to active.");
			break;
		case OK:
			pDebug(DBG_DIAG, "Temperature state changed to ok - this should"
			       "not happen at this point ->bug ->mail@renninger.de.");
			break;
		default:
			pDebug(DBG_DIAG, "Undefined temperature state ->bug ->mail@renninger.de.");
			break;
		}
	}
	/* it is getting cooler */
	else if (ret == 2) {
		switch (_cur_state.TEMP_STATE) {
		case CRITICAL:
			_eM->executeEvent("temperature.critical");
			pDebug(DBG_INFO, "Temperature state changed to critical. This should"
			       "not happen at this point ->bug ->mail@renninger.de.");
			break;
		case HOT:
			_eM->executeEvent("temperature.hot");
			pDebug(DBG_INFO, "Temperature state changed to hot.");
			break;
		case PASSIVE:
			_eM->executeEvent("temperature.passive");
			pDebug(DBG_INFO, "Temperature state changed to passive.");
			break;
			/* I do not differ between ACTIVE (fans on) and
			   OK mode (no cooling) at the moment */
		case ACTIVE:
			_eM->executeEvent("temperature.active");
			pDebug(DBG_INFO, "Temperature state changed to active.");
			break;
		case OK:
			_eM->executeEvent("temperature.ok");
			pDebug(DBG_INFO, "Temperature state changed to ok.");
			break;
		default:
			pDebug(DBG_INFO, "Undefined temperature state ->bug ->mail@renninger.de.");
			break;
		}
	}
}


bool PM_Interface::checkThrottling()
{
	return _throttleInterface.updateCPUState(_eM);
}

void PM_Interface::checkDevice(char *udi, bool added)
{
	if (added) {
		_devices.checkDeviceAdded(udi);
	}
	else {
		_devices.checkDeviceRemoved(udi);
	}
}

int PM_Interface::updateACState()
{
	int changed;
	int old_ac_state;

	old_ac_state = _cur_state.AC_STATE;
	_cur_state.AC_STATE = (AC_ADAPTER_STATE) getACAdapterStatus();

	/* there are new error codes returned 
	   (e.g. NO_DEVICE_ERROR and NO_MODULE_ERROR) */
	if (_cur_state.AC_STATE != AC_ONLINE && _cur_state.AC_STATE != AC_OFFLINE) {
		pDebug(DBG_INFO, "Could not detect AC status, assume AC_ONLINE");
		_cur_state.AC_STATE = AC_ONLINE;
	}
	if (old_ac_state != _cur_state.AC_STATE) {
		changed = 1;
		if (activateScheme(_cur_state.AC_STATE) > 0) {
			_eM->executeEvent("daemon.scheme.change");
		}
	} else
		changed = 0;
	return changed;
}

int PM_Interface::updateTemperatureState()
{

	int x;
	int state;
	/* is also THERM_MODE but C (library) forbids the declaration */
	int temp_state = OK;

	for (x = 0; config_obj->current_scheme->thermal_zones[x].present &&
		    ((state = getThermalZoneState(x)) >= OK) &&
		    x < MAX_THERMAL_ZONES; x++) {
		config_obj->current_scheme->thermal_zones[x].state = state;
		temp_state = max(state, temp_state);
	}
	/* here we need more proper error handling -> destroy allocated objects
	   probably the thermal module has been unloaded ... */
	if (config_obj->current_scheme->thermal_zones[x].present) {
		pDebug(DBG_DIAG, "Error during temperature evaluation");
		return -1;
	}
	if (temp_state > _cur_state.TEMP_STATE) {
		_cur_state.TEMP_STATE = (THERM_MODE) temp_state;
		return 1;
	} else if (temp_state < _cur_state.TEMP_STATE) {
		_cur_state.TEMP_STATE = (THERM_MODE) temp_state;
		return 2;
	} else
		return 0;
}

int PM_Interface::activateScheme(const string &scheme_name)
{

	int ret;
	if (!config_obj->current_scheme->SCHEME_NAME.compare(scheme_name)) {
		pDebug(DBG_DIAG, "Scheme already set");
		return 0;
	}
	ret = config_obj->setActiveScheme(scheme_name);
	if (ret <= 0)
		return ret;

	pDebug(DBG_DIAG, "switched scheme to: %s", scheme_name.c_str());

	if (scheme_name == "AdvancedPowersave") {
		pDebug(DBG_WARN, "AdvancedPowersave scheme activated. This scheme "
		       "is highly experimental. Please be careful!");
	}


	activateSettings();
	return 1;
}

int PM_Interface::activateScheme(AC_ADAPTER_STATE ac_state)
{

	int ret = 0;
	switch (ac_state) {
	case AC_ONLINE:
		ret = activateScheme(config_obj->getACScheme());
		break;
	case AC_OFFLINE:
		ret = activateScheme(config_obj->getBatteryScheme());
		break;
	case AC_UNKNOWN:
		// which one should we use -> I'd say better use AC
		// we could even test for battery level, or is this
		// too overdosed?
		ret = activateScheme(config_obj->getACScheme());
		break;
	}
	return ret;
}

void PM_Interface::activateSettings()
{
	// Switch kernel/userspace control during runtime
	if (config_obj->current_scheme->CPUFREQ_CONTROL != cpufreq->controlMode()
		|| !cpufreq->isSupported()) {

		pDebug(DBG_DIAG, "Change cpufreq control to: %s",
		       ((config_obj->current_scheme->CPUFREQ_CONTROL == CPUFREQ_USERSPACE)
			? "userspace" : "kernel"));

		delete cpufreq;
		cpufreq = NULL;
		cpufreq = new CpufreqManagement();
		cpufreq->init();
	}

	cpufreq->setConfigs();

	/* set new CPU freq mode */
	if (!cpufreq->setModes(config_obj->current_scheme->CPUFREQUENCY, _eM) < 0) {
		pDebug(DBG_WARN, "Could not set CPUFreq mode");
	}

	// only adjust brightness if we have no client caring about
	if (!_cC->checkClientCapabilities(CAPABILITY_BRIGHTNESS)) {
		switch (config_obj->current_scheme->DISPLAY_BRIGHTNESS) {
		case -1:		// ignore
			break;
		case -2:		// low
			_brightness->Min();
			break;
		case -3:		// medium
			_brightness->Med();
			break;
		case -4:		// high
			_brightness->Max();
		break;
		default:		// level
			_brightness->Set(config_obj->current_scheme->DISPLAY_BRIGHTNESS);
		break;
		}
	}
	
	_devices.update(config_obj->current_scheme->dpmClasses);

	main_loop->updateCallbacks();
}

void PM_Interface::throttle(int percent)
{
	if (_cur_state.TEMP_STATE < PASSIVE) {
		_throttleInterface.throttle(percent);
	}
}

void PM_Interface::dethrottle()
{
	if (_cur_state.TEMP_STATE < PASSIVE) {
		_throttleInterface.dethrottle();
	}
}

void PM_Interface::resume(CPUFREQ_MODE cpufreq_mode)
{
	pDebug(DBG_DIAG, "Daemon reinitialises system states, resume finished");
	_requested_sleep_state = PSD_NO_SLEEP_REQUEST;
	// reset frequency which may have changed without noticing it
	cpufreq->reinitSpeeds();
	/* reset CPU freq mode */
	if (cpufreq->isSupported())
		cpufreq->setModes(cpufreq_mode, _eM);

	checkACStateChanges();
	checkBatteryStateChanges();
}

void PM_Interface::sleep_wrapper(const string &event)
{
	static bool before_sleep = true;
	static struct timeval cur_time;
	static struct timezone dummy;
	static CPUFREQ_MODE cpufreq_mode;

	if (before_sleep) {
		if (event == "suspend2disk") {
			if (config_obj->current_scheme->SUSPEND2DISK_DELAY > 0)
				sleep(config_obj->current_scheme->SUSPEND2DISK_DELAY);
		} else if (event == "suspend2ram") {
			if (config_obj->current_scheme->SUSPEND2RAM_DELAY > 0)
				sleep(config_obj->current_scheme->SUSPEND2RAM_DELAY);
		} else if (event == "standby") {
			if (config_obj->current_scheme->STANDBY_DELAY > 0)
				sleep(config_obj->current_scheme->STANDBY_DELAY);
		}
		ignore_buttons(true);

		if (cpufreq->isSupported()) {
			cpufreq_mode = cpufreq->getMode();
			cpufreq->setModes(_PERFORMANCE, _eM);
		}

		pDebug(DBG_INFO, "Set machine into sleep mode");

		before_sleep = false;
		gettimeofday(&cur_time, &dummy);
	} else {
		// the time where we went to sleep in milliseconds
		unsigned long sleeptime = cur_time.tv_sec * 1000 + cur_time.tv_usec / 1000;

		gettimeofday(&cur_time, &dummy);

		// the time were we are back in milliseconds
		unsigned long wakeuptime = cur_time.tv_sec * 1000 + cur_time.tv_usec / 1000;

		// substract the time we slept from all events
		_eM->adjustTimeouts(wakeuptime - sleeptime);
		pDebug(DBG_INFO, "Back from %s inside sleep_wrapper", event.c_str());

		resume(cpufreq_mode);

		before_sleep = true;

		// throw a resume event
		_eM->executeEvent("global.resume." + event);
	}
}

void PM_Interface::suspendNetworkManager()
{
	if (!DBus_Server::serviceHasOwner(DBUS_NM_SERVICE)) {
		pDebug(DBG_DIAG, "Networkmanager not running");
		return;
	}

	DBus_Server::sendMethodCall(DBUS_NM_SERVICE, DBUS_NM_PATH,
				    DBUS_NM_INTERFACE, "sleep");
}

void PM_Interface::resumeNetworkManager()
{
	if (!DBus_Server::serviceHasOwner(DBUS_NM_INTERFACE)) {
		pDebug(DBG_DIAG, "Networkmanager not running");
		return;
	}

	DBus_Server::sendMethodCall(DBUS_NM_SERVICE, DBUS_NM_PATH,
				    DBUS_NM_INTERFACE, "wake");
}

int PM_Interface::executeScript(const string &script, const string &param)
{
	int ret = -1;
	int pid;
	int status;
	char *argv[3];
	argv[0] = new char[script.size() + 1];
	argv[1] = new char[param.size() + 1];
	strcpy(argv[0], script.c_str());
	strcpy(argv[1], param.c_str());
	argv[2] = NULL;

	string full_path_to_script(SCRIPT_DIR);
	full_path_to_script.append("/" + script);

	pid = fork();
	if (!pid) {
		signal(SIGHUP, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
		signal(SIGINT, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
		signal(SIGPIPE, SIG_DFL);
		signal(SIGCHLD, SIG_DFL);
		// close all fds
		int fdlimit = sysconf(_SC_OPEN_MAX);

		for (int fd = 0; fd < fdlimit; fd++)
			close(fd);

		// set environment variables for scripts
		setenv("KDE_BINDIR", KDE_BINDIR, 1);
		setenv("GNOME_BINDIR", GNOME_BINDIR, 1);
		setenv("SCRIPT_DIR", config_obj->script_dir.c_str(), 1);
		setenv("PUB_SCRIPT_DIR", config_obj->pub_script_dir.c_str(), 1);
		setenv("SYSCONF_DIR", config_obj->config_dir.c_str(), 1);
		setenv("BIN_DIR", BIN_DIR, 1);

		execv(full_path_to_script.c_str(), argv);

		// we should never get here.
		pDebug(DBG_ERR, "could not execv the sleep script '%s'. Argument: '%s', error: '%s'",
		       full_path_to_script.c_str(), argv[1], strerror(errno));
		_exit(127);
	} else if (pid == -1) {
		pDebug(DBG_WARN, "Could not fork for sleep script");
	} else {
		// if the script is the sleep script, wait for its return
		if (script == ACPI_SLEEP_SCRIPT) {

			// wait for the sleep script in case of a suspend
			signal(SIGCLD, SIG_DFL);

			if (waitpid(pid, &status, 0) < 0) {
				if (errno == ECHILD) {
					pDebug(DBG_DIAG, "Sleep script already exited.");
				} else {
					pDebug(DBG_ERR, "Sleep script failed - %s", strerror(errno));
				}
			} else if (WIFEXITED(status)) {
				if (WEXITSTATUS(status)) {
					pDebug(DBG_WARN,
					       "Program to trigger sleep (%s) failed and exited with status %d",
					       full_path_to_script.c_str(), WEXITSTATUS(status));
					ret = -1;
				} else {
					/* everything is alright, go on executing ... */
					pDebug(DBG_INFO, "Program %s to trigger sleep executed successfully",
					       full_path_to_script.c_str());
					ret = 0;
				}
			} else if (WIFSIGNALED(status)) {
				pDebug(DBG_DIAG, "Program %s exited on signal %d",
				       full_path_to_script.c_str(), WTERMSIG(status));
				ret = -1;
			}

			// ignore children further on
			signal(SIGCLD, SIG_IGN);
		}
		// no need to wait for script to return
	}
	return ret;
}

int PM_Interface::screenlock(const string &param)
{

	DBus_Server::emitSignal("Screenlock", param.c_str());

	if (!_cC->checkClientCapabilities(CAPABILITY_SCREENLOCK)) {
		executeScript("do_screen_saver", param);
	}
	return 1;
}

int PM_Interface::x_notification(const string &param)
{
	DBus_Server::emitSignal("Notification", param.c_str());

	if (!_cC->checkClientCapabilities(CAPABILITY_NOTIFICATIONS)) {
		executeScript("do_x_notification", param);
	}
	return 1;
}

void PM_Interface::ignore_buttons(bool ignore)
{
	pDebug(DBG_DIAG, "ignore_buttons is set to: %d", ignore);
	_sleep_triggered = ignore;
}

bool PM_Interface::haveSuspend2()
{
	if (access("/proc/suspend2", F_OK) < 0) {
                return false;
	}

	return true;
}

DBUS_ERROR PM_Interface::handleDBusRequest(DBusMessage * msg, DBusMessage * reply, POWERSAVE_MESSAGE_TYPE type)
{

	string msg_member = dbus_message_get_member(msg);
	string sender = dbus_message_get_sender(msg);
	uint32_t serial = dbus_message_get_serial(msg);

	/** @todo: Debug only, throw out when working... */
	pDebug(DBG_INFO, "Received msg: '%s'", msg_member.c_str());
	pDebug(DBG_DEBUG, "Obj.path '%s' serial '%d' from '%s'", dbus_message_get_path(msg), serial, sender.c_str());

	switch (type) {
	case MANAGER_MESSAGE:
		return handleManagerMessage(msg);
		break;
	case SCRIPTS_MESSAGE:	       
		return handleScriptsMessage( msg );
		break;
	case REQUEST_MESSAGE:
		return handleRequestMessage(msg, reply);
		break;
	case ACTION_MESSAGE:
		return handleActionMessage(msg);
		break;
	default:
		/* Should never happen -> get rid of message? */
		pDebug(DBG_ERR, "Received invalid message");
		return REPLY_INVALID_INTERFACE;
	}

	return REPLY_GENERAL_ERROR;
}

DBUS_ERROR PM_Interface::handleManagerMessage(DBusMessage * msg)
{
	string msg_member = dbus_message_get_member(msg);
	string sender = dbus_message_get_sender(msg);

	pDebug(DBG_DIAG, "Handling manager request");
	if (msg_member == "Connect") {
		int capabilities;
		/* add client. No 'request string/modes' check so far */
		if (!dbusGetMessageInteger(msg, &capabilities, 0)) {
			_cC->addClient(sender.c_str(), capabilities);
			return REPLY_SUCCESS;
		} else
			return REPLY_INVALID_PARAM;
	} else if (msg_member == "Disconnect") {
		_cC->checkClientRemoval(sender.c_str());
		return REPLY_SUCCESS;
	} else
		return REPLY_INVALID_METHOD;

	return REPLY_GENERAL_ERROR;
}

DBUS_ERROR PM_Interface::handleScriptsMessage( DBusMessage *msg ) {
 	string msg_member = dbus_message_get_member(msg);
 	string sender = dbus_message_get_sender(msg);
 
 	char *dummy = "";

	pDebug (DBG_DIAG, "Handling Scripts request");
	if (msg_member == "ScriptReturn") {
		int event_id, request;

		// get the event id
		if (dbusGetMessageInteger(msg, &event_id, 0) < 0) {
			pDebug(DBG_WARN, "Script returned but we could not get the event id");
			return REPLY_INVALID_PARAM;
		}
		// get the so called mode ( notify, progress, true, false )
		if (!dbusGetMessageInteger(msg, &request, 1) < 0) {
			pDebug(DBG_WARN, "Script returned but we could not to get the request");
			return REPLY_INVALID_PARAM;
		}
		// get the message
		if (dbusGetMessageString(msg, &dummy, 0) < 0) {
			pDebug(DBG_INFO, "Script returned but we could not to get the message."
			       " Continuing anyway...");
			dummy = "";
		}

		pDebug(DBG_DEBUG, "Script returned: event id: %d, request: %d, message: '%s'", event_id, request, dummy);

		string str_dummy = dummy;
		_eM->checkScriptReturn(event_id, request, str_dummy);

		return REPLY_SUCCESS;
	} else
		return REPLY_INVALID_METHOD;

	return REPLY_GENERAL_ERROR;
}

DBUS_ERROR PM_Interface::handleRequestMessage(DBusMessage * msg, DBusMessage * reply)
{

	string msg_member = dbus_message_get_member(msg);
	char *dummy = "";

	/** @todo Also reply REPLY_HW_NOT_SUPPORTED for suspend*/
	if (msg_member == "AllowedSuspendToDisk" || msg_member == "suspend_to_disk_allowed") {
		if (config_obj->current_scheme->USER_SUSPEND2DISK_DISABLED)
			return REPLY_DISABLED;
		else
			return REPLY_SUCCESS;
	} else if (msg_member == "AllowedSuspendToRam" || msg_member == "suspend_to_ram_allowed") {
		if (config_obj->current_scheme->USER_SUSPEND2RAM_DISABLED)
			return REPLY_DISABLED;
		else
			return REPLY_SUCCESS;
	} else if (msg_member == "AllowedStandby" || msg_member == "standby_allowed") {
		if (config_obj->current_scheme->USER_STANDBY_DISABLED)
			return REPLY_DISABLED;
		else
			return REPLY_SUCCESS;
	} else if (msg_member == "BatteryState") {
		switch (_battery.batteryState()) {
		case BAT_NONE_STATE:
			return REPLY_HW_NOT_SUPPORTED;
			break;
		case BAT_NORM_STATE:
			dummy = "normal";
			break;
		case BAT_WARN_STATE:
			dummy = "warning";
			break;
		case BAT_LOW_STATE:
			dummy = "low";
			break;
		case BAT_CRIT_STATE:
			dummy = "critical";
			break;
		default:
			return REPLY_GENERAL_ERROR;
			break;
		}
		dbus_message_append_args(reply, DBUS_TYPE_STRING, &dummy, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;
	} else if (msg_member == "CpufreqPolicy") {
		const char *policy;
		if (!cpufreq->isSupported())
			return REPLY_HW_NOT_SUPPORTED;

		switch (cpufreq->getMode()) {
		case _PERFORMANCE:
			policy = "performance";
			break;
		case _POWERSAVE:
			policy = "powersave";
			break;
		case _DYNAMIC:
			policy = "dynamic";
			break;
		default:
			return REPLY_HW_NOT_SUPPORTED;
		}
		dbus_message_append_args(reply, DBUS_TYPE_STRING, &policy, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;
	} else if (msg_member == "AcPower") {
		switch (_cur_state.AC_STATE) {
		case AC_ONLINE:
			dummy = "on";
			break;
		case AC_OFFLINE:
			dummy = "off";
			break;
		case AC_UNKNOWN:
			dummy = "unknown";
			break;
		default:
			return REPLY_HW_NOT_SUPPORTED;
		}
		dbus_message_append_args(reply, DBUS_TYPE_STRING, &dummy, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;
	} else if (msg_member == "SchemesGet") {
		int active = -1;
		int battery = -1;
		int ac_power = -1;
		SchemeConfig *scheme;
		const char *scheme_name;

		for (int x = 0; x < MAX_SCHEMES; x++) {
			scheme = config_obj->getSchemeByNumber(x);
			if (scheme == NULL)
				break;
			/* only set \n if there will be another string to append */
			scheme_name = scheme->SCHEME_NAME.c_str();
			dbus_message_append_args(reply, DBUS_TYPE_STRING, &scheme_name, DBUS_TYPE_INVALID);
			if (scheme == config_obj->current_scheme)
				active = x;
			if (scheme == config_obj->Battery_scheme)
				battery = x;
			if (scheme == config_obj->AC_scheme)
				ac_power = x;
		}
		/*
		   pDebug (DBG_DIAG, "Active: %d, battery: %d, ac_power: %d",
		   battery, active, ac_power);
		 */

		/* order of appended INT32 for request "schemes":
		 * 0: no. of currently active   scheme
		 * 1: no. of           battery  scheme
		 * 2: no. of           ac_power scheme
		 * Use the values of these appended INT32 to determine which
		 * appended strings are active/battery/ac_power scheme
		 */
		if (active == -1 || battery == -1 || ac_power == -1) {
			pDebug(DBG_ERR, "Could not find active/battery/ac_power scheme %d/%d/%d",
			       active, battery, ac_power);
			return REPLY_GENERAL_ERROR;
		} else {
			dbus_message_append_args(reply,
						 DBUS_TYPE_INT32, &active,
						 DBUS_TYPE_INT32, &battery,
						 DBUS_TYPE_INT32, &ac_power, DBUS_TYPE_INVALID);
		}

		return REPLY_SUCCESS;
	} else if (msg_member == "SchemesGetDescription") {
		SchemeConfig *scheme;
		const char *description;
		if (!dbusGetMessageString(msg, &dummy, 0)) {
			scheme = config_obj->getScheme(dummy);

			if (scheme == NULL) {
				return REPLY_INVALID_PARAM;
			}

			description = scheme->SCHEME_DESCRIPTION.c_str();

			dbus_message_append_args(reply, DBUS_TYPE_STRING, &description, DBUS_TYPE_INVALID);

			return REPLY_SUCCESS;
		}
		else {
			return REPLY_INVALID_PARAM;
		}
	} else if (msg_member == "BrightnessGet") {
		int brt = _brightness->Get();

		if (brt < 0) {
			return REPLY_HW_NOT_SUPPORTED;
		}

		dbus_message_append_args(reply, DBUS_TYPE_INT32, &brt, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;

	} else if (msg_member == "BrightnessGetPercent") {
		int brt = _brightness->GetPercent();

		if (brt < 0) {
			return REPLY_HW_NOT_SUPPORTED;
		}

		dbus_message_append_args(reply, DBUS_TYPE_INT32, &brt, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;

	} else if (msg_member == "BrightnessLevelsGet") {
		int brt = _brightness->GetLevels();

		if (brt < 0) {
			return REPLY_HW_NOT_SUPPORTED;
		}

		dbus_message_append_args(reply, DBUS_TYPE_INT32, &brt, DBUS_TYPE_INVALID);
		return REPLY_SUCCESS;
	} else if (msg_member == "DpmDevicesGet"){
		if (!config_obj->current_scheme->DPM_ENABLED) {
			pDebug(DBG_DIAG, "Device power management is disabled in config.");
			return REPLY_DISABLED;
		}

		if (!dbusGetMessageString(msg, &dummy, 0)) {
			list< string > udis;
			list< DPM_POWER_STATE > power_states;

			int i = 0;
			while (i != DPM_CLASS_MAX) {
				if (!strcmp(dummy, DPM_CLASS_NAMES[i])) {
					_devices.getDevices((DPM_DEVICE_CLASS)i, udis, power_states);
					break;
				}
				i++;
			}
			
			list< DPM_POWER_STATE >::iterator ti = power_states.begin();
			for (list< string>::iterator it = udis.begin();
			     it != udis.end(); ++it) {

				dummy = (char*)it->c_str();
				dbus_message_append_args(reply, DBUS_TYPE_STRING, &dummy,
							 DBUS_TYPE_INT32, &(*ti),
							 DBUS_TYPE_INVALID);
				
				++ti;
			}

			return REPLY_SUCCESS;
		}

		return REPLY_INVALID_PARAM;
	} else if (msg_member == "DpmClassesGet") {
		if (!config_obj->current_scheme->DPM_ENABLED) {
			pDebug(DBG_DIAG, "Device power management is disabled in config.");
			return REPLY_DISABLED;
		}

		int i = 0;
		while (i != DPM_CLASS_MAX) {
			dbus_message_append_args(reply, DBUS_TYPE_STRING, &(DPM_CLASS_NAMES[i]),
						 DBUS_TYPE_INVALID);
			i++;
		}
		
		return REPLY_SUCCESS;
	} else if (msg_member == "ClientsGet") {
		list< string > names;
		list< int > caps;
		_cC->getClients(names, caps);
		
		for (list< string >::iterator it = names.begin();
		     it != names.end(); ++it) {

 			dbus_message_append_args(reply, DBUS_TYPE_STRING, &(*it),
						 DBUS_TYPE_INVALID);
		}

		for (list< int >::iterator it = caps.begin();
		     it != caps.end(); ++it) {
			dbus_message_append_args(reply, DBUS_TYPE_INT32, &(*it),
						 DBUS_TYPE_INVALID);
		}
		return REPLY_SUCCESS;
	} else if (msg_member == "supp_sleep_states") {

	} else {
		return REPLY_INVALID_METHOD;
	}

	return REPLY_GENERAL_ERROR;
}

DBUS_ERROR PM_Interface::handleActionMessage(DBusMessage * msg)
{

	string msg_member = dbus_message_get_member(msg);
	char *dummy = "";

	if (msg_member == "SuspendToDisk" || msg_member == "suspend_to_disk") {
		string interface = dbus_message_get_interface(msg);
		if (config_obj->current_scheme->USER_SUSPEND2DISK_DISABLED
			&& interface != PS_DBUS_ADMIN_INTERFACE) {
			pDebug(DBG_DIAG, "Suspend to disk is disabled in config.");
			return REPLY_DISABLED;
		} else {
			pDebug(DBG_INFO, "Client has requested suspend2disk");
			_requested_sleep_state = PSD_SUSPEND2DISK;
			/* used for workaround for apm to remembemer triggered sleep state (not needed at
			   this point any more)**** */

			// executing event now triggers suspend2disk, using do_suspend_to_disk variable
			if (haveSuspend2())
				_eM->executeEvent("global.suspend2disk.other");
			else
				_eM->executeEvent("global.suspend2disk");
			return REPLY_SUCCESS;
		}
	} else if (msg_member == "SuspendToRam" || msg_member == "suspend_to_ram") {
		string interface = dbus_message_get_interface(msg);
		if (config_obj->current_scheme->USER_SUSPEND2RAM_DISABLED
			&& interface != PS_DBUS_ADMIN_INTERFACE) {
			pDebug(DBG_DIAG, "Suspend2ram is disabled in config.");
			return REPLY_DISABLED;
		} else if (!(_supported_sleeping_states & APM_SUSPEND) && !(_supported_sleeping_states & ACPI_S3)) {
			pDebug(DBG_DIAG, "Suspend2ram is not supported");
			return REPLY_HW_NOT_SUPPORTED;
		} else {
			pDebug(DBG_INFO, "Client has requested suspend2ram");
			_requested_sleep_state = PSD_SUSPEND2RAM;
			// executing event now triggers suspend2ram, using do_suspend2ram variable!
			_eM->executeEvent("global.suspend2ram");
			//suspend2ram();
			return REPLY_SUCCESS;
		}
	} else if (msg_member == "Standby" || msg_member == "standby") {
		string interface = dbus_message_get_interface(msg);
		if (config_obj->current_scheme->USER_STANDBY_DISABLED
		    && interface != PS_DBUS_ADMIN_INTERFACE) {
			pDebug(DBG_DIAG, "Standby is disabled in config.");
			return REPLY_DISABLED;
		} else if (!(_supported_sleeping_states & APM_STANDBY) && !(_supported_sleeping_states & ACPI_S1)) {
			pDebug(DBG_DIAG, "Standby is not supported");
			return REPLY_HW_NOT_SUPPORTED;
		} else {
			pDebug(DBG_INFO, "Client has requested standby");
			/* workaround for apm to remembemer triggered sleep state ***************
			   but defined for whole PM_Interface to avoid dynamic cast */
			_requested_sleep_state = PSD_STANDBY;
			/* workaround for apm to remembemer triggered sleep state ************** */
			// executing event now triggers suspend2ram, using do_suspend_to_ram variable!
			_eM->executeEvent("global.standby");
			//suspend2ram();
			return REPLY_SUCCESS;
		}
	} else if (msg_member == "CpufreqPerformance") {
		if (cpufreq->isSupported()) {
			if (cpufreq->getMode() != _PERFORMANCE) {
				if (cpufreq->setModes(_PERFORMANCE, _eM) < 0) {
					pDebug(DBG_ERR, "Could not set performance mode");
					return REPLY_HW_NOT_SUPPORTED;
				}
				_throttleInterface.restart();

				return REPLY_SUCCESS;
			} else
				return REPLY_ALREADY_SET;
		} else
			return REPLY_HW_NOT_SUPPORTED;
	} else if (msg_member == "CpufreqPowersave") {
		if (cpufreq->isSupported()) {
			if (cpufreq->getMode() != _POWERSAVE) {
				if (cpufreq->setModes(_POWERSAVE, _eM) < 0) {
					pDebug(DBG_ERR, "Could not set powersave mode");
					return REPLY_HW_NOT_SUPPORTED;
				}
				// this only concerns dynamic events
				_throttleInterface.restart();

				return REPLY_SUCCESS;
			} else
				return REPLY_ALREADY_SET;
		} else
			return REPLY_HW_NOT_SUPPORTED;
	} else if (msg_member == "CpufreqDynamic") {
		if (cpufreq->isSupported()) {
			if (cpufreq->getMode() != _DYNAMIC) {
				if (cpufreq->setModes(_DYNAMIC, _eM) < 0) {
					pDebug(DBG_ERR, "Could not set dynamic mode");
					return REPLY_HW_NOT_SUPPORTED;
				}
				// this only concerns dynamic events
				_throttleInterface.restart();

				return REPLY_SUCCESS;
			} else
				return REPLY_ALREADY_SET;
		} else
			return REPLY_HW_NOT_SUPPORTED;
	} else if (msg_member == "CpufreqCPUEnable") {
		int value;
		if (!dbusGetMessageInteger(msg, &value, 0)) {
			int ret = cpufreq->enableCPU(value); 
			if (ret < 0)
				return REPLY_HW_NOT_SUPPORTED;
			else if (ret == 0)
				return REPLY_ALREADY_SET;
			else if (ret > 0)
				return REPLY_SUCCESS;
		} else {
			return REPLY_GENERAL_ERROR;
		}
	} else if (msg_member == "CpufreqCPUDisable") {
		int value;
		if (!dbusGetMessageInteger(msg, &value, 0)) {
			int ret = cpufreq->disableCPU(value); 
			if (ret < 0)
				return REPLY_HW_NOT_SUPPORTED;
			else if (ret == 0)
				return REPLY_ALREADY_SET;
			else if (ret > 0)
				return REPLY_SUCCESS;
		} else {
			return REPLY_GENERAL_ERROR;
		}
	} else if (msg_member == "Ping") {
		return REPLY_SUCCESS;
	} else if (msg_member == "SchemesSet") {
		if (!dbusGetMessageString(msg, &dummy, 0)) {
			int r = activateScheme(dummy);
			if (r == 0) {
				return REPLY_ALREADY_SET;
			} else if (r < 0) {
				pDebug(DBG_DIAG, "Scheme '%s' does not exist", dummy);
				return REPLY_INVALID_PARAM;
			} else {
				pDebug(DBG_DIAG, "Scheme '%s' received and activated", dummy);

				_eM->executeEvent("daemon.scheme.change");
				return REPLY_SUCCESS;
			}
		} else
			return REPLY_INVALID_PARAM;
	} else if (msg_member == "BrightnessSet") {
		int value;
		if (!dbusGetMessageInteger(msg, &value, 0)) {
			_brightness->Set(value);
			return REPLY_SUCCESS;
		} else {
			return REPLY_GENERAL_ERROR;
		}
	} else if (msg_member == "BrightnessSetPercent") {
		int value;
		if (!dbusGetMessageInteger(msg, &value, 0)) {
			_brightness->SetPercent(value);
			return REPLY_SUCCESS;
		} else {
			return REPLY_GENERAL_ERROR;
		}
	} else if (msg_member == "BrightnessMin") {
		_brightness->Min();
		return REPLY_SUCCESS;
	} else if (msg_member == "BrightnessMed") {
		_brightness->Med();
		return REPLY_SUCCESS;
	} else if (msg_member == "BrightnessMax") {
		_brightness->Max();
		return REPLY_SUCCESS;
	} else if (msg_member == "BrightnessUp") {
		_brightness->Up();
		return REPLY_SUCCESS;
	} else if (msg_member == "BrightnessDown") {
		_brightness->Down();
		return REPLY_SUCCESS;
	} else if (msg_member == "DpmDevicesSuspend") {
		if (!config_obj->current_scheme->DPM_ENABLED) {
			pDebug(DBG_DIAG, "Device power management is disabled in config.");
			return REPLY_DISABLED;
		}
		int device_num = -1;
		if (!dbusGetMessageString(msg, &dummy, 0)
		    && !dbusGetMessageInteger(msg, &device_num, 0)) {

			int i = 0;
			while (i != DPM_CLASS_MAX) {
				if (!strcmp(dummy, DPM_CLASS_NAMES[i])) {
					_devices.suspend((DPM_DEVICE_CLASS)i, device_num);
					return REPLY_SUCCESS;
				}
				i++;
			}
		}

		return REPLY_INVALID_PARAM;
	} else if (msg_member == "DpmDevicesResume") {	
		if (!config_obj->current_scheme->DPM_ENABLED) {
			pDebug(DBG_DIAG, "Device power management is disabled in config.");
			return REPLY_DISABLED;
		}
		int device_num = -1;
		if (!dbusGetMessageString(msg, &dummy, 0)
		    && !dbusGetMessageInteger(msg, &device_num, 0)) {

			int i = 0;
			while (i != DPM_CLASS_MAX) {
				if (!strcmp(dummy, DPM_CLASS_NAMES[i])) {
					_devices.resume((DPM_DEVICE_CLASS)i, device_num);
					return REPLY_SUCCESS;
				}
				i++;
			}
		}
		return REPLY_INVALID_PARAM;
	} else {
		return REPLY_INVALID_METHOD;
	}

	return REPLY_GENERAL_ERROR;
}

BATTERY_STATES PM_Interface::batteryState() {
	return _battery.batteryState();
}
