/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <sstream>
#include <fstream>
#include <cpufreq.h>

#include "cpufreq_interface.h"

CPUFreq_Kernel::CPUFreq_Kernel(list< int > cpu_list, unsigned long rate)
	: CPUFreq_Interface(cpu_list)
{
	_sampling_rate = rate;
	POWERSAVE_STRING = "powersave";
	PERFORMANCE_STRING = "performance";
	ON_DEMAND_STRING = "ondemand";
}

CPUFreq_Kernel::~CPUFreq_Kernel()
{
}

bool CPUFreq_Kernel::init()
{
	if (setGovernor(ON_DEMAND_STRING) < 0) {
		/* no ondemand governor set */
		pDebug(DBG_INFO, "Can not set ondemand governor, maybe your cpufreq driver is too slow.");
		return false;
	}

	return true;
}

bool CPUFreq_Kernel::readFrequencies()
{
	if (!online())
		return false;

	/* this just returns the maximum frequency step to get the value
	 * for downstepping the cpu
	 */
	int freq_hi = 0, freq_lo = 0;
	float ratio = 0, tmp_ratio = 0;

	cpufreq_available_frequencies *freqs = cpufreq_get_available_frequencies(_cpu_base);
	if (freqs == NULL) {
		return false;
	}

	freq_hi = freqs->frequency;
	freqs = freqs->next;

	while (freqs != NULL) {
		freq_lo = freqs->frequency;

		tmp_ratio = ((float)freq_hi / (float)freq_lo);
		pDebug(DBG_INFO, "freq_hi: %d freq_lo: %d, ratio: %4.2f", freq_hi, freq_lo, tmp_ratio);
		freq_hi = freq_lo;
		if (tmp_ratio > ratio)
			ratio = tmp_ratio;

		if (freqs->next == NULL) {
			cpufreq_put_available_frequencies(freqs->first);
			break;
		}
		else {
			freqs = freqs->next;
		}
	}

	_down_threshold = (int)((float)_cpu_max / ratio) - _cpu_hysteresis;
	pDebug(DBG_DIAG, "cpu_max: %d, ratio: %4.3f, hysteresis: %d => down_threshold ((m/r)-h): %d", _cpu_max, ratio,
	       _cpu_hysteresis, _down_threshold);

	return true;
}

int CPUFreq_Kernel::writeOndemand(const string &name, int value)
{
	ostringstream strstr("");
	string ondemand_config_file;

	strstr << SYSFS_FILES << "cpu" << _cpu_base << "/cpufreq/ondemand/" << name;
	ondemand_config_file = strstr.str();
	if (access(ondemand_config_file.c_str(), F_OK)) {
		pDebug(DBG_DIAG, "ondemand config file '%s' does not exist.", ondemand_config_file.c_str());
		return -1;
	}
	if (access(ondemand_config_file.c_str(), W_OK)) {
		pDebug(DBG_WARN, "Could not write ondemand config file %s: %s",
		       ondemand_config_file.c_str(), strerror(errno));
		return 1;
	}
	write_line(ondemand_config_file.c_str(), "%d", value);
	pDebug(DBG_INFO, "wrote '%d' to file '%s'", value, ondemand_config_file.c_str());
	return 0;
}

void CPUFreq_Kernel::setOndemandConfig()
{
	/* probably forgot some ondemand configs ?

	   Current powersaved configs:
	   int CPUFreq_Interface::cpu_max = 90;
	   int CPUFreq_Interface::cpu_hysteresis = 5;
	   int CPUFreq_Interface::high_cpu_limit = 0;
	   int CPUFreq_Interface::consider_nice = 1;

	   Current ondemand config files:
	   down_threshold
	   sampling_down_factor
	   sampling_rate
	   sampling_rate_max
	   sampling_rate_min
	   up_threshold

	   Current dynamic vs. ondemand config mapping:
	   cpu_max       == up_threshold
	   sampling_rate == poll_interval * 1000
	 */

	/* up_threshold is always there, even in "old" kernels (2.6.11)
	   if not, there is something wrong and we return early. */
	if (writeOndemand("up_threshold", _cpu_max)) {
		pDebug(DBG_WARN, "Problems with ondemand governor, could not write up_threshold.");
		return;
	}

	/* ingore_nice sysfs-file should probably be named "consider nice":
	   0 = niced processes do not count for cpufreq calculation
	   1 = niced processes _do_ count for cpufreq calculation
	   Let me tell you that this confused me quite a bit :-)

	   in recent kernels, it was actually fixed. The file is now called
	   "ignore_nice_load" and it does actually do what the name suggests:
	   0 = niced processes do count
	   1 = niced processes do not count
	   Only one of those 2 files is present
	 */
	if ((writeOndemand("ignore_nice_load", !_consider_nice) < 0) &&
	    (writeOndemand("ignore_nice", _consider_nice) < 0)) {
		/* if the kernel is even older, it does not have neither
		   of those, but then it has "down_treshold" (which is auto-
		   calculated in newer kernels), so we at least set that.
		 */
		writeOndemand("down_threshold", _down_threshold);
	}
	writeOndemand("sampling_down_factor", 1);	// todo: use define or configure it.
	writeOndemand("sampling_rate", _sampling_rate);
	return;
}

void CPUFreq_Kernel::setConfig()
{
}

int CPUFreq_Kernel::adjustSpeed()
{
	switch (_mode) {
	case _DYNAMIC:
		if (setGovernor(ON_DEMAND_STRING) < 0) {
			pDebug(DBG_WARN, "Could not set ondemand governor.");
			return -1;
		}
		// after switching back to ondemand, the config needs to be re-set
		setOndemandConfig();
		break;
	case _PERFORMANCE:
		if (setGovernor("performance") < 0) {
			pDebug(DBG_WARN, "Could not set performance governor.");
			return -1;
		}
		break;

	case _POWERSAVE:
		if (setGovernor("powersave") < 0) {
			pDebug(DBG_WARN, "Could not set powersave governor.");
			return -1;
		}
		break;
	default:
		pDebug(DBG_WARN, "Unknown cpufreq kernel governor requested");
		return -1;
	}

	return 1;
}

void CPUFreq_Kernel::reinitSpeed()
{
	/* dummy inplementation */
}
