/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef EVENT_MANAGEMENT_H
#define EVENT_MANAGEMENT_H

#include <list>

class Event;

/** @brief Manages all pending and running events */
class EventManagement {
public:
	/** @brief constructor initializing the class */
	EventManagement();

	virtual ~EventManagement();
	
	/** @brief returns the current program of an event with the id id
	 * @param id event id
	 * @return current program to execute of this event 
	 */
	const std::string cur_prog(int id);

	/** @brief get name of an event 
	 * @param id event id
	 * @return returns the name of the event
	 */
	const std::string name(int id);

	/** @brief moves the iterator current_program_2_execute to the next program of this event
	 * @param id event id
	 * @return interger value
	 * @retval 1 if there are programs left
	 * @retval 0 if all programs returned already
	 * @retval -1 if such an event specified by id could not be found
	 */
	int moveOn(int id);

	/** @brief registers one event event and adjusts some values
	 * @param event an event object
	 * @param timeout timeout of this event in seconds 
	 * @return integer value 
	 * @retval true if successfull
	 * @retval false otherwise
	 */
	int registerEvent(Event & event, time_t timeout);

	/** @brief get number of pending events 
	 * @return number of still running/pending events 
	 */
	int count();

	/** @brief removes one event and frees its memory
	 * @param id event id
	 * @return true on success, false otherwise
	 */
	bool remove(int id);

	/** @brief double linked list of Event-pointers to manage events */
	 std::list < Event > Events;

	/** @brief returns the event which refers to id
	 * @param id event id
	 * @return pointer to event
	 * @retval pointer pointing to an event object
	 * @retval NULL if event was not found
	 */
 	Event* event(int id);

	/** @brief adjusts timeouts after resuming
	 * @param sleep_time time we slept in milliseconds 	 
	 */
	void adjustTimeouts(unsigned long sleep_time);

	/** @brief checks whether an event is still running
	 * @param eventName the name of the event
	 * @return integer 
	 * @retval 1 if running
	 * @retval 0 if not
	 */
	int eventIsRunning(const std::string &eventName);

	/** @brief checks if a returned script belongs to a pending event
	 *  and executes the appropriate actions
	 */
	void checkScriptReturn(int id, int returnValue, std::string &comment);

	/** @brief continues an event by executing the next action
	 *
	 * @param event the event to continue
	 * @param acpi_event_line the acpi event if available, else ""
	 *
	 * @return false if action could not be executed
	 */
	bool continueEvent(Event *event, std::string acpi_event_line);

	/** @brief execute a powersave event 
	 *
	 * @param eventName the name of the powersave event
	 * @param acpi_event_line acpi event if present
	 *
	 * @return true if event got executed, false if event does not
	 *         exist
	 */
	bool executeEvent(const std::string &eventName,
			  const char *acpi_event_line = "");

	/** @brief checks all pending events whether there timeouts are
	 * reached */
	void checkEventTimeouts();

private:
	/** @brief execute an internal action
	 *
	 * @return integer with result of execute requested internal action
	 * @retval 1 if the action could be found
	 * @retval -1 if this was not an internal action
	 */
	bool executeInternalAction(Event *event, std::string action);

	/** @brief every event will get a unique id */
	unsigned int id_count;
};

#endif // EVENT_MANAGEMENT_H
