/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Timo Hoenig <thoenig@suse.de>                  *
 *                          Danny Kukawka <danny.kukawka@web.de>           *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/utsname.h>
#include <sstream>

#include <sys/ioctl.h>
#include <linux/pmu.h>

#include "powerlib.h"
/* merge #include "config_pm.h" */

#include "brightness.h"

/* Probe function to identify kernel interface */

Brightness *Brightness::Probe()
{
	int fd = 0;

	if ((fd = open(ACPI_ASUS, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessASUS();
	}

	if ((fd = open(ACPI_IBM, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessIBM();
	}

	if ((fd = open(ACPI_SONY, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessSony();
	}

	if ((fd = open(ACPI_TOSHIBA, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessToshiba();
	}

	if ((fd = open(ACPI_PANASONIC, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessPanasonic();
	}

	if ((fd = open(LCD_OMNIBOOK, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessOmnibook();
	}

	if ((fd = open(DEV_PMU, O_RDONLY)) > 0) {
		close(fd);
		return new BrightnessPMU();
	}
	

	if (fd) {
		close(fd);
	}

	return new Brightness();
}

Brightness::~Brightness()
{
}

/* Brightness (dummy object) */

void Brightness::Init()
{
	last_percent = -1;
	return;
}

int Brightness::Get()
{
	return -1;
}

void Brightness::Set(int level)
{
	return;
}

void Brightness::Min()
{
	return;
}

void Brightness::Med()
{
	return;
}

void Brightness::Max()
{
	Set(GetLevels() - 1);
	return;
}

void Brightness::Up()
{
	Set(Get() + 1);
	return;
}

void Brightness::Down()
{
	Set(Get() - 1);
	return;
}

int Brightness::GetLevels()
{
	return -1;
}

// a helper function: give percent, get level
int Brightness::PercentToLevel(int percent)
{
	int b, l, comp;
	l = GetLevels();
	comp = 50 / l;	// (100/l)/2, compensation
	if (percent < 0)
		percent = 0;
	if (percent > 100)
		percent = 100;

	b = l * (percent + comp) / 100;

	return b;
}

void Brightness::SetPercent(int percent)
{
	Set(PercentToLevel(percent));
	last_percent = percent;
}

int Brightness::GetPercent()
{
	int b, l;
	b = Get();
	l = GetLevels() - 1;
	
	if ((b < 0) || (l <= 0))
		return -1;

	if ((last_percent >= 0) && (PercentToLevel(last_percent) == b))
		return last_percent;

	last_percent = ((b * 1000 + 5) / l) / 10;
	return last_percent;
}

/* Brightness (ASUS object) */

void BrightnessASUS::Init()
{
	last_percent = -1;
	iface = ACPI_ASUS;
	return;
}

int BrightnessASUS::Get()
{
	char value[3];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 2) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "%2d", &level);

out:
	close(fd);
	return level;
}

void BrightnessASUS::Set(int level)
{
	char value[3];

	if ((fd = open(iface, O_RDWR)) < 0) {
		perror(iface);
		goto out;
	}

	if (level > 15)
		level = 15;
	else if (level < 0)
		level = 0;

	sprintf(value, "%2d", level);
	if (write(fd, value, 2) < 0) {
		pDebug(DBG_WARN,
		       "Unable to write to ASUS brightness interface");
	}
 
out:
	close(fd);
	return;
}

void BrightnessASUS::Min()
{
	Set(0);
	return;
}

void BrightnessASUS::Med()
{
	Set(7);
	return;
}

int BrightnessASUS::GetLevels()
{
	return 16;
}

/* Brightness (IBM object) */

void BrightnessIBM::Init()
{
	last_percent = -1;
	iface = ACPI_IBM;
	return;
}

int BrightnessIBM::Get()
{
	char value[18];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 17) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "level:%1d", &level);

out:
	close(fd);
	return level;
}

void BrightnessIBM::Set(int level)
{
	char value[8];

	if ((fd = open(iface, O_RDWR)) < 0) {
		perror(iface);
		goto out;
	}

	if (level > 7)
		level = 7;
	else if (level < 0)
		level = 0;
	sprintf(value, "level %d", level);
	if (write(fd, value, 7) < 0) {
		pDebug(DBG_WARN,
		       "Unable to write to IBM brightness interface");
	}

out:
	close(fd);
	return;
}

void BrightnessIBM::Min()
{
	Set(0);
	return;
}

void BrightnessIBM::Med()
{
	Set(4);
	return;
}

int BrightnessIBM::GetLevels()
{
	return 8;
}

/* Brightness (Sony object) */

void BrightnessSony::Init()
{
	last_percent = -1;
	iface = ACPI_SONY;
	return;
}

/* sony_acpi has brightness levels 1 - 8, all others start with Zero.
   So we normalize sony_acpi to 0 - 7 in Get() and Set() */
int BrightnessSony::Get()
{
	char value[2];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 1) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "%1d", &level);
	level--;

out:
	close(fd);
	return level;
}

void BrightnessSony::Set(int level)
{
	char value[2];

	if ((fd = open(iface, O_RDWR)) < 0) {
		perror(iface);
		goto out;
	}

	if (level > 7)
		level = 7;
	else if (level < 0)
		level = 0;
	sprintf(value, "%1d", level + 1);
	if (write(fd, value, 1) < 0) {
		pDebug(DBG_WARN,
		       "Unable to write to SONY brightness interface");
	}

out:
	close(fd);
	return;
}

void BrightnessSony::Min()
{
	Set(0);
	return;
}

void BrightnessSony::Med()
{
	Set(3);
	return;
}

int BrightnessSony::GetLevels()
{
	return 8;
}

/* Brightness (Toshiba object) */

void BrightnessToshiba::Init()
{
	last_percent = -1;
	iface = ACPI_TOSHIBA;
	return;
}

int BrightnessToshiba::Get()
{
	char value[27];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 26) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "brightness:%1d", &level);

out:
	close(fd);
	return level;
}

void BrightnessToshiba::Set(int level)
{
	char value[27];

	if ((fd = open(iface, O_RDWR)) < 1) {
		perror(iface);
		goto out;
	}

	if (level > 7)
		level = 7;
	else if (level < 0)
		level = 0;
	sprintf(value, "brightness:%1d", level);
	if (write(fd, value, 26) < 0) {
		pDebug(DBG_WARN,
		       "Unable to write to Toshiba brightness interface");
	}

out:
	close(fd);
	return;
}

void BrightnessToshiba::Min()
{
	Set(0);
	return;
}

void BrightnessToshiba::Med()
{
	Set(4);
	return;
}

int BrightnessToshiba::GetLevels()
{
	return 8;
}

/* Brightness (Panasonic object) */
void BrightnessPanasonic::Init()
{
	char value[4];
	last_percent = -1;
	iface = ACPI_PANASONIC "ac_brightness";

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		goto error;
	}

	if (read(fd, &value, 3) == -1) {
		goto error;
	}
	close(fd);
	fd = 0;

#ifndef PANASONIC_BRIGHTNESS_INTERFACE_IS_DOG_SLOW
	iface_dc = ACPI_PANASONIC "dc_brightness";
	if ((fd = open(iface_dc, O_RDWR)) < 0) {
		perror(iface_dc);
		goto error;
	}

	if (write(fd, &value, 3) == -1) {
		goto error;
	}
	close(fd);
	fd = 0;
#endif

	if ((fd = open(ACPI_PANASONIC "ac_brightness_max", O_RDONLY)) < 0) {
		goto error;
	}
	if (read(fd, &value, 3) == -1) {
		goto error;
	}
	close(fd);
	fd = 0;
	brt_max = atoi(value);

	if ((fd = open(ACPI_PANASONIC "ac_brightness_min", O_RDONLY)) < 0) {
		goto error;
	}
	if (read(fd, &value, 3) == -1) {
		goto error;
	}
	close(fd);
	fd = 0;
	brt_min = atoi(value);
	/* brt_scale is the scale factor for the levels, multiplied by 10.
	   so the division by two gives us approx. 20 steps.
	   This avoids using float arithmetics.				*/
	brt_scale = (brt_max - brt_min) / 2;

	goto out;

error:
	brt_min = -1;
	brt_max = -1;
	brt_scale = -1;
out:
	close(fd);
	return;
}

/* pcc_acpi has brightness levels 51 - 255 (on a CF-51, see brt_min/brt_max)
   So we compensate this by adding/substracting brt_min in Get() and Set() */
int BrightnessPanasonic::Get()
{
	char value[4];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 3) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "%3d", &level);
	level = ((level - brt_min) * 10 + 5) / brt_scale;

out:
	close(fd);
	return level;
}

void BrightnessPanasonic::Set(int level)
{
	char value[4];
	int x;

	if ((fd = open(iface, O_RDWR)) < 0) {
		perror(iface);
		goto out;
	}

	x = (level * brt_scale + 5) / 10 + brt_min;
	if (x > brt_max)
		x = brt_max;
	else if (x < brt_min)
		x = brt_min;
	sprintf(value, "%3d", x);
	if (write(fd, value, 3) < 0)
		pDebug(DBG_WARN,
		       "Unable to write to Panasonic ac_brightness interface");

#ifndef PANASONIC_BRIGHTNESS_INTERFACE_IS_DOG_SLOW
	close(fd);
	fd = 0;

	if ((fd = open(iface_dc, O_RDWR)) < 0) {
		perror(iface_dc);
		goto out;
	}
	if (write(fd, value, 3) < 0)
		pDebug(DBG_WARN,
		       "Unable to write to Panasonic dc_brightness interface");
#endif

out:
	close(fd);
	return;
}

void BrightnessPanasonic::Min()
{
	Set(0);
	return;
}

void BrightnessPanasonic::Med()
{
	/* the driver (or BIOS?) default for dc_brightness seems to be 127,
	   with GetLevels()/2 it is set to 153, too high imo. TODO: fix. */
	Set(GetLevels() / 2);
	return;
}

int BrightnessPanasonic::GetLevels()
{
	return (brt_max - brt_min) * 10 / brt_scale + 1;
}

/* Brightness (Omnibook object) */

void BrightnessOmnibook::Init()
{
	last_percent = -1;
	iface = LCD_OMNIBOOK;
	return;
}

int BrightnessOmnibook::Get()
{
	char value[19];
	int level;

	if ((fd = open(iface, O_RDONLY)) < 0) {
		perror(iface);
		level = -1;
		goto out;
	}

	if (read(fd, &value, 18) == -1) {
		level = -1;
		goto out;
	}

	sscanf(value, "LCD brightness: %d", &level);

out:
	close(fd);
	return level;
}

void BrightnessOmnibook::Set(int level)
{
	char value[3];
	int n;

	if ((fd = open(iface, O_RDWR)) < 1) {
		perror(iface);
		goto out;
	}

	if (level > 10)
		level = 10;
	else if (level < 0)
		level = 0;
	n = sprintf(value, "%d", level);
	if (write(fd, value, n) < 0) {
		pDebug(DBG_WARN,
		       "Unable to write to Omnibook brightness interface");
	}

out:
	close(fd);
	return;
}

void BrightnessOmnibook::Min()
{
	Set(0);
	return;
}

void BrightnessOmnibook::Med()
{
	Set(5);
	return;
}

int BrightnessOmnibook::GetLevels()
{
	return 11;
}

/* Brightness (PMU object (for ppc/mac/iBook)) */

void BrightnessPMU::Init()
{
	last_percent = -1;
	iface = DEV_PMU;
	return;
}

int BrightnessPMU::Get()
{
	int level = -1;

	if ((fd = open(iface, O_RDWR)) < 0) {
		perror(iface);
		goto out;
	}

	if (ioctl (fd, PMU_IOC_GET_BACKLIGHT, &level) < 0 ) {
		pDebug(DBG_WARN, "Failed ioctl on /dev/pmu with PMU_IOC_GET_BACKLIGHT");
		level = -1;
	}

out:
	close(fd);
	return level;
}

void BrightnessPMU::Set(int level)
{
	if ((fd = open(iface, O_RDWR)) < 1) {
		perror(iface);
		goto out;
	}

	if (level > 15)
		level = 15;
	else if (level < 0)
		level = 0;
	
	if(ioctl (fd, PMU_IOC_SET_BACKLIGHT, &level) < 0) {
		pDebug(DBG_WARN, "Failed ioctl on /dev/pmu with PMU_IOC_GET_BACKLIGHT");
	}

out:
	close(fd);
	return;
}

void BrightnessPMU::Min()
{
	Set(1); // looks as 0 is display off, use as last 1
	return;
}

void BrightnessPMU::Med()
{
	Set(8);
	return;
}

int BrightnessPMU::GetLevels()
{
	return 16;
}

