/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>

#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <syslog.h>

#include <glib-object.h>
#include <gio/gio.h>
#include "powerd-internal.h"
#include "powerd-object.h"
#include "powerd-dbus.h"
#include "powerd-sensors.h"
#include "device-config.h"
#include "log.h"

#include <hybris/input/input_stack_compatibility_layer.h>
#include <hybris/input/input_stack_compatibility_layer_codes_key.h>

#include <hybris/surface_flinger/surface_flinger_compatibility_layer.h>

#include "libsuspend.h"

#include <android/hardware/power.h>

static GThread *powerd_mainloop_thread;

namespace
{

#define SHUTDOWN_HOLD_SECS 2

static uint button_timer = 0;

/*
 * Do not modify activity_timer directly. Instead, call
 * reset_activity_timer();
 */
static uint activity_timer = 0;
GMutex activity_timer_mutex;

static time_t curtime;
static int lasttime;
struct tm *tm;

/* The real default for this is set in the gschema file, but set 60 here
 * as a sanity check */
static const int activity_timeout = 60;
static const int dim_timeout = 45;

static GMainLoop *main_loop = NULL;
static guint name_id;

enum state {
    BUTTON_DOWN,
    BUTTON_UP,
    SHUTDOWN
};

static state button_state  = BUTTON_UP;

enum {
    SCREEN_STATE_OFF = 0,
    SCREEN_STATE_DIM,
    SCREEN_STATE_BRIGHT
};

/* Assume screen starts off when powerd starts */
int activity_timer_screen_state = SCREEN_STATE_OFF;

static int g_exit_code = 0;

/*
 * Display request for activity timer
 *
 * Stupid C++ compiler won't let me use designated initializers ...
 */
static struct powerd_display_request activity_timer_req = {
    {0,},                                   /* cookie */
    POWERD_DISPLAY_STATE_ON,                /* state */
    POWERD_DISPLAY_FLAG_BRIGHT       /* flags */
};

/* Display request for use of proximity sensor during phone calls */
static struct powerd_display_request prox_sensor_req = {
    {0,},                                   /* cookie */
    POWERD_DISPLAY_STATE_DONT_CARE,         /* state */
    POWERD_DISPLAY_FLAG_USE_PROXIMITY       /* flags */
};


gboolean activity_monitor(gpointer data);
void update_screen_state(int state);

gboolean call_shutdown(gpointer data)
{
    if (button_state == BUTTON_DOWN) {
        button_state = SHUTDOWN;
        powerd_set_brightness(0);
        display_set_power_mode(0, "off");
        powerd_shutdown();
    }

    return FALSE;
}

gboolean activity_monitor(gpointer data)
{
    int new_state;

    g_mutex_lock(&activity_timer_mutex);

    if (activity_timer_screen_state > SCREEN_STATE_OFF) {
        if (dim_timeout > 0 && activity_timeout > dim_timeout)
            new_state = activity_timer_screen_state - 1;
        else
            new_state = SCREEN_STATE_OFF;

        update_screen_state(new_state);
        if (new_state != SCREEN_STATE_OFF && activity_timer > 0)
            activity_timer = g_timeout_add_seconds(activity_timeout - dim_timeout,
                                                   activity_monitor, NULL);
    }

    g_mutex_unlock(&activity_timer_mutex);
    return FALSE;
}

static struct power_module* _power_module;

void on_new_event(Event* event, void* context)
{
    switch(event->type)
    {
        case KEY_EVENT_TYPE:
            if (event->details.key.key_code == ISCL_KEYCODE_POWER && button_state != SHUTDOWN) {
                if (event->action == 1) {
                    powerd_debug("power button pressed");
                    button_state = BUTTON_UP;
                    curtime = time(0);
                    tm = localtime (&curtime);
                    if (tm->tm_sec - lasttime < SHUTDOWN_HOLD_SECS) {
                        gboolean enabled = powerd_display_enabled();
                        if (enabled)
                            powerd_display_set_override(POWERD_OVERRIDE_REASON_POWER_BUTTON);
                        else
                            powerd_display_clear_override(POWERD_OVERRIDE_REASON_POWER_BUTTON);
                        powerd_reset_activity_timer(!enabled);
                    }
                } else if (event->action == 0) {
                    button_state = BUTTON_DOWN;
                    curtime = time(0);
                    tm = localtime (&curtime);
                    lasttime = tm->tm_sec;
                    if (button_timer > 0)
                    {
                        g_source_remove(button_timer);
                        button_timer = 0;
                    }
                    button_timer = g_timeout_add_seconds(SHUTDOWN_HOLD_SECS,
                                                         call_shutdown, NULL);
                }
            }
            break;
        default:
            if (powerd_display_enabled())
                powerd_reset_activity_timer(1);
            break;
    }
}

static void
sigterm_quit(int signal)
{
    powerd_warn("SIGTERM recieved, cleaning up");
    powerd_exit(0);
}

int update_screen_state_worker(gpointer data)
{
    long new_state = (long)data;
    int ret;

    if (new_state == activity_timer_screen_state)
        return 0;

    if (new_state == SCREEN_STATE_OFF) {
        powerd_remove_display_request(activity_timer_req.cookie);
    } else {
        if (new_state == SCREEN_STATE_DIM)
            activity_timer_req.flags &= ~POWERD_DISPLAY_FLAG_BRIGHT;
        else
            activity_timer_req.flags |= POWERD_DISPLAY_FLAG_BRIGHT;

        if (activity_timer_screen_state == SCREEN_STATE_OFF)
            ret = powerd_add_display_request(&activity_timer_req,
                                             "activity-timer");
        else
            ret = powerd_update_display_request(&activity_timer_req);

        if (ret)
            powerd_warn("Error adding display state request for activity timer");
    }

    activity_timer_screen_state = new_state;
    return 0;
}

void update_screen_state(int new_state)
{
    long data = new_state;
    if (new_state < SCREEN_STATE_OFF || new_state > SCREEN_STATE_BRIGHT)
        return;
    powerd_run_mainloop_sync(update_screen_state_worker, (gpointer)data);
}

}   //namespace

void powerd_hal_signal_activity(void)
{
    powerd_warn("signalling activity via HAL");
    if (_power_module)
        _power_module->powerHint(_power_module, POWER_HINT_INTERACTION, NULL);
}

void powerd_reset_activity_timer(int add)
{
    int timeout;

    g_mutex_lock(&activity_timer_mutex);

    if (activity_timer > 0) {
        g_source_remove(activity_timer);
        activity_timer = 0;
    }

    if (add) {
        if (dim_timeout > 0 && dim_timeout < activity_timeout)
            timeout = dim_timeout;
        else
            timeout = activity_timeout;

        update_screen_state(SCREEN_STATE_BRIGHT);
        activity_timer = g_timeout_add_seconds(timeout, activity_monitor, NULL);
    } else {
        update_screen_state(SCREEN_STATE_OFF);
    }

    g_mutex_unlock(&activity_timer_mutex);
}

/*
 * Once dbus is ready, force the screen on to make the system state
 * match the powerd iternal state
 */
void powerd_dbus_init_complete(void)
{
    powerd_reset_activity_timer(1);
}

void
on_ofono_voicecallmanager_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    powerd_debug("we get signal from %s: %s", sender_name, signal_name);
    /* from org.ofono.VoiceCallManager - a call was added */
    if (!strcmp(signal_name, "CallAdded")) {
        powerd_debug("Waking up the device - Incoming Call");
        powerd_display_clear_override(POWERD_OVERRIDE_REASON_POWER_BUTTON);
        powerd_reset_activity_timer(1);
    }
}

void
on_ofono_messagemanager_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    powerd_debug("we get signal from %s: %s", sender_name, signal_name);
    /* from org.ofono.MessageManager */
    if ((!strcmp(signal_name, "IncomingMessage")) ||
        (!strcmp(signal_name, "ImmediateMessage"))) {
        powerd_debug("Waking up the device - Incoming SMS");
        powerd_display_clear_override(POWERD_OVERRIDE_REASON_POWER_BUTTON);
        powerd_reset_activity_timer(1);
    }
}

void
on_ofono_voicecall_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    GVariant *value = NULL;
    const char *prop_name;
    const char *prop_value;
    static gboolean alerted = FALSE;

    powerd_debug("we get signal from %s: %s", sender_name, signal_name);

    /* This function enables and disables the proximity sensor based
     * on the state of the call.
     *
     * Active - meaning, the call is in progress --> Proximity On
     * Alerting - a signal made on outbound calls only which means that
     *            the phone is ringing on the other side, but has not
     *            been picked up. --> Proximity On (this signal is not
     *            sent on inbound calls)
     * Disconnected - the call is over. --> Proximity Off.
     *
     * For more details on states see voicecall-api.txt in the ofono source
     */

    /* Note: As of June 13, 2013, the phone cannot successfully
     * accept a call while another call is in progress. When this is fixed
     * the code below will need to be rewritten to handle multiple
     * active/disconnected signals. LP: #1191033 (mfisch) */

    if (!strcmp(signal_name, "PropertyChanged")) {
        g_variant_get(parameters, "(&sv)", &prop_name, &value);
        if (!strcmp(prop_name,"State")) {
            g_variant_get(value, "&s", &prop_value);
            powerd_debug("Call State = %s", prop_value);
            if (!strcmp(prop_value,"active")) {
                if (!alerted) {
                    //call is active, enable sensor
                    if (powerd_add_display_request(&prox_sensor_req, "prox-sensor"))
                        powerd_warn("Request to use proximity sensor failed");
                }
            }
            if (!strcmp(prop_value,"alerting")) {
                alerted = TRUE;
                //call is altering, enable sensor
                if (powerd_add_display_request(&prox_sensor_req, "prox-sensor"))
                    powerd_warn("Request to use proximity sensor failed");
            }
            else if (!strcmp(prop_value,"disconnected")) {
                alerted = FALSE;
                //call is disconnected, disable sensor
                powerd_remove_display_request(prox_sensor_req.cookie);
            }
            g_variant_unref(value);
        }
    }
}

/* Must be first to run on main loop */
static gboolean main_init(gpointer unused)
{
    powerd_mainloop_thread = g_thread_self();
    return FALSE;
}

/* Returns TRUE if current thread of execution is the default main loop */
int powerd_is_mainloop(void)
{
    return (g_thread_self() == powerd_mainloop_thread);
}

void powerd_shutdown(void)
{
    static char poweroff_cmd[] = "poweroff";
    char *argv[] = {poweroff_cmd, NULL};
    g_spawn_async(NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, NULL, NULL);
}

void powerd_exit(int exit_code)
{
    g_exit_code = exit_code;
    g_main_loop_quit(main_loop);
}

int main(int argc, char** argv)
{
    powerd_log_init();

    name_id = g_bus_own_name(G_BUS_TYPE_SYSTEM, "com.canonical.powerd",
        G_BUS_NAME_OWNER_FLAGS_REPLACE, powerd_bus_acquired_cb,
        powerd_name_acquired_cb, powerd_name_lost_cb, NULL, NULL);
    powerd_debug("owner id: %u", name_id);

    /* for incoming SMS signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        "/ril_0",
        "org.ofono.MessageManager",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        NULL);

    /* for incoming calls Added/Removed signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        "/ril_0",
        "org.ofono.VoiceCallManager",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        NULL);

    /* for answering a call/hanging up signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        "/ril_0/voicecall01",
        "org.ofono.VoiceCall",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        NULL);

    /* Init this first, data is used by other inits */
    device_config_init();

    libsuspend_init(0);
    powerd_stats_init();
    powerd_client_init();
    power_request_init();
    display_request_init();
    dbus_name_watch_init();
    powerd_backlight_init();
    powerd_autobrightness_init();
    powerd_display_init();
    powerd_sensors_init();
    powerd_ps_init();

    main_loop = g_main_loop_new (NULL, FALSE);
    signal(SIGTERM, sigterm_quit);

    AndroidEventListener listener;
    listener.on_new_event = on_new_event;
    listener.context = NULL;

    InputStackConfiguration config = {
    enable_touch_point_visualization : false,
    default_layer_for_touch_point_visualization : 10000,
    input_area_width : 2048,
    input_area_height : 2048
    };

    int err = hw_get_module(POWER_HARDWARE_MODULE_ID,
        (hw_module_t const**)&_power_module);

    if (!err)
        _power_module->init(_power_module);

    android_input_stack_initialize(&listener, &config);
    android_input_stack_start();

    /* Config use should be done during init, okay to free now */
    device_config_deinit();

    /* This needs to be the first thing to run on the main loop */
    g_idle_add_full(G_PRIORITY_HIGH, main_init, NULL, NULL);

    g_main_loop_run(main_loop);
    g_bus_unown_name(name_id);
    g_main_loop_unref(main_loop);

    android_input_stack_stop();
    android_input_stack_shutdown();

    powerd_ps_deinit();
    dbus_name_watch_deinit();
    powerd_autobrightness_deinit();
    powerd_backlight_deinit();
    display_request_deinit();
    power_request_deinit();
    powerd_client_deinit();
    powerd_stats_deinit();
    return g_exit_code;
}
