/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <stdlib.h>
#include <avcodec.h>
#include <avformat.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-avcodec.h"

typedef struct {
	AVFormatContext *format;
	int stream;
	AVCodecContext *codec_ctx;
	int64_t pos;
	int seekable;
} iavc;

static int input_avcodec_open(input *p, const char *fn) {
	iavc *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	p->data = a;

	for (int pass = 0; pass < 2; pass++) {
		a->format = NULL;
		a->stream = -1;
		a->codec_ctx = NULL;
		a->pos = 0;
		a->seekable = 0;

		if (av_open_input_file(&a->format, fn, NULL, 0, NULL) != 0)
			return -1;

		if (av_find_stream_info(a->format) < 0)
			return -1;

		for (int i = 0; i < a->format->nb_streams; i++) {
			AVCodecContext *ctx = a->format->streams[i]->codec;
			if (ctx->codec_type == CODEC_TYPE_AUDIO) {
				a->stream = i;
				a->codec_ctx = ctx;
			}
		}
		if (a->codec_ctx == NULL)
			return -1;

		AVCodec *codec = avcodec_find_decoder(a->codec_ctx->codec_id);
		if (codec == NULL)
			return -1;

		if (avcodec_open(a->codec_ctx, codec) < 0)
			return -1;

		if (pass == 1) {
			// We've tried once already -- must not be seekable.
			break;
		} else if (av_seek_frame(a->format, a->stream, 0, 0) == 0) {
			// Seeking worked.
			a->seekable = 1;
			break;
		} else {
			// Seeking didn't work -- reopen the file.
			avcodec_close(a->codec_ctx);
			av_close_input_file(a->format);
		}
	}

	return 0;
}

static int input_avcodec_get_audio(input *p, buffer *buf) {
	iavc *a = (iavc *) p->data;

	// Keep reading until we get some audio.
	int out_size = 0, in_total = 0;
	do {
		AVPacket packet;

		// Read until we get an audio frame.
		while (1) {
			if (av_read_frame(a->format, &packet) < 0) {
				// End of file.
				return 0;
			}

			if (packet.stream_index == a->stream)
				break;

			av_free_packet(&packet);
		}

		// Decode the frame.
		uint8_t *in_data = packet.data;
		int in_size = packet.size;
		while (in_size > 0) {
			int chunk = AVCODEC_MAX_AUDIO_FRAME_SIZE;
			unsigned char *out = buffer_reserve(buf, chunk);
			if (out == NULL)
				g_error("out of memory");

			int in_used = avcodec_decode_audio2(a->codec_ctx,
							    (int16_t *)out,
							    &chunk,
							    in_data, in_size);
			if (in_used < 0) {
				// Decoding error -- skip this frame.
				break;
			}

			in_data += in_used;
			in_size -= in_used;
			in_total += in_used;
			buf->used += chunk;
			out_size += chunk;
		}

		a->pos = packet.pts;

		av_free_packet(&packet);
	} while (out_size == 0);

	p->fmt.bits = 16;
	p->fmt.rate = a->codec_ctx->sample_rate;
	p->fmt.channels = a->codec_ctx->channels;
	p->fmt.byte_format = END_NATIVE;

	// avformat does provide the total bitrate, but we only want the stream
	// we're decoding.
	double num_samples = (1.0L * out_size)
	                     / ((p->fmt.bits / 8) * p->fmt.channels);
	p->bitrate = 8.0L * in_total
	             / ((num_samples / p->fmt.rate) * 1000.0L);

	return out_size;
}

static double convert_time(iavc *a, int64_t t) {
	const AVRational *tb = &a->format->streams[a->stream]->time_base;
	return (((double) tb->num) * t) / tb->den;
}

static int input_avcodec_get_pos(input *p, double *pos) {
	iavc *a = (iavc *) p->data;

	int64_t pts = a->pos;
	int64_t start_time = a->format->streams[a->stream]->start_time;
	if (start_time != AV_NOPTS_VALUE)
		pts -= start_time;

	*pos = convert_time(a, pts);
	return 0;
}

static int input_avcodec_get_len(input *p, double *len) {
	iavc *a = (iavc *) p->data;

	int64_t duration = a->format->streams[a->stream]->duration;
	if (duration == AV_NOPTS_VALUE) {
		return -1;
	} else {
		*len = convert_time(a, duration);
		return 0;
	}
}

static int input_avcodec_get_seekable(input *p) {
	iavc *a = (iavc *) p->data;

	return a->seekable;
}

static int input_avcodec_set_pos(input *p, double pos) {
	iavc *a = (iavc *) p->data;

	const AVRational *tb = &a->format->streams[a->stream]->time_base;
	int64_t timestamp = (tb->den * pos) / tb->num;
	timestamp += a->format->streams[a->stream]->start_time;

	int flags = (timestamp < a->pos) ? AVSEEK_FLAG_BACKWARD : 0;
	flags |= AVSEEK_FLAG_ANY;
	if (av_seek_frame(a->format, a->stream, timestamp, flags) < 0)
		return -1;

	return 0;
}

static int input_avcodec_close(input *p) {
	iavc *a = (iavc *) p->data;

	if (a->codec_ctx != NULL)
		avcodec_close(a->codec_ctx);
	if (a->format != NULL)
		av_close_input_file(a->format);

	free(a);
	free(p);

	return 0;
}

input *input_new_avcodec(void) {
	static gboolean initialised = FALSE;
	if (!initialised) {
		av_register_all();
		initialised = TRUE;
	}

	input *p = input_alloc();

	p->open = input_avcodec_open;
	p->get_audio = input_avcodec_get_audio;
	p->get_pos = input_avcodec_get_pos;
	p->get_len = input_avcodec_get_len;
	p->get_seekable = input_avcodec_get_seekable;
	p->set_pos = input_avcodec_set_pos;
	p->close = input_avcodec_close;

	return p;
}
