/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <ao/ao.h>
#include <vorbis/vorbisfile.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-vorbis.h"

typedef struct {
	OggVorbis_File *vf;
	int segment, segment_set;
} aovorbis;

static int vorbis_open(input *p, const char *fn) {
	aovorbis *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	p->data = a;

	a->segment_set = 0;

	FILE *f = fopen(fn, "r");
	if (f == NULL)
		return -1;

	a->vf = malloc(sizeof *a->vf);
	if (a->vf == NULL)
		g_error("out of memory");

	if (ov_open(f, a->vf, NULL, 0) < 0) {
		fclose(f);
		return -1;
	}
	f = NULL; // vorbisfile now owns it

	return 0;
}

static int vorbis_get_audio(input *p, buffer *buf) {
	aovorbis *a = (aovorbis *) p->data;

	const int chunk_size = 65536;
	unsigned char *out = buffer_reserve(buf, chunk_size);
	if (out == NULL)
		g_error("out of memory");

	int bitstream;
	long n = ov_read(a->vf, (char *) out, chunk_size, 0, 2, 1, &bitstream);
	if (n < 0)
		return -1;
	buf->used += n;
	if (n == 0)
		return 0;

	p->bitrate = ov_bitrate_instant(a->vf) / 1000.0;

	if (!a->segment_set || a->segment != bitstream) {
		vorbis_info *info = ov_info(a->vf, -1);
		if (info == NULL)
			return -1;

		p->fmt.bits = 16;
		p->fmt.rate = info->rate;
		p->fmt.channels = info->channels;
		p->fmt.byte_format = END_LITTLE;
		a->segment = bitstream;
		a->segment_set = 1;
	}

	return n;
}

static int vorbis_get_pos(input *p, double *pos) {
	aovorbis *a = (aovorbis *) p->data;

	*pos = ov_time_tell(a->vf);

	return 0;
}

static int vorbis_get_len(input *p, double *len) {
	aovorbis *a = (aovorbis *) p->data;

	// FIXME does this do the right thing for multiple bitstreams?
	*len = ov_time_total(a->vf, -1);

	return 0;
}

static int vorbis_get_seekable(input *p) {
	aovorbis *a = (aovorbis *) p->data;

	return ov_seekable(a->vf) != 0;
}

static int vorbis_set_pos(input *p, double pos) {
	aovorbis *a = (aovorbis *) p->data;

	if (ov_time_seek(a->vf, pos) != 0)
		return -1;

	return 0;
}

static int vorbis_close(input *p) {
	aovorbis *a = (aovorbis *) p->data;

	if (ov_clear(a->vf) != 0)
		return -1;

	free(a->vf);
	free(a);
	free(p);

	return 0;
}

input *input_new_vorbis(void) {
	input *p = input_alloc();

	p->open = vorbis_open;
	p->get_audio = vorbis_get_audio;
	p->get_pos = vorbis_get_pos;
	p->get_len = vorbis_get_len;
	p->get_seekable = vorbis_get_seekable;
	p->set_pos = vorbis_set_pos;
	p->close = vorbis_close;

	return p;
}
