/*
 Copyright (C) 2010 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Reader {
    static string module = null;
    static bool verbose = false;
    static bool version = false;
    static string[] filenames = null;
    const OptionEntry[] options = {
        { "module", 'm', 0, OptionArg.STRING, ref module, N_("Module"), null },
        { "verbose", 'v', 0, OptionArg.NONE, ref verbose, N_("Verbose"), null },
        { "version", 'V', 0, OptionArg.NONE, ref version, N_("Display program version"), null },
        { "", 0, 0, GLib.OptionArg.STRING_ARRAY, ref filenames, N_("Filenames"), null },
        { null }
    };

    static int main (string[] args) {
        Postler.App.argv0 = args[0];

#if HAVE_NLS
        Intl.bindtextdomain (Config.GETTEXT_PACKAGE, Config.LOCALEDIR);
        Intl.bind_textdomain_codeset (Config.GETTEXT_PACKAGE, "UTF-8");
        Intl.textdomain (Config.GETTEXT_PACKAGE);
#endif

        var context = new GLib.OptionContext (
            /* i18n: The meaning of the uppercase values is analogous to the
                     fields at the top of a message, ie. From, To, Copy */
            _("[mailto:][ADDRESS][?subject=SUBJECT][&body=BODY]")
          + _("[&from=FROM][&to=TO][&cc=COPY][&attach=ATTACHMENT][&part=PART]"));
        context.set_help_enabled (true);
        context.add_main_entries (options, null);
        context.add_group (Gtk.get_option_group (false));
        Environment.set_application_name (_("Postler"));
        try {
	    context.parse (ref args);
	} catch (GLib.Error error) {
            /* i18n: Command line arguments are invalid */
	    GLib.error (_("Failed to parse command line: %s"), error.message);
	}

        if (version) {
            stdout.printf ("%s %s\n\n%s\n\t%s\n",
                Config.PACKAGE_NAME, Config.PACKAGE_VERSION,
                _("Please report comments, suggestions and bugs to:"),
                Config.PACKAGE_BUGREPORT);
            return 0;
        }

        /* --verbose or POSTLER_DEBUG=1 enables debug messages */
        unowned string? debug = Environment.get_variable ("POSTLER_DEBUG");
        if (verbose || debug == "1") {
            Environment.set_variable ("POSTLER_DEBUG", "1", false);
        }
        else {
            LogFunc handler = (domain, level, message) => { };
            Log.set_handler (null, LogLevelFlags.LEVEL_DEBUG, handler);
        }

        /* mailto, ? or @ implies compose, otherwise file implies content */
        if (module == null && filenames != null && filenames[0] != null) {
            if (filenames[0].has_prefix ("mailto:")
             || filenames[0].has_prefix ("?")
             || (filenames[0].chr (-1, '@') != null && filenames[0].chr (-1, '/') == null))
                module = "compose";
            else
                module = "content";
        }

        if (module == null)
            module = "bureau";

        Environment.set_prgname ("postler-" + module);

        if (module == "service") {
            var service = new Postler.Service ();
            service.done.connect ((status) => {
                Process.exit (status);
            });
            service.run ();
            assert_not_reached ();
        }

        Gtk.init (ref args);
        /* FIXME: only for new webkit */
        WebKit.set_cache_model (WebKit.CacheModel.DOCUMENT_VIEWER);
        Postler.App.register_stock_items ();

        if (module != "bureau") {
            Gtk.Widget instance = null;
            string icon_name = null;
            string title = null;

            if (module == "content" || module == "source") {
                instance = new Postler.Content ();
                icon_name = module == "source" ? "text-html" : "emblem-mail";
                title = _("(No subject)");
                if (filenames != null && filenames[0] != null) {
                    Postler.Content content = (Postler.Content)instance;
                    string filename;
                    try {
                       filename = Filename.from_uri (filenames[0], null);
                    } catch (GLib.Error error) {
                        filename = filenames[0];
                    }
                    if (module == "source")
                        content.display_source (filename);
                    else
                        content.display (filename);
                    title = content.subject;
                }
            }
            else if (module == "compose") {
                Postler.Composer composer = new Postler.Composer ();
                instance = composer;
                string mailto = "";
                if (filenames != null && filenames[0] != null)
                    mailto = filenames[0];
                /* mailto: may or may not be stripped by the calling app */
                if (mailto.has_prefix ("mailto:"))
                    mailto = mailto.substring (7, -1);

                    string[] fields = mailto.split_set ("?&");
                    foreach (var field in fields) {
                        string[] pieces = field.split ("=");
                        bool success;
                        if (pieces[0] != null && pieces[1] != null)
                            success = composer.add_field (pieces[0], Soup.URI.decode (pieces[1]));
                        else
                            success = composer.add_field ("to", Soup.URI.decode (field));
                        if (!success)
                            GLib.warning (_("Invalid field \"%s\" was ignored."),
                                          pieces[1] != null ? pieces[0] : field);
                    }

                    if (filenames != null && filenames[1] != null) {
                        composer.prepare_reply (filenames[1],
                            composer.subject.has_prefix ("Re: ")
                         || composer.subject.has_prefix ("Fw: "));
                    } else {
                        composer.prepare_reply ();
                    }

                instance.destroy.connect (Gtk.main_quit);
                instance.show ();
                Gtk.main ();
                return 0;
            }
            else
                GLib.error ("Unknown module \"%s\". Valid modules are: %s",
                    module, "bureau content source compose service");

            /* FIXME: Escape to close window */
            var window = new Gtk.Window (Gtk.WindowType.TOPLEVEL);
            window.set_icon_name (icon_name);
            window.set_title (title);
            var screen = window.get_screen ();
            Gdk.Rectangle monitor;
            screen.get_monitor_geometry (0, out monitor);
            double window_width = monitor.width / 2.2;
            window.set_default_size ((int)window_width, (int)(monitor.height / 2.2));
            var shelf = new Gtk.VBox (false, 0);
            window.add (shelf);
            /* TODO: menubar, toolbar */
            shelf.add (new Postler.Viewer (instance as Postler.Content));
            window.destroy.connect (Gtk.main_quit);
            window.show_all ();
            Gtk.main ();
            return 0;
        }

        var app = new Postler.App ();
        if (app.is_running) {
            Unique.MessageData data = new Unique.MessageData ();
            if (filenames != null && filenames[0] != null)
                data.set_text (filenames[0], filenames[0].length);
            var response = app.send_message (Unique.Command.ACTIVATE, data);
            if (response == Unique.Response.OK)
                return 0;
            GLib.error (_("Failed to activate running instance"));
        }

        var bureau = new Postler.Bureau ();
        bureau.destroy.connect (Gtk.main_quit);
        bureau.show ();
        if (filenames != null && filenames[0] != null) {
            GLib.Idle.add (() => {
                bureau.folders.select_folder (filenames[0]);
                return false;
            });
        }
        app.watch_window (bureau);
        Gtk.main ();
        return 0;
    }
}

