/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Composer : Gtk.Window {
    Accounts accounts = new Accounts ();
    Postler.Client client = new Postler.Client ();
    Dexter.Dexter dexter = new Dexter.Dexter ();

    Gtk.UIManager ui;
    Gtk.ActionGroup actions;

    Gtk.VBox shelf;
    Gtk.Toolbar toolbar;
    Gtk.ProgressBar progressbar;
    Postler.Content content;
    Postler.Attachments attachments;
    Gtk.ComboBox combo_from;
    Postler.RecipientEntry entry_to;
    Postler.RecipientEntry entry_copy;
    Postler.RecipientEntry entry_blind_copy;
    Gtk.Entry entry_subject;
    Gtk.Button button_send;
    Gtk.Button button_save;
    uint grace_period_timer = 0;
    uint progress_timer = 0;
    bool message_saved = false;
    string? draft_filename = null;

    public string subject { get { return entry_subject.text; } }
    int part = 0;
    string? body = null;

    const string ui_markup = """
        <ui>
            <menubar>
                <menu action="Mail">
                    <menuitem action="MessageSend"/>
                    <separator/>
                    <menuitem action="MessageSave"/>
                    <separator/>
                    <menuitem action="FileAttach"/>
                    <separator/>
                    <menuitem action="Close"/>
                </menu>
                <menu action="Edit">
                    <menuitem action="MarkImportant"/>
                    <menuitem action="Quote"/>
                    <menuitem action="InsertEmoticonSmileBig"/>
                    <menuitem action="InsertEmoticonWink"/>
                    <menuitem action="InsertEmoticonSad"/>
                </menu>
            </menubar>
            <toolbar>
                <toolitem action="Quote"/>
                <toolitem action="FileAttach"/>
                <toolitem action="MarkImportant"/>
                <separator expand="true"/>
            </toolbar>
            <popup name="emoticons">
                <menuitem action="InsertEmoticonSmileBig"/>
                <menuitem action="InsertEmoticonWink"/>
                <menuitem action="InsertEmoticonSad"/>
            </popup>
        </ui>
    """;

    AccountInfo? selected_account_for_address (string address)
    {
        string[] from = Postler.Messages.parse_address (address);
        foreach (var info in accounts.get_infos ())
            if (info.address != null && from[1] in info.address)
                return info;
        return null;
    }

    void toggle_sensitivity (bool state) {
        combo_from.sensitive = state;
        entry_to.sensitive = state;
        entry_copy.sensitive = state;
        entry_blind_copy.sensitive = state;
        entry_subject.sensitive = state;
        content.sensitive = state;
        attachments.sensitive = state;
        foreach (var child in toolbar.get_children ())
            child.sensitive = state;

        if (state) {
            button_send.label = STOCK_MAIL_SEND;
            button_send.sensitive = button_save.sensitive = true;
            progressbar.hide ();
            grace_period_timer = progress_timer = 0;
        }
        else {
            button_send.label = Gtk.STOCK_CANCEL;
            button_send.parent.sensitive = true;
        }
    }

    bool send_grace_period_timer () {
        progressbar.fraction += 0.2;
        if (progressbar.fraction == 1.0) {
            send_message ();
            return false;
        }
        return true;
    }

    bool send_progress_timer () {
        progressbar.pulse ();
        return true;
    }

    void action_mail_send () {
        if (grace_period_timer > 0) {
            GLib.Source.remove (grace_period_timer);
            if (progress_timer > 0)
                GLib.Source.remove (progress_timer);
            toggle_sensitivity (true);
            return;
        }

        if (entry_subject.text == "") {
            var dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE,
                _("You did not enter a subject."));
            dialog.format_secondary_text (
                _("Do you want to send the message without a subject?"));
            dialog.add_buttons (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                _("_Send message without subject"), Gtk.ResponseType.OK);
            dialog.set_default_response (Gtk.ResponseType.CANCEL);
            int response = dialog.run ();
            dialog.destroy ();
            if (response != Gtk.ResponseType.OK)
                return;
        }

        content.execute_script ("document.title = document.body.innerText;");
        body = content.title;
        if (attachments.model.iter_n_children (null) == 0) {
            /* i18n: A warning is displayed when writing a new message
                     and there are no attachments but the text mentions
                     attachments. Words are separated by |. */
            string[] attachments = _("attachment|attach").split ("|");
            foreach (string keyword in attachments)
                if (keyword in body) {
                    var dialog = new Gtk.MessageDialog (this, 0,
                        Gtk.MessageType.ERROR, Gtk.ButtonsType.NONE,
        _("An attachment is mentioned in the text, but no files were attached."));
                    dialog.format_secondary_text (
                        _("Do you want to send the message without attachments?"));
                    dialog.add_buttons (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                        _("_Send message without attachments"), Gtk.ResponseType.OK);
                    dialog.set_default_response (Gtk.ResponseType.CANCEL);
                    int response = dialog.run ();
                    dialog.destroy ();
                    if (response != Gtk.ResponseType.OK)
                        return;
                    break;
                }
        }

        string? sender = combo_from.get_active_text ();
        if (sender == null) {
            var dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.ERROR, Gtk.ButtonsType.NONE,
                _("You have not configured any accounts for sending."));
            dialog.add_buttons (Gtk.STOCK_OK, Gtk.ResponseType.OK);
            dialog.run ();
            dialog.destroy ();
            return;
        }

        var info = selected_account_for_address (sender);
        string queue = info.path + "/" + info.get_folder (FolderType.QUEUE);
        string sent = info.path + "/" + info.get_folder (FolderType.SENT);
        if (queue.has_suffix ("/") || sent.has_suffix ("/")
         || !Postler.Messages.ensure_folder (queue)
         || !Postler.Messages.ensure_folder (sent)) {
            var dialog = new Gtk.MessageDialog (this, 0,
                 Gtk.MessageType.ERROR, Gtk.ButtonsType.NONE,
                _("Folders for sent messages and outbox can't be created. "
                + "You need to fetch mail at least once.\n"
                + "If you see this error after fetching, right-click "
                + "the folder for sent messages, \"Use as...\" and "
                + "select \"Sent\". Do the same for the outbox."));
            dialog.add_buttons (Gtk.STOCK_OK, Gtk.ResponseType.OK);
            dialog.run ();
            dialog.destroy ();
            return;
        }

        toggle_sensitivity (false);
        progressbar.fraction = 0.0;
        progressbar.show ();
        grace_period_timer = GLib.Timeout.add_seconds (1, send_grace_period_timer);
    }

    void action_mail_save () {
        save_draft ();
        if (message_saved)
            destroy();
    }

    string generate_message (string sender, AccountInfo info) {
        var now = new Soup.Date.from_now (0);
        string copy = entry_copy.text;
        string blind_copy = entry_blind_copy.text;
        string header = ("From: %s\nTo: %s\nSubject: %s\n"
                       + "Date: %s\nX-Mailer: %s\nMIME-Version: 1.0\n").printf (
                         sender,
                         entry_to.text,
                         entry_subject.text,
                         now.to_string (Soup.DateFormat.RFC2822),
                         "Postler/" + Config.PACKAGE_VERSION.substring (0, 3));
        if (info.reply != null)
            header += "Reply-To: " + info.reply + "\n";
        if (info.organization != null)
            header += "Organization: " + info.organization + "\n";
        if (copy != "")
            header += "CC: " + copy + "\n";
        if (blind_copy != "")
            header += "BCC: " + blind_copy + "\n";
        if ((actions.get_action ("MarkImportant") as Gtk.ToggleAction).active)
            header += "X-Priority: 1\n";
        if (content.message_id != null)
            header += "In-Reply-To: " + content.message_id + "\n";

        Gtk.TreeIter iter;
        if (!(attachments.model.get_iter_first (out iter))) {
            header = header
                + "Content-Transfer-Encoding: 8bit\n"
                + "Content-Type: text/plain; charset=UTF-8\n";
        }

        var clipboard = content.get_clipboard (Gdk.SELECTION_CLIPBOARD);
        clipboard.set_text ("", 0);
        content.select_all ();
        content.copy_clipboard ();
        string body = clipboard.wait_for_text ();
        content.execute_script ("window.getSelection ().removeAllRanges ();");

        if (attachments.model.get_iter_first (out iter)) {
            string boundary = GLib.Checksum.compute_for_string (
                GLib.ChecksumType.MD5,
                body + now.to_string (Soup.DateFormat.RFC2822));
            header = header
                + "Content-Type: multipart/mixed; "
                +  "boundary=" + boundary + "\n";
            StringBuilder body_builder = new StringBuilder (body);
            body_builder.prepend ("Content-Type: text/plain; charset=UTF-8\n\n");
            body_builder.prepend ("--" + boundary + "\n");
            body_builder.append_c ('\n');
            MessagePart part;
            attachments.model.get (iter, 0, out part);
            try {
                body_builder.append ("--" + boundary + "\n");
                uchar[] data = {};
                FileUtils.get_data (part.filename, out data);
                bool uncertain;
                string mime_type = g_content_type_guess (part.filename, data,
                                                         out uncertain);
                var fname_parts = part.filename.split ("/");
                var filename = fname_parts[fname_parts.length-1];
                body_builder.append ("Content-Type: " + mime_type + "\n"
                    + "Content-Disposition: attachment; "
                    + "filename=\"" + filename + "\"\n"
                    + "Content-Transfer-Encoding: base64\n\n"
                    + GLib.Base64.encode (data));
                while (attachments.model.iter_next (ref iter)) {
                    attachments.model.get (iter, 0, out part);
                    body_builder.append ("\n--" + boundary + "\n");
                    FileUtils.get_data (part.filename, out data);
                    mime_type = g_content_type_guess (part.filename, data,
                                                      out uncertain);
                    fname_parts = part.filename.split ("/");
                    filename = fname_parts[fname_parts.length - 1];
                    body_builder.append ("Content-Type: " + mime_type + "\n"
                        + "Content-Disposition: attachment; "
                        + "filename=\"" + filename + "\"\n"
                        + "Content-Transfer-Encoding: base64\n\n"
                        + GLib.Base64.encode (data));
                }
                body_builder.append ("\n--" + boundary + "--");
                body = body_builder.str;
                } catch (Error e) {
                GLib.message (_("Failed to add attachment: %s"), part.filename);
            }
        }
        return header + "\n" + body;
    }

    bool save_draft () {
        button_send.sensitive = button_save.sensitive = false;

        string? sender = combo_from.get_active_text ();
        if (sender == null) {
            var dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.ERROR, Gtk.ButtonsType.NONE,
                _("You have not configured any accounts for sending."));
            dialog.add_buttons (Gtk.STOCK_OK, Gtk.ResponseType.OK);
            dialog.run ();
            dialog.destroy ();
            return true;
        }

        var info = selected_account_for_address (sender);
        string drafts = info.path + "/" + info.get_folder (FolderType.DRAFTS);
        if (drafts.has_suffix ("/") || !Postler.Messages.ensure_folder (drafts)) {
            var dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.ERROR, Gtk.ButtonsType.NONE,
                _("Folder for drafts saving can't be created. "
                + "You need to fetch mail at least once.\n"
                + "If you see this error after fetching, right-click "
                + "the folder for drafts, \"Use as...\" and "
                + "select \"Drafts\"."));
            dialog.add_buttons (Gtk.STOCK_OK, Gtk.ResponseType.OK);
            dialog.run ();
            dialog.destroy ();
            return true;
        }

        string message = generate_message (sender, info);

        try {
            if (draft_filename == null) {
                draft_filename = Postler.Messages.generate_filename (drafts + "/cur/", "D");
            } else {
                FileUtils.remove (draft_filename);
            }
            FileUtils.set_contents (draft_filename, message, -1);
            FileUtils.chmod (draft_filename, 0700);
        } catch (GLib.Error error) {
            GLib.critical (_("Failed to save message: %s"), error.message);
        }
        message_saved = true;
        toggle_sensitivity (true);
        return true;
    }

    void send_message () {
        progress_timer = GLib.Timeout.add_seconds (1, send_progress_timer);
        button_send.sensitive = button_save.sensitive = false;

        string? sender = combo_from.get_active_text ();
        var info = selected_account_for_address (sender);
        string message = generate_message (sender, info);

                try {
                    /* TODO: Put in /Queue/new/ first and move on success */
                    string sent = info.path + "/" + info.get_folder (FolderType.SENT);
                    string filename = Postler.Messages.generate_filename (sent + "/cur/", "S");
                    FileUtils.set_contents (filename, message, -1);
                    FileUtils.chmod (filename, 0700);
                    client.sent.connect ((account, file, message) => {
                        if (account != info.name || file != filename)
                            return;
                        if (message == null) {
                            destroy ();
                            return;
                        }
                        var error_dialog = new Gtk.MessageDialog (null, 0,
                            Gtk.MessageType.ERROR, Gtk.ButtonsType.OK,
                            _("Failed to send message."));
                        error_dialog.format_secondary_text (message);
                        error_dialog.run ();
                        error_dialog.destroy ();
                        toggle_sensitivity (true);
                    });
                    client.send (info.name, filename);
                } catch (GLib.Error error) {
                    GLib.critical (_("Failed to send message: %s"), error.message);
                    toggle_sensitivity (true);
                }
    }

    void action_close () {
        if (!window_not_closed ())
            destroy ();
    }

    bool window_not_closed () {
        if (content.can_undo () && !message_saved) {
            var dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE,
                _("Do you want to discard the unsaved message?"));
            dialog.add_buttons (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                Gtk.STOCK_DISCARD, Gtk.ResponseType.OK);
            dialog.set_default_response (Gtk.ResponseType.CANCEL);
            int response = dialog.run ();
            dialog.destroy ();
            if (response != Gtk.ResponseType.OK)
                return true;
        }
        return false;
    }

    void insert_text (string text) {
        content.execute_script ("""
            var text = '%s';
            if (!document.execCommand ('inserthtml', false, text))
                document.body.innerHTML = text + document.body.innerHTML;
            """.
            printf (text.replace ("'", "\\'")));
    }

    void action_quote () {
        if (!content.can_copy_clipboard ())
            content.select_all ();
        var clipboard = content.get_clipboard (Gdk.SELECTION_PRIMARY);
        string selected = clipboard.wait_for_text ();
        insert_text ("&gt; " + selected.replace ("\n", "\\n&gt; "));
    }

    void action_insert_attachment (Gtk.Action action) {
        string filename = null;
        var filechooser = new Gtk.FileChooserDialog (_("Attach File..."),
            null, Gtk.FileChooserAction.OPEN,
            Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
            Gtk.STOCK_OPEN, Gtk.ResponseType.ACCEPT);
        filechooser.set_current_folder (Environment.get_home_dir ());
        if (filechooser.run () == Gtk.ResponseType.ACCEPT)
            filename = filechooser.get_filename ();
        else
            filename = null;
        filechooser.destroy ();
        if (filename == null)
            return;
        attachments.add_filename (filename);
    }

    void drag_received (Gdk.DragContext context, int x, int y,
        Gtk.SelectionData data, uint info, uint timestamp) {

        bool success = false;
        foreach (string uri in data.get_uris ()) {
            File file = File.new_for_uri (uri);
            if (file.query_file_type (FileQueryInfoFlags.NOFOLLOW_SYMLINKS)
                == FileType.REGULAR && file.is_native ()) {
                attachments.add_filename (file.get_path ());
                success = true;
            }
        }
        Gtk.drag_finish (context, success, false, timestamp);
    }

    void action_emoticons (Gtk.Action action) {
        var menu = ui.get_widget ("/emoticons") as Gtk.Menu;
        if (menu.get_attach_widget () == null) {
            var proxy = action.get_proxies ().nth_data (0);
            menu.attach_to_widget (proxy, (widget, menu) => { });
        }
        menu.popup (null, null, Postler.App.button_menu_position, 1,
                    Gtk.get_current_event_time ());
        menu.select_first (true);
        menu.deactivate.connect (( menu_shell ) => {
            (menu.attach_widget as Gtk.ToggleToolButton).active = false;
        });
    }

    void action_insert_emoticon (Gtk.Action action) {
        unowned string name = action.name;
        string emoticon;
        if (name.has_suffix ("SmileBig"))
            emoticon = ":-D";
        else if (name.has_suffix ("Wink"))
            emoticon = ";-)";
        else if (name.has_suffix ("Sad"))
            emoticon = ":-(";
        else
            assert_not_reached ();
        insert_text (emoticon);
    }

    void action_address_book () {
        dexter.show_window ();
    }

    const Gtk.ActionEntry[] action_entries = {
        { "Mail", null, N_("_Mail") },
        { "MessageSend", STOCK_MAIL_SEND, null, "<Ctrl>Return",
          N_("Send the message"), action_mail_send },
        { "MessageSave", Gtk.STOCK_SAVE, null, "<Ctrl>s",
          N_("Save message as draft"), null },
        { "FileAttach", STOCK_MAIL_ATTACHMENT, N_("_Attach File"), "",
          N_("Attach a file to the message"), action_insert_attachment },
        { "Close", Gtk.STOCK_CLOSE, null, "Escape",
          N_("Close the window"), action_close },
        { "Edit", null, N_("_Edit") },
        { "AddressBook", STOCK_ADDRESSBOOK, null, "<Ctrl><Shift>a",
          N_("Open the addressbook"), action_address_book },
        { "Quote", Gtk.STOCK_INDENT, N_("_Quote the selected text"), "",
          N_("Mark the selected text as a quote"), action_quote },
        { "Emoticons", STOCK_FACE_SMILE_BIG, N_("Insert _Smiley"), "", null, action_emoticons },
        { "InsertEmoticonSmileBig", STOCK_FACE_SMILE_BIG, N_("Insert Big _Smile"), "",
          N_("Insert a grinning face"), action_insert_emoticon },
        { "InsertEmoticonWink", STOCK_FACE_WINK, N_("Insert _Winking"), "",
          N_("Insert a winking face"), action_insert_emoticon },
        { "InsertEmoticonSad", STOCK_FACE_SAD, N_("Insert _Sad Face"), "",
          N_("Insert a sad face"), action_insert_emoticon }
    };

    const Gtk.ToggleActionEntry[] toggle_action_entries = {
        { "MarkImportant", STOCK_MAIL_MARK_IMPORTANT, null, "",
          N_("Mark message as important"), null, false }
    };

    public Composer () {
        GLib.Object (icon_name: STOCK_MAIL_MESSAGE_NEW,
                     title: _("Compose message"));
        /* Autosave every 5 minutes */
        GLib.Timeout.add_seconds (350, save_draft);

        var screen = get_screen ();
        Gdk.Rectangle monitor;
        screen.get_monitor_geometry (0, out monitor);
        set_default_size ((int)(monitor.width / 2.5), (int)(monitor.height / 2.0));

        ui = new Gtk.UIManager ();
        actions = new Gtk.ActionGroup ("Composer");
        actions.set_translation_domain (Config.GETTEXT_PACKAGE);
        actions.add_actions (action_entries, this);
        actions.add_toggle_actions (toggle_action_entries, this);
        ui.insert_action_group (actions, 0);
        try {
            ui.add_ui_from_string (ui_markup, -1);
            add_accel_group (ui.get_accel_group ());
        }
        catch (Error error) {
            GLib.error (_("Failed to create window: %s"), error.message);
        }

        shelf = new Gtk.VBox (false, 0);
        add (shelf);
        var menubar = ui.get_widget ("/menubar");
        /* The menubar is nice for globalmenu, but otherwise redundant */
        menubar.set_no_show_all (true);
        menubar.hide ();
        shelf.pack_start (menubar, false, false, 0);
        toolbar = ui.get_widget ("/toolbar") as Gtk.Toolbar;
        toolbar.set_icon_size (Gtk.IconSize.BUTTON);

        /* Grace period progress after activating Send */
        var toolitem = new Gtk.ToolItem ();
        var align = new Gtk.Alignment (0.0f, 0.5f, 1.0f, 0.1f);
        align.left_padding = 8;
        progressbar = new Gtk.ProgressBar ();
        progressbar.set_no_show_all (true);
        align.add (progressbar);
        toolitem.add (align);
        toolbar.insert (toolitem, toolbar.get_n_items () - 1);

        Gtk.rc_parse_string ("""
            style "postler-widebutton"
            {
                 GtkButton::inner-border = { 16, 16, 0, 0 }
            }
            widget "*PostlerWideButton" style "postler-widebutton"
            """
        );

        toolitem = new Gtk.ToolItem ();
        toolitem.set_border_width (4);
        button_save = new Gtk.Button.from_stock (Gtk.STOCK_SAVE);
        button_save.name = "PostlerWideButton";
        toolitem.add (button_save);
        actions.get_action ("MessageSave").connect_proxy (button_save);
        button_save.clicked.connect (action_mail_save);
        toolbar.insert (toolitem, -1);
        toolitem = new Gtk.ToolItem ();
        toolitem.set_border_width (4);
        button_send = new Gtk.Button.from_stock (STOCK_MAIL_SEND);
        button_send.name = "PostlerWideButton";
        toolitem.add (button_send);
        var send = actions.get_action ("MessageSend");
        send.connect_proxy (button_send);
        send.notify["sensitive"].connect ((action, pspec) => {
            button_send.sensitive = send.sensitive;
        });
        button_send.clicked.connect ((widget) => {
            action_mail_send ();
        });
        toolbar.insert (toolitem, -1);

        toolitem = new Gtk.ToggleToolButton.from_stock (STOCK_FACE_SMILE_BIG);
        (toolitem as Gtk.ToggleToolButton).label = _("Insert Smiley");
        toolitem.tooltip_text = _("Insert a Smiley into the message");
        actions.get_action ("Emoticons").connect_proxy (toolitem);
        (toolitem as Gtk.ToggleToolButton).clicked.connect ((widget) => {
            action_emoticons (actions.get_action ("Emoticons"));
        });
        toolbar.insert (toolitem, 0);

        /* Make non-tool-buttons homogeneous like in a dialogue */
        foreach (var child in toolbar.get_children ())
            (child as Gtk.ToolItem).set_homogeneous (!(child is Gtk.ToolButton));

        shelf.pack_end (toolbar, false, false, 0);
        var sizegroup = new Gtk.SizeGroup (Gtk.SizeGroupMode.HORIZONTAL);
        var box = new Gtk.HBox (false, 0);
        shelf.pack_start (box, false, false, 4);
        var label_from = new Gtk.Label.with_mnemonic (_("_From:"));
        label_from.xalign = 1.0f;
        box.pack_start (label_from, false, false, 4);
        sizegroup.add_widget (label_from);
        combo_from = new Gtk.ComboBox.text ();
        box.pack_start (combo_from, true, true, 12);
        box = new Gtk.HBox (false, 0);
        shelf.pack_start (box, false, false, 4);
        var label_and_arrow_vbox = new Gtk.VBox (false, 0);
        box.pack_start (label_and_arrow_vbox, false, false, 4);
        var label_and_arrow = new Gtk.HBox (false, 0);
        label_and_arrow_vbox.pack_start (label_and_arrow, false, false, 4);
        var button = new Gtk.Button ();
        var arrow = new Gtk.Arrow (Gtk.ArrowType.RIGHT, Gtk.ShadowType.NONE);
        button.add (arrow);
        button.set_relief (Gtk.ReliefStyle.NONE);
        button.focus_on_click = false;
        button.clicked.connect ((widget) => {
            if (arrow.arrow_type == Gtk.ArrowType.RIGHT) {
                entry_copy.parent.show ();
                entry_blind_copy.parent.show ();
                arrow.arrow_type = Gtk.ArrowType.DOWN;
            }
            else {
                entry_blind_copy.parent.hide ();
                arrow.arrow_type = Gtk.ArrowType.RIGHT;
            }
        });
        label_and_arrow.pack_start (button, false, false, 4);
        var label = new Gtk.Label.with_mnemonic (_("_To:"));
        label.xalign = 1.0f;
        label_and_arrow.pack_start (label, false, false, 4);
        sizegroup.add_widget (label_and_arrow);
        entry_to = new Postler.RecipientEntry ();
        box.pack_end (entry_to, true, true, 12);
        box = new Gtk.HBox (false, 0);
        shelf.pack_start (box, false, false, 4);
        label = new Gtk.Label.with_mnemonic (_("_Copy:"));
        label.xalign = 1.0f;
        box.pack_start (label, false, false, 4);
        sizegroup.add_widget (label);
        entry_copy = new Postler.RecipientEntry ();
        box.pack_end (entry_copy, true, true, 12);
        label.show ();
        entry_copy.show ();
        box.set_no_show_all (true);
        box = new Gtk.HBox (false, 0);
        shelf.pack_start (box, false, false, 4);
        label = new Gtk.Label.with_mnemonic (_("_Blind Copy:"));
        label.xalign = 1.0f;
        box.pack_start (label, false, false, 4);
        sizegroup.add_widget (label);
        entry_blind_copy = new Postler.RecipientEntry ();
        box.pack_end (entry_blind_copy, true, true, 12);
        label.show ();
        entry_blind_copy.show ();
        box.set_no_show_all (true);
        box = new Gtk.HBox (false, 0);
        shelf.pack_start (box, false, false, 4);
        label = new Gtk.Label.with_mnemonic (_("_Subject:"));
        label.xalign = 1.0f;
        box.pack_start (label, false, false, 4);
        sizegroup.add_widget (label);
        entry_subject = new Gtk.Entry ();
        box.pack_start (entry_subject, true, true, 12);
        content = new Postler.Content ();
        var scrolled = new Postler.ScrolledWindow (content);
        scrolled.shadow_type = Gtk.ShadowType.IN;
        shelf.pack_start (scrolled, true, true, 0);

        attachments = new Postler.Attachments ();
        attachments.editable = true;
        shelf.pack_start (attachments, false, false, 0);
        shelf.show_all ();

        Gtk.drag_dest_set (content, Gtk.DestDefaults.ALL, {}, Gdk.DragAction.COPY);
        Gtk.drag_dest_add_uri_targets (content);
        content.drag_data_received.connect (drag_received);
        Gtk.drag_dest_set (attachments, Gtk.DestDefaults.ALL, {}, Gdk.DragAction.COPY);
        Gtk.drag_dest_add_uri_targets (attachments);
        attachments.drag_data_received.connect (drag_received);

        actions.get_action ("MessageSend").sensitive = false;
        entry_to.notify["empty"].connect ((object, pspec) => {
            actions.get_action ("MessageSend").sensitive = !entry_to.empty;
        });

        foreach (var account_info in accounts.get_infos ()) {
            if (account_info.address != null) {
                string[] addresses = account_info.address.split (",");
                if (addresses[0] != null) {
                    foreach (string address in addresses) {
                        unowned string realname = account_info.realname;
                        if (realname != null && realname != "")
                            combo_from.append_text ("%s <%s>".printf (
                                realname, address));
                        else
                            combo_from.append_text (address);
                    }
                }
                else
                    combo_from.append_text (account_info.address);
            }
        }

        /* Select first account, hide From if there's only one account */
        combo_from.set_active (0);
        if (combo_from.model.iter_n_children (null) > 1)
            combo_from.grab_focus ();
        else {
            label_from.hide ();
            combo_from.hide ();
            entry_to.grab_focus ();
        }

        content.editable = true;
        var settings = content.settings;
        settings.set ("enable-scripts", true);
        if (settings.get_class ().find_property ("enable-spell-checking") != null) {
            string languages = string.joinv (",", Intl.get_language_names ());
            settings.set ("enable-spell-checking", true,
                          "spell-checking-languages", languages, null);
        }

        delete_event.connect (window_not_closed);
    }

    public bool prepare_reply (string? location=null, bool quote=false) {
        var info = selected_account_for_address (combo_from.get_active_text ());
        content.prepare_reply (location, info, entry_to.text, quote, part, body);
        return true;
    }

    public bool add_field (string field, string data) {
        if (data == "")
            return false;

        string name = field.down ();
        Gtk.Entry entry = null;
        if (name == "from") {
            var model = combo_from.model;
            Gtk.TreeIter iter;
            string from = Postler.Messages.parse_address (data)[1];
            if (model.iter_children (out iter, null)) {
                do {
                    string address;
                    model.get (iter, 0, out address);
                    if (from in address) {
                        combo_from.set_active_iter (iter);
                        break;
                    }
                } while (model.iter_next (ref iter));
            }
            return true;
        }
        else if (name == "to")
            entry = entry_to.entry;
        else if (name == "cc") {
            entry = entry_copy.entry;
            entry_copy.parent.show ();
        }
        else if (name == "bcc") {
            entry = entry_blind_copy.entry;
            entry_blind_copy.parent.show ();
        }
        else if (name == "subject") {
            entry_subject.set_text (data);
            return true;
        }
        else if (name == "body") {
            body = data;
            return true;
        }
        else if (name == "attach") {
            string filename = data.has_prefix ("file:///")
                ? data.substring (7, -1) : data;
            if (Path.is_absolute (filename)
             && FileUtils.test (filename, FileTest.EXISTS)) {
                attachments.add_filename (filename);
                return true;
            }
            GLib.warning (_("File %s doesn't exist"), data);
            return false;
        } else if (name == "part") {
            part = data.to_int ();
            return true;
        }
        else if (name == "in-reply-to")
            content.message_id = data;
        else
            return false;

        string current = entry.get_text () ?? "";
        entry.set_text (current + data + ",");
        return true;
    }
}


