/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Bureau : Gtk.Window {
    Accounts accounts = new Accounts ();
    Postler.Client client = new Postler.Client ();

    Postler.State previous_state = new State ();

    Gtk.UIManager ui;
    Gtk.ActionGroup actions;

    Gtk.VBox shelf;
    Gtk.Toolbar toolbar;
    Elementary.SearchEntry search;
    Gtk.Toolbar search_options;
    public Postler.Folders folders;
    Gtk.ProgressBar progressbar;
    Gtk.Label statuslabel;
    Gtk.VBox messages_box;
    public Postler.Messages messages;
    Postler.Content content;


    const string ui_markup = """
        <ui>
            <menubar>
                <menu action="Mail">
                    <menuitem action="MailReceive"/>
                    <separator/>
                    <menuitem action="MessageNew"/>
                    <separator/>
                    <menuitem action="MessageReply"/>
                    <menuitem action="MessageReplyAll"/>
                    <menuitem action="MessageForward"/>
                    <separator/>
                    <menuitem action="Quit"/>
                </menu>
                <menu action="Edit">
                    <menuitem action="MessageUnread"/>
                    <menuitem action="MessageFlag"/>
                    <menuitem action="MessageArchive"/>
                    <menuitem action="MessageJunk"/>
                    <menuitem action="MessageDelete"/>
                    <separator/>
                    <menuitem action="MessageNextUnread"/>
                    <menuitem action="MessagePreviousUnread"/>
                </menu>
                <menu action="View">
                    <menuitem action="HideRead"/>
                    <menuitem action="ZoomIn"/>
                    <menuitem action="ZoomOut"/>
                    <separator/>
                    <menuitem action="Fullscreen"/>
                    <menuitem action="ViewSource"/>
                    <separator/>
                    <menuitem action="Search"/>
                </menu>
                <menu action="Help">
                    <menuitem action="Shortcuts"/>
                    <menuitem action="About"/>
                </menu>
            </menubar>
            <toolbar>
                <toolitem action="MailReceive"/>
                <toolitem action="MessageNew"/>
                <separator/>
                <toolitem action="MessageReply"/>
                <toolitem action="MessageReplyAll"/>
                <toolitem action="MessageForward"/>
                <separator/>
                <toolitem action="MessageArchive"/>
                <toolitem action="MessageJunk"/>
                <toolitem action="MessageDelete"/>
                <separator expand="true"/>
            </toolbar>
            <popup name="view">
                <menuitem action="HideRead"/>
                <separator/>
                <menuitem action="Fullscreen"/>
                <menuitem action="AccountNew"/>
                <menuitem action="ReportBug"/>
                <menuitem action="Shortcuts"/>
                <menuitem action="About"/>
            </popup>
            <toolbar name="search_options">
                <toolitem action="SearchSubject"/>
                <toolitem action="SearchSender"/>
                <toolitem action="SearchRecipient"/>
                <toolitem action="SearchBody"/>
                <separator expand="true"/>
                <toolitem action="SaveSearch"/>
            </toolbar>
        </ui>
    """;

    void action_mail_receive () {
        client.receive ();
    }

    void compose_message (string? prefix=null, string? recipient=null, int part=0) {
        string? chosen_from = content.choose_from ();
        if (chosen_from == null) {
            if (messages.account_info != null)
                chosen_from = messages.account_info.address;
            else
                chosen_from = "";
        }

        var clipboard = content.get_clipboard (Gdk.SELECTION_PRIMARY);
        string? selected = null;
        if (content.can_copy_clipboard ())
            selected = GLib.Uri.escape_string (clipboard.wait_for_text (), "", true);

        StringBuilder mailto = new StringBuilder (recipient);
        char delimiter = '?';
        if (prefix != null) {
            mailto.append_printf ("?subject=%s%s", prefix, content.subject);
            delimiter = '&';
        }
        mailto.append_printf ("%cfrom=%s", delimiter, chosen_from);
        delimiter = '&';
        mailto.append_printf ("&part=%d", part);
        if (selected != null) {
            mailto.append_printf ("&body=%s", selected);
            mailto.append_printf ("&in-reply-to=%s", content.message_id);
        }

        Postler.App.spawn_module ("compose", mailto.str,
            prefix != null ? content.last_location : null);
    }

    void action_message_new () {
        compose_message ();
    }

    unowned string? list_or_sender (string reply_to) {
        if (content.list_post == "")
            return reply_to;

        Gtk.MessageDialog dialog;

        if (content.list_post.has_prefix ("NO ")) {
            dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE,
                _("This message was sent to a mailing list."));
            dialog.format_secondary_text (_("Replying is not allowed."));
            dialog.add_buttons (_("Reply to _Sender"), Gtk.ResponseType.CANCEL,
                Gtk.STOCK_CANCEL, Gtk.ResponseType.NONE);
            dialog.set_default_response (Gtk.ResponseType.NONE);
        }
        else {
            dialog = new Gtk.MessageDialog (this, 0,
                Gtk.MessageType.QUESTION, Gtk.ButtonsType.NONE,
                _("This message was sent to a mailing list."));
            dialog.format_secondary_text (_("Do you want to reply to the list?"));
            dialog.add_buttons (_("Reply to _Sender"), Gtk.ResponseType.CANCEL,
                _("Reply to _List"), Gtk.ResponseType.OK);
            dialog.set_default_response (Gtk.ResponseType.OK);
        }

        int response = dialog.run ();
        dialog.destroy ();

        if (response == Gtk.ResponseType.OK)
            return content.list_post;
        else if (response == Gtk.ResponseType.NONE)
            return null;
        return reply_to;
    }

    void action_message_reply () {
        unowned string? reply = list_or_sender (content.reply_to);
        if (reply != null)
            compose_message ("Re: ", reply, content.current_part_index);
    }

    void action_message_reply_all () {
        unowned string? reply = content.reply_to_all;
        if (reply != null)
            compose_message ("Re: ", reply, content.current_part_index);
    }

    void action_message_forward () {
        compose_message ("Fw: ", null, content.current_part_index);
    }

    void action_message_unread () {
        messages.toggle_selected_flag ('S');
    }

    void action_flag () {
        messages.toggle_selected_flag ('F');
    }

    void action_archive () {
        messages.move_selected (FolderType.ARCHIVE);
    }

    void action_junk () {
        string folder = Path.get_basename (folders.selected_location);
        if (folder != messages.account_info.get_folder (FolderType.JUNK))
            messages.move_selected (FolderType.JUNK);
        else
            messages.move_selected (FolderType.INBOX);
    }

    void action_delete () {
        string folder = Path.get_basename (folders.selected_location);
        if (folder == messages.account_info.get_folder (FolderType.TRASH))
            messages.move_selected (FolderType.INBOX);
        else
            messages.move_selected (FolderType.TRASH);
    }

    void action_previous_unread () {
        messages.select_next_unread (false);
    }

    void action_next_unread () {
        messages.select_next_unread (true);
    }

    void action_search () {
        search.grab_focus ();
    }

    void action_search_options () {
        search.activate ();
    }

    void search_entry_activated () {
        var action = actions.get_action ("SearchSubject") as Gtk.RadioAction;
        if (search.get_text () == search.hint_string || search.get_text () == "") {
            search_options.hide ();
            action.set_current_value (0);
            messages.search ("");
            return;
        }

        switch (action.get_current_value ()) {
        case 0:
            messages.search (search.get_text (), "subject");
            break;
        case 1:
            messages.search (search.get_text (), "from");
            break;
        case 2:
            messages.search (search.get_text (), "to");
            break;
        case 3:
            messages.search (search.get_text (), "fulltext");
            break;
        default:
            assert_not_reached ();
        }
        messages.grab_focus ();
        search_options.show ();
    }

    void action_save_search () {
        var account_info = new AccountInfo ();
        account_info.name = search.get_text ();
        account_info.type = AccountType.SEARCH;

        unowned string header;
        var action = actions.get_action ("SearchSubject") as Gtk.RadioAction;
        switch (action.get_current_value ()) {
        case 0:
            header = "subject";
            break;
        case 1:
            header = "from";
            break;
        case 2:
            header = "to";
            break;
        case 3:
            header = "fulltext";
            break;
        default:
            assert_not_reached ();
        }
        account_info.path = "search:" + header + "/" + search.get_text ();
        accounts.add_info (account_info);
        folders.populate ();
    }

    void action_view (Gtk.Action action) {
        var menu = ui.get_widget ("/view") as Gtk.Menu;
        if (menu.get_attach_widget () == null) {
            var proxy = action.get_proxies ().nth_data (0);
            menu.attach_to_widget (proxy, (widget, menu) => { });
        }
        menu.popup (null, null, Postler.App.button_menu_position, 1,
                    Gtk.get_current_event_time ());
        menu.select_first (true);
        menu.deactivate.connect ((menu_shell) => {
            (menu.attach_widget as Gtk.ToggleToolButton).active = false;
        });
    }

    void action_zoom_in () {
        content.zoom_in ();
    }

    void action_zoom_out () {
        content.zoom_out ();
    }

    void action_fullscreen () {
        if ((window.get_state () & Gdk.WindowState.FULLSCREEN) == 0) {
            fullscreen ();
        } else {
            unfullscreen ();
        }
        window_state_event (new Gdk.EventWindowState ());
    }

    public override bool window_state_event (Gdk.EventWindowState event) {
        Gtk.Orientation orientation;
        if ((window.get_state () & Gdk.WindowState.FULLSCREEN) != 0
         || (window.get_state () & Gdk.WindowState.MAXIMIZED) != 0) {
            /* 3 vertical columns */
            orientation = Gtk.Orientation.HORIZONTAL;
        } else {
            /* 1 : 2 layout */
            orientation = Gtk.Orientation.VERTICAL;
        }
        (messages.parent.parent.parent as Gtk.Orientable).set_orientation (orientation);
        return true;
    }

    void action_report_bug () {
        Postler.App.show_uri (get_screen (), "https://bugs.launchpad.net/postler/+filebug");
    }

    static Gtk.Dialog? shortcuts = null;
    void action_shortcuts () {
        if (shortcuts == null) {
            shortcuts = new Postler.Shortcuts (this, actions);
            shortcuts.show ();
            shortcuts.destroy.connect (() => {
                shortcuts = null;
            });
        }
        else
            shortcuts.present ();
    }

    void action_view_source () {
        Postler.App.spawn_module ("source", content.last_location);
    }

    static AccountSetup? setup = null;
    void action_account_new () {
        if (setup != null) {
            setup.present ();
            return;
        }

        setup = AccountSetup.new_account ();
        setup.destroy.connect (() => {
            setup = null;
        });
        setup.done.connect ((setup, info) => {
            accounts.add_info (info);
            client.fetch (info.name);
            folders.populate ();
        } );
    }

    void action_quit () {
        update_saved_state ();
        client.quit ();
        Gtk.main_quit ();
    }

    public override bool delete_event (Gdk.Event event) {
        update_saved_state ();
        return false;
    }

    void update_saved_state () {
        if ((window.get_state () & Gdk.WindowState.MAXIMIZED) != 0)
            previous_state.window_state = 1;
        else if ((window.get_state () & Gdk.WindowState.FULLSCREEN) != 0)
            previous_state.window_state = 2;
        else
            previous_state.window_state = 0;

        if (previous_state.window_state == 0) {
            int width = 0;
            int height = 0;
            get_size (out width, out height);
            previous_state.width = width;
            previous_state.height = height;
        }
        previous_state.open_folder = folders.selected_location ?? "";
        previous_state.update ();
    }

    void action_about () {
        string[] authors = { "Christian Dywan <christian@twotoasts.de>",
                             "Daniel Foré <daniel.p.fore@gmail.com>",
                             "Allen Lowe <lallenlowe@gmail.com>" };
        Gtk.show_about_dialog (this,
            "program-name", GLib.Environment.get_application_name (),
            "version", Config.PACKAGE_VERSION,
            "copyright", "Copyright (C) 2010-2011 Christian Dywan",
            "authors", authors,
            "logo-icon-name", STOCK_INTERNET_MAIL,
            "translator-credits", _("translator-credits"),
            "website", "https://launchpad.net/postler",
            null);
    }

    void action_hide_read () {
        messages.hide_read = !messages.hide_read;
    }

    /* TODO: Send outstanding outgoing mail */
    const Gtk.ActionEntry[] action_entries = {
        { "Mail", null, N_("_Mail") },
        { "MailReceive", STOCK_MAIL_SEND_RECEIVE, null, "<Ctrl>i",
          N_("Receive mail on all accounts"), action_mail_receive },
        { "MessageNew", STOCK_MAIL_MESSAGE_NEW, null, "<Ctrl>m",
          N_("Compose a new message"), action_message_new },
        { "MessageReply", STOCK_MAIL_REPLY_SENDER, null, "<Ctrl>r",
          N_("Reply to the sender of the message"), action_message_reply },
        { "MessageReplyAll", STOCK_MAIL_REPLY_ALL, null, "<Ctrl><Shift>r",
          N_("Reply to all recipients"), action_message_reply_all },
        { "MessageForward", STOCK_MAIL_FORWARD, null, "<Ctrl>f",
          N_("Forward message"), action_message_forward },
        { "MessageUnread", STOCK_MAIL_MARK_UNREAD, null, "<Ctrl>u",
          N_("Mark message as unread"), action_message_unread },
        { "MessageFlag", STOCK_MAIL_MARK_IMPORTANT, null, "<Ctrl>t",
          N_("Flag message"), action_flag },
        { "MessageArchive", STOCK_ARCHIVE_INSERT, null, "<Ctrl>e",
          N_("Archive message"), action_archive },
        { "MessageJunk", STOCK_MAIL_MARK_JUNK, null, "<Ctrl>j",
          N_("Mark message as junk"), action_junk },
        { "MessageDelete", Gtk.STOCK_DELETE, null, "<Ctrl>d",
          N_("Delete message"), action_delete },
        { "MessagePreviousUnread", null, N_("_Previous Unread Message"), "<Alt>Up",
          N_("Go to the previous unread message"), action_previous_unread },
        { "MessageNextUnread", null, N_("_Next Unread Message"), "<Alt>Down",
          N_("Go to the next unread message"), action_next_unread },
        { "Quit", Gtk.STOCK_QUIT, null, "<Ctrl>q",
          N_("Quit the application"), action_quit },
        { "Edit", null, N_("_Edit") },
        { "Search", Gtk.STOCK_FIND, null, "<Ctrl>s",
          N_("Search the selected folder"), action_search },
        { "SaveSearch", null, N_("S_ave Search"), "<Ctrl><Shift>s",
          N_("Save the current search"), action_save_search },
        { "View", null, N_("_View") },
        { "AppMenu", Gtk.STOCK_PROPERTIES, N_("_View"), "",
          null, action_view },
        { "ZoomIn", Gtk.STOCK_ZOOM_IN, N_("_Enlarge Text"), "<Ctrl>plus",
          N_("Enlarge message text"), action_zoom_in },
         { "ZoomOut", Gtk.STOCK_ZOOM_OUT, N_("Sh_rink Text"), "<Ctrl>minus",
          N_("Shrink message text"), action_zoom_out },
        { "Fullscreen", Gtk.STOCK_FULLSCREEN, null, "F11",
          N_("View the message in fullscreen"), action_fullscreen },
        { "ViewSource", null, N_("View _Source"), "<Ctrl><Alt>u",
          N_("View the source of the message"), action_view_source },
        { "AccountNew", STOCK_ACCOUNT_NEW, null, "",
          N_("Setup a new account"), action_account_new },
        { "ReportBug", STOCK_REPORT_BUG, N_("_Report a Problem..."), "",
          N_("Report a Problem..."), action_report_bug },
        { "Help", null, N_("_Help") },
        { "Shortcuts", null, N_("_Shortcuts"),"F1",
         N_("View keyboard shortcuts"), action_shortcuts },
        { "About", Gtk.STOCK_ABOUT, null, "",
          N_("Show information about the program"), action_about }
    };

    const Gtk.ToggleActionEntry[] toggle_entries = {
        { "HideRead", null, N_("_Hide Read"), "<Ctrl>h",
          N_("Hide read messages"), action_hide_read,
          false }
    };

    const Gtk.RadioActionEntry[] radio_entries = {
        { "SearchSubject", null, N_("_Subject"), "",
          N_("Search messages by subject"), 0 },
        { "SearchSender", null, N_("S_ender"), "",
          N_("Search messages by sender"), 1 },
        { "SearchRecipient", null, N_("_Recipient"), "",
          N_("Search messages by recipient"), 2 },
        /* i18n: The text contents of a message, when searching */
        { "SearchBody", null, N_("_Body"), "",
          N_("Search the full message text"), 3}
    };

    void account_check (AccountInfo? account_info) {
        actions.get_action ("MailReceive").sensitive = false;
        actions.get_action ("MessageNew").sensitive = false;
        actions.get_action ("MessageJunk").visible = false;
        foreach (var info in accounts.get_infos ())
            if (info.type == AccountType.IMAP) {
                actions.get_action ("MailReceive").sensitive = true;
                actions.get_action ("MessageNew").sensitive = true;
                if (info.get_folder (FolderType.JUNK) != null)
                    actions.get_action ("MessageJunk").visible = true;
            }
    }

    Gtk.InfoBar show_fetch_error_infobar (string account, string error_message) {
        var infobar = new Gtk.InfoBar ();
        infobar.set_message_type (Gtk.MessageType.ERROR);
        messages_box.pack_start (infobar, false, false, 0);
        var infobox = infobar.get_content_area () as Gtk.Box;
        var image = new Gtk.Image.from_stock (Gtk.STOCK_DIALOG_ERROR,
                                              Gtk.IconSize.BUTTON);
        infobox.pack_start (image, false, false, 0);
        var infolabel = new Gtk.Label (_("%s: %s").printf (
                                       account, error_message));
        infolabel.wrap = true;
        var attrlist = new Pango.AttrList ();
        attrlist.insert (Pango.attr_weight_new (Pango.Weight.BOLD));
        infolabel.attributes = attrlist;
        infobox.pack_start (infolabel, true, false, 0);
        infobar.show_all ();

        /* Infobar closes on button, folder change or fetching mail */
        infobar.response.connect_after ((response) => {
            infobar.destroy ();
        });
        messages.notify["selected-location"].connect ((object, pspec) => {
            infobar.destroy ();
        });
        client.progress.connect ((account_name, text, fraction) => {
            if (account == account_name && text != null && fraction == 0.0)
                infobar.destroy ();
        });
        return infobar;
    }

    void client_received (string account, string? error_message) {
        if (error_message == null)
            return;

        unowned string? debug = Environment.get_variable ("POSTLER_DEBUG");
        if (error_message.has_prefix ("PSTL/CERT/")) {
            var infobar = show_fetch_error_infobar (account,
        _("Can't verify mail server authenticity."));
            infobar.add_button (_("Fetch Without Verifying"), Gtk.ResponseType.OK);
            infobar.response.connect ((response) => {
                foreach (var info in accounts.get_infos ()) {
                    if (info.display_name == account) {
                        info.unverified = true;
                        accounts.update ();
                        client.receive ();
                        return;
                    }
                }
                GLib.critical (_("Account \"%s\" doesn't exist"), account);
            });
        }
        else if (error_message.has_prefix ("PSTL/LOGIN/"))
            show_fetch_error_infobar (account, _("Wrong username or password."));
        else if (error_message.has_prefix ("PSTL/")) {
            string[] parts = error_message.split ("/");
            string error_id = parts[1];
            string folder = parts[2];
            string error_text = error_message;
            if (error_id == "UIDVAL")
                error_text = _("Folder \"%s\" is in invalid state.").printf (folder);
            else if (error_id == "BADUID")
                error_text = _("Failed to update folder \"%s\".").printf (folder);
            else if (debug != "1")
                return;
            var infobar = show_fetch_error_infobar (account, error_text);
            infobar.set_message_type (Gtk.MessageType.WARNING);
        }
        else if (debug == "1")
            show_fetch_error_infobar (account, error_message);
    }

    public Bureau () {
        GLib.Object (icon_name: STOCK_INTERNET_MAIL,
                     title: GLib.Environment.get_application_name ());

        var screen = get_screen ();
        Gdk.Rectangle monitor;
        screen.get_monitor_geometry (0, out monitor);
        double window_width = monitor.width / 1.7;
        double window_height = monitor.height / 1.7;

        if ((double)previous_state.width != 0)
            window_width = (double) previous_state.width;
        if ((double)previous_state.height != 0)
            window_height = (double) previous_state.height;

        set_default_size ((int)window_width, (int)window_height);
        if (previous_state.window_state == 1)
            maximize ();
        if (previous_state.window_state == 2)
            fullscreen ();

        ui = new Gtk.UIManager ();
        actions = new Gtk.ActionGroup ("Bureau");
        actions.set_translation_domain (Config.GETTEXT_PACKAGE);
        actions.add_actions (action_entries, this);
        actions.add_toggle_actions  (toggle_entries, this);
        actions.add_radio_actions  (radio_entries, 0, action_search_options);
        ui.insert_action_group (actions, 0);
        try {
            ui.add_ui_from_string (ui_markup, -1);
            add_accel_group (ui.get_accel_group ());
        }
        catch (Error error) {
            GLib.error (_("Failed to create window: %s"), error.message);
        }

        var folder_actions = new Gtk.ActionGroup ("Bureau/Folders");
        for (int i = 1; i < 9; i++) {
            string name = "Inbox%d".printf (i);
            var action = new Gtk.Action (name, name, null, null);
            action.activate.connect ((action) => {
                int index = (int)action.name[5].digit_value ();
                Gtk.TreeModel model = folders.get_model ();
                Gtk.TreeIter iter;
                if (model.iter_nth_child (out iter, null, index - 1)) {
                    folders.set_cursor (model.get_path (iter), null, false);
                    folders.grab_focus ();
                }
            });
            folder_actions.add_action_with_accel (action, "<Alt>%d".printf (i));
            action.set_accel_group (ui.get_accel_group ());
            action.connect_accelerator ();
        }
        ui.insert_action_group (folder_actions, 0);

        shelf = new Gtk.VBox (false, 0);
        add (shelf);
        var menubar = ui.get_widget ("/menubar");
        /* The menubar is nice for globalmenu, but otherwise redundant */
        menubar.set_no_show_all (true);
        menubar.hide ();
        shelf.pack_start (menubar, false, false, 0);
        toolbar = ui.get_widget ("/toolbar") as Gtk.Toolbar;
        /* Do not draw separators in the toolbar */
        foreach (var toolchild in toolbar.get_children ()) {
            if (toolchild is Gtk.SeparatorToolItem)
                (toolchild as Gtk.SeparatorToolItem).draw = false;
        }
        toolbar.show_arrow = false;
        actions.get_action ("MessageNew").is_important = true;
        var toolitem = new Gtk.ToolItem ();
        toolbar.insert (toolitem, -1);
        search = new Elementary.SearchEntry ("Type To Search...");
        search.activate.connect (search_entry_activated);
        search.icon_press.connect_after ((position, event) => {
            search.activate ();
        } );
        toolitem.add (search);
        toolitem.show_all ();
        toolitem = new Gtk.ToggleToolButton.from_stock (Gtk.STOCK_PROPERTIES);
        toolitem.set ("label", _("Menu"));
        actions.get_action ("AppMenu").connect_proxy (toolitem);
        (toolitem as Gtk.ToggleToolButton).clicked.connect ((widget) => {
            action_view (actions.get_action ("AppMenu"));
        });
        toolbar.insert (toolitem, -1);
        shelf.pack_start (toolbar, false, false, 0);

        var hpaned = new Gtk.HPaned ();
        hpaned.name = "SidebarHandleLeft";
        folders = new Postler.Folders (accounts);
        folders.name = "SidebarContent";
        folders.set_size_request (100, 100);
        search.sensitive = false;
        folders.notify["selected-location"].connect ((object, pspec) => {
            search.sensitive = folders.selected_location != null;
            if (search_options.visible)
                GLib.Idle.add (() => {
                    search.activate ();
                    return false;
                });

            if (folders.get_selected_account () == null)
                return;

            string folder = Path.get_basename (folders.selected_location ?? "");
            var action = actions.get_action ("MessageJunk");
            if (folder == folders.get_selected_account ().get_folder (FolderType.JUNK)) {
                action.stock_id = STOCK_MAIL_MARK_NOT_JUNK;
                action.tooltip = _("Mark message as not junk");
            } else {
                action.stock_id = STOCK_MAIL_MARK_JUNK;
                action.tooltip = _("Mark message as junk");
            }
            action = actions.get_action ("MessageDelete");
            if (folder == folders.get_selected_account ().get_folder (FolderType.TRASH)) {
                action.stock_id = Gtk.STOCK_UNDELETE;
                action.tooltip = _("Restore message");
            } else {
                action.stock_id = Gtk.STOCK_DELETE;
                action.tooltip = _("Delete message");
            }
        });
        var folderbox = new Gtk.HBox (false, 4);
        var scrolled = new Postler.ScrolledWindow (folders);
        scrolled.name = "SidebarScrolled";
        folderbox.pack_start (scrolled, true, true, 0);

        progressbar = new Gtk.ProgressBar ();
        statuslabel = new Gtk.Label ("");
        statuslabel.justify = Gtk.Justification.CENTER;
        statuslabel.ellipsize = Pango.EllipsizeMode.END;
        var panebox = new Gtk.VBox (false, 0);
        panebox.border_width = 4;
        panebox.pack_start (folderbox, true, true, 0);
        panebox.pack_end (progressbar, false, false, 0);
        panebox.pack_end (statuslabel, false, false, 0);
        hpaned.pack1 (panebox, false, false); /* don't expand, don't shrink */
        progressbar.set_no_show_all (true);
        statuslabel.set_no_show_all (true);
        client.progress.connect ((account, text, fraction) => {
            if (text != null) {
                statuslabel.label = text;
                progressbar.fraction = fraction;
                if (!statuslabel.visible) {
                    statuslabel.show ();
                    progressbar.show ();
                    actions.get_action ("MailReceive").sensitive = false;
                }
            } else {
                if (statuslabel.visible) {
                    statuslabel.hide ();
                    progressbar.hide ();
                    actions.get_action ("MailReceive").sensitive = true;
                }
            }
        });

        shelf.pack_start (hpaned, true, true, 0);
        var vpaned = new Gtk.VPaned ();
        hpaned.pack2 (vpaned, true, false); /* do expand, don't shrink */
        messages_box = new Gtk.VBox (false, 0);
        vpaned.pack1 (messages_box, false, false); /* no expand, no shrink */
        search_options = ui.get_widget ("/search_options") as Gtk.Toolbar;
        actions.get_action ("SearchSubject").is_important = true;
        actions.get_action ("SearchSender").is_important = true;
        actions.get_action ("SearchRecipient").is_important = true;
        actions.get_action ("SearchBody").is_important = true;
        actions.get_action ("SaveSearch").is_important = true;
        /* Implement mnemonics for search buttons */
        foreach (var button in search_options.get_children ()) {
            if (button is Gtk.ToolButton) {
                var toolbutton = button as Gtk.ToolButton;
                var label = new Gtk.Label.with_mnemonic (toolbutton.label);
                toolbutton.label_widget = label;
            }
        }
        messages_box.pack_start (search_options, false, false, 0);

        client.received.connect (client_received);

        messages = new Postler.Messages (accounts);
        messages.set_size_request (250, 200);
        actions.get_action ("MessageFlag").sensitive = false;
        actions.get_action ("MessageArchive").sensitive = false;
        actions.get_action ("MessageJunk").sensitive = false;
        actions.get_action ("MessageDelete").sensitive = false;
        messages.notify["selected-location"].connect ((object, pspec) => {
            Postler.Messages messages = object as Postler.Messages;
            bool state = messages.selected_location != null;
            actions.get_action ("MessageFlag").sensitive = state;
            string folder = Path.get_basename (folders.selected_location ?? "");
            actions.get_action ("MessageArchive").sensitive = state
                && folder != messages.account_info.get_folder (FolderType.ARCHIVE);
            var action = actions.get_action ("MessageJunk");
            action.sensitive = state
                && messages.account_info.get_folder (FolderType.JUNK) != null;
            action = actions.get_action ("MessageDelete");
            action.sensitive = state && messages.account_info.can_delete ();
        });
        scrolled = new Postler.ScrolledWindow (messages);
        messages_box.pack_end (scrolled, true, true, 0);

        /* Welcome the new user with an account form embedded in the window */
        if (accounts.get_infos ().length () == 0) {
            var welcome = new AccountWidget ();
            var label = new Gtk.Label (null);
            label.set_markup ("<span size=\"x-large\" weight=\"bold\">%s</span>"
                .printf (_("Set up your account.")));
            welcome.pack_start (label, true, false, 4);
            label = new Gtk.Label (
                _("Postler needs some basic information to get your mail."));
            label.sensitive = false;
            welcome.pack_start (label, true, false, 4);
            label = new Gtk.Label (
                "<a href=\"https://mail.google.com/mail/signup\">%s</a>\n".printf (
                _("You don't have an email address yet?")));
            label.use_markup = true;
            label.activate_link.connect ((widget, uri) => {
                return Postler.App.show_uri (get_screen (), uri);
            });
            welcome.pack_start (label, true, false, 4);
            var welcome_box = new Gtk.HBox (false, 0);
            welcome_box.pack_start (welcome, true, false, 0);
            welcome_box.show_all ();
            messages_box.pack_start (welcome_box, true, false, 0);
            messages_box.show ();
            welcome.address.grab_focus ();
            panebox.set_no_show_all (true);
            messages.parent.set_no_show_all (true);
            var continue_box = new Gtk.HButtonBox ();
            continue_box.set_border_width (16);
            continue_box.layout_style = Gtk.ButtonBoxStyle.END;
            var continue_button = new Gtk.Button.with_mnemonic (_("_Continue"));
            continue_button.sensitive = false;
            welcome.address.changed.connect (() => {
                continue_button.sensitive = welcome.address.get_text ().chr (-1, '@') != null;
            });
            continue_button.clicked.connect (() => {
                welcome.apply ();
            });
            continue_box.pack_end (continue_button, false, false, 0);
            messages_box.pack_end (continue_box, false, false, 0);
            welcome.done.connect ((info) => {
                if (welcome.address.get_text ().chr (-1, '@') == null)
                    return;
                welcome_box.hide ();
                continue_box.hide ();
                panebox.set_no_show_all (false);
                panebox.show_all ();
                messages.parent.set_no_show_all (false);
                messages.parent.show_all ();
                accounts.add_info (info);
                client.fetch (info.name);
                folders.populate ();
            });
        }

        content = new Postler.Content ();
        actions.get_action ("MessageReply").sensitive = false;
        actions.get_action ("MessageReplyAll").sensitive = false;
        actions.get_action ("MessageForward").sensitive = false;
        actions.get_action ("ViewSource").sensitive = false;
        messages.notify["selected-location"].connect ((object, pspec) => {
            Postler.Messages messages = object as Postler.Messages;
            bool state = messages.selected_location != null;
            actions.get_action ("MessageReply").sensitive = state;
            actions.get_action ("MessageForward").sensitive = state;
            state = content.reply_to_all != null;
            actions.get_action ("MessageReplyAll").sensitive = state;
        });
        content.notify["reply-to-all"].connect ((object, pspec) => {
            Postler.Content content = object as Postler.Content;
            bool state = content.reply_to_all != null;
            actions.get_action ("MessageReplyAll").sensitive = state;
        });
        content.notify["last-location"].connect ((object, pspec) => {
            Postler.Content content = object as Postler.Content;
            bool state = content.last_location != null;
            actions.get_action ("ViewSource").sensitive = state;
        });

        var viewer = new Postler.Viewer (content);
        viewer.show_all ();
        viewer.set_no_show_all (true);
        viewer.hide ();
        messages.notify["location"].connect ((object, pspec) => {
            viewer.hide ();
        });
        messages.notify["selected-location"].connect ((object, pspec) => {
            if (messages.selected_location != null && !viewer.visible) {
                content.clear ();
                viewer.show ();
                /* Ensure that the selection is visible, in case of resizing */
                Gtk.TreeIter iter = Gtk.TreeIter ();
                GLib.List<Gtk.TreePath> paths;
                paths = messages.get_selection ().get_selected_rows (null);
                var path = paths.nth_data (0);
                if (path != null && messages.model.get_iter (out iter, path)) {
                    GLib.Idle.add (() => {
                        messages.scroll_to_cell (path, null, false, 0, 0);
                        return false;
                    });
                }
            }
        });
        vpaned.pack2 (viewer, false, true);
        vpaned.button_press_event.connect ((widget, event) => {
            /* Double click on handle disables or restores content view */
            if (event.type == Gdk.EventType.2BUTTON_PRESS
             && event.window == vpaned.get_handle_window ()) {
                vpaned.button_release_event (event);
                if (vpaned.position < vpaned.max_position)
                    vpaned.position = vpaned.max_position;
                else
                    vpaned.position = (int)(window_height / 3);
                return true;
            }
            return false;
        });
        shelf.show_all ();
        search_options.hide ();

        folders.set_headers_visible (false);
        messages.set_headers_visible (false);
        messages.set_rules_hint (true);
        vpaned.set_position ((int)(window_height / 3));
        hpaned.set_position ((int)(window_width / 4));

        folders.messages = messages;
        messages.content = content;

        accounts.add_info.connect_after (account_check);
        accounts.remove_info.connect_after (account_check);
        account_check (null);

        /* Never focus hidden widgets: assertion `WIDGET_REALIZED_FOR_EVENT */
        if (folders.visible)
            folders.grab_focus ();
        GLib.Idle.add (() => {
            folders.select_folder (previous_state.open_folder);
            return false;
        });
    }
}


