#include "policyd.h"

/*
 *
 *
 *                           Policy Daemon
 *
 *  policy daemon is used in conjuction with postfix to combat spam.
 *
 *  Copyright (C) 2004 Cami Sardinha (cami@mweb.co.za)
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the  GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the License, or (at your
 *  option) any later version.
 *
 *  This program  is  distributed  in the hope that  it will be useful, but
 *  WITHOUT  WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free  Software Foundation Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 *
 */

/*
 * function: w_select (function_select)
 *  purpose: wrapper for select()
 */             
int     
w_select(int nfds, fd_set *readfds, fd_set *writefds, fd_set *exceptfds, struct timeval *timeout)
{               
  int n;
  if ((n=select(nfds, readfds, writefds, exceptfds, timeout)) < 0)
  {
    logmessage("fatal: select(): %s\n", strerror(errno));
    exit(-1);
  }

  return(n);
}




/* 
 * function: w_accept (wrapped_select)
 *  purpose: wrapped for accept()
 *   return: file descriptor
 */
int
w_accept(unsigned int fd, struct sockaddr *sa, socklen_t *salenptr)
{
  int n;
   
again:
  
  if ((n = accept(fd, sa, salenptr)) < 0)
  {
#ifdef  EPROTO
    if (errno == EPROTO || errno == ECONNABORTED)
    {
#else
    if (errno == ECONNABORTED)
    {
#endif
      if(DEBUG > 2)
        logmessage("DEBUG: fd: %d accept(): %s: retrying\n", fd, strerror(errno));
      
      goto again;
    } else {
      logmessage("fatal: accept(): %s (MAXFDS=%d ENV=%d)\n", strerror(errno), MAXFDS, getdtablesize());
      exit(-1);
    }
  }

  return(n);
}




/*
 * function: w_bind (wrapper_bind)
 *  purpose: wrapper for bind()
 */
void
w_bind(unsigned int fd, const struct sockaddr *sa, socklen_t salen)
{
  u_int yes=1;
  
  /* set options on socket (reuse port/address) */
  if(setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(yes)) < 0)
  {
    logmessage("fatal: setsockopt(): %s\n", strerror(errno));
    exit(-1);
  }
  
  /* bind a name to a socket */
  if (bind(fd, sa, salen) < 0) 
  {
    logmessage("fatal: bind(): %s\n", strerror(errno));
    exit(-1);
  }
}




/*
 * function: w_listen (wrapper_listen)
 *  purpose: wrapper for listen()
 */
void
w_listen(unsigned int fd, unsigned int backlog)
{
  /* listen for connection on a socket */
  if (listen(fd, backlog) < 0)
  {
    logmessage("fatal: listen(): %s\n", strerror(errno));
    exit(-1);
  }
}




/*
 * function: w_read (function_read)
 *  purpose: read number of bytes read
 *   return: number bytes read
 */ 
ssize_t
w_read(unsigned int volatile fd, void *ptr, size_t nbytes)
{       
  ssize_t         n;

  /* catch read timeouts */
  if (sigsetjmp (sjmp, 1))
  { 
    logmessage("warning: read(): read timeout on fd: %d\n", fd);
    return (0); /* close connection */
  }

  /* set the SIGALRM handler */
  signal (SIGALRM, (void *) sigalrm_handler);
  
  /* read timeout */
  alarm(3);
  
  /* read from a file descriptor */
  if ((n = read(fd, ptr, nbytes)) == -1)
  {
    /* reset the timer & unset the signal handler */
    alarm (0);           signal (SIGALRM, SIG_DFL);

    logmessage("warning: read(): %s on fd: %d\n", strerror(errno), fd);
    return (0);                               /* close connection */
  }

  if(DEBUG > 2)
  {
    if(n > 0)
      logmessage("DEBUG: fd: %d read(%d) bytes\n", fd, n);
    else
      logmessage("DEBUG: fd: %d connection closed\n", fd);
  }

  /* reset the timer & unset the signal handler */
  alarm (0);           signal (SIGALRM, SIG_DFL);

  return (n);                                 /* bytes read */
}   




/*
 * function: w_write (wrapper_write)
 *  purpose: Write "n" bytes to a descriptor.
 *   return: bytes written
 */
int
w_write(unsigned int volatile fd, void *ptr, int volatile nbytes)
{

  /* catch write timeouts */
  if (sigsetjmp (sjmp, 1))
  { 
    logmessage("warning: write(): write timeout on fd: %d\n", fd);
    
    return (-1); /* close connection */
  }

  /* set the SIGALRM handler */
  signal (SIGALRM, (void *) sigalrm_handler);
  
  /* read timeout */
  alarm(3);
  
  /* write to a file descriptor */
  if (f_write(fd, ptr, nbytes) != nbytes)
  {
    if(DEBUG > 2)
      logmessage("DEBUG: fd: %d write(): failed, closing connection\n", fd);
    
    /* reset the timer & unset the signal handler */
    alarm (0);           signal (SIGALRM, SIG_DFL);
    return (-1);
  } else {
    if(DEBUG > 2)
      logmessage("DEBUG: fd: %d write(%d) bytes\n", fd, nbytes);
    
    /* reset the timer & unset the signal handler */
    alarm (0);           signal (SIGALRM, SIG_DFL);
    return (nbytes);
  }
}




/*
 * function: f_write (function_write)
 *  purpose: write "n" bytes to a descriptor.
 *   return: bytes written
 */
ssize_t
f_write(unsigned int fd, const void *vptr, size_t n)
{
  size_t     nleft;
  ssize_t    nwritten;
  const char *ptr;

  ptr = vptr;
  nleft = n;
  while (nleft > 0)
  {
    if ((nwritten = write(fd, ptr, nleft)) <= 0)
    {
      if (nwritten < 0 && errno == EINTR)
        nwritten = 0;           /* and call write() again */
      else
        return(-1);                     /* error */
    }

    nleft -= nwritten;
    ptr   += nwritten;
  }
  return(n);
}




/*
 * function: w_close (function_close)
 *  purpose: close a file descriptor
 */
void
w_close(unsigned int fd)
{ 
  if(DEBUG > 2)
    logmessage("DEBUG: fd: %d shutting down fd %d\n", fd, fd);

  /* close a file descriptor */
  if (close(fd) == -1)
  {
    logmessage("close(%d): %s\n", fd, strerror(errno));
  }
}   



/*
 *  function: w_tcp_conn_acl
 *   purpose: check if connecting host is allowed       
 *    return: 0=allow, -1=disallow
 */             
int                     
w_tcp_conn_acl (const char *host)
{
  char            *p, range[strlen(CONN_ACL)];
  unsigned long   ip = inet_addr (host);          
    signed int    xi;
        
  strcpy(range, CONN_ACL);                  /* tmp buffer */
  for( xi=strlen(range) ; xi >= 0 ; xi--)             
  {
    p = range + xi;                         /* cycle backwards */
    if((range[xi] == ' ') || (range[xi] == ',') || (range == p))
    {                                       /* delimiter */
      if(range == p)  p = range + xi;
      else            p = range + xi + 1;
      if((p[strlen(p)-1] == ' ') || (p[strlen(p)-1] == ','))
        p[strlen(p)-1] = 0x00;              /* chomp trailing char */

      if(cidr_ip_match(ip, p) == 1) return (1); /* pass */
      *p = 0x00;                      /* chomp old match */
    }
  }

  return (-1);                                    /* fail */
}



/*
 * function: w_socket (wrapper_socket)
 *  purpose: 
 *   return: file descriptor
 */
int     
w_socket(int family, int type, int protocol)
{
  int n;

  /* create an endpoint for communication */
  if ((n = socket(family, type, protocol)) < 0)
  {
    logmessage("fatal: socket(): %s\n", strerror(errno));
    exit(-1);
  }
  
  return (n);
}



/*
 * function: w_socket (wrapper_socket)
 *  purpose: 
 *   return: file descriptor
 */
const char *
w_inet_ntop(int family, const void *addrptr, char *strptr, size_t len)
{
  const char      *ptr;

  if (strptr == NULL)             /* check for old code */
  {
    logmessage("fatal: NULL 3rd argument to w_inet_ntop");
    exit(-1);
  }
  
  if ((ptr=inet_ntop(family, addrptr, strptr, len)) == NULL)
  {
    logmessage("fatal: inet_ntop(): %s\n", strerror(errno));
    exit(-1);
  }
  
  return(ptr);
}




/*
 * function: w_fork (wrapper_fork)
 *  purpose: background process
 *   return: pid
 */
pid_t
w_fork(void)
{
  pid_t pid;

  if ((pid = fork()) == -1)
  {
    logmessage("fatal: fork(): %s\n", strerror(errno));
    exit(-1);
  }

  return (pid);
}




/*
 *  function: daemon
 *   purpose: backgroup processes
 *    return: status
 */
int
daemonize(int nochdir, int noclose)
{
  unsigned int i;
  pid_t pid;

  if((pid=w_fork()) < 0)
    return (-1);
  else if(pid)
    _exit(0);                   /* parent terminates */

  /* child 1 continues */
  if(setsid() < 0)              /* become session leader */
    return (-1);

  if((pid=w_fork()) < 0)
    return (-1);
  else if(pid)
    _exit(0);                   /* child 1 terminates */

  /* child 2 continues */
  if(nochdir)
    chdir("/");                 /* change working directory */

  /* close off all file descriptors */
  if(noclose)
    for(i=0;i<64;i++)
      close(i);

  /* redirect stdin, stdout and stderr to /dev/null */
  open("/dev/null", O_RDONLY);
  open("/dev/null", O_RDWR);
  open("/dev/null", O_RDWR);

  return (0);
}

/* EOF */
