/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "polyxmass-ui-seqed-widget.h"
#include "polyxmass-pixbuf-rendering.h"
#include "polyxmass-ui-seqed-widget-kbd.h"
#include "polyxmass-ui-sequence-purification.h"
#include "polyxmass-ui-seqed-widget-context-menu.h"
#include "polyxmass-ui-seqed-widget-self-read.h"


enum
  {
    SEQUENCE_MODIFIED,
    SEQUENCE_NOT_MODIFIED,
    WHOLE_SEQ_MASS_UPDATE_REQUIRED,
    SELEC_SEQ_MASS_UPDATE_REQUIRED,
    SEQ_LEFT_END_MODIF_UPDATE_REQUIRED,
    SEQ_RIGHT_END_MODIF_UPDATE_REQUIRED,
    LAST_SIGNAL
  };






static void
polyxmass_seqed_widget_class_init (PxmSeqedWidgetClass *klass);

static void
polyxmass_seqed_widget_init (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_finalize (GObject *object);


static void 
polyxmass_seqed_widget_sequence_modified (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_sequence_not_modified (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_whole_seq_mass_update_required (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_selec_seq_mass_update_required (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_seq_left_end_modif_update_required (PxmSeqedWidget *seqed_widget);

static void
polyxmass_seqed_widget_seq_right_end_modif_update_required (PxmSeqedWidget *seqed_widget);


static GObjectClass *parent_class = NULL;
static guint seqed_widget_signals [LAST_SIGNAL] = {0};



GType
polyxmass_seqed_widget_get_type (void)
{
  static GType pxm_seqed_widget_type = 0;
  
  if (pxm_seqed_widget_type == 0)
    {
      static const GTypeInfo pxm_seqed_widget_info =
	{
	  sizeof (PxmSeqedWidgetClass),
	  NULL,		/* base_init */
	  NULL,		/* base_finalize */
	  (GClassInitFunc) polyxmass_seqed_widget_class_init,
	  NULL,		/* class_finalize */
	  NULL,		/* class_data */
	  sizeof (PxmSeqedWidget),
	  0,		/* n_preallocs */
	  (GInstanceInitFunc) polyxmass_seqed_widget_init,
	};

      pxm_seqed_widget_type = 
	g_type_register_static (GTK_TYPE_VBOX, "PxmSeqedWidget",
				&pxm_seqed_widget_info, 0);
    }
  
  return pxm_seqed_widget_type;
}

static void
polyxmass_seqed_widget_class_init (PxmSeqedWidgetClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = G_OBJECT_CLASS (klass);
  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->finalize = polyxmass_seqed_widget_finalize;
  
  klass->sequence_modified = polyxmass_seqed_widget_sequence_modified;

  klass->sequence_not_modified = polyxmass_seqed_widget_sequence_not_modified;

  klass->whole_seq_mass_update_required = 
    polyxmass_seqed_widget_whole_seq_mass_update_required;
  
  klass->selec_seq_mass_update_required = 
    polyxmass_seqed_widget_selec_seq_mass_update_required;
  
  klass->seq_left_end_modif_update_required = 
    polyxmass_seqed_widget_seq_left_end_modif_update_required;
  
  klass->seq_right_end_modif_update_required = 
    polyxmass_seqed_widget_seq_right_end_modif_update_required;
  
  
  seqed_widget_signals[SEQUENCE_MODIFIED] =
    g_signal_new ("seq_modified",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, sequence_modified),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  seqed_widget_signals[SEQUENCE_NOT_MODIFIED] =
    g_signal_new ("seq_not_modified",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, sequence_not_modified),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  seqed_widget_signals[WHOLE_SEQ_MASS_UPDATE_REQUIRED] =
    g_signal_new ("whole_seq_mass_update_required",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, whole_seq_mass_update_required),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED] =
    g_signal_new ("selec_seq_mass_update_required",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, selec_seq_mass_update_required),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  seqed_widget_signals[SEQ_LEFT_END_MODIF_UPDATE_REQUIRED] =
    g_signal_new ("seq_left_end_modif_update_required",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, seq_left_end_modif_update_required),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  seqed_widget_signals[SEQ_RIGHT_END_MODIF_UPDATE_REQUIRED] =
    g_signal_new ("seq_right_end_modif_update_required",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (PxmSeqedWidgetClass, seq_right_end_modif_update_required),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
}

  

/*
****************************************************************
******************** WIDGET CLASS FUNCTIONS ********************
****************************************************************
*/

static void
polyxmass_seqed_widget_init (PxmSeqedWidget *seqed_widget)
{
  GtkWidget *code_entries_hbox = NULL;
  GtkWidget *pos_scale_hbox = NULL;
   
  GtkWidget *elab_code_frame = NULL;
  GtkWidget *error_code_frame = NULL;
   
  GtkWidget *widget = NULL;
   
  /*
  We have to initialize the composite widget that will comprise :
   
  - a vertical box where all the widgets are packed: this vertical
  box is going to be the container of all the widgets we're going
  to allocate (vbox)
   
  - a hscale widget to indicate what the size of the monicons shoud
  be along with a label widget where the position of the cursor
  is indicated.
   
  - a scroll window where the canvas will be able to scroll (sw)
   
  - a GnomeCanvas where the monomer icons will be displayed to
  represent the sequence being worked on (canvas)

  - a label widget where the position of the currently pointed
  monomer icon is displayed (monomer_pos_label)

  - an entry widget where the currently elaborating code is
  displayed. That's interesting when codes are of more than one
  letter length (elab_code_entry)

  - an entry widget where error messages are displayed if the code
  being elaborated is already konown to be erroneous
  (error_code_entry)

  */

  /* The scale and position label widgets in a hbox.
   */

  /* We need to set this variable, as we'll need to initialize a number of data below.
   */
  seqed_widget->monicon_size = MONICON_SIZE;

  pos_scale_hbox = gtk_hbox_new (FALSE, 0);
  g_assert (pos_scale_hbox != NULL);
  g_object_ref (G_OBJECT (pos_scale_hbox));

  gtk_box_pack_start (GTK_BOX (seqed_widget),
		      pos_scale_hbox,
		      FALSE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  
  
  widget = gtk_frame_new (_("Current Cursor Position"));
  g_assert (widget != NULL);
  g_object_ref (G_OBJECT (widget));
  
  gtk_frame_set_shadow_type (GTK_FRAME (widget),
			     GTK_SHADOW_NONE);
  gtk_frame_set_label_align (GTK_FRAME (widget), 0.5, 0.5);
  
  seqed_widget->monomer_pos_label = gtk_label_new (" label  ");
  g_assert (seqed_widget->monomer_pos_label != NULL);
  g_object_ref (G_OBJECT (seqed_widget->monomer_pos_label));
  
  gtk_container_add (GTK_CONTAINER (widget),
		     seqed_widget->monomer_pos_label);

  gtk_box_pack_start (GTK_BOX (pos_scale_hbox),
		      widget,
		      TRUE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  

  widget = gtk_frame_new (_("Monomer Icon Size (Pixels)"));
  g_assert (widget != NULL);
  g_object_ref (G_OBJECT (widget));

  gtk_frame_set_shadow_type (GTK_FRAME (widget),
			     GTK_SHADOW_NONE);
  gtk_frame_set_label_align (GTK_FRAME (widget), 0.5, 0.5);
  
  seqed_widget->monicon_size_scale = 
    gtk_hscale_new_with_range ((gdouble) 10,
			       (gdouble) 200,
			       (gdouble) 1);
  g_assert (seqed_widget->monicon_size_scale != NULL);
  g_object_ref (G_OBJECT (seqed_widget->monicon_size_scale));

  gtk_scale_set_digits (GTK_SCALE (seqed_widget->monicon_size_scale), 
			0);
  
  gtk_scale_set_value_pos (GTK_SCALE (seqed_widget->monicon_size_scale), 
			   GTK_POS_LEFT);

  gtk_container_add (GTK_CONTAINER (widget),
		     seqed_widget->monicon_size_scale);
  

  gtk_box_pack_start (GTK_BOX (pos_scale_hbox),
		      widget,
		      TRUE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  
  /* Set the hscale widget to the value of the size of the monicons.
   */
  gtk_range_set_value (GTK_RANGE (seqed_widget->monicon_size_scale), 
		       (gdouble) seqed_widget->monicon_size);
  
  g_signal_connect (G_OBJECT (seqed_widget->monicon_size_scale),
		    "value-changed",
		    G_CALLBACK 
		    (polyxmass_seqed_widget_monicon_size_value_changed),
		    GTK_WIDGET (seqed_widget));
  

  
  /* Next we have to do is create the scrolled window where the
     sequence canvas is going to be displayed.
  */
  seqed_widget->sw = gtk_scrolled_window_new (NULL, NULL);
  g_assert (seqed_widget->sw != NULL);
  g_object_ref (G_OBJECT (seqed_widget->sw));

  gtk_widget_show (seqed_widget->sw);
  gtk_box_pack_start (GTK_BOX (seqed_widget), seqed_widget->sw, TRUE, TRUE, 3);
  
  /*  And right away the canvas that we'll pack in it !
   */
  seqed_widget->canvas = gnome_canvas_new ();
  g_assert (seqed_widget->canvas != NULL);
  g_object_ref (G_OBJECT (seqed_widget->canvas));

  gtk_widget_show (seqed_widget->canvas);
  gtk_container_add (GTK_CONTAINER (seqed_widget->sw), seqed_widget->canvas);
  gnome_canvas_set_scroll_region (GNOME_CANVAS (seqed_widget->canvas), 0, 0, 100, 100);


  /* Now do some configuration tweaking on the different widgets above.
   */
  
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (seqed_widget->sw),
				  GTK_POLICY_ALWAYS, 
				  GTK_POLICY_ALWAYS);
  
  /* Because we have to know at each instant what is the size of the
     scroll window into which the canvas displays the monicons (so
     that we can know where each monicon sits), we have to connect the
     size_allocate signal, which is emitted each time the window
     changes its geometry.
  */
  g_signal_connect (G_OBJECT (seqed_widget->sw), 
		    "size_allocate",
		    G_CALLBACK (polyxmass_seqed_widget_sw_size_allocate), 
		    GTK_WIDGET (seqed_widget));

  /* After the window has been resized, it is necessary to check that
     the cursor and (if any) the selection be visible in the
     seqed_widget. We cannot do all this stuff in
     polyxmass_seqed_widget_sw_size_allocate (), nor in
     polyxmass_seqed_widget_event () because in these functions the new
     size of the window is not known. So we have to deal with these
     calculations through a call to a function that is systematically
     called after the first handlers.
  */
  g_signal_connect_after (G_OBJECT (seqed_widget->sw),
			  "size_allocate", 
			  G_CALLBACK 
			  (polyxmass_seqed_widget_ensure_selection_and_cursor_visible), 
			  GTK_WIDGET (seqed_widget));
  

  seqed_widget->sw_width = SW_WIDTH;
  seqed_widget->sw_height = SW_HEIGHT;

  /* Since we now have the canvas, we can create the root group of
     that canvas, which is where the monomer icons are going to be displayed.
  */
  seqed_widget->canvas_main_group =
    gnome_canvas_root (GNOME_CANVAS (seqed_widget->canvas));
  
  /* Set the left margin of the polymer sequence display inside the
   * scroll window, so that there is enough space for the labels
   *indicating the position of the first monicon in the row.
   */
  seqed_widget->left_margin = LEFT_MARGIN;

  gnome_canvas_set_scroll_region (GNOME_CANVAS (seqed_widget->canvas),
				  0, 0, 0, 0);
  



  /* Also we want that when the cursor flies upon the canvas, the canvas be
     able to get the focus, so that keyboard key strokes can successfully be
     detected and processed. 
  */
  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (seqed_widget->canvas),
			GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (seqed_widget->canvas),
			GTK_CAN_FOCUS);



  /* The great many signals that we want to handle are relative to 
   * the canvas, so set a handler for the most general event category:
   * "event".
   */
  g_signal_connect (G_OBJECT (seqed_widget->canvas), "event",
		    G_CALLBACK (polyxmass_seqed_widget_canvas_event),
		    seqed_widget); 

  /* Editing of the sequence, the keyboard handling here goes
   * directly to the canvas
   */
  g_signal_connect (G_OBJECT (seqed_widget),
		      "key-press-event",
		      G_CALLBACK (polyxmass_seqed_widget_kbd_key_press_event),
		      seqed_widget);
		      
  g_signal_connect (G_OBJECT (seqed_widget),
		      "key-release-event",
		      G_CALLBACK (polyxmass_seqed_widget_kbd_key_release_event), 
		      seqed_widget);

  /* We have to tell GTK+ that we want to get these messages :
   */
  gtk_widget_add_events (seqed_widget->canvas,
			 GDK_KEY_RELEASE_MASK);
  

  /* Now that the canvas is correctly set up, we can prepare the
     contextual menu that will popup if the user right-clicks in it.
   */
  widget = polyxmass_seqed_widget_context_menu_setup (GTK_WIDGET (seqed_widget));
  g_assert (widget != NULL);
  g_object_ref (G_OBJECT (widget));

  g_object_set_data (G_OBJECT (PXM_SEQED_WIDGET (seqed_widget)->canvas), 
		     "canvas_menu", widget);
  

  /* We must create the cursor that will be in use in the widget !
   */
  seqed_widget->cursor = polyxmass_monicon_new ();
  
  /* When an edition widget is made, there is no current selection.
   */
  seqed_widget->sel_mnm_idx1 = -1;
  seqed_widget->sel_mnm_idx2 = -1;


  /* And now go on with the two widgets where the elaborating code is
     displayed and where errors might be displayed. Note that we'll
     pack them into GtkFrame widgets, that we'll pack into a GtkHBox...
  */
  code_entries_hbox = gtk_hbox_new (FALSE, 0);
  g_assert (code_entries_hbox != NULL);
  g_object_ref (G_OBJECT (code_entries_hbox));

  gtk_box_pack_start (GTK_BOX (seqed_widget),
		      code_entries_hbox,
		      FALSE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  
  elab_code_frame = gtk_frame_new (_("Currently Edited Code"));
  g_assert (elab_code_frame != NULL);
  g_object_ref (G_OBJECT (elab_code_frame));
  
  gtk_frame_set_shadow_type (GTK_FRAME (elab_code_frame),
			     GTK_SHADOW_NONE);
  gtk_frame_set_label_align (GTK_FRAME (elab_code_frame), 0.5, 0.5);  
  gtk_box_pack_start (GTK_BOX (code_entries_hbox),
		      elab_code_frame,
		      TRUE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  
  error_code_frame = gtk_frame_new (_("Code-Related Error Messages"));
  g_assert (error_code_frame != NULL);
  g_object_ref (G_OBJECT (error_code_frame));

  gtk_frame_set_shadow_type (GTK_FRAME (error_code_frame),
			     GTK_SHADOW_NONE);
  gtk_frame_set_label_align (GTK_FRAME (error_code_frame), 0.5, 0.5);
  gtk_box_pack_start (GTK_BOX (code_entries_hbox),
		      error_code_frame,
		      TRUE /* gboolean expand*/,
		      TRUE /*gboolean fill*/,
		      3 /*guint padding*/);
  
  seqed_widget->elab_code_entry = gtk_entry_new ();
  g_assert (seqed_widget->elab_code_entry != NULL);
  g_object_ref (G_OBJECT (seqed_widget->elab_code_entry));

  gtk_entry_set_has_frame (GTK_ENTRY (seqed_widget->elab_code_entry),
			   FALSE);
  gtk_editable_set_editable (GTK_EDITABLE (seqed_widget->elab_code_entry),
			     FALSE);  
  /* Finally add the entry to its containing frame.
   */
  gtk_container_add (GTK_CONTAINER (elab_code_frame), 
		     seqed_widget->elab_code_entry);
  
  seqed_widget->error_code_entry = gtk_entry_new ();
  g_assert (seqed_widget->error_code_entry != NULL);
  g_object_ref (G_OBJECT (seqed_widget->error_code_entry));

  gtk_entry_set_has_frame (GTK_ENTRY (seqed_widget->
				      error_code_entry),
			   FALSE);
  gtk_editable_set_editable (GTK_EDITABLE (seqed_widget->
					   error_code_entry),
			     FALSE);  
  
  /* Finally add the entry to its containing frame.
   */
  gtk_container_add (GTK_CONTAINER (error_code_frame), 
		     seqed_widget->error_code_entry);
  

  seqed_widget->moniconGPA = g_ptr_array_new ();
  seqed_widget->labelGPA = g_ptr_array_new ();
  seqed_widget->visualGPA = g_ptr_array_new ();
  seqed_widget->propGPA = g_ptr_array_new ();

  /* Initialize the GData keyed datalist for the monomer pixbufs.
   */
  g_datalist_init (&seqed_widget->pixbufGData);

  seqed_widget->monomer_code_completions_wnd = NULL;
  
  gtk_widget_show_all (GTK_WIDGET (seqed_widget));
}




gboolean
polyxmass_seqed_widget_post_pack_constructor (PxmSeqedWidget *seqed_widget)
{

  GdkColor green_color;
  GdkColor red_color;
  GdkColormap *colormap;

  gboolean success = FALSE;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;


  g_assert (seqed_widget != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  


  /* The error code is displayed in its related GtkEntry in red color.
   */
  red_color.red = 55000;
  red_color.blue = 10000;
  red_color.green = 5000;

  colormap = gdk_drawable_get_colormap (seqed_widget->
					error_code_entry->window);
  gdk_colormap_alloc_colors (colormap,
			     &red_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);
  
  gtk_widget_modify_text (GTK_WIDGET (seqed_widget->
				      error_code_entry),
			  GTK_STATE_NORMAL,
			  &red_color);

  
  /* The elaborating code is displayed in its related GtkEntry in
     green color.
   */
  green_color.red = 5000;
  green_color.blue = 10000;
  green_color.green = 35000;

  colormap = gdk_drawable_get_colormap (seqed_widget->
					elab_code_entry->window);
  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);
  
  gtk_widget_modify_text (GTK_WIDGET (seqed_widget->
				      elab_code_entry),
			  GTK_STATE_NORMAL,
			  &green_color);


  /* Since we know the polymer definition context for which the current
     window, and thus the seqed_widget are to be setup, we can now allocate
     the gchar slots for the codes.
  */
  seqed_widget->elab_code = 
    g_malloc0 (sizeof (gchar) * (1 + polchemdef->codelen));
  
  seqed_widget->eval_code = 
    g_malloc0 (sizeof (gchar) * (1 + polchemdef->codelen));

  /* Now that the window has its dimensions, we can set the vertical
   * and horizontal GtkAdjustments to their holders. This way we'll
   * gain access to these very easily throughout the program.
   */
  seqed_widget->sw_vadjust = 
    gtk_scrolled_window_get_vadjustment 
    (GTK_SCROLLED_WINDOW (seqed_widget->sw));

  /* Set the increment to the size of the monicons.
   */
  seqed_widget->sw_vadjust->step_increment = 
    seqed_widget->monicon_size;
  
  seqed_widget->sw_hadjust = 
    gtk_scrolled_window_get_hadjustment 
    (GTK_SCROLLED_WINDOW (seqed_widget->sw));

  /* Set the increment to the size of the monicons.
   */
  seqed_widget->sw_hadjust->step_increment = 
    seqed_widget->monicon_size;

  
  gnome_canvas_set_scroll_region (GNOME_CANVAS (seqed_widget->
						canvas), 0, 0,
				  SW_WIDTH, SW_HEIGHT);
  
  
  /* At this point we know we can render the polymer sequence in the
     seqed_widget for the first time. This is why 'FALSE' is passed as
     the second argument to the functions below: to tell this call is
     not to re-render, but to render a first time.
  */
  if (-1 == polyxmass_seqed_widget_render_polseq_monicons (GTK_WIDGET (seqed_widget),
							   FALSE))
    {
      g_warning (_("%s@%d: failed to render monicons\n"),
		 __FILE__, __LINE__);
      
      return -1;
    }
      
  /* Rendering of the cursor cannot fail, or an error is issued with
   * following stop of the program.
   */
  polyxmass_pixbuf_rendering_cursor_monicon_render (GTK_WIDGET (seqed_widget), FALSE);

  /* We are drawing the sequence for the very first time; we have to
     initialize the seqed_widget->selection_rect.x1 and
     seqed_widget->selection_rect.y1 to sane values, so that they
     correspond to the position of the cursor which is draw at
     position 0 the first time it is drawn.

     This is especially important, because if the user opens a
     sequence. The cursor is drawn left of the very first
     monomer. Now, the user clicks out of the sequence rendering area,
     which is that the mouse event handling function does not
     acknowledge that click. Thus, the sequence cursor (or point in
     Emacs terminology) does not change in any way. Now the user drags
     the mouse (button still depressed) to the sequence rendering
     area. The mouse event handling functions do take into account
     that movement. And if the user releases the mouse button, the
     selection polygon is going to be drawn. But if the location of
     the first point of the selection is not defined, CRASH. Thus this
     small calculation below. The coordinates that we ask are for the
     two '*' positions here (north west and south west):

     *----+
     |    |
     |    |
     *----+
     
  */
  polyxmass_seqed_widget_monicon_get_pixel_coord 
    (0, 
     GTK_WIDGET (seqed_widget), 
     &(seqed_widget->selection_rect), 
     COORDSYS_NW, COORDSYS_SW);
  


  return TRUE;
}



GtkWidget *
polyxmass_seqed_widget_new (void)
{
  GtkWidget *new_widget = NULL;
  
  new_widget = GTK_WIDGET (g_object_new (PXM_TYPE_SEQED_WIDGET, NULL));

  return new_widget;
}


static void
polyxmass_seqed_widget_finalize (GObject *object)
{
  PxmSeqedWidget *widget = PXM_SEQED_WIDGET (object);


  GnomeCanvasItem *label = NULL;
  
  /*
    debug_printf (("BEGIN finalizing the seqed_widget %p\n", widget));
  */
  
  /* Start with internal data:
   */
  if (widget->visualGPA != NULL)
    libpolyxmass_prop_GPA_free (widget->visualGPA);
  

  if (widget->propGPA != NULL)
    libpolyxmass_prop_GPA_free (widget->propGPA);
  
  /* Remember that we might have a completions' listview opened
     right now; so remove it...
  */
  if (widget->monomer_code_completions_wnd != NULL)
    {
      if (GTK_IS_WINDOW (widget->monomer_code_completions_wnd))
	{
	  gtk_object_destroy (GTK_OBJECT (widget->monomer_code_completions_wnd));
	}
    }

  /* Each monicon here, has a canvas_item in it, which in turn has a
     pixbuf for which it was referenced once upon creation of the
     canvas_item. Thus, freeing all the monicons of the array should
     unref the pixbufs in the widget->pixbufGData as many times as
     they were used to create all the monicons of the polymer sequence
     displayed in the sequence. If no pixbuf is used in other places
     (like the completions list window or the monomeric composition
     list window, each pixbuf of the widget->pixbufGData should have a
     single reference count (the one it had when it was created anew).

     Thus, the call to g_datalist_foreach(), below, will bring that
     reference count to 0, entirely freeing the pixbufs still present
     in it.
   */
  if (widget->moniconGPA != NULL)
    polyxmass_monicon_GPA_free (widget->moniconGPA);
  

  /* Clear the pixbfGData GData list. BUT before doing this we have to
     unref one last time all the GdkPixbuf objects that were made
     during the loading/editing of the polymer sequence.
  */
  g_datalist_foreach (&(widget->pixbufGData),
		      polyxmass_pixbuf_rendering_unref_last_pixbuf_from_gdata,
		      NULL);
  
  g_datalist_clear (&(widget->pixbufGData));
  
  /*
    debug_printf (("done clearing the widget->pixbufGData\n"));
  */

  if (widget->labelGPA != NULL)
    {
      while (widget->labelGPA->len > 0)
	{
	  label = g_ptr_array_index (widget->labelGPA, 0);
	  g_ptr_array_remove_fast (widget->labelGPA, label);
	  
	  /*
	    We do not need to gtk_object_destroy() the label as it was
	    created as a floating widget without reffing it
	    explicitely at construction.
	    
	    gtk_object_destroy (GTK_OBJECT (label));
	  */
	}

      g_ptr_array_free (widget->labelGPA, TRUE);
    }
    
  if (widget->cursor != NULL)
    polyxmass_monicon_free (widget->cursor);
  

  if (widget->elab_code != NULL)
    g_free (widget->elab_code);
  if (widget->eval_code != NULL)
    g_free (widget->eval_code);
  
  G_OBJECT_CLASS (parent_class)->finalize (object);

  /*
  debug_printf ((" END  finalizing the seqed_widget %p\n\n", widget));
  */
}


static void
polyxmass_seqed_widget_sequence_modified (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_sequence_modified\n"));
  
}

static void
polyxmass_seqed_widget_sequence_not_modified (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_sequence_not_modified"));
  
}

static void
polyxmass_seqed_widget_whole_seq_mass_update_required (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_whole_seq_mass_update_required"));

}

static void
polyxmass_seqed_widget_selec_seq_mass_update_required (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_selec_seq_mass_update_required"));

}

void
polyxmass_seqed_widget_seq_left_end_modif_update_required  (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_seq_left_end_modif_changed\n"));
  
}

void
polyxmass_seqed_widget_seq_right_end_modif_update_required  (PxmSeqedWidget *seqed_widget)
{
  //  debug_printf (("polyxmass_seqed_widget_seq_right_end_modif_update_required\n"));
  
}



void 
polyxmass_seqed_widget_signal_sequence_modified (PxmSeqedWidget *seqed_widget,
						 gboolean true_false)
{
  if (true_false == TRUE)
    g_signal_emit (G_OBJECT (seqed_widget), 
		     seqed_widget_signals [SEQUENCE_MODIFIED], 0);
  else
    g_signal_emit (G_OBJECT (seqed_widget), 
		   seqed_widget_signals [SEQUENCE_NOT_MODIFIED], 0);
}

void
polyxmass_seqed_widget_signal_mass_update_required (PxmSeqedWidget *seqed_widget, 
						    PxmMasscalcTarget whole_selec)
{
  if (whole_selec & PXM_MASSCALC_TARGET_WHOLE_SEQ)
    g_signal_emit (G_OBJECT (seqed_widget), 
		   seqed_widget_signals [WHOLE_SEQ_MASS_UPDATE_REQUIRED], 0);
  
  if (whole_selec & PXM_MASSCALC_TARGET_SELEC_SEQ)
    g_signal_emit (G_OBJECT (seqed_widget), 
		   seqed_widget_signals [SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
}

void
polyxmass_seqed_widget_signal_seq_left_right_end_modif_update_required (PxmSeqedWidget *seqed_widget,
									PxmEnd left_right)
{
  if (left_right & PXM_END_LEFT)
    g_signal_emit (G_OBJECT (seqed_widget), 
		   seqed_widget_signals [SEQ_LEFT_END_MODIF_UPDATE_REQUIRED], 0);
  
  if (left_right & PXM_END_RIGHT)
    g_signal_emit (G_OBJECT (seqed_widget), 
		   seqed_widget_signals [SEQ_RIGHT_END_MODIF_UPDATE_REQUIRED], 0);
}





/*
****************************************************************
**************** SEQUENCE-MODIFYING FUNCTIONS ******************
****************************************************************
*/
gboolean
polyxmass_seqed_widget_remove_monomer (GtkWidget *widget, gint idx)
{
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;

  gint length = 0;

  /* Attention, this function does not elicit recalculatation of
     masses, it only ensure that a monomer is removed both from the
     polymer sequence monomerGPA monomer instances, and from the
     polymer editing context moniconGPA monicon instances. The caller
     function MUST ensure that the masses are recalculated, if
     necessary (selection and or whole, appropriately).
  */

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);

  length = polymer->monomerGPA->len;

  /* Some easy checks:
   */
  if (length <= 0)
    return TRUE;

  /* Removing a monomer from a polymer involves a number of different
   * steps :
   * 1. check that index is < length;
   * 2. remove/free the monomer object from the GPtrArray of monomer
   * objects of the polymer object.
   * 3. remove/free the monicon object from the GPtrArray of 
   * monicon objects of the seqed_widget object.
   */

  g_assert (idx < length || idx >= 0);
  
  /* Now remove the monomer from the sequence GPtrArray of monomers.
   */
  monomer = g_ptr_array_remove_index (polymer->monomerGPA, idx);
  g_assert (monomer != NULL);
  pxmchem_monomer_free (monomer);


  /* Immediately inform that the sequence has changed.
   */
  g_signal_emit (G_OBJECT (widget), 
		 seqed_widget_signals[SEQUENCE_MODIFIED], 0);
  

  /* Now remove the monicon from the sequence editor's GPtrArray of
   * monicons.
   */
  monicon = 
    g_ptr_array_remove_index (PXM_SEQED_WIDGET (widget)->moniconGPA, idx);
  
  g_assert (monicon != NULL);
  
  polyxmass_monicon_free (monicon);
  
  return TRUE;
}


gint
polyxmass_seqed_widget_remove_selected_oligomer (GtkWidget *widget)
{
  gint start = -1;
  gint end = -1;

  gboolean result = FALSE;
  
  PxmPolymer *polymer = NULL;


  g_assert (widget != NULL);

  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  

  /* It is absolutely required, if we get to this place, that there is
     a selection. This is necessary to ensure the internal consistency
     of the program. Thus the g_assert () call below.
  */
  result = 
    polyxmass_seqed_widget_get_selection_indices (widget, &start, &end);
  
  g_assert (result == TRUE);
  
  /* As usual it must be noted that when selecting a portion of the
     polymer sequence, the indexes that are returned are OK for the
     start index but BAD for the end index. Indeed, this latter index
     is one unit larger (behaving like a position and not like an
     index. So we have to decrement by one in order to get a true
     index.
   */
  if (end > 0)
    end--;

  /* Now everything should work OK. The function below returns the
     location of the cursor after the operation has taken place.
   */  
  return polyxmass_seqed_widget_remove_sequence_indices (widget,
							 start, end);
}


gint
polyxmass_seqed_widget_remove_sequence_indices (GtkWidget *widget,
						gint start, gint end)
{
  gint iter = 0;
  gint length = 0;
  gint first = 0;
  gint last = 0;
  
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;
  
  /* This function returns the last_point_1_idx value.
   */

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  

  length = polymer->monomerGPA->len;
  
  /* The start and end indices passed as parameter may not be ordered,
     so that we have to put them in order, and after that is done
     ensure that their value is compatible with the polymer sequence
     array boundaries. 

     In any way, none of the passed value can be higher than (length -
     1) because we are dealing with indices.

     Note that start and end may be the same if only one monomer is
     currently selected !!!
  */

  g_assert (start <= length - 1 && start >= 0);
  g_assert (end <= length - 1 && end >= 0 );
  
  /* First step: make sure that start is less than end.
   */
  first = start < end ? start : end;
  last = start < end ? end : start;

  /* Now we are certain that (first <= last).
   */

  /* Second, make sure that we can access the monomers that
     are at the two indices start and end.
  */
  monomer = g_ptr_array_index (polymer->monomerGPA, first);
  g_assert (monomer != NULL);
    
  monomer = g_ptr_array_index (polymer->monomerGPA, last);
  g_assert (monomer != NULL);
  
  
  /* AND NOW: removing monomers from a polymer sequence 
   * involves a number of different steps :
   *
   * 1. remove/free the monomer object from the GPtrArray of monomers
   * of the polymer object.
   *
   * 2. remove/free the monicon object from the GPtrArray of 
   * monicon objects of the seqed_widget object.
   */

  /* Now remove the monomer objects from the sequence's GPtrArray of
   * monomers.
   */
  iter = last;

  while (iter >= first)
    {
      /* First remove the monomer from the polymer sequence's array
       * of monomers.
       */
      monomer = g_ptr_array_remove_index (polymer->monomerGPA, iter);
      
      g_assert (monomer != NULL);

      pxmchem_monomer_free (monomer);

      /* Immediately inform that the sequence has changed.
       */
      g_signal_emit (G_OBJECT (widget), 
		     seqed_widget_signals[SEQUENCE_MODIFIED], 0);


      /* Next, remove the monomer icon from the sequence editor's
       * array of monicons.
       */
      monicon = 
	g_ptr_array_remove_index (PXM_SEQED_WIDGET (widget)->moniconGPA, iter);

      g_assert (monicon != NULL);

      /* polyxmass_monicon_free bothers to call the gtk_object_destroy ()
       * on the moniconCanvasItem, which means the GdkPixbuf
       * is unref'ed right away
       */
      polyxmass_monicon_free (monicon);

      iter--;
    }

  /* And finally we kill the selection polygon, and ask that the cursor
   * be drawn at index 'first'. The function below will set to -1
   * both sel_mnm_idx1 and sel_mnm_idx2.
   */
  polyxmass_seqed_widget_remove_selection_polygon (widget);
  
  /* Set the cursor position at the new index:
   */
  PXM_SEQED_WIDGET (widget)->last_point_1_idx = first;
  
  polyxmass_seqed_widget_draw_cursor (widget);
  
  /* Redraw the whole sequence.
   */
  if (- 1 == polyxmass_seqed_widget_redraw_sequence (widget))
    {
      g_critical (_("%s@%d: failed to draw the sequence\n"),
	     __FILE__, __LINE__);
    }
    
  /*
  if (gtk_events_pending())
    gtk_main_iteration();
  */

  /* Return the present index, so that if this function was 
   * called as the first step in a replacement operation, the caller
   * knows at which index it should insert the replacement. This is 
   * particularly interesting if this function was called with 
   * monomer pointers in the p_seqcoords, and not indexes.
   */

  return PXM_SEQED_WIDGET (widget)->last_point_1_idx;
}



gboolean
polyxmass_seqed_widget_integrate_monomer_at_idx (PxmMonomer *monomer,
						 gint idx,
						 GtkWidget *widget,
						 gboolean render_modif)
{
  gint length = 0;

  PxmPolymer *polymer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  PxmMonicon *monicon = NULL;

  PxmProp *prop = NULL;
    
  GdkPixbuf *pixbuf = NULL;


  g_assert (widget != NULL);

  g_assert (idx >= 0);
  

  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  
  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  length = polymer->monomerGPA->len;


  
  /* Attention, this function does not elicit recalculatation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recalculated (selection and or
     whole, appropriately).
  */

  /* First ADD/INSERT the new monomer pointer to its monomerGPA.
   */

  /* -------- ATTENTION --------- that the index might be higher than
     the upper bound of the polymer sequence GPtrArray because we want
     to allow the cursor in the editor to be drawn RIGHT TO THE LAST
     MONOMERICON, which means at an index = max_index_possible + 1
     !!!!!!!!!! IN THIS PRECISE CASE WE DO NOT INSERT A NEW MONOMER
     POINTER, WE ADD IT !!!!
  */

  if (idx >= length)
    g_ptr_array_add (polymer->monomerGPA, monomer);
  else
    g_ptr_array_insert_val (polymer->monomerGPA, idx, monomer);
  
  /* Immediately inform that the sequence has changed.
   */
  g_signal_emit (G_OBJECT (widget), 
		 seqed_widget_signals[SEQUENCE_MODIFIED], 0);

  
  /* Now that we have this new monomer in the sequence we should
     integrate it into the moniconGPA also, otherwise the colinearity
     between the polymer->monomerGPA and the moniconGPA goes dead.
   */

  /* Render (or get a reference to an existing one) a pixbuf for the
     monomer's code.
   */
  if (FALSE == render_modif)
    {
      pixbuf = 
	polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
						    PXM_SEQED_WIDGET (widget)->monicon_size,
						    &(PXM_SEQED_WIDGET (widget)->pixbufGData),
						    polchemdefctxt);
    }
  else
    {
      /* Find out what modif there might be in that monomer.
       */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					  NULL,
					  NULL,
					  "MODIF",
					  NULL,
					  PXM_CMP_NO_DEEP);
      if (prop == NULL)
	{
	  pixbuf = 
	    polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
							PXM_SEQED_WIDGET (widget)->monicon_size,
							&(PXM_SEQED_WIDGET (widget)->pixbufGData),
							polchemdefctxt);
	}
      else
	{
	  pixbuf = 
	    polyxmass_pixbuf_rendering_render_modif (monomer->code,
						     (gchar *) prop->data, /*modif*/
						     PXM_SEQED_WIDGET (widget)->monicon_size,
						     &(PXM_SEQED_WIDGET (widget)->pixbufGData),
						     polchemdefctxt);
	}
    }
  
  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed to render monicon for monomer: '%s' "
		    "at index: '%d'\n"),
		  __FILE__, __LINE__, monomer->code, idx);
      
      return FALSE;
    }
  
  /* Allocate a new monicon.
   */
  monicon = polyxmass_monicon_new ();
  
  if (FALSE == 
      polyxmass_pixbuf_rendering_monicon_make_canvas_item (monicon, 
							   PXM_SEQED_WIDGET (widget)->monicon_size,
							   pixbuf,
							   idx,
							   widget))
    {
      g_critical (_("%s@%d: failed to make canvas_item for monomer code: '%s' "
	       "at index: '%d'\n"),
	     __FILE__, __LINE__, monomer->code, idx);
      
      polyxmass_monicon_free (monicon);

      return FALSE;
    }

  /* At this point, the pixbuf that we got in the first place, with
     the call to polyxmass_pixbuf_rendering_render_no_modif() got
     referenced by the gnome_canvas_item_new() that makes the canvas
     item in the polyxmass_pixbuf_rendering_monicon_make_canvas_item()
     call. Thus, when we call the pxm_monicon_free() function on the
     monicon, it should just 

     gtk_object_destroy(GTK_OBJECT(monicon->canvas_item)); 

     which will unref the pixbuf contained in it.

     So, the story goes : 

     1. we get a pixbuf (either pre-existing or allocated anew) with
        no reference count incrementing.
	
     2. we construct a canvas_item by using it, which increments its
        reference count by 1.

     3. we put that pixbuf into a monicon object that will contain it
        up to its freeing.

     4. some day we free the monicon, which will gtk_object_destroy
        the canvas_item, which will decrement the reference count by
        1. At this stage we get a reference count identical to the
        situation in point 1.
  */



  /* At this point we have a fully qualified monicon.
   *   
   * -------- ATTENTION ---------    SAME AS ABOVE
   */
  if (idx >= length)
    g_ptr_array_add (PXM_SEQED_WIDGET (widget)->moniconGPA, monicon);
  else
    g_ptr_array_insert_val (PXM_SEQED_WIDGET (widget)->moniconGPA, idx, 
			    monicon);

  polyxmass_seqed_widget_ensure_selection_and_cursor_visible
    (PXM_SEQED_WIDGET (widget)->sw, NULL, widget);
  
  return TRUE;
}


gboolean
polyxmass_seqed_widget_integrate_monomer_at_point (PxmMonomer *monomer,
						   GtkWidget *widget,
						   gboolean render_modif)
{

  g_assert (monomer != NULL);
  
  g_assert (widget != NULL);
  

  /* Attention, this function does not elicit recalculatation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recalculated (selection and or
     whole, appropriately).
  */
 
  /* All we want is to insert/add the monomer at the current cursor
     point location:
  */
  if (FALSE == 
      polyxmass_seqed_widget_integrate_monomer_at_idx (monomer,
						       PXM_SEQED_WIDGET (widget)->
						       last_point_1_idx,
						       widget,
						       render_modif))
    {
      g_critical (_("%s@%d: failed to integrate monomer: '%s' at point\n"),
		  __FILE__, __LINE__, monomer->code);
      
      return FALSE;
    }
  
  /* Since a monomer was inserted at point, the coordinates of
     that point have to be incremented by one.
  */
  PXM_SEQED_WIDGET (widget)->last_point_1_idx++;

  return TRUE;
}


gint
polyxmass_seqed_widget_integrate_sequence_at_point (GPtrArray *GPA,
						    GtkWidget *widget,
						    gboolean render_modif)
{
  PxmMonomer *monomer = NULL;
  PxmMonomer *monomer_new = NULL;
  
  PxmRect rect;
  
  gint iter = 0;
  

  /* Attention, this function does not elicit recalculatation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recalculated (selection and or
     whole, appropriately).
  */


  /* We get an array of PxmMonomer instances. That array should be
     copied into the polseqctx->polymer->monomerGPA array at the
     precise cursor point.
  */

  g_assert (widget != NULL);
  g_assert (GPA != NULL);


  /* If the array of monomer that we should integrate in the sequence
     is empty, then we return the number of successfully integrated
     monomer: 0!
  */
  if (GPA->len <= 0) return 0;
  

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == 
      polyxmass_seqed_widget_get_selection_indices (widget, NULL, NULL))
    {
      if (-1 == polyxmass_seqed_widget_remove_selected_oligomer (widget))
	{
	  g_critical (_("%s@%d: failed to remove the selected "
			"sequence portion.\n"),
		      __FILE__, __LINE__);
	  
	  return -1;
	}
    }
  

  /* OK, apparently everything is working, we can set each monomer
     object to its relevant place in the polymer sequence.
   */
  for (iter = 0; iter < GPA->len; iter++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      g_assert (monomer != NULL);
  

      /* Make a copy of that monomer, because we do not know if the
	 monomers in the GPA were allocated for us or not. Note that
	 the duplication function is DEEP, i.e. it will duplicate also
	 the array of properties of the source monomer into the
	 destination monomer_new. The function below will make sure
	 that the variable describing the cursor point location is
	 incremented by one each time a new monomer is inserted in the
	 sequence.
      */
      monomer_new = pxmchem_monomer_dup (monomer, PXM_DUP_DEEP);
      
      if (FALSE ==
	  polyxmass_seqed_widget_integrate_monomer_at_point (monomer_new,
							     widget,
							     render_modif))
	{
	  g_critical (_("%s@%d: failed to integrate the sequence element.\n"),
		      __FILE__, __LINE__);
	  
	  pxmchem_monomer_free (monomer_new);
	  
	  return -1;
	}
    }
  
  /* We should make sure that the insertion point is systematically
   * visible, even if we are inserting data on the lowest part of the
   * sequence editor window.
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						  last_point_1_idx,
						  widget,
						  &rect,
						  COORDSYS_NW, COORDSYS_NW);
  
  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
						 canvas),
				   &(PXM_SEQED_WIDGET (widget)->canvas_x_offset),
				   &(PXM_SEQED_WIDGET (widget)->canvas_y_offset));
  
  if (rect.y1 
      >= PXM_SEQED_WIDGET (widget)->canvas_y_offset 
      + PXM_SEQED_WIDGET (widget)->monicon_size 
      + PXM_SEQED_WIDGET (widget)->sw_height
      - - PXM_SEQED_WIDGET (widget)->monicon_size)
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
					    canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset +
			      PXM_SEQED_WIDGET (widget)->monicon_size);
    }
  
  polyxmass_seqed_widget_redraw_sequence (widget);
  
  return iter + 1;
}






/*
****************************************************************
*************    SEQUENCE (RE-) DRAWING FUNCTIONS   ************
****************************************************************
*/
gint
polyxmass_seqed_widget_render_polseq_monicons (GtkWidget *widget,
					       gboolean rerender)
{
  /* We get a pointer to a GTK_WIDGET (PxmSeqedWidget *), and are
     asked that the monicon rendering be performed for each monomer in
     the sequence that is associated to it (the association is
     performed by setting a pointer to a PxmPolymer object in the
     PxmSeqedWidget instance).

     If rerender is TRUE, then the array of monicons should first be
     destroyed, so that the rendering is forced for EACH monomer in
     the sequence. This function returns the number of rendered
     pixbufs.
   
     The way the task is achieved is by running a loop in which (for
     each monomer in seqed_widget->polymer->monomerGPA) three main
     achievements are performed:
   
     1. Allocate a new monicon instance.
    
     2. Render a pixbuf (or just find it in the GData of the
     polchemdefctxt) for the monomer->code.
   
     3. Use the pixbuf from step 1. to create with it a new
     monicon->canvas_item object.
   */
  gint count = 0;
  gint iter = 0;

  PxmProp *prop = NULL;

  PxmPolymer *polymer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmSeqedWidget *seqed_widget = NULL;
  
  
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;

  GdkPixbuf *pixbuf = NULL;
  

  g_assert (widget != NULL);
  seqed_widget = PXM_SEQED_WIDGET (widget);
  

  polymer = seqed_widget->polymer;
  g_assert (polymer != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  if (rerender == TRUE)
    {
      /* If we are here because we have to rerender all the monicons
       * from scratch, typically if the user asked that the dimension
       * of the monicons be changed, then update also the scroll
       * adjustement's steps after having freed the array of monicons
       * instances.
       */

      /* First empty moniconGPA and free all the monicons that were
	 prepared to display the sequence in the editor. When each
	 monicon object in the array will get freed, its contained
	 canvas_item will be gtk_object_destroy'ed with a deccrement
	 of the reference count for its contained pixbuf.

	 Upon full freeing of the seqed_widget->moniconGPA, there
	 should be no more than one reference count for each pixbuf in
	 the seqed_widget->pixbufGData, unless some are used in other
	 places, like in the monomer composition listview. But, this
	 is no problem, because what's important is that the reference
	 associated to monicon objects is removed.

	 Below, another decrement of reference count is asked, that is
	 the last one (if only the sequence is using pixbufs) that
	 belongs to the pixbufs (that is the one that was set upon
	 initial creation). After that g_datalist_foreach() call
	 below, all pixbufs not currently used in other places than
	 the seqed_widget will be freed, as their reference count
	 drops to 0!
      */
      polyxmass_monicon_GPA_empty (seqed_widget->moniconGPA);
      
      /* Clear the pxbfGData GData list. BUT before doing this
       * we have to unref one last time all the GdkPixbuf objects that
       * were made during the loading/editing of the polymer sequence.
       */
      g_datalist_foreach 
	(&(seqed_widget->pixbufGData),
	 polyxmass_pixbuf_rendering_unref_last_pixbuf_from_gdata,
	 NULL);
      
      g_datalist_clear (&(seqed_widget->pixbufGData));

      /* And now we can initialize back the GData keyed datalist.
       */
      g_datalist_init (&(seqed_widget->pixbufGData));
      
      /* Set the increment to the size of the monicons.
       */
      seqed_widget->sw_vadjust->step_increment = 
	seqed_widget->monicon_size;
      seqed_widget->sw_hadjust->step_increment = 
	seqed_widget->monicon_size;
    }
  
  
  for (iter = 0; iter < polymer->monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      /* First see if the monomer is modified. Depending on the result
       * create (or get from the GData in seqed_widget) a pixbuf for it.
       */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "MODIF",
				      NULL,
				      PXM_CMP_NO_DEEP);
      if (prop == NULL)
	pixbuf = 
	  polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
						      seqed_widget->monicon_size,
						      &(seqed_widget->pixbufGData),
						      polchemdefctxt);
      
      else
	pixbuf = 
	  polyxmass_pixbuf_rendering_render_modif (monomer->code,
						   (gchar *) prop->data ,
						   seqed_widget->monicon_size,
						   &(seqed_widget->pixbufGData),
						   polchemdefctxt);
      
      if (pixbuf == NULL)
	{
	  g_critical (_("%s@%d: failed to render monicon for monomer: '%s' "
		   "at index: '%d'\n"),
		 __FILE__, __LINE__, monomer->code, iter);
	  
	  /* Do not free it, because is will be freed later, when 
	     all the process of shutting down the editctxt is carried
	     out !!!
	     polyxmass_monicon_GPA_free (seqed_widget->moniconGPA);
	  */

	  return -1;
	}
      
      /* Apparently we could successfully render (or get a pointer to)
       * a pixbuf for the monomer being iterated. Go on with making a
       * new monicon for the monomer in question and then make the
       * canvas_item.  During the process of creating the canvas_item
       * the pixbuf is going to be reffed once.
       */
      monicon = polyxmass_monicon_new ();

      if (FALSE == 
	  polyxmass_pixbuf_rendering_monicon_make_canvas_item (monicon, 
							       seqed_widget->monicon_size,
							       pixbuf,
							       iter,
							       widget))
	{
	  g_critical (_("%s@%d: failed to make canvas_item "
			"for monomer code: '%s' at index: '%d'\n"),
		      __FILE__, __LINE__, monomer->code, iter);
	  
	  polyxmass_monicon_GPA_free (seqed_widget->moniconGPA);
	  
	  return -1;
	}

      /* At this point, the pixbuf that we got in the first place, with
	 the call to polyxmass_pixbuf_rendering_render_no_modif() got
	 referenced by the gnome_canvas_item_new() that makes the canvas
	 item in the polyxmass_pixbuf_rendering_monicon_make_canvas_item()
	 call. Thus, when we call the pxm_monicon_free() function on the
	 monicon, it should just 

	 gtk_object_destroy(GTK_OBJECT(monicon->canvas_item)); 

	 which will unref the pixbuf contained in it.

	 So, the story goes : 

	 1. we get a pixbuf (either pre-existing or allocated anew) with
	 no reference count incrementing.
	
	 2. we construct a canvas_item by using it, which increments its
	 reference count by 1.

	 3. we put that pixbuf into a monicon object that will contain it
	 up to its freeing.

	 4. some day we free the monicon, which will gtk_object_destroy
	 the canvas_item, which will decrement the reference count by
	 1. At this stage we get a reference count identical to the
	 situation in point 1.
      */




      /* At this point we have a fully qualified monicon.
       */
      g_ptr_array_add (seqed_widget->moniconGPA, monicon);

      /*
	debug_printf (("added monicon %p for monomer %s - "
	"Array is of len %d\n", monicon, monomer->name,
	seqed_widget->moniconGPA->len));
      */

      count++;
    }

  /* Now, our seqed_widget object has a GPtrArray filled with monicon
   * objects faithfully representing the polymer sequence.
   */

  return count ;
}


gint 
polyxmass_seqed_widget_redraw_sequence (GtkWidget *widget)
{
  /* We get new size of the window, which means we can calculate how
   * many icons we can set to the line:
   */
  gint iter = 0;
  gint count = 0;
  
  gdouble scroll_x = 0;
  gdouble scroll_y = 0;

  gchar *label = NULL;

  PxmMonicon *monicon = NULL;
  
  GnomeCanvasItem *label_item = NULL;

  g_assert (widget != NULL);
  g_assert (PXM_IS_SEQED_WIDGET (widget));
  
  
  if (PXM_SEQED_WIDGET (widget)->monicon_size > 0)
    {
      PXM_SEQED_WIDGET (widget)->monicons_per_line = 
	(PXM_SEQED_WIDGET (widget)->sw_width 
	 - PXM_SEQED_WIDGET (widget)->left_margin) 
	/ PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else
    {
      g_error (_("%s@%d: width of monomer icon is not > 0\n"),
	       __FILE__, __LINE__);
      
      return -1 ;
    }
  /*
    printf ("PXM_SEQED_WIDGET (widget)->monicons_per_line = %d\n", 
	  PXM_SEQED_WIDGET (widget)->monicons_per_line);
  */

  /* Empty the array of labels so that we can fill it with newly 
   * made ones.
   */
  g_assert (PXM_SEQED_WIDGET (widget)->labelGPA != NULL);
  
  while (PXM_SEQED_WIDGET (widget)->labelGPA->len)
    {
      label_item = g_ptr_array_index (PXM_SEQED_WIDGET (widget)->labelGPA, iter);
      g_ptr_array_remove_fast (PXM_SEQED_WIDGET (widget)->labelGPA, label_item);
      gtk_object_destroy (GTK_OBJECT (label_item));
    }
  
  g_ptr_array_free (PXM_SEQED_WIDGET (widget)->labelGPA, TRUE);
  PXM_SEQED_WIDGET (widget)->labelGPA = g_ptr_array_new ();


  /* Iter in the array of monicons and move them to their new place.
   */
  for (iter = 0; iter < PXM_SEQED_WIDGET (widget)->moniconGPA->len; iter++)
    {
      monicon = g_ptr_array_index (PXM_SEQED_WIDGET (widget)->moniconGPA, iter);

      polyxmass_seqed_widget_monicon_get_pixel_coord (iter, 
						      widget, 
						      &monicon_rect, 
						      COORDSYS_NW, 
						      COORDSYS_NW);
      
      if (monicon_rect.x1 == PXM_SEQED_WIDGET (widget)->left_margin)
	{
	  /* We are at first position in a row, so we need to 
	   * create and display a label indicating the monomer
	   * current position in red in the margin
	   */
	  label = g_strdup_printf ("%d", iter + 1);

	  label_item =
	    gnome_canvas_item_new (PXM_SEQED_WIDGET (widget)->canvas_main_group,
				   gnome_canvas_text_get_type (),
				   "text", label,
				   "font", "12x24",
				   "x", monicon_rect.x1 -
				   (PXM_SEQED_WIDGET (widget)->left_margin/2),
				   "y", monicon_rect.y1 +
				   (PXM_SEQED_WIDGET (widget)->monicon_size/2),
				   "fill_color", "red",
				   "anchor", GTK_ANCHOR_CENTER,
				   NULL); 
	  
	  g_ptr_array_add (PXM_SEQED_WIDGET (widget)->labelGPA, label_item);
	  
	  g_object_ref (G_OBJECT (label_item));
	  
	  g_free (label);
	}
      
      /* Now that we have calculated the new position, we can ask a move
       * using the x y position that was previously stored in the item.
       */
      gnome_canvas_item_move (monicon->canvas_item, 
			      monicon_rect.x1 - monicon->x,
			      monicon_rect.y1 - monicon->y);

      /* Do not forget to update the x y position of the item now that
       * it has been moved (for next move, of course)!
       */
      monicon->x = monicon_rect.x1;
      monicon->y = monicon_rect.y1;

      count++;
    }

  /* Below, the y+PXM_SEQED_WIDGET (widget)->monicon_size is to take
   * into account the height of the icons, since scroll measurements
   * are done according to anchor point southwest and our measurements
   * are according to anchor point northwest.
   */
  scroll_x = 
    (PXM_SEQED_WIDGET (widget)->monicons_per_line * 
     PXM_SEQED_WIDGET (widget)->monicon_size) 
    + PXM_SEQED_WIDGET (widget)->left_margin;

  scroll_y = monicon_rect.y1 + PXM_SEQED_WIDGET (widget)->monicon_size;

  gnome_canvas_set_scroll_region (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
						canvas), 
				  0, 0, scroll_x, scroll_y);


#if 0

  /* The sequence editor context (aka PXM_SEQED_WIDGET (widget)) has a
   * GPtrArray of very special prop instances: visualGPA. This array
   * contains prop instances which describe the way graphical elements
   * should be drawn onto the sequence canvas items made of all the
   * monicons.
   * 
   * For example, imagine you have searched for a sequence motif 
   * "KA" in the currently active polymer sequence and a full
   * list of found "KA" motifs is displayed to you. You may ask, by 
   * clicking onto an item (or more) of the list, that a visual clue
   * be given to you as to where the "KA" motif is found in the
   * sequence. This may be as simple as one black line at the bottom
   * of the two monicon objects that represent the "KA" sequence motif
   * present in the sequence editor.
   *
   * Now, imagine that you change the size of the seqence editor window.
   * The black line in question will no more be drawn at the proper
   * place, because it is stuck to coordinates which were calculated
   * for another size of the window. What we want here is to find
   * a way that will let the sequence redrawing function recalculate 
   * all graphical elements that need refreshing after the size of 
   * the window has changed or the sequence has been edited.
   *
   * In our example of the found motif, what we have is a 
   * prop named "FINDVISUAL_PTR" and of which the p_data member
   * points to a findvisual structure which holds absolutely all
   * the data required for full redraw of the graphical elements
   * for which it was created in the first place.
   * The strategy here is to do some kind of plugin work. We have 
   * to iterated in the array of prop instances and -for each one-
   * find a plugin function which can handle the redrawing work.
   */

  PxmProp *prop = NULL;
  
  gboolean render_result = FALSE;
  
  for (iter = 0 ; iter < PXM_SEQED_WIDGET (widget)->visualGPA->len ; iter++)
    {
      prop = g_ptr_array_index (PXM_SEQED_WIDGET (widget)->visualGPA, iter);
      g_assert (prop != NULL);


      visual_render_in_seqed_wnd_plugin_func = 
	visual_render_in_seqed_wnd_plugin_chooser (prop->str_name);
      
      if (visual_render_in_seqed_wnd_plugin_func != NULL)
	{
	  render_result = visual_render_in_seqed_wnd_plugin_func (prop);

	  if (render_result != PXM_ERR_NOERR)
	    {
	      mk_strerr (__FILE__, __LINE__);
	      g_critical ("failed to find the rendering plugin for %s\n",
		     prop->str_name);
	    }
	}
      else
	{
	  mk_strerr (__FILE__, __LINE__);
	  g_critical ("the visual rendering property is not registered: %s\n",
		 prop->str_name);
	}
    }

#endif  

  /* Now that the mere polymer sequence has been drawn again, and 
     that all the other visuals are rendered, we
     can reselect the oligomer that was selected before.
  *
  polyxmass_seqed_widget_update_sel_polygon (PXM_SEQED_WIDGET (widget));
  */
  
  /* Draw the cursor.
   */
  polyxmass_seqed_widget_draw_cursor (widget);
  
  return count;
}


void
polyxmass_seqed_widget_draw_cursor (GtkWidget *widget)
{
  /* The position of the cursor is governed by the last
   * pointed monomer index, which is stored in the variable 
   * last_point_1_idx each time something relevant is done
   * (mouse click, keyboard actions....)
   */
  PxmRect rect;

  g_assert (widget != NULL);

  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx < 0)
    PXM_SEQED_WIDGET (widget)->last_point_1_idx = 0;
  
  /* We tolerate, for graphical purposes, that the "index" of the
   * last selectable monomer be > max index of polymer length, but
   * when accessing the polymer data, always check !
   */
  if (PXM_SEQED_WIDGET (widget)->last_point_1_idx >
      PXM_SEQED_WIDGET (widget)->moniconGPA->len)
    {
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = 
	PXM_SEQED_WIDGET (widget)->moniconGPA->len;
    }
  
  polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
					       last_point_1_idx,
					       widget, 
					       &rect,
					       COORDSYS_NW, COORDSYS_NW);
  

  /* We do not want that the cursor be alone at position right of the
   * LEFT_MARGIN because cursor position actually is at end of the
   * line above the one where it would be without the correction
   * below. The exception is when the monicon index is 0, since of
   * course it is possible to set the cursor at left position of the
   * first monicon in the sequence.
   */
  if ((rect.x1 == PXM_SEQED_WIDGET (widget)->left_margin) 
      && (PXM_SEQED_WIDGET (widget)->last_point_1_idx > 0))
    {
      rect.x1 = PXM_SEQED_WIDGET (widget)->left_margin 
	+ (PXM_SEQED_WIDGET (widget)->monicons_per_line 
	   * PXM_SEQED_WIDGET (widget)->monicon_size);
      
      rect.y1 -= PXM_SEQED_WIDGET (widget)->monicon_size;
    } 
  
  gnome_canvas_item_move (PXM_SEQED_WIDGET (widget)->cursor->canvas_item,
			  rect.x1 - PXM_SEQED_WIDGET (widget)->cursor->x,
			  rect.y1 - PXM_SEQED_WIDGET (widget)->cursor->y);
  
  /* When we are redrawing the cursor because we have typed in a new
   * polymer sequence portion the newly monicon's will be on top of
   * the cursor, this is why we ask it to be on top of absolutely
   * everything !!!
   */
  gnome_canvas_item_raise_to_top (PXM_SEQED_WIDGET (widget)->
				  cursor->canvas_item);
  
  PXM_SEQED_WIDGET (widget)->cursor->x = rect.x1;
  PXM_SEQED_WIDGET (widget)->cursor->y = rect.y1;
}


void 
polyxmass_seqed_widget_remove_selection_polygon (GtkWidget *widget)
{

  g_assert (widget != NULL);
  
  PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 = -1;
  PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 = -1;
  
  if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
    {
      gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
				 canvas_selection_points);
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
    }
  
  if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
				      canvas_selection_polygon));

      PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
    }
  
  PXM_SEQED_WIDGET (widget)->selection_rect.x1 
    = PXM_SEQED_WIDGET (widget)->selection_rect.x2;

  PXM_SEQED_WIDGET (widget)->selection_rect.y1 
    = PXM_SEQED_WIDGET (widget)->selection_rect.y2;
  
  return;
}
  

void
polyxmass_seqed_widget_draw_sel_polygon (GtkWidget *widget)
{
  /* We get a pointer to a polymer editing context structure that
   * contains a selection_rect member structure that contains the
   * coordinates of the (one or two) point(s) describing the pixel
   * position in the sequence editor canvas of the current selection
   * (if any !).  We use this (these) point(s) to draw a polygon by
   * calculating all the necessary points around the selecton to draw a
   * "polyline" polygon.
   */
  gint sel_lines_count;

  gdouble end_line_EE_x = 0;


  g_assert (widget != NULL);

  /* Only consider selection is effective if the movement amplitude in
   * x or y is at least a third of the monicon width (in x).
   */
  if (abs (PXM_SEQED_WIDGET (widget)->selection_rect.x1 - 
	   PXM_SEQED_WIDGET (widget)->selection_rect.x2) 
      <= (0.3 * PXM_SEQED_WIDGET (widget)->monicon_size)
      && 
      abs (PXM_SEQED_WIDGET (widget)->selection_rect.y1 - 
	   PXM_SEQED_WIDGET (widget)->selection_rect.y2)
      <= (0.3 * PXM_SEQED_WIDGET (widget)->monicon_size))
    {
      /* the user has moved the pointer while pushing left button
       * over a too small distance to consider that it is a selection.
       * Thus we do not bother with drawing any specific selection
       * polygon, we just set the sel_monomerIdx(1,2) to -1, which
       * indicates that no real selection is there.
       */
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 = -1;
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 = -1;

      if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
	{
	  gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
				     canvas_selection_points);
	  PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
	}
      if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
	{
	  gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
					  canvas_selection_polygon));
	  PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
	}
      return;
    }
  

  /* Apparently the user has effectively wanted to select a piece
   * of polymer sequence.
   */
  /* Get the indexes of the first and last selected monomers.
   */
  PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 = 
    polyxmass_seqed_widget_get_mnm_idx_with_xy (widget,
					     PXM_SEQED_WIDGET (widget)->
					     selection_rect.x1,
					     PXM_SEQED_WIDGET (widget)->
					     selection_rect.y1);
  
  PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 = 
    polyxmass_seqed_widget_get_mnm_idx_with_xy (widget,
					     PXM_SEQED_WIDGET (widget)->
					     selection_rect.x2,
					     PXM_SEQED_WIDGET (widget)->
					     selection_rect.y2);
  
  /* We have to set an ordered polygon construction mechanism because
   * the points at each polygon corner are in a very precise order.
   */
  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 <= PXM_SEQED_WIDGET (widget)->
      sel_mnm_idx2)
    {
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx1,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx2,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
    }

  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
      > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
    {
      /* We have to interchange the two monomers, since for polygon
       * drawing we need the first monomer to be the lowest index
       * Be shure to see the difference in the function calls this time !
       */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx1,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx2,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
    }

  /* Now we can draw the polygon.
   */

  /* But first eliminate any trace of a previous polygon!
   */
  if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
    {
      gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
				 canvas_selection_points);
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
    }

  if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
				      canvas_selection_polygon));
      PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
    }

  /* Calculate the X position corresponding to EAST position of
   * last monomer icon in a ROW.
   */
  end_line_EE_x = PXM_SEQED_WIDGET (widget)->left_margin 
    + (PXM_SEQED_WIDGET (widget)->monicons_per_line 
       * PXM_SEQED_WIDGET (widget)->monicon_size);

  /* Calculate the number of lines that the selection encompasses.
   */
  sel_lines_count = (PXM_SEQED_WIDGET (widget)->mnm2rect.y1 
		     - PXM_SEQED_WIDGET (widget)->mnm1rect.y1)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  /* We only draw the polygon if there is a real polygon,
   * which means something at least a square!
   */
  if (sel_lines_count == 0)
    {
      if (PXM_SEQED_WIDGET (widget)->mnm1rect.x1 
	  == PXM_SEQED_WIDGET (widget)->mnm2rect.x1)
	{
	  /* Two points which should have different X coordinates
	   * are the same : there is no real square here.
	   */
	  if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
	    {
	      gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
					 canvas_selection_points);
	      PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
	    }
	  if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
	    {
	      gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
					      canvas_selection_polygon));
	      PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
	    }
	  return;
	}
    }

  /*
    debug_printf (("SELECTION = monomer1 idx is %d -- monomer2 idx is
    %d\n", PXM_SEQED_WIDGET (widget)->sel_mnm_idx1, PXM_SEQED_WIDGET
    (widget)->sel_mnm_idx2));
  */

  /* We do not want that the selection polygon makes a fake
   * more-than-one-line selection pattern while at last line it does
   * not select anything because selection actually stops at right end
   * of line above.
   */
  if (sel_lines_count > 0 && PXM_SEQED_WIDGET (widget)->mnm2rect.x1 == 
      PXM_SEQED_WIDGET (widget)->left_margin)
    {
      /* We are right at the MARGIN, and we want to go up one line and
       * right to the last monicon.
       */
      sel_lines_count -= 1;
      PXM_SEQED_WIDGET (widget)->mnm2rect.x1 = end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->mnm2rect.y1 -= PXM_SEQED_WIDGET (widget)->
	monicon_size;
      PXM_SEQED_WIDGET (widget)->mnm2rect.y2 -= PXM_SEQED_WIDGET (widget)->
	monicon_size;
    }  
  
  /* Now resume normal way of positioning the points of the selection
   * polygon.
   */
  if (sel_lines_count == 0)
    {
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = 
	gnome_canvas_points_new (4);

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[0] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[1] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[2] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[3] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[4] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[5] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y2;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[6] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[7] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y2;

    }

  if (sel_lines_count > 0)
    {
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = 
	gnome_canvas_points_new (8);
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[0] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[1] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[2] = 
	end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[3] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[4] = 
	end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[5] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size);
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[6] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[7] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size);
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[8] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[9] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[10] = 
	LEFT_MARGIN;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[11] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[12] = 
	LEFT_MARGIN;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[13] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1
	+ PXM_SEQED_WIDGET (widget)->monicon_size;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[14] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[15] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }

  PXM_SEQED_WIDGET (widget)->canvas_selection_polygon =
    gnome_canvas_item_new
    (PXM_SEQED_WIDGET (widget)->canvas_main_group,
     gnome_canvas_polygon_get_type (),
     "points", PXM_SEQED_WIDGET (widget)->canvas_selection_points,
     "outline_color", "black",
     "width_units", SEL_POLYGON_LINE_WIDTH,
     NULL);
  
  g_signal_connect (G_OBJECT (PXM_SEQED_WIDGET (widget)->
			      canvas_selection_polygon), 
		    "event",
		    G_CALLBACK (polyxmass_seqed_widget_sel_polygon_event), 
				PXM_SEQED_WIDGET (widget));
}


gboolean
polyxmass_seqed_widget_update_sel_polygon (GtkWidget *widget)
{
  /* We get the xy positions for the two points corresponding to
   * initial impact of mouse and to current mouse position.
   */
  gint sel_lines_count;

  gdouble end_line_EE_x = 0;


  g_assert (widget != NULL);


  /* This function returns TRUE if the selection polygon was 
     actually changed; FALSE if it remained unchanged.
  */


  /* If the window was moved or resized, the sequence is redrawn,
   * which is ok, but the selection is updated, hence we are here. BUT
   * if no selection was in operation before the moving or resizing of
   * the window, then we do not want any selection (if we let the
   * function do its work here with both values below == 0, the first
   * monomer of the sequence is selected, which we do not want)
   */
  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 == -1 
      && PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 == -1)
    return FALSE;

  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 <= PXM_SEQED_WIDGET (widget)->
      sel_mnm_idx2)
    {
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx1,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx2,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
    }
  
  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
      > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
    {
      /* we have to interchange the two monomers, since for polygon
       * drawing we need the first monomer to be the lowest index
       * Be shure to see the difference in the function calls this time !
       */
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx1,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx2,
						   widget, 
						   &(PXM_SEQED_WIDGET (widget)->
						     mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
    }
  
  /* Now we can draw the polygon.
   */
  
  /* But first eliminate any trace of a previous polygon!
   */
  if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
    {
      gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
				 canvas_selection_points);
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
    }

  if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
				      canvas_selection_polygon));
      PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
    }

  /* Calculate the X position corresponding to EAST position of
   * last monomer icon in a ROW.
   */
  end_line_EE_x = PXM_SEQED_WIDGET (widget)->left_margin 
    + (PXM_SEQED_WIDGET (widget)->monicons_per_line 
       * PXM_SEQED_WIDGET (widget)->monicon_size);
  
  /* Calculate the number of lines that the selection encompasses.
   */
  sel_lines_count = (PXM_SEQED_WIDGET (widget)->mnm2rect.y1
		     - PXM_SEQED_WIDGET (widget)->mnm1rect.y1)
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  /* We only draw the polygon if there is a real polygon,
   * which means something at least a square!
   */
  if (sel_lines_count == 0)
    {
      if (PXM_SEQED_WIDGET (widget)->mnm1rect.x1 
	  == PXM_SEQED_WIDGET (widget)->mnm2rect.x1)
	{
	  /* Two points which should have different X coordinates
	   * are the same : there is no real square here.
	   */
	  if (PXM_SEQED_WIDGET (widget)->canvas_selection_points)
	    {
	      gnome_canvas_points_unref (PXM_SEQED_WIDGET (widget)->
					 canvas_selection_points);
	      PXM_SEQED_WIDGET (widget)->canvas_selection_points = NULL;
	    }
	  if (PXM_SEQED_WIDGET (widget)->canvas_selection_polygon)
	    {
	      gtk_object_destroy (GTK_OBJECT (PXM_SEQED_WIDGET (widget)->
					      canvas_selection_polygon));
	      PXM_SEQED_WIDGET (widget)->canvas_selection_polygon = NULL;
	    }
	  return TRUE;
	}
    }

  /* We do not want that the selection polygon makes a fake
   * more-than-one-line selection pattern while at last line it does
   * not select anything because selection actually stops at right end
   * of line above.
   */
  if (sel_lines_count > 0 && PXM_SEQED_WIDGET (widget)->mnm2rect.x1 == 
      PXM_SEQED_WIDGET (widget)->left_margin)
    {
      /* We are right at the MARGIN, and we want to go up one line and
       * right to the last monicon.
       */
      sel_lines_count -= 1;
      PXM_SEQED_WIDGET (widget)->mnm2rect.x1 = end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->mnm2rect.y1 -= PXM_SEQED_WIDGET (widget)->
	monicon_size;
      PXM_SEQED_WIDGET (widget)->mnm2rect.y2 -= PXM_SEQED_WIDGET (widget)->
	monicon_size;
    }  

  /* Now resume normal way of positioning the points of the selection
   * polygon.
   */
  if (sel_lines_count == 0)
    {
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = 
	gnome_canvas_points_new (4);

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[0] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[1] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[2] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[3] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[4] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[5] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y2;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[6] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[7] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.y2;
   }

  if (sel_lines_count > 0)
    {
      PXM_SEQED_WIDGET (widget)->canvas_selection_points = 
	gnome_canvas_points_new (8);

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[0] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[1] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[2] = 
	end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[3] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[4] =
	end_line_EE_x;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[5] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size);

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[6] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[7] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size);;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[8] = 
	PXM_SEQED_WIDGET (widget)->mnm2rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[9] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[10] = 
	LEFT_MARGIN;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[11] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1 +
	(sel_lines_count * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[12] = 
	LEFT_MARGIN;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[13] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1
	+ PXM_SEQED_WIDGET (widget)->monicon_size;

      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[14] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.x1;
      PXM_SEQED_WIDGET (widget)->canvas_selection_points->coords[15] = 
	PXM_SEQED_WIDGET (widget)->mnm1rect.y1
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }

  PXM_SEQED_WIDGET (widget)->canvas_selection_polygon =
    gnome_canvas_item_new 
    (PXM_SEQED_WIDGET (widget)->canvas_main_group,
     gnome_canvas_polygon_get_type (),
     "points", PXM_SEQED_WIDGET (widget)->canvas_selection_points,
     "outline_color", "black",
     "width_units", SEL_POLYGON_LINE_WIDTH,
     NULL);
  
  g_signal_connect (G_OBJECT (PXM_SEQED_WIDGET (widget)->
			      canvas_selection_polygon), 
		    "event",
		    G_CALLBACK (polyxmass_seqed_widget_sel_polygon_event), 
		    PXM_SEQED_WIDGET (widget));
  
  return TRUE;
}


void
polyxmass_seqed_widget_monicon_size_value_changed (GtkWidget *range,
						   gpointer data)
{
  /* The user asks that the size of the monicons used to display the
     sequence be changed. We thus have to first destroy all the
     monicon objects in the array of monicon objects for the current
     sequence editor context and recreate them fully from disk.
  */

  GtkWidget *widget = data;
  
  gint new_size = 0;

  g_assert (range != NULL);
  g_assert (widget != NULL);
  
  new_size = (gint) gtk_range_get_value (GTK_RANGE (range));
  
  PXM_SEQED_WIDGET (widget)->monicon_size = new_size;
  
  polyxmass_seqed_widget_render_polseq_monicons (widget, TRUE);
  
  /* Rendering of the cursor cannot fail, or an error is issued with
   * following stop of the program.
   */
  polyxmass_pixbuf_rendering_cursor_monicon_render (widget, TRUE);
  
  if (-1 == polyxmass_seqed_widget_redraw_sequence (widget))
    {
      g_warning (_("%s@%d: failed to draw the sequence\n"),
		 __FILE__, __LINE__);
      
      return;
    }
  
  polyxmass_seqed_widget_update_sel_polygon (widget);
  
  polyxmass_seqed_widget_draw_cursor (widget);

  polyxmass_seqed_widget_ensure_selection_and_cursor_visible 
    (PXM_SEQED_WIDGET (widget)->sw, NULL, widget);

  return ;
}





/*
****************************************************************
**************** VARIOUS  UTILITY   FUNCTIONS ******************
****************************************************************
*/
gboolean
polyxmass_seqed_widget_get_selection_indices (GtkWidget *widget,
					      gint *start, gint *end)
{
  g_assert (widget != NULL);
  
  
  /* A selection must actually exist.
   */

  /* Non-selection is stated in the seqed_widget by setting any of the
   * two members seqed_widget->sel_mnm_idx1 and
   * seqed_widget->sel_mnm_idx2 to -1.
   */
  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 == -1 
       || PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 == -1)
    return FALSE;

  /* We want that the selection be ordered, which means that we need
   * that the start of the selection be less than or equal to the end
   * of the selection.
   */
  if (start != NULL)
    {
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	*start = PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 ;
      else
	*start = PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 ;
    }
  if (end != NULL)
    {
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  > PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	*end = PXM_SEQED_WIDGET (widget)->sel_mnm_idx1;
      else
	*end = PXM_SEQED_WIDGET (widget)->sel_mnm_idx2;
    }
  
  /*
    if (start != NULL && end != NULL) 
    debug_printf (("polyxmass_seqed_widget_get_selection_indices:" " [%d--%d]",
    *start, *end));
  */
  
  return TRUE;
}


gint
polyxmass_seqed_widget_set_point_at_idx (GtkWidget *widget,
					 gint idx)
{
  gint length = 0;
  
  PxmPolymer *polymer = NULL;
  
  /* This function returns the last_point_1_idx value.
   */

  g_assert (widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  

  length = polymer->monomerGPA->len;
  
  /* We just remove a selection polygon if any and set the cursor so
     that it is located at index 'idx'.*/

  g_assert (idx >=0);
  g_assert (idx <= length);
  

  /* And finally we kill the selection polygon, and ask that the cursor
   * be drawn at index 'idx'. The function below will set to -1
   * both sel_mnm_idx1 and sel_mnm_idx2.
   */
  polyxmass_seqed_widget_remove_selection_polygon (widget);
  
  /* Set the cursor position at the new index:
   */
  PXM_SEQED_WIDGET (widget)->last_point_1_idx = idx;
  
  polyxmass_seqed_widget_draw_cursor (widget);
  
  /* Redraw the whole sequence.
   */
  if (- 1 == polyxmass_seqed_widget_redraw_sequence (widget))
    {
      g_critical (_("%s@%d: failed to draw the sequence\n"),
	     __FILE__, __LINE__);
    }
    
  /*
  if (gtk_events_pending())
    gtk_main_iteration();
  */

  /* Return the present index, so that if this function was 
   * called as the first step in a replacement operation, the caller
   * knows at which index it should insert the replacement. This is 
   * particularly interesting if this function was called with 
   * monomer pointers in the p_seqcoords, and not indexes.
   */

  return PXM_SEQED_WIDGET (widget)->last_point_1_idx;
}



PxmMonomer *
polyxmass_seqed_widget_get_selected_monomer (GtkWidget *widget,
					     gint *idx)
{
  gint index = 0;
  
  PxmPolymer *polymer = NULL;
  

  /* We return a valid pointer ONLY if a single monomer is currently
   * selected. Otherwise we return NULL.
   */

  g_assert (widget != NULL);

  polymer = PXM_SEQED_WIDGET (widget)->polymer;
  g_assert (polymer != NULL);
  
  
  g_assert (polymer->monomerGPA != NULL);
  
  if (polymer->monomerGPA->len <= 0)
    return NULL;
    
  /* If no selection: return NULL.
   */
  if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 == -1 
      || PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 == -1)
    return NULL;
  
  /* If more than one monomer selected: return NULL.
   */
  if (abs (PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 -
	   PXM_SEQED_WIDGET (widget)->sel_mnm_idx1) != 1)
    return NULL;
  
  /* The index of the monomer that is valid is the lowest of the two
   * (we do not know if the selection was made from highest to lowest
   * monomer indexes).
   */
  index =
    PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
    < PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 ?
    PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 : 
    PXM_SEQED_WIDGET (widget)->sel_mnm_idx2;
  
  /* Sanity check, the selected monomer cannot be out of boundaries of 
   * the array that contains the polymer sequence:
   */
  g_assert (index < polymer->monomerGPA->len);
  
  if (idx != NULL)
    *idx = index;
  
  return g_ptr_array_index (polymer->monomerGPA, index);
}

  
gboolean
polyxmass_seqed_widget_monicon_get_pixel_coord (gint idx, 
						GtkWidget *widget, 
						PxmRect *rect, 
						PxmCoordsys coordsys1, 
						PxmCoordsys coordsys2)
{
  gint pos_line = 0;
  gint line_num = 0;
  


  g_assert (widget != NULL);
  
  g_assert (rect != NULL);
  g_assert (idx >= 0);
  
  PXM_SEQED_WIDGET (widget)->monicons_per_line =
    (PXM_SEQED_WIDGET (widget)->sw_width - 
     PXM_SEQED_WIDGET (widget)->left_margin) 
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  line_num = idx / PXM_SEQED_WIDGET (widget)->monicons_per_line;
  
  pos_line = idx - (line_num * PXM_SEQED_WIDGET (widget)->monicons_per_line);
  
  /* First calculate the first point's coordinates.
   */
  if (coordsys1 == COORDSYS_CN) 
    /* top vertically, center horizontally*/
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
      rect->y1 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  if (coordsys1 == COORDSYS_NW)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y1 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys1 == COORDSYS_SW)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y1 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys1 == COORDSYS_NE)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y1 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys1 == COORDSYS_SE)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y1 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys1 == COORDSYS_CW)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y1 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
    }
  else if (coordsys1 == COORDSYS_CE)
    {
      rect->x1 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y1 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
    }



  /* Second calculate the second point's coordinates.
   */
  if (coordsys2 == COORDSYS_CN) 
    /* top vertically, center horizontally*/
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
      rect->y2 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  if (coordsys2 == COORDSYS_NW)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y2 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys2 == COORDSYS_SW)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y2 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys2 == COORDSYS_NE)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y2 = line_num * PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys2 == COORDSYS_SE)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y2 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size) 
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
    }
  else if (coordsys2 == COORDSYS_CW)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size);
      rect->y2 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
    }
  else if (coordsys2 == COORDSYS_CE)
    {
      rect->x2 = 
	PXM_SEQED_WIDGET (widget)->left_margin + 
	(pos_line * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ PXM_SEQED_WIDGET (widget)->monicon_size;
      rect->y2 = (line_num * PXM_SEQED_WIDGET (widget)->monicon_size)
	+ (PXM_SEQED_WIDGET (widget)->monicon_size / 2);
    }
  
  return TRUE;
}



gint
polyxmass_seqed_widget_get_mnm_idx_with_xy (GtkWidget *widget, 
					    gdouble mouse_x, 
					    gdouble mouse_y)
{
  gint length = 0;
  gint line_count = 0;


  /* monicon_numb is the number (with fraction part) of the
   * icon in the row we are sitting on . For example, 2.25 means
   * we are on the third icon in the fourth part of it. We also
   * can formulate this : "we are one fourth of icon forward after icon
   * number 2 in a row".
   */
  gdouble monicon_numb_x = 0;
  gdouble fractional_part = 0;
  gdouble interger_part = 0;

  gint res_idx = 0;

  /* Easy to understand from code below.
   */
  gint monicon_numb_y = 0;
  gint icons_per_line = 0;



  g_assert (widget != NULL);

  length = PXM_SEQED_WIDGET (widget)->moniconGPA->len;
  
  
  /* First of all note that if the mouse_x value is less than the
     margin, then we should return -1. That -1 value should be tested
     against by the caller function to inhibit dangerous actions. 

     Equally, if mouse_y is less than zero, we should also return -1.
  */

  if (mouse_x < PXM_SEQED_WIDGET (widget)->left_margin)
    return -1;
  if (mouse_y < 0)
    return -1;
  
  
  /* The y dimension is easier than the x dimension.
   */
  icons_per_line = 
    (PXM_SEQED_WIDGET (widget)->sw_width 
     - PXM_SEQED_WIDGET (widget)->left_margin) 
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  line_count = mouse_y / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  monicon_numb_y = line_count * icons_per_line;

  /* If the mouse pointer is clicked in the last horizontal tenth part
   * of a given icon, the cursor will be drawn on the right of this
   * monomer icon. And the monomer index incremented by one.
   *
   * Note that this is important to allow, because it is the only
   * way we can artificially set the currently pointed monomer index
   * (thus also draw the cursor) right of last monomer of a polymer
   * sequence so that user has the feeling that she will append
   * monomer's to the end of the polymer sequence. 
   */
  monicon_numb_x = (mouse_x - PXM_SEQED_WIDGET (widget)->left_margin) 
    / PXM_SEQED_WIDGET (widget)->monicon_size;
  
  fractional_part = modf (monicon_numb_x, &interger_part);
  
  if (fractional_part > 0.9)
    ++interger_part;
  
  res_idx = (gint) interger_part + monicon_numb_y;
  
  /* Some easy checks, again, a bit paranoid, but this ensures
     consistency.
   */
  if (res_idx < 0)
    res_idx = -1 ;
  if (res_idx > length)
    res_idx = -1;

  return res_idx;
}




gboolean
polyxmass_seqed_widget_select_sequence (GtkWidget *widget,
					gint start_idx, gint end_idx)
{
  PxmRect rect;

  gint length = 0;


  g_assert (widget != NULL);


  length = PXM_SEQED_WIDGET (widget)->polymer->monomerGPA->len;
  
  /* A polymer sequence must actually exist.
   */
  if (length < 1)
    return TRUE;

  /* Note that if any of the two index parameters are -1, then that
     means that the whole sequence should be selected.
  */
  if (start_idx == -1 || end_idx == -1)
    {
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 = 0;
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 = length;
    }
  else
    {
      /* Now we should make some tests to ensure that we are not
	 accessing monomers outside of the boundaries of the polymer
	 sequence. Remember that we have to increment the end_idx
	 value by one so that the selection routines actually work as
	 expected.
      */
      g_assert (start_idx < length);
      g_assert (end_idx + 1 <= length);
      
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 = start_idx;
      PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 = end_idx + 1;
    }

  if (TRUE == 
      polyxmass_seqed_widget_update_sel_polygon (widget))
    {
      /* Update the  selection masses, since the selection has changed.
       */

      /* The selection.
       */
      g_signal_emit (G_OBJECT (widget), 
		     seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
      
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx1,
						   widget,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x1 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y1 = rect.y1;
      
      polyxmass_seqed_widget_monicon_get_pixel_coord (PXM_SEQED_WIDGET (widget)->
						   sel_mnm_idx2,
						   widget,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      PXM_SEQED_WIDGET (widget)->selection_rect.x2 = rect.x1;
      PXM_SEQED_WIDGET (widget)->selection_rect.y2 = rect.y1;  



      /* Reposition the cursor and redraw it !
       */
      PXM_SEQED_WIDGET (widget)->last_point_1_idx = end_idx + 1;
      polyxmass_seqed_widget_draw_cursor (widget);
  
      /* Since we decided above that we would select a whole portion
	 of the sequence, from top to bottom of the sequence, we must
	 make sure that the cursor is visible and that the horizontal
	 bottom line of the selection polygon is visible at the bottom
	 of the sequence display area.
      */
      polyxmass_seqed_widget_ensure_selection_and_cursor_visible 
	(PXM_SEQED_WIDGET (widget)->sw, NULL, widget);


      /* And now make sure that we claim ownership of the X PRIMARY
	 selection.
      */
      polyxmass_seqed_widget_clipboard_primary_copy (widget);      
    }
    
  return TRUE;
}



gboolean 
polyxmass_seqed_widget_ensure_region_visible (GtkWidget *widget,
					      gint start_idx,
					      gint end_idx)
{
  /* This function is called by other functions when they need to make
   * sure that a given polymer sequence region in the seqed_widget is
   * effectively visible (at least its first elements if not in its
   * entirety) in the seqed_widget. We are handed the seqed_widget
   * pointer and the two indexes of the start monomer and of the end
   * monomer which both constitute the borders of the region to
   * display.
   */
  PxmSeqedWidget *seqed_widget = PXM_SEQED_WIDGET (widget);

  gint last_vis_line_pixel_y = 0;
  gint moment_idx = 0;
  gint hscrollbar_height = 0;

  PxmRect start_rect;
  
    
  g_assert (seqed_widget != NULL);
  g_assert (seqed_widget->polymer != NULL);
  

  /* We may check some sanity issues:
   */
  if (start_idx > end_idx)
    {
      moment_idx = start_idx;
      start_idx = end_idx;
      end_idx = moment_idx;
    }
  
  if (start_idx < 0)
    return FALSE;
  
  if (end_idx > seqed_widget->polymer->monomerGPA->len -1)
    return FALSE;
  
  /* Get the pixel y coordinate of the last visible line, we'll need it
   * later.
   */ 
  last_vis_line_pixel_y =  seqed_widget->sw_vadjust->value 
    + seqed_widget->sw_height;

  /* We should get the upper left (NorthWest) y coordinate of 
   * the initial monomer in the region (index start_idx):
   */
  polyxmass_seqed_widget_monicon_get_pixel_coord (start_idx,
					       widget,
					       &start_rect,
					       COORDSYS_NW, COORDSYS_NW);
  
  /* start_rect.y1 represents now such y NW coordinate. If this 
   * y coordinate is not visible we should make a scroll, otherwise
   * do not bother. ATTENTION, the size of the scroll bar, the 
   * horizontal one (if any) is included in the calculatation of the 
   * scroll win height. This is why, in order to ENSURE that the region
   * which may be partially visible on the bottom of the seqed window is
   * indeed fully visible, I have asked that the north western 
   * coordinate of the monomer at index start_idx be less or equal to
   * the last visible line of the seqeditor window minus
   * the size of an icon and minus the horizontal scrollbar height.
   */
  hscrollbar_height = 
    GTK_WIDGET (GTK_SCROLLED_WINDOW (seqed_widget->sw)->
		hscrollbar)->allocation.height;
  
  if (start_rect.y1 <  seqed_widget->sw_vadjust->value
      || start_rect.y1 > (last_vis_line_pixel_y - 
			  seqed_widget->monicon_size -
			  hscrollbar_height))
    {
      seqed_widget->canvas_y_offset = start_rect.y1;
      
      gnome_canvas_scroll_to (GNOME_CANVAS (seqed_widget->canvas),
			      seqed_widget->canvas_x_offset,
			      seqed_widget->canvas_y_offset);
      
      scroll_size_changed = FALSE;
      
      return TRUE;
    }

  return TRUE;
}



gboolean
polyxmass_seqed_widget_ensure_selection_and_cursor_visible (GtkWidget *sw, 
							    GdkEventConfigure *
							    event, 
							    gpointer data)
{
  /* Called after the handler seqed_sw_size_allocate() was
   * called, since we connected this function through a call to 
   * gtk_signal_connect_after () for "size_allocate" signal of 
   * PXM_SEQED_WIDGET (widget)->canvas_scroll_wnd. 
   *
   * However, we saw that the size_allocate () function is called even
   * when no change in the size of the scrolled window is made, thus
   * eliciting a call to _this_ function, which we do not want, since
   * the calculations dones here are ONLY for use when the size of the
   * window EFFECTIVELY changes. SO, we have a file-scope variable
   * (gboolean scrollwin_size_changed) that is set to TRUE in
   * size_allocate () function whenever a real size change has been
   * done to the scrolled window. If this variable equals TRUE in
   * _this_ function that means we have to do the calculatations,
   * otherwise just return.
   */
  /* This function is meant to make sure that the cursor (and, if any,
   * the biggest portion of the current selection) be visible in the
   * seqeditor window.  Typically this function is called when the
   * window is resized, which might hide the cursor (and selection if
   * any) from the visible portion of the sequence.
   */
  GtkWidget *widget = data; /* The seqed_widget */
  
  /* The sw parameter is the scrolled window that contains the
     canvas. The "size_allocate" event signal connect after was set
     for this window in polyxmass_seqed_widget_init ().
   */
  
  gint last_vis_line_pixel_y = 0;
  gint hscrollbar_height = 0;


  g_assert (widget != NULL);
  

  /* Removed because it would inhibit the process when editing 
     a piece of sequence that would be below the bottom of the canvas.
     
     if (scroll_size_changed == FALSE) 
     return TRUE;
  */

  
  /* There are two cases :
   *
   * 1. there is not a selection in the sequnece editor. In this case
   * we only want to make sure that the cursor is visible in the
   * sequence editor. Best would be to make it visible in the vertical
   * center of the sequence editor.
   *
   * 2. there is a selection. In this case there are two different
   * situations :
   *
   * 2.A. if the selection was done from top to bottom (sel_mnm_idx1 <
   * sel_mnm_idx2) the window should be scrolled so that the cursor
   * (at position sel_mnm_idx2) is displayed at the bottom of the
   * seqeditor, which is the last visible line.  This way we manage
   * that the largest portion possible of the selection is displayed
   * in the seqeditor window.
   *
   * 2.B. if the selection was done from bottom to top (sel_mnm_idx2 <
   * sel_mnm_idx1) the window should be scrolled so that the cursor
   * (at position sel_mnm_idx1) is displayed at the top of the
   * seqeditor, which is the first visible line. This way we manage
   * that the largest portion possible of the selection is displayed
   * in the seqeditor window.
   */

  /* Get the pixel y coordinate of the last visible line, we'll need
   * it later.
   */ 
  last_vis_line_pixel_y = 
    PXM_SEQED_WIDGET (widget)->sw_vadjust->value
    + PXM_SEQED_WIDGET (widget)->sw_height;
  
  /* Check if there is a selection.
   */
  if ((PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 != -1) 
      && (PXM_SEQED_WIDGET (widget)->sel_mnm_idx2 != -1))
    {
      /* An oligomer is selected. We have to test the two different cases
       * described above.
       */
      if (PXM_SEQED_WIDGET (widget)->sel_mnm_idx1 
	  < PXM_SEQED_WIDGET (widget)->sel_mnm_idx2)
	{
	  /* The selection was done from top to bottom, so the scroll
	     has to be done so that the cursor is at the last visible
	     line.
	  */
	  
	  /* ATTENTION, the size of the scroll bar, the horizontal one
	     (if any) is included in the calculatation of the scroll win
	     height.  This is why, in order to ENSURE that the cursor
	     be precisely located at the bottom of the visible scroll
	     window, I have asked that the window be scrolled
	     according to the calculatation below:
	   */
	  hscrollbar_height = 
	    GTK_WIDGET (GTK_SCROLLED_WINDOW (PXM_SEQED_WIDGET (widget)->sw)->
			hscrollbar)->allocation.height;
	  
	  PXM_SEQED_WIDGET (widget)->canvas_y_offset = 
	    PXM_SEQED_WIDGET (widget)->cursor->y  
	    + PXM_SEQED_WIDGET (widget)->monicon_size
	    - PXM_SEQED_WIDGET (widget)->sw_height 
	    + hscrollbar_height 
	    + SEL_POLYGON_LINE_WIDTH ;
	}
      else
	  /* The selection was done from bottom to top, so the scroll
	   * has to be done so that the cursor is at first visible
	   * line.
	   */
	  PXM_SEQED_WIDGET (widget)->canvas_y_offset = 
	    PXM_SEQED_WIDGET (widget)->cursor->y;
      
      /* Now that the new canvas_y_offset is calculated, make the scroll
	 happen.
      */
      gnome_canvas_scroll_to (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->
					    canvas),
			      PXM_SEQED_WIDGET (widget)->canvas_x_offset,
			      PXM_SEQED_WIDGET (widget)->canvas_y_offset);
      
      scroll_size_changed = FALSE;
      
      return TRUE;
    }
  else
    /* No oligomer is currently selected, so the only task here is to
     * scroll the window to make sure the cursor is visible in it.
     */
    {
      /* Only perform the calculations and the y_scroll if the cursor
       * effectively lies either above the first visible line or below
       * the last visible line.
       */
      /* Test if cursor is not visible, in which case perform a scroll
       * of the canvas, so that the cursor is now visible on the first
       * visible line of the seqeditor.
       */
      if ((PXM_SEQED_WIDGET (widget)->cursor->y 
	   < PXM_SEQED_WIDGET (widget)->sw_vadjust->value)
	  || 
	  (PXM_SEQED_WIDGET (widget)->cursor->y 
	   > last_vis_line_pixel_y))
	{
	  PXM_SEQED_WIDGET (widget)->canvas_y_offset = 
	    PXM_SEQED_WIDGET (widget)->cursor->y;
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (PXM_SEQED_WIDGET (widget)->canvas),
	     PXM_SEQED_WIDGET (widget)->canvas_x_offset,
	     PXM_SEQED_WIDGET (widget)->canvas_y_offset);
	  
	  scroll_size_changed = FALSE;
	  
	  return TRUE;
	}
    }
  
  scroll_size_changed = FALSE;
  
  return TRUE;
}


/*
****************************************************************
***************** SELF READ HANDLING FUNCTIONS *****************
****************************************************************
*/
gint
polyxmass_seqed_widget_self_read_configuration (GtkWidget *widget)
{
  PxmPolchemdefCtxt *polchemdefctxt = NULL;

  GtkWidget *window = NULL;
  
  /* We are asked to show a window that contains all the
     configurations for the user to tell what should be exported to
     the sound playlist file.
  */
  g_assert (widget != NULL);
  

  polchemdefctxt = PXM_SEQED_WIDGET (widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);


  window = polyxmass_seqed_widget_self_read_wnd_setup (widget);
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to set up the self-read config window\n"),
	     __FILE__, __LINE__);
      
      return -1;
    }

  return 1;
}



/*
****************************************************************
************ LOW LEVEL CLIPBOARD HANDLING FUNCTIONS ************
****************************************************************
*/
void
polyxmass_seqed_widget_clipboard_selection_get (GtkWidget *widget,
						GtkSelectionData 
						*selection_data,
						guint info,
						guint time_stamp,
						gpointer data)
{
  /* widget is the seqed_widget->canvas */
  /* data is seqed_widget */


  PxmProp *prop = NULL;

  PxmSeqedWidget *seqed_widget = data;
  
  gint start_idx = -1;
  gint end_idx = -1;

  gchar *seq;

  /* This function is called as a callback to a "selection_get" signal
     that our process gets from an application (or itself) that would
     like to paste what whe may have previously put in either the
     GDK_SELECTION_PRIMARY or the GDK_SELECTION_CLIPBOARD selections!
     Whe have to make the difference between the two, because:

     1. if the GDK_SELECTION_PRIMARY is asked, then we real-time have
     to check if a sequence is currently selected. That is "HOT"
     selection.

     2. if the GDK_SELECTION_CLIPBOARD is asked, then we have to check
     if the canvas has a prop object named "CLIPBOARD_SELECTION"
     containing the sequence that was allocated when the user last did
     Ctrl + C. If such prop is there, then its data have to be
     provided to the caller with the gtk_selection_data_set () call
     below.
  */

  g_assert (widget != NULL);
  g_assert (seqed_widget != NULL);
  g_assert (seqed_widget->polymer != NULL);
  
  
  if (selection_data->selection == GDK_SELECTION_PRIMARY)
    {
      /* We first check if there is a selection in the widget. If
	 there is such a selection, we just put it in the clipboard.
      */
      if (FALSE == polyxmass_seqed_widget_get_selection_indices (GTK_WIDGET (seqed_widget),
							      &start_idx,
							      &end_idx))
	{
	  g_message (_("%s@%d: no selection in the sequence editor\n"),
		 __FILE__, __LINE__);
      
	  return;
	}
  
      /* Note that due to sequence editor-specific reasons, the end_idx
	 that is returned above contains one more monomer, so we should
	 decrement this once before using the function above.
      */
      seq = 
	pxmchem_polymer_make_codes_string (seqed_widget->polymer,
					   start_idx, end_idx - 1);
      
      g_assert (seq != NULL);
      
      /* When we return a single string, it should not be NULL-terminated,
	 that will be done for us.
      */
      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      (const guchar *) seq, strlen (seq));

      return;
    }
  
  /* We had created a prop object with the clipboard selection and set
     that prop to the conventional propGPA of the seqed_widget. Let's
     get it right now.
  */
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD)
    {
      prop = 
	libpolyxmass_prop_find_prop (seqed_widget->propGPA,
				     NULL,
				     NULL,
				     "CLIPBOARD_SELECTION",
				     NULL,
				     PXM_CMP_NO_DEEP);
      
      if (prop == NULL)
	{
	  debug_printf (("The prop is NULL !!!\n"));
	  
	  return ;
	}
      
      g_assert (prop->data != NULL);
      
      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      prop->data, strlen (prop->data));
      
      return;
    }
  
  return;
}



void
polyxmass_seqed_widget_import_raw_text_file (GtkWidget *widget)
{
  /* widget is the (GtkWidget*) seqed_widget */  
  

  /* We are asked to import a text file as raw text. Note that we have
     to make sure that the user can choose the file.
  */

  GtkWidget *window = NULL;
  GtkWidget *dialog;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  PxmEditCtxt *editctxt = NULL;
    
  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;
  
  gchar *filename = NULL;
  gchar *seq = NULL;

  gchar *line = NULL; 
  gchar *read = NULL;

  FILE *file = NULL;

  GString *gs = NULL;
  

  
  g_assert (widget != NULL);
 

  editctxt = g_object_get_data (G_OBJECT (widget), "editctxt");
  g_assert (editctxt != NULL);
  
  polchemdefctxt = PXM_SEQED_WIDGET(widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  /*
    At this point let the user select a file.
  */


  dialog = gtk_file_chooser_dialog_new ("Open Raw Text File",
					(GtkWindow *)editctxt->sequence_editor_wnd,
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);


  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_CANCEL)
    {
      gtk_widget_destroy (dialog);
      
      return;
    }

  /* Else, GTK_RESPONSE_ACCEPT
   */

  filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    
  /* Let's check that the file actually exists.
   */
  if (FALSE == g_file_test (filename, G_FILE_TEST_EXISTS))
    {
      g_warning (_("%s@%d: file '%s' not found.\n"),
		 __FILE__, __LINE__, filename);
	
      g_free (filename);

      gtk_widget_destroy (dialog);
	
      return;
    }
      
  gtk_widget_destroy (dialog);

  /* Great, the file is there.
   */
  file  = fopen (filename, "r");
    
  if (NULL == file)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file could not be opened: '%s'\n"),
	     __FILE__, __LINE__, filename);

      g_free (filename);

      return ;
    }

  g_free (filename);

  /* Allocate the memory into which each line of the file will be
   * stored for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));
  
  read = line;

  /* The GString object where we'll store all the found lines.
   */
  gs = g_string_new ("");
    
  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, file);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  g_string_append (gs, read);
	}
      /* End of
	 if (read != NULL && strlen (read) > 1)
      */
    }
  /* End of   
     while (read != NULL)
  */

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer
   */
  g_free (line);
 
  fclose (file);
 
  /* The contents of the file are now available in gs.
   */
  seq = gs->str;
  g_assert (seq != NULL);

  g_string_free (gs, FALSE);
  
  /* We now have our string that we have to interpret and to insert in
     the form of monomer icons in the editor at the insertion
     point. If there is a selection, that selection has to be erased
     and replaced.
  */

  /* First off, check that the sequence in the selection is correct by
     just asking to fill an array of monomer with that sequence string.
  */

  /* Allocate the array where the PxmMonomer objects are going to be
     stored as the seq is parsed.
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();

  /* At this point we have to make sure that the string we got
     contains characters that effectively correctly encode
     monomers. That is, we must ensure that by parsing of the sequence
     we can convert the characters parsed into monomer codes. If this
     is not possible, then that means that there are errors and we
     have to make sure the user knows this. We will provide the user
     with the possibility to edit the sequence so that it gets
     purified in order to be pasted into the sequence editor.
  */

  /* The array errorsGPA will contain allocated gint objects
     representing the index of invalid characters/codes in seq.
  */
  pxmchem_monomer_fill_array_from_string_check_errors 
    (fillGPA,
     errorsGPA,
     seq, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     FALSE /*empty_first*/);
  
  if (errorsGPA->len == 0)
    {
      /* There was not a single error during the parsing of the
	 sequence. We can free the array right away.
      */
      g_ptr_array_free (errorsGPA, TRUE);
      errorsGPA = NULL;
    }
  else
    {
      /* Apparently there were errors in the parsing process. We
	 should give the user the proper feedback. This will be
	 performed by opening a window were the user will see the seq
	 string formatted in such a way that the invalid characters
	 will be highlighted to ease correction.
      */

      /* We can immediately free the fillGPA array that contains the valid 
	 monomer codes as we are not going to use it.
      */
      pxmchem_monomer_GPA_free (fillGPA);

      /* OK, now we have to setup the window where the seq string is
	 going to be displayed. We pass to the function the array of
	 indices where errors were encountered so that the seq string
	 will be formatted so that the user will find it easy to see
	 where the errors are located. The seq string will be
	 displayed in a textview widget were formatting is possible.
      */
      window = 
	polyxmass_sequence_purification_wnd_setup (GTK_WIDGET (widget),
							   seq,
							   errorsGPA);
      
      if (window == NULL)
	{
	  g_critical (_("%s@%d: failed to set up the sequence purification "
			"window.\n"),
		      __FILE__, __LINE__);
	  
	  /* NOTE that we do not free the 'errorsGPA' nor the 'seq'
	     objects because that freeing took place in the called
	     function above.
	  */
	  return;
	}
      
      /* At this point we should have setup the window and we can consider
	 that we have finished doing our work here. Let the work go on
	 if the seq string contained no error whatsoever.
      */
      
      /* NOTE that we do not free the errorsGPA nor the seq objects
	 because that freeing took place in the called function above.
      */
      
      return;
    }
  

  /* At this point we know that no error was found during the seq
     string parsing. We thus have an array filled with all the
     PxmMonomer instances corresponding to each monomer code in the
     seq string. Of course, the number of monomer instances in the
     fillGPA MUST be indentical to the number of monomer codes in the
     sequence string. We do not check that because we know all the
     monomer codes could be validated in the function call
     above. Otherwise the errorsGPA->len value would have been != 0.
  */

  /* And now the tough part: the graphical part, and also rememember
     that the PxmMonomer instances that we got are not yet in the
     polymer sequence !
  */

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == polyxmass_seqed_widget_get_selection_indices (widget,
							    NULL,
							    NULL))
    {      
      if (-1 == 
	  polyxmass_seqed_widget_remove_selected_oligomer (widget))
	{
	  g_critical (_("%s@%d: failed to remove the selected "
			"sequence portion.\n"),
		      __FILE__, __LINE__);
	  
	  g_free (seq);
	  
	  pxmchem_monomer_GPA_free (fillGPA);
	  
	  return;
	}
    }
  

  /* Now we should be able to insert the seq in the polymer sequence
     (paste, that is).
  */
  if (-1 == 
      polyxmass_seqed_widget_integrate_sequence_at_point (fillGPA,
							  widget,
							  FALSE /*render_modif*/))
    {
      g_critical (_("%s@%d: failed to integrate the selected "
		    "sequence portion.\n"),
		  __FILE__, __LINE__);
      
      g_free (seq);
      
      pxmchem_monomer_GPA_free (fillGPA);
      
      return;
    }
  
  /* The sequence and the selection.
   */
  g_signal_emit (G_OBJECT (widget), 
		 seqed_widget_signals[WHOLE_SEQ_MASS_UPDATE_REQUIRED], 0);

  g_signal_emit (G_OBJECT (widget), 
		 seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
  
  /* At this point we should be done. 
   */
  pxmchem_monomer_GPA_free (fillGPA);

  g_free (seq);

  return;
}






/*
****************************************************************
*************** CLIPBOARD  HANDLING  FUNCTIONS *****************
****************************************************************
*/
void
polyxmass_seqed_widget_clipboard_primary_text_received (GtkClipboard *clipboard,
							const gchar *text,
							gpointer data)
{
  polyxmass_seqed_widget_clipboard_text_received (clipboard,
						  text,
						  data);
}

void
polyxmass_seqed_widget_clipboard_clipboard_text_received (GtkClipboard *clipboard,
							  const gchar *text,
							  gpointer data)
{
  polyxmass_seqed_widget_clipboard_text_received (clipboard,
						  text,
						  data);
}


void
polyxmass_seqed_widget_clipboard_text_received (GtkClipboard *clipboard,
						const gchar *text,
						gpointer data)
{
  /* data is the seqed_widget */  
  
  GtkWidget *window = NULL;
  
  PxmSeqedWidget *seqed_widget = data;

  gchar *seq = NULL;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;
  


  g_assert (seqed_widget != NULL);
  g_assert (clipboard != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  


  if (text == NULL)
    return;
    
  seq = g_strdup (text);
  g_assert (seq != NULL);

  /* We now have our string that we have to interpret and to insert in
     the form of monomer icons in the editor at the insertion
     point. If there is a selection, that selection has to be erased
     and replaced.
  */

  /* First off, check that the sequence in the selection is correct by
     just asking to fill an array of monomer with that sequence string.
  */

  /* Allocate the array where the PxmMonomer objects are going to be
     stored as the seq is parsed.
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();

  /* At this point we have to make sure that the string we got
     contains characters that effectively correctly encode
     monomers. That is, we must ensure that by parsing of the sequence
     we can convert the characters parsed into monomer codes. If this
     is not possible, then that means that there are errors and we
     have to make sure the user knows this. We will provide the user
     with the possibility to edit the sequence so that it gets
     purified in order to be pasted into the sequence editor.
  */

  /* The array errorsGPA will contain allocated gint objects
     representing the index of invalid characters/codes in seq.
  */
  pxmchem_monomer_fill_array_from_string_check_errors 
    (fillGPA,
     errorsGPA,
     seq, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     FALSE /*empty_first*/);
  
  if (errorsGPA->len == 0)
    {
      /* There was not a single error during the parsing of the
	 sequence. We can free the array right away.
       */
      g_ptr_array_free (errorsGPA, TRUE);
      errorsGPA = NULL;
    }
  else
    {
      /* Apparently there were errors in the parsing process. We
	 should give the user the proper feedback. This will be
	 performed by opening a window were the user will see the seq
	 string formatted in such a way that the invalid characters
	 will be highlighted to ease correction.
      */

      /* We can immediately free the fillGPA array that contains the valid 
	 monomer codes as we are not going to use it.
      */
      pxmchem_monomer_GPA_free (fillGPA);

      /* OK, now we have to setup the window where the seq string is
	 going to be displayed. We pass to the function the array of
	 indices where errors were encountered so that the seq string
	 will be formatted so that the user will find it easy to see
	 where the errors are located. The seq string will be
	 displayed in a textview widget were formatting is possible.
      */
      window = 
	polyxmass_sequence_purification_wnd_setup (GTK_WIDGET (seqed_widget),
							   seq,
							   errorsGPA);
      
      if (window == NULL)
	{
	  g_critical (_("%s@%d: failed to set up the sequence purification "
			"window.\n"),
		      __FILE__, __LINE__);
	  
	  /* NOTE that we do not free the 'errorsGPA' nor the 'seq'
	     objects because that freeing took place in the called
	     function above.
	  */
	  return;
	}
      
      /* At this point we should have setup the window and we can consider
	 that we have finished doing our work here. Let the work go on
	 if the seq string contained no error whatsoever.
      */
      
      /* NOTE that we do not free the errorsGPA nor the seq objects
	 because that freeing took place in the called function above.
      */
      
      return;
    }
  

  /* At this point we know that no error was found during the seq
     string parsing. We thus have an array filled with all the
     PxmMonomer instances corresponding to each monomer code in the
     seq string. Of course, the number of monomer instances in the
     fillGPA MUST be indentical to the number of monomer codes in the
     sequence string. We do not check that because we know all the
     monomer codes could be validated in the function call
     above. Otherwise the errorsGPA->len value would have been != 0.
  */

  /* And now the tough part: the graphical part, and also rememember
     that the PxmMonomer instances that we got are not yet in the
     polymer sequence !
  */

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == polyxmass_seqed_widget_get_selection_indices (GTK_WIDGET (seqed_widget),
							    NULL,
							    NULL))
    {      
      if (-1 == 
	  polyxmass_seqed_widget_remove_selected_oligomer (GTK_WIDGET (seqed_widget)))
	{
	  g_critical (_("%s@%d: failed to remove the selected "
			"sequence portion.\n"),
		      __FILE__, __LINE__);
	  
	  g_free (seq);
	  
	  pxmchem_monomer_GPA_free (fillGPA);
	  
	  return;
	}
    }
  

  /* Now we should be able to insert the seq in the polymer sequence
     (paste, that is).
  */
  if (-1 == 
      polyxmass_seqed_widget_integrate_sequence_at_point (fillGPA,
							  GTK_WIDGET (seqed_widget),
							  FALSE /*render_modif*/))
    {
      g_critical (_("%s@%d: failed to integrate the selected "
		    "sequence portion.\n"),
		  __FILE__, __LINE__);
      
      g_free (seq);
      
      pxmchem_monomer_GPA_free (fillGPA);
      
      return;
    }
  
  /* The sequence and the selection.
   */
  g_signal_emit (G_OBJECT (seqed_widget), 
		 seqed_widget_signals[WHOLE_SEQ_MASS_UPDATE_REQUIRED], 0);

  g_signal_emit (G_OBJECT (seqed_widget), 
		 seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
  
  /* At this point we should be done. 
   */
  pxmchem_monomer_GPA_free (fillGPA);

  g_free (seq);

  return;
}


void
polyxmass_seqed_widget_clipboard_primary_copy (GtkWidget *widget)
{
  /* widget is the seqed_widget */

  gint start_idx = -1;
  gint end_idx = -1;

  gchar *seq = NULL;
  
  GtkClipboard *clipboard = NULL;
  
  g_assert (widget != NULL);
  g_assert (PXM_SEQED_WIDGET (widget)->canvas != NULL);

  /* This function is called each time a new selection is performed in
     the sequence editor (the canvas, actually). That selection owner
     setting, below, corresponds to the X selection mechanism that
     will set asynchronously the selected portion of the sequence as
     the PRIMARY selection, which is the one that --according to X
     tradition-- is pasted using the mouse middle button. No clipboard
     stuff here. Check the corresponding functions if you are
     interested in clipboard stuff.
  */

  clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
  
 
  /* We first check if there is a selection in seqed_widget. If
     there is such a selection, we just put it in the
     clipboard. Otherwise we do nothing, so that we do not change the
     current selection in another process if any.
  */

  if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
							     &start_idx,
							     &end_idx))
    return ;
  
  seq = 
    pxmchem_polymer_make_codes_string (PXM_SEQED_WIDGET (widget)->polymer,
				       start_idx, end_idx - 1);
  
  g_assert (seq != NULL);

  gtk_clipboard_set_text (clipboard, seq, -1);

  g_free (seq);

  return ;
}



void
polyxmass_seqed_widget_clipboard_clipboard_copy (GtkWidget *widget)
{
  /* widget is the seqed_widget */

  gint start_idx = -1;
  gint end_idx = -1;

  gchar *seq = NULL;
  
  GtkClipboard *clipboard = NULL;
  
  g_assert (widget != NULL);
  g_assert (PXM_SEQED_WIDGET (widget)->canvas != NULL);

  /* This function is called each time a new selection is performed in
     the sequence editor (the canvas, actually). That selection owner
     setting, below, corresponds to the X selection mechanism that
     will set asynchronously the selected portion of the sequence as
     the PRIMARY selection, which is the one that --according to X
     tradition-- is pasted using the mouse middle button. No clipboard
     stuff here. Check the corresponding functions if you are
     interested in clipboard stuff.
  */

  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  
 
  /* We first check if there is a selection in seqed_widget. If
     there is such a selection, we just put it in the
     clipboard. Otherwise we do nothing, so that we do not change the
     current selection in another process if any.
  */

  if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
							     &start_idx,
							     &end_idx))
    return ;
  
  seq = 
    pxmchem_polymer_make_codes_string (PXM_SEQED_WIDGET (widget)->polymer,
				       start_idx, end_idx - 1);
  
  g_assert (seq != NULL);

  gtk_clipboard_set_text (clipboard, seq, -1);

  g_free (seq);
  
  return;
}



void
polyxmass_seqed_widget_clipboard_cut (GtkWidget *widget)
{
  /* widget is the seqed_widget */
  
  g_assert (widget != NULL);
  
  if (FALSE == polyxmass_seqed_widget_get_selection_indices (widget,
							     NULL, NULL))
    {
      return;
    }
  

  /* OK, there is a selection. We first put it in the clipboard and next
     we cut it from the polymer sequence (remove the selection.
  */
  polyxmass_seqed_widget_clipboard_clipboard_copy (widget);
  /*
    debug_printf (("first index %d second index %d",
    PXM_SEQED_WIDGET (widget)sel_mnm_idx1, PXM_SEQED_WIDGET
    (widget)sel_mnm_idx2));
  */
  
  if (-1 == polyxmass_seqed_widget_remove_selected_oligomer (widget))
    return;
  else 
    {
      /* The sequence and the selection, since the selected
	 oligomer was remove, the selection and the sequence
	 actually were changed.
      */
      g_signal_emit (G_OBJECT (widget), 
		     seqed_widget_signals[WHOLE_SEQ_MASS_UPDATE_REQUIRED], 0);
      
      g_signal_emit (G_OBJECT (widget), 
		     seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
      
      return;
    }
  return;
}



void
polyxmass_seqed_widget_clipboard_primary_paste (GtkWidget *widget)
{
  /* widget is the seqed_widget */

  GtkClipboard *clipboard = NULL;



  g_assert (widget != NULL);
  g_assert (PXM_SEQED_WIDGET (widget)->canvas != NULL);



  clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);

  gtk_clipboard_request_text 
    (clipboard,
     (GtkClipboardTextReceivedFunc) polyxmass_seqed_widget_clipboard_primary_text_received,
     (gpointer) widget);
  
  return;
}


void
polyxmass_seqed_widget_clipboard_clipboard_paste (GtkWidget *widget)
{
  /* widget is the seqed_widget */

  GtkClipboard *clipboard = NULL;



  g_assert (widget != NULL);
  g_assert (PXM_SEQED_WIDGET (widget)->canvas != NULL);



  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);

  gtk_clipboard_request_text 
    (clipboard,
     (GtkClipboardTextReceivedFunc) polyxmass_seqed_widget_clipboard_clipboard_text_received,
     (gpointer) widget);
  
  return;
}
















/*
****************************************************************
*************** EVENT/SIGNAL HANDLING FUNCTIONS ****************
****************************************************************
*/

gboolean
polyxmass_seqed_widget_canvas_event (GtkWidget *widget,
				     GdkEvent *event,
				     gpointer data)
{
  /* 'widget' is the seqed_widget->canvas */

  PxmSeqedWidget *seqed_widget = data;

  GtkMenu *canvas_menu = NULL;
  
  gdouble click_x = 0;
  gdouble click_y = 0;

  gchar *label = NULL;

  gint idx = 0;
  
  PxmRect rect;
    
  g_assert (seqed_widget != NULL);


  /* store the position of the mouse event
   */
  click_x = event->button.x;
  click_y = event->button.y;

  gnome_canvas_window_to_world (GNOME_CANVAS (widget),
				event->button.x,
				event->button.y,
				&click_x,
				&click_y);

  gtk_widget_grab_focus (seqed_widget->sw);
  
  /* Analyze the event and produce the effects if required.
   */
  switch (event->type)
    {
    case GDK_BUTTON_PRESS:

      switch (event->button.button)
	{
	case 1:
	  
	  if (kbd_shift_down == TRUE)
	    {
	      /* Calculate the index of the monicon that was clicked, in
		 order to know at each instant where the "current
		 position" is in the polymer sequence. If -1 is
		 returned, that means that the mouse cursor was not
		 onto a monomer icon (in the margins of the sequence
		 drawing area, thus we just return).
	       */
	      idx = 
		polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget),
							    click_x,
							    click_y);
	      if (-1 == idx)
		return FALSE;

	      seqed_widget->last_point_1_idx = idx;

	      /* Usually, in text editors, when the cursor is
	       * somewhere and that the user presses the shift key
	       * while clicking the mouse in some other place, it is
	       * considered that the user wanted to select the text
	       * between the first cursor position and the newly
	       * pointed position.  This is what we do here.
	       */
	      seqed_widget->selection_rect.x2 = click_x;
	      seqed_widget->selection_rect.y2 = click_y;
	      
	      
	      /*
		debug_printf (("the 1_idx is %d\n", 
		seqed_widget->last_point_1_idx));
	      */
	    }
	  else
	    {
	      /* Calculate the index of the monicon that was clicked, in
		 order to know at each instant where the "current
		 position" is in the polymer sequence. If -1 is
		 returned, that means that the mouse cursor was not
		 onto a monomer icon (in the margins of the sequence
		 drawing area, thus we just return).
	       */
	      idx = 
		polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget),
							    click_x,
							    click_y);
	      if (-1 == idx)
		return FALSE;
	      
	      seqed_widget->last_point_1_idx = idx;
		
	      /* Left mouse button is clicked without ongoing
	       * selection, so store the impact point's coordinates
	       * into the GTK_WIDGET (seqed_widget)'s selrect first selpoint member
	       * data.
	       */
	      seqed_widget->selection_rect.x1 = click_x;
	      seqed_widget->selection_rect.y1 = click_y;
	      
	      seqed_widget->selection_rect.x2 = click_x;
	      seqed_widget->selection_rect.y2 = click_y;

	      /*
		debug_printf (("the 1_idx is %d\n", 
		seqed_widget->last_point_1_idx));
	      */
	    }
	  
	  /* The selection polygon drawing function below will calculate
	     the indices of the currently selected first and last
	     monomers.
	  */
	  polyxmass_seqed_widget_draw_sel_polygon (GTK_WIDGET (seqed_widget));
	    
	  /* Since we know where the impact occurred from a monicon
	   * index standpoint, we can ask that the cursor be drawn at
	   * this precise position. Drawing of the cursor can happen
	   * thanks to updating the seqed_widget->last_point_1_idx datum,
	   * which is then used to calculate precise pixel position for
	   * the cursor.
	   */
	  polyxmass_seqed_widget_draw_cursor (GTK_WIDGET (seqed_widget));
	  
	  /* The selection.
	   */
	  g_signal_emit (G_OBJECT (seqed_widget), 
			 seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
	  
	  /* We might well be starting a mouse_dragging operation, if the
	   * user wants to select a portion of the sequence. Dragging
	   * will be set to FALSE as soon as the left button is
	   * released. 
	   */
	  mouse_dragging = TRUE;
	  break;

	case 2:

	  /* We are asking that a paste be done with the contents of
	     the PRIMARY selection clipboard, the X traditional
	     selection. What we do here is acutally ask for that
	     data. We are not pasting that. The real pasting activity
	     will be performed when the data arrives to our process,
	     and at that time only the "selection received" signal
	     will be sent to our process. It is the handler of that
	     signal (that we have coded ourselves) that will actually
	     do the pasting with the received data. (Remember that the
	     selection/paste stuff in X (traditional) is asynchronous,
	     because X may be distributed across the network, which
	     may have some latency.)
	  */
	  polyxmass_seqed_widget_clipboard_primary_paste (GTK_WIDGET (seqed_widget));
	  break;
	  
	case 3:

	  /* We'll maybe need the monomer index of the monomer that
	     lies beneath the mouse cursor right now, so calculate that
	     right away.
	  */
	  rect.x1 = click_x;
	  rect.y1 = click_y;
	  
	  /* Calculate the index of the monicon that was clicked, in
	     order to know at each instant where the "current
	     position" is in the polymer sequence. For this specific
	     case, we are OK if -1 is returned.
	  */
	  seqed_widget->last_point_3_idx =
	    polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget),
							rect.x1,
							rect.y1);
	  
	  /*
	    debug_printf (("the 3_idx is %d\n", 
	    seqed_widget->last_point_3_idx));
	  */

	  /* We want to display a contextual menu now. Note that a
	     pointer to the menu is set by the setup function that
	     sets up the window in which the seqed_widget is
	     installed. This might be, for example,
	     polyxedit_seqed_wnd_setup ().

	     But if the window that needs this seqed_widget does not
	     need a contextual menu, the 'canvas_menu' variable below
	     can be NULL. So, do not assert anything against the value
	     returned by the call below.
	   */
	  canvas_menu = 
	    (GtkMenu *) g_object_get_data (G_OBJECT (widget), 
					   "canvas_menu");
	  if (canvas_menu != NULL)
	    {
	      /* The 1 below, as sixth param is to tell that the first button
		 of the mouse is going to be used to trigger the "activate"
		 signal of the menu item to be selected.
	      */
	      gtk_menu_popup (canvas_menu, NULL, NULL, NULL, NULL, 
			      1  /* ((GdkEventButton *)event)->button */, 
			      ((GdkEventButton *)event)->time);
	      break;
	    }
	}
      
      /* end of 
	 switch (event->button.button)
      */
      break;

      
    case GDK_MOTION_NOTIFY:

      /* It is interesting to trap this message:
       *
       * 1. if we are pressing the left mouse button, in which 
       * case the "mouse_dragging" variable was set to TRUE.
       *
       * 2. because we just want to update the index of the last
       * monomer onto which the mouse was passed over.
       */
      if (mouse_dragging && (event->motion.state & GDK_BUTTON1_MASK))
	{
	  /* Calculate the monicon index where we are mouse_dragging the mouse
	     so that later the cursor drawing function can do its job
	     by calculating precisely the cursor position from the
	     seqed_widget->last_point_1_idx value. If -1 is returned,
	     that means that the mouse cursor was not onto a monomer
	     icon (in the margins of the sequence drawing area, thus
	     we just return).
	   */
	  idx = 
	    polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget),
							click_x,
							click_y);
	  if (-1 == idx)
	    return FALSE;
	  

	  seqed_widget->last_point_1_idx = idx;
	  
	  /* Update the mouse cursor position because we need to use
	   * new position to draw the selection polygon.
	   */
	  seqed_widget->selection_rect.x2 = click_x;
	  seqed_widget->selection_rect.y2 = click_y;

	  /* Draw the cursor, so that we see the movement of the mouse
	   * real time. This function takes as parameter the pointer
	   * to the polymer sequence context that we got in this
	   * function as param "data".
	   */
	  polyxmass_seqed_widget_draw_cursor (GTK_WIDGET (seqed_widget));
	  
	  /* Draw the selection polygon, by specifying in the unique
	   * parameter a pointer to the polymer sequence context,
	   * where the called function can get values to guide its
	   * polygon drawing process.
	   */
	  polyxmass_seqed_widget_draw_sel_polygon (GTK_WIDGET (seqed_widget));
	  
	  /* Deal with scrolling of the window if pointer approaches
	   * the top/bottom borders of the window and that scrolling
	   * is allowed.
	   */
	  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget),
					   &(seqed_widget->canvas_x_offset),
					   &(seqed_widget->canvas_y_offset));

	  if (click_y >= seqed_widget->canvas_y_offset
	      + seqed_widget->sw_height - 
	      seqed_widget->monicon_size)
	    {
	      /* We are near the bottom of viewable sequence, so we
	       * ask for a scroll to happen toward higher monomer
	       * index (scroll downward).
	       */
	      gnome_canvas_scroll_to (GNOME_CANVAS (widget),
				      seqed_widget->canvas_x_offset,
				      seqed_widget->canvas_y_offset +
				      seqed_widget->monicon_size);
	    }
	  else if (click_y <= seqed_widget->canvas_y_offset
		   + seqed_widget->monicon_size / 3)
	    {
	      /* We are near the top of viewable sequence, so we ask
	       * for a scroll to happen toward lower monomer index
	       * (scroll upward).
	       */
	      gnome_canvas_scroll_to (GNOME_CANVAS (seqed_widget->
						    canvas),
				      seqed_widget->canvas_x_offset,
				      seqed_widget->canvas_y_offset -
				      seqed_widget->monicon_size);
	    }

	  /* The selection.
	   */
	  g_signal_emit (G_OBJECT (seqed_widget), 
			 seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
	}
      
      seqed_widget->last_mouse_idx =
	polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget), 
						    click_x, click_y);
      break;
      
    case GDK_BUTTON_RELEASE:
      
      switch (event->button.button)
	{
	case 1:
	  
	  /* Calculate the monicon index where we are unclicking the mouse
	   * so that later the cursor drawing function can do its job by
	   * calculating precisely the cursor position from the
	   * seqed_widget->last_point_1_idx value. If -1 is returned,
	     that means that the mouse cursor was not onto a monomer
	     icon (in the margins of the sequence drawing area, thus
	     we just return).
	   */
	  idx =
	    polyxmass_seqed_widget_get_mnm_idx_with_xy (GTK_WIDGET (seqed_widget),
							click_x,
							click_y);
	  if (-1 == idx)
	    return FALSE;

	  /* However, that is not enough. Imagine that the user left
	     clicked the mouse in the margins and dragged it still
	     clicked to the sequence. At that moment she would release
	     the mouse button. We are here. The value calculated above
	     is not going to be -1 because we are IN the sequence
	     drawing area. But we started mouse_dragging the mouse out of
	     the sequence area. So the system is crippled because we
	     are going to ask that a selection polygon be drawn, but
	     we do not know from where !
	  */

	  seqed_widget->last_point_1_idx = idx;
	  
	  /* Store the position of the unclicking of the mouse button
	   * so that we always know where we are in the polymer sequence.
	   */
	  seqed_widget->selection_rect.x2 = click_x;
	  seqed_widget->selection_rect.y2 = click_y;


	  /*
	    debug_printf (("the 1_idx is %d\n", 
	    seqed_widget->last_point_1_idx));
	  */
	  
	  /* Draw the cursor, so that we see the last position of the
	   * mouse real time. This function takes as parameter the pointer
	   * to the polymer context that we got in this function as param
	   * "data".
	   */
	  polyxmass_seqed_widget_draw_cursor (GTK_WIDGET (seqed_widget));

	  /* Draw the selection polygon, because this way, if one exists
	   * and we are clicking somewhere else, we can destroy it.
	   */
	  polyxmass_seqed_widget_draw_sel_polygon (GTK_WIDGET (seqed_widget));
      
	  /* Set "mouse_dragging" to FALSE to clearly indicate that
	   * the mouse was released and mouse_dragging (if any) operation is
	   * terminated.
	   */
	  mouse_dragging = FALSE;

	  /* The selection.
	   */
	  g_signal_emit (G_OBJECT (seqed_widget), 
			 seqed_widget_signals[SELEC_SEQ_MASS_UPDATE_REQUIRED], 0);
	  
	  /* We can automatically make the selection owner...  This will
	     allow next putting on the primary clipboard the sequence
	     corresponding to the selection (if any only).
	  */
	  polyxmass_seqed_widget_clipboard_primary_copy (GTK_WIDGET (seqed_widget)) ;
	  break;
	  
	default:
	  break;
	}
      /* end of 
	 switch (event->button.button)
      */


    default:
      break;
    }
  /* end 
     switch (event->type)
  */
      
  /* Set the monomer index that was "moused" (passed over with the
     mouse) to the lable sitting on top of the sequence
     editor. Attention we want monomer positions and not monomer
     indexes, which is why we add 1 before converting to string. Only
     if the index is > -1. (Remember that -1 indicates that the cursor
     is in the margin of the sequence display area.) That's true for
     all positions but not the virtual monomer that is right of the
     very last monomer of the sequence. This is why we do not display
     anything if the last_mouse_idx is greater than the index of the
     last monomer.
   */
  if (seqed_widget->last_mouse_idx > -1
      && seqed_widget->last_mouse_idx < seqed_widget->polymer->monomerGPA->len)
    label = g_strdup_printf ("%d", seqed_widget->last_mouse_idx + 1);
  else
    label = g_strdup (" ");
  
  gtk_label_set_text (GTK_LABEL(seqed_widget->monomer_pos_label),
		      label);
  g_free (label);
  
  /* Let the canvas do the actual pixel graphics redrawing work.
   */
  return FALSE;
}



gboolean 
polyxmass_seqed_widget_sw_size_allocate (GtkWidget *widget, 
					 GtkAllocation *allocation,
					 gpointer data)
{
  /* This function is meant to provide to the sequence editor context
   * the dimensions of the scrolled window "real time". We cannot just
   * try to get the dimensions of the window, because the scrolled
   * window that contains the gnome canvas is not the only widget in
   * the window.
   */

  /* Strangely, this function seems to get called even when the mouse
   * is not resizing the window, but for example, simply going from
   * one monomer canvas item to another. This means that there is a huge
   * overhead I want to eliminate : I first check if the size
   * of the window effectively changed. 
   *
   * If not: I return.
   *
   * If yes: I grasp the new dimensions, put them to proper variables
   * and also set to TRUE a file-scope variable that will let the
   * "after-connected handler" know that the scrolled window size
   * effectively changed.  (see
   * polyxedit_seqed_wnd_ensure_select_cursor_visible () for details.
   */
  PxmSeqedWidget *seqed_widget = PXM_SEQED_WIDGET (data);

  g_assert (widget != NULL);
  g_assert (seqed_widget != NULL);


  /* This function is called (I do not know why) each time the mouse
     cursor goes from a monicon to another one. Of course I don't want
     this, so I only update the seqed_widget->canvas_scroll_wndwin
     dimensions if they are changed.
   */
  if ((seqed_widget->sw_width != allocation->width)
      || (seqed_widget->sw_height != allocation->height))
    {
      /* If the width of the window goes less than
       * (seqed_widget->left_margin + seqed_widget->monicon_size) it makes
       * a segfault, so prevent this behaviour.
       */
      if (allocation->width > 
	  (seqed_widget->left_margin + seqed_widget->monicon_size))
	{
	  seqed_widget->sw_width = allocation->width;
	}
      
      seqed_widget->sw_height = allocation->height;

      scroll_size_changed = TRUE;

      if (-1 == polyxmass_seqed_widget_redraw_sequence (GTK_WIDGET (seqed_widget)))
	{
	  g_warning (_("%s@%d: failed to draw the sequence\n"),
		     __FILE__, __LINE__);
	  
	  return TRUE;
	}

      polyxmass_seqed_widget_update_sel_polygon (GTK_WIDGET (seqed_widget));
      polyxmass_seqed_widget_draw_cursor (GTK_WIDGET (seqed_widget));
    }

  return TRUE;
}


gboolean
polyxmass_seqed_widget_canvas_item_event (GnomeCanvasItem *canvas_item,
					  GdkEvent *event, gpointer data)
{
  /* Data is the GTK_WIDGET (seqed_widget).
   */

  /* This function handler will be used later to display data specific
   * of each canvas item (like a id card of the monomer that is 
   * represented by the canvas item, in the form of a tooltip, for 
   * example).
   */
  /*
    debug_printf (("polyxmass_seqed_widget_canvas_item_event\n"));
  */
  
  return FALSE;
}



gboolean
polyxmass_seqed_widget_sel_polygon_event (GtkWidget *widget,
					  GdkEvent *event,
					  gpointer data)
{
  /*
    debug_printf (("polyxmass_seqed_widget_sel_polygon_event\n"));
  */
  
  return FALSE;
}




