/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "polyxmass-ui-note.h"
#include "polyxmass-ui-seqed-widget.h"

enum
  {
    COLUMN_NOTE_NAME = 0,
    COLUMN_NOTE_VALUE,
    COLUMN_NOTE_VALUE_TYPE,
    COLUMN_NOTE_POINTER,
    COLUMN_NOTEVAL_POINTER,
    COLUMN_PROP_POINTER,
    COLUMN_NOTE_COL_COUNT
  };

  
  

/* Window setup.
 */
GtkWidget *
polyxmass_note_setup_wnd (GtkWidget *seqed_widget,
			  gint target,
			  gint start_idx,
			  gint end_idx,
			  gint index)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;
  GtkWidget *vbox = NULL;
  
  PxmMonomer *monomer = NULL;
  PxmPolymer *polymer = NULL;
  
  
  gint *help_int = NULL;


  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  /* This window is set up because notes are going edited either for
     the polymer sequence as a whole (target == NOTE_TARGET_POLYMER),
     or for a determinated monomer (target == NOTE_TARGET_MONOMER, and
     index is the index of the monomer in question).
  */
  
  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxmass-note.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "note_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "note_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed to create the note management window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }


  /* Now that we have the window pointer, immediately set to it the
     pointer of the seqed_widget as a datum, so that we can get to it
     later.
   */
  g_object_set_data (G_OBJECT (window), "seqed_widget", seqed_widget);
  

  /* We want to make sure that the different parameters are always 
     available to this window.
  */
  help_int = g_malloc0 (sizeof (gint));
  *help_int = target;
  
  g_object_set_data_full (G_OBJECT (window), "TARGET", help_int,
			  (GDestroyNotify) g_free);
  
  help_int = g_malloc0 (sizeof (gint));
  *help_int = index;
  
  g_object_set_data_full (G_OBJECT (window), "INDEX", help_int,
			  (GDestroyNotify) g_free);
  
  help_int = g_malloc0 (sizeof (gint));
  *help_int = start_idx;
  
  g_object_set_data_full (G_OBJECT (window), "START_IDX", help_int,
			  (GDestroyNotify) g_free);
  
  help_int = g_malloc0 (sizeof (gint));
  *help_int = end_idx;
  
  g_object_set_data_full (G_OBJECT (window), "END_IDX", help_int,
			  (GDestroyNotify) g_free);
  

  /* There is a GtkEntry in the window that is commonly used to
     display messages. We need to set this pointer to window
     immediately because we'll need this entry.
  */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);

  /* The hbox where the polymer/monomer data widgets are located that
     we may make invisible by clicking onto the button on its
     container frame (see below).
  */
  widget = glade_xml_get_widget (xml, "polymer_monomer_data_hbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polymer_monomer_data_hbox", widget);

  /* The seqed_widget must have a pointer to the polymer which in turn
     must have a plminfo member, which in turn might have a name set
     to it. If so we put that name in the corresponding widget!
  */
  
  /* Set the polymer sequence name to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);
  
  if (polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			polymer->plminfo->name);
  else 
    gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));
  

  /* Set the sequence widget id number (its pointer) to its
     correspondent GtkEntry.
  */
  widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),  "polseq_id_number_entry", widget);

  help = g_strdup_printf ("%p", seqed_widget);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  widget = glade_xml_get_widget (xml, "monomer_code_pos_label");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "monomer_code_pos_label", widget);


  widget = glade_xml_get_widget (xml, "monomer_range_label");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "monomer_range_label", widget);


  if (target == NOTE_TARGET_MONOMER_SINGLE 
      || target == NOTE_TARGET_MONOMER_RANGE)
    {
      /* Whatever the kind of monomer target, we have to set the code
	 of the reference monomer and its position to the label. That
	 monomer is the monomer from which the user will select a note
	 to remove, if removal is concerned.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "monomer_code_pos_label");
      g_assert (widget != NULL);
      
      /* Evidently, the index of the monomer of which the notes are to
	 be displayed, that is passed as parameter, must not be
	 greater or equal to the nuber of monomers in the polymer
	 sequence.
      */
      g_assert (index < polymer->monomerGPA->len);
      
      monomer = g_ptr_array_index (polymer->monomerGPA, index);
      g_assert (monomer != NULL);
      
      help = g_strdup_printf ("%s / %d", monomer->code, index + 1 );
      
      gtk_label_set_text (GTK_LABEL (widget), help);
      
      g_free (help);

      /* Now, if the target is a RANGE, then we'll have to synthesize a
	 string to show that range in the relevant label.
      */
      if (target == NOTE_TARGET_MONOMER_RANGE)
	{
	  /* We want to show the positions and not the indices, hence
	     the '+ 1' below.
	  */
	  help = g_strdup_printf ("[%d-%d]", start_idx + 1, end_idx + 1);
	  
	  widget = g_object_get_data (G_OBJECT (window), 
				      "monomer_range_label");
	  g_assert (widget != NULL);

	  gtk_label_set_text (GTK_LABEL (widget), help);

	  g_free (help);
	}
    }
  
  widget = glade_xml_get_widget (xml, "note_hpaned");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "note_hpaned", widget);

  vbox = glade_xml_get_widget (xml, "notes_treeview_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "notes_treeview_vbox", vbox);

  widget = glade_xml_get_widget (xml, "note_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "note_name_entry", widget);

  widget = glade_xml_get_widget (xml, "value_type_str_radiobutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_type_str_radiobutton", widget);

  widget = glade_xml_get_widget (xml, "value_type_int_radiobutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_type_int_radiobutton", widget);

  widget = glade_xml_get_widget (xml, "value_type_dbl_radiobutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_type_dbl_radiobutton", widget);

  widget = glade_xml_get_widget (xml, "value_contents_textview");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_contents_textview", widget);

  widget = glade_xml_get_widget (xml, "value_add_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_add_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_note_wnd_value_add_button), 
		    window);

  widget = glade_xml_get_widget (xml, "value_apply_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "value_apply_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_note_wnd_value_apply_button), 
		    window);

  widget = glade_xml_get_widget (xml, "note_add_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "note_add_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_note_wnd_note_add_button), 
		    window);

  widget = glade_xml_get_widget (xml, "note_apply_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "note_apply_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_note_wnd_note_apply_button), 
		    window);

  widget = glade_xml_get_widget (xml, "notes_menubar");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "notes_menubar", widget);

  /* Menu items in the menu bar.
   */
  widget = glade_xml_get_widget (xml, "selected_item_remove_single");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "selected_item_remove_single", widget);
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK 
		    (polyxmass_note_wnd_selected_item_remove_single),
		    window);

  widget = glade_xml_get_widget (xml, "selected_item_remove_range");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "selected_item_remove_range", widget);
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK 
		    (polyxmass_note_wnd_selected_item_remove_range),
		    window);
  if (target == NOTE_TARGET_MONOMER_SINGLE 
      || target == NOTE_TARGET_POLYMER)
    gtk_widget_set_sensitive (widget, FALSE);
  
  widget = glade_xml_get_widget (xml, "selected_item_propagate");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "selected_item_propagate", widget);
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK 
		    (polyxmass_note_wnd_selected_item_propagate),
		    window);
  if (target == NOTE_TARGET_MONOMER_SINGLE 
      || target == NOTE_TARGET_POLYMER)
    gtk_widget_set_sensitive (widget, FALSE);


  /******************** TEMPLATE *******************************
  widget = glade_xml_get_widget (xml, "");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "", widget);
  */

  /* We don't need the GladeXML object any more, so unref it to save
     some memory
  */
  g_object_unref (G_OBJECT (xml));



  /* Now that all the widgets have been created, we need to fill the
     listview of notes for the correct target. We know that notes are
     encapsulated into PxmProp object named "NOTE". These PxmProp
     objects are stored in each target's propGPA standard
     GPtrArray. All we need to have is a pointer to that GPtrArray of
     PxmProp objects.
  */

  if (target == NOTE_TARGET_MONOMER_SINGLE
      || target == NOTE_TARGET_MONOMER_RANGE)
    {
      if (-1 == 
	  polyxmass_note_wnd_setup_note_treeview (window,
						  monomer->propGPA))
	{
	  g_critical (_("%s@%d: failed to fill treeview with monomer note(s)\n"),
		 __FILE__, __LINE__);
	  
	  gtk_object_destroy (GTK_OBJECT (window));
	  
	  return NULL;
	}
    }
  else if (target == NOTE_TARGET_POLYMER)
    {
      if (-1 == 
	  polyxmass_note_wnd_setup_note_treeview (window,
						  polymer->propGPA))
	{
	  g_critical (_("%s@%d: failed to fill treeview with polymer note(s)\n"),
		 __FILE__, __LINE__);
	  
	  g_object_unref (G_OBJECT (window));

	  return NULL;
	}
    }

  
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxmass_note_wnd_delete_event),
		    seqed_widget);
  

  /* Set this window pointer as a full datum to the seqed_widget , so
     that when it is destroyed, this window is destroyed also. There
     might be more than one note window opened for a given
     seqed_widget, and we do not want that the second window destroys
     the datum name of the first window, so we create an unambiguous
     datum name each time.
  */
  help = g_strdup_printf ("note_wnd-%p", window);
  
  g_object_set_data_full (G_OBJECT (seqed_widget),
			  help, GTK_WIDGET (window), 
			  (GDestroyNotify) polyxmass_note_wnd_really_close);
  
  g_free (help);
  
  
  /* Finally we can show the whole sequence editor.
   */
  gtk_widget_show_all (window);
  
  return window;
}





/* Note treeview setup and housekeeping functions.
 */
gint
polyxmass_note_wnd_setup_note_treeview (GtkWidget *widget,
					GPtrArray *GPA)
{
  GtkWidget *window = widget;
  GtkWidget *vbox = NULL;
  GtkWidget *sw = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;
  GtkTreeSelection* selection = NULL;


  gint col_offset = 0;


  g_assert (window != NULL);
  g_assert (GPA != NULL);
  

  /* Get a pointer to the vertical box where we'll pack the scrolled window
     and its treeview.
  */
  vbox = g_object_get_data (G_OBJECT (window), "notes_treeview_vbox");
  g_assert (vbox != NULL);
  
  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxmass_note_wnd_setup_note_treemodel (GPA);
  
  /* Set to the model a datum with a pointer to GPA, so that
   * the array of PxmProp that has been used to fill the model
   * is accessible later. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "propGPA", GPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove/add item handler).
   */
  g_object_set_data (G_OBJECT (window), "notes_treeview", treeview);

  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);

  /* "changed" is emitted by the selection object in the treeview each
     time the selection changes.
  */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_signal_connect 
    (G_OBJECT (selection), "changed", 
     G_CALLBACK 
     (polyxmass_note_wnd_treeview_selection_changed),
     window);
    
  /*Note name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     COLUMN_NOTE_NAME);
  
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, 
						 "text",
						 
						 COLUMN_NOTE_NAME,
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /*Note value column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     COLUMN_NOTE_NAME);
  
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Value"),
						 renderer, 
						 "text",
						 
						 COLUMN_NOTE_VALUE,
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /*Note value type column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     COLUMN_NOTE_NAME);
  
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Type"),
						 renderer, 
						 "text",
						 
						 COLUMN_NOTE_VALUE_TYPE,
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return 1;
}


GtkTreeModel *
polyxmass_note_wnd_setup_note_treemodel (GPtrArray *GPA)
{
  GtkTreeStore *model = NULL;
  GtkTreeIter treeiter;
  
  gint idx = 0;
  gint iter = 0;

  gchar *help = NULL;
  
  PxmProp *prop = NULL;
  PxmNote *note = NULL;
  PxmNoteVal *noteval = NULL;
  

  model = gtk_tree_store_new (COLUMN_NOTE_COL_COUNT,
			      /* note name */
			      G_TYPE_STRING,
			      /* note value */
			      G_TYPE_STRING,
			      /* note value type */
			      G_TYPE_STRING,
			      /* note pointer */
			      G_TYPE_POINTER,
			      /* noteval pointer */
			      G_TYPE_POINTER,
			      /* prop pointer */
			      G_TYPE_POINTER);


  /* The GPA that we get as parameter contains a number of PxmProp objects,
     and we have to select only the ones that have "NOTE" as a name.
  */
  while (1)
    {
      prop = libpolyxmass_prop_find_prop (GPA,
				      &idx,
				      NULL,
				      "NOTE",
				      NULL,
				      PXM_CMP_NO_DEEP);
      if (prop == NULL)
	break;

      /* We found a prop object named "NOTE", all we have is to extract the
	 PxmNote instance and put the relevant data to the treeview.
      */
    
      note = (PxmNote *) prop->data;
      g_assert (note != NULL);
      g_assert (note->name != NULL);
      g_assert (note->notevalGPA != NULL);
    
      gtk_tree_store_append (model, &treeiter, NULL);

      gtk_tree_store_set (model, 
			  &treeiter,
			  COLUMN_NOTE_NAME, note->name,
			  COLUMN_NOTE_POINTER, note,
			  COLUMN_PROP_POINTER, prop,
			  -1);
    
      /* We now have to iterate in the array of noteval objects and
	 for each object insert the data. Note that a note may have an
	 empty array of noteval objects.
      */
      for (iter = 0; iter < note->notevalGPA->len; iter++)
	{
	  GtkTreeIter treejter;


	  noteval = g_ptr_array_index (note->notevalGPA, iter);
	  g_assert (noteval != NULL);
	  if (noteval->value == NULL)
	    continue;
	  
	  if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	    {
	      gtk_tree_store_append (model, &treejter, &treeiter);
	      gtk_tree_store_set (model, 
				  &treejter,
				  COLUMN_NOTE_VALUE, noteval->value,
				  COLUMN_NOTE_VALUE_TYPE, "str",
				  COLUMN_NOTE_POINTER, note,
				  COLUMN_NOTEVAL_POINTER, noteval,
				  COLUMN_PROP_POINTER, prop,
				  -1);

	      continue;
	    }
	  
	  if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	    {
	      help = g_strdup_printf ("%d", *(gint*) noteval->value);
	      
	      gtk_tree_store_append (model, &treejter, &treeiter);
	      gtk_tree_store_set (model, 
				  &treejter,
				  COLUMN_NOTE_VALUE, help,
				  COLUMN_NOTE_VALUE_TYPE, "int",
				  COLUMN_NOTE_POINTER, note,
				  COLUMN_NOTEVAL_POINTER, noteval,
				  COLUMN_PROP_POINTER, prop,
				  -1);
	      g_free (help);

	      continue;
	    }
	  
	  if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	    {
	      help = g_strdup_printf ("%.9f", 
				      *(gdouble*) noteval->value);
	      
	      gtk_tree_store_append (model, &treejter, &treeiter);
	      gtk_tree_store_set (model, 
				  &treejter,
				  COLUMN_NOTE_VALUE, help,
				  COLUMN_NOTE_VALUE_TYPE, "dbl",
				  COLUMN_NOTE_POINTER, note,
				  COLUMN_NOTEVAL_POINTER, noteval,
				  COLUMN_PROP_POINTER, prop,
				  -1);
	      g_free (help);

	      continue;
	    }
	  else
	    g_error (_("%s@%d: the type of the note's value is incorrect.\n"),
		   __FILE__, __LINE__);
	}
	  
      /* Go to the next prop object if there is any, and ask that it
	 be searched in GPA starting from the next prop object,
	 because we do not want to iterate again on the same prop. If,
	 after incrementation, idx >= GPA->len, then NULL is returned
	 and we can safely go out of this while loop.
      */
      idx++;
    }
  
  return GTK_TREE_MODEL (model);
}


void
polyxmass_note_wnd_treeview_selection_changed (GtkTreeSelection 
						     *selection,
						     gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = data;
  GtkWidget *note_name_entry = NULL;
  GtkWidget *type_radiobutton = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter_start;
  GtkTextIter text_iter_end;
  GtkTextBuffer *buffer = NULL;

  GtkTreeModel *model = NULL;
  GtkTreePath *path = NULL;
  GtkTreeView *treeview;
  GtkTreeIter treeiter;

  gboolean valid = FALSE;

  gchar *help = NULL;
  
  gint idx = -1;
  gint depth = -1;
  
  PxmNote *note = NULL;
  PxmNoteVal *noteval = NULL;
  

  /* The selection in the treeview of currently available notes has
     changed, so we have to set its member data in their respective
     widgets. This only if the item that has been selected is actually
     a pxmnoteval item, and not only the "header" note name.
  */

  g_assert (selection != NULL);
  g_assert (window != NULL);
  
  /* The selection object has a pointer to its related treeview.
   */
  treeview = gtk_tree_selection_get_tree_view (selection);
  
  /* Get the model from the treeview, so that we can iterate later
     in the treeview.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we should clean the different
	 note-related data from their respective widget.
       */
      polyxmass_note_wnd_reset_note_related_widgets (window);
      
      return;
    }
  
  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  /* Get the index of the currently selected item.
   */
  idx = gtk_tree_path_get_indices (path) [0];

  /* Get the depth of the path, so that we know if we are dealing
     with the name of the note or one of the possibly-more-than-one
     values of that same note.
  */
  depth = gtk_tree_path_get_depth (path);

  if (depth == 2)
    {
      idx = gtk_tree_path_get_indices (path) [1];
    }
  
  /* We can free the path now.
   */
  gtk_tree_path_free (path);
  
  /* Get a pointer to all the widgets we are going to use next:
   */
  
  note_name_entry = g_object_get_data (G_OBJECT (window),
				       "note_name_entry");
  g_assert (note_name_entry != NULL);
  
  textview = g_object_get_data (G_OBJECT (window),
				"value_contents_textview");
  g_assert (textview != NULL);
  
  buffer = gtk_text_view_get_buffer (textview);
  
  /* If depth is 1, just set the name of the Note to its GtkEntry.
   */
  if (depth == 1)
    {
      /* First of all we want to make sensitive the menu item
	 that asks for propagation of the selected item in the
	 monomer range (indeed we can propagate note objects!).
      */
      widget = g_object_get_data (G_OBJECT (window),
				"selected_item_propagate");
      g_assert (widget != NULL);
      
      gtk_widget_set_sensitive (widget, TRUE);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTE_POINTER, &note,
			  -1);
      
      g_assert (note != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (note_name_entry), note->name);
      
      /* Set the other fields to nothing.
       */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);
      gtk_text_buffer_delete (buffer, &text_iter_start, &text_iter_end);
    }
  
  else if (depth == 2)
    {
      /* First of all we want to make unsensitive the menu item
	 that asks for propagation of the selected item in the
	 monomer range (indeed we cannot propagate noteval objects, only
	 note objects).
      */
      widget = g_object_get_data (G_OBJECT (window),
				"selected_item_propagate");
      g_assert (widget != NULL);
      
      gtk_widget_set_sensitive (widget, FALSE);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTE_POINTER, &note,
			  -1);
      
      g_assert (note != NULL);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTEVAL_POINTER, &noteval,
			  -1);
      
      g_assert (noteval != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (note_name_entry), note->name);
      
      /* We have to select the proper radiobutton, depending on the
	 value of the noteval->type variable. Since the glade file
	 states that they all belong to the same group, if one is
	 activated, the others are deactivated automatically.
      */
      if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	{
	  type_radiobutton = 
	    g_object_get_data (G_OBJECT (window),
			       "value_type_str_radiobutton");
	  g_assert (type_radiobutton != NULL);
	  
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (type_radiobutton),
					TRUE);
	}
      else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	{
	  type_radiobutton = 
	    g_object_get_data (G_OBJECT (window),
			       "value_type_int_radiobutton");
	  g_assert (type_radiobutton != NULL);
	  
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (type_radiobutton),
					TRUE);
	}
      else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	{
	  type_radiobutton = 
	    g_object_get_data (G_OBJECT (window),
			       "value_type_dbl_radiobutton");
	  g_assert (type_radiobutton != NULL);
	  
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (type_radiobutton),
					TRUE);
	}

      /* And finally, we have to set the noteval->value to the textview.
	 But we have to convert it first.
      */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);
      gtk_text_buffer_delete (buffer, &text_iter_start, &text_iter_end);

      if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
	{
	  gtk_text_buffer_insert (buffer, &text_iter_start, 
				  (gchar *) noteval->value, -1);  
	}
      else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
	{ 
	  help = g_strdup_printf ("%d", *(gint*) noteval->value);
	  
	  gtk_text_buffer_insert (buffer, &text_iter_start, 
				  help, -1);  
	  g_free (help);
	}
      else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
	{
	  help = g_strdup_printf ("%.9f", 
				  *(gdouble*) noteval->value);
	  
	  gtk_text_buffer_insert (buffer, &text_iter_start, 
				  help, -1);  
	  g_free (help);
	}
    }
	 
  return;
}


void
polyxmass_note_wnd_value_add_button (GtkWidget *widget,
				     gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *type_radiobutton = NULL;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeIter treejter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter_start;
  GtkTextIter text_iter_end;
  GtkTextBuffer *buffer = NULL;


  gboolean valid = FALSE;

  gint depth = 0;

  gchar *help = NULL;
  gchar *text = NULL;

  PxmProp *prop = NULL;
  PxmNote *note = NULL;
  PxmNoteVal *noteval = NULL;
  
  

  /* We can only add a value if a note is currently selected in the
     treeview, otherwise one asks to which note the value is to be
     added?
  */
  g_assert (window != NULL);

  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) window,
	 _("Select or create a note first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);
      
      return;
    }

  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  /* From the treeiter that we got we can extract the pointer to the
     note object that was used to make that selected item (be the item
     a depth-1 or depth-2 item; see the tree model).
  */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_NOTE_POINTER, &note,
		      -1);
  g_assert (note != NULL);
  
  /* Now that we know that the note is OK, we can start analyzing 
     the data that will make up a new value. Adding a new value requires
     that a new PxmNoteVal object be allocated and that the following
     bits of data are correctly set:

     - the value type (string, int or double)
     - the value contents.
  */
  /* At this moment, allocating a new noteval object makes sense.
   */
  noteval = libpolyxmass_noteval_new ();
  
  /* Get the note value type immediately into the allocated noteval.
   */
  polyxmass_note_wnd_get_active_valtype_radiobutton (window,
						     &type_radiobutton,
						     &noteval->type);
  
  g_assert (type_radiobutton != NULL);
  
  textview = g_object_get_data (G_OBJECT (window),
				"value_contents_textview");
  g_assert (textview != NULL);
    
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);

  text = gtk_text_buffer_get_text (buffer, 
				   &text_iter_start, &text_iter_end,
				   FALSE);
  g_assert (text != NULL);

  /* We now have our PxmNoteValue->value, in the form of text. But we 
     must convert it to the right type before.
  */
  if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
    {
      noteval->value = (gpointer) g_strdup (text);
    }
  else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
    {
      noteval->value = g_malloc0 (sizeof (gint));
      
      if (FALSE == 
	  libpolyxmass_globals_strtoi (text, (gint *) noteval->value, 10))
	{
	  libpolyxmass_noteval_free (noteval);
	  g_free (text);
	  
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("The value content is not an integer. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return;
	}
      
      /* At this point the noteval->value variable holds the proper
	 value!
      */
    }
  else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
    {
      noteval->value = g_malloc0 (sizeof (gdouble));

      if (FALSE == 
	  libpolyxmass_globals_strtod (text, (gdouble *) noteval->value))
	{
	  libpolyxmass_noteval_free (noteval);
	  g_free (text);
	  
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("The value content is not a double. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return;
	}      
      /* At this point the noteval->value variable holds the proper
	 value!
      */
    }
  else
    g_assert_not_reached ();

  /* Later we will need to now the value of noteval->type so as to 
     display it in the treeview like a string. So prepare that string
     immediately.
  */
  if (noteval->type == PXM_NOTE_VAL_TYPE_STR)
    help = g_strdup ("str");
  else if (noteval->type == PXM_NOTE_VAL_TYPE_INT)
    help = g_strdup ("int");
  else if (noteval->type == PXM_NOTE_VAL_TYPE_DBL)
    help = g_strdup ("dbl");
  else
    g_assert_not_reached ();
  
/* At this point our new value is fully qualified, we have to add
     this PxmNoteValue instance to the array of the note.
  */
  g_ptr_array_add (note->notevalGPA, noteval);
  
  /* Next, we have to add that new noteval instance to the currently
     selected treeview item (note depth 1). If the selected item (we
     got the path to it at the beginning of this function) is of depth
     1, that means that a note is currently selected, so we just
     append the new item to it. If the depth is 2, that means that a
     notevalue item is selected, and we need to first get grasp of the
     parent (the note) before appending a new item.
  */
  depth = gtk_tree_path_get_depth (path);

  /* We can free the path now.
   */
  gtk_tree_path_free (path);

  if (depth == 1)
    {
      /* Append the item to the treeview's store.
       */
      gtk_tree_store_append ((GtkTreeStore *) model, &treejter, &treeiter);
      
      gtk_tree_store_set ((GtkTreeStore *) model, 
			  &treejter,
			  COLUMN_NOTE_VALUE, text,
			  COLUMN_NOTE_VALUE_TYPE, help,
			  COLUMN_NOTE_POINTER, note,
			  COLUMN_NOTEVAL_POINTER, noteval,
			  COLUMN_PROP_POINTER, prop,
			  -1);
      /* Free the string that we show in the treeview to indicate what
	 is the type of the noteval->value.
      */
      g_free (help);

      /* Select the item that we just inserted so that the user
	 gets an immediate visual feedback of what happened.
      */
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
      
      gtk_tree_selection_select_iter (selection, &treejter);
    }
  else if (depth == 2)
    {
      gtk_tree_store_insert_after ((GtkTreeStore *) model, &treeiter, 
				   NULL, &treeiter);

      gtk_tree_store_set ((GtkTreeStore *) model, 
			  &treeiter,
			  COLUMN_NOTE_VALUE, text,
			  COLUMN_NOTE_VALUE_TYPE, help,
			  COLUMN_NOTE_POINTER, note,
			  COLUMN_NOTEVAL_POINTER, noteval,
			  COLUMN_PROP_POINTER, prop,
			  -1);
      
      /* Free the string that we show in the treeview to indicate what
	 is the type of the noteval->value.
      */
      g_free (help);

     /* Select the item that we just inserted so that the user
	 gets an immediate visual feedback of what happened.
      */
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
      
      gtk_tree_selection_select_iter (selection, &treeiter);
    }
  else
    g_assert_not_reached ();

  return;
}


void
polyxmass_note_wnd_value_apply_button (GtkWidget *widget,
				       gpointer data)
{
  /* The user has modified values for a given PxmNoteVal object that
     is selected in the treeview. The modifications were performed in
     the note value widgets (value type radiobutton, value contents
     textview). If no item is selected in the treeview, then we just
     return. If an item is selected, but is not of depth 2, then that
     means that the item selected is a note, and not a note value
     item. So we return also.
  */
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  GtkWidget *type_radiobutton = NULL;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter_start;
  GtkTextIter text_iter_end;
  GtkTextBuffer *buffer = NULL;

  gboolean valid = FALSE;

  gint depth = 0;

  gchar *help = NULL;
  gchar *text = NULL;

  PxmNote *note = NULL;
  PxmNoteVal *noteval_old = NULL;
  PxmNoteVal *noteval_new = NULL;
  
  g_assert (window != NULL);

  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);

  
  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) window,
	 _("Select a value item in the treeview first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  /* If the depth is 1, we just return: we want to apply changes to a
     row that describes a note value, not a note.
  */
  depth = gtk_tree_path_get_depth (path);
  
  /* We can free the path now.
   */
  gtk_tree_path_free (path);
  
  if (depth == 1)
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) window,
	 _("Select a value item in the treeview first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* At this point we know that a notevalue item is currently selected in
     the treeview. We also know that the user pretends he has modified
     its data in the related widgets. So, we have to modify the
     PxmNoteVal item in question with the new data, after having
     established that these data are OK.
  */

  /* We want to be extremely conservative: if anything goes wrong here
     we do not want that the contents of the selected noteval object be
     lost. We do no overwrite anything: allocate a new PxmNoteVal instance,
     fill it with data and check that everything is correct. Only at 
     that moment do the replacement of the old instance with the new one.
  */
  noteval_new = libpolyxmass_noteval_new ();
  
  /* Get the note value type immediately into the allocated noteval_new.
   */
  polyxmass_note_wnd_get_active_valtype_radiobutton (window,
						     &type_radiobutton,
						     &noteval_new->type);
  g_assert (type_radiobutton != NULL);
  
  textview = g_object_get_data (G_OBJECT (window),
				"value_contents_textview");
  g_assert (textview != NULL);
    
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);

  text = gtk_text_buffer_get_text (buffer, 
				   &text_iter_start, &text_iter_end,
				   FALSE);
  g_assert (text != NULL);

  /* We now have our PxmNoteValue->value, in the form of text. But we 
     must convert it to the right type before.
  */
  if (noteval_new->type == PXM_NOTE_VAL_TYPE_STR)
    {
      noteval_new->value = (gpointer) g_strdup (text);
    }
  else if (noteval_new->type == PXM_NOTE_VAL_TYPE_INT)
    {
      noteval_new->value = g_malloc0 (sizeof (gint));
      
      if (FALSE == 
	  libpolyxmass_globals_strtoi (text, (gint *) noteval_new->value, 10))
	{
	  libpolyxmass_noteval_free (noteval_new);
	  g_free (text);
	  
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("The value content is not an integer. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return;
	}
      
      /* At this point the noteval_new->value variable holds the proper
	 value!
      */
    }
  else if (noteval_new->type == PXM_NOTE_VAL_TYPE_DBL)
    {
      noteval_new->value = g_malloc0 (sizeof (gdouble));

      if (FALSE == 
	  libpolyxmass_globals_strtod (text, (gdouble *) noteval_new->value))
	{
	  libpolyxmass_noteval_free (noteval_new);
	  g_free (text);
	  
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("The value content is not a double. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return;
	}      
      /* At this point the noteval_new->value variable holds the proper
	 value!
      */
    }
  else
    g_assert_not_reached ();

  /* Later we will need to now the value of noteval_new->type so as to 
     display it in the treeview like a string. So prepare that string
     immediately.
  */
  if (noteval_new->type == PXM_NOTE_VAL_TYPE_STR)
    help = g_strdup ("str");
  else if (noteval_new->type == PXM_NOTE_VAL_TYPE_INT)
    help = g_strdup ("int");
  else if (noteval_new->type == PXM_NOTE_VAL_TYPE_DBL)
    help = g_strdup ("dbl");
  else
    g_assert_not_reached ();
  
  /* At this point our new value is fully qualified, we have to replace 
     the old noteval object with the new one.
  */

  /* From the treeiter that we got we can extract the pointer to the
     note object that contains the array of noteval instances of which
     one is the instance to remove.
  */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_NOTE_POINTER, &note,
		      -1);
  g_assert (note != NULL);

  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_NOTEVAL_POINTER, &noteval_old,
		      -1);
  g_assert (noteval_old != NULL);

  /* Remove the old noteval from the array and free it.
   */
  g_assert (TRUE == g_ptr_array_remove (note->notevalGPA, noteval_old));
  libpolyxmass_noteval_free (noteval_old);
  
  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						   TRUE);
  
  /* And now make sure that the new one replaces the old one!
   */
  g_ptr_array_add (note->notevalGPA, noteval_new);
  
  /* At this point the replacement has been done. We now have to perform
     the feedback to the treeview item that was selected in the first 
     place and that the user wanted to modify.
   */
  gtk_tree_store_set ((GtkTreeStore *) model, 
		      &treeiter,
		      COLUMN_NOTE_VALUE, text,
		      COLUMN_NOTE_VALUE_TYPE, help,
		      COLUMN_NOTEVAL_POINTER, noteval_new,
		      -1);
  
  /* Free the string that we show in the treeview to indicate what
     is the type of the noteval->value.
  */
  g_free (help);
  
  /* Select the item that we just inserted so that the user
     gets an immediate visual feedback of what happened.
  */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  
  gtk_tree_selection_select_iter (selection, &treeiter);
  
}


void
polyxmass_note_wnd_note_add_button (GtkWidget *widget,
				    gpointer data)
{
  /* The user wants to add a note to one of the following
     destinations:

     - the polymer
     
     - a single monomer

     For this to happen properly, the name of the note must be correct
     and non-empty. Further, that name must not be encountered in 
     any of the previously existent notes in the destination. 

     If the value data is existent, then it is taken into account,
     otherwise it defaults to no PxmNoteval object in the array that
     belongs to the PxmNote object (notevalGPA array of PxmNoteVal
     objects). Indeed, that array can be empty (but cannot be NULL
     since it is allocated by the constructor of the PxmNote object).

     Do not forget that a note is actually an allocate PxmProp object
     of name "NOTE" and data a PxmNote object !
  */
  GtkWidget *window = data;
  GtkWidget *entry = NULL;
  GtkWidget *seqed_widget = NULL;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeIter treejter;
  GtkTreeSelection *selection = NULL;


  PxmProp *prop = NULL;
  PxmProp *prop_new = NULL;

  PxmNote *note = NULL;
  PxmNote *note_new = NULL;

  GPtrArray *GPA = NULL;
  
  
  gint *target = NULL;
  gint idx = 0;
  gboolean valid = FALSE;
  
  gchar *help = NULL;
  
  
  g_assert (window != NULL);
  

  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);


  target = g_object_get_data (G_OBJECT (window), "TARGET");
  g_assert (target != NULL);


  /* First get the note name, in its entry.
   */
  entry = g_object_get_data (G_OBJECT (window),
			     "note_name_entry");
  g_assert (entry != NULL);
  
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);

  if (strlen (help) < 1)
    {
      g_free (help);

      polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	 _("The name of the note cannot empty. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* We know we have a name, so we can allocate a note prop, since
     we'll have a means to name the note! The prop is named
     automatically "NOTE" by the constructor;
  */  
  prop_new = libpolyxmass_note_prop_new ();
  note_new = (PxmNote *) prop_new->data;  

  libpolyxmass_note_set_name (note_new, help);

  g_free (help);

  /* We are going to need these data!
   */
  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  
  /* Since we are creating a new note, we do not take into account the
     data in the note value-related widgets, because the user has the
     other means to set new noteval objects to an existing note. This
     is to maintain a clean interface with simple tasks for each
     button at the user's disposal.
  */

  /* At this point we have successfully created a new note property that
     we will have to add to the destination (polymer or monomer). However,
     in both cases we will have to make sure that a note by the same name
     does not exist already.
  */

  /* When the target is the polymer, the pointer to the array of
     prop objects is stored in the "propGPA" datum. This is where
     the new prop object will land if there is not a pre-existing
     note prop by the same name.
  */
  GPA = g_object_get_data (G_OBJECT (model), "propGPA");
  g_assert (GPA != NULL);
      
  /* Now try to find a property that is identical to ours, and
     see...
  */
  prop = libpolyxmass_prop_find_prop (GPA,
				  &idx,
				  NULL,
				  "NOTE",
				  NULL,
				  PXM_CMP_NO_DEEP);
  while (prop != NULL)
    { 
      /* We have to make sure that the found prop of name "NOTE" 
	 has a datum that is a PxmNote object of which the name
	 is actually the one we are trying to monitor.
      */
      note = (PxmNote *) prop->data;
	  
      if (0 == strcmp (note->name, note_new->name))
	{
	  libpolyxmass_prop_free (prop_new);
	      
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("A note of the same name exists. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return;
	}

      idx++;

      prop = libpolyxmass_prop_find_prop (GPA,
				      &idx,
				      NULL,
				      "NOTE",
				      NULL,
				      PXM_CMP_NO_DEEP);
    }
      
  /* At this point we know that the polymer does not have a prop
     of name "NOTE" with the same note name. So we can add
     our newly allocated prop object right away.
  */
  g_ptr_array_add (GPA, prop_new);

  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget), TRUE);
  
      
  /* Append the item to the treeview's store.
   */
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
      
  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treejter);
  if (valid == FALSE)
    {
      gtk_tree_store_append ((GtkTreeStore *) model, 
			     &treeiter, NULL);
    }
  else
    {
      gtk_tree_store_insert_after ((GtkTreeStore *) model, 
				   &treeiter,
				   NULL,
				   &treejter);
    }
      
  gtk_tree_store_set ((GtkTreeStore *) model, 
		      &treeiter,
		      COLUMN_NOTE_NAME, note_new->name,
		      COLUMN_NOTE_POINTER, note_new,
		      COLUMN_PROP_POINTER, prop_new,
		      -1);

  /* Now that this item is appended we want to select it.
   */
  /* Get a tree iterator to the selection.
   */
  gtk_tree_selection_select_iter (selection, &treeiter);
}



void
polyxmass_note_wnd_note_apply_button (GtkWidget *widget,
				    gpointer data)
{
  /* The user wants to perform a modification in the name of a note.
     No other data in the note are changed. Only the note name. For
     this to happen clearly, the user has to select in the treeview
     the note for which the changes apply. Next, make the changes in
     the note name entry widget. Finally we are here.

     We will first make sure that a note name item (or note value
     item, in which case we'll work out the note name item) is selected.

     We will verify that the new name is correct, it must not be
     empty. And if correct, the change will be performed. After having
     made sure however that no other note has the same name as the
     changed one already.
  */
  GtkWidget *window = data;
  GtkWidget *entry = NULL;
  GtkWidget *seqed_widget = NULL;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeIter treejter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;

  gboolean valid = FALSE;

  gint depth = 0;
  gint idx = 0;
  
  gchar *help = NULL;

  PxmProp *prop = NULL;
  PxmNote *note = NULL;
  PxmNote *note_new = NULL;

  GPtrArray *GPA = NULL;
  


  g_assert (window != NULL);

  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);

 
  /* We are going to need these data!
   */
  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
      polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	 _("Select a note name item in the treeview first. "
	   "Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  depth = gtk_tree_path_get_depth (path);

  /* We can free the path now.
   */
  gtk_tree_path_free (path);
  


  /* First get the note name, in its entry.
   */
  entry = g_object_get_data (G_OBJECT (window),
			     "note_name_entry");
  g_assert (entry != NULL);
  
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);

  if (strlen (help) < 1)
    {
      g_free (help);
  
      return;
    }

  /* We now have to get to the pointer of the note object for which
     the note name change is to be performed. That means that we must
     know what item is currently selected. Whatever item is selected
     (could be a note name item or a note value item, they both have a
     pointer to the note, so the work is easy.
  */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_NOTE_POINTER, &note_new,
		      -1);
  g_assert (note_new != NULL);

  /* Be careful not to perform the change if another note pre-existed
     that had the same name as the new one. For this to be checked
     properly, we have to get a pointer to the propGPA on which we are
     working.
  */
  GPA = g_object_get_data (G_OBJECT (model), "propGPA");
  g_assert (GPA != NULL);

  /* Now try to find a property that is identical to ours, and
     see...
  */
  prop = libpolyxmass_prop_find_prop (GPA,
				  &idx,
				  NULL,
				  "NOTE",
				  NULL,
				  PXM_CMP_NO_DEEP);
  while (prop != NULL)
    { 
      /* We have to make sure that the found prop of name "NOTE" 
	 has a datum that is a PxmNote object of which the name
	 is actually the one we are trying to monitor.
      */
      note = (PxmNote *) prop->data;
	  
      if (0 == strcmp (note->name, help))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("A note of the same name exists. Operation cancelled."),
	     POLYXMASS_NORM_MSG_TIMEOUT);

	  g_free (help);
	  
	  return;
	}

      idx++;

      prop = libpolyxmass_prop_find_prop (GPA,
				      &idx,
				      NULL,
				      "NOTE",
				      NULL,
				      PXM_CMP_NO_DEEP);
    }

  /* At this point we know that the new note name is unique, we can
     perform the change.
  */
  libpolyxmass_note_set_name (note_new, help);
  
  /* Immediately set the polymer sequence to be modified.
   */
  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
                                                   TRUE);

  /* Show that change in the treeview. But be aware that we only show
     that name change in the note name items. So we must ensure that
     the treeiter that we got when looking for the selected item was
     indeed of a note name item. Otherwise we would have to go to the
     parent item.
   */
  if (depth == 1)
    {
      gtk_tree_store_set ((GtkTreeStore *) model, 
			  &treeiter,
			  COLUMN_NOTE_NAME, help,
			  -1);

      gtk_tree_selection_select_iter (selection, &treeiter);
    }
  else if (depth == 2)
    {
      g_assert (TRUE == 
		gtk_tree_model_iter_parent (model,
                                            &treejter, &treeiter));
      
      gtk_tree_store_set ((GtkTreeStore *) model, 
			  &treejter,
			  COLUMN_NOTE_NAME, help,
			  -1);

      gtk_tree_selection_select_iter (selection, &treejter);
    }
  
  g_free (help);
}




void
polyxmass_note_wnd_selected_item_remove_single (GtkMenuItem *menuitem,
						gpointer data)
{
  /* The user wants to remove the selected item from the treeview. We
     have to determine if the selected item is a note name item (whole
     note would be concerned by the operation) or a note value item
     (only that value item would be concerned by the operation).
  */
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeIter treejter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;
  gchar *path_str = NULL;

  gboolean valid = FALSE;

  gint depth = 0;
  gint idx_depth_1 = 0;
  gint idx_depth_2 = 0;
  gint count = 0;
  
  PxmProp *prop = NULL;
  PxmNote *note = NULL;
  PxmNoteVal *noteval = NULL;

  GPtrArray *GPA = NULL;
 

  g_assert (window != NULL);

  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);

  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
      polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	 _("Select a value item in the treeview first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  depth = gtk_tree_path_get_depth (path);
  
  
  if (depth == 1)
    {
      /* Get the index of the currently selected item. We are dealing with
	 indexes of items at depth 1 (index [0] below).
       */
      idx_depth_1 = gtk_tree_path_get_indices (path) [0];
	
      /* Get the number of note-name items in the treeview (NULL param).
       */
      count = gtk_tree_model_iter_n_children (model,
					      NULL);

      /* Prepare the land so that the item to be selected after the
	 currently selected one is remove is ok and that the behaviour
	 allows rapid removal of the other items in the treeview.
      */
      if (count == 1)
	idx_depth_1 = -1;
      if (idx_depth_1 == count - 1)
	idx_depth_1--;
      else
	idx_depth_1++;

      /* Synthesize a path string with the idx where the selection will
	 have to be performed after removal of the current item.
      */
      if (idx_depth_1 != -1)
	{
	  path_str = g_strdup_printf ("%d", idx_depth_1);
	  
	  gtk_tree_model_get_iter_from_string (model, 
					       &treejter, path_str);
	  g_free (path_str);
	  
	  gtk_tree_selection_select_iter (selection, &treejter);
	}
      
      /* We are being asked to remove an entire note (since a note
	 name is currently selected). We DO remember that the truth
	 about the notes is that they actually are PxmProp objects. If
	 we want to remove a note we acutally have to get the pointer
	 to the prop object that contains it. Fortunately, we have it
	 easily:
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_PROP_POINTER, &prop,
			  -1);
      g_assert (prop != NULL);

      /* We have to remove that prop object from the propGPA in which it
	 must be sitting patiently.
      */
      GPA = g_object_get_data (G_OBJECT (model), "propGPA");
      g_assert (GPA != NULL);

      g_assert (TRUE == g_ptr_array_remove (GPA, prop));

      /* Immediately set the polymer sequence to be modified.
       */
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						       TRUE);
      
      /* Free the removed prop object. 
       */
      libpolyxmass_prop_free (prop);
      
      /* Now that the note has been freed, we can remove the item
	 from the treeview.
      */
      gtk_tree_store_remove ((GtkTreeStore *) model, &treeiter);
    }

  if (depth == 2)
    {
      /* Get the indices of the items (note name and note value) at
	 the two different depths, so that we'll be able to fully
	 localize the item to be selected later.
       */
      idx_depth_1 = gtk_tree_path_get_indices (path) [0];
      idx_depth_2 = gtk_tree_path_get_indices (path) [1];

       /* Get the number of note-value items in the current note name
	 item. To do that we must get an iter to the parent item (the
	 note name item that contains our note-value item).
       */
      g_assert (TRUE == gtk_tree_model_iter_parent (model,
						    &treejter,
						    &treeiter));
		
      count = gtk_tree_model_iter_n_children (model,
					      &treejter);
      /* Prepare the land so that the item to be selected after the
	 currently selected one is removed is ok and that the
	 behaviour allows rapid removal of the other items in the
	 treeview.
      */
      if (count == 1)
	idx_depth_2 = -1;
      if (idx_depth_2 == count - 1)
	idx_depth_2--;
      else
	idx_depth_2++;

      /* Synthesize a path string with the idx where the selection
	 will have to be performed after removal of the current
	 item. However we only do prepare that string if the
	 idx_depth_2 is not -1. Indeed, if idx_depth_2 is -1, that
	 after the removal of the currently selected note value item
	 no more items will be present in the parent note name item.
	 So we do not select anything in this case.
      */
      if (idx_depth_2 != -1)
	{
	  path_str = g_strdup_printf ("%d:%d", idx_depth_1, idx_depth_2);
	  
	  gtk_tree_model_get_iter_from_string (model, 
					       &treejter, path_str);
	  g_free (path_str);
	  
	  gtk_tree_selection_select_iter (selection, &treejter);
	}

      /* We are being asked to remove a note-value item. We DO
	 remember that the truth about the note values is that they
	 actually are PxmNoteVal objects that are stored in the
	 GPtrArray of noteval objects in the PxmNote instances. If we
	 want to remove a note value we acutally have to get the
	 pointer to the PxmNoteVal object that contains
	 it. Fortunately, we have both the PxmNoteVal and PxmNote
	 instances' pointers very easily:
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTE_POINTER, &note,
			  -1);
      g_assert (note != NULL);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTEVAL_POINTER, &noteval,
			  -1);
      g_assert (noteval != NULL);

      /* We have to remove that noteval object from the notevalGPA in
	 which it must be sitting patiently.
      */
      g_assert (note->notevalGPA != NULL);

      g_assert (TRUE == g_ptr_array_remove (note->notevalGPA, noteval));

      /* Immediately set the polymer sequence to be modified.
       */
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						       TRUE);

      /* Free the removed noteval object. 
       */
      libpolyxmass_noteval_free (noteval);
      
      /* Now that the noteval has been freed, we can remove the item
	 from the treeview.
      */
      gtk_tree_store_remove ((GtkTreeStore *) model, &treeiter);
    }

  /* We can free the path now.
   */
  gtk_tree_path_free (path);
}


void
polyxmass_note_wnd_selected_item_remove_range (GtkMenuItem *menuitem,
						gpointer data)
{
  /* The item that is currently selected must be removed from all the 
     monomers in the range. This is similar to the "single" case, unless
     an iteration in the monomers of the range must allow to perform the 
     same action in all the monomers contained in the range.
  */
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeIter treejter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;
  gchar *path_str = NULL;

  gboolean valid = FALSE;

  gint depth = 0;
  gint idx_depth_1 = 0;
  gint idx_depth_2 = 0;
  gint count = 0;

  gint iter = 0;
  gint jter = 0;
  gint idx = 0;
  gint *start = 0;
  gint *end = 0;
  gint *index = NULL;
  
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmProp *prop = NULL;
  PxmProp *prop_iter = NULL;
  PxmNote *note = NULL;
  PxmNote *note_iter = NULL;
  PxmNoteVal *noteval = NULL;
  PxmNoteVal *noteval_iter = NULL;


  g_assert (window != NULL);

  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);

  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);
  
  start = g_object_get_data (G_OBJECT (window), "START_IDX");
  g_assert (start != NULL);
  
  end = g_object_get_data (G_OBJECT (window), "END_IDX");
  g_assert (end != NULL);
  
  index = g_object_get_data (G_OBJECT (window), "INDEX");
  g_assert (index != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
      polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	 _("Select a value item in the treeview first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  depth = gtk_tree_path_get_depth (path);
  
  
  if (depth == 1)
    {
      /* Get the index of the currently selected item. We are dealing with
	 indexes of items at depth 1 (index [0] below).
       */
      idx_depth_1 = gtk_tree_path_get_indices (path) [0];
	
      /* Get the number of note-name items in the treeview (NULL param).
       */
      count = gtk_tree_model_iter_n_children (model,
					      NULL);

      /* Prepare the land so that the item to be selected after the
	 currently selected one is remove is ok and that the behaviour
	 allows rapid removal of the other items in the treeview.
      */
      if (count == 1)
	idx_depth_1 = -1;
      if (idx_depth_1 == count - 1)
	idx_depth_1--;
      else
	idx_depth_1++;

      /* Synthesize a path string with the idx where the selection will
	 have to be performed after removal of the current item.
      */
      if (idx_depth_1 != -1)
	{
	  path_str = g_strdup_printf ("%d", idx_depth_1);
	  
	  gtk_tree_model_get_iter_from_string (model, 
					       &treejter, path_str);
	  g_free (path_str);
	  
	  gtk_tree_selection_select_iter (selection, &treejter);
	}
      
      /* We are being asked to remove an entire note (since a note
	 name is currently selected). We DO remember that the truth
	 about the notes is that they actually are PxmProp objects. If
	 we want to remove a note we acutally have to get the pointer
	 to the prop object that contains it. Fortunately, we have it
	 easily:
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_PROP_POINTER, &prop,
			  -1);
      g_assert (prop != NULL);

      /* We have to remove that prop object from the propGPA in which it
	 must be sitting patiently; and this for all the monomers
	 in the range.
      */
      for (iter = *start ; iter < *end + 1 ; iter++)
	{
	  /* We do not want to remove and free the prop object that
	     serves us as a template to search the other monomers in 
	     the range! We will remove and free that prop object after
	     all the others.
	  */
	  if (iter == *index)
	    continue;
	  
	  monomer = g_ptr_array_index (polymer->monomerGPA,
				       iter);
	  g_assert (monomer != NULL);
	  g_assert (monomer->propGPA != NULL);
	  
	  for (jter = 0; jter < monomer->propGPA->len ; jter++)
	    {
	      prop_iter = g_ptr_array_index (monomer->propGPA, jter);
	      g_assert (prop_iter != NULL);
	      
	      if (0 == libpolyxmass_prop_cmp 
		  (prop, prop_iter,
		   PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1));
		{
		  /* The iterated prop object is identical to the 
		     one we are asked to remove from all the monomers
		     in the range. Do it!
		  */
		  g_assert (TRUE == g_ptr_array_remove (monomer->propGPA,
							prop_iter));
		  
		  /* Immediately set the polymer sequence to be modified.
		   */
		  polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
								   TRUE);

		  libpolyxmass_prop_free (prop_iter);

		  break;
		}
	    }
	}
      
      /* At this point all of the prop objects were removed but one:
	 the one that belongs to the monomer that is used as
	 reference. So we still have to remove it and also update the
	 treeview.
      */
      monomer = g_ptr_array_index (polymer->monomerGPA,
				   *index);
      g_assert (monomer != NULL);
      g_assert (monomer->propGPA != NULL);

      /* We know from the beginning that the prop to remove is 'prop'
	 since this is the pointer we got from the treeview!
      */
      g_assert (TRUE == g_ptr_array_remove (monomer->propGPA, prop));
      
      /* Immediately set the polymer sequence to be modified.
       */
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						       TRUE);
      
      libpolyxmass_prop_free (prop);

      /* Now that the note has been freed, we can remove the item
	 from the treeview.
      */
      gtk_tree_store_remove ((GtkTreeStore *) model, &treeiter);
    }

  if (depth == 2)
    {
      /* Get the indices of the items (note name and note value) at
	 the two different depths, so that we'll be able to fully
	 localize the item to be selected later.
       */
      idx_depth_1 = gtk_tree_path_get_indices (path) [0];
      idx_depth_2 = gtk_tree_path_get_indices (path) [1];

       /* Get the number of note-value items in the current note name
	 item. To do that we must get an iter to the parent item (the
	 note name item that contains our note-value item).
       */
      g_assert (TRUE == gtk_tree_model_iter_parent (model,
						    &treejter,
						    &treeiter));
		
      count = gtk_tree_model_iter_n_children (model,
					      &treejter);
      /* Prepare the land so that the item to be selected after the
	 currently selected one is removed is ok and that the
	 behaviour allows rapid removal of the other items in the
	 treeview.
      */
      if (count == 1)
	idx_depth_2 = -1;
      if (idx_depth_2 == count - 1)
	idx_depth_2--;
      else
	idx_depth_2++;

      /* Synthesize a path string with the idx where the selection
	 will have to be performed after removal of the current
	 item. However we only do prepare that string if the
	 idx_depth_2 is not -1. Indeed, if idx_depth_2 is -1, that
	 after the removal of the currently selected note value item
	 no more items will be present in the parent note name item.
	 So we do not select anything in this case.
      */
      if (idx_depth_2 != -1)
	{
	  path_str = g_strdup_printf ("%d:%d", idx_depth_1, idx_depth_2);
	  
	  gtk_tree_model_get_iter_from_string (model, 
					       &treejter, path_str);
	  g_free (path_str);
	  
	  gtk_tree_selection_select_iter (selection, &treejter);
	}

      /* We are being asked to remove a note-value item. We DO
	 remember that the truth about the note values is that they
	 actually are PxmNoteVal objects that are stored in the
	 GPtrArray of noteval objects in the PxmNote instances. If we
	 want to remove a note value we acutally have to get the
	 pointer to the PxmNoteVal object that contains
	 it. Fortunately, we have both the PxmNoteVal and PxmNote
	 instances' pointers very easily:
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_PROP_POINTER, &prop,
			  -1);
      g_assert (prop != NULL);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTE_POINTER, &note,
			  -1);
      g_assert (note != NULL);

      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_NOTEVAL_POINTER, &noteval,
			  -1);
      g_assert (noteval != NULL);

      /* We have to remove that noteval object from all the prop objects 
	 that are identical to prop.
      */
      for (iter = *start ; iter < *end + 1 ; iter++)
	{
	  /* We do not want to remove and free the prop' noteval
	     object that serves us as a template to search in the other
	     monomers in the range! We will remove and free that prop
	     object after all the others.
	  */
	  if (iter == *index)
	    continue;
	  
	  monomer = g_ptr_array_index (polymer->monomerGPA,
				       iter);
	  g_assert (monomer != NULL);
	  g_assert (monomer->propGPA != NULL);
	  
	  for (jter = 0; jter < monomer->propGPA->len ; jter++)
	    {
	      prop_iter = g_ptr_array_index (monomer->propGPA, jter);
	      g_assert (prop_iter != NULL);
	      
	      if (0 == libpolyxmass_prop_cmp 
		  (prop, prop_iter,
		   PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1))
		{
		  /* The iterated prop object is identical to the one
		     in which we are asked to remove the noteval
		     object. So find that noteval object in this note.
		  */
		  note_iter = (PxmNote *) prop_iter->data;
		
		  idx = 
		    libpolyxmass_noteval_get_index_by_noteval (noteval,
							   note_iter->
							   notevalGPA);
		  
		  if (idx != -1)
		    {
		      /* We found a prop object (actually a note,
			 as we know) that contains a noteval object
			 identical to the one we have to remove. Do it!
		      */
		     noteval_iter = 
		       g_ptr_array_remove_index (note_iter->notevalGPA,
						 idx);
		     g_assert (noteval_iter != NULL);
		     
		     /* Immediately set the polymer sequence to be modified.
		      */
		     polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
								      TRUE);

		     libpolyxmass_noteval_free (noteval_iter);

		     break;
		    }
		}
	    }
	}
      
      /* At this point we still have to remove the noteval object from
	 the note that patiently sits in the reference monomer's
	 monomer->GPA of prop objects...
      */
      monomer = g_ptr_array_index (polymer->monomerGPA,
				   *index);
      g_assert (monomer != NULL);
      g_assert (monomer->propGPA != NULL);

      /* We know from the beginning that the prop in which the note
	 resides is prop, since we got its pointer directly from the
	 treeview. In this prop, the member 'data' is actually the
	 note of which we also know the pointer from the treeview. In
	 this note object we do have a notevalGPA array of noteval
	 objects. In this array, there is the noteval object of which
	 we also got a pointer from the treeview.
      */
      g_assert (TRUE == g_ptr_array_remove (note->notevalGPA, noteval));
      
      /* Immediately set the polymer sequence to be modified.
       */
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						       TRUE);

      libpolyxmass_noteval_free (noteval);
      
      /* Now that the noteval has been freed, we can remove the item
	 from the treeview.
      */
      gtk_tree_store_remove ((GtkTreeStore *) model, &treeiter);
    }

  /* We can free the path now.
   */
  gtk_tree_path_free (path);
}


void
polyxmass_note_wnd_selected_item_propagate (GtkMenuItem *menuitem,
					    gpointer data)
{
  /* The user is asking that the selected item be propagated into all
     the monomers of the range. Note that if we are not in range mode,
     this menu does not have any meaning. And in fact it cannot be
     called because we have made it insensitive.
  */
  GtkWidget *window = data;
  GtkWidget *dialog = NULL;
  GtkWidget *seqed_widget = NULL;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;

  gboolean valid = FALSE;

  gint depth = 0;

  gint iter = 0;
  gint *start = 0;
  gint *end = 0;
  gint *index = NULL;
  
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmProp *prop = NULL;
  PxmProp *prop_iter = NULL;

  PxmNote *note = NULL;
  PxmNote *note_iter = NULL;
  


  g_assert (window != NULL);


  seqed_widget =  g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);


  start = g_object_get_data (G_OBJECT (window), "START_IDX");
  g_assert (start != NULL);
  
  end = g_object_get_data (G_OBJECT (window), "END_IDX");
  g_assert (end != NULL);
  
  index = g_object_get_data (G_OBJECT (window), "INDEX");
  g_assert (index != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "notes_treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  /* Get a tree iterator to the selection.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  valid = gtk_tree_selection_get_selected (selection,
					   &model,
					   &treeiter);
  
  if (valid == FALSE)
    {
      /* There is no selected item, so we just return.
       */
     polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	 _("Select a value item in the treeview first. Operation cancelled."),
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return;
    }

  /* Get the path to the selected item, which can be a Note name row
     or a NoteVal value/value-type row. These two different cases must
     have a different handling. Specifically, if depth == 1, that means 
     that the selected row is a Note name row, while if it is 2, that
     means that the selected row is a NoteVal value/value-type row.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &treeiter);
  g_assert (path != NULL);
  
  depth = gtk_tree_path_get_depth (path);
  
  if (depth == 1)
    {
      /* We are dealing with a note name (a whole note object, thus)
	 item. Get the prop object that is referenced by the item that
	 is currently selected.
      */
      gtk_tree_model_get ((GtkTreeModel *) model, &treeiter, 
			  COLUMN_PROP_POINTER, &prop,
			  -1);
      g_assert (prop != NULL);

      note = (PxmNote *) prop->data;
      
      /* We now have to iterate in the range, and for each monomer in the
	 range we have to perform these tasks in order:
	 
	 1. check that the iterated monomer does not already have a
	 "NOTE"-named PxmProp object of which the name is the same as
	 the one we are trying to set. 

	 In other words, a monomer may not have two different PxmNote
	 objects of the same note->name name. This is why we call
	 libpolyxmass_note_cmp () below with the "PXM_NOTE_CMP_NAME"
	 parameter below.
	 
	 2. add a copy of 'prop' in that same array.
      */
      for (iter = *start ; iter < *end + 1 ; iter++)
	{
	  /* We do not want to add one more copy of 'prop' in the
	     reference monomer, because that 'prop' object is already 
	     in its array !
	  */
	  if (iter == *index)
	    continue;

	  monomer = g_ptr_array_index (polymer->monomerGPA,
				       iter);
	  g_assert (monomer != NULL);
	  g_assert (monomer->propGPA != NULL);	  
	  
	  prop_iter = libpolyxmass_prop_find_prop (monomer->propGPA,
					       NULL,
					       NULL,
					       "NOTE",
					       NULL,
					       PXM_CMP_NO_DEEP);
	  
	  if (prop_iter != NULL)
	    {
	      /* The iterated monomer has a PxmProp object by name
		 "NOTE", that is it has a note object. We have to make
		 sure that it is not identical to the one we are
		 adding.
	      */
	      note_iter = (PxmNote *) prop_iter->data;
	      
	      if (0 == libpolyxmass_note_cmp (note_iter,
					      note,
					      PXM_CMP_NO_DEEP))
		{
		  dialog = gtk_message_dialog_new 
		    (GTK_WINDOW (window),
		     GTK_DIALOG_DESTROY_WITH_PARENT,
		     GTK_MESSAGE_WARNING,
		     GTK_BUTTONS_OK,
		     _("Skipped monomer: '%s' at position: "
		       "'%d' already having a note by the "
		       "same name: '%s'"),
		     monomer->code, iter + 1, 
		     note_iter->name);
		  
		    /* We do not care of what "response" value is
		       returned, the only thing the user can do is
		       either close the dialog using the window
		       decorations or clicking the OK button.
		    */
		    gtk_dialog_run (GTK_DIALOG (dialog));

		    /* Now that the user has closed the dialog, we are
		       responsible for closing it.
		     */
		    gtk_widget_destroy (dialog);
		    
		    continue;
		  }
	    }
	  
	  /* At this point, clearly, no pre-existing note by the same name
	     as the one we are trying to propagate was found in the
	     currently iterated monomer.  
	  */
	  prop_iter = libpolyxmass_prop_dup (prop, PXM_DUP_DEEP);
	  
	  g_ptr_array_add (monomer->propGPA, prop_iter);
	}
      
      /* At this point we have finished propagating the note property.
       */
    }
  
  /* The case for which depth == 2 is not dealt with at it is only
     possible to propagate full notes, and not noteval objects 
     taken individually.
  */
}


void
polyxmass_note_wnd_reset_note_related_widgets (GtkWidget *widget)
{
  GtkWidget *window = widget;
  GtkWidget *note_name_entry = NULL;
  GtkWidget *type_radiobutton = NULL;
  
  GtkTextView *textview = NULL;
  GtkTextIter text_iter_start;
  GtkTextIter text_iter_end;
  GtkTextBuffer *buffer = NULL;


  g_assert (window != NULL);
  


  textview = g_object_get_data (G_OBJECT (window),
				"value_contents_textview");
  g_assert (textview != NULL);
    
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);
  gtk_text_buffer_delete (buffer, &text_iter_start, &text_iter_end);

  note_name_entry = g_object_get_data (G_OBJECT (window),
				       "note_name_entry");
  g_assert (note_name_entry != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (note_name_entry), "");
  
  type_radiobutton = g_object_get_data (G_OBJECT (window),
					"value_type_str_radiobutton");
  g_assert (type_radiobutton != NULL);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (type_radiobutton),
				TRUE);  
  
}


/* Window house keeping functions.
 */

void 
polyxmass_note_wnd_get_active_valtype_radiobutton (GtkWidget *window,
						   GtkWidget **radiobutton,
						   PxmNoteValType *valtype)
{
  GtkWidget *type_radiobutton = NULL;
  
  g_assert (window != NULL);
  g_assert (radiobutton != NULL);
  g_assert (valtype != NULL);
  
  type_radiobutton = g_object_get_data (G_OBJECT (window),
					"value_type_str_radiobutton");
  if (TRUE == 
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (type_radiobutton)))
    {
      *radiobutton = type_radiobutton;
      *valtype = PXM_NOTE_VAL_TYPE_STR;

      return;
    }
    
  type_radiobutton = g_object_get_data (G_OBJECT (window),
					"value_type_int_radiobutton");
  if (TRUE == 
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (type_radiobutton)))
    {
      *radiobutton = type_radiobutton;
      *valtype = PXM_NOTE_VAL_TYPE_INT;

      return;
    }
    
  type_radiobutton = g_object_get_data (G_OBJECT (window),
					"value_type_dbl_radiobutton");
  if (TRUE == 
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (type_radiobutton)))
    {
      *radiobutton = type_radiobutton;
      *valtype = PXM_NOTE_VAL_TYPE_DBL;

      return;
    }
    
  g_assert_not_reached ();
}


/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxmass_note_wnd_really_close (GtkWidget *window)
{
  gint count = 0;
  

  g_assert (window != NULL);
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /*
    printf ("polyxmass_note_wnd_really_close, count is %d\n",
    count);
  */

  gtk_widget_destroy (window);
}

gboolean
polyxmass_note_wnd_delete_event (GtkWidget *window, 
				 GdkEventAny *event, 
				 gpointer data)
{
  GtkWidget *seqed_widget = data;

  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (seqed_widget != NULL);
  
  help = g_strdup_printf ("note_wnd-%p", window);
  
  g_object_set_data (G_OBJECT (seqed_widget),
		     help, NULL);
  g_free (help);
  
  return TRUE;
}













  
