/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "pxmchem-elemcompcalc.h"
#include "pxmchem-monomer.h"
#include "pxmchem-formula.h"


PxmCompcalcRes
pxmchem_elemcompcalc_polymer (PxmPolymer *polymer,
			      PxmPolchemdef *polchemdef,
			      PxmCalcOpt *calcopt,
			      PxmIonizerule *ionizerule, 
			      GPtrArray *acGPA,
			      gchar **formula)
{
  PxmCompcalcRes compcalcres = PXM_COMPCALC_SUCCESS;

  GPtrArray *atomcountGPA = NULL;
  
  gboolean res = FALSE;
  gboolean array_allocated = FALSE;
   


  g_assert (polymer != NULL && polymer->monomerGPA != NULL);

  /* Both acGPA and formula may be NULL, but not at the same time.
   */
  g_assert (acGPA != NULL || formula != NULL);
  
  if (formula != NULL)
    {
      /* The string into which the formula is returned is going to be
	 allocated, and its address is returned into 'formula'. That's
	 why formula needs to be non-NULL but *formula has to be NULL
	 so that we ensure it is empty and has no allocated data
	 pointed to by it.
      */
      g_assert (*formula == NULL);
    }
  
  /* We want to allow that the user calls this function with an allocated
     GPtrArray of PxmAtomcount objects because that enables her to 
     perform incremental elemental composition calculations by 
     stacking onto the same array the calculatations from different 
     function calls.

     If the acGPA param is NULL, we still need one such array, so we
     allocate it.
  */
  if (acGPA == NULL)
    {
      /* We have to locally allocate a GPtrArray in which PxmAtomcount 
	 instances will be stored.
      */
      atomcountGPA = g_ptr_array_new ();

      array_allocated = TRUE;
    }
  else
    {
      /* We just copy the pointer from the param.
       */
      atomcountGPA = acGPA;
    }
  
  g_assert (polchemdef != NULL);

  g_assert (calcopt != NULL);
  g_assert (ionizerule != NULL);


  /* Check that there is a sequence element for which to calculate an
     elemental composition.  If not, this is not necessarily an
     error. We continue working because many other things have to be
     taken into account. But inform the user.
     *
     if (polymer->monomerGPA->pdata == NULL)
     {
     g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
     _("%s@%d: the monomer array is empty.\n"),
     __FILE__, __LINE__);
     }
  */
  
  /*********************** CAPPED OR NON CAPPED ************************/

  /* Now that we know we have a GPtrArray of monomer instances
     corresponding to the sequence of the polymer, we can continue the
     work, with the actual composition calculations based upon the
     contents of the monomer GPtrArray.

     Note that if the caller has asked that the monomer entities (like
     modifications) be taken into account during the mass calculation,
     this will be done in the function call below.
  */
  res = pxmchem_elemcompcalc_noncapped_monomer_GPA (polymer->monomerGPA,
						    polchemdef, 
						    atomcountGPA, 
						    calcopt);
  if (TRUE != res)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to calculate the non-capped masses of the polymer\n"),
	     __FILE__, __LINE__);
      
      compcalcres = PXM_COMPCALC_FAILURE;
    }
  
  
  /* We now have in the atomcountGPA array, a number of actomcoung
     instances describing the elemental composition of the non-capped
     residual chain of monomers constituting the polymer. We now have
     to check if user wants composition for a CAPPED polymer or
     not. If yes we delegate this calculation to a helper function
  */
  if (calcopt->capping != PXM_CAPPING_NONE)
    {
      res =
	pxmchem_elemcompcalc_polymer_account_caps (atomcountGPA, 
						   calcopt->capping, 
						   polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for caps of the polymer\n"),
		 __FILE__, __LINE__);
	  
	  compcalcres = PXM_COMPCALC_FAILURE;
	}
      
      
    }
  
  
  /*** THE CHEMICAL ENTITIES THAT MAY BELONG TO THE POLYMER SEQUENCE ***/
  
  if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF)
    {
      /* Take into account the left end modif.
       */
      res = 
	pxmchem_elemcompcalc_polymer_account_left_end_modif (polymer,
							     atomcountGPA,
							     polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for polymer's "
		   "left end modif.\n"),
		 __FILE__, __LINE__);
	  
	  compcalcres = PXM_COMPCALC_FAILURE;
	}
      
    }
  
  
  if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
    {
      /* Take into account the right end modif.
       */
      res = 
	pxmchem_elemcompcalc_polymer_account_right_end_modif (polymer,
							      atomcountGPA,
							      polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for polymer's "
		   "right end modif.\n"),
		 __FILE__, __LINE__);
	  
	  compcalcres = PXM_COMPCALC_FAILURE;
	}
      
    }
  

  /******************** IONIZATION REQUIREMENTS**********************/

  /* For the moment the elemental composition calculated is for a
     non-charged polymer: let's check if caller wants some kind of
     charge to be put on it.  For this we also need to know what atom
     is responsible for the charge formation (proton or whatever).
  */
  res = pxmchem_elemcompcalc_polymer_account_ionizerule (ionizerule,
							 polchemdef->atomGPA,
							 atomcountGPA);
  if (FALSE == res)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the ionizerule.\n"),
	     __FILE__, __LINE__);
      
      compcalcres = PXM_COMPCALC_FAILURE;
    }
  

  /* We now have a GPtrArray full of allocated atomcount instances,
   * from which we must construct a formula:
   */
  *formula = pxmchem_formula_make_string_from_atomcount_GPA (atomcountGPA);
  
  if (*formula == NULL)
    compcalcres = PXM_COMPCALC_FAILURE;

  /* We can now free the atomcountGPA, but only if we had to allocate it 
     at the beginning of our work.
   */
  if (array_allocated == TRUE)
    pxmchem_atomcount_GPA_free (atomcountGPA);
  
  return compcalcres;
}



gboolean
pxmchem_elemcompcalc_noncapped_monomer_GPA (GPtrArray *GPA,
					    PxmPolchemdef *polchemdef, 
					    GPtrArray *acGPA, 
					    PxmCalcOpt *calcopt)
{
  gint len = 0;
  gint iter = 0;

  PxmMonomer *monomer = NULL;

  /* We will iterate in the monomerGPA passed as parameter and for
     each monomer iterated we will add its elemental contributions to
     the object 'acGPA' passed as parameter. Note that if
     calcopt states that the monomer chemical entities (modifs) must
     be taken into account, then, this is done.

     Note that the calcopt struct passed as param contains two members
     that will allow us to know which monomer stretch from the GPA
     should be taken into account during the calculatation.
  */

  /* Some easy checks to start with, in particular the array of
     monomers, GPA, cannot be NULL.
   */
  g_assert (GPA != NULL);
  g_assert (polchemdef != NULL);
  g_assert (acGPA != NULL);

  len = GPA->len;

  if (len == 0)
    {
      /*
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: the monomer array is empty.\n"),
	__FILE__, __LINE__);
      */
      return TRUE;
    }

  /* Now check some of the options passed as param pertaining to
   * elemental composition calculatation of the non capped chain (make
   * code FOOL PROOF).
   */

  /* Left end first. 
     ---------------

     It may happen that the user clicks and drags right of the last 
     polymer sequence's monicon and that this function is called with a 
     calcopt->start_idx = len. We just need to check this.
  */
  if (calcopt->start_idx > len)
    calcopt->start_idx = len;
  
  /* Special value: if -1, then that means the whole sequence should be
     taken into account. Thus set start_idx to 0.
  */
  if (calcopt->start_idx == -1)
    calcopt->start_idx = 0;

  /* And now make sure that start_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (calcopt->start_idx >= 0);
  

  /* Right end, second.
   */
  g_assert (calcopt->end_idx <= len);

  /* Special value: if -1, then that means the whole sequence should
     be taken into account. Thus set end_idx to len (because the for
     loop later will squeeze stop at iter == (len - 1).
  */
  if (calcopt->end_idx == -1)
    calcopt->end_idx = len;

  /* And now make sure that end_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (calcopt->end_idx >= 0);
  
  /* NOW start calculating the elemental composition of the non-capped
   * polymer by accumulating the formula of each iterated monomer in
   * the GPtrArray of monomer passed as param.
   */
  for (iter = calcopt->start_idx; iter < calcopt->end_idx; iter++)
    {
      monomer = g_ptr_array_index (GPA, iter);

      if (FALSE ==
	  pxmchem_formula_parse (monomer->formula, polchemdef->atomGPA,
				 1, acGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account formula: '%s' "
		   "for monomer: '%s'\n"),
		 __FILE__, __LINE__, monomer->formula, monomer->name);
	  
	  return FALSE;
	}

      /* If monomer modifs are asked to be taken into account, do it:
       */
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	{



	  if (FALSE == 
	      pxmchem_monomer_account_elemcompos_for_modif (monomer,
							    polchemdef->modifGPA,
							    polchemdef->atomGPA,
							    1, /*times*/
							    acGPA))
	    {	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for modif of "
		       "monomer: '%s'\n"),
		     __FILE__, __LINE__, monomer->name);


	      return FALSE;
	    }
	}	
      /* end of 
	 if (calcopt->mnm_chement == PXMCHEMENT_MNM_MODIF)
      */

    }
  /* end iterating in the GPtrArray of monomers */

  /* At this stage we have the NON CAPPED elemental composition
     calculated for the given polymer stretch comprised of monomers
     [calcopt->start_idx - calcopt->end_idx]. If the options stated
     that the monomer modif's had to be taken into account, this has
     been done! Also, here both the elemental composition have been
     calculated.
   */

  return TRUE;
  
}


gboolean
pxmchem_elemcompcalc_polymer_account_caps (GPtrArray *acGPA,
					   PxmCapping cap,
					   PxmPolchemdef *polchemdef)
{
  /* There are two kinds of caps: the left end cap and the right end cap.
     Each should be treated separately depending on the value of cap.
  */
  g_assert (acGPA != NULL);
  g_assert (polchemdef != NULL);
  
  if (cap & PXM_CAPPING_NONE)
    return TRUE;
  
  if (FALSE != (cap & PXM_CAPPING_LEFT))
    {
      if (FALSE == 
	  pxmchem_actform_account_elemcompos (polchemdef->leftcap, 
					      polchemdef->atomGPA,
					      1 /*times*/, acGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for left cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->leftcap);
	  return FALSE;
	}
    }
  
  if (FALSE != (cap & PXM_CAPPING_RIGHT))
    {
      if (FALSE == 
	  pxmchem_actform_account_elemcompos (polchemdef->rightcap, 
					      polchemdef->atomGPA,
					      1 /*times*/, acGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for right cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->rightcap);
	  return FALSE;
	}
    }
  
  return TRUE;
}

  
gboolean
pxmchem_elemcompcalc_polymer_account_left_end_modif (PxmPolymer *polymer,
						     GPtrArray *acGPA,
						     PxmPolchemdef *polchemdef)
{
  PxmProp *prop = NULL;

  /* We need to get access to the LEFT_END_MODIF prop instance in 
     the polymer, if there is any.
  */
  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
				  NULL,
				  NULL,
				  "LEFT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);
  if (prop == NULL)
    return TRUE;
  
  if (FALSE == 
      pxmchem_modif_account_elemcompos_by_name ((gchar *) prop->data,
						polchemdef->modifGPA,
						polchemdef->atomGPA,
						1,
						acGPA))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for left end modif of the polymer: '%s'\n"),
	     __FILE__, __LINE__, (gchar *) prop->data);

      return FALSE;
    }
  
  return TRUE;
}

  
gboolean
pxmchem_elemcompcalc_polymer_account_right_end_modif (PxmPolymer *polymer,
						      GPtrArray *acGPA,
						      PxmPolchemdef *polchemdef)
{
  PxmProp *prop = NULL;

  /* We need to get access to the LEFT_END_MODIF prop instance in 
     the polymer, if there is any.
  */
  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
				  NULL,
				  NULL,
				  "RIGHT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);
  if (prop == NULL)
    return TRUE;
  
  if (FALSE == 
      pxmchem_modif_account_elemcompos_by_name ((gchar *) prop->data,
						polchemdef->modifGPA,
						polchemdef->atomGPA,
						1,
						acGPA))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for right end modif of polymer: '%s'\n"),
	     __FILE__, __LINE__, (gchar *) prop->data);
      return FALSE;
    }
  
  return TRUE;
}


gboolean
pxmchem_elemcompcalc_polymer_account_ionizerule (PxmIonizerule *ionizerule,
						 GPtrArray *atom_refGPA,
						 GPtrArray *acGPA)
{
  

  g_assert (ionizerule != NULL);
  g_assert (ionizerule->actform != NULL);
  
  g_assert (acGPA != NULL);
  g_assert (atom_refGPA != NULL);
  

  /* We get an array of atomcount instances and we need to update it
     in order to comply with the ionization options asked in the
     'ionizerule'.
   */
  /* We cannot have a charge == 0, because it will be in the
   * denominator in the following calculations, so return without
   * error, which simply means that the caller does not want to have
   * any ionization! Same for level. See below.
   */
  if (ionizerule->charge == 0 || ionizerule->level == 0)
    return TRUE;
  
  
  /* In the ionizerule argument, we have a member actform of the kind
     "+Cd1-H1", for example.
   */
  if (FALSE == 
      pxmchem_actform_account_elemcompos (ionizerule->actform, 
					  atom_refGPA,
					  ionizerule->level /*times*/, acGPA))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the actform of the ionization "
	       "rule: '%s'\n"),
	     __FILE__, __LINE__, ionizerule->actform);
      
      return FALSE;
    }
 
  /* OK we have accounted the acGPA for the ionizerule->actform by a
     number of times ionizerule->level.
  */

  return TRUE;
}







PxmCompcalcRes
pxmchem_elemcompcalc_oligomer (PxmOligomer *oligomer,
			       PxmPolchemdef *polchemdef,
			       PxmCalcOpt *calcopt,
			       PxmCleaveOpt *cleaveopt,
			       PxmIonizerule *ionizerule,
			       GPtrArray *acGPA,
			       gchar **formula)
{
  PxmCompcalcRes compcalcres = PXM_COMPCALC_SUCCESS;

  gint len = 0;
  gint iter = 0;

  PxmPolymer *polymer = NULL;

  GPtrArray *atomcountGPA = NULL;

  GPtrArray *clrGPA = NULL;
  
  gboolean res = FALSE;
  gboolean array_allocated = FALSE;
  


  g_assert (oligomer != NULL);
  polymer = oligomer->polymer;

  g_assert (polymer != NULL);
  g_assert (polymer->monomerGPA != NULL);
  
  /* Both acGPA and formula may be NULL, but not at the same time.
   */
  g_assert (acGPA != NULL || formula != NULL);
  
  if (formula != NULL)
    {
      /* The string into which the formula is returned is going to be
	 allocated, and its address is returned into 'formula'. That's
	 why formula needs to be non-NULL but *formula has to be NULL
	 so that we ensure it is empty and has no allocated data
	 pointed to by it.
      */
      g_assert (*formula == NULL);
    }
  
  /* We want to allow that the user calls this function with an allocated
     GPtrArray of PxmAtomcount objects because that enables her to 
     perform incremental elemental composition calculations by 
     stacking onto the same array the calculatations from different 
     function calls.
     
     If the acGPA param is NULL, we still need one such array, so we
     allocate it.
  */
  if (acGPA == NULL)
    {
      /* We have to locally allocate a GPtrArray in which PxmAtomcount 
	 instances will be stored.
      */
      atomcountGPA = g_ptr_array_new ();
      
      array_allocated = TRUE;
    }
  else
    {
      /* We just copy the pointer from the param.
       */
      atomcountGPA = acGPA;
    }
  
  g_assert (polchemdef != NULL);

  g_assert (calcopt != NULL);
  g_assert (ionizerule != NULL);
  
  len = oligomer->polymer->monomerGPA->len;
  
  if (len <= 0)
    {
      /*
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: the polymer sequence is empty.\n"),
	__FILE__, __LINE__);
      */
      return TRUE;
    }
  

  /* For the elemental calculation process we must not take
     coordinates of the oligomer in the oligomer object itself since
     these have not been set for elemental composition
     calculations. The values have been set correctly for the
     elemental composition calculations in the calcopt structure
     passed as parameter.
  */
  g_assert (calcopt->start_idx >=0 && calcopt->end_idx >= 0);
  g_assert (calcopt->start_idx < len);
  g_assert (calcopt->end_idx <= len);

  res = pxmchem_elemcompcalc_noncapped_monomer_GPA (polymer->monomerGPA,
						    polchemdef, 
						    atomcountGPA, 
						    calcopt);
  if (FALSE == res)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to calculate the composition of the oligomer\n"),
	     __FILE__, __LINE__);
      
      return PXM_COMPCALC_FAILURE;
    }
  

  /* At this stage we have the NON CAPPED elemental composition
     calculated for the given polymer stretch comprised of monomers
     [calcopt->start_idx - calcopt->end_idx]. If the options stated
     that the monomer modif's had to be taken into account, this has
     been done!
  */
  
  /* Here is one part that is a bit tricky with the calculating of
     oligomer elemental composition: the cleaverule handling. Each
     oligomer may result from a cleavage that has as side effects the
     chemical modification of the monomers that were bordering the
     chemical bond that is cleaved. See the cyanogen bromide example
     in the proteins' world.

     The cleaveopt parameter has a cleavespec object in it, which, in
     turn has an array of cleaverules, that we are going to handle
     right now. Note that the array should exist, but may be empty.

     However, remember that the cleaveopt param may be NULL if the
     oligomer was not resulting from a chemical cleavage. See the case
     of searching masses in the polymer sequence.
  */
  if (cleaveopt != NULL)
    {
      PxmCleaveSpec *cleavespec = NULL;
      PxmCleaveRule *lr_rule = NULL;
      
      cleavespec = cleaveopt->cleavespec;
      g_assert (cleavespec != NULL);
     
      clrGPA = cleavespec->clrGPA;
      g_assert (clrGPA != NULL);
      
      
      for (iter = 0; iter < clrGPA->len; iter++)
	{
	  lr_rule = g_ptr_array_index (clrGPA, iter);
	  g_assert (lr_rule != NULL);
	  
	  /* Now account for this rule.
	   */
	  res = 
	    pxmchem_elemcompcalc_oligomer_account_cleaverule 
	    (lr_rule,
	     oligomer,
	     polchemdef->atomGPA,
	     atomcountGPA);
	  
	  if (FALSE == res)
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for cleaverule\n"),
		     __FILE__, __LINE__);
	      
	      return PXM_COMPCALC_FAILURE;
	    }
	}

      /* Done accounting for cleaverule objects (if any). OUF.
       */
    }
  
  /* We now have in the atomcountGPA array all the atomcount instances that
     reflect the elemental composition of the non-capped residual
     chain of monomers constituting the oligomer. We now have to check
     if user wants the elemental composition for a CAPPED oligomer or
     not. If yes we delegate this calculation to a helper function.
  */
  if (calcopt->capping != PXM_CAPPING_NONE)
    {
      res = pxmchem_elemcompcalc_polymer_account_caps (atomcountGPA, 
						       calcopt->capping, 
						       polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for caps of the polymer\n"),
		 __FILE__, __LINE__);
	  
	  return PXM_COMPCALC_FAILURE;
	}
    }


  /*** THE CHEMICAL ENTITIES THAT MAY BELONG TO THE POLYMER SEQUENCE ***/

  if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
      && oligomer->start_idx == 0)
    {
      /* Take into account the left end modif. We only do this if
	  the current oligomer contains as its first monomer the 
	  monomer that was the left end monomer of the inital polymer 
	  sequence:
       */
      res = 
	pxmchem_elemcompcalc_polymer_account_left_end_modif (polymer,
							     atomcountGPA,
							     polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for polymer's "
		   "left end modif.\n"),
		 __FILE__, __LINE__);
	  
	  return PXM_COMPCALC_FAILURE;
	}
    }
    
  
  if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF
      && oligomer->end_idx == len - 1)
    {
      /* Take into account the right end modif. We only do this if the
	 current oligomer contains as its last monomer the monomer
	 that was the right end monomer of the inital polymer
	 sequence:
      */
      res =
	pxmchem_elemcompcalc_polymer_account_right_end_modif (polymer,
							      atomcountGPA,
							      polchemdef);
      if (FALSE == res)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for polymer's "
		   "right end modif.\n"),
		 __FILE__, __LINE__);
	  
	  return PXM_COMPCALC_FAILURE;
	}
    }
  
  

  /******************** IONIZATION REQUIREMENTS**********************/

  /* For the moment the masses calculated are for a non-charged polymer:
     let's check if caller wants some kind of charge to be put on it.
     For this we also need to know what atom is responsible for the
     charge formation (proton or whatever).
  */
  res = pxmchem_elemcompcalc_polymer_account_ionizerule (ionizerule,
							 polchemdef->atomGPA,
							 atomcountGPA);
  if (FALSE == res)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the ionizerule.\n"),
	     __FILE__, __LINE__);
      
      return PXM_COMPCALC_FAILURE;
    }
  
  /* We now have a GPtrArray full of allocated atomcount instances,
   * from which we must construct a formula:
   */
  *formula = pxmchem_formula_make_string_from_atomcount_GPA (atomcountGPA);
  
  if (*formula == NULL)
    compcalcres = PXM_COMPCALC_FAILURE;

  /* We can now free the atomcountGPA, but only if we had to allocate it 
     at the beginning of our work.
   */
  if (array_allocated == TRUE)
    pxmchem_atomcount_GPA_free (atomcountGPA);
  
  return compcalcres;
}


gboolean
pxmchem_elemcompcalc_oligomer_account_cleaverule (PxmCleaveRule *lr_rule,
						     PxmOligomer *oligomer,
						     GPtrArray *atom_refGPA,
						     GPtrArray *acGPA)
{
  gint len = 0;
 
  gboolean res = FALSE;
  
  PxmMonomer *monomer = NULL;

  /* We get an array of atomcount instances into which we are asked to
     account for the cleaverule that is passed as parameter. A
     cleaverule is nothing but a conditional actform. The condition
     exercises on two criteria: the identity of the monomer that is
     concerned and the location of this monomer (is it the left end
     monomer of the oligomer or is it the monomer at the right end of
     the oligomer.

     Note that any of the two monomer codes (even both actually) in
     the cleaverule may be NULL. If they are non-NULL, then their
     corresponding actform has to be also non-NULL. There is a
     specific case where the condition may be respected and the
     actform must not be enforced: when the monomer is at the
     left/right end of the oligomer, but it is there not by virtue of
     the cleavage of the initial polymer sequence. For example:

     ATJQFFGM/LQJDFQSDFUERM/QSDFIOPUFNQDFOPIERM is a sequence that
     gets cleaved right of each M (show with the '/' char.). The
     cleavage thus leads to the formation of three oligomers:

     ATJQFFGM   LQJDFQSDFUERM   QSDFIOPUFNQDFOPIERM

     The cleaverule states that if 'M' is found on the right end of
     the generated oligomers, then the actform must be enforced.

     We would quietly perform this chemical modification on the three
     oligomers. But this would be erroneous for the third one, because
     the 'M' was actually NEVER cleaved, since it corresponded to the
     right end monomer of the initial polymer. This is why we have to
     not only check if the monomer identity/location condition is
     verified but also if it makes sense to enforce the chemical
     modification.
  */
  g_assert (acGPA != NULL);
  g_assert (atom_refGPA != NULL);
  
  len = oligomer->polymer->monomerGPA->len;


  /* The left couple of data.
   */
  if (lr_rule->left_code != NULL)
    {
      /* If there is a left code, then the actform must also be
	 there. Assert this.
      */
      g_assert (lr_rule->left_actform != NULL);
	  
      /* Get the monomer that is at the left end of the current
	 oligomer.
      */     
      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA,
				   oligomer->start_idx);
      g_assert (monomer != NULL);
	  
      /* Now check if the monomer's code matches the code that
	 is specified in the cleaverule.
      */
      if (0 == strcmp (lr_rule->left_code, monomer->code))
	{
	  if (oligomer->start_idx != 0)
	    {
	      /* The monomer at left end of current monomer matches!
		 And monomer was not the left end monomer of the
		 polymer sequence. Now do the chemical work proper.
	      */
	      res = pxmchem_actform_account_elemcompos (lr_rule->
							left_actform, 
							atom_refGPA,
							1 /*times*/, 
							acGPA);
	      if (FALSE == res)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			 _("%s@%d: failed to account for actform of cleaverule: '%s'\n"),
			 __FILE__, __LINE__, lr_rule->left_actform);
		  
		  return PXM_COMPCALC_FAILURE;
		}
	    }
	}
    }
  /* end of
     if (lr_rule->left_code != NULL)
  */
      
  /* The right couple of data.
   */
  if (lr_rule->right_code != NULL)
    {
      /* If there is a right code, then the actform must also be
	 there. Assert this.
      */
      g_assert (lr_rule->right_actform != NULL);
	  
      /* Get the monomer that is at the right end of the current
	 oligomer.
      */     
      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA,
				   oligomer->end_idx);
      g_assert (monomer != NULL);
	  
      /* Now check if the monomer's code matches the code that
	 is specified in the cleaverule.
      */
      if (0 == strcmp (lr_rule->right_code, monomer->code))
	{
	  if (oligomer->end_idx == len - 1)
	    {
	      /* The monomer at right end of current monomer matches!
		 And monomer was not the right end monomer of the
		 polymer sequence. Now do the chemical work proper.
	      */
	      res = pxmchem_actform_account_elemcompos (lr_rule->
							right_actform, 
							atom_refGPA,
							1 /*times*/, 
							acGPA);
	      if (FALSE == res)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			 _("%s@%d: failed to account for actform of cleaverule: '%s'\n"),
			 __FILE__, __LINE__, lr_rule->right_actform);
		  
		  return PXM_COMPCALC_FAILURE;
		}
	    }
	  
	}
    }
  /* end of
     if (lr_rule->right_code != NULL)
  */
  return PXM_COMPCALC_SUCCESS;
}
