/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "polyxedit-globals.h"
#include "polyxmass-window-mngmt.h"
#include "polyxmass-ui-window-mngmt.h"


/* columns */

enum
{
  /* The string representing the module owning the window.
   */
  COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING,

  /* The string representation of the window's pointer.
   */
  COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING,
  
  /* The string of the sequence's name.
   */
  COLUMN_AVAIL_WINDOWS_SEQ_NAME,

  /* The string standard description of the purpose of the window.
   */
  COLUMN_AVAIL_WINDOWS_WND_DESC,

  /* The string comment for the window.
   */
  COLUMN_AVAIL_WINDOWS_WND_COMMENT,

  /* The pointer of the window.
   */
  COLUMN_AVAIL_WINDOWS_WND_POINTER,

  /* The pointer of the sequence editor window.
   */
  COLUMN_AVAIL_WINDOWS_SEQED_WND_POINTER,

  /* The pointer of the parent window.
   */
  COLUMN_AVAIL_WINDOWS_PARENT_WND_POINTER,

  /* The pointer of the editor context.
   */
  COLUMN_AVAIL_WINDOWS_EDITCTXT_POINTER,

  /* The gboolean value telling of the window can report.
   */
  COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT,

  /* The pointer to the PxmWinMngmt instance.
   */
  COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER,

  /* The pointer to the module array containing the PxmWinMngmt
     instance..
   */
  COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER,


  COLUMN_AVAIL_WINDOWS_COL_COUNT
};



/* This window will allow the user to manage all the windows that have
   registered themselves upon creation.
*/
GtkWidget *
polyxmass_window_mngmt_wnd_setup (GtkWidget *parent_wnd)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;



  g_assert (parent_wnd != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxmass-window-mngmt.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "window_mngmt_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "window_mngmt_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the window mngmt window\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt passed as
     parameter.
  */
  g_object_set_data (G_OBJECT (parent_wnd), "window_mngmt_wnd",
		     window);
  

  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);

  widget = glade_xml_get_widget (xml, "available_windows_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "available_windows_vbox", widget);

  widget = glade_xml_get_widget (xml, "wnd_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "wnd_id_number_entry", widget);

  widget = glade_xml_get_widget (xml, "sequence_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "sequence_id_number_entry", widget);

  widget = glade_xml_get_widget (xml, "wnd_description_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "wnd_description_entry", widget);

  widget = glade_xml_get_widget (xml, "wnd_comment_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "wnd_comment_entry", widget);

  widget = glade_xml_get_widget (xml, "wnd_can_report_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "wnd_can_report_checkbutton", widget);



  widget = glade_xml_get_widget (xml, "show_window_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "show_window_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_show_window_button), 
		    window);

  widget = glade_xml_get_widget (xml, "hide_window_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "hide_window_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_hide_window_button), 
		    window);


  widget = glade_xml_get_widget (xml, "report_to_clipboard_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "report_to_clipboard_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_report_clipboard_button), 
		    window);

  widget = glade_xml_get_widget (xml, "report_file_choose_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "report_file_choose_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_report_file_choose_button), 
		    window);

  widget = glade_xml_get_widget (xml, "overwrite_report_to_file_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "overwrite_report_to_file_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_report_file_overwrite_button), 
		    window);

  widget = glade_xml_get_widget (xml, "append_report_to_file_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "append_report_to_file_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_window_mngmt_wnd_report_file_append_button), 
		    window);

  widget = glade_xml_get_widget (xml, "current_file_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "current_file_entry", widget);



  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxmass_window_mngmt_wnd_delete_event), 
		    parent_wnd);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxmass_window_mngmt_wnd_destroy_event), 
		    parent_wnd);

  gtk_selection_add_target (GTK_WIDGET (window),
			    GDK_SELECTION_PRIMARY,
			    GDK_SELECTION_TYPE_STRING,
			    1);

  gtk_selection_add_target (GTK_WIDGET (window),
			    GDK_SELECTION_CLIPBOARD,
			    GDK_SELECTION_TYPE_STRING,
			    1);

  g_signal_connect (G_OBJECT (window),
		    "selection_get",
		    G_CALLBACK (polyxmass_window_mngmt_wnd_clipboard_selection_get),
		    NULL);

  g_signal_connect (G_OBJECT (window),
		    "selection_clear_event",
		    G_CALLBACK (polyxmass_window_mngmt_wnd_clipboard_selection_clear),
		    NULL);


  /* Now that we did all the first-level setting-up, we can go through the
     preparation of the treeview in the available_windows_vbox vbox widget.
  */
  polyxmass_window_mngmt_wnd_setup_windows_treeview (window);
  
  gtk_widget_show_all (GTK_WIDGET (window));


  /* We have finished setting up the window, and so also using the xml
     data, unref them.
   */
  g_object_unref (G_OBJECT (xml));
  
  
  return window;
}


gboolean
polyxmass_window_mngmt_wnd_setup_windows_treeview (GtkWidget *window)
{
 
  GtkWidget *sw = NULL;
  GtkWidget *available_windows_vbox = NULL;
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeIter treeiter_module ;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;
  GtkTreeSelection* selection = NULL;

  PxmWinMngmt *winmngmt = NULL;
  
  gchar *wnd_ptr_str = NULL;
  
  gint iter = 0;
  gint jter = 0;
    
  GPtrArray *winmngmtGPA = NULL;
  

  g_assert (window != NULL);
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  

  /* Get a pointer to the vbox where we'll pack all the stuff.
   */
  available_windows_vbox = g_object_get_data (G_OBJECT (window),
					      "available_windows_vbox");
  g_assert (available_windows_vbox != NULL);
  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (available_windows_vbox), sw, TRUE, TRUE, 0);  
  
  
  /* Create the treeview model.
   */
  model = (GtkTreeModel *) gtk_tree_store_new 
    (COLUMN_AVAIL_WINDOWS_COL_COUNT,

     /* The string representing the module owning the window.
      */
     G_TYPE_STRING,

     /* The string representation of the window's pointer.
      */
     G_TYPE_STRING,
  
     /* The string of the sequence's name.
      */
     G_TYPE_STRING,

     /* The string standard description of the purpose of the window.
      */
     G_TYPE_STRING,

     /* The string comment for the window.
      */
     G_TYPE_STRING,

     /* The pointer of the window.
      */
     G_TYPE_POINTER,

     /* The pointer of the sequence editor window.
      */
     G_TYPE_POINTER,

     /* The pointer of the parent window.
      */
     G_TYPE_POINTER,

     /* The pointer of the editor context.
      */
     G_TYPE_POINTER,

     /* The gboolean value telling of the window can report.
      */
     G_TYPE_BOOLEAN,

     /* The pointer to the PxmWinMngmt instance.
      */
     G_TYPE_POINTER,

     /* The pointer to the module array containing the PxmWinMngmt
	instance..
      */
     G_TYPE_POINTER);
  


  /* We will fill the model with data, but we have to iterate in the
     array that stores the data for all the windows...
  */


  /* Remember that the array that we get is an array of arrays.  This
     array contains an array for each group of PxmWinMngmt instances
     corresponding to different modules (polyxedit,
     polyxcalc...).
  */
  for (iter = 0; iter < polyxmass_winmngmtGPA_GPA->len; iter++)
    {
      winmngmtGPA = g_ptr_array_index (polyxmass_winmngmtGPA_GPA, iter);
      g_assert (winmngmtGPA != NULL);

      /* We have an array of PxmWinMngmt instances. If we are here for
	 the first time for this new array, we have to append a new
	 item to the store. This item will only contain the first
	 column item : the module string. That string is contained in
	 the PxmWinMngmt instances, so we just get the first of the
	 array (an array cannot be empty).
      */
      winmngmt = g_ptr_array_index (winmngmtGPA, 0);
      g_assert (winmngmt != NULL);

      gtk_tree_store_append ((GtkTreeStore *) model, 
			     &treeiter_module, NULL);

      gtk_tree_store_set 
	((GtkTreeStore *) model, &treeiter_module,
	 
	 /* The string representing the module owning the window.
	  */
	 COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, winmngmt->module,

	 /* The pointer to the module array containing the PxmWinMngmt
	    instance..
	 */
	 COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER, winmngmtGPA,

	 -1);
      	 
      /* And now iterate in the array and analyze each winmngmt in it.
       */
      for (jter = 0 ; jter < winmngmtGPA->len ; jter++)
	{
	  winmngmt = g_ptr_array_index (winmngmtGPA, jter);
	  g_assert (winmngmt != NULL);

	  wnd_ptr_str = g_strdup_printf ("%p", winmngmt->wnd);
	  
	  gtk_tree_store_append ((GtkTreeStore *) model, 
				 &treeiter, &treeiter_module);

	  gtk_tree_store_set 
	    ((GtkTreeStore *) model, &treeiter,

	     /* The string representing the module owning the window.
	      *
	      COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, 
	     */

	     /* The string representation of the window's pointer.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING, wnd_ptr_str,
  
	     /* The string of the sequence's name.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQ_NAME, winmngmt->sequence_name,

	     /* The string standard description of the purpose of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_DESC, winmngmt->desc,

	     /* The string comment for the window.
	      *
	      COLUMN_AVAIL_WINDOWS_WND_COMMENT,
	     */

	     /* The pointer of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER, winmngmt->wnd,

	     /* The pointer of the sequence editor window.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQED_WND_POINTER, 
	     winmngmt->sequence_editor_wnd,

	     /* The pointer of the parent window.
	      */
	     COLUMN_AVAIL_WINDOWS_PARENT_WND_POINTER, winmngmt->parent_wnd,

	     /* The pointer of the editor context (that is the pointer
		that is used to identify sequences (Sequence ID
		Number).
	      */
	     COLUMN_AVAIL_WINDOWS_EDITCTXT_POINTER, winmngmt->editctxt,

	     /* The gboolean value telling of the window can report.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT, winmngmt->can_report,

	     /* The pointer to the PxmWinMngmt instance.
	      */
	     COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, winmngmt,

	     /* The pointer to the module array containing the PxmWinMngmt
		instance..
	     */
	     COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER,  winmngmtGPA,

	     -1);

	  g_free (wnd_ptr_str);
	}
      /* end of 
	 for (jter = 0 ; jter < winmngmtGPA->len ; jter++)
      */
    }
  /* end of 
     for (iter = 0; iter < GPA->len; iter++)
  */
  

  /* Set the window a datum with a pointer to the mode, so that later
     the model is accessible (add/remove button handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "winmngmt_treeview_model", model);
    
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  
  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "winmngmt_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);

  /* "changed" is emitted by the selection object in the treeview each
     time the selection changes.
   */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_signal_connect 
    (G_OBJECT (selection), "changed", 
     G_CALLBACK 
     (polyxmass_window_mngmt_wnd_winmngmt_treeview_selection_changed),
     window);
  
  /* The string representing the module owning the window.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes 
    (_("Module"),
     renderer,
     "text",
     COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING,
     NULL);
  
  gtk_tree_view_column_set_sort_column_id 
    (column, 
     COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* The string representation of the window's pointer.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes 
    (_("Window ID"),
     renderer,
     "text",
     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING,
     NULL);
  
  gtk_tree_view_column_set_sort_column_id 
    (column, 
     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* The string of the sequence's name.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes 
    (_("Seq. Name"),
     renderer,
     "text",
     COLUMN_AVAIL_WINDOWS_SEQ_NAME,
     NULL);
  
  gtk_tree_view_column_set_sort_column_id 
    (column, 
     COLUMN_AVAIL_WINDOWS_SEQ_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* The string standard description of the purpose of the window.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes 
    (_("Window Desc."),
     renderer,
     "text",
     COLUMN_AVAIL_WINDOWS_WND_DESC,
     NULL);
  
  gtk_tree_view_column_set_sort_column_id 
    (column, 
     COLUMN_AVAIL_WINDOWS_WND_DESC);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* The gboolean value telling of the window can report.
   */
  renderer = gtk_cell_renderer_toggle_new ();

  column = gtk_tree_view_column_new_with_attributes 
    (_("Can Report"),
     renderer,
     "active",
     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT,
     NULL);
  
  gtk_tree_view_column_set_sort_column_id 
    (column, 
     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Finally add the treeview to the scrooll window.
   */
  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (available_windows_vbox);
  
  return TRUE;
}


void
polyxmass_window_mngmt_wnd_winmngmt_treeview_add_winmngmt (GtkWidget *window,
							   PxmWinMngmt *winmngmt,
							   GPtrArray *winmngmtGPA)
{
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeIter treeiter_module ;
 
  
  gboolean module_item_valid = FALSE;

  GPtrArray *moduleGPA = NULL;
  
   gchar *wnd_ptr_str = NULL;

     

 
  /* This function is called when the user has created a new window
     that has registered itself. We want to update the list of windows
     currently opened.
  */

  g_assert (window != NULL);
  g_assert (winmngmt != NULL);
  g_assert (winmngmtGPA != NULL);
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  
  
  model =  g_object_get_data (G_OBJECT (window), "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  /* Get the first treeiter in the model (this is a module iter, since
     the first we get in the model is of first level). The module item
     has a pointer to the GPtrArray of items that it contains. See
     polyxmass_window_mngmt_wnd_setup_windows_treeview() for
     reference.
   */
  module_item_valid = gtk_tree_model_get_iter_first (model, &treeiter_module);

  while (module_item_valid == TRUE)
    {
      gtk_tree_model_get (model, &treeiter_module, 
			  COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER, &moduleGPA,
			  -1);

      if (moduleGPA == winmngmtGPA)
	{
	  /* Great, we are iterating in the module_item that
	     corresponds to the winmngmtGPA passed as parameter. That means that
	     all we have to do is to add an item there.
	  */
	  gtk_tree_store_append ((GtkTreeStore *) model,
				 &treeiter, &treeiter_module);

	  wnd_ptr_str = g_strdup_printf ("%p", winmngmt->wnd);
	  
	  gtk_tree_store_set 
	    ((GtkTreeStore *) model, &treeiter,

	     /* The string representing the module owning the window.
	      *
	      COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, 
	     */

	     /* The string representation of the window's pointer.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING, wnd_ptr_str,
  
	     /* The string of the sequence's name.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQ_NAME, winmngmt->sequence_name,

	     /* The string standard description of the purpose of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_DESC, winmngmt->desc,

	     /* The string comment for the window.
	      *
	      COLUMN_AVAIL_WINDOWS_WND_COMMENT,
	     */

	     /* The pointer of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER, winmngmt->wnd,

	     /* The pointer of the sequence editor window.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQED_WND_POINTER, 
	     winmngmt->sequence_editor_wnd,

	     /* The pointer of the parent window.
	      */
	     COLUMN_AVAIL_WINDOWS_PARENT_WND_POINTER, winmngmt->parent_wnd,

	     /* The pointer of the editor context (that is the pointer
		that is used to identify sequences (Sequence ID
		Number).
	      */
	     COLUMN_AVAIL_WINDOWS_EDITCTXT_POINTER, winmngmt->editctxt,

	     /* The gboolean value telling of the window can report.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT, winmngmt->can_report,

	     /* The pointer to the PxmWinMngmt instance.
	      */
	     COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, winmngmt,

	     /* The pointer to the module array containing the PxmWinMngmt
		instance..
	     */
	     COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER,  winmngmtGPA,

	     -1);

	  g_free (wnd_ptr_str);	  

	  return;
	}
      
      module_item_valid = gtk_tree_model_iter_next (model, &treeiter_module);

      continue;
    }
  
  /* If we are here, that means that we have not found any moduleGPA
     corresponding to our parameter. This must be because the
     'winmngmt' object is the sole item in the 'winmngmtGPA'. This is
     because being the first item in hat array, the array is not yet
     represented in the treeview. So we have to do that work: create a
     new root item.
  */
  gtk_tree_store_append ((GtkTreeStore *) model, 
			 &treeiter_module, NULL);
  
  gtk_tree_store_set 
    ((GtkTreeStore *) model, &treeiter_module,
     
     /* The string representing the module owning the window.
      */
     COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, winmngmt->module,
     
     /* The pointer to the module array containing the PxmWinMngmt
	instance..
     */
     COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER, winmngmtGPA,
     
     -1);
  
  /* And now that we have the parent item, we can put the winmngmt
     item in it.
  */
  wnd_ptr_str = g_strdup_printf ("%p", winmngmt->wnd);
	  
  gtk_tree_store_append ((GtkTreeStore *) model, 
			 &treeiter, &treeiter_module);

  gtk_tree_store_set 
    ((GtkTreeStore *) model, &treeiter,

     /* The string representing the module owning the window.
      *
      COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, 
     */

     /* The string representation of the window's pointer.
      */
     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING, wnd_ptr_str,
  
     /* The string of the sequence's name.
      */
     COLUMN_AVAIL_WINDOWS_SEQ_NAME, winmngmt->sequence_name,

     /* The string standard description of the purpose of the window.
      */
     COLUMN_AVAIL_WINDOWS_WND_DESC, winmngmt->desc,

     /* The string comment for the window.
      *
      COLUMN_AVAIL_WINDOWS_WND_COMMENT,
     */

     /* The pointer of the window.
      */
     COLUMN_AVAIL_WINDOWS_WND_POINTER, winmngmt->wnd,

     /* The pointer of the sequence editor window.
      */
     COLUMN_AVAIL_WINDOWS_SEQED_WND_POINTER, 
     winmngmt->sequence_editor_wnd,

     /* The pointer of the parent window.
      */
     COLUMN_AVAIL_WINDOWS_PARENT_WND_POINTER, winmngmt->parent_wnd,

     /* The pointer of the editor context (that is the pointer
	that is used to identify sequences (Sequence ID
	Number).
     */
     COLUMN_AVAIL_WINDOWS_EDITCTXT_POINTER, winmngmt->editctxt,

     /* The gboolean value telling of the window can report.
      */
     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT, winmngmt->can_report,

     /* The pointer to the PxmWinMngmt instance.
      */
     COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, winmngmt,

     /* The pointer to the module array containing the PxmWinMngmt
	instance..
     */
     COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER,  winmngmtGPA,

     -1);

  g_free (wnd_ptr_str);
  
}



void
polyxmass_window_mngmt_wnd_winmngmt_treeview_remove_winmngmt (GtkWidget *window,
							      PxmWinMngmt *winmngmt)
{
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeIter treeiter_module ;
 
  PxmWinMngmt *winmngmt_iter = NULL;
  
  //  gint iter = 0;
  // gint jter = 0;

  gboolean module_item_valid = FALSE;
  gboolean item_valid = FALSE;
      
 
  /* This function is called when the user has closed a window, that
     that window has un_registered, and we want to update the list of
     windows currently opened. This is required because otherwise
     there is a risk that the user asks some action to be performed on
     a window that does not exist anymore.
   */

  g_assert (window != NULL);
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  
  
  model =  g_object_get_data (G_OBJECT (window), "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  /* Get the first treeiter in the model (this is a module iter, since
     the first we get in the model is of first level).
   */
  module_item_valid = gtk_tree_model_get_iter_first (model, &treeiter_module);

  while (module_item_valid == TRUE)
    {
      /* Now get an iterator to the children items, which are indeed the
	 winmngmt items...
      */
      item_valid = gtk_tree_model_iter_children (model,
						 &treeiter, &treeiter_module);

      while (item_valid == TRUE)
	{
	  gtk_tree_model_get (model, &treeiter, 
			      COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, &winmngmt_iter,
			      -1);
	  
	  if (winmngmt_iter == winmngmt)
	    {
	      gtk_tree_store_remove ((GtkTreeStore *) model,
				     &treeiter);

	      /* At this point, if we have removed the very last item, we
		 should remove the module_item also.
	      */
	      if (FALSE == gtk_tree_model_iter_has_child (model, &treeiter_module))
		gtk_tree_store_remove ((GtkTreeStore *) model,
				       &treeiter_module);

	      return ;
	    }
	  
	  item_valid = gtk_tree_model_iter_next (model, &treeiter);

	  continue;
	}
      
      /* At this point, we have to go to the next module item, as we still
	 have not found the proper winmngmt item.
      */
      module_item_valid = gtk_tree_model_iter_next (model, &treeiter_module);

      continue;
    }
}






void
polyxmass_window_mngmt_wnd_winmngmt_treeview_update (GtkWidget *window)
{
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeIter treeiter_module ;
 
  PxmWinMngmt *winmngmt = NULL;
  
  gchar *wnd_ptr_str = NULL;
  
  gint iter = 0;
  gint jter = 0;
    
  GPtrArray *winmngmtGPA = NULL;
  
  /* This function is called when the user has closed a window, that
     that window has un_registered, and we want to update the list of
     windows currently opened. This is required because otherwise
     there is a risk that the user asks some action to be performed on
     a window that does not exist anymore.
   */

  g_assert (window != NULL);
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  
  
  model =  g_object_get_data (G_OBJECT (window), "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  /* Clear all the items in the tree store. If the array is empty,
     just return after having cleared it.
   */
  gtk_tree_store_clear (GTK_TREE_STORE (model));

  if (polyxmass_winmngmtGPA_GPA->len == 0)
    return;
  
  for (iter = 0; iter < polyxmass_winmngmtGPA_GPA->len; iter++)
    {
      winmngmtGPA = g_ptr_array_index (polyxmass_winmngmtGPA_GPA, iter);
      g_assert (winmngmtGPA != NULL);

      /* We have an array of PxmWinMngmt instances. If we are here for
	 the first time for this new array, we have to append a new
	 item to the store. This item will only contain the first
	 column item : the module string. That string is contained in
	 the PxmWinMngmt instances, so we just get the first of the
	 array (an array cannot be empty).
      */
      winmngmt = g_ptr_array_index (winmngmtGPA, 0);
      g_assert (winmngmt != NULL);

      gtk_tree_store_append ((GtkTreeStore *) model, 
			     &treeiter_module, NULL);

      gtk_tree_store_set 
	((GtkTreeStore *) model, &treeiter_module,
	 
	 /* The string representing the module owning the window.
	  */
	 COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, winmngmt->module,

	 /* The pointer to the module array containing the PxmWinMngmt
	    instance..
	 */
	 COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER, winmngmtGPA,

	 -1);
      	 
      /* And now iterate in the array and analyze each winmngmt in it.
       */
      for (jter = 0 ; jter < winmngmtGPA->len ; jter++)
	{
	  winmngmt = g_ptr_array_index (winmngmtGPA, jter);
	  g_assert (winmngmt != NULL);

	  wnd_ptr_str = g_strdup_printf ("%p", winmngmt->wnd);
	  
	  gtk_tree_store_append ((GtkTreeStore *) model, 
				 &treeiter, &treeiter_module);

	  gtk_tree_store_set 
	    ((GtkTreeStore *) model, &treeiter,

	     /* The string representing the module owning the window.
	      *
	      COLUMN_AVAIL_WINDOWS_OWNER_MODULE_STRING, 
	     */

	     /* The string representation of the window's pointer.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER_STRING, wnd_ptr_str,
  
	     /* The string of the sequence's name.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQ_NAME, winmngmt->sequence_name,

	     /* The string standard description of the purpose of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_DESC, winmngmt->desc,

	     /* The string comment for the window.
	      *
	      COLUMN_AVAIL_WINDOWS_WND_COMMENT,
	     */

	     /* The pointer of the window.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_POINTER, winmngmt->wnd,

	     /* The pointer of the sequence editor window.
	      */
	     COLUMN_AVAIL_WINDOWS_SEQED_WND_POINTER, 
	     winmngmt->sequence_editor_wnd,

	     /* The pointer of the parent window.
	      */
	     COLUMN_AVAIL_WINDOWS_PARENT_WND_POINTER, winmngmt->parent_wnd,

	     /* The pointer of the editor context (that is the pointer
		that is used to identify sequences (Sequence ID
		Number).
	      */
	     COLUMN_AVAIL_WINDOWS_EDITCTXT_POINTER, winmngmt->editctxt,

	     /* The gboolean value telling of the window can report.
	      */
	     COLUMN_AVAIL_WINDOWS_WND_CAN_REPORT, winmngmt->can_report,

	     /* The pointer to the PxmWinMngmt instance.
	      */
	     COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, winmngmt,

	     /* The pointer to the module array containing the PxmWinMngmt
		instance..
	     */
	     COLUMN_AVAIL_WINDOWS_MODULE_ARRAY_POINTER,  winmngmtGPA,

	     -1);

	  g_free (wnd_ptr_str);
	}
      /* end of 
	 for (jter = 0 ; jter < winmngmtGPA->len ; jter++)
      */
    }
  /* end of 
     for (iter = 0; iter < GPA->len; iter++)
  */
  
  return;
}


void
polyxmass_window_mngmt_wnd_winmngmt_treeview_selection_changed (GtkTreeSelection 
								*selection,
								gpointer data)
{
  GtkWidget *window = data;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;


  gint depth = -1;

  PxmWinMngmt *winmngmt = NULL;
  
      


  g_assert (window != NULL);
  
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == result)
    return;
  
  /* OK, at this point we know a selection is there.
   */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    {
      polyxmass_window_mngmt_wnd_reset_winmngmt_data (window); 

      return;
    }
  
  /* OK, now we now the selection is a winmngmt instance item and not
     a root item, a module item. Get the pointer to the winmngmt
     instance right off the item and set the corresponding data to
     their different widgets...
  */
  gtk_tree_model_get (model, &tree_iter_sel, 
		      COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, &winmngmt, -1);
  
  g_assert (winmngmt != NULL);
  
  /* Now that we have the pointer to the winmngmt instance for which
     we will document the member data to their respective widgets, we
     can start doing the work.
  */
  polyxmass_window_mngmt_wnd_update_winmngmt_data (window, winmngmt);
  
}


gint
polyxmass_window_mngmt_wnd_update_winmngmt_data (GtkWidget *window, 
						 PxmWinMngmt *winmngmt)
{
  GtkWidget *widget = NULL;
  
  g_assert (window != NULL);
  g_assert (winmngmt != NULL);
  
  gchar *help = NULL;
  
  widget = g_object_get_data (G_OBJECT (window), "wnd_id_number_entry");
  g_assert (widget != NULL);

  help = g_strdup_printf ("%p", winmngmt->wnd);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  widget = g_object_get_data (G_OBJECT (window), "sequence_id_number_entry");
  g_assert (widget != NULL);

  help = g_strdup_printf ("%p", winmngmt->editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_description_entry");
  g_assert (widget != NULL);
  g_assert (winmngmt->desc != NULL);
  gtk_entry_set_text (GTK_ENTRY (widget), winmngmt->desc);
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_comment_entry");
  g_assert (widget != NULL);
  if (winmngmt->comment != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), winmngmt->comment);
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_can_report_checkbutton");
  g_assert (widget != NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
				winmngmt->can_report);
  
  return 1;
}


gint
polyxmass_window_mngmt_wnd_reset_winmngmt_data (GtkWidget *window)
{
  GtkWidget *widget = NULL;
  
  g_assert (window != NULL);
  
  
  widget = g_object_get_data (G_OBJECT (window), "wnd_id_number_entry");
  g_assert (widget != NULL);

  gtk_entry_set_text (GTK_ENTRY (widget), "");
  
  widget = g_object_get_data (G_OBJECT (window), "sequence_id_number_entry");
  g_assert (widget != NULL);

  gtk_entry_set_text (GTK_ENTRY (widget), "");
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_description_entry");
  g_assert (widget != NULL);
  gtk_entry_set_text (GTK_ENTRY (widget), "");
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_comment_entry");
  g_assert (widget != NULL);
    gtk_entry_set_text (GTK_ENTRY (widget), "");
  

  widget = g_object_get_data (G_OBJECT (window), "wnd_can_report_checkbutton");
  g_assert (widget != NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
				FALSE);
  
  return 1;
}




gint
polyxmass_window_mngmt_wnd_show_window_button (GtkWidget *button,
					       gpointer data)
{
  GtkWidget *window = data;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;


  gint depth = -1;

  PxmWinMngmt *winmngmt = NULL;
  
      


  g_assert (window != NULL);
  
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == result)
    return 0;
  
  /* OK, at this point we know a selection is there.
   */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    return 0;
  
  
  /* OK, now we now the selection is a winmngmt instance item and not
     a root item, a module item. Get the pointer to the winmngmt
     instance right off the item and set the corresponding data to
     their different widgets...
  */
  gtk_tree_model_get (model, &tree_iter_sel, 
		      COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, &winmngmt, -1);
  
  g_assert (winmngmt != NULL);
  
  /* At this point, we can make sure the window is visible.
   */
  gtk_window_present (GTK_WINDOW (winmngmt->wnd));
    
  return 1;
}


gint
polyxmass_window_mngmt_wnd_report_file_choose_button (GtkWidget *button,
						      gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *dialog = NULL;
  GtkWidget *entry = NULL;
  
  gchar *filename = NULL;


  g_assert (window != NULL);
  
  /* Create the file selector */
  dialog = 
    gtk_file_chooser_dialog_new (_("Write Report To File"),
				 GTK_WINDOW (window),
				 GTK_FILE_CHOOSER_ACTION_SAVE,
				 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				 GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				 NULL);
  
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      
      /* Display the selected file to the correponding entry widget.
       */
      entry = g_object_get_data (G_OBJECT (window), "current_file_entry");
      g_assert (entry != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (entry), filename);
      
      g_free (filename);
      
      /* We can destroy the file chooser dialog.
       */
      gtk_widget_destroy (dialog);
      
      return 1;
    }
  else
    {
      g_free (filename);
      
      /* We can destroy the file chooser dialog.
       */
      gtk_widget_destroy (dialog);
      
      return 1;
    }
  
  /* We can destroy the file chooser dialog.
   */
  gtk_widget_destroy (dialog);

  return 1;
}


gint
polyxmass_window_mngmt_wnd_hide_window_button (GtkWidget *button,
					       gpointer data)
{
  GtkWidget *window = data;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;


  gint depth = -1;

  PxmWinMngmt *winmngmt = NULL;
  
      


  g_assert (window != NULL);
  
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == result)
    return 0;
  
  /* OK, at this point we know a selection is there.
   */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    return 0;
  
  
  /* OK, now we now the selection is a winmngmt instance item and not
     a root item, a module item. Get the pointer to the winmngmt
     instance right off the item and set the corresponding data to
     their different widgets...
  */
  gtk_tree_model_get (model, &tree_iter_sel, 
		      COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, &winmngmt, -1);
  
  g_assert (winmngmt != NULL);
  
  /* At this point, we can make sure the window is visible.
   */
  gtk_widget_hide (GTK_WIDGET (winmngmt->wnd));
    
  return 1;
}



void
polyxmass_window_mngmt_wnd_clipboard_selection_get (GtkWidget *widget,
						    GtkSelectionData *
						    selection_data,
						    guint info,
						    guint time_stamp,
						    gpointer data)
{
  gchar *string = NULL;
  
  /* This function is called when another process asks to paste the contents
     that we have claimed previously to have as clipboard data. 

     We only deal with GDK_SELECTION_CLIPBOARD selections here: the
     data we are providing to other processes are not "selectable"
     using the mouse.
  */
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD)
    {
      string = g_object_get_data (G_OBJECT (widget),
				  "CLIPBOARD_SELECTION");
      
      if (string == NULL)
	return;
      
      /*
	debug_printf (("The string is %s\n", string));
      */
      
      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      (const guchar *) string, strlen (string));
      
      return;
    }
  else if (selection_data->selection == GDK_SELECTION_PRIMARY)
    {
      string = g_object_get_data (G_OBJECT (widget),
				  "CLIPBOARD_SELECTION");
      
      if (string == NULL)
	return;
      
      /*
	debug_printf (("The string is %s\n", string));
      */

      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      (const guchar *) string, strlen (string));
      
      return;
    }

  return;
}

gboolean
polyxmass_window_mngmt_wnd_clipboard_selection_clear (GtkWidget *widget,
						      GdkEventSelection *
						      event,
						      gpointer data)
{

  /* Some other process has claimed ownership of the selection.
     If the selection is of type GDK_SELECTION_CLIPBOARD, then we
     have to clear the selection that we have set in a previous call.
  */
  gchar *string = NULL;


  if (event->selection == GDK_SELECTION_CLIPBOARD
      || event->selection == GDK_SELECTION_PRIMARY)
    {
      string = g_object_get_data (G_OBJECT (widget),
				  "CLIPBOARD_SELECTION");
      
      if (string == NULL)
	/* Propagate the signal because we did not perform anything.
	 */
	return FALSE;

      /* Setting to NULL the data, will automatically g_free() it 
	 because it was set as a full string datum.
      */
      g_object_set_data (G_OBJECT (widget),
			 "CLIPBOARD_SELECTION", NULL);
      
      return TRUE;
    }
  
  return FALSE;
}





/* ACTUAL WINDOW CONTENTS-REPORTING
 */
gchar *
polyxmass_window_mngmt_wnd_report_prepare (GtkWidget *window)
{
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;


  gint depth = -1;

  PxmWinMngmt *winmngmt = NULL;
  PxmReportOpt *reportopt = NULL;
  
  PxmProp *prop = NULL;

  gchar *report = NULL;
  
  g_assert (window != NULL);
  
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == result)
    return NULL;
  
  /* OK, at this point we know a selection is there.
   */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    return NULL;
  
  
  /* OK, now we now the selection is a winmngmt instance item and not
     a root item, a module item. Get the pointer to the winmngmt
     instance right off the item and set the corresponding data to
     their different widgets...
  */
  gtk_tree_model_get (model, &tree_iter_sel, 
		      COLUMN_AVAIL_WINDOWS_WINMNGMT_POINTER, &winmngmt, -1);
  
  g_assert (winmngmt != NULL);

  /* Check if we are dealing with a window coming from a polymer
     sequence editing context... If so we should have a PxmReportOpt
     prop object in the editctxt->propGPA array of prop... If there is
     not such PxmReportOpt already, then that means that the user did
     not configure the reporting options yet for the current sequence
     editor context. Never mind, we create one with default values !
   */
  if (winmngmt->editctxt != NULL)
    {
      prop = libpolyxmass_prop_find_prop (winmngmt->editctxt->propGPA,
					  NULL, NULL,
					  "REPORTOPT", 
					  NULL, PXM_UNDEF_PROP_CMP);
      
      if (prop == NULL)
	{
	  /* There is no such property, thus we have to create one.
	   */
	  prop = libpolyxmass_reportopt_prop_new ();

	  reportopt = (PxmReportOpt *) prop->data;

	  libpolyxmass_reportopt_set_default (reportopt);

	  g_ptr_array_add (winmngmt->editctxt->propGPA, prop);
	}
      else
	{
	  reportopt = (PxmReportOpt *) prop->data;
	}
    }


  report = winmngmt->make_report (reportopt, winmngmt);

  return report;
}



gint
polyxmass_window_mngmt_wnd_report_clipboard_button (GtkWidget *button,
							      gpointer data)
{
  GtkWidget *window = data;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gchar *report = NULL;
  
  gint depth = -1;

  gboolean result = FALSE;

  g_assert (window != NULL);
  

  /* Make sure we do not ask that a report be made if no window
     item is currently selected (and available in the treeview).
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }
  
  /* OK, at this point we know a selection is there.
   */
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }

  /* OK, we can finally ask that a report be made for the currently
     selected window item.
  */
  report = polyxmass_window_mngmt_wnd_report_prepare (window);
  g_assert (report != NULL);

  /* At this point we have the report string. Since the user asked 
     that the report be sent to the clipboard, we have to do that.
 
     We can claim ownership of the clipboard selection after having
     set the 'report' string available to the window, in case some
     process asks this text. See
     polyxmass_window_mngmt_wnd_clipboard_selection_get () to see how
     this is done.
  */
  g_object_set_data_full (G_OBJECT (window),
			  "CLIPBOARD_SELECTION", report,
			  (GDestroyNotify) g_free);
  
  gtk_selection_owner_set (GTK_WIDGET (window),
			   GDK_SELECTION_CLIPBOARD,
			   GDK_CURRENT_TIME);
  
  gtk_selection_owner_set (GTK_WIDGET (window),
			   GDK_SELECTION_PRIMARY,
			   GDK_CURRENT_TIME);
  
  
  return 1;
}



gint
polyxmass_window_mngmt_wnd_report_file_overwrite_button (GtkWidget *button,
							 gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *entry = NULL;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  FILE *filep = NULL;
    
  gchar *report = NULL;
  gchar *filename = NULL;
  
  gboolean selection_result = FALSE;
 
  gint result = -1;
  gint depth = -1;



  g_assert (window != NULL);


  /* Make sure we do not ask that a report be made if no window
     item is currently selected (and available in the treeview).
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  selection_result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == selection_result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }
  
  /* OK, at this point we know a selection is there.
   */
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }

  /* OK, we can finally ask that a report be made for the currently
     selected window item. This is the string we'll write to file.
  */
  report = polyxmass_window_mngmt_wnd_report_prepare (window);
  g_assert (report != NULL);


  entry = g_object_get_data (G_OBJECT (window), "current_file_entry");
  g_assert (entry != NULL);
  
  /* filename is NOT allocated.
   */
  filename = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  g_assert (filename != NULL);
  
  if (strlen (filename) <= 0)
    {
      /* Echo a failure message to the user.
       */
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Filename is not set"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      g_free (report);
      
      return -1;
    }
  
  /* Now that we have a filename, we should try to open it in overwrite mode.
   */
  filep = fopen (filename, "w");
  
  if (filep == NULL)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to open file"),
					POLYXMASS_NORM_MSG_TIMEOUT);

      g_critical (_("%s@%d: failed to open file: '%s'\n"),
		  __FILE__, __LINE__, filename);
      
      g_free (report);
      
      return -1;
    }

  /* We can now write to the file.
   */
  result = fputs (report, filep);

  fclose (filep);
  
  /* At this point we can free the report string.
   */
  g_free (report);
  
  /* Check the writing operation.
   */    
  if (result == EOF || result < 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to write to file"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      
      g_critical (_("%s@%d: failed to write report to file: '%s'\n"),
	     __FILE__, __LINE__, filename);
      
      return -1;
    }
  
  return 1;
}


gint
polyxmass_window_mngmt_wnd_report_file_append_button (GtkWidget *button,
					       gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *entry = NULL;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  FILE *filep = NULL;
    
  gchar *report = NULL;
  gchar *filename = NULL;
  
  gboolean selection_result = FALSE;

  gint result = -1;
  gint depth = -1;



  g_assert (window != NULL);

  /* Make sure we do not ask that a report be made if no window
     item is currently selected (and available in the treeview).
  */
  treeview= g_object_get_data (G_OBJECT (window), 
			       "winmngmt_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_assert (tree_selection != NULL);
    
  selection_result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);
  if (FALSE == selection_result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }
  
  /* OK, at this point we know a selection is there.
   */
  /* We only deal with this function if the item that is selected is
     an item of depth > 1. That is we want to deal with an item that
     is not a root item.
  */
  model = g_object_get_data (G_OBJECT (window),
			     "winmngmt_treeview_model");
  g_assert (model != NULL);
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  depth = gtk_tree_path_get_depth (path);

  if (depth != 2)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Select a window item in treeview"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      return -1;
    }

  /* OK, we can finally ask that a report be made for the currently
     selected window item. This is the string we'll write to file.
  */
  report = polyxmass_window_mngmt_wnd_report_prepare (window);
  g_assert (report != NULL);


  entry = g_object_get_data (G_OBJECT (window), "current_file_entry");
  g_assert (entry != NULL);
  
  /* filename is NOT allocated.
   */
  filename = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  g_assert (filename != NULL);
  
  if (strlen (filename) <= 0)
    {
      /* Echo a failure message to the user.
       */
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Filename is not set"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      g_free (report);
      
      return -1;
    }
  
  /* Now that we have a filename, we should try to open it in overwrite mode.
   */
  filep = fopen (filename, "a");
  
  if (filep == NULL)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to open file"),
					POLYXMASS_NORM_MSG_TIMEOUT);

      g_critical (_("%s@%d: failed to open file: '%s'\n"),
		  __FILE__, __LINE__, filename);
      
      g_free (report);
      
      return -1;
    }

  /* We can now write to the file.
   */
  result = fputs (report, filep);

  fclose (filep);
  
  /* At this point we can free the report string.
   */
  g_free (report);
  
  /* Check the writing operation.
   */    
  if (result == EOF || result < 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to write to file"),
					POLYXMASS_NORM_MSG_TIMEOUT);
      
      g_critical (_("%s@%d: failed to write report to file: '%s'\n"),
	     __FILE__, __LINE__, filename);
      
      return -1;
    }

  return 1;
}




gint
polyxmass_window_mngmt_wnd_delete_event  (GtkWidget *window, 
				      GdkEventAny *event, 
					  gpointer data)
{
  GtkWidget *parent_wnd = data;
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);

  g_object_set_data (G_OBJECT (parent_wnd), "window_mngmt_wnd", NULL);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}



gint
polyxmass_window_mngmt_wnd_destroy_event  (GtkWidget *window, 
					   GdkEventAny *event, 
					   gpointer data)
{
  GtkWidget *parent_wnd = data;
  
  g_assert (parent_wnd != NULL);
  g_assert (window != NULL);
  
  return FALSE;
}


