/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-seqed-kbd.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-seqed-clipboard.h"
#include "polyxedit-ui-completions.h"
#include "polyxedit-ui-masses-display-wnd.h"



gboolean
polyxedit_seqed_kbd_key_press_event (GtkWidget *widget,
				     GdkEventKey *event,
				     gpointer data)
{
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  GtkMenu *canvas_menu = NULL;

  gchar *key_name = NULL;
  
  PxmEditCtxt *editctxt = data;
  
  /* Often, key combinations like Shift + arrow down are use to
   * perform actions like select a sequence stretch. When a key is
   * pressed it is thus necessary to make sure if it is either a
   * shift, ctrl or alt key or a simple ascii character key (an
   * alphabetic character). If a control character is being type and
   * caught here, we have to set to true the corresponding globabl
   * variable ('kbd_alt_down' for example). Each time a character key is
   * caught here, we check if a control key was down at that precise
   * time, so that it becomes clear what the user wanted to do.
   */

  switch (gdk_event_key->keyval)
    {
    case GDK_Shift_L:
    case GDK_Shift_R:
      kbd_shift_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Control_L:
    case GDK_Control_R:
      kbd_control_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Alt_L:
    case GDK_Alt_R:
      kbd_alt_down = TRUE;
      return TRUE;     
      break;

    case GDK_Meta_L:
    case GDK_Meta_R:
      kbd_meta_down = TRUE;
      return TRUE;     
      break;
	  
    case GDK_Left:
      return polyxedit_seqed_kbd_left_handler (editctxt);
      break;
	  
    case GDK_Right:
      return polyxedit_seqed_kbd_right_handler (editctxt);
      break;
	  
    case GDK_Up:
      return polyxedit_seqed_kbd_up_handler (editctxt);
      break;
	  
    case GDK_Down:
      return polyxedit_seqed_kbd_down_handler (editctxt);
      break;
	  
    case GDK_Home:
      return polyxedit_seqed_kbd_home_handler (editctxt);
      break;
	  
    case GDK_End:
      return polyxedit_seqed_kbd_end_handler (editctxt);
      break;
	  
    case GDK_Page_Up:
      return polyxedit_seqed_kbd_page_up_handler (editctxt);
      break;
	  
    case GDK_Page_Down:
      return polyxedit_seqed_kbd_page_down_handler (editctxt);
      break;
    }
      
  if (GDK_Escape == gdk_event_key->keyval)
    {
      if (FALSE == 
	  polyxedit_seqed_kbd_del_last_char_elab_code (editctxt))
	{
	  g_error (_("%s@%d: failed to delete last char from elab code\n"),
		 __FILE__, __LINE__);
	      
	  return FALSE;
	}
	  
      return TRUE;
    }
      
  if (GDK_Tab == gdk_event_key->keyval)
    {
      /* TAB makes the completion. The function that is called
       * below will check if the current elab_code is unambiguous
       * and if so will evaluate it. If not, a list of all
       * available completions will be displayed for information
       * to the user.
       */
      if (FALSE == 
	  polyxedit_seqed_kbd_check_complete_elab_code (editctxt))
	{
	  g_error (_("%s@%d: failed to check complete elab code\n"),
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}
	  
      return TRUE;
    }
	    
  if (GDK_Return == gdk_event_key->keyval)
    {
      return polyxedit_seqed_kbd_enter_handler (editctxt);
    }
      
  if (GDK_BackSpace == gdk_event_key->keyval)
    {
      return polyxedit_seqed_kbd_backspace_handler (editctxt);
    }
      
  if (GDK_Delete == gdk_event_key->keyval)
    {
      return polyxedit_seqed_kbd_delete_handler (editctxt);
    }

  if (GDK_F10 == gdk_event_key->keyval)
    {
      /* That key should elicit the appearance of the contextual menu
	 on the sequence editor canvas. The menu will be popped up at
	 the precise location of the mouse cursor (wherever in the
	 polseq editor window, and not only onto the canvas sequence
	 display area.
      */
      canvas_menu = 
	(GtkMenu *) g_object_get_data (G_OBJECT (editctxt->
						 seqeditorctxt->sequence_canvas), 
				       "polyxedit_canvas_menu");
      g_assert (canvas_menu != NULL);
      
      /* The 1 below, as sixth param is to tell that the first button
	 of the mouse is going to be used to trigger the "activate"
	 signal of the menu item to be selected.
      */
      gtk_menu_popup (canvas_menu, NULL, NULL, NULL, NULL, 
		      1, 
		      gdk_event_key->time);
    }
  

  /* Now, check if the key that was pressed is for an alpha character.
   * If not, return, otherwise we ought to start analyzing it.
   */

  /* First make a duplicate of the key name (that could be either 'a'
     or 'A' or 'Enter') for our use:
  */
  key_name = g_strdup (gdk_keyval_name (gdk_event_key->keyval));
  
  /* Now we can start doing work on it:
     ----------------------------------
     
     First, check that the string is only one character long (that is
     we are keying in a single letter !)
     
     Second, check that the character is indeed a alphabetic
     non-numerical character
     
   */
  
  if (strlen (key_name) > 1)
    {
      g_free (key_name);
      
      return TRUE;
    }
  
  
  if (0 == isalpha (key_name [0]))
    {
      g_free (key_name);
      
      return TRUE;
    }
  
      
  /* At this point we know that we can deal with the character, now.
   */
  if (FALSE == polyxedit_seqed_kbd_alpha_handler (key_name,
						  editctxt))
    {
      g_error (_("%s@%d: failed to handle alpha character\n"),
	     __FILE__, __LINE__);
	  
      return FALSE;
    }

  /* Now we can free the allocated copy of the key' name, because
     we do not need it anymore.
  */

  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
     
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_key_release_event (GtkWidget *widget,
				       GdkEventKey *event,
				       gpointer data)
{
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  
  switch (gdk_event_key->keyval)
    {
    case GDK_Shift_L:
    case GDK_Shift_R:
      kbd_shift_down = FALSE;

      /* If the shift was because the user wanted to select a sequence
       * element, now, the selection would be stopped.
       */
      kbd_selecting = FALSE;
      break;
	  
    case GDK_Control_L:
    case GDK_Control_R:
      kbd_control_down = FALSE;
      break;
	  
    case GDK_Alt_L:
    case GDK_Alt_R:
      kbd_alt_down = FALSE;
      break;

    case GDK_Meta_L:
    case GDK_Meta_R:
      kbd_meta_down = FALSE;
      break;
    }
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_alpha_handler (gchar *key_name,
				   PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  PxmMonomer *monomer = NULL;
  
  GPtrArray *refGPA = NULL;

  gchar *err = NULL;

  gint complete = 0;
  

  /* The array to receive all the mononers that have a code
   * completing a given partial monomer code.
   */
  GPtrArray *fillGPA = NULL;
  


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  refGPA = polchemdefctxt->polchemdef->monomerGPA;
  
  fillGPA = g_ptr_array_new ();

  /* First of all we must ensure that no Ctrl Key is being pressed, otherwise
     we are not actually editing the polymer sequence by adding into it
     monomer codes, but we are editing it using 3 possibilities:

     1. Ctrl + A (select all)

     2. Ctrl + C (copy selection in the clipboard)

     3. Ctrl + X (remove selection and put in the clipboard)

     4. Ctrl + V (paste selection from the clipboard to the insertion
        point)
  */

  /* 
     debug_printf (("key_name is %s\n", key_name));
  */

  if ('C' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_copy (editctxt);
  if ('c' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_copy (editctxt);

  if ('X' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_cut (editctxt);
  if ('x' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_cut (editctxt);

  if ('V' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_paste (editctxt);
  if ('v' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_clipboard_paste (editctxt);
  
  if ('A' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_wnd_select_sequence (editctxt, -1, -1);
  if ('a' == key_name [0] && kbd_control_down == TRUE)
    return polyxedit_seqed_wnd_select_sequence (editctxt, -1, -1);


  /* Ok, some keys can be used with a Ctrl key down, but these are
     limited to the ones above. If Ctrl is down, we now return because
     we do not want that monomer codes be considered if Ctrl is
     down. Same for the Alt key.
  */
  if (kbd_control_down == TRUE || kbd_alt_down == TRUE)
    return TRUE;
    
  /* We are allowed more than one character per monomer code. So the
   * rule is that the first character of a monomer code must be
   * UPPERcase, while the remaining characters must be lowerCASE. The
   * total number of characters is governed by the
   * polchemdefctxt->polchemdef->codelen variable.
   */  

  /* In seqeditorctxt, we have three variables that are there to store
   * monomer code elements during the parsing of newly types monomer
   * codes:
   * 
   * elab_code - where the code that is being elaborated is stored.
   *
   * eval_code - where a code that is apparently completed is stored
   * for its further evaluation as a valid monomer code.
   *
   * kb_typed_chars - where the number of characters being typed to
   * make a monomer code is stored.
   */
  if (0 != isupper (key_name [0]))
    {
      /* The character is uppercase, by necessity this character
       * starts a new monomer code. That means that, if there were an
       * elab_code, we first have to evaluate it. After the
       * evaluation, we reset all variables and feed the elab_code
       * variable this uppercase character so as to start fresh with
       * the parsing of a new monomer code.
       *
       * We can know if a code was elaborating because if so the
       * kb_typed_chars variable should be different than 0.
       */
      if (seqeditorctxt->kb_typed_chars > 0)
	{
	  /* Some characters were typed already. The new uppercase
	   * character that we have read is starting a new monomer
	   * code, which also means that it is ending the elab_code.
	   */
	  strcpy (seqeditorctxt->eval_code, seqeditorctxt->elab_code);
	  
	  if (FALSE == 
	      polyxedit_seqed_kbd_evaluate_code (seqeditorctxt->eval_code,
						 seqeditorctxt->last_point_1_idx,
						 editctxt))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     seqeditorctxt->eval_code);

	      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  err);
	  
	      g_free (err);

	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (seqeditorctxt->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	     
	      /* The elab_code is found erroneous, so we give the user
	       * a chance to modify it so that it complies with the
	       * codes of the polymer chemistry definition with which
	       * we are working. To do that we remove the last
	       * character from the elaborating code, so that the user
	       * may change it into another character that completes a
	       * valid monomer code. When this is done, we just ask
	       * that the shortened elaborating monomer code be echoed
	       * to the proper GtkEntry widget.
	       */
	      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	      polyxedit_seqed_kbd_echo_elab_code (editctxt);
	      
	      /* Because the code that was elaborated is invalid, it
	       * makes no sense that we take into account the
	       * uppercase character that we were parsing in the first
	       * place, so we just return.
	       */
	      g_ptr_array_free (fillGPA, TRUE);

	      return TRUE;
	    }
	  
	  /* If we are here, that means that the evaluation of the
	   * eval_code has been successful, so we have to do the
	   * following:
	   * 
	   * 1. emtpy the eval_code, so that we later can use it
                fresh.
	   *
	   *
	   * 2. set to 0 the count of typed characters, since we have
                finished dealing witht he previously elaborating code.
	  */
	  memset (seqeditorctxt->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);
	  
	  seqeditorctxt->kb_typed_chars = 0;
	}

      /* At this point we have finished dealing with the case that the
       * uppercase character is effectively ending the elaboration of
       * a monomer code. Because the uppercase character is starting a
       * new elaborated code, we just copy this character in the
       * elab_code at position 0 [which is the pointer of elab_code +
       * seqeditorctxt->kb_typed_chars (which is 0)]:
       */
      memset (seqeditorctxt->elab_code + seqeditorctxt->kb_typed_chars,
	      key_name [0], 1);

      /* Increment the count of typed characters.
       */
      seqeditorctxt->kb_typed_chars++;
    
      
      /* And now effectively deal with the uppercase character that
       * was parsed in the first place: check if this uppercase
       * character already characterizes with no ambiguity a monomer
       * (ie corresponds already to a unique monomer code).
       */
      complete = 
	pxmchem_monomer_get_ptrs_by_completing_code (seqeditorctxt->elab_code,
						     refGPA, fillGPA);
      
      if (complete == 0)
	{
	  /* No one single monomer in refGPA does have a monomer code
	   * starting with the currently parsed uppercase letter. This
	   * is an error condition.
	   */
	  err = g_strdup_printf (_("code already invalid: '%s'"),
				 seqeditorctxt->elab_code);
	  
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      err);
	  
	  g_free (err);

	  polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      if (complete == 1)
	{
	  /* There is no degeneration here, the uppercase character
	   * that was entered corresponds without ambiguity to a
	   * single monomer so that we can go through all the
	   * evaluation process right now.
	   */
	  monomer = g_ptr_array_index (fillGPA, 0);
	  
	  if (FALSE == polyxedit_seqed_kbd_evaluate_code (monomer->code,
							  seqeditorctxt->
							  last_point_1_idx,
							  editctxt))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     monomer->code);
	      
	      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	      polyxedit_seqed_kbd_echo_elab_code (editctxt);
	      
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }

	  /* If we are here that means that we successfully evaluated
	   * the monomer->code, which means that we can reinitialize
	   * some variables for the next character to be entered to be
	   * handled freshly.
	   */
	  memset (seqeditorctxt->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (seqeditorctxt->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  seqeditorctxt->kb_typed_chars = 0;
	  
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* Do not forget to tell the polymer sequence that it is no
	   * more unmodified!
	   */
	  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  "");

	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && seqeditorctxt->kb_typed_chars != polchemdefctxt->polchemdef->codelen)
	{
	  /* The monomer code that is represented by
	   * seqeditorctxt->elab_code corresponds to more than one valid
	   * monomer code. There is ambiguity, and there is still
	   * place for another discriminating character, so we just do
	   * nothing special here, awaiting for another character that
	   * may perform the discrimitation between all the possible
	   * monomer codes.
	   */
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      "");
	  
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && seqeditorctxt->kb_typed_chars == polchemdefctxt->polchemdef->codelen)
	{
	  /* The code that is being elaborated has now the maximum
	   * number of authorized character. That means that we have
	   * to consider it for validity without waiting for any
	   * character more. Let's evaluate it.
	   */
	  strcpy (seqeditorctxt->eval_code, seqeditorctxt->elab_code);

	  if (FALSE == 
	      polyxedit_seqed_kbd_evaluate_code (seqeditorctxt->eval_code,
						 seqeditorctxt->last_point_1_idx,
						 editctxt))
	    {
	      err = g_strdup_printf (_("failed to eval code: '%s'"),
				     seqeditorctxt->eval_code);
	      
	      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (seqeditorctxt->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	      
	      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	      polyxedit_seqed_kbd_echo_elab_code (editctxt);
	      
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }
	  
	  /* At this point we have successfully evaluated the eval_code,
	   * reinitialize the usual variables:
	   */
	  memset (seqeditorctxt->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (seqeditorctxt->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  seqeditorctxt->kb_typed_chars = 0;
	  
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* Do not forget to tell the polymer sequence that it is no
	   * more unmodified!
	   */
	  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      "");
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
    }
  /* end of:
     if (0 != isupper (key_name [0]))
  */
  
  else /* character is lowercase */
    {
      /* We are pursuing the elaboration of a monomer code.
       */
      if  (seqeditorctxt->kb_typed_chars == 0)
	{
	  err = g_strdup_printf (_("cannot start a code with lowercase: '%s'"),
				 key_name);
	  
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      err);
	  
	  g_free (err);
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      /* Apparently we are pursuing effectively the elaboration of a
       * monomer code, add the character to the elab_code.
       */
      memset (seqeditorctxt->elab_code + seqeditorctxt->kb_typed_chars,
	      key_name [0], 1);

      /* Increment the count of typed characters.
       */
      seqeditorctxt->kb_typed_chars++;
      
      /* By the addition of this new character to the elab_code, we
       * may have gotten an unambiguous monomer code. Let us check
       * this immediately.
       */
      complete = 
	pxmchem_monomer_get_ptrs_by_completing_code (seqeditorctxt->elab_code,
						     refGPA, fillGPA);
      
      if (complete == 0)
	{
	  /* No one single monomer in refGPA does have a monomer code
	   * resembling seqeditorctxt->elab_code. This is an error
	   * condition.
	   */
	  err = g_strdup_printf (_("code already invalid: '%s'"),
				 seqeditorctxt->elab_code);
	  
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      err);
	  
	  g_free (err);

	  polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      if (complete == 1)
	{
	  /* There is no degeneration here, the uppercase character
	   * that was entered corresponds without ambiguity to a
	   * single monomer so that we can go through all the
	   * evaluation process right now.
	   */
	  monomer = g_ptr_array_index (fillGPA, 0);
	  
	  if (FALSE == polyxedit_seqed_kbd_evaluate_code (monomer->code,
							  seqeditorctxt->
							  last_point_1_idx,
							  editctxt))
	    {
	      err = g_strdup_printf (_("failed eval code: '%s'"),
				     monomer->code);
	      
	      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	      polyxedit_seqed_kbd_echo_elab_code (editctxt);
	      
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }

	  /* If we are here that means that we successfully evaluated
	   * the monomer->code, which means that we can reinitialize
	   * some variables for the next character to be entered to be
	   * handled freshly.
	   */
	  memset (seqeditorctxt->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (seqeditorctxt->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  seqeditorctxt->kb_typed_chars = 0;
	  
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* Do not forget to tell the polymer sequence that it is no
	   * more unmodified!
	   */
	  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      "");
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
      
      if (complete > 1 
	  && seqeditorctxt->kb_typed_chars != polchemdefctxt->polchemdef->codelen)
	{
	  /* The monomer code that is represented by
	   * seqeditorctxt->elab_code corresponds to more than one valid
	   * monomer code. There is ambiguity, and there is still
	   * place for another discriminating character, so we just do
	   * nothing special here, awaiting for another character that
	   * may perform the discrimitation between all the possible
	   * monomer codes.
	   */
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      "");
	  
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return TRUE;
	}
      
      if (complete > 1 
	  && seqeditorctxt->kb_typed_chars == polchemdefctxt->polchemdef->codelen)
	{
	  /* The code that is being elaborated has now the maximum
	   * number of authorized character. That means that we have
	   * to consider it for validity without waiting for any
	   * character more. Let's evaluate it.
	   */
	  strcpy (seqeditorctxt->eval_code, seqeditorctxt->elab_code);

	  if (FALSE == 
	      polyxedit_seqed_kbd_evaluate_code (seqeditorctxt->eval_code,
						 seqeditorctxt->last_point_1_idx,
						 editctxt))
	    {
	      err = g_strdup_printf (_("failed eval code: '%s'"),
				     seqeditorctxt->eval_code);
	      
	      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
				  err);
	      
	      g_free (err);
	      
	      /* Since we know that eval_code is invalid, we can remove it.
	       */
	      memset (seqeditorctxt->eval_code, '\x0', 
		      polchemdefctxt->polchemdef->codelen + 1);
	      
	      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
	      polyxedit_seqed_kbd_echo_elab_code (editctxt);
	      
	      g_ptr_array_free (fillGPA, TRUE);
	      
	      return TRUE;
	    }
	  
	  /* At this point we have successfully evaluated the eval_code,
	   * reinitialize the usual variables:
	   */
	  memset (seqeditorctxt->eval_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  memset (seqeditorctxt->elab_code, '\x0', 
		  polchemdefctxt->polchemdef->codelen + 1);

	  seqeditorctxt->kb_typed_chars = 0;
	  
	  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
	  /* Do not forget to tell the polymer sequence that it is no
	   * more unmodified!
	   */
	  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
	  
	  /* If there was an error message, just erase it.
	   */
	  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			      "");
	  
	  g_ptr_array_free (fillGPA, TRUE);

	  return TRUE;
	}
    }
  /* end of:
     else character is lowercase
  */
  
  return TRUE; 
}


gboolean
polyxedit_seqed_kbd_backspace_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint idx = -1;
  

  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* There are different situations here:
   * 
   * If there is an active selection in the sequence editor, we should
   * destroy that selected oligomer first, and return after that.
   * 
   * If there is no active selection, then we have to remove the
   * monomer that is located prior to the current cursor position.
   */
  
  /*
    debug_printf (("last_point_1_idx = %d, sel_mnm_idx1 = %d, "
    "sel_mnm_idx2 = %d\n",
    seqeditorctxt->last_point_1_idx,
    seqeditorctxt->sel_mnm_idx1, seqeditorctxt->sel_mnm_idx2));
  */
  
  if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							 NULL,
							 NULL))
    {
      /* There is no selection. We are thus ready to perform the removal
       * of a monomer in the sequence editor, if the cursor is not located
       * left of the beginning of the sequence.
       */
  
      /* Imagine we have the following sequence:
       * 
       * A R N ^ D D E Q, with the cursor being locate where ^ is.
       *
       * In this situation, the variable seqeditorctxt->last_point_1_idx ==
       * 3, which is actually the index of the monomer located at the
       * right of the 'N'. If we issue a backspace now, what we want
       * to remove is not the monomer of index 3, but the monomer at
       * the left of the cursor position, which is 'N', thus monomer
       * 3-1 = 2. So we should issue a removal order with an index of
       * value (seqeditorctxt->last_point_1_idx - 1).
       * 
       * But what if the cursor is at the following position:
       * 
       * ^ A R N D D E Q 
       *
       * It would not make sense to send a removal order at all:
       */
      if (seqeditorctxt->last_point_1_idx == 0)
	return TRUE;
      
      idx = seqeditorctxt->last_point_1_idx - 1;
      
      polyxedit_seqed_wnd_remove_monomer (editctxt, idx);
      
      /* Each time a monicon is removed (towards the left, lesser
       * indexes) from the GPtrArray, the last_point_1_idx is
       * decremented by one unit BUT only if the removed monomer is
       * not removed from the first (index 0) GPtrArray index !!!
       */
      if (seqeditorctxt->last_point_1_idx > 0)
	seqeditorctxt->last_point_1_idx--;
      
      polyxedit_seqed_wnd_redraw_sequence (editctxt);
      
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
      
    }
  /* end of:
     if (seqeditorctxt->sel_mnm_idx1 == -1 || seqeditorctxt->sel_mnm_idx2 == -1)
  */
  
  else
    {
      /* An oligomer is currently selected.
       */
      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	{
	  g_critical (_("%s@%d: failed to remove the selected oligomer\n"),
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}
        
      polyxedit_seqed_wnd_redraw_sequence (editctxt);
      
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
    }

  /* The sequence and the selection.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_delete_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint idx = -1;
  gint length = 0;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* There are different situations here:
   * 
   * If there is an active selection in the sequence editor, we should
   * destroy that selected oligomer first, and return after that.
   * 
   * If there is no active selection, then we have to remove the
   * monomer that is located after the current cursor position.
   */
  
  if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  NULL,
							 NULL))
    {
      /* There is no selection. We are thus ready to perform the removal
       * of a monomer in the sequence editor, if the cursor is not located
       * right of the last monomer in the sequence.
       */
  
      /* Imagine we have the following sequence:
       * 
       * A R N ^ D D E Q, with the cursor being locate where ^ is.
       *
       * In this situation, the variable seqeditorctxt->last_point_1_idx ==
       * 3, which is actually the index of the monomer located at the
       * right of the 'N'. If we issue a delete now, what we want to
       * remove is precisely the monomer of index 3. So we should
       * issue a removal order with an index of value
       * seqeditorctxt->last_point_1_idx.
       * 
       * But what if the cursor is at the following position:
       * 
       * A R N D D E Q ^
       *
       * It would not make sense to send a removal order, because we
       * would be asking to remove a monomer at index 7 which does not
       * exist !
       */
      if (seqeditorctxt->last_point_1_idx >= length)
	return TRUE;
      
      idx = seqeditorctxt->last_point_1_idx;
      
      polyxedit_seqed_wnd_remove_monomer (editctxt, idx);
      
      /* In the present case we do not need to modify the
       * seqeditorctxt->last_point_1_idx because we removed a monomer right
       * of the cursor.
       */

      polyxedit_seqed_wnd_redraw_sequence (editctxt);
      
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
    }
  /* end of:
     if (seqeditorctxt->sel_mnm_idx1 == -1 || seqeditorctxt->sel_mnm_idx2 == -1)
  */
  
  else
    {
      /* An oligomer is currently selected.
       */
      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	{
	  g_error (_("%s@%d: failed to remove the selected oligomer\n"),
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}

      polyxedit_seqed_wnd_redraw_sequence (editctxt);
      
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
    }
  
  /* The sequence and the selection.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_right_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0;
  
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx >= length)
    return TRUE;

  /* Be careful that if the user is pressing the right arrow key in
   * combination with a shift down, that means that she wants to
   * select a portion of polymer sequence.
   */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
	
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;
	  
	  seqeditorctxt->last_point_1_idx++;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;

	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx++;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1 
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* And now increment the last pointed monomer index.
       */
      seqeditorctxt->last_point_1_idx++;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, 
					       COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
     &seqeditorctxt->x_offset, 
     &seqeditorctxt->y_offset);
  
  if (rect.y1 > (seqeditorctxt->y_offset 
		 + seqeditorctxt->canvas_scroll_wnd_height 
		 - seqeditorctxt->monicon_size))
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset +
			      seqeditorctxt->monicon_size);
    }

  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);

  /* In case there was a selection going on, let's inform that this is 
     going on, so that the selected sequence is available in the
     PRIMARY selection.
  */
  polyxedit_seqed_clipboard_primary_copy (editctxt);  
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_left_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0;
  
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx <=0)
    return TRUE;

  /* Be careful that if the user is pressing the right arrow key in
   * combination with a shift down, that means that she wants to
   * select a portion of polymer sequence.
   */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;
	  

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx--;

	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx--;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
      */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1 
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* And now just decrement the last pointed monomer index.
       */
      seqeditorctxt->last_point_1_idx--;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, 
					       COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
				  &seqeditorctxt->x_offset, 
				  &seqeditorctxt->y_offset);
  
  if (rect.y1 < (seqeditorctxt->y_offset + seqeditorctxt->monicon_size))
    {
      /* We are near the top of viewable sequence, so we ask
       * for a scroll to happen towards lower monomer indexes
       * (scroll upwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset -
			      seqeditorctxt->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* In case there was a selection going on, let's inform that this is 
     going on, so that the selected sequence is available in the
     PRIMARY selection.
  */
  polyxedit_seqed_clipboard_primary_copy (editctxt);

  return TRUE;
}


gboolean
polyxedit_seqed_kbd_home_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0;
  gint line = 0;
  gint target_idx = 0;
    
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx <=0)
    return TRUE;


  /* There are two cases:

  1. if the control key is not pressed, the target position is the
  beginning of the current line. 

  2. if the control key is pressed, the target position is the 
  beginning of the whole sequence.
  */

  if (kbd_control_down == TRUE)
    target_idx = 0;
  else
    {
      /* Calculate the line number (position not index) on which the cursor
	 is currently located.
      */
      line = 
	((seqeditorctxt->last_point_1_idx -1) / 
	 seqeditorctxt->monicons_per_line) + 1;
      
      target_idx = seqeditorctxt->monicons_per_line * (line - 1);
    }
  

  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the beginning of the
     current line.
   */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  	    	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* We are changing the last point index.
       */
      seqeditorctxt->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);

  /*
    debug_printf (("cursor y : %f\n", seqeditorctxt->cursor->y));
  */

  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_ensure_select_cursor_visible (seqeditorctxt->sequence_editor_wnd,
						    NULL,
						    editctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_end_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0;
  gint line = 0;
  gint target_idx = 0;
    
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx >= length)
    return TRUE;


  /* There are two cases:

  1. if the control key is not pressed, the target position is the
  end of the current line. 

  2. if the control key is pressed, the target position is the 
  end of the whole sequence.
  */

  if (kbd_control_down == TRUE)
    target_idx = length;
  else
    {
      /* Calculate the line number (position not index) on which the cursor
	 is currently located.
      */
      line = 
	((seqeditorctxt->last_point_1_idx -1) 
	 / seqeditorctxt->monicons_per_line) + 1;
      
      target_idx = seqeditorctxt->monicons_per_line * line;
    }
  

  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the end of the current
     line.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1 
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* We are changing the last point index.
       */
      seqeditorctxt->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);

  /*
    debug_printf (("cursor y : %f\n", seqeditorctxt->cursor->y));
  */

  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_ensure_select_cursor_visible (seqeditorctxt->sequence_editor_wnd,
						    NULL,
						    editctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_up_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint target_idx = -1;
  gint length = 0;
    
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* If cursor is already at the beginning of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx <=0)
    return TRUE;

  /* We want to draw the cursor one line above the one where it sits
     right now, ideally in the exact same column. That means going
     back in the sequence by monicons_per_line positions.
   */

  /* Calculate the number of monicons per line:
   */
  seqeditorctxt->monicons_per_line = 
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin)
    / seqeditorctxt->monicon_size;
  
  target_idx = seqeditorctxt->last_point_1_idx 
    - seqeditorctxt->monicons_per_line;

  if (target_idx < 0)
    return TRUE;
  
  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the line above, ideally
     in the exact same column. That means going back in the sequence
     by monicons_per_line positions.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1 
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* We are changing the last point index.
       */
      seqeditorctxt->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, 
					       COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
     &seqeditorctxt->x_offset, 
     &seqeditorctxt->y_offset);
  
  if (rect.y1 < (seqeditorctxt->y_offset + seqeditorctxt->monicon_size))
    {
      /* We are near the top of viewable sequence, so we ask
       * for a scroll to happen towards lower monomer indexes
       * (scroll upwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset -
			      seqeditorctxt->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
  /*
    debug_printf (("cursor y : %f\n", seqeditorctxt->cursor->y));
  */

  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_down_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint target_idx = -1;
  gint length = 0;
  
  PxmRect rect;
  

  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* If cursor is already at the end of the polymer sequence, no
   * matter doing anything here.
   */
  if (seqeditorctxt->last_point_1_idx >= length)
    return TRUE;

  /* We want to draw the cursor one line below the one where it sits
   * right now, ideally in the exact same column. That means going
     forward in the sequence by monicons_per_line positions.
   */

  /* Calculate the number of monicons per line:
   */
  seqeditorctxt->monicons_per_line = 
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin)
    / seqeditorctxt->monicon_size;

  target_idx = seqeditorctxt->last_point_1_idx 
    + seqeditorctxt->monicons_per_line;

  if (target_idx > length)
    target_idx = length;
  
  /* If the shift key is down, that means that the user wants to
     select a sequence portion from "here" to the line below, ideally
     in the exact same column. That means going forward in the sequence
     by monicons_per_line positions.
  */
  if (kbd_shift_down == TRUE)
    {
      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  else
    {
      /* Normal processing. If a selection is active, it is destroyed.
       */
      if (seqeditorctxt->sel_mnm_idx1 != -1 
	  && seqeditorctxt->sel_mnm_idx2 != -1)
	polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);
      
      /* We are changing the last point index.
       */
      seqeditorctxt->last_point_1_idx = target_idx;

      /* Finally, set the selection rectangle to nothing exactly as if
	 we were left-clicking onto this monomer.
      */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   last_point_1_idx,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  
    }
  
  /* Now continue with normal processing.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, 
					       COORDSYS_NW);
  
  /* If the pointer approaches the top or the bottom of the
   * canvas, then it may be necessary that the canvas be scrolled.
   */
  gnome_canvas_get_scroll_offsets 
    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
     &seqeditorctxt->x_offset, 
     &seqeditorctxt->y_offset);
  
  if (rect.y1 > (seqeditorctxt->y_offset 
		 + seqeditorctxt->canvas_scroll_wnd_height 
		 - seqeditorctxt->monicon_size))
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset +
			      seqeditorctxt->monicon_size);
    }
  
  /* We can now draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
  /*
    debug_printf (("cursor y : %f\n", seqeditorctxt->cursor->y));
  */

  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_page_up_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0; 
  gint y_new_scroll = 0;
  gint count_vis_lines = 0;
  gint target_idx = -1;
    
  PxmRect rect;


  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* There are two cases:
     
  1. No shift key is down, in which case we only make a scroll of the
  page. That scroll won't change the insertion point.

  2. kbd_shift_down is true, in which case we select the sequence portion
  that goes from "here" to the same column, a page up.
  */

  /* If we want to scroll one page up, that means that we want to
     display some lines that currently are hidden on top of the
     window. But how many lines? The exact number of lines that
     are currently displayed, which is calculatable:
  */
  count_vis_lines = seqeditorctxt->canvas_scroll_wnd_height 
    / seqeditorctxt->monicon_size;
  
  /* Calculate the number of monicons per line:
   */
  seqeditorctxt->monicons_per_line = 
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin)
    / seqeditorctxt->monicon_size;


  if (kbd_shift_down == FALSE)
    {
      /* Get the scroll offsets. The y scroll offset is the number of
	 pixels from start of the sequence picture to the first
	 visible area. For example, if four rows of monomers are
	 hidden above the currently visible sequence, that each
	 monicon has a size of 100 pixels, we'll get a y scroll offset
	 of 400.
       */
      gnome_canvas_get_scroll_offsets 
	(GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	 &seqeditorctxt->x_offset, 
	 &seqeditorctxt->y_offset);
      
      /* Since we are scrolling upwards, that means that we want to
	 diminish the amount of hidden pixels above the visible region
	 of the window. That means that we want to decrement this
	 number of pixels by the amount of (count_vis_lines *
	 seqeditorctxt->monicon_size).
       */
      y_new_scroll = seqeditorctxt->y_offset - 
	(count_vis_lines * seqeditorctxt->monicon_size);
  
      if (y_new_scroll < 0)
	y_new_scroll = 0;
  
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      y_new_scroll);
      
      
      /* The selection does not change and cursor point neither, so do
	 not ask that masses be recalculated.
      */
      return TRUE;
    }
  else 
    /* (kbd_shift_down == TRUE)
     */
    {
      /* The selection should go from current cursor point to the same
	 column, one page up. That is the number of visibles lines
	 upwards.
      */
      target_idx = seqeditorctxt->last_point_1_idx - 
	(count_vis_lines * seqeditorctxt->monicons_per_line) ;
      
      if (target_idx < 0)
	target_idx = 0;


      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);


      /* The way we scroll the window depends on the manner in which
	 the selection is performed: 

	 1. if seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2,
	 then that means that we are selecting from top to bottom,
	 which is that we want to have the bottom horizontal line of
	 the selection drawn at the bottom of the visible area of the
	 canvas (Note the COORDSYS_SW param to the pixel_coord
	 function call !).

	 2. if seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2,
	 then that means that we are selecting from bottom to top,
	 which means that we want to display the upper horizontal line
	 of the selection polygon at the top of the visible area of
	 the canvas (Note the COORDSYS_NW param to the pixel_coord
	 function call !).
      */

      if (seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2)
	{
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_SW, 
						       COORDSYS_SW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     &seqeditorctxt->x_offset, 
	     &seqeditorctxt->y_offset);
	  
	  y_new_scroll = rect.y1 
	    - (count_vis_lines * seqeditorctxt->monicon_size);
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     seqeditorctxt->x_offset,
	     y_new_scroll);
	}
      
      if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
	{
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     &seqeditorctxt->x_offset, 
	     &seqeditorctxt->y_offset);
	  
	  y_new_scroll = rect.y1;

	    if (y_new_scroll < 0)
	      y_new_scroll = 0;
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     seqeditorctxt->x_offset,
	     y_new_scroll);
	}
      
      /* Finally we draw the cursor to the new cursor point location !
       */
      polyxedit_seqed_wnd_draw_cursor (editctxt->seqeditorctxt);
      
      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  

      /* The selection.
       */
      polyxedit_ui_masses_display_wnd_update_selection_masses
	((PxmEditCtxt *) polyxedit_last_editctxt);

      return TRUE;
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_page_down_handler (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  gint length = 0;
  gint count_max_pixels = 0;
  gint y_new_scroll = 0;
  gint count_lines = 0;
  gint count_vis_lines = 0;
  gint target_idx = -1;
      
  PxmRect rect;


  g_assert (editctxt != NULL);
  
  g_assert (editctxt->polymer != NULL);
  length = editctxt->polymer->monomerGPA->len;
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* There are two cases:
     
  1. No shift key is down, in which case we only make a scroll of the
  page. That scroll won't change the insertion point.

  2. kbd_shift_down is true, in which case we select the sequence portion
  that goes from "here" to the same column, a page down.
  */

  /* If we want to scroll one page down, that means that we want to
     display some lines that currently are hidden on bottom of the
     window. But how many lines? The exact number of lines that
     are currently displayed, which is calculatable:
  */
  count_vis_lines = seqeditorctxt->canvas_scroll_wnd_height 
    / seqeditorctxt->monicon_size;

  /* Calculate the number of monicons per line:
   */
  seqeditorctxt->monicons_per_line = 
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin)
    / seqeditorctxt->monicon_size;


  if (kbd_shift_down == FALSE)
    {
      /* Get the scroll offsets. The y scroll offset is the number of
	 pixels from start of the sequence picture to the first
	 visible area. For example, if four rows of monomers are
	 hidden above the currently visible sequence, that each
	 monicon has a size of 100 pixels, we'll get a y scroll offset
	 of 400.
       */
      gnome_canvas_get_scroll_offsets 
	(GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	 &seqeditorctxt->x_offset, 
	 &seqeditorctxt->y_offset);
      
      /* Since we are scrolling downwards, that means that we want to
	 increase the amount of hidden pixels above the visible region
	 of the window. That means that we want to increment this
	 number of pixels by the amount of (count_vis_lines *
	 seqeditorctxt->monicon_size).
       */
      y_new_scroll = seqeditorctxt->y_offset + 
	(count_vis_lines * seqeditorctxt->monicon_size);
  
      /* Test if the y_new_scroll position is not greater than the
	 number of pixels of the entire sequence picture!
      */
      count_lines = (length - 1) / seqeditorctxt->monicons_per_line;
      
      count_max_pixels = count_lines * seqeditorctxt->monicon_size;
      
      if (y_new_scroll > count_max_pixels)
	y_new_scroll = count_max_pixels;
      
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      y_new_scroll);
      
      
      /* The selection does not change and cursor point neither, so do
	 not ask that masses be recalculated.
      */
      return TRUE;
    }
  else 
    /* (kbd_shift_down == TRUE)
     */
    {
      /* The selection should go from current cursor point to the same
	 column, one page down. That is the number of visibles lines
	 downwards.
      */
      target_idx = seqeditorctxt->last_point_1_idx + 
	(count_vis_lines * seqeditorctxt->monicons_per_line) ;
      
      if (target_idx > length)
	target_idx = length;


      /* There are two cases:
	 
      1. there is no current selection. We are initiating a 
      selection operation from scratch. 
      
      2. there is a selection, we are just continuing it.
      */
      
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      NULL, NULL))
	{
	  /* No selection is presently made in the sequence editor: we
	   * are initiating a selection operation. Set current
	   * monicon's coordinates to the first point of the rect
	   * PRIOR TO CHANGING THE CURRENTLY POINTED MONOMER INDEX!!!
	   */
	  kbd_selecting = TRUE;
	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x1 = rect.x1;
	  seqeditorctxt->selection_rect.y1 = rect.y1;

	  /* Now move according to the cursor key that was keyed in.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;
	  
	  /* And set the new monicon's coordinates to the first point
	     of the rect.
	   */	  
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      else
	{
	  /* There is a selection right now, so we just resume it by
	     setting kbd_selecting to TRUE, even if that key was
	     released. If the user wants to stop the selection and
	     start a fresh selection, she'll have to move cursor
	     without pressing shift, which will elicit the removal of
	     the selection polygon. After that, the selection process
	     can be started over by putting the cursor at the beginnig
	     of the desired selection, pressing shift while moving the
	     cursor with the cursor keys...
	  */
	  kbd_selecting = TRUE;
	  
	  /* We are changing the second point of a running selection.
	   */
	  seqeditorctxt->last_point_1_idx = target_idx;

	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);

	  seqeditorctxt->selection_rect.x2 = rect.x2;
	  seqeditorctxt->selection_rect.y2 = rect.y2;
	}
      
      /* Now that we have the correct points, we can draw the
	 selection polygon.
       */
      polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);


      /* The way we scroll the window depends on the manner in which
	 the selection is performed: 

	 1. if seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2,
	 then that means that we are selecting from top to bottom,
	 which is that we want to have the bottom horizontal line of
	 the selection drawn at the bottom of the visible area of the
	 canvas (Note the COORDSYS_SW param to the pixel_coord
	 function call !).

	 2. if seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2,
	 then that means that we are selecting from bottom to top,
	 which means that we want to display the upper horizontal line
	 of the selection polygon at the top of the visible area of
	 the canvas (Note the COORDSYS_NW param to the pixel_coord
	 function call !).
      */

      if (seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2)
	{
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_SW, 
						       COORDSYS_SW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     &seqeditorctxt->x_offset, 
	     &seqeditorctxt->y_offset);
	  
	  y_new_scroll = rect.y1 
	    - (count_vis_lines * seqeditorctxt->monicon_size);
        
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     seqeditorctxt->x_offset,
	     y_new_scroll);
	}
      
      if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
	{
	  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						       last_point_1_idx,
						       seqeditorctxt,
						       &rect,
						       COORDSYS_NW, 
						       COORDSYS_NW);
	  
	  gnome_canvas_get_scroll_offsets 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     &seqeditorctxt->x_offset, 
	     &seqeditorctxt->y_offset);
	  
	  y_new_scroll = rect.y1;

	    if (y_new_scroll < 0)
	      y_new_scroll = 0;
	  
	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     seqeditorctxt->x_offset,
	     y_new_scroll);
	}
      
      /* Finally we draw the cursor to the new cursor point location !
       */
      polyxedit_seqed_wnd_draw_cursor (editctxt->seqeditorctxt);

      /* In case there was a selection going on, let's inform that this is 
	 going on, so that the selected sequence is available in the
	 PRIMARY selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);  

      /* The selection.
       */
      polyxedit_ui_masses_display_wnd_update_selection_masses
	((PxmEditCtxt *) polyxedit_last_editctxt);

      return TRUE;
    }
  /* end of:
     if (kbd_shift_down == TRUE)
  */
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_enter_handler (PxmEditCtxt *editctxt)
{

  g_assert (editctxt != NULL);
  
  polyxedit_seqed_kbd_force_complete_elab_code (editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_del_last_char_elab_code (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* We have to go back one character in the currently elaborating
   * code.  To do this, we need to have at least one character in the
   * elab_code!
   */
  if (seqeditorctxt->kb_typed_chars == 0)
    return TRUE;
  
  /* Calculate the index where we will erase the character by putting
   * over it a 'x\0' character.
   */
  /* Imagine that 1 character was in elab_code (ie
   * . seqeditorctxt->kb_typed_chars = 1). That would mean that
   * (seqeditorctxt->elab_code + 0) contains that character. And this is
   * precisely the slot that we want to anihilate. 
   * 
   * So the logic is that if we have type n characters, then we want to 
   * anihilate the (n-1) slot.
   *
   * Since we also want to decrement seqeditorctxt->kb_typed_chars, so that
   * it still reflects correctly the number of characters that have
   * been typed in the elab_code, we could use the C construct that first
   * decrements the variable, and next uses it.
   */
  memset (seqeditorctxt->elab_code + (--seqeditorctxt->kb_typed_chars), '\x0', 1);

  /* To maintain the systematic consistency between the contents of 
   * seqeditorctxt->elab_code and what is displayed in the elab_code_entry
   * let's call the function below:
   */
  polyxedit_seqed_kbd_echo_elab_code (editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_echo_elab_code (PxmEditCtxt *editctxt)
{
  g_assert (editctxt != NULL);
  g_assert (editctxt->seqeditorctxt != NULL);
  

  gtk_entry_set_text (GTK_ENTRY (editctxt->seqeditorctxt->elab_code_entry),
		      editctxt->seqeditorctxt->elab_code);

  return TRUE;
}


gboolean
polyxedit_seqed_kbd_check_complete_elab_code (PxmEditCtxt *editctxt)
{
  /* The array to receive all the mononers that have a code
   * completing a given partial monomer code.
   */
  GPtrArray *fillGPA = NULL;
  
  gint complete = 0;
  
  PxmMonomer *monomer = NULL;
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  g_assert (editctxt != NULL);
  
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  fillGPA = g_ptr_array_new ();
  


  /* Let us say that we have 3-lett monomer codes in the polymer
   * definition, of which Gly, Gl, G. We have entered G and this
   * function is called. What will happen here is that we'll first
   * check how many codes are available as codes which comply with the
   * first G chracter.  This function calls FIXME: name of called
   * function and gets back Gly, Gl, and G.  Now imagine we have
   * entered Gl and this function is called. We would get Gl and Gly
   * back. If we had previously entered Gly, and this function was
   * called, then we would have gotten only one monomer code as an
   * available code to comply with Gly: Gly (!). So we just would
   * evaluate this code. In the two former cases, we would have
   * displayed a list with the available completions.
   */
  strcpy (seqeditorctxt->eval_code, seqeditorctxt->elab_code);
  
  complete = 
  pxmchem_monomer_get_ptrs_by_completing_code (seqeditorctxt->eval_code, 
					       polchemdef->monomerGPA, 
					       fillGPA);
 if (complete == 0)
    {
      g_error (_("%s@%d: code is already invalid: '%s'\n"),
	     __FILE__, __LINE__, seqeditorctxt->eval_code);
      
      g_ptr_array_free (fillGPA, TRUE);
      
      return FALSE;
    }
  
  if (complete == 1)
    {
      /* Only one monomer can complete with current elab_code, so that
       * it is easy: we just evaluate it now.
       */
      monomer = g_ptr_array_index (fillGPA, 0);
      
      if (FALSE == polyxedit_seqed_kbd_evaluate_code (monomer->code,
						      seqeditorctxt->
						      last_point_1_idx,
						      editctxt))
	{
	  g_error (_("%s@%d: failed to evaluate code: '%s'\n"),
		 __FILE__, __LINE__, monomer->code);
	  
	  g_ptr_array_free (fillGPA, TRUE);
	  
	  return FALSE;
	}
      
      /* Now that we have successfully evaluated monomer->code, we
       * reinitialize some variables for next typed character.
       */
      memset (seqeditorctxt->eval_code, '\x0', 
	      polchemdefctxt->polchemdef->codelen + 1);
      
      memset (seqeditorctxt->elab_code, '\x0', 
	      polchemdefctxt->polchemdef->codelen + 1);
      
      seqeditorctxt->kb_typed_chars = 0;
      
      polyxedit_seqed_kbd_echo_elab_code (editctxt);
    }
  
  if (complete > 1)
    {
      /* There are more than one monomer that have a code that is 
       * compatible with eval_code. So we should display these monomers
       * in a graphical list packed with a button for each monomer so
       * that the user may see them and click on one of them to finally
       * make a "graphical" completion.
       */
      polyxedit_completions_mnm_code_show_list (editctxt, fillGPA);

      g_ptr_array_free (fillGPA, TRUE);
    }
  
  return TRUE;
}


gboolean
polyxedit_seqed_kbd_force_complete_elab_code (PxmEditCtxt *editctxt)
{
  PxmMonomer *monomer = NULL;
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  gchar *err = NULL;

  
  g_assert (editctxt != NULL);
  
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  /* Let us say that we have a 3-letter monomer codes polymer
   * definition, of which "Gly", "Gl" and "G". We have entered "G" and
   * this function is called. What we want here to happen is that if
   * "G" corresponds to a valid monomer code it be evaluated without
   * bothering with other available completions.
   */
  strcpy (seqeditorctxt->eval_code, seqeditorctxt->elab_code);
  
  monomer = pxmchem_monomer_get_ptr_by_code (seqeditorctxt->eval_code,
					     polchemdef->monomerGPA);
  
  if (monomer == NULL)
    {
      /* No one single monomer in refGPA does have a monomer code
       * resembling seqeditorctxt->elab_code. This is an error
       * condition.
       */
      err = g_strdup_printf (_("code already invalid: '%s'"),
			     seqeditorctxt->eval_code);
      
      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			  err);
      
      g_free (err);
      
      polyxedit_seqed_kbd_del_last_char_elab_code (editctxt);
      polyxedit_seqed_kbd_echo_elab_code (editctxt);

      return FALSE;
    }

  /* Apparently we have a monomer, so just evaluate its code.
   */
  if (FALSE == polyxedit_seqed_kbd_evaluate_code (monomer->code,
						  seqeditorctxt->last_point_1_idx,
						  editctxt))
    {
      err = g_strdup_printf (_("failed eval code: '%s'"),
			     monomer->code);
      
      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			  err);
      
      g_free (err);
      
      return FALSE;
    }

  /* If we are here that means that we successfully evaluated
   * the monomer->code, which means that we can reinitialize
   * some variables for the next character to be entered to be
   * handled freshly.
   */
  memset (seqeditorctxt->eval_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);

  memset (seqeditorctxt->elab_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);

  seqeditorctxt->kb_typed_chars = 0;
	  
  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
  /* Do not forget to tell the polymer sequence that it is no
   * more unmodified!
   */
  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
	  
  /* If there was an error message, just erase it.
   */
  gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
		      "");

  return TRUE;
}



gboolean
polyxedit_seqed_kbd_evaluate_code (gchar *code, gint idx,
				   PxmEditCtxt *editctxt)
{
  PxmMonomer *monomer = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmRect rect;
  
  GPtrArray *refGPA = NULL;
  


  g_assert (code != NULL);
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  refGPA = editctxt->polchemdefctxt->polchemdef->monomerGPA;
  g_assert (refGPA != NULL);
  

  if (-1 == pxmchem_monomer_get_index_by_code (code, refGPA))
    {
      g_critical (_("%s@%d: monomer of code: '%s' is not known\n"),
	     __FILE__, __LINE__, code);
      
      return FALSE;
    }
  
  if (TRUE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							 NULL,
							 NULL))
    {
      /*
	debug_printf (("selection to be removed is [%d-%d]\n",
	seqeditorctxt->sel_mnm_idx1, seqeditorctxt->sel_mnm_idx2));
      */

      /* The function below returns the location of the cursor after the
       * operation or -1 if an error occurred.
       */
      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	{
	  g_error (_("%s@%d: failed to remove the selected oligomer\n"),
		 __FILE__, __LINE__);
	}
    }
  
  /* Create a new monomer from its code.
   */
  monomer = pxmchem_monomer_new_by_code (code, refGPA);

  /*
    debug_printf (("new code is %s\n", monomer->code));
  */
    
  /* Integrate the newly created monomer into the polymer sequence's
   * GPtrArray:
   */

  /*
    debug_printf (("last point idx before integration of monomer: %d\n", 
    seqeditorctxt->last_point_1_idx));
  */

  if (FALSE == polyxedit_seqed_wnd_integrate_monomer_at_idx (monomer, 
							     idx,
							     editctxt))
    {
      g_critical (_("%s@%d: failed to integrate new monomer: '%s' in sequence\n"),
	     __FILE__, __LINE__, monomer->code);
      
      pxmchem_monomer_free (monomer);
      
      return FALSE;
    }
  
  seqeditorctxt->last_point_1_idx++;
  
  /*
    debug_printf (("last point idx after integration of monomer: %d\n", 
    seqeditorctxt->last_point_1_idx));
  */

  /* We should make sure that the insertion point is systematically
     visible, even if we are inserting data on the lowest part of the
     sequence editor window.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, COORDSYS_NW);

  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
				   &(seqeditorctxt->x_offset),
				   &(seqeditorctxt->y_offset));
  
  if (rect.y1 
      >= seqeditorctxt->y_offset 
      + seqeditorctxt->monicon_size 
      + seqeditorctxt->canvas_scroll_wnd_height
      - - seqeditorctxt->monicon_size)
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset +
			      seqeditorctxt->monicon_size);
    }
      
  polyxedit_seqed_wnd_redraw_sequence (editctxt);
  
  /* The sequence and the selection.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return TRUE;
}



