// ------------------------------------------------------------------------- //
// $Id: primitive.h,v 1.27 2003/05/20 21:04:02 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// ------------------------------------------------------------------------- //

#ifndef _PRIMITIVE_H_
#define _PRIMITIVE_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_VECTOR
# include <vector>
#endif

#ifndef __GNUC__
# ifdef _WIN32
#  include <windows.h>
# endif
#endif

#if HAVE_GL_GL_H
# include <GL/gl.h>
#endif

#if HAVE_OPENGL_GL_H	// MAC OS X
# include <OpenGL/gl.h>	
#endif

#include "common.h"
#include "refcount.h"
#include "image.h"

// ------------------------------------------------------------------------- //

class Texture;
class Tile;

//! Encapsulates the OpenGL primitives
/*!
 * Each Node may contain one or more Primitives. A Primitive may contain
 * many polygons but all must use the same Texture.
 */
class Primitive
{
public:
	Primitive();
	Primitive(Texture* tex, uint width, uint height);
	Primitive(Ref<Tile> tile);
	virtual ~Primitive();
	void blend_colors(rgba& color);
	virtual void draw(rgba &color);
	void default_prim();
	void box_prim();
	void line_prim(const v3& end_pos);
	void set_prim_type(GLenum pt) { _GL_prim_type = pt; }
	void set_num_vertices(int i)  { _num_vertices = i; }
	void set_vertices(v3* vertices)    { _vertices  = vertices; }
	void set_colors(rgba* colors);
	void set_tex_coords(v2* texcoords) { _tex_coords = texcoords; }
	void set_use_colors(bool b)     { _use_colors     = b; }
	void set_use_tex_coords(bool b) { _use_tex_coords = b; }
	void set_texture(Texture* tex) { _texture = tex; }
	void offset(float x, float y, float z); // This should probably go away 
	static int _num_prims_drawn;
private:
	void     init();
	GLenum   _GL_prim_type;
	int      _num_vertices;
	v3      *_vertices;
	bool     _use_tex_coords;
	v2      *_tex_coords;
	Texture *_texture;
	bool     _use_colors;
	rgba    *_colors;
	GLubyte *_blended_colors;
	Ref<Tile> _tile;
};

// ------------------------------------------------------------------------- //

class Tile : public Rect, public Refcount
{
friend class TiledTexture;
public:
	typedef Ref<Tile> handle;
	~Tile();
	TiledTexture* get_texture() const { // Making this const is a HACK
		return _texture; } 
	void copy_from_subimage(Image *img, uint xoffset, uint yoffset);
private:
	Tile(Rect r, TiledTexture* t) : Rect(r), Refcount(true), _texture(t) {};
	TiledTexture *_texture;
};

//! Singleton
/*!
 * FIXME
 */

class TileBank
{
public:
	static TileBank* instance();
	~TileBank();
	Tile::handle get_tile(iv2 size);
	void free_tile(Tile* tile, TiledTexture *texture);
	Texture* debug_texture(uint i = 0);
	void load_textures();
protected:
	TileBank();
private:
	static TileBank _instance;
	bool   new_texture();
	std::vector<TiledTexture*> _textures;
	typedef std::vector<TiledTexture*>::iterator _texture_iterator;
	iv2    _texture_size;
};

// ------------------------------------------------------------------------- //

//! Texture encapsulates OpenGL's textures
/*!
 * When a Texture is constructed it is also registred with the OpenGL
 * renderer. Use get_handle() to get the OpenGL handle (sometimes known
 * as the texture name).
 */
class Texture
{
public:
	Texture(uint w, uint h);
//	Texture(const Image& img);
	~Texture();
	GLuint      get_handle() const { return _tex_name; };
	uint        get_width()  const { return _image->get_width(); };
	uint        get_height() const { return _image->get_height(); };
	void        load_from_subimage(const Image& img, uint x, uint y, 
			        uint w, uint h);
	void        load();
	Image*      get_image() { return _image; }
private:
	void        gl_bind();
	GLuint      _tex_name;
	Image       *_image;
};

//! TiledTexture 
/*!
 * FIXME
 */

class TiledTexture : public Texture
{
public:
	TiledTexture(uint w, uint h)
		: Texture(w,h),
		  _tiles_used(0),
		  _baseline(0),
		  _current_line_used(0, 0)
	{ }
	~TiledTexture() { };
	Tile::handle get_tile(iv2 size);
	bool         free_tile(Tile* tile);
private:
	//std::vector<Tile> _tiles;
	//typedef std::vector<Tile>::iterator
	             //_tile_iterator;
	uint         _tiles_used;
	uint         _baseline;
	iv2          _current_line_used;
};

#endif /* _PRIMITIVE_H_ */

// ------------------------------------------------------------------------- //

