//
// LocalDateTime.cpp
//
// $Id: //poco/1.2/Foundation/src/LocalDateTime.cpp#1 $
//
// Library: Foundation
// Package: DateTime
// Module:  LocalDateTime
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/LocalDateTime.h"
#include "Poco/Timezone.h"
#include "Poco/Timespan.h"
#include <algorithm>


namespace Poco {


LocalDateTime::LocalDateTime():
	_tzd(Timezone::tzd())
{
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


LocalDateTime::LocalDateTime(int year, int month, int day, int hour, int minute, int second, int millisecond, int microsecond):
	_dateTime(year, month, day, hour, minute, second, millisecond, microsecond),
	_tzd(Timezone::tzd())
{
}


LocalDateTime::LocalDateTime(int tzd, int year, int month, int day, int hour, int minute, int second, int millisecond, int microsecond):
	_dateTime(year, month, day, hour, minute, second, millisecond, microsecond),
	_tzd(tzd)
{
}


LocalDateTime::LocalDateTime(double julianDay):
	_dateTime(julianDay),
	_tzd(Timezone::tzd())
{
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


LocalDateTime::LocalDateTime(int tzd, double julianDay):
	_dateTime(julianDay),
	_tzd(tzd)
{
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


LocalDateTime::LocalDateTime(const DateTime& dateTime):
	_dateTime(dateTime),
	_tzd(Timezone::tzd())
{
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


LocalDateTime::LocalDateTime(int tzd, const DateTime& dateTime):
	_dateTime(dateTime),
	_tzd(tzd)
{
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


LocalDateTime::LocalDateTime(const LocalDateTime& dateTime):
	_dateTime(dateTime._dateTime),
	_tzd(dateTime._tzd)
{
}


LocalDateTime::LocalDateTime(Timestamp::UtcTimeVal utcTime, Timestamp::TimeDiff diff, int tzd):
	_dateTime(utcTime, diff),
	_tzd(tzd)
{
}

	
LocalDateTime::~LocalDateTime()
{
}

	
LocalDateTime& LocalDateTime::operator = (const LocalDateTime& dateTime)
{
	if (&dateTime != this)
	{
		_dateTime = dateTime._dateTime;
		_tzd      = dateTime._tzd;
	}
	return *this;
}


LocalDateTime& LocalDateTime::operator = (const Timestamp& timestamp)
{
	if (timestamp != this->timestamp())
		_dateTime = timestamp;

	return *this;
}


LocalDateTime& LocalDateTime::operator = (double julianDay)
{
	_tzd      = Timezone::tzd();
	_dateTime = julianDay;
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
	return *this;
}

	
LocalDateTime& LocalDateTime::assign(int year, int month, int day, int hour, int minute, int second, int millisecond, int microseconds)
{
	_dateTime.assign(year, month, day, hour, minute, second, millisecond, microseconds);
	_tzd = Timezone::tzd();
	return *this;
}


LocalDateTime& LocalDateTime::assign(int tzd, int year, int month, int day, int hour, int minute, int second, int millisecond, int microseconds)
{
	_dateTime.assign(year, month, day, hour, minute, second, millisecond, microseconds);
	_tzd = tzd;
	return *this;
}


LocalDateTime& LocalDateTime::assign(int tzd, double julianDay)
{
	_tzd      = tzd;
	_dateTime = julianDay;
	_dateTime += Timespan(((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
	return *this;
}


void LocalDateTime::swap(LocalDateTime& dateTime)
{
	_dateTime.swap(dateTime._dateTime);
	std::swap(_tzd, dateTime._tzd);
}


DateTime LocalDateTime::utc() const
{
	return DateTime(_dateTime.utcTime(), -((Timestamp::TimeDiff) _tzd)*Timespan::SECONDS);
}


bool LocalDateTime::operator == (const LocalDateTime& dateTime) const
{
	return utcTime() == dateTime.utcTime();
}


bool LocalDateTime::operator != (const LocalDateTime& dateTime) const	
{
	return utcTime() != dateTime.utcTime();
}


bool LocalDateTime::operator <  (const LocalDateTime& dateTime) const	
{
	return utcTime() < dateTime.utcTime();
}


bool LocalDateTime::operator <= (const LocalDateTime& dateTime) const	
{
	return utcTime() <= dateTime.utcTime();
}


bool LocalDateTime::operator >  (const LocalDateTime& dateTime) const	
{
	return utcTime() > dateTime.utcTime();
}


bool LocalDateTime::operator >= (const LocalDateTime& dateTime) const	
{
	return utcTime() >= dateTime.utcTime();
}


LocalDateTime LocalDateTime::operator + (const Timespan& span) const
{
	return LocalDateTime(_dateTime.utcTime(), span.totalMicroseconds(), _tzd);	
}


LocalDateTime LocalDateTime::operator - (const Timespan& span) const
{
	return LocalDateTime(_dateTime.utcTime(), -span.totalMicroseconds(), _tzd);	
}


Timespan LocalDateTime::operator - (const LocalDateTime& dateTime) const
{
	return Timespan((utcTime() - dateTime.utcTime())/10);
}


LocalDateTime& LocalDateTime::operator += (const Timespan& span)
{
	_dateTime += span;
	return *this;
}


LocalDateTime& LocalDateTime::operator -= (const Timespan& span)
{
	_dateTime -= span;
	return *this;
}


} // namespace Poco
