

//////////////////////////////////////////////////////////////////
//                                                              //
//           PLINK (c) 2005-2008 Shaun Purcell                  //
//                                                              //
// This file is distributed under the GNU General Public        //
// License, Version 2.  Please see the file COPYING for more    //
// details                                                      //
//                                                              //
//////////////////////////////////////////////////////////////////


#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <vector>
#include <map>
#include <iterator>
#include "plink.h"
#include "helper.h"
#include "options.h"
#include "perm.h"

const double EPS_OVERLAP = 1e-6;

bool intersects(set<Range>&,set<Range>&,int,int,int);
int count_intersects(set<Range>&,int,int,int);
vector<int> segmentCountCaseControls(Plink*,bool);

void Plink::setUpForCNVList()
{

  if ( ! par::cnv_makemap ) 
    {

      ///////////////////////////////////////////////
      // .map file 

      checkFileExists(par::mapfile);

      printLOG("Reading marker information from [ " 
	       + par::mapfile + " ]\n");

      vector<bool> include;
      vector<int> include_pos(0);
      int nl_actual=0;
      
      readMapFile(par::mapfile,include,include_pos,nl_actual);


      ///////////////////////////////////////////////
      // .fam file 
      
      printLOG("Reading individual information from [ " 
	       + par::famfile + " ]\n");
      checkFileExists(par::famfile);
      readFamFile(par::famfile);  


      // Set some basics that we would have skipped by 
      // doing the above manually

      nl_all = locus.size();
      n = sample.size();
      prettyPrintLengths();

    }


}

void Plink::readCNVList()
{


  // These should be done already, but check

  nl_all = locus.size();
  n = sample.size();
  prettyPrintLengths();


  ///////////////////////////////////////////////
  // Intersect with a range file?

  set<Range> isection;
  map<Range,string> idescription;
  set<Range> iintersected;

  if ( par::cnv_intersect )
    {
      checkFileExists( par::cnv_intersect_file );

      printLOG("Reading CNV intersection list from [ " 
	       + par::cnv_intersect_file + " ]\n");
      ifstream IN(par::cnv_intersect_file.c_str(), ios::in);
      
      while ( ! IN.eof() )
	{
	  // First three fields are CHR, M1, M2
	  // Ignore rest of the line
	  
	  char cline[par::MAX_LINE_LENGTH];
	  IN.getline(cline,par::MAX_LINE_LENGTH,'\n');
	  string sline = cline;
	  if (sline=="") continue;
      
	  string buf; 
	  stringstream ss(sline); 
	  vector<string> tokens; 
	  while (ss >> buf)
	    tokens.push_back(buf);
	  
	  if ( tokens.size() < 3 ) 
	    error("Problem with line:\n" + sline );

	  string chr = tokens[0];
	  string m1 = tokens[1];
	  string m2 = tokens[2];
	  
	  int p1,p2;
	  Range r;
	  if (chr=="") 
	    continue;
	  if ( ! from_string<int>( r.start,m1, std::dec ) )
	    error("Problem with position : " + m1 );
 	  if ( ! from_string<int>( r.stop,m2, std::dec ) )
	    error("Problem with position : " + m2 );

	  // Add any border
	  r.start -= par::cnv_region_border;
	  r.stop += par::cnv_region_border;

	  // Check for consistency
	  if ( r.start > r.stop ) 
	    error("Badly defined region:\n"+sline);

	  r.chr = getChromosomeCode(chr);

	  isection.insert(r);
	  idescription.insert(make_pair(r,sline));
	}
      IN.close();

      if ( par::cnv_exclude ) 
	printLOG("Read " + int2str( isection.size() ) 
		 + " ranges to exclude from CNV list\n");
      else
	printLOG("Read " + int2str( isection.size() ) 
		 + " ranges to intersect with CNV list\n");

    }


  ///////////////////////////////////////////////
  // .cnv file 

  printLOG("\nReading segment list (CNVs) from [ " 
	   + par::cnv_listname + " ]\n");
  
  checkFileExists( par::cnv_listname );

  ifstream IN;
  IN.open( par::cnv_listname.c_str() , ios::in );

  ofstream MOUT;
  if ( par::cnv_writelist )
    {      

      if ( par::output_file_name + ".cnv" == par::cnv_listname )
	error("CNV input and output file names cannot be the same, " 
	      + par::cnv_listname );
    }

  map<string,Individual*> uid;
  map<int2,int> mlocus;

  if ( ! par::cnv_makemap )
    {

      makePersonMap( *this, uid );

      for (int l=0; l<nl_all; l++)
	{
	  int2 p;
	  p.p1 = locus[l]->chr;
	  p.p2 = locus[l]->bp;
	  mlocus.insert(make_pair(p,l));
	}
    }


  map<string,Individual*>::iterator ii;
  map<int2,int>::iterator il1;
  map<int2,int>::iterator il2;
  
  int nseg=0;
  int n_mapped_to_person=0;
  int n_passed_filters=0;
  int n_intersects=0;
  int nall=0;

  set<int2> positions;


  while ( ! IN.eof() )
    {

      string fid, iid, chr, bp1, bp2;
      string type, scorestr, sitesstr;

      // FID IID CHR BP1 BP2 TYPE SCORE SITES
      
      IN >> fid >> iid 
	 >> chr >> bp1 >> bp2 
	 >> type >> scorestr >> sitesstr;
      
      
      if ( fid == "FID" || 
	   fid == "" )
	continue;

      nall++;

      // Lookup person
      if ( ! par::cnv_makemap )
	{
	  ii = uid.find( fid + "_" + iid );      
	  if ( ii == uid.end() )
	    continue;
	}
      
      ++n_mapped_to_person;

      // Correct type? 
      int t;
      if ( ! from_string<int>( t,type, std::dec ) )
	error("Problem with type specifier: " + type );
	    
      if ( par::cnv_del_only && t > 1 ) 
	continue;

      if ( par::cnv_dup_only && t < 3 ) 
	continue;

      
      int p1,p2;
      if ( ! from_string<int>( p1, bp1, std::dec ) )
	error("Problem with first position: " + bp1 );
      if ( ! from_string<int>( p2, bp2, std::dec ) )
	error("Problem with second position: " + bp2 );

      if ( p1 > p2 ) 
	error("Badly defined segment, " + bp1 + " > " + bp2);


      double score;
      int sites;
      
      if ( ! from_string<double>( score, scorestr, std::dec ) )
	error("Problem with score : " + scorestr );
	    
      if ( ! from_string<int>( sites, sitesstr, std::dec ) )
	error("Problem with sites : " + sitesstr );
      
      
      // Filters:

      double kb = (double)(p2 - p1) / 1000.0;

      if ( par::cnv_min_sites > 0 && par::cnv_min_sites > sites )
	continue;
      if ( par::cnv_max_sites > 0 && par::cnv_max_sites < sites )
	continue;

      if ( par::cnv_min_score > 0 && par::cnv_min_score > score )
	continue;
      if ( par::cnv_max_score > 0 && par::cnv_max_score < score )
	continue;

      if ( par::cnv_min_kb > 0 && par::cnv_min_kb > kb )
	continue;
      if ( par::cnv_max_kb > 0 && par::cnv_max_kb < kb )
	continue;

      ++n_passed_filters;


      ///////////////////////////////////////////////
      // Intersect with range as specified from file
      // (optionally counting overaps, instead)

      int segment_contains = 0;
      if ( par::cnv_intersect ) 
	{
	  if ( par::cnv_count ) 
	    segment_contains += 
	      count_intersects(isection,getChromosomeCode(chr),p1,p2);
	  else
	    {
	      if ( par::cnv_exclude && 
		   intersects(isection,iintersected,getChromosomeCode(chr),p1,p2) )
		continue;
	      if ( (!par::cnv_exclude) && 
		   !intersects(isection,iintersected,getChromosomeCode(chr),p1,p2) )
		continue;
	    }	  
	  ++n_intersects;
	}


      int2 p;
      p.p1 = getChromosomeCode( chr );
      
      if ( par::cnv_makemap )
	{

	  int2 p;
	  p.p1 = getChromosomeCode( chr );

	  // Start
	  p.p2 = p1;
	  positions.insert(p);

	  // End
	  p.p2 = p2;
	  positions.insert(p);
	  
	  // One position past end
	  p.p2++;
	  positions.insert(p);
	  
	}
      else
	{

	  // Can we map this to an exact marker?

	  p.p2 = p1;
	  il1 = mlocus.find( p );
	  
	  p.p2 = p2;
	  il2 = mlocus.find( p );

	  if ( il1 == mlocus.end() || 
	       il2 == mlocus.end() )
	    continue;

	  
	  // Seems okay, add segment to list

	  Segment s;
	  
	  s.start = il1->second;
	  s.finish = il2->second;
	  s.p1 = s.p2 = ii->second;
	  s.count = segment_contains;
	  s.type = t;
	  s.score = score;
	  s.sites = sites;
	  segment.push_back(s);
	  
	}
        
    }
  

  /////////////////////////////////////////////////////////////
  // Write-back any intersected segments (or, non-intersected 
  // segments, if in exclude mode)

  if ( par::cnv_intersect_writeback )
    {
      
      if ( par::cnv_exclude ) 
	printLOG("Writing back list to non-intersected regions to [ " 
		 + par::output_file_name + ".reg ]\n");
      else
	printLOG("Writing back list to intersected regions to [ " 
		 + par::output_file_name + ".reg ]\n");
      
      ofstream ROUT;
      ROUT.open( ( par::output_file_name+".reg").c_str(), ios::out );
	 
      set<Range>::iterator ir = isection.begin();     
      while ( ir != isection.end() )
	{

	  set<Range>::iterator i = iintersected.find(*ir);
	  
	  bool writeback = par::cnv_exclude ? 
	    i == iintersected.end() :
	    i != iintersected.end() ; 
	  
	  if ( writeback )
	    {	      
	      map<Range,string>::iterator is = idescription.find(*ir);	  
	      ROUT << idescription[*ir] << "\n";
	    }
	  ++ir;	  

	}
      
      ROUT.close();
    }
  


  /////////////////////////////////////////////////////////////
  // Drop segments that are above or below a certain threshold

  int removed_freq_filter = 0;
  
  // Old approach was to define genomic regions 
  // and then filter based on these. 
  
  // New approach (below) is to focus on each event more directly, 
  // and ask how many others intersect with it.  Populate "freq"
  // for each event;  
  // Overlap definition wants to be 
  //  1) NOT "disrupt"
  //  2) Inforce union overlap so that frequency groups are 
  //     symmetric 
 
  // Clean up this code later (decide if we want to keep these old 
  // definitions

  if ( par::cnv_freq_method2 )
    {
      if ( par::cnv_freq_include ) 
	{
	  
	  if ( par::cnv_freq_include )
	    printLOG("Filtering segments based on frequencies\n");	  
	  
	  
	  // Turn off potential write-back function for 
	  // ranges; and manually set the intersect function
	  // to work as needed here
	  
	  par::cnv_intersect_writeback = false;
	  par::cnv_disrupt = false;
	  par::cnv_union_overlap = true;
	  par::cnv_region_overlap = false;
	  
	  for (int s1=0; s1<segment.size(); s1++)
	    {
	      Segment * seg1 = &(segment[s1]);;
	      int chr = locus[ seg1->start ]->chr;
	      int bp1 = locus[ seg1->start ]->bp;
	      int bp2 = locus[ seg1->finish ]->bp;
	      
	      // Count overlap with self, once
	      ++seg1->freq;
	      
	      for (int s2=s1+1; s2<segment.size(); s2++)
		{
		  Segment * seg2 = &(segment[s2]);
		  
		  // Skip if different chromosome
		  if ( locus[ seg2->start ]->chr != chr ) 
		    continue;
		  
		  // Do these overlap?
		  
		  int t1 = locus[seg2->start ]->bp;
		  int t2 = locus[seg2->finish ]->bp;
		  
		  // This seg ends before the other starts?
		  if ( t2 < bp1 )
		    continue;
		  
		  // Or does this seg start after the other ends?
		  if ( t1 > bp2 ) 
		    continue;
		  
		  // Calculate overlap (union)
		  
		  double consensusStart = t1 > bp1 ? t1 : bp1;
		  double consensusStop = t2 < bp2 ? t2 : bp2;	      
		  double numerator = consensusStop - consensusStart + 1.0;
		  double unionStart = t1 < bp1 ? t1 : bp1;
		  double unionStop = t2 > bp2 ? t2 : bp2;
		  double denom = unionStop - unionStart + 1.0;
		  double overlap = numerator / denom ;
		  overlap += EPS_OVERLAP;
		  
		  if ( overlap > par::cnv_freq_method2_threshold )
		    {
		      seg1->freq++;
		      seg2->freq++;
		    }
		}
	    }
	  
	  
	  vector<Segment>::iterator s = segment.begin();
	  
	  while ( s != segment.end() )
	    {
	      //cout << "seg count = " << s->freq << "\n";
	      
	      if ( par::cnv_freq_include_exact 
		   && s->freq != par::cnv_freq_include_cnt )
		{
		  s = segment.erase(s);
		  ++removed_freq_filter;
		}
	      else if ( par::cnv_freq_include_below 
			&& s->freq > par::cnv_freq_include_cnt )
		{
		  s = segment.erase(s);
		  ++removed_freq_filter;
		}
	      else if ( s->freq < par::cnv_freq_include_cnt )
		{
		  s = segment.erase(s);
		  ++removed_freq_filter;
		}
	      else
		s++;
	    }
	  
	  printLOG("Will remove " + int2str( removed_freq_filter ) 
		   + " CNVs based on frequency (after other filters)\n");
	  
	}
    }


  ////////////////////////////
  // Old frequency filter code

  if ( ! par::cnv_freq_method2 ) 
    {
      if ( par::cnv_freq_include || par::cnv_unique )
	{
	
	if ( par::cnv_unique )
	  printLOG("Filtering segments unique to cases/controls\n");
	
	if ( par::cnv_freq_include )
	  printLOG("Filtering segments based on frequencies\n");	  
	  
	  // Clear any existing segments (we are done with gene lists, etc)
	  // by now
	  
	  isection.clear();
	  
	  // 1) Find common regions
	  // 2) intersect or exclude as is fit
	  // 3) remove from list
	  
	  vector<int> caseCount = segmentCountCaseControls(this,true);
	  vector<int> controlCount = segmentCountCaseControls(this,false);
	  
	  // Determine regions to exclude
	  
	  bool inRegion = false;
	  Range r;
	  
	  for (int l=0; l<nl_all; l++)
	    {
	      
	      //////////////////////////////
	      // End of chromosome/genome?
	      
	      bool endOfChromosome = false;
	      
	      if ( inRegion && ( l == nl_all-1 || locus[l+1]->chr != locus[l]->chr ) )
		{
		  endOfChromosome = true;
		}
	  
	      int count = caseCount[l] + controlCount[l];
	      bool uniq = caseCount[l] == 0 || controlCount[l] == 0;
	      
	      bool acceptable;
	      
	      if ( par::cnv_freq_include_exact ) 
		{
		  acceptable = count == par::cnv_freq_include_cnt;
		}
	      else 
		{
		  
		  // Use inclusive thresholds for inclusion
		  // --cnv-freq-exclude-above X
		  if ( par::cnv_freq_include_below ) // inclusive thresholds
		    acceptable = count <= par::cnv_freq_include_cnt; 
		  else // --cnv-freq-exclude-below X
		    acceptable = count >= par::cnv_freq_include_cnt;  
		}
	      
	      
	      /////////////////////////////////////////////////////////
	      // Filter based on uniquenes to either cases or controls
	      
	      if ( inRegion )
		{
		  
		  // End of an acceptable or unique region?
		  if ( ( ( par::cnv_unique && ! uniq ) || ( ! par::cnv_unique ) ) &&
		       ( ( par::cnv_freq_include && ! acceptable ) || ( ! par::cnv_freq_include ) ) )
		    {
		      // Range goes up to the position just before this region
		      r.stop = locus[l]->bp-1;
		      if ( r.stop < 0 ) r.stop = 0;
		      isection.insert(r);
		      inRegion = false;
		    }
		  else if ( endOfChromosome ) // of we just have to stop anyway?
		    {
		      inRegion = false;
		      r.stop = locus[l]->bp;
		      isection.insert(r);
		      continue;
		    }
		}
	      // ...or, the start of a new region?
	      else if ( ( par::cnv_unique && uniq ) || 
			( par::cnv_freq_include && acceptable ) )
		{
		  r.start = locus[l]->bp;
		  r.chr = locus[l]->chr;
		  inRegion = true;
		}	      
	      
	    } // Next SNP
	  
	  
	  /////////////////////////////////////////////////////////////
	  // We now have a list of "acceptable sections"  -- intersect 
	  // based on these
	  
	  // Turn off potential write-back function for 
	  // ranges
	  
	  par::cnv_intersect_writeback = false;
	  
	  vector<Segment>::iterator s = segment.begin();
	  
	  while ( s != segment.end() )
	    {
	      
	      bool doesIntersect = intersects(isection,
					      iintersected,
					      locus[s->start]->chr,
					      locus[s->start]->bp,
					      locus[s->finish]->bp);
	      
	      if ( par::cnv_freq_include_exact_exclude )
		doesIntersect = ! doesIntersect;
	      
	      // Remove this segment or keep?
	      if ( ! doesIntersect )
		{
		  s = segment.erase(s);
		  ++removed_freq_filter;
		}
	      else
		++s;
	    }
	  
	  printLOG("Will remove " + int2str( removed_freq_filter ) 
		   + " CNVs based on frequency (after other filters)\n");
	  
	}
}
  
  
  
  

  // Drop individuals for whom we do not see any segments of this
  // variety?

  if ( par::cnv_drop_no_segment )
    {
      vector<bool> indel(n,false);      
      indivSegmentSummaryCalc(segmentCount, segmentLength, true, true);
      
      for ( int i = 0; i < n; i++)
	{
	  indivPair t;
	  t.p1 = t.p2 = sample[i];
	  map<indivPair,int>::iterator ic = segmentCount.find(t);
	  map<indivPair,double>::iterator il = segmentLength.find(t);
	  	  
	  if ( ic != segmentCount.end() ) 
	    {
	      if ( ic->second == 0 )
		indel[i] = true;
	    }
	  else
	    indel[i] =true;
	}
      
      int n_removed = deleteIndividuals(indel);
      printLOG("Removed " + int2str(n_removed) + " individuals with <1 CNV\n");      
    }


  ///////////////////////////////////////////////////////////
  // Check that CNVs do not overlap at all (within a person)
    
  if ( par::cnv_check_overlap ) 
    {
      error("Not yet implemented...\n");

      vector<Segment>::iterator s1 = segment.begin();
      while ( s1 != segment.end() )
	{
	  vector<Segment>::iterator s2 = s1 + 1;
	  while ( s2 != segment.end() )
	    {
	      //if ( s1->p1 == s2->p1 )		
	      ++s2;
	    }
	  ++s1;
	}
    }


  ////////////////////////////////////////////////////////
  // Get full typeCount; for case/control data, split out 
  // by cases and controls
  
  map<int,int> typeCount;
  map<int,int> typeCaseCount;
  
  vector<Segment>::iterator s = segment.begin();
  while ( s != segment.end() )
    {
      
      //////////////////////////////////
      // Record in full 0/1, 3/4 space
      
      map<int,int> & myCount = par::bt && s->p1->aff ? 
	typeCaseCount : typeCount;
      
      map<int,int>::iterator it = myCount.find( s->type );
            
      if ( it == myCount.end() )
	{
	  myCount.insert(make_pair(s->type,1));
	}
      else
	{
	  ++(it->second);
	}

      // Next segment
      ++s;
    }



  //////////////////////////////
  // Make a new map file

  if ( ! par::cnv_makemap )
    {

      printLOG(int2str( n_mapped_to_person ) + " mapped to a person, of which " 
	       + int2str( n_passed_filters) + " passed filters\n");

      if ( par::cnv_intersect )
	{
	  if ( par::cnv_exclude )
	    printLOG( int2str( n_intersects ) 
		      + " kept after excluding specific regions\n");
	  else
	    printLOG( int2str( n_intersects ) 
		      + " intersected with one or more specified region\n");
	}
      
      int t = par::cnv_intersect ? n_intersects : n_passed_filters;
      t -= removed_freq_filter;
      if ( t - segment.size()  > 0  )
	printLOG( int2str( t - segment.size() ) 
		   + " did not map to any marker\n");

      printLOG(int2str( segment.size() ) 
	       + " of " + int2str(nall) 
	       + " mapped as valid segments\n");
      

      map<int,int>::iterator it1 = typeCaseCount.begin();
      map<int,int>::iterator it2 = typeCount.begin();
      set<int> obsCounts;

      while ( it1 != typeCaseCount.end() )
	{
	  obsCounts.insert( it1->first );
	  ++it1;
	}
      while ( it2 != typeCount.end() )
	{
	  obsCounts.insert( it2->first );
	  ++it2;
	}
      
      stringstream s2;
      
      if ( par::bt )
	s2 << setw(6) << "CopyN" << " " 
	   << setw(12) << "Case/Control" << "\n";
      else
	s2 << setw(6) << "CopyN" << " "
	   << setw(8) << "Count" << "\n";
      
      printLOG( s2.str() );
      s2.clear();
      
      set<int>::iterator it = obsCounts.begin();

      while ( it != obsCounts.end() )
	{
	  map<int,int>::iterator i1 = typeCaseCount.find( *it );
	  map<int,int>::iterator i2 = typeCount.find( *it );
	  
	  int n1 = 0, n2 = 0;

	  if ( i1 != typeCaseCount.end() )
	    n1 = i1->second;
	  
	  if ( i2 != typeCount.end() )
	    n2 = i2->second;

	  stringstream s;

	  if ( par::bt )
	    s << setw(6) << *it << " "
	      << setw(12) << (int2str(n1)+" / "+int2str(n2)) << "\n";
	  else
	    s << setw(6) << *it << " "
	      << setw(8) << n2 << "\n";
	  	  
	  printLOG( s.str() );

	  ++it;
	}
      printLOG("\n");
    }
  
  IN.close();
  


  ////////////////////////////////
  // Write a CNV list file? 

  if ( par::cnv_writelist )
    {

      printLOG("Writing new CNV list to [ " 
	       + par::output_file_name 
	       + ".cnv ]\n");
      
      MOUT.open( ( par::output_file_name + ".cnv").c_str() , ios::out );
      MOUT << setw(par::pp_maxfid) << "FID" << " "
	   << setw(par::pp_maxiid) << "IID" << " ";
      if (par::dump_covar_with_phenotype )
	MOUT << setw(8) << "PHE" << " ";
      MOUT << setw(4) << "CHR" << " "
	   << setw(12) << "BP1" << " "   
	   << setw(12) << "BP2" << " "
	   << setw(6) << "TYPE" << " "
	   << setw(12) << "SCORE" << " "
	   << setw(8) << "SITES" << "\n";
      
      vector<Segment>::iterator s = segment.begin();
      while ( s != segment.end() )
	{
	  
	  Individual * person = s->p1;
	  Locus * loc1 = locus[s->start];
	  Locus * loc2 = locus[s->finish];

	  MOUT << setw(par::pp_maxfid) << person->fid << " "
	       << setw(par::pp_maxiid) << person->iid << " ";
	  if (par::dump_covar_with_phenotype )
	    MOUT << setw(8) << person->phenotype << " ";
	  MOUT  << setw(4) << loc1->chr << " "
		<< setw(12) << loc1->bp << " "   
		<< setw(12) << loc2->bp << " "
		<< setw(6) << s->type << " "
		<< setw(12) << s->score << " "
		<< setw(8) << s->sites << "\n";
	 	  
	  ++s;
	}
      MOUT.close();

      printLOG("Writing new FAM file to [ "
               + par::output_file_name
               + ".fam ]\n");
      MOUT.open( ( par::output_file_name + ".fam").c_str() , ios::out );

      for (int i=0;i<n;i++)
	{
	  Individual * person = sample[i];
	  MOUT << person->fid << " "
	       << person->iid << " "
	       << person->pat << " "
	       << person->mat << " "
	       << person->sexcode << " ";
	  if (par::bt)
	    MOUT << (int)person->phenotype << "\n";
	  else
	    MOUT << person->phenotype << "\n";
	}      
      MOUT.close();
    }  


  /////////////////////////////////////
  // Collapse type (0/1 -> 2, 3/4 -> 2)

  s = segment.begin();
  while ( s != segment.end() )
    {
      s->type = s->type < 2 ? 1 : 2;
      ++s;
    }


  ////////////////////////////////
  // Write a new MAP file out?

  if ( par::cnv_makemap )
    {
      ofstream MOUT;
      printLOG("Writing new MAP file to [ " 
	       + par::output_file_name + ".cnv.map ]\n");
      MOUT.open( ( par::output_file_name + ".cnv.map").c_str() , ios::out );
      set<int2>::iterator ip = positions.begin();
      int nseg = 1;
      while ( ip != positions.end() )
	{
	  MOUT << ip->p1 << "\t"
	       << "p"+int2str(ip->p1)+"-"+int2str(ip->p2) << "\t"
	       << "0\t"
	       << ip->p2 << "\n"; 
	  ++ip;
	}
      MOUT.close();
      
      printLOG("Wrote " + int2str( positions.size() ) 
	       + " unique positions to file\n");
    }

 
}


void Plink::processCNVList()
{

  if (  par::cnv_writelist || par::cnv_makemap )
    return;

  // Per-individual summaries
  printLOG("Writing per-individual summary to [ " 
	   + par::output_file_name +
	   ".cnv.indiv ]\n");
  indivSegmentSummary();

  // Display convenient segment view
  if (par::display_segment_long)
    displaySegmentsLong();

  // Display convenient BED/UCSC track
  if (par::display_cnv_track)
    displaySegmentsBED();

  // Find overlap in segments?
  if (par::segment_overlap)
    summariseHomoRuns();  

  // Utilise existing tests for homozygous segments
  printLOG("Writing positional summary to [ " 
	   + par::output_file_name 
	   + ".cnv.summary ]\n");

  summaryIBSsegments(*pperm);
}


bool intersects(set<Range>& isection , set<Range> & iintersects, int chr, int p1, int p2)
{

  // Does this particular CNV intersect with atleast one range?
  // Potentially allowing for fractional overlaps)

  // Either consider all ranges and report back true when 
  // first intersection is seen
  
  // Or; consider all ranges, no matter what, keeping track
  // of which ranges have been intersected (i.e. if we have
  // overlapping ranges, for example)
  
  // Intersect: default -- if any of segment overlaps
  //      param 0.2     -- if at least20% of segment overlaps
  
  // Alternatively, "disrupt" mode simply asks whether the start 
  // or end of a CNV is within the region

  bool doesIntersect = false;
  
  set<Range>::iterator ir = isection.begin();
  
  while ( ir != isection.end() )
    {
      if ( ir->chr != chr )
	{
	  ++ir;
	  continue;
	}

      // Either use disrupt mode, or standard
      // intersect mode (which might include 
      // and overlap)

      if ( par::cnv_disrupt )
	{
	  
	  if (  ( p1 >= ir->start && p1 <= ir->stop ) ||	   
		( p2 >= ir->start && p2 <= ir->stop ) )
	    {
	      if ( par::cnv_intersect_writeback )
		{
		  doesIntersect = true;
		  iintersects.insert(*ir);
		}
	      else
		return true;
	    }
	}
      else // ... intersect or exclude mode
	{
	  if ( p1 < ir->stop && p2 > ir->start )
	    {
	      
	      if ( par::cnv_overlap < 0 )
		{
		  if ( par::cnv_intersect_writeback )
		    {
		      doesIntersect = true;
		      iintersects.insert(*ir);
		    }
		  else
		    return true;
		}
	      
	      /////////////////////////////
	      // Overlap-based comparison
		
	      // The CNV spans p1 to p2
	      // Region spans ir->start/stop

              // Denominator either of CNV itself, 
              // or is union

	      double consensusStart = p1 > ir->start ? p1 : ir->start;
	      double consensusStop = p2 < ir->stop ? p2 : ir->stop;
	      
	      double numerator = consensusStop - consensusStart + 1.0;

	      double denom;
	      if ( par::cnv_union_overlap )
		{
		  double unionStart = p1 < ir->start ? p1 : ir->start;
		  double unionStop = p2 > ir->stop ? p2 : ir->stop;
		  denom = unionStop - unionStart + 1.0;
		}
	      else if ( par::cnv_region_overlap )
		denom = ir->stop-ir->start+1.0;
	      else
		denom = p2-p1+1.0;
	      
	      double overlap = numerator / denom ;
	      overlap += EPS_OVERLAP;

	      if ( overlap >= par::cnv_overlap )
		{
		  if ( par::cnv_intersect_writeback )
		    {
		      doesIntersect = true;
		      iintersects.insert(*ir);
		    }
		  else
		    return true;
		}
	    }
	}
      
      // Consider next range
      ++ir;
    }
  
  return doesIntersect;

}


int count_intersects(set<Range>& isection ,int chr,int p1,int p2)
{

  // How many ranges does this CNV intersect with?
  // Potentially allowing for fractional overlaps)

  // Intersect: default -- if any of segment overlaps
  //      param 0.2     -- if at least20% of segment overlaps

  int iCount = 0;

  set<Range>::iterator ir = isection.begin();
  while ( ir != isection.end() )
    {
      if ( ir->chr != chr )
	{
	  ++ir;
	  continue;
	}

      // Either use disrupt mode, or standard
      // intersect mode (which might include 
      // and overlap)

      if ( par::cnv_disrupt )
	{
	  
	  if (  ( p1 >= ir->start && p1 <= ir->stop ) ||	   
		( p2 >= ir->start && p2 <= ir->stop ) )
	    {
	      ++iCount;
	    }
	}
      else // ... intersect or exclude mode
	{	  
	  
	  if ( p1 < ir->stop && p2 > ir->start )
	    {
	      
	      if ( par::cnv_overlap < 0 ) 
		++iCount;
	      else
		{
		  // The CNV spans p1 to p2
		  double consensusStart = p1 > ir->start ? p1 : ir->start;
		  double consensusStop = p2 < ir->stop ? p2 : ir->stop;

		  double numerator = consensusStop - consensusStart + 1.0;

		  double denom;
		  if ( par::cnv_union_overlap )
		    {
		      double unionStart = p1 < ir->start ? p1 : ir->start;
		      double unionStop = p2 > ir->stop ? p2 : ir->stop;
		      denom = unionStop - unionStart + 1.0;
		    }
		  else if ( par::cnv_region_overlap )
		    denom = ir->stop-ir->start+1.0;
		  else
		    denom = p2-p1+1.0;
		  
		  double overlap = numerator / denom ;
		  
		  overlap += EPS_OVERLAP;
		  
		  if ( overlap >= par::cnv_overlap )
		    ++iCount;
		}
	    }
	}
      
      ++ir;
    }
  return iCount;
}



vector<int> segmentCountCaseControls(Plink * P, bool countCases)
{

  // Helper function to count segments (for CNVs, homozygous segments
  // only)

  vector<int> count(P->nl_all,0);
    
  vector<Segment>::iterator s = P->segment.begin();
  
  while ( s != P->segment.end() )
    {          
      if ( s->p1->pperson->aff == countCases ) 
	for (int l = s->start ; l <= s->finish; l++) 
	  count[l]++;
      s++;
    }
  return count;
}
