/************************************************************************
 * $Id: decode.c,v 1.2 2002/07/21 23:14:55 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  Ulrich Weigand
 *
 *  decode.c:   Intel i386 instruction decoder.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "decode.h"

#define FALSE 0
#define TRUE  1

/* Operand Size dependent instruction names */

#define SizeDep( opw, opd )       ( (opw) | ((opd) << 16) )
#define IsSizeDep( instr )        ( ((instr) >> 16) != 0 )
#define GetSizeDep( instr, size ) ( ((instr) >> ((size)==2? 0:16)) & 0xffff )

#define INS_cbw_cwde   SizeDep( INS_cbw,   INS_cwde   )
#define INS_cwd_cdq    SizeDep( INS_cwd,   INS_cdq    )
#define INS_jEcxz      SizeDep( INS_jcxz,  INS_jecxz  )
#define INS_pushav     SizeDep( INS_pusha, INS_pushad )
#define INS_popav      SizeDep( INS_popa,  INS_popad  )
#define INS_pushfv     SizeDep( INS_pushf, INS_pushfd )
#define INS_popfv      SizeDep( INS_popf,  INS_popfd  )

/* Instruction Groups */

#define INS_Grp1   (INS_LAST+101)
#define INS_Grp2   (INS_LAST+102)
#define INS_Grp3   (INS_LAST+103)
#define INS_Grp4   (INS_LAST+104)
#define INS_Grp5   (INS_LAST+105)
#define INS_Grp6   (INS_LAST+106)
#define INS_Grp7   (INS_LAST+107)
#define INS_Grp8   (INS_LAST+108)
#define INS_Grp9   (INS_LAST+109)
#define INS_Grp13  (INS_LAST+110)
#define INS_Grp14  (INS_LAST+111)
#define INS_Grp15  (INS_LAST+112)
#define INS_Grp16  (INS_LAST+113)
#define INS_Grp17  (INS_LAST+114)

#define IsInstrGroup( instr )   ( (instr) >= INS_Grp1 && (instr) <= INS_Grp17 )
#define GetInstrGroup( instr )  ( (instr) - INS_Grp1 )

/* Escape Instructions */

#define INS_Esc1   (INS_LAST+151)
#define INS_Esc2   (INS_LAST+152)
#define INS_Esc3   (INS_LAST+153)
#define INS_Esc4   (INS_LAST+154)
#define INS_Esc5   (INS_LAST+155)
#define INS_Esc6   (INS_LAST+156)
#define INS_Esc7   (INS_LAST+157)
#define INS_Esc8   (INS_LAST+158)

#define IsInstrEsc( instr )   ( (instr) >= INS_Esc1 && (instr) <= INS_Esc8 )
#define GetInstrEsc( instr )  ( (instr) - INS_Esc1 )

/* Special Prefixes */

#define INS_InsPrefix        (INS_LAST+201)
#define INS_AddrSizePrefix   (INS_LAST+202)
#define INS_OpSizePrefix     (INS_LAST+203)
#define INS_SegPrefix        (INS_LAST+204)
#define INS_TwoBytePrefix    (INS_LAST+205)


/* Operand identifiers */

#define OP_General   (0x01 << 16)
#define OP_Register  (0x02 << 16)
#define OP_Segment   (0x04 << 16)
#define OP_FPUStack  (0x08 << 16)
#define OP_Immediate (0x10 << 16)

#define AddrMode_0   0
#define AddrMode_A   1  /* direct address */
#define AddrMode_C   2  /* control reg in reg field */
#define AddrMode_D   3  /* debug reg in reg field */
#define AddrMode_E   4  /* effective address */
#define AddrMode_F   5  /* flags reg */
#define AddrMode_G   6  /* general reg in reg field */
#define AddrMode_I   7  /* immediate data */
#define AddrMode_J   8  /* (E)IP relative offset */
#define AddrMode_M   9  /* effective memory address */
#define AddrMode_O  10  /* memory offset */
#define AddrMode_P  11  /* packed quadword MMX reg */
#define AddrMode_Q  12  /* MMX reg or effective address */
#define AddrMode_R  13  /* general reg in mod field */
#define AddrMode_S  14  /* segment reg in reg field */
#define AddrMode_T  15  /* test reg in reg field */
#define AddrMode_V  16  /* packed SIMD floating-point reg */
#define AddrMode_W  17  /* SSE reg or effective address */
#define AddrMode_X  18  /* memory addressed by DS:SI */
#define AddrMode_Y  19  /* memory addressed by ES:DI */

int AddrHasModRM[] = { FALSE, FALSE, TRUE, TRUE, TRUE, FALSE, TRUE, FALSE, 
                       FALSE, TRUE, FALSE, TRUE, TRUE, TRUE, TRUE, TRUE, 
                       TRUE, TRUE, FALSE, FALSE };

#define OpType_0     0
#define OpType_a     1  /* two WORD/DWORD sized operands in memory */
#define OpType_b     2  /* BYTE */
#define OpType_c     3  /* BYTE expanded to WORD/DWORD */
#define OpType_d     4  /* DWORD */
#define OpType_p     5  /* segmented pointer, offset size WORD/DWORD */
#define OpType_o     6  /* octalword (16 byte) */
#define OpType_q     7  /* quadword (8 byte) */
#define OpType_s     8  /* six-byte pseudo-descriptor */
#define OpType_v     9  /* WORD/DWORD */
#define OpType_w    10  /* WORD */
#define OpType_x    11  /* single-precision real (4 byte) */
#define OpType_y    12  /* double-precision real (8 byte) */
#define OpType_z    13  /* extended-precision real (10 byte) */

#define RegCode_AX   0
#define RegCode_CX   1
#define RegCode_DX   2
#define RegCode_BX   3
#define RegCode_SP   4
#define RegCode_BP   5
#define RegCode_SI   6
#define RegCode_DI   7

#define op(mode,type)     ( OP_General|(AddrMode_##mode << 8)|(OpType_##type) )
#define IsOpGeneral(op)   ( ((op) & OP_General) != 0 )
#define GenAddrMode(op)   ( ((op) >> 8) & 0xff )
#define GenOpType(op)     ( (op) & 0xff )

#define reg(name,type)    ( OP_Register|(RegCode_##name << 8)|(OpType_##type) )
#define IsOpRegister(op)  ( ((op) & OP_Register) != 0 )
#define RegName(op)       ( ((op) >> 8) & 0xff )
#define RegType(op)       ( (op) & 0xff )

#define seg(name)         ( OP_Segment | SEG_##name  )
#define IsOpSegment(op)   ( ((op) & OP_Segment) != 0 )
#define SegName(op)       ( (op) & 0xffff )

#define st(name)          ( OP_FPUStack | P86_REG_ST##name  )
#define IsOpFPUStack(op)  ( ((op) & OP_FPUStack) != 0 )
#define FPUStackName(op)  ( (op) & 0xffff )

#define imm(value)        ( OP_Immediate | value )
#define IsOpImmediate(op) ( ((op) & OP_Immediate) != 0 )
#define ImmValue(op)      ( (op) & 0xffff )


/* Instruction Tables */

struct instruction
{
    int instr;
    int op0;
    int op1;
    int op2;
};

struct instruction OneByteMap[] =
{
/*00*/  { INS_add,    op(E,b),   op(G,b) },
/*01*/  { INS_add,    op(E,v),   op(G,v) },
/*02*/  { INS_add,    op(G,b),   op(E,b) },
/*03*/  { INS_add,    op(G,v),   op(E,v) },
/*04*/  { INS_add,    reg(AX,b), op(I,b) },
/*05*/  { INS_add,    reg(AX,v), op(I,v) },
/*06*/  { INS_push,   seg(ES) },
/*07*/  { INS_pop,    seg(ES) },

/*08*/  { INS_or,     op(E,b),   op(G,b) },
/*09*/  { INS_or,     op(E,v),   op(G,v) },
/*0a*/  { INS_or,     op(G,b),   op(E,b) },
/*0b*/  { INS_or,     op(G,v),   op(E,v) },
/*0c*/  { INS_or,     reg(AX,b), op(I,b) },
/*0d*/  { INS_or,     reg(AX,v), op(I,v) },
/*0e*/  { INS_push,   seg(CS) },
/*0f*/  { INS_TwoBytePrefix },

/*10*/  { INS_adc,    op(E,b),   op(G,b) },
/*11*/  { INS_adc,    op(E,v),   op(G,v) },
/*12*/  { INS_adc,    op(G,b),   op(E,b) },
/*13*/  { INS_adc,    op(G,v),   op(E,v) },
/*14*/  { INS_adc,    reg(AX,b), op(I,b) },
/*15*/  { INS_adc,    reg(AX,v), op(I,v) },
/*16*/  { INS_push,   seg(SS) },
/*17*/  { INS_pop,    seg(SS) },

/*18*/  { INS_sbb,    op(E,b),   op(G,b) },
/*19*/  { INS_sbb,    op(E,v),   op(G,v) },
/*1a*/  { INS_sbb,    op(G,b),   op(E,b) },
/*1b*/  { INS_sbb,    op(G,v),   op(E,v) },
/*1c*/  { INS_sbb,    reg(AX,b), op(I,b) },
/*1d*/  { INS_sbb,    reg(AX,v), op(I,v) },
/*1e*/  { INS_push,   seg(DS) },
/*1f*/  { INS_pop,    seg(DS) },

/*20*/  { INS_and,    op(E,b),   op(G,b) },
/*21*/  { INS_and,    op(E,v),   op(G,v) },
/*22*/  { INS_and,    op(G,b),   op(E,b) },
/*23*/  { INS_and,    op(G,v),   op(E,v) },
/*24*/  { INS_and,    reg(AX,b), op(I,b) },
/*25*/  { INS_and,    reg(AX,v), op(I,v) },
/*26*/  { INS_SegPrefix, SEG_ES },
/*27*/  { INS_daa },

/*28*/  { INS_sub,    op(E,b),   op(G,b) },
/*29*/  { INS_sub,    op(E,v),   op(G,v) },
/*2a*/  { INS_sub,    op(G,b),   op(E,b) },
/*2b*/  { INS_sub,    op(G,v),   op(E,v) },
/*2c*/  { INS_sub,    reg(AX,b), op(I,b) },
/*2d*/  { INS_sub,    reg(AX,v), op(I,v) },
/*2e*/  { INS_SegPrefix, SEG_CS },
/*2f*/  { INS_das },

/*30*/  { INS_xor,    op(E,b),   op(G,b) },
/*31*/  { INS_xor,    op(E,v),   op(G,v) },
/*32*/  { INS_xor,    op(G,b),   op(E,b) },
/*33*/  { INS_xor,    op(G,v),   op(E,v) },
/*34*/  { INS_xor,    reg(AX,b), op(I,b) },
/*35*/  { INS_xor,    reg(AX,v), op(I,v) },
/*36*/  { INS_SegPrefix, SEG_SS },
/*37*/  { INS_aaa },

/*38*/  { INS_cmp,    op(E,b),   op(G,b) },
/*39*/  { INS_cmp,    op(E,v),   op(G,v) },
/*3a*/  { INS_cmp,    op(G,b),   op(E,b) },
/*3b*/  { INS_cmp,    op(G,v),   op(E,v) },
/*3c*/  { INS_cmp,    reg(AX,b), op(I,b) },
/*3d*/  { INS_cmp,    reg(AX,v), op(I,v) },
/*3e*/  { INS_SegPrefix, SEG_DS },
/*3f*/  { INS_aas },

/*40*/  { INS_inc,    reg(AX,v) },
/*41*/  { INS_inc,    reg(CX,v) },
/*42*/  { INS_inc,    reg(DX,v) },
/*43*/  { INS_inc,    reg(BX,v) },
/*44*/  { INS_inc,    reg(SP,v) },
/*45*/  { INS_inc,    reg(BP,v) },
/*46*/  { INS_inc,    reg(SI,v) },
/*47*/  { INS_inc,    reg(DI,v) },

/*48*/  { INS_dec,    reg(AX,v) },
/*49*/  { INS_dec,    reg(CX,v) },
/*4a*/  { INS_dec,    reg(DX,v) },
/*4b*/  { INS_dec,    reg(BX,v) },
/*4c*/  { INS_dec,    reg(SP,v) },
/*4d*/  { INS_dec,    reg(BP,v) },
/*4e*/  { INS_dec,    reg(SI,v) },
/*4f*/  { INS_dec,    reg(DI,v) },

/*50*/  { INS_push,   reg(AX,v) },
/*51*/  { INS_push,   reg(CX,v) },
/*52*/  { INS_push,   reg(DX,v) },
/*53*/  { INS_push,   reg(BX,v) },
/*54*/  { INS_push,   reg(SP,v) },
/*55*/  { INS_push,   reg(BP,v) },
/*56*/  { INS_push,   reg(SI,v) },
/*57*/  { INS_push,   reg(DI,v) },

/*58*/  { INS_pop,    reg(AX,v) },
/*59*/  { INS_pop,    reg(CX,v) },
/*5a*/  { INS_pop,    reg(DX,v) },
/*5b*/  { INS_pop,    reg(BX,v) },
/*5c*/  { INS_pop,    reg(SP,v) },
/*5d*/  { INS_pop,    reg(BP,v) },
/*5e*/  { INS_pop,    reg(SI,v) },
/*5f*/  { INS_pop,    reg(DI,v) },

/*60*/  { INS_pushav },
/*61*/  { INS_popav },
/*62*/  { INS_bound,  op(G,v),   op(M,a) },
/*63*/  { INS_arpl,   op(E,w),   op(R,w) },
/*64*/  { INS_SegPrefix, SEG_FS },
/*65*/  { INS_SegPrefix, SEG_GS },
/*66*/  { INS_OpSizePrefix },
/*67*/  { INS_AddrSizePrefix },

/*68*/  { INS_push,   op(I,v) },
/*69*/  { INS_imul,   op(G,v),   op(E,v),   op(I,v) },
/*6a*/  { INS_push,   op(I,c) },
/*6b*/  { INS_imul,   op(G,v),   op(E,v),   op(I,c) },
/*6c*/  { INS_ins,    op(Y,b),   reg(DX,w) },
/*6d*/  { INS_ins,    op(Y,v),   reg(DX,w) },
/*6e*/  { INS_outs,   reg(DX,w), op(X,b) },
/*6f*/  { INS_outs,   reg(DX,w), op(X,v) },

/*70*/  { INS_jo,     op(J,b) },
/*71*/  { INS_jno,    op(J,b) },
/*72*/  { INS_jb,     op(J,b) },
/*73*/  { INS_jae,    op(J,b) },
/*74*/  { INS_je,     op(J,b) },
/*75*/  { INS_jne,    op(J,b) },
/*76*/  { INS_jbe,    op(J,b) },
/*77*/  { INS_ja,     op(J,b) },

/*78*/  { INS_js,     op(J,b) },
/*79*/  { INS_jns,    op(J,b) },
/*7a*/  { INS_jp,     op(J,b) },
/*7b*/  { INS_jnp,    op(J,b) },
/*7c*/  { INS_jl,     op(J,b) },
/*7d*/  { INS_jge,    op(J,b) },
/*7e*/  { INS_jle,    op(J,b) },
/*7f*/  { INS_jg,     op(J,b) },

/*80*/  { INS_Grp1,   op(E,b),   op(I,b) },
/*81*/  { INS_Grp1,   op(E,v),   op(I,v) },
/*82*/  { INS_Grp1,   op(E,b),   op(I,b) },
/*83*/  { INS_Grp1,   op(E,v),   op(I,c) },
/*84*/  { INS_test,   op(E,b),   op(G,b) },
/*85*/  { INS_test,   op(E,v),   op(G,v) },
/*86*/  { INS_xchg,   op(E,b),   op(G,b) },
/*87*/  { INS_xchg,   op(E,v),   op(G,v) },

/*88*/  { INS_mov,    op(E,b),   op(G,b) },
/*89*/  { INS_mov,    op(E,v),   op(G,v) },
/*8a*/  { INS_mov,    op(G,b),   op(E,b) },
/*8b*/  { INS_mov,    op(G,v),   op(E,v) },
/*8c*/  { INS_mov,    op(E,w),   op(S,w) },
/*8d*/  { INS_lea,    op(G,v),   op(M,0) },
/*8e*/  { INS_mov,    op(S,w),   op(E,w) },
/*8f*/  { INS_pop,    op(E,v) },

/*90*/  { INS_nop },
/*91*/  { INS_xchg,   reg(AX,v), reg(CX,v) },
/*92*/  { INS_xchg,   reg(AX,v), reg(DX,v) },
/*93*/  { INS_xchg,   reg(AX,v), reg(BX,v) },
/*94*/  { INS_xchg,   reg(AX,v), reg(SP,v) },
/*95*/  { INS_xchg,   reg(AX,v), reg(BP,v) },
/*96*/  { INS_xchg,   reg(AX,v), reg(SI,v) },
/*97*/  { INS_xchg,   reg(AX,v), reg(DI,v) },

/*98*/  { INS_cbw_cwde },
/*99*/  { INS_cwd_cdq },
/*9a*/  { INS_call,   op(A,p) },
/*9b*/  { INS_fwait },
/*9c*/  { INS_pushfv },
/*9d*/  { INS_popfv },
/*9e*/  { INS_sahf, },
/*9f*/  { INS_lahf, },

/*a0*/  { INS_mov,    reg(AX,b), op(O,b) },
/*a1*/  { INS_mov,    reg(AX,v), op(O,v) },
/*a2*/  { INS_mov,    op(O,b),   reg(AX,b) },
/*a3*/  { INS_mov,    op(O,v),   reg(AX,v) },
/*a4*/  { INS_movs,   op(Y,b),   op(X,b) },
/*a5*/  { INS_movs,   op(Y,v),   op(X,v) },
/*a6*/  { INS_cmps,   op(Y,b),   op(X,b) },
/*a7*/  { INS_cmps,   op(Y,v),   op(X,v) },

/*a8*/  { INS_test,   reg(AX,b), op(I,b) },
/*a9*/  { INS_test,   reg(AX,v), op(I,v) },
/*aa*/  { INS_stos,   op(Y,b),   reg(AX,b) },
/*ab*/  { INS_stos,   op(Y,v),   reg(AX,v) },
/*ac*/  { INS_lods,   reg(AX,b), op(X,b) },
/*ad*/  { INS_lods,   reg(AX,v), op(X,v) },
/*ae*/  { INS_scas,   op(Y,b),   reg(AX,b) },
/*af*/  { INS_scas,   op(Y,v),   reg(AX,v) },

/*b0*/  { INS_mov,    reg(AX,b), op(I,b) },
/*b1*/  { INS_mov,    reg(CX,b), op(I,b) },
/*b2*/  { INS_mov,    reg(DX,b), op(I,b) },
/*b3*/  { INS_mov,    reg(BX,b), op(I,b) },
/*b4*/  { INS_mov,    reg(SP,b), op(I,b) },
/*b5*/  { INS_mov,    reg(BP,b), op(I,b) },
/*b6*/  { INS_mov,    reg(SI,b), op(I,b) },
/*b7*/  { INS_mov,    reg(DI,b), op(I,b) },

/*b8*/  { INS_mov,    reg(AX,v), op(I,v) },
/*b9*/  { INS_mov,    reg(CX,v), op(I,v) },
/*ba*/  { INS_mov,    reg(DX,v), op(I,v) },
/*bb*/  { INS_mov,    reg(BX,v), op(I,v) },
/*bc*/  { INS_mov,    reg(SP,v), op(I,v) },
/*bd*/  { INS_mov,    reg(BP,v), op(I,v) },
/*be*/  { INS_mov,    reg(SI,v), op(I,v) },
/*bf*/  { INS_mov,    reg(DI,v), op(I,v) },

/*c0*/  { INS_Grp2,   op(E,b),   op(I,b) },
/*c1*/  { INS_Grp2,   op(E,v),   op(I,b) },
/*c2*/  { INS_ret,    op(I,w) },
/*c3*/  { INS_ret },
/*c4*/  { INS_les,    op(G,v),   op(M,p) },
/*c5*/  { INS_lds,    op(G,v),   op(M,p) },
/*c6*/  { INS_mov,    op(E,b),   op(I,b) },
/*c7*/  { INS_mov,    op(E,v),   op(I,v) },

/*c8*/  { INS_enter,  op(I,w),   op(I,b) },
/*c9*/  { INS_leave },
/*ca*/  { INS_retf,   op(I,w) },
/*cb*/  { INS_retf },
/*cc*/  { INS_int,    imm(3) },
/*cd*/  { INS_int,    op(I,b) },
/*ce*/  { INS_into },
/*cf*/  { INS_iret },

/*d0*/  { INS_Grp2,   op(E,b),   imm(1) },
/*d1*/  { INS_Grp2,   op(E,v),   imm(1) },
/*d2*/  { INS_Grp2,   op(E,b),   reg(CX,b) },
/*d3*/  { INS_Grp2,   op(E,v),   reg(CX,b) },
/*d4*/  { INS_aam,    op(I,b) },
/*d5*/  { INS_aad,    op(I,b) },
/*d6*/  { 0 },  /* salc */
/*d7*/  { INS_xlat },

/*d8*/  { INS_Esc1 },
/*d9*/  { INS_Esc2 },
/*da*/  { INS_Esc3 },
/*db*/  { INS_Esc4 },
/*dc*/  { INS_Esc5 },
/*dd*/  { INS_Esc6 },
/*de*/  { INS_Esc7 },
/*df*/  { INS_Esc8 },

/*e0*/  { INS_loopne, op(J,b) },
/*e1*/  { INS_loope,  op(J,b) },
/*e2*/  { INS_loop,   op(J,b) },
/*e3*/  { INS_jEcxz,  op(J,b) },
/*e4*/  { INS_in,     reg(AX,b), op(I,b) },
/*e5*/  { INS_in,     reg(AX,v), op(I,b) },
/*e6*/  { INS_out,    op(I,b),   reg(AX,b) },
/*e7*/  { INS_out,    op(I,b),   reg(AX,v) },

/*e8*/  { INS_call,   op(J,v) },
/*e9*/  { INS_jmp,    op(J,v) },
/*ea*/  { INS_jmp,    op(A,p) },
/*eb*/  { INS_jmp,    op(J,b) },
/*ec*/  { INS_in,     reg(AX,b), reg(DX,w) },
/*ed*/  { INS_in,     reg(AX,v), reg(DX,w) },
/*ee*/  { INS_out,    reg(DX,w), reg(AX,b) },
/*ef*/  { INS_out,    reg(DX,w), reg(AX,v) },

/*f0*/  { INS_InsPrefix, INS_lock },
/*f1*/  { 0 },   /* icebp */
/*f2*/  { INS_InsPrefix, INS_repne },
/*f3*/  { INS_InsPrefix, INS_rep },
/*f4*/  { INS_hlt },
/*f5*/  { INS_cmc },
/*f6*/  { INS_Grp3,   op(0,b),    op(0,b) },
/*f7*/  { INS_Grp3,   op(0,v),    op(0,v) },

/*f8*/  { INS_clc },
/*f9*/  { INS_stc },
/*fa*/  { INS_cli },
/*fb*/  { INS_sti },
/*fc*/  { INS_cld },
/*fd*/  { INS_std },
/*fe*/  { INS_Grp4 },
/*ff*/  { INS_Grp5 }
};

struct instruction TwoByteMap[] =
{
/*00*/  { INS_Grp6 },
/*01*/  { INS_Grp7 },
/*02*/  { INS_lar,    op(G,v),   op(E,w) },
/*03*/  { INS_lsl,    op(G,v),   op(E,w) },
/*04*/  { 0 },  /* loadall ? */
/*05*/  { 0 },  /* loadall 286 */
/*06*/  { INS_clts },
/*07*/  { 0 },  /* loadall 386 */

/*08*/  { INS_invd },
/*09*/  { INS_wbinvd },
/*0a*/  { 0 },
/*0b*/  { INS_ud2 },
/*0c*/  { 0 },
/*0d*/  { 0 },
/*0e*/  { 0 },
/*0f*/  { 0 },

/*10*/  { INS_movups, op(V,o),   op(W,o) },  /* movss d */
/*11*/  { INS_movups, op(W,o),   op(V,o) },  /* movss d */
/*12*/  { INS_movlps, op(V,q),   op(W,q) },  /* movhlps */
/*13*/  { INS_movlps, op(W,q),   op(V,q) },
/*14*/  { INS_unpcklps,op(V,o),  op(W,q) },
/*15*/  { INS_unpckhps,op(V,o),  op(W,q) },
/*16*/  { INS_movhps, op(V,q),   op(W,q) },  /* movlhps */
/*17*/  { INS_movhps, op(W,q),   op(V,q) },

/*18*/  { INS_Grp17 },
/*19*/  { 0 },
/*1a*/  { 0 },
/*1b*/  { 0 },
/*1c*/  { 0 },
/*1d*/  { 0 },
/*1e*/  { 0 },
/*1f*/  { 0 },

/*20*/  { INS_mov,    op(R,d),   op(C,d) },
/*21*/  { INS_mov,    op(R,d),   op(D,d) },
/*22*/  { INS_mov,    op(C,d),   op(R,d) },
/*23*/  { INS_mov,    op(D,d),   op(R,d) },
/*24*/  { INS_mov,    op(R,d),   op(T,d) },
/*25*/  { 0 },
/*26*/  { INS_mov,    op(T,d),   op(R,d) },
/*27*/  { 0 },

/*28*/  { INS_movaps, op(V,o),   op(W,o) },
/*29*/  { INS_movaps, op(W,o),   op(V,o) },
/*2a*/  { INS_cvtpi2ps,op(V,o),  op(Q,q) }, /* cvtsi2ss d,q */
/*2b*/  { INS_movntps,op(W,o),   op(V,o) },
/*2c*/  { INS_cvttps2pi,op(Q,q), op(W,o) }, /* cvttss2si d */
/*2d*/  { INS_cvtps2pi,op(Q,q),  op(W,o) }, /* cvtss2si d */
/*2e*/  { INS_ucomiss,op(V,d),   op(W,d) },
/*2f*/  { INS_comiss, op(V,o),   op(W,o) },

/*30*/  { INS_wrmsr },
/*31*/  { INS_rdtsc },
/*32*/  { INS_rdmsr },
/*33*/  { INS_rdpmc },
/*34*/  { INS_sysenter },
/*35*/  { INS_sysexit },
/*36*/  { 0 },
/*37*/  { 0 },

/*38*/  { 0 },
/*39*/  { 0 },
/*3a*/  { 0 },
/*3b*/  { 0 },
/*3c*/  { 0 },
/*3d*/  { 0 },
/*3e*/  { 0 },
/*3f*/  { 0 },

/*40*/  { INS_cmovo,  op(G,v),   op(E,v) },
/*41*/  { INS_cmovno, op(G,v),   op(E,v) },
/*42*/  { INS_cmovb,  op(G,v),   op(E,v) },
/*43*/  { INS_cmovae, op(G,v),   op(E,v) },
/*44*/  { INS_cmove,  op(G,v),   op(E,v) },
/*45*/  { INS_cmovne, op(G,v),   op(E,v) },
/*46*/  { INS_cmovbe, op(G,v),   op(E,v) },
/*47*/  { INS_cmova,  op(G,v),   op(E,v) },

/*48*/  { INS_cmovs,  op(G,v),   op(E,v) },
/*49*/  { INS_cmovns, op(G,v),   op(E,v) },
/*4a*/  { INS_cmovp,  op(G,v),   op(E,v) },
/*4b*/  { INS_cmovnp, op(G,v),   op(E,v) },
/*4c*/  { INS_cmovl,  op(G,v),   op(E,v) },
/*4d*/  { INS_cmovge, op(G,v),   op(E,v) },
/*4e*/  { INS_cmovle, op(G,v),   op(E,v) },
/*4f*/  { INS_cmovg,  op(G,v),   op(E,v) },

/*50*/  { INS_movmskps,op(E,d),  op(V,o) },
/*51*/  { INS_sqrtps, op(V,o),   op(W,o) }, /* sqrtss d */
/*52*/  { INS_rsqrtps,op(V,o),   op(W,o) }, /* rsqrtss d */
/*53*/  { INS_rcpps,  op(V,o),   op(W,o) }, /* rcpss d */
/*54*/  { INS_andps,  op(V,o),   op(W,o) },
/*55*/  { INS_andnps, op(V,o),   op(W,o) },
/*56*/  { INS_orps,   op(V,o),   op(W,o) },
/*57*/  { INS_xorps,  op(V,o),   op(W,o) },

/*58*/  { INS_addps,  op(V,o),   op(W,o) }, /* addss d */
/*59*/  { INS_mulps,  op(V,o),   op(W,o) }, /* mulss d */
/*5a*/  { 0 },
/*5b*/  { 0 },
/*5c*/  { INS_subps,  op(V,o),   op(W,o) }, /* subss d */
/*5d*/  { INS_minps,  op(V,o),   op(W,o) }, /* minss d */
/*5e*/  { INS_divps,  op(V,o),   op(W,o) }, /* divps d */
/*5f*/  { INS_maxps,  op(V,o),   op(W,o) }, /* maxss d */

/*60*/  { INS_punpcklbw, op(P,q),  op(Q,d) },
/*61*/  { INS_punpcklwd, op(P,q),  op(Q,d) },
/*62*/  { INS_punpckldq, op(P,q),  op(Q,d) },
/*63*/  { INS_packsswb,  op(P,q),  op(Q,q) },
/*64*/  { INS_pcmpgtb,   op(P,q),  op(Q,q) },
/*65*/  { INS_pcmpgtw,   op(P,q),  op(Q,q) },
/*66*/  { INS_pcmpgtd,   op(P,q),  op(Q,q) },
/*67*/  { INS_packuswb,  op(P,q),  op(Q,q) },

/*68*/  { INS_punpckhbw, op(P,q),  op(Q,q) },
/*69*/  { INS_punpckhwd, op(P,q),  op(Q,q) },
/*6a*/  { INS_punpckhdq, op(P,q),  op(Q,q) },
/*6b*/  { INS_packssdw,  op(P,q),  op(Q,q) },
/*6c*/  { 0 },
/*6d*/  { 0 },
/*6e*/  { INS_movd,      op(P,d),  op(E,d) },
/*6f*/  { INS_movq,      op(P,q),  op(Q,q) },

/*70*/  { 0 },
/*71*/  { INS_Grp13 },
/*72*/  { INS_Grp14 },
/*73*/  { INS_Grp15 },
/*74*/  { INS_pcmpeqb,  op(P,q),  op(Q,q) },
/*75*/  { INS_pcmpeqw,  op(P,q),  op(Q,q) },
/*76*/  { INS_pcmpeqd,  op(P,q),  op(Q,q) },
/*77*/  { INS_emms },

/*78*/  { 0 },
/*79*/  { 0 },
/*7a*/  { 0 },
/*7b*/  { 0 },
/*7c*/  { 0 },
/*7d*/  { 0 },
/*7e*/  { INS_movd,     op(E,d),  op(P,d) },
/*7f*/  { INS_movq,     op(Q,q),  op(P,q) },

/*80*/  { INS_jo,     op(J,v) },
/*81*/  { INS_jno,    op(J,v) },
/*82*/  { INS_jb,     op(J,v) },
/*83*/  { INS_jae,    op(J,v) },
/*84*/  { INS_je,     op(J,v) },
/*85*/  { INS_jne,    op(J,v) },
/*86*/  { INS_jbe,    op(J,v) },
/*87*/  { INS_ja,     op(J,v) },

/*88*/  { INS_js,     op(J,v) },
/*89*/  { INS_jns,    op(J,v) },
/*8a*/  { INS_jp,     op(J,v) },
/*8b*/  { INS_jnp,    op(J,v) },
/*8c*/  { INS_jl,     op(J,v) },
/*8d*/  { INS_jge,    op(J,v) },
/*8e*/  { INS_jle,    op(J,v) },
/*8f*/  { INS_jg,     op(J,v) },

/*90*/  { INS_seto,   op(E,b) },
/*91*/  { INS_setno,  op(E,b) },
/*92*/  { INS_setb,   op(E,b) },
/*93*/  { INS_setae,  op(E,b) },
/*94*/  { INS_sete,   op(E,b) },
/*95*/  { INS_setne,  op(E,b) },
/*96*/  { INS_setbe,  op(E,b) },
/*97*/  { INS_seta,   op(E,b) },

/*98*/  { INS_sets,   op(E,b) },
/*99*/  { INS_setns,  op(E,b) },
/*9a*/  { INS_setp,   op(E,b) },
/*9b*/  { INS_setnp,  op(E,b) },
/*9c*/  { INS_setl,   op(E,b) },
/*9d*/  { INS_setge,  op(E,b) },
/*9e*/  { INS_setle,  op(E,b) },
/*9f*/  { INS_setg,   op(E,b) },

/*a0*/  { INS_push,   seg(FS) },
/*a1*/  { INS_pop,    seg(FS) },
/*a2*/  { INS_cpuid },
/*a3*/  { INS_bt,     op(E,v),   op(G,v) },
/*a4*/  { INS_shld,   op(E,v),   op(G,v),   op(I,b) },
/*a5*/  { INS_shld,   op(E,v),   op(G,v),   reg(CX,b) },
/*a6*/  { 0 },
/*a7*/  { 0 },

/*a8*/  { INS_push,   seg(GS) },
/*a9*/  { INS_pop,    seg(GS) },
/*aa*/  { INS_rsm },
/*ab*/  { INS_bts,    op(E,v),   op(G,v) },
/*ac*/  { INS_shrd,   op(E,v),   op(G,v),  op(I,b) },
/*ad*/  { INS_shrd,   op(E,v),   op(G,v),  reg(CX,b) },
/*ae*/  { INS_Grp16 },
/*af*/  { INS_imul,   op(G,v),   op(E,v) },

/*b0*/  { INS_cmpxchg,op(E,b),   op(G,b) },
/*b1*/  { INS_cmpxchg,op(E,v),   op(G,v) },
/*b2*/  { INS_lss,    op(G,v),   op(M,p) },
/*b3*/  { INS_btr,    op(E,v),   op(G,v) },
/*b4*/  { INS_lfs,    op(G,v),   op(M,p) },
/*b5*/  { INS_lgs,    op(G,v),   op(M,p) },
/*b6*/  { INS_movzx,  op(G,v),   op(E,b) },
/*b7*/  { INS_movzx,  op(G,v),   op(E,w) },

/*b8*/  { 0 },
/*b9*/  { INS_ud2 },  /* Grp11 */
/*ba*/  { INS_Grp8,   op(E,v),   op(I,b) },
/*bb*/  { INS_btc,    op(E,v),   op(G,v) },
/*bc*/  { INS_bsf,    op(G,v),   op(E,v) },
/*bd*/  { INS_bsr,    op(G,v),   op(E,v) },
/*be*/  { INS_movsx,  op(G,v),   op(E,b) },
/*bf*/  { INS_movsx,  op(G,v),   op(E,w) },

/*c0*/  { INS_xadd,   op(E,b),   op(G,b) },
/*c1*/  { INS_xadd,   op(E,v),   op(G,v) },
/*c2*/  { INS_cmpps,  op(V,o),   op(W,o),  op(I,b) },  /* cmpss */
/*c3*/  { 0 },
/*c4*/  { INS_pinsrw, op(P,q),   op(E,d),  op(I,b) },
/*c5*/  { INS_pextrw, op(G,d),   op(P,q),  op(I,b) },
/*c6*/  { INS_shufps, op(V,o),   op(W,o),  op(I,b) },
/*c7*/  { INS_Grp9 },

/*c8*/  { INS_bswap,  reg(AX,d) },
/*c9*/  { INS_bswap,  reg(CX,d) },
/*ca*/  { INS_bswap,  reg(DX,d) },
/*cb*/  { INS_bswap,  reg(BX,d) },
/*cc*/  { INS_bswap,  reg(SP,d) },
/*cd*/  { INS_bswap,  reg(BP,d) },
/*ce*/  { INS_bswap,  reg(SI,d) },
/*cf*/  { INS_bswap,  reg(DI,d) },

/*d0*/  { 0 },
/*d1*/  { INS_psrlw,  op(P,q),  op(Q,q) },
/*d2*/  { INS_psrld,  op(P,q),  op(Q,q) },
/*d3*/  { INS_psrlq,  op(P,q),  op(Q,q) },
/*d4*/  { 0 },
/*d5*/  { INS_pmullw, op(P,q),  op(Q,q) },
/*d6*/  { 0 },
/*d7*/  { INS_pmovmskb,op(G,d), op(P,q) },

/*d8*/  { INS_psubusb,op(P,q),  op(Q,q) },
/*d9*/  { INS_psubusw,op(P,q),  op(Q,q) },
/*da*/  { INS_pminub, op(P,q),  op(Q,q) },
/*db*/  { INS_pand,   op(P,q),  op(Q,q) },
/*dc*/  { INS_paddusb,op(P,q),  op(Q,q) },
/*dd*/  { INS_paddusw,op(P,q),  op(Q,q) },
/*de*/  { INS_pmaxub, op(P,q),  op(Q,q) },
/*df*/  { INS_pandn,  op(P,q),  op(Q,q) },

/*e0*/  { INS_pavgb,  op(P,q),  op(Q,q) },
/*e1*/  { INS_psraw,  op(P,q),  op(Q,q) },
/*e2*/  { INS_psrad,  op(P,q),  op(Q,q) },
/*e3*/  { INS_pavgw,  op(P,q),  op(Q,q) },
/*e4*/  { INS_pmulhuw,op(P,q),  op(Q,q) },
/*e5*/  { INS_pmulhw, op(P,q),  op(Q,q) },
/*e6*/  { 0 },
/*e7*/  { INS_movntq, op(W,q),  op(V,q) },

/*e8*/  { INS_psubsb, op(P,q),  op(Q,q) },
/*e9*/  { INS_psubsw, op(P,q),  op(Q,q) },
/*ea*/  { INS_pminsw, op(P,q),  op(Q,q) },
/*eb*/  { INS_por,    op(P,q),  op(Q,q) },
/*ec*/  { INS_paddsb, op(P,q),  op(Q,q) },
/*ed*/  { INS_paddsw, op(P,q),  op(Q,q) },
/*ee*/  { INS_pmaxsw, op(P,q),  op(Q,q) },
/*ef*/  { INS_pxor,   op(P,q),  op(Q,q) },

/*f0*/  { 0 },
/*f1*/  { INS_psllw,  op(P,q),  op(Q,q) },
/*f2*/  { INS_pslld,  op(P,q),  op(Q,q) },
/*f3*/  { INS_psllq,  op(P,q),  op(Q,q) },
/*f4*/  { 0 },
/*f5*/  { INS_pmaddwd,op(P,q),  op(Q,q) },
/*f6*/  { INS_psadbw, op(P,q),  op(Q,q) },
/*f7*/  { 0 },

/*f8*/  { INS_psubb,  op(P,q),  op(Q,q) },
/*f9*/  { INS_psubw,  op(P,q),  op(Q,q) },
/*fa*/  { INS_psubd,  op(P,q),  op(Q,q) },
/*fb*/  { 0 },
/*fc*/  { INS_paddb,  op(P,q),  op(Q,q) },
/*fd*/  { INS_paddw,  op(P,q),  op(Q,q) },
/*fe*/  { INS_paddd,  op(P,q),  op(Q,q) },
/*ff*/  { INS_ud2 }
};

struct instruction GroupMap[] =
{
/* Group 1: */
/*0*/   { INS_add },
/*1*/   { INS_or },
/*2*/   { INS_adc },
/*3*/   { INS_sbb },
/*4*/   { INS_and },
/*5*/   { INS_sub },
/*6*/   { INS_xor },
/*7*/   { INS_cmp },

/* Group 2: */
/*0*/   { INS_rol },
/*1*/   { INS_ror },
/*2*/   { INS_rcl },
/*3*/   { INS_rcr },
/*4*/   { INS_shl },
/*5*/   { INS_shr },
/*6*/   { INS_sal },
/*7*/   { INS_sar },

/* Group 3: */
/*0*/   { INS_test,   op(E,0),   op(I,0) },
/*1*/   { INS_test,   op(E,0),   op(I,0) },
/*2*/   { INS_not,    op(E,0) },
/*3*/   { INS_neg,    op(E,0) },
/*4*/   { INS_mul,    op(E,0) },
/*5*/   { INS_imul,   op(E,0) },
/*6*/   { INS_div,    op(E,0) },
/*7*/   { INS_idiv,   op(E,0) },

/* Group 4: */
/*0*/   { INS_inc,    op(E,b) },
/*1*/   { INS_dec,    op(E,b) },
/*2*/   { 0 },
/*3*/   { 0 },
/*4*/   { 0 },
/*5*/   { 0 },
/*6*/   { 0 },
/*7*/   { 0 },

/* Group 5: */
/*0*/   { INS_inc,    op(E,v) },
/*1*/   { INS_dec,    op(E,v) },
/*2*/   { INS_call,   op(E,v) },
/*3*/   { INS_call,   op(E,p) },
/*4*/   { INS_jmp,    op(E,v) },
/*5*/   { INS_jmp,    op(E,p) },
/*6*/   { INS_push,   op(E,v) },
/*7*/   { 0 },

/* Group 6: */
/*0*/   { INS_sldt,   op(E,w) },
/*1*/   { INS_str,    op(E,w) },
/*2*/   { INS_lldt,   op(E,w) },
/*3*/   { INS_ltr,    op(E,w) },
/*4*/   { INS_verr,   op(E,w) },
/*5*/   { INS_verw,   op(E,w) },
/*6*/   { 0 },
/*7*/   { 0 },

/* Group 7: */
/*0*/   { INS_sgdt,   op(M,s) },
/*1*/   { INS_sidt,   op(M,s) },
/*2*/   { INS_lgdt,   op(M,s) },
/*3*/   { INS_lidt,   op(M,s) },
/*4*/   { INS_smsw,   op(E,w) },
/*5*/   { 0 },
/*6*/   { INS_lmsw,   op(E,w) },
/*7*/   { INS_invlpg, op(M,0) },

/* Group 8: */
/*0*/   { 0 },
/*1*/   { 0 },
/*2*/   { 0 },
/*3*/   { 0 },
/*4*/   { INS_bt },
/*5*/   { INS_bts },
/*6*/   { INS_btr },
/*7*/   { INS_btc },

/* Group 9: */
/*0*/   { 0 },
/*1*/   { INS_cmpxchg8b, op(M,q) },
/*2*/   { 0 },
/*3*/   { 0 },
/*4*/   { 0 },
/*5*/   { 0 },
/*6*/   { 0 },
/*7*/   { 0 },

/* Group 13: */
/*0*/   { 0 },
/*1*/   { 0 },
/*2*/   { INS_psrlw,  op(P,q),  op(I,b) },
/*3*/   { 0 },
/*4*/   { INS_psraw,  op(P,q),  op(I,b) },
/*5*/   { 0 },
/*6*/   { INS_psllw,  op(P,q),  op(I,b) },
/*7*/   { 0 },

/* Group 14: */
/*0*/   { 0 },
/*1*/   { 0 },
/*2*/   { INS_psrld,  op(P,q),  op(I,b) },
/*3*/   { 0 },
/*4*/   { INS_psrad,  op(P,q),  op(I,b) },
/*5*/   { 0 },
/*6*/   { INS_pslld,  op(P,q),  op(I,b) },
/*7*/   { 0 },

/* Group 15: */
/*0*/   { 0 },
/*1*/   { 0 },
/*2*/   { INS_psrlq,  op(P,q),  op(I,b) },
/*3*/   { 0 },
/*4*/   { 0 },
/*5*/   { 0 },
/*6*/   { INS_psllq,  op(P,q),  op(I,b) },
/*7*/   { 0 },

/* Group 16: */
/*0*/   { INS_fxsave,  op(M,0) },
/*1*/   { INS_fxrstor, op(M,0) },
/*2*/   { INS_ldmxcsr, op(M,d) },
/*3*/   { INS_stmxcsr, op(M,d) },
/*4*/   { 0 },
/*5*/   { 0 },
/*6*/   { 0 },
/*7*/   { INS_sfence },

/* Group 17: */
/*0*/   { INS_prefetchnta, op(M,0) },
/*1*/   { INS_prefetcht0, op(M,0) },
/*2*/   { INS_prefetcht1, op(M,0) },
/*3*/   { INS_prefetcht2, op(M,0) },
/*4*/   { 0 },
/*5*/   { 0 },
/*6*/   { 0 },
/*7*/   { 0 }
};

struct instruction EscMainMap[] =
{
/* D8 ESC Opcode: */
/*0*/   { INS_fadd,    op(M,x) },
/*1*/   { INS_fmul,    op(M,x) },
/*2*/   { INS_fcom,    op(M,x) },
/*3*/   { INS_fcomp,   op(M,x) },
/*4*/   { INS_fsub,    op(M,x) },
/*5*/   { INS_fsubr,   op(M,x) },
/*6*/   { INS_fdiv,    op(M,x) },
/*7*/   { INS_fdivr,   op(M,x) },

/* D9 ESC Opcode: */
/*0*/   { INS_fld,     op(M,x) },
/*1*/   { 0 },
/*2*/   { INS_fst,     op(M,x) },
/*3*/   { INS_fstp,    op(M,x) },
/*4*/   { INS_fldenv,  op(M,0) },
/*5*/   { INS_fldcw,   op(M,w) },
/*6*/   { INS_fstenv,  op(M,0) },
/*7*/   { INS_fstcw,   op(M,w) },

/* DA ESC Opcode: */
/*0*/   { INS_fiadd,   op(M,d) },
/*1*/   { INS_fimul,   op(M,d) },
/*2*/   { INS_ficom,   op(M,d) },
/*3*/   { INS_ficomp,  op(M,d) },
/*4*/   { INS_fisub,   op(M,d) },
/*5*/   { INS_fisubr,  op(M,d) },
/*6*/   { INS_fidiv,   op(M,d) },
/*7*/   { INS_fidivr,  op(M,d) },

/* DB ESC Opcode: */
/*0*/   { INS_fild,    op(M,d) },
/*1*/   { 0 },
/*2*/   { INS_fist,    op(M,d) },
/*3*/   { INS_fistp,   op(M,d) },
/*4*/   { 0 },
/*5*/   { INS_fld,     op(M,z) },
/*6*/   { 0 },
/*7*/   { INS_fstp,    op(M,z) },

/* DC ESC Opcode: */
/*0*/   { INS_fadd,    op(M,y) },
/*1*/   { INS_fmul,    op(M,y) },
/*2*/   { INS_fcom,    op(M,y) },
/*3*/   { INS_fcomp,   op(M,y) },
/*4*/   { INS_fsub,    op(M,y) },
/*5*/   { INS_fsubr,   op(M,y) },
/*6*/   { INS_fdiv,    op(M,y) },
/*7*/   { INS_fdivr,   op(M,y) },

/* DD ESC Opcode: */
/*0*/   { INS_fld,     op(M,y) },
/*1*/   { 0 },
/*2*/   { INS_fst,     op(M,y) },
/*3*/   { INS_fstp,    op(M,y) },
/*4*/   { INS_frstor,  op(M,0) },
/*5*/   { 0 },
/*6*/   { INS_fsave,   op(M,0) },
/*7*/   { INS_fstsw,   op(M,w) },

/* DE ESC Opcode: */
/*0*/   { INS_fiadd,   op(M,w) },
/*1*/   { INS_fimul,   op(M,w) },
/*2*/   { INS_ficom,   op(M,w) },
/*3*/   { INS_ficomp,  op(M,w) },
/*4*/   { INS_fisub,   op(M,w) },
/*5*/   { INS_fisubr,  op(M,w) },
/*6*/   { INS_fidiv,   op(M,w) },
/*7*/   { INS_fidivr,  op(M,w) },

/* DF ESC Opcode: */
/*0*/   { INS_fild,    op(M,w) },
/*1*/   { 0 },
/*2*/   { INS_fist,    op(M,w) },
/*3*/   { INS_fistp,   op(M,w) },
/*4*/   { INS_fbld,    op(M,0) },
/*5*/   { INS_fild,    op(M,q) },
/*6*/   { INS_fbstp,   op(M,0) },
/*7*/   { INS_fistp,   op(M,q) }
};

struct instruction EscAuxMap[] =
{
/* D8 ESC Opcode: */
/*c0*/  { INS_fadd,    st(0), st(0) },
/*c1*/  { INS_fadd,    st(0), st(1) },
/*c2*/  { INS_fadd,    st(0), st(2) },
/*c3*/  { INS_fadd,    st(0), st(3) },
/*c4*/  { INS_fadd,    st(0), st(4) },
/*c5*/  { INS_fadd,    st(0), st(5) },
/*c6*/  { INS_fadd,    st(0), st(6) },
/*c7*/  { INS_fadd,    st(0), st(7) },

/*c8*/  { INS_fmul,    st(0), st(0) },
/*c9*/  { INS_fmul,    st(0), st(1) },
/*ca*/  { INS_fmul,    st(0), st(2) },
/*cb*/  { INS_fmul,    st(0), st(3) },
/*cc*/  { INS_fmul,    st(0), st(4) },
/*cd*/  { INS_fmul,    st(0), st(5) },
/*ce*/  { INS_fmul,    st(0), st(6) },
/*cf*/  { INS_fmul,    st(0), st(7) },

/*d0*/  { INS_fcom,    st(0), st(0) },
/*d1*/  { INS_fcom,    st(0), st(1) },
/*d2*/  { INS_fcom,    st(0), st(2) },
/*d3*/  { INS_fcom,    st(0), st(3) },
/*d4*/  { INS_fcom,    st(0), st(4) },
/*d5*/  { INS_fcom,    st(0), st(5) },
/*d6*/  { INS_fcom,    st(0), st(6) },
/*d7*/  { INS_fcom,    st(0), st(7) },

/*d8*/  { INS_fcomp,   st(0), st(0) },
/*d9*/  { INS_fcomp,   st(0), st(1) },
/*da*/  { INS_fcomp,   st(0), st(2) },
/*db*/  { INS_fcomp,   st(0), st(3) },
/*dc*/  { INS_fcomp,   st(0), st(4) },
/*dd*/  { INS_fcomp,   st(0), st(5) },
/*de*/  { INS_fcomp,   st(0), st(6) },
/*df*/  { INS_fcomp,   st(0), st(7) },

/*e0*/  { INS_fsub,    st(0), st(0) },
/*e1*/  { INS_fsub,    st(0), st(1) },
/*e2*/  { INS_fsub,    st(0), st(2) },
/*e3*/  { INS_fsub,    st(0), st(3) },
/*e4*/  { INS_fsub,    st(0), st(4) },
/*e5*/  { INS_fsub,    st(0), st(5) },
/*e6*/  { INS_fsub,    st(0), st(6) },
/*e7*/  { INS_fsub,    st(0), st(7) },

/*e8*/  { INS_fsubr,   st(0), st(0) },
/*e9*/  { INS_fsubr,   st(0), st(1) },
/*ea*/  { INS_fsubr,   st(0), st(2) },
/*eb*/  { INS_fsubr,   st(0), st(3) },
/*ec*/  { INS_fsubr,   st(0), st(4) },
/*ed*/  { INS_fsubr,   st(0), st(5) },
/*ee*/  { INS_fsubr,   st(0), st(6) },
/*ef*/  { INS_fsubr,   st(0), st(7) },

/*f0*/  { INS_fdiv,    st(0), st(0) },
/*f1*/  { INS_fdiv,    st(0), st(1) },
/*f2*/  { INS_fdiv,    st(0), st(2) },
/*f3*/  { INS_fdiv,    st(0), st(3) },
/*f4*/  { INS_fdiv,    st(0), st(4) },
/*f5*/  { INS_fdiv,    st(0), st(5) },
/*f6*/  { INS_fdiv,    st(0), st(6) },
/*f7*/  { INS_fdiv,    st(0), st(7) },

/*f8*/  { INS_fdivr,   st(0), st(0) },
/*f9*/  { INS_fdivr,   st(0), st(1) },
/*fa*/  { INS_fdivr,   st(0), st(2) },
/*fb*/  { INS_fdivr,   st(0), st(3) },
/*fc*/  { INS_fdivr,   st(0), st(4) },
/*fd*/  { INS_fdivr,   st(0), st(5) },
/*fe*/  { INS_fdivr,   st(0), st(6) },
/*ff*/  { INS_fdivr,   st(0), st(7) },

/* D9 ESC Opcode: */
/*c0*/  { INS_fld,     st(0), st(0) },
/*c1*/  { INS_fld,     st(0), st(1) },
/*c2*/  { INS_fld,     st(0), st(2) },
/*c3*/  { INS_fld,     st(0), st(3) },
/*c4*/  { INS_fld,     st(0), st(4) },
/*c5*/  { INS_fld,     st(0), st(5) },
/*c6*/  { INS_fld,     st(0), st(6) },
/*c7*/  { INS_fld,     st(0), st(7) },

/*c8*/  { INS_fxch,    st(0), st(0) },
/*c9*/  { INS_fxch,    st(0), st(1) },
/*ca*/  { INS_fxch,    st(0), st(2) },
/*cb*/  { INS_fxch,    st(0), st(3) },
/*cc*/  { INS_fxch,    st(0), st(4) },
/*cd*/  { INS_fxch,    st(0), st(5) },
/*ce*/  { INS_fxch,    st(0), st(6) },
/*cf*/  { INS_fxch,    st(0), st(7) },

/*d0*/  { INS_fnop },
/*d1*/  { 0 },
/*d2*/  { 0 },
/*d3*/  { 0 },
/*d4*/  { 0 },
/*d5*/  { 0 },
/*d6*/  { 0 },
/*d7*/  { 0 },

/*d8*/  { 0 },
/*d9*/  { 0 },
/*da*/  { 0 },
/*db*/  { 0 },
/*dc*/  { 0 },
/*dd*/  { 0 },
/*de*/  { 0 },
/*df*/  { 0 },

/*e0*/  { INS_fchs },
/*e1*/  { INS_fabs },
/*e2*/  { 0 },
/*e3*/  { 0 },
/*e4*/  { INS_ftst },
/*e5*/  { INS_fxam },
/*e6*/  { 0 },
/*e7*/  { 0 },

/*e8*/  { INS_fld1 },
/*e9*/  { INS_fldl2t },
/*ea*/  { INS_fldl2e },
/*eb*/  { INS_fldpi },
/*ec*/  { INS_fldlg2 },
/*ed*/  { INS_fldln2 },
/*ee*/  { INS_fldz },
/*ef*/  { 0 },

/*f0*/  { INS_f2xm1 },
/*f1*/  { INS_fyl2x },
/*f2*/  { INS_fptan },
/*f3*/  { INS_fpatan },
/*f4*/  { INS_fxtract },
/*f5*/  { INS_fprem1 },
/*f6*/  { INS_fdecstp },
/*f7*/  { INS_fincstp },

/*f8*/  { INS_fprem },
/*f9*/  { INS_fyl2xp1 },
/*fa*/  { INS_fsqrt },
/*fb*/  { INS_fsincos },
/*fc*/  { INS_frndint },
/*fd*/  { INS_fscale },
/*fe*/  { INS_fsin },
/*ff*/  { INS_fcos },

/* DA ESC Opcode: */
/*c0*/  { INS_fcmovb,  st(0), st(0) },
/*c1*/  { INS_fcmovb,  st(0), st(1) },
/*c2*/  { INS_fcmovb,  st(0), st(2) },
/*c3*/  { INS_fcmovb,  st(0), st(3) },
/*c4*/  { INS_fcmovb,  st(0), st(4) },
/*c5*/  { INS_fcmovb,  st(0), st(5) },
/*c6*/  { INS_fcmovb,  st(0), st(6) },
/*c7*/  { INS_fcmovb,  st(0), st(7) },

/*c8*/  { INS_fcmove,  st(0), st(0) },
/*c9*/  { INS_fcmove,  st(0), st(1) },
/*ca*/  { INS_fcmove,  st(0), st(2) },
/*cb*/  { INS_fcmove,  st(0), st(3) },
/*cc*/  { INS_fcmove,  st(0), st(4) },
/*cd*/  { INS_fcmove,  st(0), st(5) },
/*ce*/  { INS_fcmove,  st(0), st(6) },
/*cf*/  { INS_fcmove,  st(0), st(7) },

/*d0*/  { INS_fcmovbe, st(0), st(0) },
/*d1*/  { INS_fcmovbe, st(0), st(1) },
/*d2*/  { INS_fcmovbe, st(0), st(2) },
/*d3*/  { INS_fcmovbe, st(0), st(3) },
/*d4*/  { INS_fcmovbe, st(0), st(4) },
/*d5*/  { INS_fcmovbe, st(0), st(5) },
/*d6*/  { INS_fcmovbe, st(0), st(6) },
/*d7*/  { INS_fcmovbe, st(0), st(7) },

/*d8*/  { INS_fcmovu,  st(0), st(0) },
/*d9*/  { INS_fcmovu,  st(0), st(1) },
/*da*/  { INS_fcmovu,  st(0), st(2) },
/*db*/  { INS_fcmovu,  st(0), st(3) },
/*dc*/  { INS_fcmovu,  st(0), st(4) },
/*dd*/  { INS_fcmovu,  st(0), st(5) },
/*de*/  { INS_fcmovu,  st(0), st(6) },
/*df*/  { INS_fcmovu,  st(0), st(7) },

/*e0*/  { 0 },
/*e1*/  { 0 },
/*e2*/  { 0 },
/*e3*/  { 0 },
/*e4*/  { 0 },
/*e5*/  { 0 },
/*e6*/  { 0 },
/*e7*/  { 0 },

/*e8*/  { 0 },
/*e9*/  { INS_fucompp },
/*ea*/  { 0 },
/*eb*/  { 0 },
/*ec*/  { 0 },
/*ed*/  { 0 },
/*ee*/  { 0 },
/*ef*/  { 0 },

/*f0*/  { 0 },
/*f1*/  { 0 },
/*f2*/  { 0 },
/*f3*/  { 0 },
/*f4*/  { 0 },
/*f5*/  { 0 },
/*f6*/  { 0 },
/*f7*/  { 0 },

/*f8*/  { 0 },
/*f9*/  { 0 },
/*fa*/  { 0 },
/*fb*/  { 0 },
/*fc*/  { 0 },
/*fd*/  { 0 },
/*fe*/  { 0 },
/*ff*/  { 0 },

/* DB ESC Opcode: */
/*c0*/  { INS_fcmovnb, st(0), st(0) },
/*c1*/  { INS_fcmovnb, st(0), st(1) },
/*c2*/  { INS_fcmovnb, st(0), st(2) },
/*c3*/  { INS_fcmovnb, st(0), st(3) },
/*c4*/  { INS_fcmovnb, st(0), st(4) },
/*c5*/  { INS_fcmovnb, st(0), st(5) },
/*c6*/  { INS_fcmovnb, st(0), st(6) },
/*c7*/  { INS_fcmovnb, st(0), st(7) },

/*c8*/  { INS_fcmovne, st(0), st(0) },
/*c9*/  { INS_fcmovne, st(0), st(1) },
/*ca*/  { INS_fcmovne, st(0), st(2) },
/*cb*/  { INS_fcmovne, st(0), st(3) },
/*cc*/  { INS_fcmovne, st(0), st(4) },
/*cd*/  { INS_fcmovne, st(0), st(5) },
/*ce*/  { INS_fcmovne, st(0), st(6) },
/*cf*/  { INS_fcmovne, st(0), st(7) },

/*d0*/  { INS_fcmovnbe,st(0), st(0) },
/*d1*/  { INS_fcmovnbe,st(0), st(1) },
/*d2*/  { INS_fcmovnbe,st(0), st(2) },
/*d3*/  { INS_fcmovnbe,st(0), st(3) },
/*d4*/  { INS_fcmovnbe,st(0), st(4) },
/*d5*/  { INS_fcmovnbe,st(0), st(5) },
/*d6*/  { INS_fcmovnbe,st(0), st(6) },
/*d7*/  { INS_fcmovnbe,st(0), st(7) },

/*d8*/  { INS_fcmovnu, st(0), st(0) },
/*d9*/  { INS_fcmovnu, st(0), st(1) },
/*da*/  { INS_fcmovnu, st(0), st(2) },
/*db*/  { INS_fcmovnu, st(0), st(3) },
/*dc*/  { INS_fcmovnu, st(0), st(4) },
/*dd*/  { INS_fcmovnu, st(0), st(5) },
/*de*/  { INS_fcmovnu, st(0), st(6) },
/*df*/  { INS_fcmovnu, st(0), st(7) },

/*e0*/  { 0 },
/*e1*/  { 0 },
/*e2*/  { INS_fclex },
/*e3*/  { INS_finit },
/*e4*/  { 0 },
/*e5*/  { 0 },
/*e6*/  { 0 },
/*e7*/  { 0 },

/*e8*/  { INS_fucomi,  st(0), st(0) },
/*e9*/  { INS_fucomi,  st(0), st(1) },
/*ea*/  { INS_fucomi,  st(0), st(2) },
/*eb*/  { INS_fucomi,  st(0), st(3) },
/*ec*/  { INS_fucomi,  st(0), st(4) },
/*ed*/  { INS_fucomi,  st(0), st(5) },
/*ee*/  { INS_fucomi,  st(0), st(6) },
/*ef*/  { INS_fucomi,  st(0), st(7) },

/*f0*/  { INS_fcomi,   st(0), st(0) },
/*f1*/  { INS_fcomi,   st(0), st(1) },
/*f2*/  { INS_fcomi,   st(0), st(2) },
/*f3*/  { INS_fcomi,   st(0), st(3) },
/*f4*/  { INS_fcomi,   st(0), st(4) },
/*f5*/  { INS_fcomi,   st(0), st(5) },
/*f6*/  { INS_fcomi,   st(0), st(6) },
/*f7*/  { INS_fcomi,   st(0), st(7) },

/*f8*/  { 0 },
/*f9*/  { 0 },
/*fa*/  { 0 },
/*fb*/  { 0 },
/*fc*/  { 0 },
/*fd*/  { 0 },
/*fe*/  { 0 },
/*ff*/  { 0 },

/* DC ESC Opcode: */
/*c0*/  { INS_fadd,    st(0), st(0) },
/*c1*/  { INS_fadd,    st(1), st(0) },
/*c2*/  { INS_fadd,    st(2), st(0) },
/*c3*/  { INS_fadd,    st(3), st(0) },
/*c4*/  { INS_fadd,    st(4), st(0) },
/*c5*/  { INS_fadd,    st(5), st(0) },
/*c6*/  { INS_fadd,    st(6), st(0) },
/*c7*/  { INS_fadd,    st(7), st(0) },

/*c8*/  { INS_fmul,    st(0), st(0) },
/*c9*/  { INS_fmul,    st(1), st(0) },
/*ca*/  { INS_fmul,    st(2), st(0) },
/*cb*/  { INS_fmul,    st(3), st(0) },
/*cc*/  { INS_fmul,    st(4), st(0) },
/*cd*/  { INS_fmul,    st(5), st(0) },
/*ce*/  { INS_fmul,    st(6), st(0) },
/*cf*/  { INS_fmul,    st(7), st(0) },

/*d0*/  { 0 },
/*d1*/  { 0 },
/*d2*/  { 0 },
/*d3*/  { 0 },
/*d4*/  { 0 },
/*d5*/  { 0 },
/*d6*/  { 0 },
/*d7*/  { 0 },

/*d8*/  { 0 },
/*d9*/  { 0 },
/*da*/  { 0 },
/*db*/  { 0 },
/*dc*/  { 0 },
/*dd*/  { 0 },
/*de*/  { 0 },
/*df*/  { 0 },

/*e0*/  { INS_fsubr,   st(0), st(0) },
/*e1*/  { INS_fsubr,   st(1), st(0) },
/*e2*/  { INS_fsubr,   st(2), st(0) },
/*e3*/  { INS_fsubr,   st(3), st(0) },
/*e4*/  { INS_fsubr,   st(4), st(0) },
/*e5*/  { INS_fsubr,   st(5), st(0) },
/*e6*/  { INS_fsubr,   st(6), st(0) },
/*e7*/  { INS_fsubr,   st(7), st(0) },

/*e8*/  { INS_fsub,    st(0), st(0) },
/*e9*/  { INS_fsub,    st(1), st(0) },
/*ea*/  { INS_fsub,    st(2), st(0) },
/*eb*/  { INS_fsub,    st(3), st(0) },
/*ec*/  { INS_fsub,    st(4), st(0) },
/*ed*/  { INS_fsub,    st(5), st(0) },
/*ee*/  { INS_fsub,    st(6), st(0) },
/*ef*/  { INS_fsub,    st(7), st(0) },

/*f0*/  { INS_fdivr,   st(0), st(0) },
/*f1*/  { INS_fdivr,   st(1), st(0) },
/*f2*/  { INS_fdivr,   st(2), st(0) },
/*f3*/  { INS_fdivr,   st(3), st(0) },
/*f4*/  { INS_fdivr,   st(4), st(0) },
/*f5*/  { INS_fdivr,   st(5), st(0) },
/*f6*/  { INS_fdivr,   st(6), st(0) },
/*f7*/  { INS_fdivr,   st(7), st(0) },

/*f8*/  { INS_fdiv,    st(0), st(0) },
/*f9*/  { INS_fdiv,    st(1), st(0) },
/*fa*/  { INS_fdiv,    st(2), st(0) },
/*fb*/  { INS_fdiv,    st(3), st(0) },
/*fc*/  { INS_fdiv,    st(4), st(0) },
/*fd*/  { INS_fdiv,    st(5), st(0) },
/*fe*/  { INS_fdiv,    st(6), st(0) },
/*ff*/  { INS_fdiv,    st(7), st(0) },

/* DD ESC Opcode: */
/*c0*/  { INS_ffree,   st(0) },
/*c1*/  { INS_ffree,   st(1) },
/*c2*/  { INS_ffree,   st(2) },
/*c3*/  { INS_ffree,   st(3) },
/*c4*/  { INS_ffree,   st(4) },
/*c5*/  { INS_ffree,   st(5) },
/*c6*/  { INS_ffree,   st(6) },
/*c7*/  { INS_ffree,   st(7) },

/*c8*/  { 0 },
/*c9*/  { 0 },
/*ca*/  { 0 },
/*cb*/  { 0 },
/*cc*/  { 0 },
/*cd*/  { 0 },
/*ce*/  { 0 },
/*cf*/  { 0 },

/*d0*/  { INS_fst,     st(0) },
/*d1*/  { INS_fst,     st(1) },
/*d2*/  { INS_fst,     st(2) },
/*d3*/  { INS_fst,     st(3) },
/*d4*/  { INS_fst,     st(4) },
/*d5*/  { INS_fst,     st(5) },
/*d6*/  { INS_fst,     st(6) },
/*d7*/  { INS_fst,     st(7) },

/*d8*/  { INS_fstp,    st(0) },
/*d9*/  { INS_fstp,    st(1) },
/*da*/  { INS_fstp,    st(2) },
/*db*/  { INS_fstp,    st(3) },
/*dc*/  { INS_fstp,    st(4) },
/*dd*/  { INS_fstp,    st(5) },
/*de*/  { INS_fstp,    st(6) },
/*df*/  { INS_fstp,    st(7) },

/*e0*/  { INS_fucom,   st(0), st(0) },
/*e1*/  { INS_fucom,   st(1), st(0) },
/*e2*/  { INS_fucom,   st(2), st(0) },
/*e3*/  { INS_fucom,   st(3), st(0) },
/*e4*/  { INS_fucom,   st(4), st(0) },
/*e5*/  { INS_fucom,   st(5), st(0) },
/*e6*/  { INS_fucom,   st(6), st(0) },
/*e7*/  { INS_fucom,   st(7), st(0) },

/*e8*/  { INS_fucomp,  st(0) },
/*e9*/  { INS_fucomp,  st(1) },
/*ea*/  { INS_fucomp,  st(2) },
/*eb*/  { INS_fucomp,  st(3) },
/*ec*/  { INS_fucomp,  st(4) },
/*ed*/  { INS_fucomp,  st(5) },
/*ee*/  { INS_fucomp,  st(6) },
/*ef*/  { INS_fucomp,  st(7) },

/*f0*/  { 0 },
/*f1*/  { 0 },
/*f2*/  { 0 },
/*f3*/  { 0 },
/*f4*/  { 0 },
/*f5*/  { 0 },
/*f6*/  { 0 },
/*f7*/  { 0 },

/*f8*/  { 0 },
/*f9*/  { 0 },
/*fa*/  { 0 },
/*fb*/  { 0 },
/*fc*/  { 0 },
/*fd*/  { 0 },
/*fe*/  { 0 },
/*ff*/  { 0 },

/* DE ESC Opcode: */
/*c0*/  { INS_faddp,   st(0), st(0) },
/*c1*/  { INS_faddp,   st(1), st(0) },
/*c2*/  { INS_faddp,   st(2), st(0) },
/*c3*/  { INS_faddp,   st(3), st(0) },
/*c4*/  { INS_faddp,   st(4), st(0) },
/*c5*/  { INS_faddp,   st(5), st(0) },
/*c6*/  { INS_faddp,   st(6), st(0) },
/*c7*/  { INS_faddp,   st(7), st(0) },

/*c8*/  { INS_fmulp,   st(0), st(0) },
/*c9*/  { INS_fmulp,   st(1), st(0) },
/*ca*/  { INS_fmulp,   st(2), st(0) },
/*cb*/  { INS_fmulp,   st(3), st(0) },
/*cc*/  { INS_fmulp,   st(4), st(0) },
/*cd*/  { INS_fmulp,   st(5), st(0) },
/*ce*/  { INS_fmulp,   st(6), st(0) },
/*cf*/  { INS_fmulp,   st(7), st(0) },

/*d0*/  { 0 },
/*d1*/  { 0 },
/*d2*/  { 0 },
/*d3*/  { 0 },
/*d4*/  { 0 },
/*d5*/  { 0 },
/*d6*/  { 0 },
/*d7*/  { 0 },

/*d8*/  { 0 },
/*d9*/  { INS_fcompp },
/*da*/  { 0 },
/*db*/  { 0 },
/*dc*/  { 0 },
/*dd*/  { 0 },
/*de*/  { 0 },
/*df*/  { 0 },

/*e0*/  { INS_fsubrp,  st(0), st(0) },
/*e1*/  { INS_fsubrp,  st(1), st(0) },
/*e2*/  { INS_fsubrp,  st(2), st(0) },
/*e3*/  { INS_fsubrp,  st(3), st(0) },
/*e4*/  { INS_fsubrp,  st(4), st(0) },
/*e5*/  { INS_fsubrp,  st(5), st(0) },
/*e6*/  { INS_fsubrp,  st(6), st(0) },
/*e7*/  { INS_fsubrp,  st(7), st(0) },

/*e8*/  { INS_fsubp,   st(0), st(0) },
/*e9*/  { INS_fsubp,   st(1), st(0) },
/*ea*/  { INS_fsubp,   st(2), st(0) },
/*eb*/  { INS_fsubp,   st(3), st(0) },
/*ec*/  { INS_fsubp,   st(4), st(0) },
/*ed*/  { INS_fsubp,   st(5), st(0) },
/*ee*/  { INS_fsubp,   st(6), st(0) },
/*ef*/  { INS_fsubp,   st(7), st(0) },

/*f0*/  { INS_fdivrp,  st(0), st(0) },
/*f1*/  { INS_fdivrp,  st(1), st(0) },
/*f2*/  { INS_fdivrp,  st(2), st(0) },
/*f3*/  { INS_fdivrp,  st(3), st(0) },
/*f4*/  { INS_fdivrp,  st(4), st(0) },
/*f5*/  { INS_fdivrp,  st(5), st(0) },
/*f6*/  { INS_fdivrp,  st(6), st(0) },
/*f7*/  { INS_fdivrp,  st(7), st(0) },

/*f8*/  { INS_fdivp,   st(0), st(0) },
/*f9*/  { INS_fdivp,   st(1), st(0) },
/*fa*/  { INS_fdivp,   st(2), st(0) },
/*fb*/  { INS_fdivp,   st(3), st(0) },
/*fc*/  { INS_fdivp,   st(4), st(0) },
/*fd*/  { INS_fdivp,   st(5), st(0) },
/*fe*/  { INS_fdivp,   st(6), st(0) },
/*ff*/  { INS_fdivp,   st(7), st(0) },

/* DF ESC Opcode: */
/*c0*/  { 0 },
/*c1*/  { 0 },
/*c2*/  { 0 },
/*c3*/  { 0 },
/*c4*/  { 0 },
/*c5*/  { 0 },
/*c6*/  { 0 },
/*c7*/  { 0 },

/*c8*/  { 0 },
/*c9*/  { 0 },
/*ca*/  { 0 },
/*cb*/  { 0 },
/*cc*/  { 0 },
/*cd*/  { 0 },
/*ce*/  { 0 },
/*cf*/  { 0 },

/*d0*/  { 0 },
/*d1*/  { 0 },
/*d2*/  { 0 },
/*d3*/  { 0 },
/*d4*/  { 0 },
/*d5*/  { 0 },
/*d6*/  { 0 },
/*d7*/  { 0 },

/*d8*/  { 0 },
/*d9*/  { 0 },
/*da*/  { 0 },
/*db*/  { 0 },
/*dc*/  { 0 },
/*dd*/  { 0 },
/*de*/  { 0 },
/*df*/  { 0 },

/*e0*/  { INS_fstsw,   reg(AX,w) },
/*e1*/  { 0 },
/*e2*/  { 0 },
/*e3*/  { 0 },
/*e4*/  { 0 },
/*e5*/  { 0 },
/*e6*/  { 0 },
/*e7*/  { 0 },

/*e8*/  { INS_fucomip, st(0), st(0) },
/*e9*/  { INS_fucomip, st(0), st(1) },
/*ea*/  { INS_fucomip, st(0), st(2) },
/*eb*/  { INS_fucomip, st(0), st(3) },
/*ec*/  { INS_fucomip, st(0), st(4) },
/*ed*/  { INS_fucomip, st(0), st(5) },
/*ee*/  { INS_fucomip, st(0), st(6) },
/*ef*/  { INS_fucomip, st(0), st(7) },

/*f0*/  { INS_fcomip,  st(0), st(0) },
/*f1*/  { INS_fcomip,  st(0), st(1) },
/*f2*/  { INS_fcomip,  st(0), st(2) },
/*f3*/  { INS_fcomip,  st(0), st(3) },
/*f4*/  { INS_fcomip,  st(0), st(4) },
/*f5*/  { INS_fcomip,  st(0), st(5) },
/*f6*/  { INS_fcomip,  st(0), st(6) },
/*f7*/  { INS_fcomip,  st(0), st(7) },

/*f8*/  { 0 },
/*f9*/  { 0 },
/*fa*/  { 0 },
/*fb*/  { 0 },
/*fc*/  { 0 },
/*fd*/  { 0 },
/*fe*/  { 0 },
/*ff*/  { 0 }
};

/* Various maps and tables */

#define OpHasModRM(op)     (   IsOpGeneral((op)) \
                            && AddrHasModRM[ GenAddrMode((op)) ] )
#define InstrHasModRM(ip)  (   IsInstrGroup((ip)->instr) \
                            || IsInstrEsc((ip)->instr)   \
                            || OpHasModRM((ip)->op0)   \
                            || OpHasModRM((ip)->op1) )
#define OpHasEA(op)        (    IsOpGeneral((op)) \
                            && (   GenAddrMode((op)) == AddrMode_E \
                                || GenAddrMode((op)) == AddrMode_M \
                                || GenAddrMode((op)) == AddrMode_Q))
#define InstrHasEA(ip)     (   OpHasEA((ip)->op0) \
                            || OpHasEA((ip)->op1))
#define OpHasMO(op)        (    IsOpGeneral((op)) \
                            &&  GenAddrMode((op)) == AddrMode_O)
#define InstrHasMO(ip)     (   OpHasMO((ip)->op0) \
                            || OpHasMO((ip)->op1))


#define FieldMod(modrm)     ( (modrm) >> 6       )
#define FieldReg(modrm)     (((modrm) >> 3) & 0x7)
#define FieldRM(modrm)      ( (modrm)       & 0x7)

#define FieldSIBScale(sib)  ( (sib) >> 6       )
#define FieldSIBIndex(sib)  (((sib) >> 3) & 0x7)
#define FieldSIBBase(sib)   ( (sib)       & 0x7)


int MapSeg[] = 
  { SEG_ES,  SEG_CS,  SEG_SS,  SEG_DS,  SEG_FS,  SEG_GS,   0,       0       };
int MapControl[] = 
  { P86_REG_CR0, P86_REG_CR1, P86_REG_CR2, P86_REG_CR3, P86_REG_CR4, P86_REG_CR5,  P86_REG_CR6, P86_REG_CR7 };
int MapDebug[] = 
  { P86_REG_DR0, P86_REG_DR1, P86_REG_DR2, P86_REG_DR3, P86_REG_DR4, P86_REG_DR5,  P86_REG_DR6, P86_REG_DR7 };
int MapTest[] = 
  { P86_REG_TR0, P86_REG_TR1, P86_REG_TR2, P86_REG_TR3, P86_REG_TR4, P86_REG_TR5,  P86_REG_TR6, P86_REG_TR7 };
int MapMMX[] = 
  { P86_REG_MM0, P86_REG_MM1, P86_REG_MM2, P86_REG_MM3, P86_REG_MM4, P86_REG_MM5,  P86_REG_MM6, P86_REG_MM7 };
int MapXMMX[] = 
  { P86_REG_XMM0, P86_REG_XMM1, P86_REG_XMM2, P86_REG_XMM3, P86_REG_XMM4, P86_REG_XMM5,  P86_REG_XMM6, P86_REG_XMM7 };

int MapReg8[] = 
  { P86_REG_AL,  P86_REG_CL,  P86_REG_DL,  P86_REG_BL,  P86_REG_AH,  P86_REG_CH,  P86_REG_DH,  P86_REG_BH  };
int MapReg16[] = 
  { P86_REG_AX,  P86_REG_CX,  P86_REG_DX,  P86_REG_BX,  P86_REG_SP,  P86_REG_BP,  P86_REG_SI,  P86_REG_DI  };
int MapReg32[] = 
  { P86_REG_EAX, P86_REG_ECX, P86_REG_EDX, P86_REG_EBX, P86_REG_ESP, P86_REG_EBP, P86_REG_ESI, P86_REG_EDI };

int EffBase16[]  = 
  { P86_REG_BX,  P86_REG_BX,  P86_REG_BP,  P86_REG_BP,  0,       0,       P86_REG_BP,  P86_REG_BX  };
int EffIndex16[] = 
  { P86_REG_SI,  P86_REG_DI,  P86_REG_SI,  P86_REG_DI,  P86_REG_SI,  P86_REG_DI,  0,       0       };

int EffBase32[]  = 
  { P86_REG_EAX, P86_REG_ECX, P86_REG_EDX, P86_REG_EBX, P86_REG_ESP, P86_REG_EBP, P86_REG_ESI, P86_REG_EDI };
int EffIndex32[] = 
  { P86_REG_EAX, P86_REG_ECX, P86_REG_EDX, P86_REG_EBX, 0,       P86_REG_EBP, P86_REG_ESI, P86_REG_EDI };


static int sign_extend( int val, int nFromSize, int nToSize )
{
    switch ( nFromSize )
    {
    default: assert( FALSE );
    case 1: if ( val & 0x00000080 ) val |= 0xffffff00; break;
    case 2: if ( val & 0x00008000 ) val |= 0xffff0000; break;
    case 4: break;
    }
    switch ( nToSize )
    {
    default: assert( FALSE );
    case 1: val &= 0x000000ff; break;
    case 2: val &= 0x0000ffff; break;
    case 4: break;
    }
    return val;
}

static int i386_decode_raw( struct i386_context *ctx, int *value, int size )
{
    /* FIXME */

    switch ( size )
    {
    default: assert( FALSE );
    case 1: *value = *(unsigned char  *)(ctx->base + ctx->offset);  break;
    case 2: *value = *(unsigned short *)(ctx->base + ctx->offset);  break;
    case 4: *value = *(unsigned int   *)(ctx->base + ctx->offset);  break;
    }

    ctx->offset += size;
    return TRUE;
}

static int i386_decode_offset( struct i386_context *ctx, 
                               struct i386_decode_data *data, int size )
{
    memset(data, '\0', sizeof(struct i386_decode_data));
    if ( size != 1 && size != 2 && size != 4 ) return FALSE;

    return i386_decode_raw( ctx, &data->offset, size );
}

static int i386_decode_segment( struct i386_context *ctx,
                                int *segment, int size )
{
    *segment = 0;
    if ( size != 2 && size != 4 ) return FALSE;

    return i386_decode_raw( ctx, segment, size );
}

static int i386_decode_pointer( struct i386_context *ctx,
                                struct i386_decode_data *data, int size )
{
    if ( size != 4 && size != 6 ) return FALSE;

    return    i386_decode_offset( ctx, data, size-2 )
           && i386_decode_segment( ctx, &data->segment, 2 );
}


/* Decoding routine */

int i386_decode( struct i386_context *ctx, struct i386_decode *dc )
{
    int old_offset = ctx->offset;
    int base = 0, index = 0, scale = 0;
    int reg = 0, mod = 0, rm = 0;
    int opcode, i, displ, size = 0, prefix = TRUE;
    struct instruction ip;
    struct i386_decode_data eff;

    memset(dc, '\0', sizeof(struct i386_decode));
    memset(&eff, '\0', sizeof(eff));

    /* Default address and operand size */
    dc->address_size = dc->operand_size = ctx->mode == CODE32 ? 4 : 2;

    while ( prefix )
    {
        /* Get instruction byte */
        if (!i386_decode_raw( ctx, &opcode, 1 )) goto fault;
        ip = OneByteMap[opcode];

        /* Check for prefixes */
        switch ( ip.instr )
        {
        case INS_InsPrefix: 
            dc->prefix = ip.op0;
            break;

        case INS_AddrSizePrefix: 
            dc->address_size = ctx->mode == CODE32 ? 2 : 4;
            break;

        case INS_OpSizePrefix: 
            dc->operand_size = ctx->mode == CODE32 ? 2 : 4;
            break;

        case INS_SegPrefix: 
            dc->segment_override = ip.op0;
            break;

        default:
            prefix = FALSE;
            break;
        }
    }
   
    /* Check for two-byte opcode */
    if (ip.instr == INS_TwoBytePrefix)
    {
        if (!i386_decode_raw( ctx, &opcode, 1 )) goto fault;
        ip = TwoByteMap[opcode];
    }

    /* Get ModR/M byte if present */
    if (InstrHasModRM( &ip ))
    {
        if (!i386_decode_raw( ctx, &opcode, 1 )) goto fault;
        mod = FieldMod( opcode );
        reg = FieldReg( opcode );
        rm  = FieldRM ( opcode );
    }

    /* Check for instruction-group opcode */
    if (IsInstrGroup( ip.instr ))
    {
        struct instruction *group = GroupMap + GetInstrGroup(ip.instr)*8 + reg;
        ip.instr = group->instr;
        ip.op0  |= group->op0;
        ip.op1  |= group->op1;
        ip.op2  |= group->op2;
    }

    /* Check for escape (coprocessor) instruction */
    if (IsInstrEsc( ip.instr )) {
        if ( mod != 3 )
            ip = EscMainMap[ GetInstrEsc( ip.instr ) * 8 + reg ];
        else
            ip = EscAuxMap [ GetInstrEsc( ip.instr ) * 64 + reg * 8 + rm ];
    }
    /* Check for size-dependent instruction name */
    if (IsSizeDep( ip.instr ))
        ip.instr = GetSizeDep( ip.instr, dc->operand_size );

    /* Check for illegal instruction */
    if (!ip.instr || ip.instr >= INS_LAST) goto fault;
    dc->instruction = ip.instr;

    /* Special REP prefix check */
    if ( dc->prefix == INS_rep )
        if ( dc->instruction == INS_cmps || dc->instruction == INS_scas )
            dc->prefix = INS_repe;

    /* i386_decode Effective Memory Address if present */
    if (InstrHasEA( &ip ) && mod != 3)
    {
        if ( dc->address_size == 2 )
        {
            base  = EffBase16 [rm];
            index = EffIndex16[rm];
            if (base == P86_REG_BP && !index && !mod) base = 0, mod = 2;
        }
        else
        {
            /* Get SIB byte if present */
            if ((base = EffBase32[rm]) == P86_REG_ESP)
            {
                if (!i386_decode_raw( ctx, &opcode, 1 )) goto fault;
                base  = EffBase32 [ FieldSIBBase(opcode)  ];
                index = EffIndex32[ FieldSIBIndex(opcode) ];
                scale = index? 1 << FieldSIBScale(opcode) : 0;
            }
            if (base == P86_REG_EBP && !mod) base = 0, mod = 2;
        }

        /* Get displacement if present */
        if (mod)
        {
            dc->displacement_size = mod == 1 ? 1 : dc->address_size;
            if (!i386_decode_offset( ctx, &eff, dc->displacement_size )) goto fault;
            eff.offset = sign_extend( eff.offset, dc->displacement_size, 4 );
        }
    }

    /* i386_decode Memory Offset if present */
    if (InstrHasMO( &ip ))
    {
        dc->displacement_size = dc->address_size;
        if (!i386_decode_offset( ctx, &eff, dc->displacement_size )) goto fault;
    }

    /* Assign arguments (and i386_decode immediate values) */
    for (i = 0; i < 3; i++)
    {
        int addr_mode = 0, op_type = 0, *MapReg = 0;
        struct i386_decode_operand *da = dc->op + i;

        int op = (i == 0)? ip.op0 : (i == 1)? ip.op1 : ip.op2;
        if ( !op ) break;

        if (IsOpGeneral( op ))
        {
            addr_mode = GenAddrMode( op );
            op_type   = GenOpType  ( op );
            if (!addr_mode) break;

            if (addr_mode == AddrMode_E) 
                addr_mode = (mod == 3)? AddrMode_R : AddrMode_M;
            if (addr_mode == AddrMode_Q) 
                addr_mode = (mod == 3)? AddrMode_Q : AddrMode_M;
            if (addr_mode == AddrMode_W) 
                addr_mode = (mod == 3)? AddrMode_W : AddrMode_M;
        }
        else if (IsOpRegister( op ))
        {
            addr_mode = AddrMode_G;
            op_type   = RegType( op );
            reg       = RegName( op );
        }
        else if (IsOpSegment( op ))
        {
            da->type = OT_VALUE;
            da->size = 2;
            da->mode = OM_REGISTER;
            da->u.reg = SegName( op );
            continue;
        }
        else if (IsOpFPUStack( op ))
        {
            da->type = OT_VALUE;
            da->size = 10;
            da->mode = OM_REGISTER;
            da->u.reg = FPUStackName( op );
            continue;
        }
        else if (IsOpImmediate( op ))
        {
            da->type = OT_VALUE;
            da->size = 1;
            da->mode = OM_IMMEDIATE;
            da->u.immediate.offset = ImmValue( op );
            continue;
        }

        switch (op_type)
        {
        case OpType_0: da->size = size = 0; 
                       da->type = OT_VALUE; break;
        case OpType_b: da->size = size = 1; 
                       da->type = OT_VALUE; break;
        case OpType_w: da->size = size = 2;
                       da->type = OT_VALUE; break;
        case OpType_d: da->size = size = 4; 
                       da->type = OT_VALUE; break;
        case OpType_q: da->size = size = 8; 
                       da->type = OT_VALUE; break;
        case OpType_o: da->size = size = 16; 
                       da->type = OT_VALUE; break;
        case OpType_x: da->size = size = 4; 
                       da->type = OT_VALUE; break;
        case OpType_y: da->size = size = 8; 
                       da->type = OT_VALUE; break;
        case OpType_z: da->size = size = 10; 
                       da->type = OT_VALUE; break;
        case OpType_v: da->size = size = dc->operand_size;
                       da->type = OT_VALUE; break;
        case OpType_c: da->size = dc->operand_size; size = 1;
                       da->type = OT_VALUE; break;
        case OpType_p: da->size = size = dc->operand_size + 2; 
                       da->type = OT_POINTER; break;
        case OpType_s: da->size = size = 6;
                       da->type = OT_DESCRIPTOR; break;
        case OpType_a: da->size = size = dc->operand_size * 2; 
                       da->type = OT_VALUEPAIR; break;
        }
        switch (da->size)
        {
        case 1: MapReg = MapReg8;  break;
        case 2: MapReg = MapReg16; break;
        case 4: MapReg = MapReg32; break;
        }

        switch (addr_mode)
        {
        case AddrMode_G:    /* general register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapReg[reg];
            break;

        case AddrMode_R:    /* general register in R/M field */
            if (da->type != OT_VALUE) goto fault;
            if (mod != 3) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapReg[rm];
            break;

        case AddrMode_S:    /* segment register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapSeg[reg];
            if (!da->u.reg) goto fault;
            break;

        case AddrMode_F:    /* flags register */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = size == 2? P86_REG_FL : P86_REG_EFL;
            break;

        case AddrMode_C:    /* control register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapControl[reg];
            break;

        case AddrMode_D:    /* debug register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapDebug[reg];
            break;

        case AddrMode_T:    /* test register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapTest[reg];
            break;

        case AddrMode_P:    /* MMX register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapMMX[reg];
            break;

        case AddrMode_Q:    /* MMX register in R/M field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapMMX[rm];
            break;

        case AddrMode_V:    /* SSE register in reg field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapXMMX[reg];
            break;

        case AddrMode_W:    /* SSE register in R/M field */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_REGISTER;
            da->u.reg = MapXMMX[rm];
            break;

        case AddrMode_I:    /* immediate data */
            if (da->type != OT_VALUE) goto fault;
            da->mode = OM_IMMEDIATE;
            if (!i386_decode_offset( ctx, &da->u.immediate, size )) goto fault;
            da->u.immediate.offset = sign_extend( da->u.immediate.offset, size, da->size );
            break;

        case AddrMode_A:    /* direct address */
            da->mode = OM_IMMEDIATE;
            if (da->type == OT_POINTER)
            {
                if (!i386_decode_pointer( ctx, &da->u.immediate, size )) goto fault; 
            }
            else
            {
                if (!i386_decode_offset( ctx, &da->u.immediate, size )) goto fault; 
                da->u.immediate.segment = ctx->segment;
            }
            break;

        case AddrMode_J:    /* direct address, (E)IP relative */
            da->mode = OM_IMMEDIATE;
            da->size = ctx->mode == CODE32 ? 6 : 4;
            if (!i386_decode_raw( ctx, &displ, size )) goto fault;
            displ = sign_extend( displ, size, da->size-2 );
            da->u.immediate.segment = ctx->segment;
            da->u.immediate.offset  = sign_extend( ctx->offset + displ, da->size-2, da->size-2 );
            break;

        case AddrMode_M:    /* effective memory address (already decoded) */
        case AddrMode_O:    /* memory offset (already decoded) */
            if (mod == 3) goto fault;
            da->mode = OM_MEMORY;

            if (dc->segment_override)
                da->u.memory.segment = dc->segment_override;
            else if (   base == P86_REG_BP  || base == P86_REG_SP
                     || base == P86_REG_EBP || base == P86_REG_ESP)
                da->u.memory.segment = SEG_SS;
            else
                da->u.memory.segment = SEG_DS;

            da->u.memory.base  = base;
            da->u.memory.index = index;
            da->u.memory.scale = scale;
            da->u.memory.displacement = eff.offset;
            break;

        case AddrMode_X:    /* memory addressed by (DS):(E)SI */
            da->mode = OM_MEMORY;

            if (dc->segment_override)
                da->u.memory.segment = dc->segment_override;
            else
                da->u.memory.segment = SEG_DS;

            da->u.memory.base = dc->address_size == 2 ? P86_REG_SI : P86_REG_ESI;
            break;

        case AddrMode_Y:    /* memory addressed by ES:(E)DI */
            da->mode = OM_MEMORY;
            da->u.memory.segment = SEG_ES; /* no override possible */
            da->u.memory.base = dc->address_size == 2 ? P86_REG_DI : P86_REG_EDI;
            break;
        }
    }

    /* Special cmpps handling */
    if ( dc->instruction == INS_cmpps )
    {
        assert( dc->op[2].mode == OM_IMMEDIATE );
        if (    dc->op[2].u.immediate.segment 
             || dc->op[2].u.immediate.offset >= 8 )
            goto fault;

        dc->op[2].type = OT_NONE;
        switch ( dc->op[2].u.immediate.offset )
        {
        case 0:  dc->instruction = INS_cmpeqps;    break;
        case 1:  dc->instruction = INS_cmpltps;    break;
        case 2:  dc->instruction = INS_cmpleps;    break;
        case 3:  dc->instruction = INS_cmpunordps; break;
        case 4:  dc->instruction = INS_cmpneqps;   break;
        case 5:  dc->instruction = INS_cmpnltps;   break;
        case 6:  dc->instruction = INS_cmpnleps;   break;
        case 7:  dc->instruction = INS_cmpordps;   break;
        }
    }

    /* Special cmpss handling */
    if ( dc->instruction == INS_cmpss )
    {
        assert( dc->op[2].mode == OM_IMMEDIATE );
        if (    dc->op[2].u.immediate.segment 
             || dc->op[2].u.immediate.offset >= 8 )
            goto fault;

        dc->op[2].type = OT_NONE;
        switch ( dc->op[2].u.immediate.offset )
        {
        case 0:  dc->instruction = INS_cmpeqss;    break;
        case 1:  dc->instruction = INS_cmpltss;    break;
        case 2:  dc->instruction = INS_cmpless;    break;
        case 3:  dc->instruction = INS_cmpunordss; break;
        case 4:  dc->instruction = INS_cmpneqss;   break;
        case 5:  dc->instruction = INS_cmpnltss;   break;
        case 6:  dc->instruction = INS_cmpnless;   break;
        case 7:  dc->instruction = INS_cmpordss;   break;
        }
    }

    dc->length = ctx->offset - old_offset;
    return TRUE;

 fault:
    ctx->offset = old_offset;
    return FALSE;
}


/* Text generation */

char *RegNamesIntel[] =  /* must correspond to enum i386_decode_register! */
{
    "",

    /* Byte registers */
    "al", "bl", "cl", "dl",
    "ah", "bh", "ch", "dh",
    /* Word registers */
    "ax", "bx", "cx", "dx",
    "si", "di", "bp", "sp",
    /* DWord registers */
    "eax", "ebx", "ecx", "edx",
    "esi", "edi", "ebp", "esp",

    /* Segment registers */
    "cs", "ss",
    "ds", "es", "fs", "gs",

    /* Flags register */
    "fl", "efl",

    /* Instruction pointer */
    "ip", "eip",

    /* Control registers */
    "cr0", "cr1", "cr2", "cr3",
    "cr4", "cr5", "cr6", "cr7",

    /* Debug registers */
    "dr0", "dr1", "dr2", "dr3",
    "dr4", "dr5", "dr6", "dr7",

    /* Test registers */
    "tr0", "tr1", "tr2", "tr3",
    "tr4", "tr5", "tr6", "tr7",

    /* FPU register stack */
    "st(0)", "st(1)", "st(2)", "st(3)",
    "st(4)", "st(5)", "st(6)", "st(7)",

    /* MMX registers */
    "mm0", "mm1", "mm2", "mm3",
    "mm4", "mm5", "mm6", "mm7",

    /* SSE registers */
    "xmm0", "xmm1", "xmm2", "xmm3",
    "xmm4", "xmm5", "xmm6", "xmm7"
};

char *InsNamesIntel[] = /* must correspond to enum i386_decode_instruction! */
{
    "",

    /* Data Transfer */
    "mov",
    "cmove",   "cmovne",  "cmovp",   "cmovnp",
    "cmova",   "cmovae",  "cmovb",   "cmovbe",
    "cmovg",   "cmovge",  "cmovl",   "cmovle",
    "cmovo",   "cmovno",  "cmovs",   "cmovns",
    "xchg",    "bswap",   "xadd",    "cmpxchg", "cmpxchg8b",
    "push",    "pusha",   "pushad",
    "pop",     "popa",    "popad",
    "in",      "out",
    "cwd",     "cdq",     "cbw",     "cwde",
    "movsx",   "movzx",
    /* Binary Arithmetic */
    "add",     "adc",     "sub",     "sbb",
    "imul",    "mul",     "idiv",    "div",
    "inc",     "dec",     "neg",     "cmp",
    /* Decimal Arithmetic */
    "daa",     "das",
    "aaa",     "aas",     "aam",     "aad",
    /* Logic */
    "and",     "or",      "not",     "xor",
    /* Shift and Rotate */
    "sar",     "shr",     "sal",     "shl",   "shld",  "shrd",
    "ror",     "rol",     "rcr",     "rcl",
    /* Bit and Byte */
    "bt",      "bts",     "btr",     "btc",   "bsf",   "bsr",
    "sete",    "setne",   "setp",    "setnp",
    "seta",    "setae",   "setb",    "setbe",
    "setg",    "setge",   "setl",    "setle",
    "seto",    "setno",   "sets",    "setns",
    "test",
    /* Control Transfer */
    "jmp",
    "je",      "jne",     "jp",      "jnp",
    "ja",      "jae",     "jb",      "jbe",
    "jg",      "jge",     "jl",      "jle",
    "jo",      "jno",     "js",      "jns",
    "jcxz",    "jecxz",
    "loop",    "loope",   "loopne",
    "call",    "ret",     "retf",    "iret",
    "int",     "into",    "bound",
    "enter",   "leave",
    /* String */
    "movs",    "cmps",    "scas",    "lods",    "stos",
    "rep",     "repe",    "repne",
    "ins",     "outs",
    /* Flag Control */
    "stc",     "clc",     "cmc",     "std",     "cld",
    "lahf",    "sahf",
    "pushf",   "pushfd",  "popf",    "popfd",
    "sti",     "cli",
    /* Segment Register Operations */
    "lds",     "les",     "lfs",     "lgs",     "lss",
    /* Miscellaneous */
    "lea",     "nop",     "ud2",     "xlatb",
    "cpuid",
    /* Systems Instructions */
    "lgdt",    "sgdt",    "lldt",    "sldt",
    "ltr",     "str",     "lidt",    "sidt",
    "lmsw",    "smsw",    "clts",
    "arpl",    "lar",     "lsl",     "verr",    "verw",
    "invd",    "wbinvd",  "invlpg",
    "lock",    "hlt",     "rsm",
    "rdmsr",   "wrmsr",   "rdpmc",   "rdtsc",
    "sysenter","sysexit",
    /* MMX Data Transfer */
    "movd",    "movq",
    /* MMX Conversion */
    "packsswb",  "packssdw",  "packuswb",
    "punpckhbw", "punpckhwd", "punpckhdq",
    "punpcklbw", "punpcklwd", "punpckldq",
    /* MMX Packed Arithmetic */
    "paddb",   "paddw",   "paddd",
    "paddsb",  "paddsw",  "paddusb", "paddusw",
    "psubb",   "psubw",   "psubd",
    "psubsb",  "psubsw",  "psubusb", "psubusw",
    "pmulhw",  "pmullw",  "pmaddwd",
    /* MMX Comparison */
    "pcmpeqb", "pcmpeqw", "pcmpeqd",
    "pcmpgtb", "pcmpgtw", "pcmpgtd",
    /* MMX Logic */
    "pand",    "pandn",   "por",     "pxor",
    /* MMX Shift and Rotate */
    "psllw",   "pslld",   "psllq",
    "psrlw",   "psrld",   "psrlq",
    "psraw",   "psrad",
    /* MMX State Management */
    "emms",
    /* SSE Floating Point Data Transfer */
    "movss",   "movaps",  "movups",  "movmskps",
    "movlps",  "movhps",  "movhlps", "movlhps",
    /* SSE Floating Point Conversion */
    "cvtpi2ps","cvtps2pi","cvttps2pi",
    "cvtsi2ss","cvtss2si","cvttss2si",
    "unpcklps","unpckhps","shufps",
    /* SSE Floating Point Arithmetic */
    "addps",   "addss",   "subps",   "subss",
    "mulps",   "mulss",   "divps",   "divss",
    "maxps",   "maxss",   "minps",   "minss",
    "sqrtps",  "sqrtss",  "rsqrtps", "rsqrtss",
    "rcpps",   "rcpss",
    /* SSE Floating Point Logic */
    "andps",   "andnps",  "orps",    "xorps",
    /* SSE Floating Point Comparision */
    "cmpps",   "cmpordps","cmpunordps",
    "cmpeqps", "cmpltps", "cmpleps",
    "cmpneqps","cmpnltps","cmpnleps",
    "cmpss",   "cmpordss","cmpunordss",
    "cmpeqss", "cmpltss", "cmpless",
    "cmpneqss","cmpnltss","cmpnless",
    "comiss",  "ucomiss",
    /* SSE Integer Data Transfer */
    "pinsrw",  "pextrw",  "pshufw",  "pmovmskb",
    /* SSE Integer Arithmetic */
    "pmaxub",  "pmaxsw",  "pminub",  "pminsw",
    "pavgb",   "pavgw",
    "pmulhuw", "psadbw",
    /* SSE Cachability Control */
    "movntq",  "movntps", "maskmovq","sfence",
    "prefetcht0", "prefetcht1",
    "prefetcht2", "prefetchnta",
    /* SSE State Management */
    "fxsave",  "fxrstor",
    "ldmxcsr", "stmxcsr",
    /* FPU Data Transfer */
    "fld",     "fst",     "fstp",
    "fild",    "fist",    "fistp",
    "fbld",    "fbstp",   "fxch",
    "fcmove",  "fcmovne", "fcmovu",  "fcmovnu",
    "fcmovb",  "fcmovbe", "fcmovnb", "fcmovnbe",
    /* FPU Basic Arithmetic */
    "fadd",    "faddp",   "fiadd",
    "fsub",    "fsubp",   "fisub",
    "fsubr",   "fsubrp",  "fisubr",
    "fmul",    "fmulp",   "fimul",
    "fdiv",    "fdivp",   "fidiv",
    "fdivr",   "fdivrp",  "fidivr",
    "fprem",   "fprem1",
    "fabs",    "fchs",    "frndint",
    "fscale",  "fsqrt",   "fxtract",
    /* FPU Comparison */
    "fcom",    "fcomp",   "fcompp",
    "fucom",   "fucomp",  "fucompp",
    "ficom",   "ficomp",
    "fcomi",   "fucomi",  "fcomip",  "fucomip",
    "ftst",    "fxam",
    /* FPU Transcendental */
    "fsin",    "fcos",    "fsincos",
    "fptan",   "fpatan",
    "f2xm1",   "fyl2x",   "fyl2xp1",
    /* FPU Load Constants */
    "fld1",    "fldz",    "fldpi",
    "fldl2e",  "fldln2",  "fldl2t",  "fldlg2",
    /* FPU Control */
    "fincstp", "fdecstp", "ffree",
    "finit",   "fninit",  "fclex",   "fnclex",
    "fstcw",   "fnstcw",  "fldcw",
    "fstenv",  "fnstenv", "fldenv",
    "fsave",   "fnsave",  "frstor",
    "fstsw",   "fnstsw",
    "fnop",    "fwait",

    NULL
};

int i386_decode_text_intel( struct i386_context *ctx, struct i386_decode *dc, char *string )
{
    int i, override;

    *string = '\0';

    /* Instruction prefix */
    if ( dc->prefix )
    {
        assert( dc->prefix < INS_LAST );
        sprintf( string, "%s ", InsNamesIntel[dc->prefix]);
    }

    /* Instruction name */
    assert( dc->instruction < INS_LAST );
    strcat(string, InsNamesIntel[dc->instruction]);

    /* Formatting */
    if (strlen(string) < 7)
    {
        strcat(string, "       ");
        string[7] = '\0';
    }

    /* Check whether to display segment overrides */
    override = (dc->segment_override != 0);
    if (dc->op[0].mode == OM_MEMORY && dc->op[0].u.memory.segment == SEG_ES)
        override = TRUE;
    if (dc->op[1].mode == OM_MEMORY && dc->op[1].u.memory.segment == SEG_ES)
        override = TRUE;


    /* Arguments */
    for ( i = 0; i < 3; i++ )
    {
        struct i386_decode_operand *da = dc->op + i;
        if (!da->type) break;

        strcat(string, i? ", " : " ");

        /* Output data */
        switch (da->mode)
        {
        case OM_REGISTER:
            assert( da->u.reg < P86_REG_LAST );
            strcat( string, RegNamesIntel[da->u.reg] );
            break;

        case OM_IMMEDIATE:
        {
            int size = da->u.immediate.segment? da->size - 2 : da->size;
            if ( da->u.immediate.segment )
                sprintf( string+strlen(string), "%04X:", da->u.immediate.segment );

            switch ( size )
            {
            case 1: sprintf( string+strlen(string), "%02X", da->u.immediate.offset ); break;
            case 2: sprintf( string+strlen(string), "%04X", da->u.immediate.offset ); break;
            case 4: sprintf( string+strlen(string), "%08X", da->u.immediate.offset ); break;
            }
        }
        break;

        case OM_MEMORY:
        {
            struct i386_decode_address *addr = &da->u.memory;
            int size  = da->type == OT_POINTER ? da->size - 2 : da->size;
            int value = addr->displacement;
            int first;


            /* Output 'ptr' size specification */
            if ( dc->instruction == INS_jmp || dc->instruction == INS_call )
                strcat( string, da->type == OT_POINTER? "far " : "near " );

            switch (size)
            {
            case 1: strcat(string, "byte ptr "); break;
            case 2: strcat(string, "word ptr "); break;
            case 4: strcat(string, "dword ptr "); break;
            case 8: strcat(string, "qword ptr "); break;
            case 16: strcat(string, "oword ptr "); break;
            }


            /* Output segment override */
            if ( override )
            {
                assert( addr->segment < P86_REG_LAST );
                sprintf( string+strlen(string), "%s:", RegNamesIntel[addr->segment] );
            }

            strcat( string, "[" );
            first = TRUE;

            /* Output base */
            if ( addr->base )
            {
                if (!first) strcat( string, "+" );
                first = FALSE;

                assert( addr->base < P86_REG_LAST );
                strcat( string, RegNamesIntel[addr->base]);
            }

            /* Output index and scale */
            if ( addr->index )
            {
                if (!first) strcat( string, "+" );
                first = FALSE;

                assert( addr->index < P86_REG_LAST );
                strcat( string, RegNamesIntel[addr->index] );

                if ( addr->scale > 1 )
                    sprintf( string+strlen(string), "*%d", addr->scale );
            }

            /* Output displacement */
            if ( !first && value < 0 && value >= -32768 ) 
            {
                strcat( string, "-" ); value = -value; 
                first = TRUE;
            }
            if ( value || first )
            {
                if (!first) strcat( string, "+" );
                first = FALSE;

                switch ( dc->displacement_size )
                {
                    case 1: sprintf( string+strlen(string), "%02X", value & 0xff   ); break;
                    case 2: sprintf( string+strlen(string), "%04X", value & 0xffff ); break;
                    case 4: sprintf( string+strlen(string), "%08X", value          ); break;
                    default: assert( FALSE );
                }
            }

            strcat(string, "]");
        }
        break;
        }
    }

    return TRUE;
}


char *RegNamesATT[] =  /* must correspond to enum i386_decode_register! */
{
    "",

    /* Byte registers */
    "%al", "%bl", "%cl", "%dl",
    "%ah", "%bh", "%ch", "%dh",
    /* Word registers */
    "%ax", "%bx", "%cx", "%dx",
    "%si", "%di", "%bp", "%sp",
    /* DWord registers */
    "%eax", "%ebx", "%ecx", "%edx",
    "%esi", "%edi", "%ebp", "%esp",

    /* Segment registers */
    "%cs", "%ss",
    "%ds", "%es", "%fs", "%gs",

    /* Flags register */
    "%fl", "%efl",

    /* Instruction pointer */
    "%ip", "%eip",

    /* Control registers */
    "%cr0", "%cr1", "%cr2", "%cr3",
    "%cr4", "%cr5", "%cr6", "%cr7",

    /* Debug registers */
    "%db0", "%db1", "%db2", "%db3",
    "%db4", "%db5", "%db6", "%db7",

    /* Test registers */
    "%tr0", "%tr1", "%tr2", "%tr3",
    "%tr4", "%tr5", "%tr6", "%tr7",

    /* FPU register stack */
    "%st(0)", "%st(1)", "%st(2)", "%st(3)",
    "%st(4)", "%st(5)", "%st(6)", "%st(7)",

    /* MMX registers */
    "%mm0", "%mm1", "%mm2", "%mm3",
    "%mm4", "%mm5", "%mm6", "%mm7",

    /* SSE registers */
    "%xmm0", "%xmm1", "%xmm2", "%xmm3",
    "%xmm4", "%xmm5", "%xmm6", "%xmm7"
};

char *InsNamesATT[] = /* must correspond to enum i386_decode_instruction! */
{
    "",

    /* Data Transfer */
    "movD",
    "cmoveD",   "cmovneD",  "cmovpD",   "cmovnpD",
    "cmovaD",   "cmovaeD",  "cmovbD",   "cmovbeD",
    "cmovgD",   "cmovgeD",  "cmovlD",   "cmovleD",
    "cmovoD",   "cmovnoD",  "cmovsD",   "cmovnsD",
    "xchgD",    "bswap",    "xaddD",    "cmpxchgD",    "cmpxchg8b",
    "pushS",    "pusha",    "pushad",
    "popS",     "popa",     "popad",
    "inD",      "outS",
    "cwtd",     "cltd",     "cbtw",     "cwtl",
    "movsSD",   "movzSD",
    /* Binary Arithmetic */
    "addD",     "adcD",     "subD",     "sbbD",
    "imulD",    "mulD",     "idivD",    "divD",
    "incS",     "decS",     "negS",     "cmpD",
    /* Decimal Arithmetic */
    "daa",     "das",
    "aaa",     "aas",     "aam",     "aad",
    /* Logic */
    "andD",    "orD",     "notD",    "xorD",
    /* Shift and Rotate */
    "sarD",    "shrD",    "salD",    "shlD",  "shldD", "shrdD",
    "rorD",    "rolD",    "rcrD",    "rclD",
    /* Bit and Byte */
    "btD",     "btsD",    "btrD",    "btcD",  "bsfD",  "bsrD",
    "seteD",   "setneD",  "setpD",   "setnpD",
    "setaD",   "setaeD",  "setbD",   "setbeD",
    "setgD",   "setgeD",  "setlD",   "setleD",
    "setoD",   "setnoD",  "setsD",   "setnsD",
    "testD",
    /* Control Transfer */
    "jmp",
    "je",      "jne",     "jp",      "jnp",
    "ja",      "jae",     "jb",      "jbe",
    "jg",      "jge",     "jl",      "jle",
    "jo",      "jno",     "js",      "jns",
    "jcxz",    "jecxz",
    "loop",    "loope",   "loopne",
    "call",    "ret",     "lret",    "iret",
    "int",     "into",    "boundD",
    "enter",   "leave",
    /* String */
    "movsD",   "cmpsD",   "scasD",   "lodsD",   "stosD",
    "rep",     "repe",    "repne",
    "insD",     "outsS",
    /* Flag Control */
    "stc",     "clc",     "cmc",     "std",     "cld",
    "lahf",    "sahf",
    "pushf",   "pushfd",  "popf",    "popfd",
    "sti",     "cli",
    /* Segment Register Operations */
    "ldsD",    "lesD",    "lfsD",    "lgsD",    "lssD",
    /* Miscellaneous */
    "leaD",    "nop",     "ud2",     "xlatb",
    "cpuid",
    /* Systems Instructions */
    "lgdt",    "sgdt",    "lldt",    "sldt",
    "ltr",     "str",     "lidt",    "sidt",
    "lmsw",    "smsw",    "clts",
    "arpl",    "larD",    "lslD",    "verr",    "verw",
    "invd",    "wbinvd",  "invlpg",
    "lock",    "hlt",     "rsm",
    "rdmsr",   "wrmsr",   "rdpmc",   "rdtsc",
    "sysenter","sysexit",
    /* MMX Data Transfer */
    "movd",    "movq",
    /* MMX Conversion */
    "packsswb",  "packssdw",  "packuswb",
    "punpckhbw", "punpckhwd", "punpckhdq",
    "punpcklbw", "punpcklwd", "punpckldq",
    /* MMX Packed Arithmetic */
    "paddb",   "paddw",   "paddd",
    "paddsb",  "paddsw",  "paddusb", "paddusw",
    "psubb",   "psubw",   "psubd",
    "psubsb",  "psubsw",  "psubusb", "psubusw",
    "pmulhw",  "pmullw",  "pmaddwd",
    /* MMX Comparison */
    "pcmpeqb", "pcmpeqw", "pcmpeqd",
    "pcmpgtb", "pcmpgtw", "pcmpgtd",
    /* MMX Logic */
    "pand",    "pandn",   "por",     "pxor",
    /* MMX Shift and Rotate */
    "psllw",   "pslld",   "psllq",
    "psrlw",   "psrld",   "psrlq",
    "psraw",   "psrad",
    /* MMX State Management */
    "emms",
    /* SSE Floating Point Data Transfer */
    "movss",   "movaps",  "movups",  "movmskps",
    "movlps",  "movhps",  "movhlps", "movlhps",
    /* SSE Floating Point Conversion */
    "cvtpi2ps","cvtps2pi","cvttps2pi",
    "cvtsi2ss","cvtss2si","cvttss2si",
    "unpcklps","unpckhps","shufps",
    /* SSE Floating Point Arithmetic */
    "addps",   "addss",   "subps",   "subss",
    "mulps",   "mulss",   "divps",   "divss",
    "maxps",   "maxss",   "minps",   "minss",
    "sqrtps",  "sqrtss",  "rsqrtps", "rsqrtss",
    "rcpps",   "rcpss",
    /* SSE Floating Point Logic */
    "andps",   "andnps",  "orps",    "xorps",
    /* SSE Floating Point Comparision */
    "cmpps",   "cmpordps","cmpunordps",
    "cmpeqps", "cmpltps", "cmpleps",
    "cmpneqps","cmpnltps","cmpnleps",
    "cmpss",   "cmpordss","cmpunordss",
    "cmpeqss", "cmpltss", "cmpless",
    "cmpneqss","cmpnltss","cmpnless",
    "comiss",  "ucomiss",
    /* SSE Integer Data Transfer */
    "pinsrw",  "pextrw",  "pshufw",  "pmovmskb",
    /* SSE Integer Arithmetic */
    "pmaxub",  "pmaxsw",  "pminub",  "pminsw",
    "pavgb",   "pavgw",
    "pmulhuw", "psadbw",
    /* SSE Cachability Control */
    "movntq",  "movntps", "maskmovq","sfence",
    "prefetcht0", "prefetcht1",
    "prefetcht2", "prefetchnta",
    /* SSE State Management */
    "fxsave",  "fxrstor",
    "ldmxcsr", "stmxcsr",
    /* FPU Data Transfer */
    "fld",     "fst",     "fstp",
    "fild",    "fist",    "fistp",
    "fbld",    "fbstp",   "fxch",
    "fcmove",  "fcmovne", "fcmovu",  "fcmovnu",
    "fcmovb",  "fcmovbe", "fcmovnb", "fcmovnbe",
    /* FPU Basic Arithmetic */
    "fadd",    "faddp",   "fiadd",
    "fsub",    "fsubp",   "fisub",
    "fsubr",   "fsubrp",  "fisubr",
    "fmul",    "fmulp",   "fimul",
    "fdiv",    "fdivp",   "fidiv",
    "fdivr",   "fdivrp",  "fidivr",
    "fprem",   "fprem1",
    "fabs",    "fchs",    "frndint",
    "fscale",  "fsqrt",   "fxtract",
    /* FPU Comparison */
    "fcom",    "fcomp",   "fcompp",
    "fucom",   "fucomp",  "fucompp",
    "ficom",   "ficomp",
    "fcomi",   "fucomi",  "fcomip",  "fucomip",
    "ftst",    "fxam",
    /* FPU Transcendental */
    "fsin",    "fcos",    "fsincos",
    "fptan",   "fpatan",
    "f2xm1",   "fyl2x",   "fyl2xp1",
    /* FPU Load Constants */
    "fld1",    "fldz",    "fldpi",
    "fldl2e",  "fldln2",  "fldl2t",  "fldlg2",
    /* FPU Control */
    "fincstp", "fdecstp", "ffree",
    "finit",   "fninit",  "fclex",   "fnclex",
    "fstcw",   "fnstcw",  "fldcw",
    "fstenv",  "fnstenv", "fldenv",
    "fsave",   "fnsave",  "frstor",
    "fstsw",   "fnstsw",
    "fnop",    "fwait",

    NULL
};

int i386_decode_text_att( struct i386_context *ctx, struct i386_decode *dc, char *string )
{
    int i, k, override;

    *string = '\0';

    /* Instruction prefix */
    if ( dc->prefix )
    {
        assert( dc->prefix < INS_LAST );
        sprintf( string, "%s ", InsNamesATT[dc->prefix]);
    }

    /* Long specifier */
    if ( dc->op[0].mode == OM_IMMEDIATE && dc->op[0].u.immediate.segment )
        strcat( string, "l" );

    /* Instruction name */
    assert( dc->instruction < INS_LAST );
    strcat(string, InsNamesATT[dc->instruction]);

    /* Operand size */
    for ( i = 0; string[i]; i++ )
        switch ( string[i] )
        {

        case 'D':  switch ( dc->op[0].size )
        {
            case 1: string[i] = 'b'; break;
            case 2: string[i] = 'w'; break;
            case 4: string[i] = 'l'; break;
        }
        break;

        case 'S':  switch ( dc->op[1].size )
        {
            case 1: string[i] = 'b'; break;
            case 2: string[i] = 'w'; break;
            case 4: string[i] = 'l'; break;
        }
        break;

        }

    /* Formatting */
    if (strlen(string) < 7)
    {
        strcat(string, "       ");
        string[7] = '\0';
    }

    /* Check whether to display segment overrides */
    override = (dc->segment_override != 0);
    if (dc->op[0].mode == OM_MEMORY && dc->op[0].u.memory.segment == SEG_ES)
        override = TRUE;
    if (dc->op[1].mode == OM_MEMORY && dc->op[1].u.memory.segment == SEG_ES)
        override = TRUE;


    /* Arguments */
    for ( k = 0, i = 2; i >= 0; i-- )
    {
        struct i386_decode_operand *da;

        /* Get operand */
        if ( dc->instruction == INS_enter )
            da = dc->op + 2 - i;
        else
            da = dc->op + i;
        if (!da->type) continue;

        strcat(string, k? ", " : " ");
        k++;

        /* Output data */
        switch (da->mode)
        {
        case OM_REGISTER:
            assert( da->u.reg < P86_REG_LAST );
            if( dc->instruction == INS_call || dc->instruction == INS_jmp )
                strcat( string, "*" );

            strcat( string, RegNamesATT[da->u.reg] );
            break;

        case OM_IMMEDIATE:
        {
            int size = da->size;

            if ( da->u.immediate.segment )
            {
                size -= 2;
                sprintf( string+strlen(string), "$0x%04x, ", da->u.immediate.segment );
            }

            switch ( size )
            {
            case 1: sprintf( string+strlen(string), "$0x%02x", da->u.immediate.offset ); break;
            case 2: sprintf( string+strlen(string), "$0x%04x", da->u.immediate.offset ); break;
            case 4: sprintf( string+strlen(string), "$0x%08x", da->u.immediate.offset ); break;
            }
        }
        break;

        case OM_MEMORY:
        {
            struct i386_decode_address *addr = &da->u.memory;
            int value = addr->displacement;

            /* Output segment override */
            if ( override )
            {
                assert( addr->segment < P86_REG_LAST );
                sprintf( string+strlen(string), "%s:", RegNamesATT[addr->segment] );
            }

            /* Output displacement */
            if ( value < 0 && value >= -32768 )
            {
                strcat( string, "-" ); value = -value;
            }
            switch ( dc->displacement_size )
            {
                case 0: /* no displacement */ break;
                case 1: sprintf( string+strlen(string), "%02x", value & 0xff ); break;
                case 2: sprintf( string+strlen(string), "%04x", value & 0xffff ); break;
                case 4: sprintf( string+strlen(string), "%08x", value ); break;
                default: assert( FALSE );
            }

            if( addr->base || addr->index )
                strcat( string, "(" );

            /* Output base */
            if ( addr->base )
            {
                assert( addr->base < P86_REG_LAST );
                strcat( string, RegNamesATT[addr->base]);
            }

            /* Output index and scale */
            if ( addr->index )
            {
                assert( addr->index < P86_REG_LAST );
                sprintf( string+strlen(string), ",%s", RegNamesATT[addr->index] );

                if ( addr->scale > 1 )
                    sprintf( string+strlen(string), ",%d", addr->scale );
            }

            if( addr->base || addr->index )
                strcat( string, ")" );
        }
        break;
        }
    }
    return TRUE;
}


int i386_decode_text( struct i386_context *ctx, struct i386_decode *dc, char *string, enum i386_asm_syntax syntax )
{
    assert( syntax < SX_NONE );

    switch (syntax)
    {
    case SX_INTEL:
        return i386_decode_text_intel (ctx, dc, string);
    case SX_ATT:
        return i386_decode_text_att (ctx, dc, string);
    default:
        return FALSE;
    }
}
