/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  stack_pro.c:  stack oriented framework operations
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  unsigned
can_push(vm_t *vm, descriptor_cache_t *ss_cache, Bit32u esp, unsigned bytes)
{
  /* No need for cache_sreg(), we're being passed a cache */

  if ( !RealMode(vm) ) { /* CR0.PE==1, EFLAGS.VM==0 */
    unsigned d_b;

    d_b = ss_cache->desc.d_b;

    /* small stack compares against 16-bit SP */
    if (!d_b)
      esp &= 0x0000ffff;

    if (ss_cache->desc.p==0) {
      monpanic(vm, "can_push: not present\n");
      return(0);
      }

    if (ss_cache->desc.type & D_EXDOWN) { /* expand down segment */
      Bit32u expand_down_limit;

      if (d_b)
        expand_down_limit = 0xffffffff;
      else
        expand_down_limit = 0x0000ffff;

      if (esp==0) {
        monpanic(vm, "can_push(): esp=0, wraparound?\n");
        return(0);
        }

      if (esp < bytes) {
        monpanic(vm, "can_push(): expand-down: esp < N\n");
        return(0);
        }
      if ( (esp - bytes) <= ss_cache->limit_scaled ) {
        monpanic(vm, "can_push(): expand-down: esp-N < limit\n");
        return(0);
        }
      if ( esp > expand_down_limit ) {
        monpanic(vm, "can_push(): esp > expand-down-limit\n");
        return(0);
        }
      return(1);
      }
    else { /* normal (expand-up) segment */
      if (ss_cache->limit_scaled==0) {
        monpanic(vm, "can_push(): found limit of 0\n");
        return(0);
        }

      /* Look at case where esp==0.  Possibly, it's an intentional wraparound */
      /* If so, limit must be the maximum for the given stack size */
      if (esp==0) {
        if (d_b && (ss_cache->limit_scaled==0xffffffff))
          return(1);
        if ((d_b==0) && (ss_cache->limit_scaled>=0xffff))
          return(1);
        monpanic(vm, "can_push(): esp=0, normal, wraparound? limit=%08x\n",
          ss_cache->limit_scaled);
        return(0);
        }

      if (esp < bytes) {
        monpanic(vm, "can_push(): expand-up: esp < N\n");
        return(0);
        }
      if ((esp-1) > ss_cache->limit_scaled) {
        monpanic(vm, "can_push(): expand-up: SP > limit\n");
        return(0);
        }
      return(1);
      }
    }
  else {
    monpanic(vm, "can_push: RM\n");
    /* need to fill in for real/virtual mode */
#if 0
/* from stack_pro.cc */
if ((ESP>=1) && (ESP<=3)) {
  monpanic(vm, "push_32: ESP=%08x\n", (unsigned) ESP);
  }
#endif
    }
  return(0);
}

  unsigned
can_pop(vm_t *vm, unsigned bytes)
{
  Bit32u temp_ESP, expand_down_limit;

  /* +++ needed? */
  if (RealMode(vm)) monpanic(vm, "can_pop: called in real mode?\n");

  /* +++ Need this call ? */
  cache_sreg(vm, SRegSS);
  if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b) { /* use ESP */
    temp_ESP = G_ESP(vm);
    expand_down_limit = 0xFFFFFFFF;
    }
  else { /* use SP */
    temp_ESP = G_SP(vm);
    expand_down_limit = 0xFFFF;
    }

  if (vm->guest_cpu.desc_cache[SRegSS].valid==0) {
    monpanic(vm, "can_pop: SS invalidated.\n");
    }

  if (vm->guest_cpu.desc_cache[SRegSS].desc.p==0) {
    monpanic(vm, "can_pop: SS.p = 0\n");
    }


  if (vm->guest_cpu.desc_cache[SRegSS].desc.type & D_EXDOWN) { /* expand down */
    if ( temp_ESP == expand_down_limit ) {
      monpanic(vm, "can_pop: found SP=ffff\n");
      }
    if ( ((expand_down_limit - temp_ESP) + 1) >= bytes )
      return(1);
    return(0);
    }
  else { /* normal (expand-up) segment */
    if (vm->guest_cpu.desc_cache[SRegSS].limit_scaled==0) {
      monpanic(vm, "can_pop: SS.limit = 0\n");
      }
    if ( temp_ESP == expand_down_limit ) {
      monpanic(vm, "can_pop: found SP=ffff\n");
      }
    if ( temp_ESP > vm->guest_cpu.desc_cache[SRegSS].limit_scaled ) {
      monpanic(vm, "can_pop: eSP > SS.limit\n");
      }
    if ( ((vm->guest_cpu.desc_cache[SRegSS].limit_scaled - temp_ESP) + 1) >= bytes )
      return(1);
    return(0);
    }
}

  void
push32(vm_t *vm, Bit32u val32)
{
  Bit32u old_esp, new_esp;
  unsigned d_b;

  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b) {
    old_esp = vm->guest.addr.guest_context->esp;
    new_esp = old_esp - 4;
    }
  else {
    old_esp = G_SP(vm);
    new_esp = (old_esp - 4) & 0x0000ffff;
    }
  if (ProtectedMode(vm)) {
    if (!can_push(vm, &vm->guest_cpu.desc_cache[SRegSS], old_esp, 4)) {
      monpanic(vm, "push32: cant push 4\n");
      }
    }
  else {
    if ((old_esp>=1) && (old_esp<=3))
      monpanic(vm, "push32: ESP=%08x\n", (unsigned) old_esp);
    }
  write_virtual_dword(vm, SRegSS, new_esp, &val32);
  if (d_b) {
    /* modify ESP (all 32 bits) */
    G_ESP(vm) = new_esp;
    }
  else {
    /* only modify SP (lower 16 bits) */
    G_SP(vm) = new_esp;
    }
}

  void
push16(vm_t *vm, Bit16u val16)
{
  Bit32u old_esp, new_esp;
  unsigned d_b;

  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b) {
    old_esp = vm->guest.addr.guest_context->esp;
    new_esp = old_esp - 2;
    }
  else {
    old_esp = G_SP(vm);
    new_esp = (old_esp - 2) & 0x0000ffff;
    }
  if (ProtectedMode(vm)) {
    if (!can_push(vm, &vm->guest_cpu.desc_cache[SRegSS], old_esp, 2)) {
      monpanic(vm, "push16: cant push 2\n");
      }
    }
  else {
    if (old_esp == 1)
      monpanic(vm, "CPU shutting down due to lack of space, eSP==1\n");
    }
  write_virtual_word(vm, SRegSS, new_esp, &val16);
  if (d_b) {
    /* modify ESP (all 32 bits) */
    G_ESP(vm) = new_esp;
    }
  else {
    /* only modify SP (lower 16 bits) */
    G_SP(vm) = new_esp;
    }
}


  void
pop32(vm_t *vm, Bit32u *val32)
{
  Bit32u temp_ESP;
  unsigned d_b;

  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);

  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 4) ) {
      monpanic(vm, "pop32: can't pop from stack\n");
      exception(vm, ExceptionSS, 0);
      }
    }

  read_virtual_dword(vm, SRegSS, temp_ESP, val32);

  if (d_b)
    G_ESP(vm) += 4;
  else
    G_SP(vm) += 4;
}

  void
pop16(vm_t *vm, Bit16u *val16)
{
  Bit32u temp_ESP;
  unsigned d_b;

  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);

  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 2) ) {
      monpanic(vm, "pop16: can't pop from stack\n");
      exception(vm, ExceptionSS, 0);
      }
    }

  read_virtual_word(vm, SRegSS, temp_ESP, val16);

  if (d_b)
    G_ESP(vm) += 2;
  else
    G_SP(vm) += 2;
}
