/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  protect_ctrl.c:  emulation of instructions which deal with
 *    guest IA32 protection features.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
ARPL_EwGw(vm_t *vm)
{
  Bit16u op2_16, op1_16;
 
  if (ProtectedMode(vm)) {
    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
      }
 
    op2_16 = ReadReg16(vm, vm->i.nnn);
 
    if ( (op1_16 & 0x03) < (op2_16 & 0x03) ) {
      op1_16 = (op1_16 & 0xfffc) | (op2_16 & 0x03);
      /* now write back to destination */
      if (vm->i.mod == 0xc0) {
        if (vm->i.os_32) {
          /* if 32bit opsize, then 0xff3f is or'd into
           * upper 16bits of register */
          Bit32u op1_32;
 
          op1_32 = ReadReg32(vm, vm->i.rm);
          op1_32 = (op1_32 & 0xffff0000) | op1_16;
          op1_32 |= 0xff3f0000;
          WriteReg32(vm, vm->i.rm, op1_32);
          }
        else {
          WriteReg16(vm, vm->i.rm, op1_16);
          }
        }
      else {
        write_RMW_virtual_word(vm, &op1_16);
        }
      G_SetZF(vm, 1);
      }
    else {
      G_SetZF(vm, 0);
      }
    }
  else {
    /* ARPL not recognized in real or v8086 mode */
    UndefinedOpcode(vm);
    }
}


  void
LGDT_Ms(vm_t *vm)
{
  Bit32u base;
  Bit16u limit;

  /* If source op is not a memory location, #UD */
  if (vm->i.mod == 0xc0) {
    monpanic(vm, "LGDT: modrm is reg\n");
    exception(vm, ExceptionUD, 0);
    }
  if (ProtectedMode(vm) && (vm->guest_cpu.cpl!=0)) {
    monpanic(vm, "LGDT: cpl!=0\n");
    exception(vm, ExceptionGP, 0);
    }

  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &limit);
  read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr+2, &base);

  if (!vm->i.os_32) {
    /* 24-bit base for 16bit opsize */
    base &= 0x00ffffff;
    }
  vm->guest_cpu.gdtr.base  = base;
  vm->guest_cpu.gdtr.limit = limit;
  virtualize_lconstruct(vm, base, base+limit, PagePermEmulate);
}

  void
LIDT_Ms(vm_t *vm)
{
  Bit16u limit_16;
  Bit32u base_32;

  if (V8086Mode(vm)) monpanic(vm, "protect_ctrl: v8086 mode\n");

  if (ProtectedMode(vm)) {
    if (G_CPL(vm) != 0) {
      monpanic(vm, "LIDT(): CPL(%u) != 0\n", (unsigned) G_CPL(vm));
      exception(vm, ExceptionGP, 0);
      }
    }

  if (vm->i.mod == 0xc0) {
    monpanic(vm, "LIDT generating exception 6\n");
    UndefinedOpcode(vm);
    }

  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &limit_16);
  read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr + 2, &base_32);

  if (vm->i.os_32==0)
    base_32 &= 0x00ffffff; /* ignore upper 8 bits */
  vm->guest_cpu.idtr.limit = limit_16;
  vm->guest_cpu.idtr.base = base_32;
  virtualize_lconstruct(vm, base_32, base_32+limit_16, PagePermEmulate);
}

  void
LTR_Ew(vm_t *vm)
{
  if (V8086Mode(vm)) monpanic(vm, "LTR_Ew: v8086\n");

  if (ProtectedMode(vm)) {
    descriptor_cache_t  cache;
    selector_t          selector;


    /* #GP(0) if the current privilege level is not 0 */
    if (G_CPL(vm) != 0) {
      monpanic(vm, "LTR: CPL != 0\n");
      exception(vm, ExceptionGP, 0);
      }

    if (vm->i.mod == 0xc0) {
      selector.raw = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector);
      }

    /* if selector is NULL, invalidate and done */
    if (IsNullSelector(selector)) {
      monpanic(vm, "ltr: loading with NULL selector!\n");
      /* if this is OK, then invalidate and load selector & descriptor cache */
      /* load here */
      vm->guest_cpu.tr_selector.raw = 0;
      vm->guest_cpu.tr_cache.valid = 0;
      return;
      }

    if (selector.fields.ti) {
      monpanic(vm, "ltr: selector.ti != 0\n");
      }

    fetch_raw_descriptor(vm, selector, &cache.desc, ExceptionGP);
    descriptor2cache(vm, &cache);

    /* #GP(selector) if object is not a TSS or is already busy */
    if ( (cache.valid==0) ||
         (cache.desc.type!=1 && cache.desc.type!=9) ) {
      monpanic(vm, "ltr: doesn't point to an available TSS descriptor!\n");
      exception(vm, ExceptionGP, selector.raw & 0xfffc); /* 0 ??? */
      }

    /* #NP(selector) if TSS descriptor is not present */
    if (cache.desc.p==0) {
      monpanic(vm, "ltr: LDT descriptor not present!\n");
      exception(vm, ExceptionNP, selector.raw & 0xfffc); /* 0 ??? */
      }

    if ((cache.desc.type==1) && (cache.limit_scaled<43)) {
      monpanic(vm, "ltr:286TSS: loading tr.limit < 43\n");
      }
    else if ((cache.desc.type==9) && (cache.limit_scaled<103)) {
      monpanic(vm, "ltr:386TSS: loading tr.limit < 103\n");
      }

    vm->guest_cpu.tr_selector = selector;
    vm->guest_cpu.tr_cache = cache;

    cache.desc.type |= 0x2; /* Set busy bit */
    access_linear(vm, vm->guest_cpu.gdtr.base + selector.fields.index*8 + 5,
                  1, 0, OP_WRITE, ((Bit8u *)&cache.desc)+5);
    }
  else {
    monpanic(vm, "ltr_ew: not recognized in real-mode!\n");
    UndefinedOpcode(vm);
    }
}

  void
LLDT_Ew(vm_t *vm)
{
  if (V8086Mode(vm))
    monpanic(vm, "LLDT_Ew: v8086 mode unsupported\n");

  invalidate_prefetch_q();

  if (RealMode(vm)) {
    monpanic(vm, "lldt: not recognized in real mode\n");
    UndefinedOpcode(vm);
    }
  else {
    descriptor_cache_t cache;
    selector_t         selector;
    Bit32u base;

    /* #GP(0) if the current privilege level is not 0 */
    if (G_CPL(vm) != 0) {
      monpanic(vm, "LLDT: CPL != 0\n");
      exception(vm, ExceptionGP, 0);
      }

    if (vm->i.mod == 0xc0) {
      selector.raw = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector.raw);
      }

    /* if selector is NULL, invalidate and done */
    if (IsNullSelector(selector)) {
      /* +++ does this load 0 or the requested selector? */
      vm->guest_cpu.ldtr_selector.raw = selector.raw;
      vm->guest_cpu.ldtr_cache.valid = 0;
      return;
      }

    /* #GP(selector) if the selector operand does not point into GDT */
    if (selector.fields.ti != 0) {
      monprint(vm, "LLDT: selector.ti != 0\n");
      exception(vm, ExceptionGP, selector.raw & 0xfffc);
      }

    fetch_raw_descriptor(vm, selector, &cache.desc, ExceptionGP);
    descriptor2cache(vm, &cache);

    /* if selector doesn't point to an LDT descriptor #GP(selector) */
    if ( !cache.valid ||
         (cache.desc.type!=2) ) {
      monprint(vm, "lldt: doesn't point to an LDT descriptor!\n");
      exception(vm, ExceptionGP, selector.raw & 0xfffc);
      }

    /* #NP(selector) if LDT descriptor is not present */
    if (cache.desc.p==0) {
      monprint(vm, "lldt: LDT descriptor not present!\n");
      exception(vm, ExceptionNP, selector.raw & 0xfffc);
      }

    if (cache.limit_scaled < 7) {
      monpanic(vm, "lldt: ldtr.limit < 7\n");
      /* +++ */
      }

    vm->guest_cpu.ldtr_selector = selector;
    vm->guest_cpu.ldtr_cache = cache;
    base = cache.base;
    virtualize_lconstruct(vm, base, base+cache.limit_scaled,
                          PagePermEmulate);
    }
}

  void
SLDT_Ew(vm_t *vm)
{
  Bit16u val16;

  if ( !ProtectedMode(vm) ) {
    monpanic(vm, "SLDT_Ew: not PM.\n");
    UndefinedOpcode(vm);
    }

  val16 = vm->guest_cpu.ldtr_selector.raw;
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, val16);
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &val16);
    }
}

  void
STR_Ew(vm_t *vm)
{
  Bit16u val16;

  if ( !ProtectedMode(vm) ) {
    monpanic(vm, "STR_Ew: not PM.\n");
    UndefinedOpcode(vm);
    }

  val16 = vm->guest_cpu.tr_selector.raw;
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, val16);
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &val16);
    }
}

  void
LAR_GvEw(vm_t *vm)
{
  descriptor_cache_t cache;
  selector_t         selector;
  Bit32u dword2;

  if (!ProtectedMode(vm)) {
    monpanic(vm, "LAR: not recognized in real mode\n");
    UndefinedOpcode(vm);
    }

  if (vm->i.mod == 0xc0) {
    selector.raw = ReadReg16(vm, vm->i.rm);
    }
  else {
    /* pointer, segment address pair */
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector.raw);
    }

  /* if selector null, clear ZF and done */
  if ( IsNullSelector(selector) ) {
    G_SetZF(vm, 0);
    return;
    }

  if ( !fetch_descriptor2(vm, selector, &cache.desc) ) {
    /* Not within descriptor table */
    G_SetZF(vm, 0);
    return;
    }
  dword2 = ((Bit32u *) &cache.desc)[1];

  descriptor2cache(vm, &cache);

  if (!cache.valid) {
    G_SetZF(vm, 0);
    return;
    }

  /* if source selector is visible at CPL & RPL,
   * within the descriptor table, and of type accepted by LAR instruction,
   * then load register with segment limit and set ZF
   */

  if ( cache.desc.type & D_S ) { /* normal segment */
    if ( (cache.desc.type & (D_EXECUTE | D_CONFORM)) ==
         (D_EXECUTE | D_CONFORM) ) {
      /* ignore DPL for conforming segments */
      }
    else {
      if ( (cache.desc.dpl < G_CPL(vm)) ||
           (cache.desc.dpl < selector.fields.rpl) ) {
        G_SetZF(vm, 0);
        return;
        }
      }
    G_SetZF(vm, 1);
    if (vm->i.os_32) {
      /* masked by 00FxFF00, where x is undefined */
      WriteReg32(vm, vm->i.nnn, dword2 & 0x00ffff00);
      }
    else {
      WriteReg16(vm, vm->i.nnn, dword2 & 0xff00);
      }
    return;
    }
  else { /* system or gate segment */
    switch ( cache.desc.type ) {
      case 1: /* available TSS */
      case 2: /* LDT */
      case 3: /* busy TSS */
      case 4: /* 286 call gate */
      case 5: /* task gate */
      case 9:  /* available 32bit TSS */
      case 11: /* busy 32bit TSS */
      case 12: /* 32bit call gate */
        break;
      default: /* rest not accepted types to LAR */
        G_SetZF(vm, 0);
        monprint(vm, "lar(): not accepted type\n");
        return;
      }

    if ( (cache.desc.dpl < G_CPL(vm)) ||
         (cache.desc.dpl < selector.fields.rpl) ) {
      G_SetZF(vm, 0);
      return;
      }
    G_SetZF(vm, 1);
    if (vm->i.os_32) {
      /* masked by 00FxFF00, where x is undefined ??? */
      WriteReg32(vm, vm->i.nnn, dword2 & 0x00ffff00);
      }
    else {
      WriteReg16(vm, vm->i.nnn, dword2 & 0xff00);
      }
    return;
    }
}

  void
LSL_GvEw(vm_t *vm)
{
  Bit32u limit32;
  descriptor_cache_t cache;
  selector_t   selector;
  Bit32u dword1, dword2;

  if (!ProtectedMode(vm)) {
    monpanic(vm, "LSL: not recognized in real mode\n");
    UndefinedOpcode(vm);
    }

  if (vm->i.mod == 0xc0) {
    selector.raw = ReadReg16(vm, vm->i.rm);
    }
  else {
    /* pointer, segment address pair */
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector.raw);
    }


  /* if selector null, clear ZF and done */
  if ( IsNullSelector(selector) ) {
    G_SetZF(vm, 0);
    return;
    }

  if ( !fetch_descriptor2(vm, selector, &cache.desc) ) {
    /* not within descriptor table */
    G_SetZF(vm, 0);
    return;
    }

  dword1 = ((Bit32u *) &cache.desc)[0];
  dword2 = ((Bit32u *) &cache.desc)[1];

/* +++ convert this over to using fields rather than raw bits */
  if ( (dword2 & 0x00001000) == 0 ) { /* system segment */
    Bit32u type;

    type = (dword2 >> 8) & 0x0000000f;
    switch (type) {
      case 1: /* 16bit TSS */
      case 3: /* 16bit TSS */
      case 2: /* LDT */
      case 9: /* 32bit TSS */
      case 11:/* 32bit TSS */
        limit32 = (dword1 & 0x0000ffff) | (dword2 & 0x000f0000);
        if ( dword2 & 0x00800000 )
          limit32 = (limit32 << 12) | 0x00000fff;
        if ( (cache.desc.dpl < G_CPL(vm)) ||
             (cache.desc.dpl < selector.fields.rpl) ) {
          G_SetZF(vm, 0);
          return;
          }
        goto lsl_ok;
        break;
      default:
        G_SetZF(vm, 0);
        return;
      }
    }
  else { /* data & code segment */
    limit32 = (dword1 & 0x0000ffff) | (dword2 & 0x000f0000);
    if ( dword2 & 0x00800000 )
      limit32 = (limit32 << 12) | 0x00000fff;
    if ( (dword2 & 0x00000c00) == 0x00000c00 ) {
      /* conforming code segment, no check done */
      goto lsl_ok;
      }

    if ( (cache.desc.dpl < G_CPL(vm)) ||
         (cache.desc.dpl < selector.fields.rpl) ) {
      G_SetZF(vm, 0);
      return;
      }
    goto lsl_ok;
    }

lsl_ok:
  /* all checks pass, limit32 is now byte granular, write to op1 */
  G_SetZF(vm, 1);

  if (vm->i.os_32)
    WriteReg32(vm, vm->i.nnn, limit32);
  else
    /* chop off upper 16 bits */
    WriteReg16(vm, vm->i.nnn, (Bit16u) limit32);
}

  void
VERW_Ew(vm_t *vm)
{
  /* for 16 bit operand size mode */
  descriptor_cache_t cache;
  selector_t selector;

  if (V8086Mode(vm))
    monpanic(vm, "VERW: v8086 mode\n");

  if (RealMode(vm)) {
    monpanic(vm, "VERW: not recognized in real mode\n");
    UndefinedOpcode(vm);
    }

  if (vm->i.mod == 0xc0) {
    selector.raw = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector.raw);
    }

  /* if selector null, clear ZF and done */
  if ( IsNullSelector(selector) ) {
    G_SetZF(vm, 0);
    monprint(vm, "VERW: null selector\n");
    return;
    }

  /* if source selector is visible at CPL & RPL,
   * within the descriptor table, and of type accepted by VERW instruction,
   * then load register with segment limit and set ZF */
  if ( !fetch_descriptor2(vm, selector, &cache.desc) ) {
    /* not within descriptor table */
    G_SetZF(vm, 0);
    monprint(vm, "VERW: not in table\n");
    return;
    }

  descriptor2cache(vm, &cache);

  /* rule out system segments & code segments */
  if ( !(cache.desc.type & D_S) ||
       (cache.desc.type & D_EXECUTE) ) {
    G_SetZF(vm, 0);
    monprint(vm, "VERW: system seg or code\n");
    return;
    }

  if ( cache.valid==0 ) {
    G_SetZF(vm, 0);
    monprint(vm, "VERW: valid bit cleared\n");
    return;
    }

  /* data segment */
  if ( cache.desc.type & D_WRITE ) { /* writable */
    if ( (cache.desc.dpl < G_CPL(vm)) ||
         (cache.desc.dpl < selector.fields.rpl) ) {
      G_SetZF(vm, 0); /* not accessible */
      monprint(vm, "VERW: writable data seg not within priv level\n");
      return;
      }
    G_SetZF(vm, 1); /* accessible */
    monprint(vm, "VERW: data seg writable\n");
    return;
    }

  G_SetZF(vm, 0); /* not accessible */
  monprint(vm, "VERW: data seg not writable\n");
}

  void
VERR_Ew(vm_t *vm)
{
  /* for 16 bit operand size mode */
  descriptor_cache_t cache;
  selector_t   selector;

  if (V8086Mode(vm))
    monpanic(vm, "VERR: v8086 mode\n");

  if (RealMode(vm)) {
    monpanic(vm, "VERR: not recognized in real mode\n");
    UndefinedOpcode(vm);
    }

  if (vm->i.mod == 0xc0) {
    selector.raw = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &selector.raw);
    }

  /* if selector null, clear ZF and done */
  if ( IsNullSelector(selector) ) {
    G_SetZF(vm, 0);
    monprint(vm, "VERR: null selector\n");
    return;
    }

  /* if source selector is visible at CPL & RPL,
   * within the descriptor table, and of type accepted by VERR instruction,
   * then load register with segment limit and set ZF */
  if ( !fetch_descriptor2(vm, selector, &cache.desc) ) {
    /* not within descriptor table */
    G_SetZF(vm, 0);
    monprint(vm, "VERR: not in table\n");
    return;
    }

  descriptor2cache(vm, &cache);

  if ( !(cache.desc.type & D_S) ) { /* system or gate descriptor */
    G_SetZF(vm, 0); /* inaccessible */
    monprint(vm, "VERR: system descriptor\n");
    return;
    }

  if ( cache.valid==0 ) {
    G_SetZF(vm, 0);
    monprint(vm, "VERR: valid bit cleared\n");
    return;
    }

  /* normal data/code segment */
  if ( cache.desc.type & D_EXECUTE ) { /* code segment */
    /* ignore DPL for readable conforming segments */
    if ( (cache.desc.type & D_CONFORM) &&
         (cache.desc.type & D_READ) ) {
      G_SetZF(vm, 1); /* accessible */
      monprint(vm, "VERR: conforming code, OK\n");
      return;
      }
    if ( !(cache.desc.type & D_READ) ) {
      G_SetZF(vm, 0); /* inaccessible */
      monprint(vm, "VERR: code not readable\n");
      return;
      }
    /* readable, non-conforming code segment */
    if ( (cache.desc.dpl < G_CPL(vm)) ||
         (cache.desc.dpl < selector.fields.rpl) ) {
      G_SetZF(vm, 0); /* inaccessible */
      monprint(vm, "VERR: non-conforming code not within priv level\n");
      return;
      }
    G_SetZF(vm, 1); /* accessible */
    monprint(vm, "VERR: code seg readable\n");
    return;
    }
  else { /* data segment */
    if ( (cache.desc.dpl < G_CPL(vm)) ||
         (cache.desc.dpl < selector.fields.rpl) ) {
      G_SetZF(vm, 0); /* not accessible */
      monprint(vm, "VERR: data seg not within priv level\n");
      return;
      }
    G_SetZF(vm, 1); /* accessible */
    monprint(vm, "VERR: data segment OK\n");
    return;
    }
}
