/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file status_layer.cpp
 * \brief Implementation of the ptb::status_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/status_layer.hpp"

#include "engine/game.hpp"
#include "visual/bitmap_writing.hpp"
#include "visual/scene_sprite.hpp"
#include "visual/scene_writing.hpp"
#include "visual/scene_line.hpp"
#include "ptb/player.hpp"
#include "ptb/level_variables.hpp"
#include "ptb/throwable_item/throwable_items_container.hpp"
#include "ptb/defines.hpp"
#include "ptb/gauge.hpp"
#include "ptb/util.hpp"

/*----------------------------------------------------------------------------*/
const double ptb::status_layer::player_status::s_bar_length = 100;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param glob Level globals in which we take the resources.
 * \param f The font to use to display the texts.
 * \param p The player from which we take the data.
 */
ptb::status_layer::player_status::player_status
( bear::engine::level_globals& glob, const bear::visual::font& f,
  bear::universe::const_derived_item_handle_maker<player>::handle_type& p )
  : energy( glob, s_bar_length, "bar (green)", "bar (red)", "heart", true ),
    oxygen( glob, s_bar_length, "bar (blue)", "bubble" ),
    fire_gauge( glob, s_bar_length, "bar (yellow)", "sun" ),
    ice_gauge( glob, s_bar_length, "bar (white)", "snowflake" ),
    m_player(p),
    oxygen_active(false), oxygen_offset_x(oxygen.width() + 5),
    fire_gauge_active(false), fire_gauge_offset_x(fire_gauge.width() + 5),
    ice_gauge_active(false), ice_gauge_offset_x(ice_gauge.width() + 5)
{
  score.create(f, "0pt");
  throwable_items.create(f, "0");

  m_air_power = false;
  m_fire_power = false;
  m_water_power = false;

  m_last_tries = game_variables::get_lives_count(m_player->get_index());
  m_tries_time = 0;

  if ( p.get() )
    update_throwable_item_animation();
} // status_layer::player_status::player_status()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the values.
 * \param glob The level globals in which we load the media.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::status_layer::player_status::progress
(  bear::engine::level_globals& glob, bear::universe::time_type elapsed_time )
{
  energy.progress(elapsed_time);
  oxygen.progress(elapsed_time);
  fire_gauge.progress(elapsed_time);
  ice_gauge.progress(elapsed_time);

  if (m_player.get())
    {
      energy.set_length
        (game_variables::get_max_energy(m_player->get_index()));
      energy.set_max_level
        (game_variables::get_max_energy(m_player->get_index()));
      energy.set_level( m_player->get_energy() );

      progress_gauge
        (oxygen, m_player->get_gauge(player::oxygen_gauge)->is_activated(),
         m_player->get_gauge(player::oxygen_gauge)->get_value(),
         player::s_max_oxygen, oxygen_active, oxygen_offset_x);

      progress_gauge
        (fire_gauge, m_player->get_gauge(player::fire_gauge)->is_activated(),
         m_player->get_gauge(player::fire_gauge)->get_value(),
         player::s_max_fire_gauge, fire_gauge_active, fire_gauge_offset_x);

      progress_gauge
        (ice_gauge, m_player->get_gauge(player::ice_gauge)->is_activated(),
         m_player->get_gauge(player::ice_gauge)->get_value(),
         player::s_max_ice_gauge, ice_gauge_active, ice_gauge_offset_x);

      bear::visual::font f = glob.get_font("font/bouncy.fnt");

      std::ostringstream oss1;
      oss1 << game_variables::get_score(m_player->get_index());
      score.create(f, oss1.str());
      score->set_intensity(1, 0.8, 0);

      std::ostringstream oss2;
      oss2 << m_player->get_throwable_items()->get_current_throwable_item
        ()->get_stock();
      throwable_items.create(f, oss2.str());
      throwable_items->set_intensity(1, 0.8, 0);

      update_tries(glob, elapsed_time, m_player->get_index());

      if ( update_throwable_item(m_player->get_index()) )
        update_throwable_item_animation();

      throwable_item_animation.next(elapsed_time);
    }
} // status_layer::player_status::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the throwable_item.
 * Return true if the powers have changed.
 * \param index The index of the player.
 */
bool ptb::status_layer::player_status::update_throwable_item
( unsigned int index )
{
  bool result = false;

  if ( m_air_power != game_variables::get_air_power(index) )
    {
      m_air_power = game_variables::get_air_power(index);
      result = true;
    }

  if ( m_fire_power != game_variables::get_fire_power(index) )
    {
      m_fire_power = game_variables::get_fire_power(index);
      result = true;
    }

  if ( m_water_power != game_variables::get_water_power(index) )
    {
      m_water_power = game_variables::get_water_power(index);
      result = true;
    }

  if ( m_last_throwable_item !=
       m_player->get_throwable_items()->get_current_throwable_item() )
    {
      m_last_throwable_item =
        m_player->get_throwable_items()->get_current_throwable_item();
      result = true;
    }

  if ( m_last_throw_status !=
       m_player->get_throwable_items()->get_current_throwable_item
       ()->can_throw() )
    {
      m_last_throw_status =
        m_player->get_throwable_items()->get_current_throwable_item
        ()->can_throw();
      result = true;
    }

  return result;
} // player_status::update_powers

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the tries count.
 * \param glob The level globals in which we load the media.
 * \param index The index of the player.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::status_layer::player_status::update_tries
( bear::engine::level_globals& glob, bear::universe::time_type elapsed_time,
  unsigned int index )
{
  unsigned int nb = game_variables::get_lives_count(m_player->get_index());

  if ( nb > m_last_tries )
    {
      m_last_tries = nb;
      m_tries_time = elapsed_time;
    }
  else if ( m_tries_time != 0 )
    {
      m_tries_time += elapsed_time;
      if ( m_tries_time > 1 )
        m_tries_time = 0;
    }
  else
    m_last_tries = nb;

  std::ostringstream oss;
  oss << nb;
  bear::visual::font f = glob.get_font("font/bouncy.fnt");
  tries.create(f, oss.str());

  double red = 1;
  double green = 0.8;

  if ( m_tries_time != 0 )
    {
      if ( m_tries_time > 0.5 )
        {
          red = m_tries_time;
          green = 2*(m_tries_time-0.5);
        }
      else if ( m_tries_time > 0 )
        {
          red = 1 - m_tries_time;
          green = 0.8 - 1.6*m_tries_time;
        }
    }
  tries->set_intensity(red, green, 0);
} // player_status::update_tries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the throwable_item animation.
 * \param glob The level globals in which we load the media.
 * \param index The index of the player.
 */
void ptb::status_layer::player_status::update_throwable_item_animation()
{
  throwable_item_animation =
    m_player->get_throwable_items()
    ->get_current_throwable_item()->get_animation();
} // player_status::update_stone_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update a gauge.
 * \param gauge The gauge to update.
 * \param is_active Indicates if the gauge must be active.
 * \param value The level of the gauge.
 * \param max_value The max level of the gauge.
 * \param new_active (out) The status of gauge activation.
 * \param offset_x (out) The offset of the gauge in x axis.
 */
void ptb::status_layer::player_status::progress_gauge
( vertical_gauge& gauge, bool is_active, double value, double max_value,
  bool& new_active, bear::visual::coordinate_type& offset_x)
{
  gauge.set_level( s_bar_length * value / max_value );

  if ( is_active )
    {
      new_active = true;

      if ( offset_x != 0 )
        --offset_x;
    }
  else
    {
      if ( offset_x != gauge.width() + 5 )
        ++offset_x;
      else
        new_active = false;
    }
} // status_layer::player_status::progress_gauge()




/*----------------------------------------------------------------------------*/
const unsigned int ptb::status_layer::s_margin = 10;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 */
ptb::status_layer::status_layer()
  : m_data_1(NULL), m_data_2(NULL), m_first_player(NULL), m_second_player(NULL),
    m_timer(NULL), m_corrupting_bonus(NULL)
{

} // status_layer::status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::status_layer::~status_layer()
{
  delete m_data_1;
  delete m_data_2;
  delete m_corrupting_bonus;
} // status_layer::~status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the layer.
 */
void ptb::status_layer::build()
{
  bear::engine::level_globals& glob = get_level_globals();

  bear::visual::font f = glob.get_font("font/bouncy.fnt");

  m_text_time.create(f, "00:00");
  m_time_on = true;

  m_small_player =
    bear::visual::sprite( glob.auto_sprite("gfx/status.png", "plee"));

  m_last_corrupting_bonus = game_variables::get_corrupting_bonus_count();
  m_corrupting_bonus_time = 0;
  m_text_corrupting_bonus.create(f, "0");

  m_hazelnut = new bear::visual::animation
    (glob.get_animation("animation/owl/hazelnut.canim"));

  m_honeypot = glob.auto_sprite("gfx/status.png", "shiny jar");
} // status_layer::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players status and update local values.
 */
void ptb::status_layer::progress( bear::universe::time_type elapsed_time )
{
  search_players();

  progress_time(elapsed_time);
  progress_corrupting_bonus(elapsed_time);

  if ( m_data_1 != NULL )
    m_data_1->progress(get_level_globals(),elapsed_time);

  if ( m_data_2 != NULL )
    m_data_2->progress(get_level_globals(),elapsed_time);
} // status_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the players status.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render( scene_element_list& e ) const
{
  // render first player status
  if (m_data_1 != NULL)
    render_player1( e );

  // render second player status (if any)
  if (m_data_2 != NULL)
    render_player2( e );

  if ( m_time_on )
    e.push_back
      ( bear::visual::scene_writing
        ( (get_size().x - m_text_time.get_width()) / 2,
          get_size().y - s_margin - m_text_time.get_height(), m_text_time ) );

  render_corrupting_bonus( e );
  render_hazelnut( e );
  render_honeypots( e );
} // status_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the timer to use for the time.
 */
void ptb::status_layer::set_timer( const timer_handle& t )
{
  m_timer = t;
} // status_layer::set_timer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the time text.
 */
void ptb::status_layer::progress_time( bear::universe::time_type elapsed_time )
{
  if ( m_timer != (bear::timer*)NULL )
    {
      m_time_on = true;

      const bear::universe::time_type time = m_timer->get_time();
      const std::string t
        ( bear::systime::format_time_s( time,  gettext("%M:%S") ) );

      if ( m_timer->is_countdown() && (time <= 30) &&
           ((time - (int)time) < 0.5) )
        m_text_time->set_intensity(1, 0, 0);
      else
        m_text_time->set_intensity(1, 0.8, 0);

      m_text_time.create( get_level_globals().get_font("font/bouncy.fnt"), t );
    }
  else
    m_time_on = false;
} // status_layer::progress_time()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the corrupting_bonus state.
 */
void ptb::status_layer::progress_corrupting_bonus
( bear::universe::time_type elapsed_time )
{
  if ( !m_corrupting_bonus )
    {
      const std::string name
        ( game_variables::get_corrupting_bonus_animation_name() );
      if ( !name.empty() )
        {
          m_corrupting_bonus =
            new bear::visual::animation
            ( get_level_globals().get_animation(name));

          m_corrupting_bonus->set_size( m_corrupting_bonus->get_size() / 2 );
        }
    }

  if ( m_corrupting_bonus )
    {
      bear::engine::level_globals& glob = get_level_globals();
      unsigned int nb = game_variables::get_corrupting_bonus_count();

      if ( nb < m_last_corrupting_bonus )
        {
          m_last_corrupting_bonus = nb;
          m_corrupting_bonus_time = elapsed_time;
        }
      else if ( m_corrupting_bonus_time != 0 )
        {
          m_corrupting_bonus_time += elapsed_time;
          if ( m_corrupting_bonus_time > 1 )
            m_corrupting_bonus_time = 0;
        }
      else
        m_last_corrupting_bonus = nb;

      std::ostringstream oss;
      oss << nb;
      bear::visual::font f = glob.get_font("font/bouncy.fnt");
      m_text_corrupting_bonus.create(f, oss.str());

      double red = 1;
      double green = 0.8;

      if ( m_corrupting_bonus_time != 0 )
        {
          if ( m_corrupting_bonus_time > 0.5 )
            {
              red = m_corrupting_bonus_time;
              green = 2*(m_corrupting_bonus_time-0.5);
            }
          else if ( m_corrupting_bonus_time > 0 )
            {
              red = 1 - m_corrupting_bonus_time;
              green = 0.8 - 1.6*m_corrupting_bonus_time;
            }
        }
      m_text_corrupting_bonus->set_intensity(red, green, 0);
    }
} // status_layer::progress_corrupting_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the corrupting_bonus state.
 */
void ptb::status_layer::render_corrupting_bonus
( scene_element_list& e) const
{
  if ( m_corrupting_bonus )
    {
      double scale = 1;
      if ( m_corrupting_bonus_time > 0.5 )
        scale = 2 - m_corrupting_bonus_time;
      else if ( m_corrupting_bonus_time > 0 )
        scale = 1 + m_corrupting_bonus_time;

      bear::universe::coordinate_type width =
        m_corrupting_bonus->width() +
        m_text_corrupting_bonus.get_width() + s_margin;

      bear::universe::position_type pos
        ( get_size().x/2 - width/2, m_honeypot.height() + 2*s_margin);

      if (m_corrupting_bonus->height() > m_text_corrupting_bonus.get_height())
        {
          bear::visual::scene_sprite
            sp( pos.x, pos.y, m_corrupting_bonus->get_sprite() );

          sp.set_scale_factor(scale,scale);
          e.push_back( sp );

          pos.x += m_corrupting_bonus->width() + s_margin;
          pos.y +=
            (m_corrupting_bonus->height()
             - m_text_corrupting_bonus.get_height()) / 2;
        }
      else
        {
          bear::visual::scene_sprite
            sp(pos.x,
               pos.y + (m_text_corrupting_bonus.get_height()
                        - m_corrupting_bonus->height()) / 2,
               m_corrupting_bonus->get_sprite() );

          sp.set_scale_factor(scale,scale);
          e.push_back( sp );

          pos.x += m_corrupting_bonus->width() + s_margin;
        }

      bear::visual::scene_writing s( pos.x, pos.y, m_text_corrupting_bonus );
      s.set_scale_factor(scale,scale);
      e.push_back( s );
    }
} // status_layer::render_corrupting_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the hazelnut.
 */
void ptb::status_layer::render_hazelnut
( scene_element_list& e) const
{
  if ( level_variables::get_hazelnut(get_level()) )
    {
      e.push_back
        ( bear::visual::scene_sprite
          ( (get_size().x-m_hazelnut->width())/2,
            get_size().y - 2*s_margin - m_text_time.get_height()
            - m_hazelnut->get_sprite().height(),
            m_hazelnut->get_sprite()) );
    }
} // status_layer::render_hazelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render honeypots.
 */
void ptb::status_layer::render_honeypots
( scene_element_list& e) const
{
  unsigned int nb = level_variables::get_honeypots_found(get_level());
  bear::universe::coordinate_type width =
    nb * m_honeypot.width() + (nb-1)*s_margin;

  if ( nb > 0 )
    for ( unsigned int i = 0; i != nb; ++i )
      e.push_back
        ( bear::visual::scene_sprite
          ( ( get_size().x - width )/ 2 +
            i * (s_margin + m_honeypot.width()),
            s_margin,
            m_honeypot) );
} // status_layer::render_honeypots()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the status of the first player.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render_player1( scene_element_list& e ) const
{
  bear::visual::position_type pos
    ( s_margin, get_size().y - s_margin - m_data_1->energy.height() );

  m_data_1->energy.render(e, pos);

  pos.y -= s_margin + m_data_1->score.get_height();
  bear::visual::scene_writing s1( pos.x, pos.y, m_data_1->score );
  e.push_back( s1 );

  bear::visual::coordinate_type y_saved(pos.y);
  pos.y -= s_margin + m_data_1->throwable_item_animation.get_sprite().height();
  bear::visual::scene_sprite s_item
    (pos.x, pos.y, m_data_1->throwable_item_animation.get_sprite());

  pos.x += m_data_1->throwable_item_animation.get_sprite().width() + s_margin;
  pos.y +=
    ( m_data_1->throwable_item_animation.get_sprite().height()
      - m_data_1->throwable_items.get_height() ) / 2;

  bear::visual::scene_writing s2( pos.x, pos.y, m_data_1->throwable_items );

  if ( !m_first_player->get_throwable_items()->get_current_throwable_item
       ()->can_throw() )
    {
      s_item.get_rendering_attributes().set_opacity(0.5);
      s2.get_rendering_attributes().set_opacity(0.5);
      s2.get_rendering_attributes().set_intensity(1,0.4,0.4);
    }

  e.push_back( s_item );
  e.push_back( s2 );

  pos.x = s_margin;
  pos.y = std::min(pos.y, y_saved) - s_margin - m_data_1->oxygen.height();

  if ( m_data_1->oxygen_active )
    {
      pos.x -= m_data_1->oxygen_offset_x;
      m_data_1->oxygen.render(e, pos);
    }
  pos.set( s_margin, pos.y - m_data_1->fire_gauge.height() - s_margin );

  if ( m_data_1->fire_gauge_active )
    {
      pos.x -= m_data_1->fire_gauge_offset_x;
      m_data_1->fire_gauge.render(e, pos);
    }
  pos.set( s_margin, pos.y - m_data_1->ice_gauge.height() - s_margin );

  if ( m_data_1->ice_gauge_active )
    {
      pos.x -= m_data_1->ice_gauge_offset_x;
      m_data_1->ice_gauge.render(e, pos);
    }
  pos.set( s_margin, pos.y - s_margin );

  // lifes
  pos.set( s_margin, s_margin );
  e.push_back( bear::visual::scene_sprite(pos.x, pos.y, m_small_player) );

  double scale = 1;
  if ( m_data_1->m_tries_time > 0.5 )
    scale = 2 - m_data_1->m_tries_time;
  else if ( m_data_1->m_tries_time > 0 )
    scale = 1 + m_data_1->m_tries_time;

  pos.x += m_small_player.width() + s_margin;
  pos.y =
    s_margin
    + (m_small_player.height() - m_data_1->tries.get_height() * scale) / 2;

  bear::visual::scene_writing s3( pos.x, pos.y, m_data_1->tries );
  s3.set_scale_factor(scale,scale);
  e.push_back( s3 );
} // status_layer::render_player1()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the status of the second player.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render_player2( scene_element_list& e ) const
{
  bear::visual::position_type pos
    ( get_size().x - m_data_2->energy.width() - s_margin,
      get_size().y - s_margin - m_data_2->energy.height() );

  m_data_2->energy.render(e, pos);

  pos.set( get_size().x - s_margin - m_data_2->score.get_size().x,
           pos.y - s_margin - m_data_2->score.get_height() );
  bear::visual::scene_writing s1( pos.x, pos.y, m_data_2->score );
  e.push_back( s1 );

  bear::visual::coordinate_type y_saved(pos.y);
  pos.set( get_size().x - s_margin -
           m_data_2->throwable_item_animation.width(),
           pos.y - s_margin -
           m_data_2->throwable_item_animation.get_sprite().height() );
  bear::visual::scene_sprite s_item
    (pos.x, pos.y, m_data_2->throwable_item_animation.get_sprite());

  pos.set( get_size().x - 2 * s_margin -
           m_data_2->throwable_item_animation.width()
           - m_data_2->throwable_items.get_size().x,
           pos.y
           + ( m_data_2->throwable_item_animation.get_sprite().height()
               - m_data_2->throwable_items.get_height() ) / 2 );

  bear::visual::scene_writing s2( pos.x, pos.y, m_data_2->throwable_items );

  if ( !m_second_player->get_throwable_items()->get_current_throwable_item
       ()->can_throw() )
    {
      s_item.get_rendering_attributes().set_opacity(0.5);
      s2.get_rendering_attributes().set_opacity(0.5);
      s2.get_rendering_attributes().set_intensity(1,0.4,0.4);
    }

  e.push_back( s_item );
  e.push_back( s2 );

  pos.x = get_size().x - s_margin;
  pos.y = std::min(pos.y, y_saved) - s_margin - m_data_2->oxygen.height();

  if ( m_data_2->oxygen_active )
    {
      pos.x = get_size().x - m_data_2->oxygen.width() - s_margin
        + m_data_2->oxygen_offset_x;
      m_data_2->oxygen.render(e, pos);
    }
  pos.set( get_size().x - s_margin,
           pos.y - m_data_2->fire_gauge.height() - s_margin );

  if ( m_data_2->fire_gauge_active )
    {
      pos.x = get_size().x - m_data_2->fire_gauge.width() - s_margin
        + m_data_2->fire_gauge_offset_x;
      m_data_2->fire_gauge.render(e, pos);
    }
  pos.set( get_size().x - s_margin,
           pos.y - m_data_2->ice_gauge.height() - s_margin );

  if ( m_data_2->ice_gauge_active )
    {
      pos.x = get_size().x - m_data_2->ice_gauge.width() - s_margin
        + m_data_2->ice_gauge_offset_x;
      m_data_2->ice_gauge.render(e, pos);
    }
  pos.set( get_size().x - s_margin, pos.y - s_margin );

  // lifes
  pos.set( get_size().x - s_margin - m_small_player.width(), s_margin );
  e.push_back( bear::visual::scene_sprite(pos.x, pos.y, m_small_player) );

  double scale = 1;
  if ( m_data_2->m_tries_time > 0.5 )
    scale = 2 - m_data_2->m_tries_time;
  else if ( m_data_2->m_tries_time > 0 )
    scale = 1 + m_data_2->m_tries_time;

  pos.set( get_size().x - 2*s_margin - m_small_player.width()
           - m_data_2->tries.get_width()*scale,
           s_margin
           + ( m_small_player.height()
               - m_data_2->tries.get_height() * scale ) / 2 );

  bear::visual::scene_writing s3( pos.x, pos.y, m_data_2->tries );
  s3.set_scale_factor(scale,scale);
  e.push_back( s3 );
} // status_layer::render_player2()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search players.
 */
void ptb::status_layer::search_players( )
{
  if ( m_first_player == (player*)NULL )
    {
      m_first_player = util::find_player( get_level_globals(), 1 );

      if ( m_first_player != (player*)NULL )
        m_data_1 = new player_status
          ( get_level_globals(),
            get_level_globals().get_font("font/bouncy.fnt"), m_first_player );
    }

  if ( game_variables::get_players_count() == 2 )
    if ( m_second_player == (player*)NULL )
      {
        m_second_player = util::find_player( get_level_globals(), 2 );

        if ( m_second_player != (player*)NULL )
          m_data_2 = new player_status
            ( get_level_globals(),
              get_level_globals().get_font("font/bouncy.fnt"),
              m_second_player );
      }
} // status_layer::search_players( )
