/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file model_mark_placement.cpp
 * \brief Implementation of the bear::engine::model_mark_placement class.
 * \author Julien Jorge
 */
#include "engine/model/model_mark_placement.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::engine::model_mark_placement::model_mark_placement()
{

} // model_mark_placement::model_mark_placement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param id The identifier of the mark.
 * \param x The x-position.
 * \param y The y-position.
 * \param depth The depth-position.
 * \param angle The angle.
 * \param visible The visibility.
 */
bear::engine::model_mark_placement::model_mark_placement
( unsigned int id, universe::coordinate_type x, universe::coordinate_type y,
  int depth, double angle, bool visible )
  : m_mark_id(id), m_position(x, y), m_depth(depth), m_angle(angle),
    m_visible(visible)
{

} // model_mark_placement::model_mark_placement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the identifier or the mark concerned by this placement.
 * \param i The identifier of the mark.
 */
void bear::engine::model_mark_placement::set_mark_id( unsigned int i )
{
  m_mark_id = i;
} // model_mark_placement::set_mark_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the identifier or the mark concerned by this placement.
 */
unsigned int bear::engine::model_mark_placement::get_mark_id() const
{
  return m_mark_id;
} // model_mark_placement::get_mark_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the mark.
 * \param x The new x-position.
 * \param y The new y-position.
 */
void bear::engine::model_mark_placement::set_position
( universe::coordinate_type x, universe::coordinate_type y )
{
  m_position.set(x, y);
} // model_mark_placement::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the mark.
 * \param pos The new position.
 */
void bear::engine::model_mark_placement::set_position
( const universe::position_type& pos )
{
  m_position.set(pos.x,pos.y);
} // model_mark_placement::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the mark.
 */
const bear::universe::position_type&
bear::engine::model_mark_placement::get_position() const
{
  return m_position;
} // model_mark_placement::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the depth-position of the mark.
 * \param z The new depth-position.
 */
void bear::engine::model_mark_placement::set_depth_position( int z )
{
  m_depth = z;
} // model_mark_placement::set_depth_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the depth-position of the mark.
 */
int bear::engine::model_mark_placement::get_depth_position() const
{
  return m_depth;
} // model_mark_placement::get_depth_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the angle of the mark.
 * \param a The new angle.
 */
void bear::engine::model_mark_placement::set_angle( double a )
{
  m_angle = a;
} // model_mark_placement::set_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the angle of the mark.
 */
double bear::engine::model_mark_placement::get_angle() const
{
  return m_angle;
} // model_mark_placement::get_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the mark is visible or not.
 * \param v Visible or not.
 */
void bear::engine::model_mark_placement::set_visibility( bool v )
{
  m_visible = v;
} // model_mark_placement::set_visibility()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the mark is visible or not.
 */
bool bear::engine::model_mark_placement::is_visible() const
{
  return m_visible;
} // model_mark_placement::is_visible()
