/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file status_layer.cpp
 * \brief Implementation of the ptb::status_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/status_layer.hpp"

#include "engine/game.hpp"
#include "ptb/item/timer_kill_players.hpp"
#include "visual/bitmap_writing.hpp"
#include "visual/scene_sprite.hpp"
#include "visual/scene_writing.hpp"
#include "ptb/item/plee/plee.hpp"
#include "ptb/level_variables.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param glob Level globals in which we take the resources.
 * \param level_sprite The position in "gfx/tube.png" of the image to use for
 *        the main level.
 * \param loss_sprite The position in "gfx/tube.png" of the image to use for the
 *        loss level.
 * \param length The length of the bar.
 */
ptb::status_layer::horizontal_bar::horizontal_bar
( bear::engine::level_globals& glob,
  const bear::visual::rectangle_type& level_sprite,
  const bear::visual::rectangle_type& loss_sprite,
  const std::string& icon_name, unsigned int length )
  : m_level_value(length)
{
  const bear::visual::image& bar_resource = glob.get_image("gfx/tube.png");

  m_level = bear::visual::sprite( bar_resource, level_sprite );
  m_loss = bear::visual::sprite( bar_resource, loss_sprite );
  m_icon = glob.auto_sprite("gfx/tube-vertical.png", icon_name);

  m_loss.set_opacity(0);

  m_tube_clamp =
    bear::visual::sprite
    ( bar_resource, claw::math::rectangle<unsigned int>( 0, 0, 19, 24 ) );

  m_glass_tube =
    bear::visual::sprite
    ( bar_resource, claw::math::rectangle<unsigned int>( 19, 0, 1, 20 ) );

  m_glass_tube.set_width(length);
  m_level.set_width(length);
} // status_layer::horizontal_bar::horizontal_bar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the length.
 */
void ptb::status_layer::horizontal_bar::set_length(unsigned int length)
{
  m_glass_tube.set_width(length);
} // status_layer::horizontal_bar::set_length()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual length of the bar.
 */
unsigned int ptb::status_layer::horizontal_bar::length() const
{
  return m_glass_tube.width();
} // status_layer::horizontal_bar::length()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual width of the bar.
 */
unsigned int ptb::status_layer::horizontal_bar::width() const
{
  return 2 * m_tube_clamp.width() + m_glass_tube.width();
} // status_layer::horizontal_bar::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual height of the bar.
 */
unsigned int ptb::status_layer::horizontal_bar::height() const
{
  return std::max
    ( m_tube_clamp.height() + m_icon.height() / 2, m_glass_tube.height() );
} // status_layer::horizontal_bar::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the level of the bar.
 * \param lev The new level.
 */
void ptb::status_layer::horizontal_bar::set_level( double lev )
{
  if ( lev != m_level_value )
    {
      if ( lev < m_level_value )
        m_loss.set_opacity(1);
      else
        m_loss.set_opacity(0);

      m_level_value = lev;
    }
} // status_layer::horizontal_bar::set_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the bar.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::status_layer::horizontal_bar::progress
( bear::universe::time_type elapsed_time )
{
  if ( m_loss.get_opacity() >= 0 )
    m_loss.set_opacity
      ( std::max(0.0, m_loss.get_opacity() - elapsed_time) );

  m_level.set_width( (unsigned int)(m_level_value + 0.5) );
  m_loss.set_width( m_level.width() );
} // status_layer::horizontal_bar::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Display the bar.
 * \param e (out) The scene elements.
 * \param pos The top-left position of the bar.
 */
void ptb::status_layer::horizontal_bar::render
( scene_element_list& e, const bear::visual::position_type& pos )
{
  bear::universe::position_type p(pos + m_icon.get_size() / 2);
  bear::visual::sprite clamp(m_tube_clamp);
  const bear::visual::coordinate_type d =
    ( clamp.height() - m_level.height() ) / 2;

  if (d < 0)
    p.y -= d;

  e.push_back( bear::visual::scene_sprite(p.x, p.y, clamp) );
  e.push_back
    ( bear::visual::scene_sprite
      (p.x - m_icon.width() / 2, p.y - m_icon.height() / 2, m_icon) );

  p.x += clamp.width();
  p.y += d;
  e.push_back( bear::visual::scene_sprite(p.x, p.y, m_level) );

  if ( m_loss.get_opacity() != 0 )
    e.push_back( bear::visual::scene_sprite(p.x, p.y, m_loss) );

  e.push_back( bear::visual::scene_sprite(p.x, p.y, m_glass_tube) );

  p.x += m_glass_tube.width();
  p.y -= d;
  clamp.mirror(true);
  e.push_back( bear::visual::scene_sprite(p.x, p.y, clamp) );
} // status_layer::horizontal_bar::render()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param glob Level globals in which we take the resources.
 * \param level_sprite The position in "gfx/tube-vertical.png" of the image to
 *        use for the main level.
 * \param icon_sprite The position in "gfx/tube-vertical.png" of the image to
 *        use for the icon of the bar.
 * \param length The length of the bar.
 */
ptb::status_layer::vertical_bar::vertical_bar
( bear::engine::level_globals& glob,
  const bear::visual::rectangle_type& level_sprite,
  const bear::visual::rectangle_type& icon_sprite, unsigned int length )
  : m_level_value(length)
{
  const bear::visual::image& bar_resource =
    glob.get_image("gfx/tube-vertical.png");

  m_level = bear::visual::sprite( bar_resource, level_sprite );
  m_icon = bear::visual::sprite( bar_resource, icon_sprite );

  m_tube_clamp =
    bear::visual::sprite
    ( bar_resource, claw::math::rectangle<unsigned int>( 0, 0, 24, 19 ) );

  m_glass_tube =
    bear::visual::sprite
    ( bar_resource, claw::math::rectangle<unsigned int>( 0, 21, 20, 1 ) );

  m_glass_tube.set_height(length);
  m_level.set_height(length);
} // status_layer::vertical_bar::vertical_bar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual width of the bar.
 */
unsigned int ptb::status_layer::vertical_bar::width() const
{
  return std::max( m_tube_clamp.width() + m_icon.width() / 2,
                   m_glass_tube.width() );
} // status_layer::vertical_bar::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visual height of the bar.
 */
unsigned int ptb::status_layer::vertical_bar::height() const
{
  return 2 * m_tube_clamp.height() + m_glass_tube.height()
    + m_icon.height() / 2;
} // status_layer::vertical_bar::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the level of the bar.
 * \param lev The new level.
 */
void ptb::status_layer::vertical_bar::set_level( double lev )
{
  m_level_value = lev;
} // status_layer::vertical_bar::set_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the bar.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::status_layer::vertical_bar::progress
( bear::universe::time_type elapsed_time )
{
  m_level.set_height( (unsigned int)(m_level_value + 0.5) );
} // status_layer::vertical_bar::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Display the bar.
 * \param e (out) The scene elements.
 * \param pos The top-left position of the bar.
 */
void ptb::status_layer::vertical_bar::render
( scene_element_list& e, const bear::visual::position_type& pos )
{
  bear::visual::position_type p(pos.x, pos.y + m_icon.height() / 2);
  bear::visual::sprite clamp(m_tube_clamp);
  const bear::visual::coordinate_type d =
    ( clamp.width() - m_level.width() ) / 2;

  if (d < 0)
    p.x -= d;

  e.push_back( bear::visual::scene_sprite(p.x, p.y, clamp) );

  p.x = pos.x + clamp.width() - m_icon.width() / 2;
  p.y = pos.y;
  e.push_back( bear::visual::scene_sprite(p.x, p.y, m_icon) );

  p.x = pos.x + d;
  p.y = pos.y + m_icon.height() / 2 + clamp.height()
    + m_glass_tube.height() - m_level.height();
  e.push_back( bear::visual::scene_sprite(p.x, p.y, m_level) );

  p.y = pos.y + clamp.height() + m_icon.height() / 2;
  e.push_back( bear::visual::scene_sprite(p.x, p.y, m_glass_tube) );

  p.x -= d;
  p.y += m_glass_tube.height();
  clamp.flip(true);
  e.push_back( bear::visual::scene_sprite(p.x, p.y, clamp) );
} // status_layer::vertical_bar::render()




/*----------------------------------------------------------------------------*/
const double ptb::status_layer::player_status::s_bar_length = 100;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param glob Level globals in which we take the resources.
 * \param f The font to use to display the texts.
 * \param p The player from which we take the data.
 */
ptb::status_layer::player_status::player_status
( bear::engine::level_globals& glob, const bear::visual::font& f,
  bear::universe::const_derived_item_handle_maker<plee>::handle_type& p )
  : energy( glob, bear::visual::rectangle_type( 22, 0, 1, 20 ),
            bear::visual::rectangle_type( 21, 0, 1, 20 ), "heart", s_bar_length ),
    oxygen( glob, bear::visual::rectangle_type( 0, 20, 20, 1 ),
            bear::visual::rectangle_type( 48, 0, 16, 16 ), s_bar_length ),
    fire_gauge( glob, bear::visual::rectangle_type( 0, 19, 20, 1 ),
            bear::visual::rectangle_type( 48, 16, 16, 16 ), s_bar_length ),
    ice_gauge( glob, bear::visual::rectangle_type( 0, 22, 20, 1 ),
            bear::visual::rectangle_type( 32, 0, 16, 16 ), s_bar_length ),
    player(p),
    oxygen_active(false), oxygen_offset_x(oxygen.width() + 5),
    fire_gauge_active(false), fire_gauge_offset_x(fire_gauge.width() + 5),
    ice_gauge_active(false), ice_gauge_offset_x(ice_gauge.width() + 5)
{
  score.create(f, "0pt");
  stones.create(f, "0");

  m_air_power = false;
  m_fire_power = false;
  m_water_power = false;

  m_last_tries = game_variables::get_lives_count(player->get_index());
  m_tries_time = 0;
  
  if ( p.get() ) 
    update_stone_animation(glob,p->get_index());
} // status_layer::player_status::player_status()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the values.
 * \param glob The level globals in which we load the media.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::status_layer::player_status::progress
(  bear::engine::level_globals& glob, bear::universe::time_type elapsed_time )
{
  energy.progress(elapsed_time);
  oxygen.progress(elapsed_time);
  fire_gauge.progress(elapsed_time);
  ice_gauge.progress(elapsed_time);
 
  if (player.get())
    {
      energy.set_length(game_variables::get_max_energy(player->get_index()));
      energy.set_level
        ( energy.length() * player->get_energy() / 
          game_variables::get_max_energy(player->get_index()) );
      
      progress_gauge
        (oxygen, player->get_gauge(plee::oxygen_gauge)->is_activated(),
         player->get_gauge(plee::oxygen_gauge)->get_value(),
         plee::s_max_oxygen, oxygen_active, oxygen_offset_x);
      
      progress_gauge
        (fire_gauge, player->get_gauge(plee::fire_gauge)->is_activated(),
         player->get_gauge(plee::fire_gauge)->get_value(),
         plee::s_max_fire_gauge, fire_gauge_active, fire_gauge_offset_x);
      
      progress_gauge
        (ice_gauge, player->get_gauge(plee::ice_gauge)->is_activated(),
         player->get_gauge(plee::ice_gauge)->get_value(),
         plee::s_max_ice_gauge, ice_gauge_active, ice_gauge_offset_x);
      
      bear::visual::font f = glob.get_font("font/bouncy.fnt");

      std::ostringstream oss1;
      oss1 << game_variables::get_score(player->get_index());
      if ( game_variables::get_score(player->get_index()) > 0 ) 
	oss1 << "pts";
      else
	oss1 << "pt";
      score.create(f, oss1.str());  
      score->set_intensity(1, 0.8, 0);
      
      std::ostringstream oss2;
      oss2 << game_variables::get_stones_count(player->get_index());
      stones.create(f, oss2.str());
      stones->set_intensity(1, 0.8, 0);
      
      update_tries(glob, elapsed_time, player->get_index());
      
      if ( update_powers(player->get_index()) ) 
        update_stone_animation(glob,player->get_index());
      
      stone_animation.next(elapsed_time);
    }
} // status_layer::player_status::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the powers.
 * Return true if the powers have changed.
 * \param index The index of the player.
 */
bool ptb::status_layer::player_status::update_powers( unsigned int index )
{
  bool result = false;

  if ( m_air_power != game_variables::get_air_power(index) ) 
    {
      m_air_power = game_variables::get_air_power(index);
      result = true;
    }

  if ( m_fire_power != game_variables::get_fire_power(index) ) 
    {
      m_fire_power = game_variables::get_fire_power(index);
      result = true;
    }

  if ( m_water_power != game_variables::get_water_power(index) ) 
    {
      m_water_power = game_variables::get_water_power(index);
      result = true;
    }

  return result;
} // player_status::update_powers

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the tries count.
 * \param glob The level globals in which we load the media.
 * \param index The index of the player.
 */
void ptb::status_layer::player_status::update_tries
( bear::engine::level_globals& glob, bear::universe::time_type elapsed_time, 
  unsigned int index )
{
  unsigned int nb = game_variables::get_lives_count(player->get_index());

  if ( nb > m_last_tries ) 
    {
      m_last_tries = nb;
      m_tries_time = elapsed_time;
    }
  else if ( m_tries_time != 0 )
    {
      m_tries_time += elapsed_time;
      if ( m_tries_time > 1 )
	m_tries_time = 0;
    }
  
  std::ostringstream oss;
  oss << nb;
  bear::visual::font f = glob.get_font("font/bouncy.fnt");
  tries.create(f, oss.str());
  
  double red = 1;
  double green = 0.8;
      
  if ( m_tries_time != 0 ) 
    {
      if ( m_tries_time > 0.5 )
	{
	  red = m_tries_time;
	  green = 2*(m_tries_time-0.5); 
	}
      else if ( m_tries_time > 0 ) 
	{
	  red = 1 - m_tries_time;
	  green = 0.8 - 1.6*m_tries_time;
	}
    }
  tries->set_intensity(red, green, 0);
} // player_status::update_tries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the stone animation.
 * \param glob The level globals in which we load the media.
 * \param index The index of the player.
 */
void ptb::status_layer::player_status::update_stone_animation
( bear::engine::level_globals& glob, unsigned int index ) 
{
  if ( game_variables::get_air_power(index) )
    if ( game_variables::get_fire_power(index) )
      if ( game_variables::get_water_power(index) )
        stone_animation = glob.get_animation
          ("animation/stones/air_fire_water_stone.canim");
      else
        stone_animation = glob.get_animation
          ("animation/stones/air_fire_stone.canim");
    else
      if ( game_variables::get_water_power(index) )
        stone_animation = glob.get_animation
          ("animation/stones/air_water_stone.canim");
      else
        stone_animation = glob.get_animation
          ("animation/stones/air_stone.canim");
  else
    if ( game_variables::get_fire_power(index) )
      if ( game_variables::get_water_power(index) )
        stone_animation = glob.get_animation
          ("animation/stones/water_fire_stone.canim");
      else
        stone_animation = glob.get_animation
          ("animation/stones/fire_stone.canim");
    else
      if ( game_variables::get_water_power(index) )
        stone_animation = glob.get_animation
          ("animation/stones/water_stone.canim");
      else
        stone_animation = glob.get_animation
          ("animation/stones/stone.canim");
} // player_status::update_stone_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update a gauge.
 * \param gauge The gauge to update.
 * \param is_active Indicates if the gauge must be active.
 * \param value The level of the gauge.
 * \param max_value The max level of the gauge.
 * \param new_active (out) The status of gauge activation.
 * \param offset_x (out) The offset of the gauge in x axis.
 */
void ptb::status_layer::player_status::progress_gauge
( vertical_bar& gauge, bool is_active, double value, double max_value,
  bool& new_active, bear::visual::coordinate_type& offset_x)
{
  gauge.set_level( s_bar_length * value / max_value );

  if ( is_active )
    {
      new_active = true;

      if ( offset_x != 0 )
        --offset_x;
    }
  else
    {
      if ( offset_x != gauge.width() + 5 )
        ++offset_x;
      else
        new_active = false;
    }
} // status_layer::player_status::progress_gauge()


/*----------------------------------------------------------------------------*/
const unsigned int ptb::status_layer::s_margin = 10;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 */
ptb::status_layer::status_layer()
  : m_data_1(NULL), m_data_2(NULL), 
    m_first_player(NULL), m_second_player(NULL), m_corrupting_bonus(NULL)
{

} // status_layer::status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::status_layer::~status_layer()
{
  delete m_data_1;
  delete m_data_2;

  if ( m_corrupting_bonus )
    delete m_corrupting_bonus;
} // status_layer::~status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the layer.
 */
void ptb::status_layer::build()
{
  bear::engine::level_globals& glob = get_level_globals();

  bear::visual::font f = glob.get_font("font/bouncy.fnt");

  m_text_time.create(f, "00:00");
  m_time_on = true;
  
  m_small_plee = 
    bear::visual::sprite( glob.auto_sprite("gfx/status.png", "plee"));
  
  m_text_corrupting_bonus.create(f, "0");
  
  m_azelnut = new bear::visual::animation
    (glob.get_animation("animation/owl/azelnut.canim"));

  m_honeypot = glob.auto_sprite("gfx/status.png", "shiny jar");
} // status_layer::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players status and update local values.
 */
void ptb::status_layer::progress( bear::universe::time_type elapsed_time )
{
  search_players();

  progress_time(elapsed_time);
  progress_corrupting_bonus(elapsed_time);
  
  if ( m_data_1 != NULL )
    m_data_1->progress(get_level_globals(),elapsed_time);
  
  if ( m_data_2 != NULL )
    m_data_2->progress(get_level_globals(),elapsed_time);
} // status_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the players status.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render( scene_element_list& e ) const
{
  // render first player status
  if (m_data_1 != NULL)
    render_player1( e );

  // render second player status (if any)
  if (m_data_2 != NULL)
    render_player2( e );

  if ( m_time_on ) 
    e.push_back
      ( bear::visual::scene_writing
	( (get_size().x - m_text_time.get_width()) / 2, 
	  s_margin, m_text_time ) );
  
  render_corrupting_bonus( e );
  render_azelnut( e );
  render_honeypots( e );
} // status_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the time text.
 */
void ptb::status_layer::progress_time( bear::universe::time_type elapsed_time )
{
  timer_kill_players::msg_get_value msg;
  bear::engine::level_globals& glob = get_level_globals();

  if ( glob.send_message( "level_timer", msg ) )
    {
      m_time_on = true;

      bear::universe::time_type time = msg.value;
      unsigned int sec = (unsigned int)time % 60;
      unsigned int min = (unsigned int)time / 60;

      std::ostringstream oss_time;
      if ( min < 10 )
        oss_time << " ";
      oss_time << min;

      oss_time << ":";

      if ( sec < 10 )
        oss_time << "0";
      oss_time << sec;

      if ( msg.countdown && ( min == 0 ) && ( sec <= 30 ) &&
	   ( (unsigned int)((time - 60*min - sec )*4)%2 == 1) )
        m_text_time->set_intensity(1, 0, 0);
      else
        m_text_time->set_intensity(1, 0.8, 0);

      m_text_time.create
        ( get_level_globals().get_font("font/bouncy.fnt"), oss_time.str() );
    }
  else
     m_time_on = false;
} // status_layer::progress_time()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the corrupting_bonus state.
 */
void ptb::status_layer::progress_corrupting_bonus
( bear::universe::time_type elapsed_time )
{
  if ( ! m_corrupting_bonus ) 
    {
      const std::string name
        (game_variables::get_corrupting_bonus_animation_name());
      if ( ! name.empty() ) 
        m_corrupting_bonus = new bear::visual::animation
          ( get_level_globals().get_animation(name));
    }
  
  if ( m_corrupting_bonus )   
    {
      std::ostringstream oss;
      oss << game_variables::get_corrupting_bonus_count();
      m_text_corrupting_bonus.create
        ( get_level_globals().get_font("font/bouncy.fnt"), oss.str() );
      
      m_text_corrupting_bonus->set_intensity(1, 0.8, 0);
    }
} // status_layer::progress_corrupting_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the corrupting_bonus state.
 */
void ptb::status_layer::render_corrupting_bonus
( scene_element_list& e) const
{
  if ( m_corrupting_bonus )   
    {
      bear::universe::coordinate_type width = 
        m_corrupting_bonus->width() +  
        m_text_corrupting_bonus.get_width() + s_margin;
      
      bear::universe::position_type pos
        ( get_size().x/2 - width/2, 
          get_size().y - m_corrupting_bonus->height() - 
	  m_honeypot.height() - 2*s_margin);
      
      e.push_back
        ( bear::visual::scene_sprite
          (pos.x, pos.y, m_corrupting_bonus->get_sprite()) );
      
      e.push_back
	( bear::visual::scene_writing
	  ( get_size().x/2  + width/2 - m_text_corrupting_bonus.get_width(),  
	    get_size().y - 2*s_margin -  m_honeypot.height() -
	    m_corrupting_bonus->height()/2 - 
	    m_text_corrupting_bonus.get_width()/4,
	    m_text_corrupting_bonus ) );
    }
} // status_layer::render_corrupting_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the azelnut.
 */
void ptb::status_layer::render_azelnut
( scene_element_list& e) const
{
  if ( level_variables::get_azelnut(get_level()) )   
    {
      e.push_back
        ( bear::visual::scene_sprite
          ( (get_size().x-m_azelnut->width())/2, 
            2*s_margin + m_text_time.get_height(), 
            m_azelnut->get_sprite()) );
    }
} // status_layer::render_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render honeypots.
 */
void ptb::status_layer::render_honeypots
( scene_element_list& e) const
{
  unsigned int nb = level_variables::get_honeypots_found(get_level());
  bear::universe::coordinate_type width = 
    nb * m_honeypot.width() + (nb-1)*s_margin;

  if ( nb > 0 )
    for ( unsigned int i = 0; i != nb; ++i )
      e.push_back
        ( bear::visual::scene_sprite
          ( ( get_size().x - width )/ 2 + 
            i * (s_margin + m_honeypot.width()),
            get_size().y - m_honeypot.height() - s_margin, 
            m_honeypot) );
} // status_layer::render_honeypots()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the status of the first player.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render_player1( scene_element_list& e ) const
{
  bear::visual::position_type pos(s_margin, s_margin);

  m_data_1->energy.render(e, pos);

  pos.set( s_margin , 2 * s_margin + m_data_1->energy.height() );
  bear::visual::scene_writing s1( pos.x, pos.y, m_data_1->score );
  e.push_back( s1 );

  pos.set( s_margin,
	   3*s_margin + m_data_1->energy.height() +
	   m_data_1->score.get_height());
  e.push_back
    ( bear::visual::scene_sprite
      (pos.x, pos.y, m_data_1->stone_animation.get_sprite()) );
  
  pos.set( 2*s_margin + m_data_1->stone_animation.width(),
	   3*s_margin + m_data_1->energy.height() + 
	   m_data_1->score.get_height());
  if ( m_data_1->stone_animation.height() > m_data_1->stones.get_height() )
    pos.y += ( m_data_1->stone_animation.height() - 
	       m_data_1->stones.get_height() ) / 2;
  bear::visual::scene_writing s2( pos.x, pos.y, m_data_1->stones );
  e.push_back( s2 );
  
  pos.y += m_data_1->stone_animation.height() + s_margin;

  if ( m_data_1->oxygen_active )
    {
      pos.x -= m_data_1->oxygen_offset_x;
      m_data_1->oxygen.render(e, pos);
    }
  pos.set( s_margin, pos.y + m_data_1->oxygen.height() + s_margin );

  if ( m_data_1->fire_gauge_active )
    {
      pos.x -= m_data_1->fire_gauge_offset_x;
      m_data_1->fire_gauge.render(e, pos);
    }
  pos.set( s_margin, pos.y + m_data_1->fire_gauge.height() + s_margin );

  if ( m_data_1->ice_gauge_active )
    {
      pos.x -= m_data_1->ice_gauge_offset_x;
      m_data_1->ice_gauge.render(e, pos);
    }
  pos.set( s_margin, pos.y + m_data_1->ice_gauge.height() + s_margin );

  // lifes
  pos.set( s_margin, get_size().y - s_margin - m_small_plee.height() );
  e.push_back( bear::visual::scene_sprite(pos.x, pos.y, m_small_plee) );

  double scale = 1;
  if ( m_data_1->m_tries_time > 0.5 )
    scale = 2 - m_data_1->m_tries_time;
  else if ( m_data_1->m_tries_time > 0 ) 
    scale = 1 + m_data_1->m_tries_time;

  pos.set( m_small_plee.width() + 2*s_margin, 
	   get_size().y - s_margin - 
	   ( m_small_plee.height() + m_data_1->tries.get_height()*scale)/2 );
  
  bear::visual::scene_writing s3( pos.x, pos.y, m_data_1->tries );
  s3.set_scale_factor(scale,scale);  
  e.push_back( s3 );
} // status_layer::render_player1()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the status of the second player.
 * \param e (out) The scene elements.
 */
void ptb::status_layer::render_player2( scene_element_list& e ) const
{
  bear::visual::position_type pos
    (get_size().x - m_data_2->energy.width() - s_margin, s_margin);

  m_data_2->energy.render(e, pos);  

  pos.set( get_size().x - s_margin - m_data_2->score.get_size().x,
           2*s_margin + m_data_2->energy.height());
  bear::visual::scene_writing s1( pos.x, pos.y, m_data_2->score );
  e.push_back( s1 );

  pos.set( get_size().x - s_margin - m_data_2->stone_animation.width(), 
	   3*s_margin + m_data_2->energy.height() + 
	   m_data_2->score.get_height());
  
  e.push_back
    ( bear::visual::scene_sprite
      (pos.x, pos.y, m_data_2->stone_animation.get_sprite()) );  

  pos.set( get_size().x - 2 * s_margin - m_data_2->stone_animation.width()
               - m_data_2->stones.get_size().x, 3*s_margin + 
               m_data_2->energy.height() + m_data_2->score.get_height());

  if ( m_data_2->stone_animation.height() > m_data_2->stones.get_height() )
    pos.y += ( m_data_2->stone_animation.height() - 
	       m_data_2->stones.get_height() ) / 2;
  
  bear::visual::scene_writing s2( pos.x, pos.y, m_data_2->stones );
  e.push_back( s2 );

  pos.y += m_data_2->stone_animation.height() + s_margin;

  if ( m_data_2->oxygen_active )
    {
      pos.x = get_size().x - m_data_2->oxygen.width() - s_margin
        + m_data_2->oxygen_offset_x;
      m_data_2->oxygen.render(e, pos);
    }
  pos.set( get_size().x - s_margin,
           pos.y + m_data_2->oxygen.height() + s_margin );

  if ( m_data_2->fire_gauge_active )
    {
      pos.x = get_size().x - m_data_2->fire_gauge.width() - s_margin
        + m_data_2->fire_gauge_offset_x;
      m_data_2->fire_gauge.render(e, pos);
    }
  pos.set( get_size().x - s_margin,
           pos.y + m_data_2->fire_gauge.height() + s_margin );

  if ( m_data_2->ice_gauge_active )
    {
      pos.x = get_size().x - m_data_2->ice_gauge.width() - s_margin
        + m_data_2->ice_gauge_offset_x;
      m_data_2->ice_gauge.render(e, pos);
    }
  pos.set( get_size().x - s_margin,
           pos.y + m_data_2->ice_gauge.height() + s_margin );

  // lifes
  pos.set( get_size().x - s_margin - m_small_plee.width(),
           get_size().y - s_margin - m_small_plee.height() );
  e.push_back( bear::visual::scene_sprite(pos.x, pos.y, m_small_plee) );
  
  double scale = 1;
  if ( m_data_2->m_tries_time > 0.5 )
    scale = 2 - m_data_2->m_tries_time;
  else if ( m_data_2->m_tries_time > 0 ) 
    scale = 1 + m_data_2->m_tries_time;

  pos.set( get_size().x - 2*s_margin - m_small_plee.width()
	   - m_data_2->tries.get_width()*scale,
           get_size().y - s_margin - 
	   ( m_data_2->tries.get_height()*scale + m_small_plee.height())/2 );

  bear::visual::scene_writing s3( pos.x, pos.y, m_data_2->tries );
  s3.set_scale_factor(scale,scale);

  e.push_back( s3 );
} // status_layer::render_player2()


/*----------------------------------------------------------------------------*/
/**
 * \brief Search players.
 */
void ptb::status_layer::search_players( )
{
  if ( m_first_player.get() == NULL )
    {
      plee::get_instance_message msg;
      bear::engine::level_globals& glob = get_level_globals();
      bear::visual::font f = glob.get_font("font/bouncy.fnt");
      glob.send_message(plee::player_name(1),msg);
      m_first_player = msg.get_instance();

      if ( m_first_player.get() != NULL )
        m_data_1 = new player_status( glob, f, m_first_player );
    }

  if ( game_variables::get_players_count() == 2 )
    {
      if ( m_second_player.get() == NULL )
        {
	   plee::get_instance_message msg2;
	   bear::engine::level_globals& glob = get_level_globals();
	   bear::visual::font f = glob.get_font("font/bouncy.fnt");
	   
	   glob.send_message(plee::player_name(2),msg2);
	   m_second_player = msg2.get_instance();
	   
	   if ( m_second_player.get() != NULL )
	     m_data_2 = new player_status( glob, f, m_second_player );
	}
    }
} // status_layer::search_players( )
