/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/history/code/action_remove_item.cpp
 * \brief Implementation of the bf::action_remove_item class.
 * \author Julien Jorge
 */
#include "bf/history/action_remove_item.hpp"

#include "bf/gui_level.hpp"
#include "bf/wx_facilities.hpp"

#include <claw/assert.hpp>
#include <wx/intl.h>

/*----------------------------------------------------------------------------*/
bf::action_remove_item::action_remove_item
( item_instance* item, unsigned int lay )
  : m_item(item), m_layer(lay), m_done(false)
{

} // action_remove_item::action_remove_item()

/*----------------------------------------------------------------------------*/
bf::action_remove_item::~action_remove_item()
{
  /* If the action has been done, the item must be deleted here. Otherwise, it
     will be deleted by the level. */
  if ( m_done )
    delete m_item;
} // action_remove_item::~action_remove_item()

/*----------------------------------------------------------------------------*/
void bf::action_remove_item::execute( gui_level& lvl )
{
  CLAW_PRECOND( !m_done );
  CLAW_PRECOND( lvl.layers_count() > m_layer );

  m_selected = lvl.item_is_selected( m_layer, m_item );
  m_main_selection = lvl.item_is_main_selection( m_layer, m_item );

  lvl.remove_item( m_layer, m_item );

  m_done = true;
} // action_remove_item::execute()

/*----------------------------------------------------------------------------*/
void bf::action_remove_item::undo( gui_level& lvl )
{
  CLAW_PRECOND( m_done );
  CLAW_PRECOND( lvl.layers_count() > m_layer );

  lvl.get_layer( m_layer ).add_item( m_item );

  if ( m_selected )
    lvl.add_to_selection( m_layer, m_item, m_main_selection );

  m_done = false;
} // action_remove_item::undo()

/*----------------------------------------------------------------------------*/
bool bf::action_remove_item::is_identity( const gui_level& lvl ) const
{
  return false;
} // action_remove_item::is_identity()

/*----------------------------------------------------------------------------*/
wxString bf::action_remove_item::get_description() const
{
  return _("Remove %s")
    + std_to_wx_string(m_item->get_class().get_class_name());
} // action_remove_item::get_description()
