/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/item_choice_frame.cpp
 * \brief Implementation of the bf::item_choice_frame class.
 * \author Sebastien Angibaud
 */
#include "bf/item_choice_frame.hpp"

#include "bf/item_instance.hpp"
#include "bf/wx_facilities.hpp"

#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this window.
 * \param choice The items choice.
 * \param selected_item The item already selected.
 */
bf::item_choice_frame::item_choice_frame
( wxWindow* parent, const item_selection& selected_group,
  const std::set<item_instance*>& choice )
  : wxDialog(parent, wxID_ANY, wxString(_("Items choice"))),
    m_choice(choice), m_selected_group(selected_group)
{
  create_controls();
  Fit();
} // item_choice_frame::item_choice_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the selected items.
 */
void bf::item_choice_frame::get_selection
(std::set<item_instance*>& selected_items)
{
  std::set<item_instance*>::const_iterator it;
  unsigned int i;

  for ( i=0, it=m_choice.begin();
        i!=m_item_checklistbox->GetCount(); i++, it++ )
    if ( m_item_checklistbox->IsChecked(i)
         != m_selected_group.is_selected(*it) )
      selected_items.insert(*it);
} // item_choice_frame::get_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 * \param selected_item The item already selected.
 */
void bf::item_choice_frame::create_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  sizer->Add( create_list_sizer(), 0, wxEXPAND );

  sizer->Add( CreateStdDialogButtonSizer(wxOK |wxCANCEL), 0,
              wxALL | wxCENTER, 5 );

  SetSizer(sizer);
} // item_choice_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the item list.
 */
wxSizer* bf::item_choice_frame::create_list_sizer()
{
  wxStaticBoxSizer* result =
    new wxStaticBoxSizer( wxHORIZONTAL, this, _("Choice") );

  m_item_checklistbox = new wxCheckListBox( this, IDC_LAYER_LIST );

  std::set<item_instance*>::iterator it;
  unsigned int i;
  for ( it = m_choice.begin(), i=0; it != m_choice.end(); ++it, ++i)
    {
      std::ostringstream stm;
      stm << "class=" << (*it)->get_class().get_class_name() << ", "
          << "pos=[" << (*it)->get_rendering_parameters().get_left() << ", "
          << (*it)->get_rendering_parameters().get_bottom() << "] "
          << "size=[" << (*it)->get_rendering_parameters().get_width() << ", "
          << (*it)->get_rendering_parameters().get_height() << "]";

      if ( !(*it)->get_id().empty() )
        stm << " id='" << (*it)->get_id() << "'";

      wxString str = std_to_wx_string(stm.str());

      m_item_checklistbox->Append(str);
      m_item_checklistbox->Check(i, m_selected_group.is_selected(*it));
    }

  result->Add( m_item_checklistbox, 0, wxALL, 5 );

  return result;
} // item_choice_frame::create_list_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Ok" button.
 * \param event This event occured.
 */
void bf::item_choice_frame::on_ok(wxCommandEvent& event)
{
  EndModal(wxID_OK);
} // item_choice_frame::on_ok()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::item_choice_frame, wxDialog)
  EVT_BUTTON( wxID_OK, bf::item_choice_frame::on_ok )
END_EVENT_TABLE()
