/******************************************************************************
*                             plasma-am4rok                                   *
*******************************************************************************
*                                                                             *
*            Copyright (C) 2008 Giulio Camuffo				      *
*	     Copyright (C) 2008 John Varouhakis                               *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA  *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/

/* 19/03/2008 , author Giulio Camuffo 
adapted file to work with amaroK 2*/

/* 23/03/2008 , author Giulio Camuffo 
added some information in the dialog*/

/* 25/03/2008 , author Giulio Camuffo 
added a configuration dialog*/

/* 30/03/2008 , author Giulio Camuffo 
moved all dbus methods on another class*/

///am4rok headers
#include "am4rok_client.h"

///Plasma
#include <plasma/containment.h>

///QT
#include <KConfigGroup>

Am4rokClient::Am4rokClient(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args)
{
    setHasConfigurationInterface(true);
    setAspectRatioMode(Plasma::IgnoreAspectRatio);

    controller = new PlayerController();
    infoWidget = new MMInfoWidget(this);

    m_width = 230;
    m_height = 250;

    resize(m_width,m_height);
    setMinimumSize(QSizeF(m_width+25,10)); //if I set m_width instead of m_width+25 it covers part of the next applet

}

void Am4rokClient::init()
{

    KConfigGroup cg = config();

    QColor textColor = QColor("#ffffff"); //default color


    m_width = cg.readEntry("width", m_width); //defines the width of the applet
    textColor = cg.readEntry("textColor", textColor); //defines the color of the text
    int player = cg.readEntry("useWithPlayer", 0); //defines what player to use
    QColor textDialogColor = cg.readEntry("textDialogColor", textColor); //defines the color of the text in the dialog
    closeDialogOnFocusOut = cg.readEntry("closeDialogOnFocusOut", true); //defines is close or not the dialog on focus out
    bool scrollStrings = cg.readEntry("scrollStrings", false); //defines if scroll or resize too long strings
    showStrings = cg.readEntry("showStrings", true); //defines if show or not infos in panel

    controller->playerStatus();
    controller->setPlayer(player);

    infoWidget->setTextColor(textColor);
    infoWidget->setPlayer(player);
    infoWidget->setDialogTextColor(textDialogColor);
    infoWidget->setCloseDialogOnFocusOut(closeDialogOnFocusOut);
    infoWidget->setScrollStrings(scrollStrings);
    infoWidget->setShowStrings(showStrings);

    connect(infoWidget->progressBarObj(), SIGNAL(valueChanged(const int &)),
	    this,SLOT(changeProgressBarValue(const int &)));


    connect(infoWidget->controlWidgetObj(), SIGNAL(buttonPressed(const QString &)),
            controller,SLOT(changeState(const QString &)));
            
    connect(controller, SIGNAL(timeChanged(const QString &)),
            infoWidget,SLOT(updateTime(const QString &)));
    connect(controller, SIGNAL(titleChanged(const QString &)),
            infoWidget,SLOT(updateTitle(const QString &)));
    connect(controller, SIGNAL(albumChanged(const QString &)),
            infoWidget,SLOT(updateAlbum(const QString &)));
    connect(controller, SIGNAL(artistChanged(const QString &)),
            infoWidget,SLOT(updateArtist(const QString &)));
    connect(controller, SIGNAL(intTimeChanged(const int &)),
            infoWidget,SLOT(updateBar(const int &)));
    connect(controller, SIGNAL(trackChanged(const QString &)),
            infoWidget,SLOT(updateTrack(const QString &)));

    connect(controller, SIGNAL(statusChanged(const QString &)),
            infoWidget,SLOT(updateStatus(const QString &)));


}

Am4rokClient::~Am4rokClient(){
//  destroy();
  delete controller;
  delete infoWidget;
}

bool Am4rokClient::isOnDesktop(void)
{
    return (containment()->containmentType() == Plasma::Containment::DesktopContainment); //gets the tipe of containment
}
// 
void Am4rokClient::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::SizeConstraint)
	updateContentsGeometry();
}

QSizeF Am4rokClient::contentSizeHint(void)
{
    if (formFactor() == Plasma::Horizontal) {
	m_height = contentsRect().height();
    } else if (formFactor() == Plasma::Vertical) {
	m_width = contentsRect().width();
    } else if (formFactor() == Plasma::Planar) {
	m_width = contentsRect().width();
	m_height = contentsRect().height();
    }

    return QSizeF(m_width,m_height);
}

void Am4rokClient::updateContentsGeometry(void)
{
    infoWidget->setIsOnPanel(not isOnDesktop());    
    infoWidget->setContentSize(contentSizeHint());
}

void Am4rokClient::createConfigurationInterface(KConfigDialog *parent)
{   
    QWidget *widget = new QWidget();
    ui.setupUi(widget);
    parent->setButtons( KDialog::Ok | KDialog::Cancel | KDialog::Apply );
    parent->addPage(widget, parent->windowTitle(), icon());
    connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
    parent->resize(430,380);

    ui.textColor->setColor(infoWidget->getTextColor());
    ui.useWithPlayer->setCurrentIndex(controller->player());
    
    if (not isOnDesktop()){
	if (infoWidget->getCloseDialogOnFocusOut())
	    ui.closeDialogOnFocusOut->setCheckState(Qt::Checked);
	else
	    ui.closeDialogOnFocusOut->setCheckState(Qt::Unchecked);    

	ui.width->setValue(m_width);
	ui.tabConfig->setTabEnabled(1,true);
	
	ui.scrollStrings->setChecked(infoWidget->getScrollStrings());
	
	ui.textDialogColor->setColor(infoWidget->getDialogTextColor());
    
	if (showStrings)
	    ui.showStrings->setCheckState(Qt::Checked);
	else
	    ui.showStrings->setCheckState(Qt::Unchecked);
    }
    else
	ui.tabConfig->setTabEnabled(1,false);
}

void Am4rokClient::configAccepted()
{
    KConfigGroup cg = config();

    if (infoWidget->getTextColor() != ui.textColor->color()){
	infoWidget->setTextColor(ui.textColor->color());
	controller->forceUpdateInfos();
    }

    if (infoWidget->getDialogTextColor() != ui.textDialogColor->color()){
	infoWidget->setDialogTextColor(ui.textDialogColor->color());
	controller->forceUpdateInfos();    	
    }
    
    cg.writeEntry("textColor" ,ui.textColor->color());
    cg.writeEntry("textDialogColor" ,ui.textDialogColor->color());
    
    if (not isOnDesktop()){
      if (infoWidget->getCloseDialogOnFocusOut() != ui.closeDialogOnFocusOut->isChecked())
	infoWidget->setCloseDialogOnFocusOut(ui.closeDialogOnFocusOut->isChecked());
	cg.writeEntry("closeDialogOnFocusOut", infoWidget->getCloseDialogOnFocusOut());
	 
      if (infoWidget->getScrollStrings() != ui.scrollStrings->isChecked())
	infoWidget->setScrollStrings(ui.scrollStrings->isChecked()); 
      cg.writeEntry("scrollStrings", infoWidget->getScrollStrings());

      if (m_width != ui.width->value()){
	m_width = ui.width->value();
	setMinimumSize(contentSizeHint());

      }
      cg.writeEntry("width", m_width);

      if (showStrings != ui.showStrings->isChecked()){
	showStrings = ui.showStrings->isChecked();
	infoWidget->setShowStrings(showStrings); 
	infoWidget->recalculateGeometry();
      }
      cg.writeEntry("showStrings", showStrings); //show strings in panel     
    }

    int player = ui.useWithPlayer->currentIndex();
    if (controller->player() != ui.useWithPlayer->currentIndex()){
      controller->setPlayer(player);
      infoWidget->recalculateGeometry();
    }
    cg.writeEntry("useWithPlayer", player);

    emit configNeedsSaving();
}

void Am4rokClient::changeProgressBarValue(const int &value){

  double tot_time;
  int cur_time;
  if ((tot_time=controller->playerIntQuery("trackTotalTime")) >0){
    cur_time=(tot_time*value/100);
    controller->playerQuery(QString("seek"), cur_time);
    
  }
}

void Am4rokClient::wheelEvent(QGraphicsSceneWheelEvent *event){
  if (event->delta() > 0)
    controller->playerQuery("seekForward");
  else
    controller->playerQuery("seekBack");
}

void Am4rokClient::hoverEnterEvent(QGraphicsSceneHoverEvent *)
{
  if ((not isOnDesktop()) and (showStrings)){
   infoWidget->controlWidgetObj()->show();  
  }
}

void Am4rokClient::hoverLeaveEvent(QGraphicsSceneHoverEvent *){
  if ((not isOnDesktop()) and (showStrings)){
    infoWidget->controlWidgetObj()->hide();  
  }
}

#include "am4rok_client.moc"
