/***********************************************************************************
* Spell Check: Plasmoid for fast spell checking.
* Copyright (C) 2008 - 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "plasma-spellcheck.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsLinearLayout>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QClipboard>
#include <QApplication>
#include <QMenu>

#include <Plasma/Theme>
#include <Plasma/ToolTipManager>

#include <KIcon>
#include <KLocale>

void SpellCheckTextEdit::contextMenuEvent(QContextMenuEvent *event)
{
    QTextCursor cursor = cursorForPosition(event->pos());
    cursor.select(QTextCursor::WordUnderCursor);

    QMenu *menu = createStandardContextMenu();

    if (!cursor.selectedText().isEmpty() && !m_checker->checkWord(cursor.selectedText()))
    {
        QStringList suggestions = m_checker->suggest(cursor.selectedText());

        if (!suggestions.isEmpty())
        {
            menu->addSeparator();

            QMenu *spellMenu = menu->addMenu(i18n("Suggestions"));

            for (int i = 0; i < suggestions.size(); ++i)
            {
                QAction *action = spellMenu->addAction(suggestions.at(i));
                action->setData(event->pos());
            }

            connect(spellMenu, SIGNAL(triggered(QAction*)), this, SLOT(replaceWord(QAction*)));
        }
    }

    menu->exec(event->globalPos());

    delete menu;
}

void SpellCheckTextEdit::setSpellChecker(Sonnet::BackgroundChecker *checker)
{
    m_checker = checker;
}

void SpellCheckTextEdit::replaceWord(QAction *action)
{
    if (action->data().type() == QVariant::Point)
    {
        QTextCursor cursor = cursorForPosition(action->data().toPoint());
        cursor.select(QTextCursor::WordUnderCursor);

        setText(toPlainText().replace(cursor.anchor(), (cursor.position() - cursor.anchor()), action->text()));
    }
}


SpellCheck::SpellCheck(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    setAspectRatioMode(Plasma::ConstrainedSquare);
    setAcceptDrops(true);

    int iconSize = IconSize(KIconLoader::Desktop);

    resize((iconSize * 2), (iconSize * 2));
}

SpellCheck::~SpellCheck()
{
    m_dialog->hide();
}

void SpellCheck::init()
{
    KConfigGroup configuration = config();

    m_icon = new Plasma::IconWidget(KIcon("tools-check-spelling"), QString(), this);

    registerAsDragHandle(m_icon);

    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addItem(m_icon);

    Plasma::ToolTipContent toolTipData(i18n("Check spelling"), i18n("Check spelling of clipboard content"), m_icon->icon().pixmap(IconSize(KIconLoader::Desktop)));

    Plasma::ToolTipManager::self()->setContent(this, toolTipData);

    m_textEdit = new SpellCheckTextEdit;

    m_highlighter = new Sonnet::Highlighter(m_textEdit);

    m_checker = new Sonnet::BackgroundChecker;

    m_textEdit->setSpellChecker(m_checker);

    m_checkerDialog = new Sonnet::Dialog(m_checker, NULL);

    m_dictionaryComboBox = new Sonnet::DictionaryComboBox;
    m_dictionaryComboBox->setToolTip(i18n("Language"));

    m_copyButton = new QToolButton;
    m_copyButton->setIcon(KIcon("edit-copy"));
    m_copyButton->setToolTip(i18n("Copy"));

    m_spellingButton = new QToolButton;
    m_spellingButton->setIcon(KIcon("tools-check-spelling"));
    m_spellingButton->setToolTip(i18n("Spell checking"));

    QHBoxLayout *horizontalLayout = new QHBoxLayout;
    horizontalLayout->addWidget(m_dictionaryComboBox);
    horizontalLayout->addWidget(m_spellingButton);
    horizontalLayout->addWidget(m_copyButton);

    m_dialog = new Plasma::Dialog();
    m_dialog->setFocusPolicy(Qt::NoFocus);
    m_dialog->setWindowFlags(Qt::Tool);
    m_dialog->setResizeHandleCorners(Plasma::Dialog::All);
    m_dialog->resize(configuration.readEntry("dialogSize", m_dialog->size()));

    QVBoxLayout *verticalLayout = new QVBoxLayout(m_dialog);
    verticalLayout->setSpacing(0);
    verticalLayout->setMargin(0);
    verticalLayout->addWidget(m_textEdit);
    verticalLayout->addLayout(horizontalLayout);

    setLanguage(configuration.readEntry("dictionary", m_highlighter->currentLanguage()));

    connect(this, SIGNAL(activate()), this, SLOT(toggleDialog()));
    connect(m_icon, SIGNAL(clicked()), this, SLOT(toggleDialog()));
    connect(m_dialog, SIGNAL(dialogResized()), this, SLOT(dialogResized()));
    connect(m_dictionaryComboBox, SIGNAL(dictionaryChanged(QString)), this, SLOT(setLanguage(QString)));
    connect(m_copyButton, SIGNAL(clicked()), this, SLOT(copyToClipboard()));
    connect(m_spellingButton, SIGNAL(clicked()), this, SLOT(showCheckerDialog()));
}

void SpellCheck::toggleDialog()
{
    if (m_dialog->isVisible())
    {
        m_dialog->hide();

        m_textEdit->clear();
    }
    else
    {
        m_dialog->move(popupPosition(m_dialog->sizeHint()));

        m_dialog->show();

        m_textEdit->setText(QApplication::clipboard()->text());
    }
}

void SpellCheck::dialogResized()
{
    KConfigGroup configuration = config();

    configuration.writeEntry("dialogSize", m_dialog->size());

    emit configNeedsSaving();
}

void SpellCheck::copyToClipboard()
{
    if (!m_textEdit->textCursor().selectedText().isEmpty())
    {
        QApplication::clipboard()->setText(m_textEdit->textCursor().selectedText());
    }
    else
    {
        QApplication::clipboard()->setText(m_textEdit->toPlainText());
    }
}

void SpellCheck::showCheckerDialog()
{
    QString buffer;

    m_selectionStart = 0;

    if (!m_textEdit->textCursor().selectedText().isEmpty())
    {
        buffer = m_textEdit->textCursor().selectedText();

        m_selectionStart = m_textEdit->textCursor().anchor();
    }
    else
    {
        buffer = m_textEdit->toPlainText();
    }

    m_checkerDialog->setBuffer(buffer);
    m_checkerDialog->show();
}

void SpellCheck::setLanguage(QString language)
{
    KConfigGroup configuration = config();

    m_highlighter->setCurrentLanguage(language);
    m_highlighter->slotRehighlight();

    m_checker->changeLanguage(language);

    m_checkerDialog->disconnect();

    delete m_checkerDialog;

    m_checkerDialog = new Sonnet::Dialog(m_checker, NULL);

    m_dictionaryComboBox->setCurrentByDictionary(language);

    configuration.writeEntry("dictionary", language);

    emit configNeedsSaving();

    connect(m_checkerDialog, SIGNAL(languageChanged(QString)), this, SLOT(setLanguage(QString)));
    connect(m_checkerDialog, SIGNAL(misspelling(QString, int)), this, SLOT(highlightWord(QString, int)));
    connect(m_checkerDialog, SIGNAL(replace(QString, int, QString)), this, SLOT(replaceWord(QString, int, QString)));
}

void SpellCheck::highlightWord(const QString word, int start)
{
    QTextCursor textCursor(m_textEdit->textCursor());
    textCursor.setPosition((m_selectionStart + start), QTextCursor::MoveAnchor);
    textCursor.setPosition((m_selectionStart + start + word.length()), QTextCursor::KeepAnchor);

    m_textEdit->setTextCursor(textCursor);
}

void SpellCheck::replaceWord(const QString oldWord, int start, const QString newWord)
{
    m_textEdit->setText(m_textEdit->toPlainText().replace((m_selectionStart + start), oldWord.length(), newWord));
}

#include "plasma-spellcheck.moc"
