;;; planner-timeclock.el --- Timeclock integration for the Emacs Planner

;; Copyright (C) 2001 John Wiegley

;; Author: John Wiegley <johnw@gnu.org>
;; Maintainer: Sacha Chua <sacha@free.net.ph>
;; Keywords: planner, timeclock
;; URL: http://sacha.free.net.ph/notebook/wiki/PlannerMode.php#timeclock

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This module allows you to clock in and clock out of your projects.
;; You can also generate reports with the <timeclock-report> tag.
;; timeclock.el is part of GNU Emacs and can be downloaded from
;;
;;    http://sacha.free.net.ph/notebook/emacs/timeclock.el

;;; Code:

(require 'planner)
(require 'timeclock)
(require 'planner-schedule)
(require 'easymenu)

(defvar planner-timeclock-current-task nil
  "Description of current task.")

(defalias 'planner-timeclock-in 'planner-task-in-progress)

(defadvice planner-task-in-progress (after planner-timeclock activate)
  "Clock in on this task."
  (let* ((task-info (planner-current-task-info))
         (project (if (planner-task-plan task-info)
                      (concat (planner-task-plan task-info) ": "
                              (planner-task-description task-info))
                    (planner-task-description task-info))))
    (setq planner-timeclock-current-task
          (planner-task-description task-info))
    (if (timeclock-currently-in-p)
        (timeclock-change nil project)
      (timeclock-in nil project))))

(defadvice timeclock-out (after planner-timeclock activate)
  "Clear `planner-timeclock-current-task.'"
  (setq planner-timeclock-current-task nil))

(defadvice planner-task-done (after planner-timeclock activate)
  "Clock out if the task description matches the one clocked in."
  (when (and planner-timeclock-current-task
             (string= (planner-task-description (planner-current-task-info))
                      planner-timeclock-current-task)
             (timeclock-currently-in-p))
    (timeclock-out 1)))

(defun planner-timeclock-report-tag (beg end highlight-p)
  "Replace the region BEG to END with a timeclock report.
If HIGHLIGHT-P is non-nil, the output is also displayed."
  (require 'timeclock)
  (if highlight-p
      (add-text-properties
       beg end (list 'display
                     (with-temp-buffer
                       (timeclock-generate-report emacs-wiki-publishing-p)
                       (buffer-string))))
    (delete-region beg end)
    (timeclock-generate-report emacs-wiki-publishing-p)
    (add-text-properties beg (point) '(read-only t))))

(add-to-list 'planner-markup-tags
             '("timeclock-report" nil nil t planner-timeclock-report-tag))
(planner-option-customized 'planner-markup-tags planner-markup-tags)

(let ((map planner-mode-map))
  (define-key map "\C-c\C-i" 'planner-task-in-progress)
  (define-key map "\C-c\C-o" 'timeclock-out)
  (if (featurep 'xemacs)
      (progn
        (define-key map (kbd "C-c C-T C-i") 'planner-task-in-progress)
        (define-key map (kbd "C-c C-T C-o") 'timeclock-out))
    (define-key map (kbd "C-c C-S-t C-i") 'planner-task-in-progress)
    (define-key map (kbd "C-c C-S-t C-o") 'timeclock-out)))

(easy-menu-add-item planner-mode-map
                    '("Planner")
                    (easy-menu-create-menu
                     "Timeclock"
                     '(["Clock in" planner-timeclock-in]
                       ["Clock out" timeclock-out])))

(provide 'planner-timeclock)

;;; planner-timeclock.el ends here
