/*
 *  PlanFacile (Easy plan, in french) is a small tool to help people to
 *  write a document on a particular subject.
 *  Copyright (C) 2005  Julien BRUGUIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "environnement.h"

#define REPTAILLEINIT			128
#define REPMULTTAILLE			2
#define CARACTEREREPERTOIRE		'/'
#define CARACTEREECHAPREPERTOIRE	'\\'
#define CARACTEREREPERTOIREUTILISATEUR	'~'
#define CHEMINREPERTOIREUTILISATEUR	"HOME"

Resultat environnement_initialisation(TRAVAIL(Environnement) environnement)
{
	/* Initialise un environnement.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE en cas d'echec d'une allocation.
	 */
	DEBUTZONESECURISEE
	if((S_T(environnement)=(STOCKAGE(Environnement))(malloc(SIZEOF(CONTENEUR(Environnement)))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	SECURISE(options_initialisation(CHAMP_TRAVAIL(environnement , options)));
	SECURISE(pilenomfichier_initialisation(CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	CHAMP(environnement , fichiersortie)=NULL;
	CHAMP(environnement , sortie)=NULL;
	CHAMP(environnement , erreur)=NULL;
	CHAMP(environnement , langue)=LANGUE_INCONNUE;
	CHAMP(environnement , charset)=CHARSET_INCONNU;
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_definition_options(TRAVAIL(Environnement) environnement , TRAVAIL(Options) options)
{
	/* Dfinit une liste d'options pour un environnement.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(options_copie(options,CHAMP_TRAVAIL(environnement , options)));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_definition_fichiersortie(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(NomFichier) fichier)
{
	/* Dfinit un nom de fichier de sortie pour un environnement.
	 * Si le nom est NULL, la sortie standard sera utilise.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	if(CHAMP(environnement , fichiersortie)!=NULL)
		free(CHAMP(environnement , fichiersortie));
	if(S_T_(fichier)==NULL)
		CHAMP(environnement , fichiersortie)=NULL;
	else
	{
		if((CHAMP(environnement , fichiersortie)=(STOCKAGE_SCALAIRE(NomFichier))(strdup((char*)(fichier))))==NULL)
			return RESULTAT_ERREUR_MEMOIRE;
	}
	return RESULTAT_OK;
}

Resultat environnement_definition_sortie(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(DescripteurFichier) sortie)
{
	/* Dfinit un descripteur de fichier de sortie pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	CHAMP(environnement , sortie)=S_T_(sortie);
	return RESULTAT_OK;
}

Resultat environnement_definition_erreur(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(DescripteurFichier) erreur)
{
	/* Dfinit un descripteur de fichier d'erreur pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	CHAMP(environnement , erreur)=S_T_(erreur);
	return RESULTAT_OK;
}

Resultat environnement_definition_pilerepertoirecourant(TRAVAIL(Environnement) environnement , TRAVAIL(PileNomFichier) pilerepertoirecourant)
{
	/* Dfinit une pile de rpertoires courants pour un environnement.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_copie(pilerepertoirecourant,CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_definition_langue(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(Langue) langue)
{
	/* Dfinit une langue pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	CHAMP(environnement , langue)=S_T_(langue);
	return RESULTAT_OK;
}

Resultat environnement_definition_charset(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(Charset) charset)
{
	/* Dfinit un jeu de caractres pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	CHAMP(environnement , charset)=S_T_(charset);
	return RESULTAT_OK;
}

Resultat environnement_lecture_options(TRAVAIL(Environnement) environnement , REFERENCE(Options) options)
{
	/* Lit une liste d'options pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R(options)=CHAMP_TRAVAIL(environnement , options);
	return RESULTAT_OK;
}

Resultat environnement_lecture_fichiersortie(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(NomFichier) fichier)
{
	/* Lit un nom de fichier de sortie pour un environnement.
	 * Si la valeur renvoye est NULL, la sortie standard doit
	 * tre utilise.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R_(fichier)=T_S_(CHAMP(environnement , fichiersortie));
	return RESULTAT_OK;
}

Resultat environnement_lecture_sortie(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(DescripteurFichier) sortie)
{
	/* Lit un descripteur de fichier de sortie pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R_(sortie)=T_S_(CHAMP(environnement , sortie));
	return RESULTAT_OK;
}

Resultat environnement_lecture_erreur(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(DescripteurFichier) erreur)
{
	/* Lit un descripteur de fichier d'erreur pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R_(erreur)=T_S_(CHAMP(environnement , erreur));
	return RESULTAT_OK;
}

Resultat environnement_lecture_pilerepertoirecourant(TRAVAIL(Environnement) environnement , REFERENCE(PileNomFichier) pilerepertoirecourant)
{
	/* Lit une pile de noms de fichier pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R(pilerepertoirecourant)=CHAMP_TRAVAIL(environnement , pilerepertoirecourant);
	return RESULTAT_OK;
}

Resultat environnement_lecture_langue(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(Langue) langue)
{
	/* Lit une langue pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R_(langue)=T_S_(CHAMP(environnement , langue));
	return RESULTAT_OK;
}

Resultat environnement_lecture_charset(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(Charset) charset)
{
	/* Lit un jeu de caractres pour un environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL.
	 */
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	T_R_(charset)=T_S_(CHAMP(environnement , charset));
	return RESULTAT_OK;
}

Resultat environnement_lecture_repertoirecourant(REFERENCE_SCALAIRE(NomFichier) repertoirecourant)
{
	/* Renvoie le nom du rpertoire courant.
	 * Alloue directement la chaine, qui devra tre libre
	 * avec free().
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	STOCKAGE_SCALAIRE(NomFichier) repertoire;
	STOCKAGE_SCALAIRE(Taille) taille;
	if((repertoire=((STOCKAGE_SCALAIRE(NomFichier))(malloc(SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*T_S_(REPTAILLEINIT)))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	taille=REPTAILLEINIT;
	for(;;)
	{
		if(getcwd((char*)(T_S_(repertoire)),(size_t)(T_S_(taille)))==T_S_(NULL))
		{
			STOCKAGE_SCALAIRE(NomFichier) repertoiretemp;
			if((repertoiretemp=((STOCKAGE_SCALAIRE(NomFichier))(realloc((void*)(repertoire),SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*T_S_(taille)*T_S_(REPMULTTAILLE)))))==NULL)
				return RESULTAT_ERREUR_MEMOIRE;
			repertoire=repertoiretemp;
			taille*=T_S_(REPMULTTAILLE);
		}
		else
		{
			if((((char*)(T_S_(repertoire)))[1])==T_S_('\0'))
			{
				T_R_(repertoirecourant)=T_S_(repertoire);
				return RESULTAT_OK;
			}
			if(((T_R_(repertoirecourant))=(STOCKAGE_SCALAIRE(NomFichier))(realloc((void*)(repertoire),SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*strlen(T_S_(repertoire))+T_S_(2))))==NULL)
				return RESULTAT_ERREUR_MEMOIRE;
			strcat((char*)(T_R_(repertoirecourant)),T_S_("/"));
			return RESULTAT_OK;
		}
	}
	return RESULTAT_OK;
}

Resultat environnement_lecture_repertoirefichier(TRAVAIL_SCALAIRE(NomFichier) repertoirefichier , REFERENCE_SCALAIRE(NomFichier) repertoire , REFERENCE_SCALAIRE(NomFichier) fichier)
{
	/* Renvoie la partie rpertoire d'un nom de fichier
	 * dans une chaine automatiquement alloue. Il faudra
	 * librer cette chaine par free(). De mme pour le
	 * nom de fichier.
	 * Renvoie RESULTAT_ERREUR si fichier est NULL, et
	 * RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	enum
	{
		ETAT_NORMAL,
		ETAT_ECHAP
	} etat;
	REFERENCE_SCALAIRE(Caractere) position;
	REFERENCE_SCALAIRE(Caractere) parcours;
	STOCKAGE_SCALAIRE(Booleen) cheminvide;
	etat=ETAT_NORMAL;
	position=repertoirefichier;
	cheminvide=VRAI;
	for(parcours=((REFERENCE_SCALAIRE(Caractere))(repertoirefichier)) ; (T_R_(parcours))!=T_S_('\0') ; parcours++)
	{
		switch(T_S_(etat))
		{
			case T_S_(ETAT_NORMAL):
				if(T_R_(parcours)==T_S_(CARACTEREREPERTOIRE))
				{
					position=parcours;
					cheminvide=FAUX;
				}
				etat=ETAT_NORMAL;
				if(T_R_(parcours)==T_S_(CARACTEREECHAPREPERTOIRE))
					etat=ETAT_ECHAP;
				break;
			case T_S_(ETAT_ECHAP):
				etat=ETAT_NORMAL;
				break;
			default:
				return RESULTAT_ERREUR;
				break;
		}
	}
	if(T_S_(cheminvide)==T_S_(VRAI))
	{
		if((T_R_(repertoire)=T_S_((STOCKAGE_SCALAIRE(NomFichier))(strdup(T_S_("./")))))==NULL)
			return RESULTAT_ERREUR_MEMOIRE;
		if((T_R_(fichier)=T_S_((STOCKAGE_SCALAIRE(NomFichier))(strdup((char*)(T_S_(repertoirefichier))))))==NULL)
			return RESULTAT_ERREUR_MEMOIRE;
		return RESULTAT_OK;
	}
	else
	{
		REFERENCE_SCALAIRE(Caractere) ecriture;
		if((T_R_(repertoire)=T_S_((STOCKAGE_SCALAIRE(NomFichier))(malloc(SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*((TRAVAIL_SCALAIRE(Entier))(position-((REFERENCE_SCALAIRE(Caractere))(repertoirefichier)))+T_S_(2))))))==T_S_(NULL))
			return RESULTAT_ERREUR_MEMOIRE;
		ecriture=(REFERENCE_SCALAIRE(Caractere))(T_R_(repertoire));
		for(parcours=((REFERENCE_SCALAIRE(Caractere))(repertoirefichier)) ; parcours<=position ; parcours++)
			T_R_(ecriture++)=T_R_(parcours);
		T_R_(ecriture)=T_S_('\0');
		if(((T_R_(fichier))=T_S_((STOCKAGE_SCALAIRE(NomFichier))(strdup((char *)(position+1)))))==T_S_(NULL))
			return RESULTAT_ERREUR_MEMOIRE;
		return RESULTAT_OK;
	}
}

Resultat environnement_ajout_repertoirefichier(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(NomFichier) fichier , REFERENCE_SCALAIRE(NomFichier) fichierabsolu , REFERENCE_SCALAIRE(Booleen) chrep)
{
	/* Prpare l'inclusion ou l'ouverture d'un fichier,
	 * en se plaant dans le rpertoire du fichier,
	 * et en renvoyant le nom du fichier avec son chemin
	 * absolu.
	 * Le boolen indique si l'on a bien pu changer de
	 * rpertoire.
	 */
	TRAVAIL_SCALAIRE(NomFichier) chemin;
	TRAVAIL_SCALAIRE(NomFichier) repertoire;
	TRAVAIL_SCALAIRE(NomFichier) nomfichier;
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(environnement_lecture_repertoirecourant(R_T_(chemin)));
	SECURISE(pilenomfichier_ajout(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),chemin));
	free(S_T_(chemin));
	SECURISE(environnement_lecture_repertoirefichier(fichier,R_T_(repertoire),R_T_(nomfichier)));
	if((T_S_(((char*)(repertoire))[0])==T_S_(CARACTEREREPERTOIREUTILISATEUR))&&(T_S_(((char*)(repertoire))[0])!=T_S_('\0'))&&(T_S_(((char*)(repertoire))[1])==T_S_(CARACTEREREPERTOIRE)))
	{
		STOCKAGE_SCALAIRE(NomFichier) home;
		if((home=getenv(T_S_(CHEMINREPERTOIREUTILISATEUR)))==NULL)
		{
			free(S_T_(repertoire));
			free(S_T_(nomfichier));
			{
				STOCKAGE_SCALAIRE(NomFichier) repertoire;
				SECURISE(pilenomfichier_retrait(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),C_S_(repertoire)));
				free(repertoire);
			}
			T_R_(fichierabsolu)=T_S_(NULL);
			T_R_(chrep)=T_S_(FAUX);
			return RESULTAT_OK;
		}
		if((S_T_(chemin)=(STOCKAGE_SCALAIRE(NomFichier))(malloc(SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*(T_S_(strlen(repertoire))+T_S_(strlen(T_S_(home)))+T_S_(1)))))==NULL)
			return RESULTAT_ERREUR_MEMOIRE;
		strcpy((char*)(T_S_(chemin)),(char*)(T_S_(home)));
		strcat((char*)(T_S_(chemin)),((char*)(T_S_(repertoire))+1));
		free(S_T_(repertoire));
		S_T_(repertoire)=S_T_(chemin);
	}
	if(chdir(repertoire)!=0)
	{
		free(S_T_(repertoire));
		free(S_T_(nomfichier));
		{
			STOCKAGE_SCALAIRE(NomFichier) repertoire;
			SECURISE(pilenomfichier_retrait(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),C_S_(repertoire)));
			free(repertoire);
		}
		T_R_(fichierabsolu)=T_S_(NULL);
		T_R_(chrep)=T_S_(FAUX);
		return RESULTAT_OK;
	}
	free(S_T_(repertoire));
	SECURISE(environnement_lecture_repertoirecourant(R_T_(chemin)));
	if(((T_R_(fichierabsolu))=T_S_((STOCKAGE_SCALAIRE(NomFichier))(malloc(SIZEOF_(CONTENEUR_SCALAIRE(Caractere))*(T_S_(strlen(chemin))+T_S_(strlen(nomfichier))+T_S_(1))))))==T_S_(NULL))
		return RESULTAT_ERREUR_MEMOIRE;
	strcpy((char*)(T_R_(fichierabsolu)),(char*)(chemin));
	strcat((char*)(T_R_(fichierabsolu)),(char*)(nomfichier));
	free(S_T_(chemin));
	free(S_T_(nomfichier));
	T_R_(chrep)=T_S_(VRAI);
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_retrait_repertoirefichier(TRAVAIL(Environnement) environnement)
{
	/* Effectue les oprations sur le rpertoire courant
	 * lors de la fin de la lecture d'un fichier source.
	 */
	STOCKAGE_SCALAIRE(NomFichier) repertoire;
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_retrait(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),C_S_(repertoire)));
	if(chdir(T_S_(repertoire))!=T_S_(0))
		return RESULTAT_ERREUR;
	free(repertoire);
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_initialisation_pilenomfichier(TRAVAIL(Environnement) environnement)
{
	/* Cre une pile de nom de fichiers.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_initialisation(CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_ajout_pilenomfichier(TRAVAIL(Environnement) environnement , TRAVAIL_SCALAIRE(NomFichier) nomfichier)
{
	/* Ajoute un nom de fichier au sommet de la pile.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_ajout(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),nomfichier));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_retrait_pilenomfichier(TRAVAIL(Environnement) environnement , COREFERENCE_SCALAIRE(NomFichier) nomfichier)
{
	/* Retire le sommet de la pile, et renvoye le nom de fichier.
	 * Renvoie RESULTAT_ERREUR si la pile est vide.
	 * Il faudra librer le nom de fichier renvoy  la main.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_retrait(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),nomfichier));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_vide_pilenomfichier(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(Booleen) vide)
{
	/* Renvoie VRAI dans vide si la pile est vide.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_vide(CHAMP_TRAVAIL(environnement , pilerepertoirecourant),vide));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_destruction_pilenomfichier(TRAVAIL(Environnement) environnement)
{
	/* Dtruit une pile de nom de fichiers.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(pilenomfichier_destruction(CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_copie(TRAVAIL(Environnement) environnement , TRAVAIL(Environnement) copie)
{
	/* Cre une copie de l'environnement.
	 * Renvoie RESULTAT_ERREUR si environnement est NULL, et
	 * RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 * Si la copie est non vide, elle est au pralable supprime.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(environnement_destruction(copie));
	SECURISE(environnement_initialisation(copie));
	SECURISE(environnement_definition_options(copie,CHAMP_TRAVAIL(environnement , options)));
	SECURISE(environnement_definition_fichiersortie(copie,T_S_(CHAMP(environnement , fichiersortie))));
	SECURISE(environnement_definition_sortie(copie,T_S_(CHAMP(environnement , sortie))));
	SECURISE(environnement_definition_erreur(copie,T_S_(CHAMP(environnement , erreur))));
	SECURISE(environnement_definition_pilerepertoirecourant(copie,CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	SECURISE(environnement_definition_langue(copie,T_S_(CHAMP(environnement , langue))));
	SECURISE(environnement_definition_charset(copie,T_S_(CHAMP(environnement , charset))));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat environnement_destruction(TRAVAIL(Environnement) environnement)
{
	/* Dtruit un environnement.
	 */
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_OK;
	free(CHAMP(environnement , fichiersortie));
	SECURISE(options_destruction(CHAMP_TRAVAIL(environnement , options)));
	SECURISE(pilenomfichier_destruction(CHAMP_TRAVAIL(environnement , pilerepertoirecourant)));
	free(S_T(environnement));
	S_T(environnement)=NULL;
	FINZONESECURISEE
	return RESULTAT_OK;
}

