/***************************************************************************
 *            qof-expenses.c
 *
 *  Thu Oct 21 07:59:13 2004-2005
 *  Copyright  2004-2005  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
/** @addtogroup QOF
	@{ */
/** @addtogroup Exp
	@{ */
/** @file  qof-expenses.c
	@brief QOF expense implementation for pilot-link
	@author Copyright (c) 2004-2005 Neil Williams <linux@codehelp.co.uk>
*/

#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <qof.h>
#include "pi-expense.h"
#include "qof-main.h"
#include "pilot-qof.h"
#include "qof-expenses.h"

#define QOF_EXPENSES_DESC "Pilot-link QOF expenses"

/** Inherited from pilot-link because the exported form using makelong
is not constant. */
#define EXPENSE_CREATOR "exps"

static QofLogModule log_module = PQ_MOD_PILOT;

/** \brief Currency Table data
	
Table relating the currencies[5] to the actual currency names, mnemonics
 and symbols. In pilot-qof it is indexed by GINT_TO_POINTER(pq_code). */
static GHashTable *PQCurrencyTable = NULL;

/** \brief Pilot-expenses currency data wrapper. */
typedef struct PQCurrency_s
{
	gint pq_code;			   /**< pilot-link integer ID */
	gint fraction;			   /**< currency fraction, pennies/cents etc. */
	const gchar *symbol;	   /**< locale symbol. */
	const gchar *mnemonic;	   /**< gnucash/cashutil mnemonic. */
	gboolean non_utf8;		   /**< TRUE if symbol is not UTF-8. (Currently,
					this is only useful if the symbol can still be shown
					as iso-8859-1.) */
} PQCurrency;

/** \brief Populate the currency table with the known currencies.

 Custom currencies are not included.

 All mnemonics are from gnucash and use the ISO4217 namespace

 custom currencies are the same and may end up with the same
 struct: gchar name[16], gchar symbol[4], gchar rate[8].

 Ignore currencies[5] (the five available currencies
 from the main table above) but:

 \todo store currencies[4] from AppInfo which
 are the four custom currencies.
*/

static void
populate_currencies (void)
{
	PQCurrencyTable = g_hash_table_new (g_direct_hash, g_direct_equal);
	{	/* Australia    0   100     AU$     AUD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 0;
		c->fraction = 100;
		c->symbol = "AU$";
		c->mnemonic = "AUD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Austria  1   100     €     ATS */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 1;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ATS";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Belgium      2   100     €     BEF */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 2;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "BEF";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Brazil   3   100     R$      BRL */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 3;
		c->fraction = 100;
		c->symbol = "R$";
		c->mnemonic = "BRL";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Canada   4   100     $CN     CAD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 4;
		c->fraction = 100;
		c->symbol = "$CN";
		c->mnemonic = "CAD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Denmark      5   100     DKK     DKK */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 5;
		c->fraction = 100;
		c->symbol = "DKK";
		c->mnemonic = "DKK";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Finland      6   100     €     FIM */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 6;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "FIM";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* France       7   100     €     FRF */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 7;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "FRF";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Germany      8   100     €     DEM */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 8;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "DEM";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Hong Kong    9   100     HK$     HKD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 9;
		c->fraction = 100;
		c->symbol = "HK$";
		c->mnemonic = "HKD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Iceland  10  100     ISK     ISK */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 10;
		c->fraction = 100;
		c->symbol = "ISK";
		c->mnemonic = "ISK";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Ireland      11  100     €     IEP */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 11;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "IEP";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Italy    12  1       EUR     ITL */
		/* The Italian Lira had a fraction == 1 */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 12;
		c->fraction = 100;
		c->symbol = "EUR";
		c->mnemonic = "ITL";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Japan        13  1   &#165;  ¥      JPY */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 13;
		c->fraction = 1;
		c->symbol = "¥";
		c->mnemonic = "JPY";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Luxembourg   14  100 &#136;  €     LUF */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 14;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "LUF";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Mexico       15  100     MXP     MXP */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 15;
		c->fraction = 100;
		c->symbol = "MXP";
		c->mnemonic = "MXP";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Netherlands  16  100     €     ANG */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 16;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ANG";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* New Zealand  17  100     $NZ     NZD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 17;
		c->fraction = 100;
		c->symbol = "$NZ";
		c->mnemonic = "NZD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Norway       18  100     NOK     NOK */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 18;
		c->fraction = 100;
		c->symbol = "NOK";
		c->mnemonic = "NOK";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Spain    19  100     €     ESP */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 19;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ESP";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Sweden   20  100     SEK     SEK */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 20;
		c->fraction = 100;
		c->symbol = "SEK";
		c->mnemonic = "SEK";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Switzerland  21  100     CHF     CHF */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 21;
		c->fraction = 100;
		c->symbol = "CHF";
		c->mnemonic = "CHF";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* United Kingdom 22    100     £      GBP */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 22;
		c->fraction = 100;
		c->symbol = "£";
		c->mnemonic = "GBP";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* United States  23    100     $US     USD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 23;
		c->fraction = 100;
		c->symbol = "$US";
		c->mnemonic = "USD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* India        24  100     Rs      INR */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 24;
		c->fraction = 100;
		c->symbol = "Rs";
		c->mnemonic = "INR";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Indonesia    25  1       Rp      IDR */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 25;
		c->fraction = 1;
		c->symbol = "Rp";
		c->mnemonic = "IDR";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Korea    26  100     KRW     KRW  (South) */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 26;
		c->fraction = 100;
		c->symbol = "KRW";
		c->mnemonic = "KRW";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Malaysia     27  100     RM      MYR */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 27;
		c->fraction = 100;
		c->symbol = "RM";
		c->mnemonic = "MYR";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* P.R.C. // People's Rep. China 28 100 RMB     CNY */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 28;
		c->fraction = 100;
		c->symbol = "RMB";
		c->mnemonic = "CNY";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Phillipines  29  100     P       PHP */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 29;
		c->fraction = 100;
		c->symbol = "P";
		c->mnemonic = "PHP";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Singapore    30  100     $       SGD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 30;
		c->fraction = 100;
		c->symbol = "$";
		c->mnemonic = "SGD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Thailand     31  100     BHT     THB */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 31;
		c->fraction = 100;
		c->symbol = "BHT";
		c->mnemonic = "THB";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* Taiwan   32  100     NT$     TWD */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 32;
		c->fraction = 100;
		c->symbol = "NT$";
		c->mnemonic = "TWD";
		c->non_utf8 = FALSE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
	{	/* EU (Euro)    133 100     €     EUR */
		PQCurrency *c = g_new0 (PQCurrency, 1);
		c->pq_code = 133;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "EUR";
		c->non_utf8 = TRUE;
		g_hash_table_insert (PQCurrencyTable,
			GINT_TO_POINTER (c->pq_code), (gpointer) c);
	}
}

static PQCurrency *
pq_currency_lookup (QofInstance * inst, gint currency_code)
{
	PQCurrency *currency;

	currency = NULL;
	if (!PQCurrencyTable)
		populate_currencies ();
	currency =
		(PQCurrency *) g_hash_table_lookup (PQCurrencyTable,
		GINT_TO_POINTER (currency_code));
	if (!currency)
	{
		PERR (" unsupported currency! %d", currency_code);
		return NULL;
	}
	kvp_frame_set_string (qof_instance_get_slots (inst),
		PQ_CURRENCY_MNEMONIC, currency->mnemonic);
	kvp_frame_set_gint64 (qof_instance_get_slots (inst),
		PQ_CURRENCY_FRACTION, currency->fraction);
	if (currency->non_utf8)
	{
		kvp_frame_set_string (qof_instance_get_slots (inst),
			PQ_CURRENCY_SYMBOL,
			qof_main_make_utf8 ((gchar *) currency->symbol));
	}
	else
	{
		kvp_frame_set_string (qof_instance_get_slots (inst),
			PQ_CURRENCY_SYMBOL, currency->symbol);
	}
	return currency;
}

/** \brief Wraps QOF around Expense_t

This allows QOF to build as an option rather than adding QofInstance to
Expense_t and requiring QOF for all builds.

*/
typedef struct
{
	QofInstance inst;
	Expense_t wrap;				// 0.12
	enum ExpenseDistance distance_unit;
	gchar *category;
	const gchar *print_string;
	PQCurrency *currency;
	gdouble temp_amount;	 /**<  The amount is set before the
		currency code is available so this acts as a buffer until both
		pieces of data are available. */
	gboolean reset_amount;	 /**< Denote whether the buffer should be used. */
} QofExpense;

static QofExpense *
expense_create (QofBook * book)
{
	Expense_t *qe;
	QofExpense *obj;
	QofCollection *coll;
	GList *all;

	obj = g_new0 (QofExpense, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_EXPENSES, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_EXPENSES);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qe = &obj->wrap;
	qe->amount = "0";
	/* 0 == AU$ so use an init value */
	qe->currency = -1;
	return obj;
}

static const gchar *
qof_exp_distanceAsString (enum ExpenseDistance distance)
{
	gchar *string;
	gint i;

	string = "Miles";			// following install-expenses convention.
	for (i = 0; ExpenseDistanceNames[i] != NULL; i++)
	{
		if (i == (gint) distance)
			string = ExpenseDistanceNames[(gint) distance];
	}
	return string;
}

static const gchar *
qof_exp_paymentAsString (enum ExpensePayment payment)
{
	gchar *string;
	gint i;

	string = "Cash";			// following install-expenses convention.
	for (i = 0; ExpensePaymentNames[i] != NULL; i++)
	{
		if (i == (gint) payment)
			string = ExpensePaymentNames[(gint) payment];
	}
	return string;
}

static const gchar *
qof_exp_typeAsString (enum ExpenseType type)
{
	gchar *string;
	gint i;

	string = "Bus";				// following install-expenses convention.
	for (i = 0; ExpenseTypeNames[i] != NULL; i++)
	{
		if (i == (gint) type)
			string = ExpenseTypeNames[(gint) type];
	}
	return string;
}

static enum ExpensePayment
qof_exp_paymentFromString (const gchar * payment_string)
{
	guint i;

	for (i = 0; ExpensePaymentNames[i] != NULL; i++)
	{
		if (0 == safe_strcmp (ExpensePaymentNames[i], payment_string))
		{
			return (enum ExpensePayment) i;
		}
	}
	return epCash;
}

static enum ExpenseType
qof_exp_typeFromString (const gchar * type_string)
{
	guint i;

	for (i = 0; ExpenseTypeNames[i] != NULL; i++)
	{
		if (0 == safe_strcmp (ExpenseTypeNames[i], type_string))
			return (enum ExpenseType) i;
	}
	return etBus;
}

static QofTime*
exp_getTime (QofExpense * e)
{
	Expense_t *qe;
	QofTime *qt;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	qt = qof_time_from_tm (&qe->date, 0);
	return qt;
}

static const gchar *
exp_getType (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qof_exp_typeAsString (qe->type);
}

static const gchar *
exp_getPayment (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qof_exp_paymentAsString (qe->payment);
}

static gint
exp_getCurrency (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, -1);
	qe = &e->wrap;
	return qe->currency;
}

static QofNumeric
exp_getAmount (QofExpense * e)
{
	Expense_t *qe;
	QofNumeric amount;
	gdouble pi_amount;
	gchar *numeric_char;

	amount = qof_numeric_zero ();
	g_return_val_if_fail (e != NULL, amount);
	qe = &e->wrap;
	if (qe->amount == 0)
		return amount;
	/* floating point as a string converts to gnc_numeric */
	pi_amount = strtod (qe->amount, NULL);
	if (e->currency)
		amount = qof_numeric_from_double (pi_amount, e->currency->fraction,
			QOF_HOW_DENOM_EXACT | QOF_HOW_RND_ROUND);
	else						/* default: use the most common fraction in the Palm currency list. */
		amount = qof_numeric_from_double (pi_amount, 100,
			QOF_HOW_DENOM_EXACT | QOF_HOW_RND_ROUND);
	numeric_char = qof_numeric_to_string (amount);
	g_free (numeric_char);
	if (qof_numeric_check (amount) == QOF_ERROR_OK)
		return amount;
	return qof_numeric_zero ();
}

static const gchar *
exp_getVendor (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->vendor;
}

static const gchar *
exp_getCity (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->city;
}

static const gchar *
exp_getAttendees (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->attendees;
}

static const gchar *
exp_getNote (QofExpense * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->note;
}

static const gchar *
exp_getDistance (QofExpense * e)
{
	guint i;
	Expense_t *qe;
	enum ExpenseDistance type;

	qe = &e->wrap;
	type = edMiles;
	if (qe->type != etMileage)
		return NULL;
	for (i = 0; ExpenseDistanceNames[i] != NULL; i++)
	{
		if (i == e->distance_unit)
			type = (enum ExpenseDistance) i;
	}
	return qof_exp_distanceAsString (type);
}

static const gchar *
exp_getCategory (QofExpense * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->category;
}

static void
exp_setTime (QofExpense * e, QofTime *h)
{
	Expense_t *qe;
	gboolean result;
	QofDate *qd;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qd = qof_date_from_qtime (h);
	result = qof_date_to_struct_tm (qd, &qe->date, 0);
	if(!result)
		PERR (" Date too large for begin.");
	qof_date_free (qd);
}

static void
exp_setType (QofExpense * e, const gchar * type_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->type = qof_exp_typeFromString (type_string);
}

static void
exp_setPayment (QofExpense * e, const gchar * payment_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->payment = qof_exp_paymentFromString (payment_string);
}

static void
exp_combine_currency_with_amount (QofExpense * e)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	if (!e->currency || qe->currency < 0)
	{
		/* the gint32 currency is always set AFTER the numeric amount. :-( */
		e->currency = pq_currency_lookup ((QofInstance *) e, 
			qe->currency);
	}
	if (!e->currency)
	{
		PERR (" Unable to identify currency fraction."
			" Using two decimal places.");
		/* Amount is stored in the Palm as a string version
		   of a floating point number. */
		qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
		return;
	}
	switch (e->currency->fraction)
	{
	case 1:
		{
			qe->amount = g_strdup_printf ("%.0f", e->temp_amount);
			break;
		}
	case 10:
		{
			qe->amount = g_strdup_printf ("%.1f", e->temp_amount);
			break;
		}
	case 100:
		{
			qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
			break;
		}
	case 1000:
		{
			qe->amount = g_strdup_printf ("%.3f", e->temp_amount);
			break;
		}
	default:
		{
			PERR (" Invalid currency fraction."
				" Using two decimal places as default.");
			qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
		}
	}
}

static void
exp_setCurrency (QofExpense * e, gint code)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->currency = code;
	e->currency = pq_currency_lookup ((QofInstance *) e, code);
	if (e->reset_amount)
		exp_combine_currency_with_amount (e);
	e->reset_amount = FALSE;
}

static void
exp_setAmount (QofExpense * e, QofNumeric h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	e->temp_amount = qof_numeric_to_double (h);
	e->reset_amount = TRUE;
	/* if an amount can ever be set without a currency_code,
	   this needs to be reviewed. */
	/** \todo FIXME: INSERT handler can set one without the
	other. Need to use the pref.default_currency? */
	if (e->currency)
		exp_combine_currency_with_amount (e);
}

static void
exp_setVendor (QofExpense * e, gchar * h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->vendor = g_strdup (qof_main_make_utf8 (h));
}

static void
exp_setCity (QofExpense * e, gchar * h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->city = g_strdup (qof_main_make_utf8 (h));
}

static void
exp_setAttendees (QofExpense * e, gchar * h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->attendees = g_strdup (qof_main_make_utf8 (h));
}

static void
exp_setNote (QofExpense * e, gchar * h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->note = g_strdup (qof_main_make_utf8 (h));
}

static void
exp_setDistance (QofExpense * e, const gchar * distance_name)
{
	gint i;
/** \todo Convert when setting into ExpensePref */
	for (i = 0; ExpenseDistanceNames[i] != NULL; i++)
	{
		if (0 == safe_strcmp (ExpenseDistanceNames[i], distance_name))
		{
			e->distance_unit = i;
		}
	}
}

static void
exp_setCategory (QofExpense * e, gchar * n)
{
	g_return_if_fail (e != NULL);
	e->category = g_strdup (qof_main_make_utf8 (n));
}

static gint
exp_unpack (QofEntity * ent, gpointer user_data)
{
	Expense_t *qe;
	QofExpense *obj;
	pi_buffer_t *pi_buf;
	PQContext *context;
	size_t len;
	gint size;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	g_return_val_if_fail (PQCurrencyTable != NULL, -1);
	obj = (QofExpense *) ent;
	qe = &obj->wrap;
	pi_buf = (pi_buffer_t *) context->pi_buf;
	len = sizeof (pi_buf->data);
	size = unpack_Expense (qe, pi_buf->data, pi_buf->allocated);
	exp_setCategory (obj, context->names[context->ent_category]);
	/* lookup currency_code in currency table. */
	obj->currency = pq_currency_lookup ((QofInstance *) ent, qe->currency);
	return size;
}

static gint
exp_pack (QofEntity * ent, gpointer user_data)
{
	PQContext *context;
	Expense_t *qe;
	QofExpense *obj;
	gint size, len;

	size = 0;
	len = PQ_DEF_BUFSZ;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context), -1);
	obj = (QofExpense *) ent;
	qe = &obj->wrap;
	/* pack_Expense still uses the old prototype
	   using len instead of pi_buf->used. */
	size = pack_Expense (qe, context->pi_buf->data, len);
	/* work around the old prototype */
	if (size > 0)
		context->pi_buf->used = size;
	return size;
}

static gint
exp_pref_unpack (QofEntity * ent, gpointer user_data)
{
	struct ExpensePref pref_e;
	PQContext *context;

	/* There is never an entity at this stage */
	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_ExpensePref (&pref_e, context->pref_buf, PQ_DEF_BUFSZ);
	context->pi_exp_pref.default_currency = pref_e.defaultCurrency;
	context->pi_exp_pref.unit_of_distance = pref_e.unitOfDistance;
	populate_currencies ();
	LEAVE (" ");
	return 0;
}

static gint
exp_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	ExpenseAppInfo_t app_e;
	PQContext *context;
	gint name_count;

	/* There is never an entity at this stage */
	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_ExpenseAppInfo (&app_e, context->app_buf->data, PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s",
			app_e.category.name[name_count]);
	}
	context->pi_cat = &app_e.category;
	LEAVE (" ");
	return 0;
}

static gint
qof_exp_free (QofEntity * ent, gpointer user_data)
{
	Expense_t *qe;
	QofExpense *obj;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	obj = (QofExpense *) ent;
	qe = &obj->wrap;
	free_Expense (qe);
	LEAVE (" ");
	return 0;
}

static const gchar *
expensePrintable (gpointer instance)
{
	QofExpense *obj;

	obj = (QofExpense *) instance;
	if (!obj)
		return NULL;
	if (exp_getType (obj))
		return g_strconcat (exp_getType (obj), " ",
			exp_getVendor (obj), " ", exp_getCity (obj), NULL);
	return NULL;
}

static QofObject expenses_object_def = {
  interface_version:QOF_OBJECT_VERSION,
  e_type:PILOT_LINK_QOF_EXPENSES,
  type_label:QOF_EXPENSES_DESC,
  create:(gpointer) expense_create,
  book_begin:NULL,
  book_end:NULL,
  is_dirty:qof_collection_is_dirty,
  mark_clean:qof_collection_mark_clean,
  foreach:qof_collection_foreach,
  printable:expensePrintable,
  version_cmp:(gint (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

static PQPack expenses_pack_def = {
  e_type:PILOT_LINK_QOF_EXPENSES,
  pack_func:exp_pack,
  unpack_func:exp_unpack,
  free_pack_func:qof_exp_free,
  palm_db_name:Expense_DB,
  app_info_unpack:exp_appinfo_unpack,
  db_pref_unpack:exp_pref_unpack,
  pref_creator:EXPENSE_CREATOR,
  pref_flag:Expense_Pref,
};

gboolean
ExpensesRegister (void)
{
	static QofParam params[] = {
		{EXP_DATE, QOF_TYPE_TIME, (QofAccessFunc) exp_getTime,
				(QofSetterFunc) exp_setTime},
		{EXP_TYPE, QOF_TYPE_STRING, (QofAccessFunc) exp_getType,
				(QofSetterFunc) exp_setType},
		{EXP_PAYMENT, QOF_TYPE_STRING, (QofAccessFunc) exp_getPayment,
				(QofSetterFunc) exp_setPayment},
		{EXP_CURRENCY, QOF_TYPE_INT32, (QofAccessFunc) exp_getCurrency,
				(QofSetterFunc) exp_setCurrency},
		{EXP_AMOUNT, QOF_TYPE_NUMERIC, (QofAccessFunc) exp_getAmount,
				(QofSetterFunc) exp_setAmount},
		{EXP_VENDOR, QOF_TYPE_STRING, (QofAccessFunc) exp_getVendor,
				(QofSetterFunc) exp_setVendor},
		{EXP_CITY, QOF_TYPE_STRING, (QofAccessFunc) exp_getCity,
				(QofSetterFunc) exp_setCity},
		{EXP_ATTENDEES, QOF_TYPE_STRING, (QofAccessFunc) exp_getAttendees,
				(QofSetterFunc) exp_setAttendees},
		{EXP_NOTE, QOF_TYPE_STRING, (QofAccessFunc) exp_getNote,
				(QofSetterFunc) exp_setNote},
		{EXP_DISTANCE, QOF_TYPE_STRING, (QofAccessFunc) exp_getDistance,
				(QofSetterFunc) exp_setDistance},
		{EXP_CATEGORY, QOF_TYPE_STRING, (QofAccessFunc) exp_getCategory,
				(QofSetterFunc) exp_setCategory},
		{EXP_KVP, QOF_TYPE_KVP, (QofAccessFunc) qof_instance_get_slots,
				NULL},
		{QOF_PARAM_BOOK, QOF_ID_BOOK,
				(QofAccessFunc) qof_instance_get_book, NULL},
		{QOF_PARAM_GUID, QOF_TYPE_GUID,
				(QofAccessFunc) qof_instance_get_guid, NULL},
		{NULL},
	};

	qof_class_register (PILOT_LINK_QOF_EXPENSES, NULL, params);

	pilot_qof_pack_register (&expenses_pack_def);

	return qof_object_register (&expenses_object_def);
}

/** @} */
/** @} */
