/***************************************************************************
 *            pilot-qof.h
 *
 *  Sun Apr 17 17:56:17 2005
 *  Copyright  2005-2006  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#ifndef _PILOT_QOF_H
#define _PILOT_QOF_H

/** @addtogroup QOF
    @{ */
/** @addtogroup pilot Using QOF as a framework around pilot-link

pilot-qof provides the executable interface to the QOF
external framework. It supports writing the QSF XML offline
storage and SQL-type queries.

The types of SQL queries that are allowed at this point are a little limited.
In general, only the following types of queries are supported: \n
SELECT * FROM SomeObj WHERE (param_a < 10.0) AND (param_b = "asdf") SORT BY param_c DESC;\n
INSERT INTO SomeObj (param_a, param_b, param_c) VALUES ("value_a", true, "0/1");

Joins are not supported directly.\n
SELECT * FROM ObjA,ObjB WHERE (ObjA.param_id = ObjB.param_other_id);\n
The problem with the above is that the search requires a nested search loop, aka a 'join', 
which is not currently supported in the underlying QofQuery code.

However, by repeating queries and adding the entities to a new session using 
::qof_entity_copy_list, a series of queries can be added to a single book. 
e.g. You can insert multiple entities and save out as a QSF XML file or use multiple 
SELECT queries to build a precise list - this can be used to replicate most of the 
functionality of a SQL join.

SELECT * from ObjA where param_id = value; SELECT * from ObjB where param_other_id = value;

Equivalent to:\n
SELECT * from ObjA,ObjB where param_id = param_other_id and param_id = value;

When combined with a foreach callback on the value of param_id for each entity in the 
QofBook, you can produce the effect of a join from running the two SELECT queries for each 
value of param_id held in 'value'.

See ::QofEntityForeachCB and ::qof_object_foreach.

SELECT a,b,c FROM ... 

Used to convert QOF objects between applications by using the returned parameter values
to create a second object. One application using QOF could register objects from two 
applications and convert data from one to the other by using\n
SELECT a,b,c FROM ObjA; SELECT d,f,k FROM ObjB; qof_object_new_instance(); 
ObjC_set_a(value_c); ObjC_set_b(value_k) etc. 

What's needed is for the SELECT to return a complete object that only contains the
parameters selected.

Unsupported: UPDATE, DELETE.

It will not be possible to support CREATE, AMEND or DROP for understandable reasons.

\todo Link with libpisync and do a fast-sync with the offline storage.

@{ */

/** @file pilot-qof.h
  @brief Executable interface to the QOF external framework
  @author Copyright (c) 2005 Neil Williams <linux@codehelp.co.uk>
*/

#include "bridge.h"    /* remove for 0.12 */
//#include "pi-buffer.h" /* included by bridge.h */
#include "qof-main.h"
#include "pilot-todo.h"
#include "qof-address.h"
#include "qof-datebook.h"
#include "qof-expenses.h"

/** the debug log file. */
#define PILOT_QOF_LOG "/tmp/pilot-qof.trace"

/** debug module identifier */
#define PQ_MOD_CLI   "pilotqof-cli"

/** The DLP Card Number reference */
#define PQ_DLP_CARD 0

/** The DLP Offset */
#define PQ_DLP_OFFSET 0

/** The DLP Record Index */
#define PQ_DLP_RECORD 0

/** The DLP Record ID - 0 for a new record */
#define PQ_DLP_NEW_REC 0

/** The DLP Record ID value, after writing - currently ignored. */
#define PQ_DLP_SET_ID 0

/** The DLP Record attributes - not currently used.

See dlpRecAttributes in pilot-link, includes/pi-dlp.h
*/
#define PQ_DLP_REC_ATTR 0

/** Default pi_buffer size */
#define PQ_DEF_BUFSZ 0xffff

#ifndef COMPATIBILITY_MODE
/** unsupported in 0.11.8 */
#define ENABLE_QUIET_MODE 	\
{ "quiet",   'q', POPT_ARG_NONE,  &plu_quiet,  0 , \
 _("Suppress HotSync connection messages"), NULL},
#define PILOT_LINK_SUPPORT "0.12"
extern int plu_connect();
#endif

/** \name Packing, unpacking and freeing wrapped objects.

 QOF is wrapped around existing pilot-link structs. These objects
 have their own methods for writing to and reading from the Palm
 databases. Objects are packed before being written to the Palm,
 and unpacked to create new objects when the database is read.

 Packing and unpacking are only used when connecting to the Palm.

 The free_ent_func routines are used whenever any QofBook created
 from this framework is closed.
@{
*/

/** \brief Free the wrapped object

 Set the pointer to the free function for the object within
 this entity. The function itself is called
 automatically by QOF when the QofBook is being closed.
*/
void pilot_entity_finaliser(QofBook *book, gpointer key, gpointer data);

/** \brief Pack this object

 Retrieve the pointer to the pack function for the
 object within this entity. Entity objects are
 packed prior to being written to a Palm database.
 Currently, this is hypothetical as support for
 merge and sync is not implemented in the pilot-qof
 application. Functionality to merge two QofBooks
 to handle the merge and sync does exist in QOF.
*/
void pilot_qof_pack(QofEntity *ent, gpointer user_data);

/** \brief Unpack this object

 Retrieve the pointer to the unpack function for this
 entity and the object it contains. Entities are unpacked
 after being read from the Palm during a HotSync. The
 unpack routine populates the object with the Palm data
 in a usable form. The QOF framework then accesses this
 data through QOF wrappers.
*/
void pilot_qof_unpack(QofEntity *ent, gpointer user_data);

/** \brief Unpack the database information.

The Palm uses a separate structure for information about the
current database, including the list of available categories.

This is a per-database structure, not per record. The records
only contain an integer - the value of the category in the
index of the CategoryAppInfo. QOF converts this into a string
value for the category name for portability.

It does not call any qof_pack routines as there is no
QofEntity at this stage.

@param e_type - determines the type of application information
to unpack from the pilot_pack wrapper.

@param user_data - gpointer to the ::pqd context.
*/
void pilot_app_unpack(QofIdTypeConst e_type, gpointer user_data);

/** \brief Packing function pointer type.

 The pack routines are built into the QOF object using the
 qof_pack function pointer and the pilot_pack wrapper. The
 same type is used for the free object memory function,
 free_pack_func.
*/
typedef int (* qof_pack)(QofEntity *ent, gpointer user_data);

/** \brief Packing function wrapper.

 The pack routines are linked to the QOF object using the
 qof_pack function pointer and the pilot_pack wrapper. The
 wrapper also includes the name of the database as held on
 the Palm.

 The e_type must be the same as the e_type of the QofObject,
 as defined in the object_def definition.
*/
typedef struct pilot_pack_s
{
	QofIdTypeConst e_type;          /**< The same e_type as the object itself. */
	qof_pack       pack_func;       /**< Calls the pack routine for the underlying
	pilot-link object to write data to the Palm. */
	qof_pack       unpack_func;     /**< Calls the unpack routine for the underlying
	pilot_link object to read data from the Palm. */
	qof_pack       free_pack_func;  /**< Calls the free routine for the underlying
	pilot-link object - used when the QofBook is closed.*/
	char           *palm_db_name;   /**< The name of the database on the Palm that
	corresponds to and will be packed / unpacked for this object. */
	qof_pack       app_info_unpack; /**< Calls the unpack routine for the current
	database to retrieve the category list. */
}pilot_pack;

/** \brief Shamelessly copied from QOF

Pilot-link requires a few functions to pack and unpack
the database records prior to  / post HotSync. These
are specific to each object and to allow new objects
to be added easily, need to be referenced generically.

qof-main implements a mini-QofObject alongside QOF that
just contains this pilot-link data.
*/
gboolean
pilot_qof_pack_register(const pilot_pack *p);

/** \brief provide for looking up an object

Returns the pilot_pack that matches this object type.
Uses the same object type as the QofObject->e_type.
*/
const pilot_pack *
pilot_qof_pack_lookup(QofIdTypeConst object_type);

/** @} */

/** \brief The pilot-qof context struct, extends qof_main_context */
typedef struct pqd {
	qof_main_context qof;      /**< Wrap the QOF context in our own. */
	int sd;                    /**< pilot connection handle. */
	int db;                    /**< pilot database handle. */
	pi_buffer_t *pi_buf;       /**< Record buffer from Palm database. */
	pi_buffer_t *app_buf;      /**< Application information buffer from Palm. */
	CategoryAppInfo_t pi_cat;  /**< category info from the Palm for this database. */
	char *port;                /**< dummy copy of popt variable for 0.11.8 */
	int quiet;                 /**< bridging variable. */ 
	int ent_category;          /**< The index of the category in pi_cat. */
	gboolean invoice_city;     /**< Shorthand to relate a contact to a timed event
								using expense_city as lookup index */
	gboolean invoice_vendor;   /**< Shorthand to relate a contact to a timed event
								using expense_vendor as lookup index */
}pilot_qof_data;

/** \brief Activate/HotSync and query the Palm, ignoring the offline storage.

Connects the the Palm and queries the live database. If -d or -e
are not used, query all supported databases in one HotSync operation. 
*/
void qof_cmd_hotsync (pilot_qof_data *context);

/** \brief Shutdown the QOF framework
*/
void pilot_qof_close(void);

/** \brief Register all QOF objects.

If new objects are added, call the register func()
here. Ensure you have the pack routines for your object.

Follow the template of other objects to create your own callbacks.

pilot_qof_init must be called by any program wanting to
use the QOF framework with pilot-link objects.

\return NULL on error, otherwise a usable pilot_qof_data* context.
*/
pilot_qof_data*  pilot_qof_init (void);

/** @} */
/** @} */

#endif /* _PILOT_QOF_H */
