#pike __REAL_VERSION__

//! This module implements various algorithms specified by
//! the Internationalizing Domain Names in Applications (IDNA) memo by
//! the Internet Engineering Task Force (IETF), see
//! @url{ftp://ftp.rfc-editor.org/in-notes/rfc3490.txt@}.

//! Punycode transcoder, see
//! @url{ftp://ftp.rfc-editor.org/in-notes/rfc3492.txt@}.
//! Punycode is used by @[to_ascii] as an "ASCII Compatible Encoding" when
//! needed.
object Punycode = class {

    inherit String.Bootstring;

    string decode(string s) {
      array(string) x = s/"-";
      x[-1] = lower_case(x[-1]);
      return ::decode(x*"-");
    }

    static void create() { ::create(36, 1, 26, 38, 700, 72, 128, '-',
				    "abcdefghijklmnopqrstuvwxyz0123456789"); }
  }();


// stringprep tables...

// Table B.2: Case folding for use with NFKC
static constant stringprep_casefold_src =
// 1 to 1
"\x0041\x0042\x0043\x0044\x0045\x0046\x0047\x0048\x0049\x004A\x004B\x004C"
"\x004D\x004E\x004F\x0050\x0051\x0052\x0053\x0054\x0055\x0056\x0057\x0058"
"\x0059\x005A\x00B5\x00C0\x00C1\x00C2\x00C3\x00C4\x00C5\x00C6\x00C7\x00C8"
"\x00C9\x00CA\x00CB\x00CC\x00CD\x00CE\x00CF\x00D0\x00D1\x00D2\x00D3\x00D4"
"\x00D5\x00D6\x00D8\x00D9\x00DA\x00DB\x00DC\x00DD\x00DE\x0100\x0102\x0104"
"\x0106\x0108\x010A\x010C\x010E\x0110\x0112\x0114\x0116\x0118\x011A\x011C"
"\x011E\x0120\x0122\x0124\x0126\x0128\x012A\x012C\x012E\x0132\x0134\x0136"
"\x0139\x013B\x013D\x013F\x0141\x0143\x0145\x0147\x014A\x014C\x014E\x0150"
"\x0152\x0154\x0156\x0158\x015A\x015C\x015E\x0160\x0162\x0164\x0166\x0168"
"\x016A\x016C\x016E\x0170\x0172\x0174\x0176\x0178\x0179\x017B\x017D\x017F"
"\x0181\x0182\x0184\x0186\x0187\x0189\x018A\x018B\x018E\x018F\x0190\x0191"
"\x0193\x0194\x0196\x0197\x0198\x019C\x019D\x019F\x01A0\x01A2\x01A4\x01A6"
"\x01A7\x01A9\x01AC\x01AE\x01AF\x01B1\x01B2\x01B3\x01B5\x01B7\x01B8\x01BC"
"\x01C4\x01C5\x01C7\x01C8\x01CA\x01CB\x01CD\x01CF\x01D1\x01D3\x01D5\x01D7"
"\x01D9\x01DB\x01DE\x01E0\x01E2\x01E4\x01E6\x01E8\x01EA\x01EC\x01EE\x01F1"
"\x01F2\x01F4\x01F6\x01F7\x01F8\x01FA\x01FC\x01FE\x0200\x0202\x0204\x0206"
"\x0208\x020A\x020C\x020E\x0210\x0212\x0214\x0216\x0218\x021A\x021C\x021E"
"\x0220\x0222\x0224\x0226\x0228\x022A\x022C\x022E\x0230\x0232\x0345\x0386"
"\x0388\x0389\x038A\x038C\x038E\x038F\x0391\x0392\x0393\x0394\x0395\x0396"
"\x0397\x0398\x0399\x039A\x039B\x039C\x039D\x039E\x039F\x03A0\x03A1\x03A3"
"\x03A4\x03A5\x03A6\x03A7\x03A8\x03A9\x03AA\x03AB\x03C2\x03D0\x03D1\x03D2"
"\x03D3\x03D4\x03D5\x03D6\x03D8\x03DA\x03DC\x03DE\x03E0\x03E2\x03E4\x03E6"
"\x03E8\x03EA\x03EC\x03EE\x03F0\x03F1\x03F2\x03F4\x03F5\x0400\x0401\x0402"
"\x0403\x0404\x0405\x0406\x0407\x0408\x0409\x040A\x040B\x040C\x040D\x040E"
"\x040F\x0410\x0411\x0412\x0413\x0414\x0415\x0416\x0417\x0418\x0419\x041A"
"\x041B\x041C\x041D\x041E\x041F\x0420\x0421\x0422\x0423\x0424\x0425\x0426"
"\x0427\x0428\x0429\x042A\x042B\x042C\x042D\x042E\x042F\x0460\x0462\x0464"
"\x0466\x0468\x046A\x046C\x046E\x0470\x0472\x0474\x0476\x0478\x047A\x047C"
"\x047E\x0480\x048A\x048C\x048E\x0490\x0492\x0494\x0496\x0498\x049A\x049C"
"\x049E\x04A0\x04A2\x04A4\x04A6\x04A8\x04AA\x04AC\x04AE\x04B0\x04B2\x04B4"
"\x04B6\x04B8\x04BA\x04BC\x04BE\x04C1\x04C3\x04C5\x04C7\x04C9\x04CB\x04CD"
"\x04D0\x04D2\x04D4\x04D6\x04D8\x04DA\x04DC\x04DE\x04E0\x04E2\x04E4\x04E6"
"\x04E8\x04EA\x04EC\x04EE\x04F0\x04F2\x04F4\x04F8\x0500\x0502\x0504\x0506"
"\x0508\x050A\x050C\x050E\x0531\x0532\x0533\x0534\x0535\x0536\x0537\x0538"
"\x0539\x053A\x053B\x053C\x053D\x053E\x053F\x0540\x0541\x0542\x0543\x0544"
"\x0545\x0546\x0547\x0548\x0549\x054A\x054B\x054C\x054D\x054E\x054F\x0550"
"\x0551\x0552\x0553\x0554\x0555\x0556\x1E00\x1E02\x1E04\x1E06\x1E08\x1E0A"
"\x1E0C\x1E0E\x1E10\x1E12\x1E14\x1E16\x1E18\x1E1A\x1E1C\x1E1E\x1E20\x1E22"
"\x1E24\x1E26\x1E28\x1E2A\x1E2C\x1E2E\x1E30\x1E32\x1E34\x1E36\x1E38\x1E3A"
"\x1E3C\x1E3E\x1E40\x1E42\x1E44\x1E46\x1E48\x1E4A\x1E4C\x1E4E\x1E50\x1E52"
"\x1E54\x1E56\x1E58\x1E5A\x1E5C\x1E5E\x1E60\x1E62\x1E64\x1E66\x1E68\x1E6A"
"\x1E6C\x1E6E\x1E70\x1E72\x1E74\x1E76\x1E78\x1E7A\x1E7C\x1E7E\x1E80\x1E82"
"\x1E84\x1E86\x1E88\x1E8A\x1E8C\x1E8E\x1E90\x1E92\x1E94\x1E9B\x1EA0\x1EA2"
"\x1EA4\x1EA6\x1EA8\x1EAA\x1EAC\x1EAE\x1EB0\x1EB2\x1EB4\x1EB6\x1EB8\x1EBA"
"\x1EBC\x1EBE\x1EC0\x1EC2\x1EC4\x1EC6\x1EC8\x1ECA\x1ECC\x1ECE\x1ED0\x1ED2"
"\x1ED4\x1ED6\x1ED8\x1EDA\x1EDC\x1EDE\x1EE0\x1EE2\x1EE4\x1EE6\x1EE8\x1EEA"
"\x1EEC\x1EEE\x1EF0\x1EF2\x1EF4\x1EF6\x1EF8\x1F08\x1F09\x1F0A\x1F0B\x1F0C"
"\x1F0D\x1F0E\x1F0F\x1F18\x1F19\x1F1A\x1F1B\x1F1C\x1F1D\x1F28\x1F29\x1F2A"
"\x1F2B\x1F2C\x1F2D\x1F2E\x1F2F\x1F38\x1F39\x1F3A\x1F3B\x1F3C\x1F3D\x1F3E"
"\x1F3F\x1F48\x1F49\x1F4A\x1F4B\x1F4C\x1F4D\x1F59\x1F5B\x1F5D\x1F5F\x1F68"
"\x1F69\x1F6A\x1F6B\x1F6C\x1F6D\x1F6E\x1F6F\x1FB8\x1FB9\x1FBA\x1FBB\x1FBE"
"\x1FC8\x1FC9\x1FCA\x1FCB\x1FD8\x1FD9\x1FDA\x1FDB\x1FE8\x1FE9\x1FEA\x1FEB"
"\x1FEC\x1FF8\x1FF9\x1FFA\x1FFB\x2102\x2107\x210B\x210C\x210D\x2110\x2111"
"\x2112\x2115\x2119\x211A\x211B\x211C\x211D\x2124\x2126\x2128\x212A\x212B"
"\x212C\x212D\x2130\x2131\x2133\x213E\x213F\x2145\x2160\x2161\x2162\x2163"
"\x2164\x2165\x2166\x2167\x2168\x2169\x216A\x216B\x216C\x216D\x216E\x216F"
"\x24B6\x24B7\x24B8\x24B9\x24BA\x24BB\x24BC\x24BD\x24BE\x24BF\x24C0\x24C1"
"\x24C2\x24C3\x24C4\x24C5\x24C6\x24C7\x24C8\x24C9\x24CA\x24CB\x24CC\x24CD"
"\x24CE\x24CF\xFF21\xFF22\xFF23\xFF24\xFF25\xFF26\xFF27\xFF28\xFF29\xFF2A"
"\xFF2B\xFF2C\xFF2D\xFF2E\xFF2F\xFF30\xFF31\xFF32\xFF33\xFF34\xFF35\xFF36"
"\xFF37\xFF38\xFF39\xFF3A\x10400\x10401\x10402\x10403\x10404\x10405\x10406"
"\x10407\x10408\x10409\x1040A\x1040B\x1040C\x1040D\x1040E\x1040F\x10410"
"\x10411\x10412\x10413\x10414\x10415\x10416\x10417\x10418\x10419\x1041A"
"\x1041B\x1041C\x1041D\x1041E\x1041F\x10420\x10421\x10422\x10423\x10424"
"\x10425\x1D400\x1D401\x1D402\x1D403\x1D404\x1D405\x1D406\x1D407\x1D408"
"\x1D409\x1D40A\x1D40B\x1D40C\x1D40D\x1D40E\x1D40F\x1D410\x1D411\x1D412"
"\x1D413\x1D414\x1D415\x1D416\x1D417\x1D418\x1D419\x1D434\x1D435\x1D436"
"\x1D437\x1D438\x1D439\x1D43A\x1D43B\x1D43C\x1D43D\x1D43E\x1D43F\x1D440"
"\x1D441\x1D442\x1D443\x1D444\x1D445\x1D446\x1D447\x1D448\x1D449\x1D44A"
"\x1D44B\x1D44C\x1D44D\x1D468\x1D469\x1D46A\x1D46B\x1D46C\x1D46D\x1D46E"
"\x1D46F\x1D470\x1D471\x1D472\x1D473\x1D474\x1D475\x1D476\x1D477\x1D478"
"\x1D479\x1D47A\x1D47B\x1D47C\x1D47D\x1D47E\x1D47F\x1D480\x1D481\x1D49C"
"\x1D49E\x1D49F\x1D4A2\x1D4A5\x1D4A6\x1D4A9\x1D4AA\x1D4AB\x1D4AC\x1D4AE"
"\x1D4AF\x1D4B0\x1D4B1\x1D4B2\x1D4B3\x1D4B4\x1D4B5\x1D4D0\x1D4D1\x1D4D2"
"\x1D4D3\x1D4D4\x1D4D5\x1D4D6\x1D4D7\x1D4D8\x1D4D9\x1D4DA\x1D4DB\x1D4DC"
"\x1D4DD\x1D4DE\x1D4DF\x1D4E0\x1D4E1\x1D4E2\x1D4E3\x1D4E4\x1D4E5\x1D4E6"
"\x1D4E7\x1D4E8\x1D4E9\x1D504\x1D505\x1D507\x1D508\x1D509\x1D50A\x1D50D"
"\x1D50E\x1D50F\x1D510\x1D511\x1D512\x1D513\x1D514\x1D516\x1D517\x1D518"
"\x1D519\x1D51A\x1D51B\x1D51C\x1D538\x1D539\x1D53B\x1D53C\x1D53D\x1D53E"
"\x1D540\x1D541\x1D542\x1D543\x1D544\x1D546\x1D54A\x1D54B\x1D54C\x1D54D"
"\x1D54E\x1D54F\x1D550\x1D56C\x1D56D\x1D56E\x1D56F\x1D570\x1D571\x1D572"
"\x1D573\x1D574\x1D575\x1D576\x1D577\x1D578\x1D579\x1D57A\x1D57B\x1D57C"
"\x1D57D\x1D57E\x1D57F\x1D580\x1D581\x1D582\x1D583\x1D584\x1D585\x1D5A0"
"\x1D5A1\x1D5A2\x1D5A3\x1D5A4\x1D5A5\x1D5A6\x1D5A7\x1D5A8\x1D5A9\x1D5AA"
"\x1D5AB\x1D5AC\x1D5AD\x1D5AE\x1D5AF\x1D5B0\x1D5B1\x1D5B2\x1D5B3\x1D5B4"
"\x1D5B5\x1D5B6\x1D5B7\x1D5B8\x1D5B9\x1D5D4\x1D5D5\x1D5D6\x1D5D7\x1D5D8"
"\x1D5D9\x1D5DA\x1D5DB\x1D5DC\x1D5DD\x1D5DE\x1D5DF\x1D5E0\x1D5E1\x1D5E2"
"\x1D5E3\x1D5E4\x1D5E5\x1D5E6\x1D5E7\x1D5E8\x1D5E9\x1D5EA\x1D5EB\x1D5EC"
"\x1D5ED\x1D608\x1D609\x1D60A\x1D60B\x1D60C\x1D60D\x1D60E\x1D60F\x1D610"
"\x1D611\x1D612\x1D613\x1D614\x1D615\x1D616\x1D617\x1D618\x1D619\x1D61A"
"\x1D61B\x1D61C\x1D61D\x1D61E\x1D61F\x1D620\x1D621\x1D63C\x1D63D\x1D63E"
"\x1D63F\x1D640\x1D641\x1D642\x1D643\x1D644\x1D645\x1D646\x1D647\x1D648"
"\x1D649\x1D64A\x1D64B\x1D64C\x1D64D\x1D64E\x1D64F\x1D650\x1D651\x1D652"
"\x1D653\x1D654\x1D655\x1D670\x1D671\x1D672\x1D673\x1D674\x1D675\x1D676"
"\x1D677\x1D678\x1D679\x1D67A\x1D67B\x1D67C\x1D67D\x1D67E\x1D67F\x1D680"
"\x1D681\x1D682\x1D683\x1D684\x1D685\x1D686\x1D687\x1D688\x1D689\x1D6A8"
"\x1D6A9\x1D6AA\x1D6AB\x1D6AC\x1D6AD\x1D6AE\x1D6AF\x1D6B0\x1D6B1\x1D6B2"
"\x1D6B3\x1D6B4\x1D6B5\x1D6B6\x1D6B7\x1D6B8\x1D6B9\x1D6BA\x1D6BB\x1D6BC"
"\x1D6BD\x1D6BE\x1D6BF\x1D6C0\x1D6D3\x1D6E2\x1D6E3\x1D6E4\x1D6E5\x1D6E6"
"\x1D6E7\x1D6E8\x1D6E9\x1D6EA\x1D6EB\x1D6EC\x1D6ED\x1D6EE\x1D6EF\x1D6F0"
"\x1D6F1\x1D6F2\x1D6F3\x1D6F4\x1D6F5\x1D6F6\x1D6F7\x1D6F8\x1D6F9\x1D6FA"
"\x1D70D\x1D71C\x1D71D\x1D71E\x1D71F\x1D720\x1D721\x1D722\x1D723\x1D724"
"\x1D725\x1D726\x1D727\x1D728\x1D729\x1D72A\x1D72B\x1D72C\x1D72D\x1D72E"
"\x1D72F\x1D730\x1D731\x1D732\x1D733\x1D734\x1D747\x1D756\x1D757\x1D758"
"\x1D759\x1D75A\x1D75B\x1D75C\x1D75D\x1D75E\x1D75F\x1D760\x1D761\x1D762"
"\x1D763\x1D764\x1D765\x1D766\x1D767\x1D768\x1D769\x1D76A\x1D76B\x1D76C"
"\x1D76D\x1D76E\x1D781\x1D790\x1D791\x1D792\x1D793\x1D794\x1D795\x1D796"
"\x1D797\x1D798\x1D799\x1D79A\x1D79B\x1D79C\x1D79D\x1D79E\x1D79F\x1D7A0"
"\x1D7A1\x1D7A2\x1D7A3\x1D7A4\x1D7A5\x1D7A6\x1D7A7\x1D7A8\x1D7BB"
// 1 to 2
"\x00DF\x0130\x0149\x01F0\x037A\x0587\x1E96\x1E97\x1E98\x1E99\x1E9A\x1F50"
"\x1F80\x1F81\x1F82\x1F83\x1F84\x1F85\x1F86\x1F87\x1F88\x1F89\x1F8A\x1F8B"
"\x1F8C\x1F8D\x1F8E\x1F8F\x1F90\x1F91\x1F92\x1F93\x1F94\x1F95\x1F96\x1F97"
"\x1F98\x1F99\x1F9A\x1F9B\x1F9C\x1F9D\x1F9E\x1F9F\x1FA0\x1FA1\x1FA2\x1FA3"
"\x1FA4\x1FA5\x1FA6\x1FA7\x1FA8\x1FA9\x1FAA\x1FAB\x1FAC\x1FAD\x1FAE\x1FAF"
"\x1FB2\x1FB3\x1FB4\x1FB6\x1FBC\x1FC2\x1FC3\x1FC4\x1FC6\x1FCC\x1FD6\x1FE4"
"\x1FE6\x1FF2\x1FF3\x1FF4\x1FF6\x1FFC\x20A8\x2103\x2109\x2116\x2120\x2122"
"\x3373\x3375\x3380\x3381\x3382\x3383\x3384\x3385\x3386\x3387\x338A\x338B"
"\x338C\x3390\x33A9\x33B4\x33B5\x33B6\x33B7\x33B8\x33B9\x33BA\x33BB\x33BC"
"\x33BD\x33BE\x33BF\x33C0\x33C1\x33C3\x33C8\x33C9\x33CB\x33CD\x33CE\x33D7"
"\x33DA\x33DC\x33DD\xFB00\xFB01\xFB02\xFB05\xFB06\xFB13\xFB14\xFB15\xFB16"
"\xFB17"
// 1 to 3
"\x0390\x03B0\x1F52\x1F54\x1F56\x1FB7\x1FC7\x1FD2\x1FD3\x1FD7\x1FE2\x1FE3"
"\x1FE7\x1FF7\x2121\x3371\x3391\x3392\x3393\x3394\x33AA\x33AB\x33AC\x33C7"
"\x33D9\xFB03\xFB04"
// 1 to 4
"\x33C6"/1;

static constant stringprep_casefold_dest =
// 1 to 1
"\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B\x006C"
"\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077\x0078"
"\x0079\x007A\x03BC\x00E0\x00E1\x00E2\x00E3\x00E4\x00E5\x00E6\x00E7\x00E8"
"\x00E9\x00EA\x00EB\x00EC\x00ED\x00EE\x00EF\x00F0\x00F1\x00F2\x00F3\x00F4"
"\x00F5\x00F6\x00F8\x00F9\x00FA\x00FB\x00FC\x00FD\x00FE\x0101\x0103\x0105"
"\x0107\x0109\x010B\x010D\x010F\x0111\x0113\x0115\x0117\x0119\x011B\x011D"
"\x011F\x0121\x0123\x0125\x0127\x0129\x012B\x012D\x012F\x0133\x0135\x0137"
"\x013A\x013C\x013E\x0140\x0142\x0144\x0146\x0148\x014B\x014D\x014F\x0151"
"\x0153\x0155\x0157\x0159\x015B\x015D\x015F\x0161\x0163\x0165\x0167\x0169"
"\x016B\x016D\x016F\x0171\x0173\x0175\x0177\x00FF\x017A\x017C\x017E\x0073"
"\x0253\x0183\x0185\x0254\x0188\x0256\x0257\x018C\x01DD\x0259\x025B\x0192"
"\x0260\x0263\x0269\x0268\x0199\x026F\x0272\x0275\x01A1\x01A3\x01A5\x0280"
"\x01A8\x0283\x01AD\x0288\x01B0\x028A\x028B\x01B4\x01B6\x0292\x01B9\x01BD"
"\x01C6\x01C6\x01C9\x01C9\x01CC\x01CC\x01CE\x01D0\x01D2\x01D4\x01D6\x01D8"
"\x01DA\x01DC\x01DF\x01E1\x01E3\x01E5\x01E7\x01E9\x01EB\x01ED\x01EF\x01F3"
"\x01F3\x01F5\x0195\x01BF\x01F9\x01FB\x01FD\x01FF\x0201\x0203\x0205\x0207"
"\x0209\x020B\x020D\x020F\x0211\x0213\x0215\x0217\x0219\x021B\x021D\x021F"
"\x019E\x0223\x0225\x0227\x0229\x022B\x022D\x022F\x0231\x0233\x03B9\x03AC"
"\x03AD\x03AE\x03AF\x03CC\x03CD\x03CE\x03B1\x03B2\x03B3\x03B4\x03B5\x03B6"
"\x03B7\x03B8\x03B9\x03BA\x03BB\x03BC\x03BD\x03BE\x03BF\x03C0\x03C1\x03C3"
"\x03C4\x03C5\x03C6\x03C7\x03C8\x03C9\x03CA\x03CB\x03C3\x03B2\x03B8\x03C5"
"\x03CD\x03CB\x03C6\x03C0\x03D9\x03DB\x03DD\x03DF\x03E1\x03E3\x03E5\x03E7"
"\x03E9\x03EB\x03ED\x03EF\x03BA\x03C1\x03C3\x03B8\x03B5\x0450\x0451\x0452"
"\x0453\x0454\x0455\x0456\x0457\x0458\x0459\x045A\x045B\x045C\x045D\x045E"
"\x045F\x0430\x0431\x0432\x0433\x0434\x0435\x0436\x0437\x0438\x0439\x043A"
"\x043B\x043C\x043D\x043E\x043F\x0440\x0441\x0442\x0443\x0444\x0445\x0446"
"\x0447\x0448\x0449\x044A\x044B\x044C\x044D\x044E\x044F\x0461\x0463\x0465"
"\x0467\x0469\x046B\x046D\x046F\x0471\x0473\x0475\x0477\x0479\x047B\x047D"
"\x047F\x0481\x048B\x048D\x048F\x0491\x0493\x0495\x0497\x0499\x049B\x049D"
"\x049F\x04A1\x04A3\x04A5\x04A7\x04A9\x04AB\x04AD\x04AF\x04B1\x04B3\x04B5"
"\x04B7\x04B9\x04BB\x04BD\x04BF\x04C2\x04C4\x04C6\x04C8\x04CA\x04CC\x04CE"
"\x04D1\x04D3\x04D5\x04D7\x04D9\x04DB\x04DD\x04DF\x04E1\x04E3\x04E5\x04E7"
"\x04E9\x04EB\x04ED\x04EF\x04F1\x04F3\x04F5\x04F9\x0501\x0503\x0505\x0507"
"\x0509\x050B\x050D\x050F\x0561\x0562\x0563\x0564\x0565\x0566\x0567\x0568"
"\x0569\x056A\x056B\x056C\x056D\x056E\x056F\x0570\x0571\x0572\x0573\x0574"
"\x0575\x0576\x0577\x0578\x0579\x057A\x057B\x057C\x057D\x057E\x057F\x0580"
"\x0581\x0582\x0583\x0584\x0585\x0586\x1E01\x1E03\x1E05\x1E07\x1E09\x1E0B"
"\x1E0D\x1E0F\x1E11\x1E13\x1E15\x1E17\x1E19\x1E1B\x1E1D\x1E1F\x1E21\x1E23"
"\x1E25\x1E27\x1E29\x1E2B\x1E2D\x1E2F\x1E31\x1E33\x1E35\x1E37\x1E39\x1E3B"
"\x1E3D\x1E3F\x1E41\x1E43\x1E45\x1E47\x1E49\x1E4B\x1E4D\x1E4F\x1E51\x1E53"
"\x1E55\x1E57\x1E59\x1E5B\x1E5D\x1E5F\x1E61\x1E63\x1E65\x1E67\x1E69\x1E6B"
"\x1E6D\x1E6F\x1E71\x1E73\x1E75\x1E77\x1E79\x1E7B\x1E7D\x1E7F\x1E81\x1E83"
"\x1E85\x1E87\x1E89\x1E8B\x1E8D\x1E8F\x1E91\x1E93\x1E95\x1E61\x1EA1\x1EA3"
"\x1EA5\x1EA7\x1EA9\x1EAB\x1EAD\x1EAF\x1EB1\x1EB3\x1EB5\x1EB7\x1EB9\x1EBB"
"\x1EBD\x1EBF\x1EC1\x1EC3\x1EC5\x1EC7\x1EC9\x1ECB\x1ECD\x1ECF\x1ED1\x1ED3"
"\x1ED5\x1ED7\x1ED9\x1EDB\x1EDD\x1EDF\x1EE1\x1EE3\x1EE5\x1EE7\x1EE9\x1EEB"
"\x1EED\x1EEF\x1EF1\x1EF3\x1EF5\x1EF7\x1EF9\x1F00\x1F01\x1F02\x1F03\x1F04"
"\x1F05\x1F06\x1F07\x1F10\x1F11\x1F12\x1F13\x1F14\x1F15\x1F20\x1F21\x1F22"
"\x1F23\x1F24\x1F25\x1F26\x1F27\x1F30\x1F31\x1F32\x1F33\x1F34\x1F35\x1F36"
"\x1F37\x1F40\x1F41\x1F42\x1F43\x1F44\x1F45\x1F51\x1F53\x1F55\x1F57\x1F60"
"\x1F61\x1F62\x1F63\x1F64\x1F65\x1F66\x1F67\x1FB0\x1FB1\x1F70\x1F71\x03B9"
"\x1F72\x1F73\x1F74\x1F75\x1FD0\x1FD1\x1F76\x1F77\x1FE0\x1FE1\x1F7A\x1F7B"
"\x1FE5\x1F78\x1F79\x1F7C\x1F7D\x0063\x025B\x0068\x0068\x0068\x0069\x0069"
"\x006C\x006E\x0070\x0071\x0072\x0072\x0072\x007A\x03C9\x007A\x006B\x00E5"
"\x0062\x0063\x0065\x0066\x006D\x03B3\x03C0\x0064\x2170\x2171\x2172\x2173"
"\x2174\x2175\x2176\x2177\x2178\x2179\x217A\x217B\x217C\x217D\x217E\x217F"
"\x24D0\x24D1\x24D2\x24D3\x24D4\x24D5\x24D6\x24D7\x24D8\x24D9\x24DA\x24DB"
"\x24DC\x24DD\x24DE\x24DF\x24E0\x24E1\x24E2\x24E3\x24E4\x24E5\x24E6\x24E7"
"\x24E8\x24E9\xFF41\xFF42\xFF43\xFF44\xFF45\xFF46\xFF47\xFF48\xFF49\xFF4A"
"\xFF4B\xFF4C\xFF4D\xFF4E\xFF4F\xFF50\xFF51\xFF52\xFF53\xFF54\xFF55\xFF56"
"\xFF57\xFF58\xFF59\xFF5A\x10428\x10429\x1042A\x1042B\x1042C\x1042D\x1042E"
"\x1042F\x10430\x10431\x10432\x10433\x10434\x10435\x10436\x10437\x10438"
"\x10439\x1043A\x1043B\x1043C\x1043D\x1043E\x1043F\x10440\x10441\x10442"
"\x10443\x10444\x10445\x10446\x10447\x10448\x10449\x1044A\x1044B\x1044C"
"\x1044D\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B"
"\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077"
"\x0078\x0079\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069"
"\x006A\x006B\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075"
"\x0076\x0077\x0078\x0079\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067"
"\x0068\x0069\x006A\x006B\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073"
"\x0074\x0075\x0076\x0077\x0078\x0079\x007A\x0061\x0063\x0064\x0067\x006A"
"\x006B\x006E\x006F\x0070\x0071\x0073\x0074\x0075\x0076\x0077\x0078\x0079"
"\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B"
"\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077"
"\x0078\x0079\x007A\x0061\x0062\x0064\x0065\x0066\x0067\x006A\x006B\x006C"
"\x006D\x006E\x006F\x0070\x0071\x0073\x0074\x0075\x0076\x0077\x0078\x0079"
"\x0061\x0062\x0064\x0065\x0066\x0067\x0069\x006A\x006B\x006C\x006D\x006F"
"\x0073\x0074\x0075\x0076\x0077\x0078\x0079\x0061\x0062\x0063\x0064\x0065"
"\x0066\x0067\x0068\x0069\x006A\x006B\x006C\x006D\x006E\x006F\x0070\x0071"
"\x0072\x0073\x0074\x0075\x0076\x0077\x0078\x0079\x007A\x0061\x0062\x0063"
"\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B\x006C\x006D\x006E\x006F"
"\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077\x0078\x0079\x007A\x0061"
"\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B\x006C\x006D"
"\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077\x0078\x0079"
"\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x006A\x006B"
"\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077"
"\x0078\x0079\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069"
"\x006A\x006B\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073\x0074\x0075"
"\x0076\x0077\x0078\x0079\x007A\x0061\x0062\x0063\x0064\x0065\x0066\x0067"
"\x0068\x0069\x006A\x006B\x006C\x006D\x006E\x006F\x0070\x0071\x0072\x0073"
"\x0074\x0075\x0076\x0077\x0078\x0079\x007A\x03B1\x03B2\x03B3\x03B4\x03B5"
"\x03B6\x03B7\x03B8\x03B9\x03BA\x03BB\x03BC\x03BD\x03BE\x03BF\x03C0\x03C1"
"\x03B8\x03C3\x03C4\x03C5\x03C6\x03C7\x03C8\x03C9\x03C3\x03B1\x03B2\x03B3"
"\x03B4\x03B5\x03B6\x03B7\x03B8\x03B9\x03BA\x03BB\x03BC\x03BD\x03BE\x03BF"
"\x03C0\x03C1\x03B8\x03C3\x03C4\x03C5\x03C6\x03C7\x03C8\x03C9\x03C3\x03B1"
"\x03B2\x03B3\x03B4\x03B5\x03B6\x03B7\x03B8\x03B9\x03BA\x03BB\x03BC\x03BD"
"\x03BE\x03BF\x03C0\x03C1\x03B8\x03C3\x03C4\x03C5\x03C6\x03C7\x03C8\x03C9"
"\x03C3\x03B1\x03B2\x03B3\x03B4\x03B5\x03B6\x03B7\x03B8\x03B9\x03BA\x03BB"
"\x03BC\x03BD\x03BE\x03BF\x03C0\x03C1\x03B8\x03C3\x03C4\x03C5\x03C6\x03C7"
"\x03C8\x03C9\x03C3\x03B1\x03B2\x03B3\x03B4\x03B5\x03B6\x03B7\x03B8\x03B9"
"\x03BA\x03BB\x03BC\x03BD\x03BE\x03BF\x03C0\x03C1\x03B8\x03C3\x03C4\x03C5"
"\x03C6\x03C7\x03C8\x03C9\x03C3"/1+
// 1 to 2
"\x0073\x0073\x0069\x0307\x02BC\x006E\x006A\x030C\x0020\x03B9\x0565\x0582"
"\x0068\x0331\x0074\x0308\x0077\x030A\x0079\x030A\x0061\x02BE\x03C5\x0313"
"\x1F00\x03B9\x1F01\x03B9\x1F02\x03B9\x1F03\x03B9\x1F04\x03B9\x1F05\x03B9"
"\x1F06\x03B9\x1F07\x03B9\x1F00\x03B9\x1F01\x03B9\x1F02\x03B9\x1F03\x03B9"
"\x1F04\x03B9\x1F05\x03B9\x1F06\x03B9\x1F07\x03B9\x1F20\x03B9\x1F21\x03B9"
"\x1F22\x03B9\x1F23\x03B9\x1F24\x03B9\x1F25\x03B9\x1F26\x03B9\x1F27\x03B9"
"\x1F20\x03B9\x1F21\x03B9\x1F22\x03B9\x1F23\x03B9\x1F24\x03B9\x1F25\x03B9"
"\x1F26\x03B9\x1F27\x03B9\x1F60\x03B9\x1F61\x03B9\x1F62\x03B9\x1F63\x03B9"
"\x1F64\x03B9\x1F65\x03B9\x1F66\x03B9\x1F67\x03B9\x1F60\x03B9\x1F61\x03B9"
"\x1F62\x03B9\x1F63\x03B9\x1F64\x03B9\x1F65\x03B9\x1F66\x03B9\x1F67\x03B9"
"\x1F70\x03B9\x03B1\x03B9\x03AC\x03B9\x03B1\x0342\x03B1\x03B9\x1F74\x03B9"
"\x03B7\x03B9\x03AE\x03B9\x03B7\x0342\x03B7\x03B9\x03B9\x0342\x03C1\x0313"
"\x03C5\x0342\x1F7C\x03B9\x03C9\x03B9\x03CE\x03B9\x03C9\x0342\x03C9\x03B9"
"\x0072\x0073\x00B0\x0063\x00B0\x0066\x006E\x006F\x0073\x006D\x0074\x006D"
"\x0061\x0075\x006F\x0076\x0070\x0061\x006E\x0061\x03BC\x0061\x006D\x0061"
"\x006B\x0061\x006B\x0062\x006D\x0062\x0067\x0062\x0070\x0066\x006E\x0066"
"\x03BC\x0066\x0068\x007A\x0070\x0061\x0070\x0076\x006E\x0076\x03BC\x0076"
"\x006D\x0076\x006B\x0076\x006D\x0076\x0070\x0077\x006E\x0077\x03BC\x0077"
"\x006D\x0077\x006B\x0077\x006D\x0077\x006B\x03C9\x006D\x03C9\x0062\x0071"
"\x0064\x0062\x0067\x0079\x0068\x0070\x006B\x006B\x006B\x006D\x0070\x0068"
"\x0070\x0072\x0073\x0076\x0077\x0062\x0066\x0066\x0066\x0069\x0066\x006C"
"\x0073\x0074\x0073\x0074\x0574\x0576\x0574\x0565\x0574\x056B\x057E\x0576"
"\x0574\x056D"/2+
// 1 to 3
"\x03B9\x0308\x0301\x03C5\x0308\x0301\x03C5\x0313\x0300\x03C5\x0313\x0301"
"\x03C5\x0313\x0342\x03B1\x0342\x03B9\x03B7\x0342\x03B9\x03B9\x0308\x0300"
"\x03B9\x0308\x0301\x03B9\x0308\x0342\x03C5\x0308\x0300\x03C5\x0308\x0301"
"\x03C5\x0308\x0342\x03C9\x0342\x03B9\x0074\x0065\x006C\x0068\x0070\x0061"
"\x006B\x0068\x007A\x006D\x0068\x007A\x0067\x0068\x007A\x0074\x0068\x007A"
"\x006B\x0070\x0061\x006D\x0070\x0061\x0067\x0070\x0061\x0063\x006F\x002E"
"\x0070\x0070\x006D\x0066\x0066\x0069\x0066\x0066\x006C"/3+
// 1 to 4
"\x0063\x2215\x006B\x0067"/4;



//! Prepare a Unicode string for ACE transcoding.  Used by @[to_ascii].
//! Nameprep is a profile of Stringprep, which is described in RFC 3454.
//!
//! @param s
//!   The string to prep.
//! @param allow_unassigned
//!   Set this flag the the string to transform is a "query string",
//!   and not a "stored string".  See RFC 3454.
//!
string nameprep(string s, int(0..1)|void allow_unassigned)
{
  // Mapping:  Table B.1 - Map to nothing
  s = replace(s,
	      "\x00AD\x034F\x1806\x180B\x180C\x180D\x200B\x200C\x200D"
	      "\x2060\xFE00\xFE01\xFE02\xFE03\xFE04\xFE05\xFE06\xFE07"
	      "\xFE08\xFE09\xFE0A\xFE0B\xFE0C\xFE0D\xFE0E\xFE0F\xFEFF"/"",
	      ({""})*27);

  // Mapping:  Table B.2 - Case folding for use with NFKC
  s = replace(s, stringprep_casefold_src, stringprep_casefold_dest);
  
  // Normalize
  s = Unicode.normalize(s, "NFKC");

  // Prohobit
  if(array_sscanf(s, "%*[^\x80-\x9f\xa0\x340\x341\x6dd\x70f\x180e\x1680"
		  "\x2000-\x200f\x2028-\x202f\x205f-\x2063"
		  "\x206a-\x206f\x2ff0-\x2ffb\x3000\xd800-\xf8ff"
		  "\xfdd0-\xfdef\xfeff\xfff9-\xffff\x1fffe\x1ffff"
		  "\x2fffe\x2ffff\x3fffe\x3ffff\x4fffe\x4ffff"
		  "\x5fffe\x5ffff\x6fffe\x6ffff\x7fffe\x7ffff"
		  "\x8fffe\x8ffff\x9fffe\x9ffff\xafffe\xaffff"
		  "\xbfffe\xbffff\xcfffe\xcffff\xdfffe\xdffff"
		  "\xe0001\xe0020-\xe007f\xefffe-\x10fffd]%n")[0]
     != sizeof(s))
    error("Nameprep failed: Prohibited character encountered\n");

  // Bidirectional check
  // FIXME!
#if 0
  if(array_sscanf(s, "%*[^\x5be\x5c0\x5c3\x5d0-\x5ea\x5f0-\x5f4"
		  "\x61b\x61f\x621-\x63a\x640-\x64a\x66d-\x66f"
		  "\x671-\x6d5\x6dd\x6e5-\x6e6\x6fa-\x6fe\x700-\x70d"
		  "\x710\x712-\x72c\x780-\x7ac\x7b1\x200f\xfb1d"
		  "\xfb1f-\xfb28\xfb2a-\xfb36\xfb38-\xfb3c\xfb3e"
		  "\xfb40\xfb41\xfb43\xfb44\xfb46-\xfbb1\xfbd3-\xfd3d"
		  "\xfd50-\xfd8f\xfd92-\xfdc7\xfdf0-\xfdfc\xfe70-\fe74"
		  "\xfe76-\xfefc]%n")[0] != sizeof(s)) {
    // s contains RandALCat characters
    // Check that it does not contain any LCat characters
    // and that it both starts and ends with an RandALCat character.
  }
#endif

  // Unassigned check
  if(!allow_unassigned) {
    // FIXME!  Check that no characters which are unassigned in
    // Unicode 3.2 are in s
  }

  return s;
}

//! The to_ascii operation takes a sequence of Unicode code points that
//! make up one label and transforms it into a sequence of code points in
//! the ASCII range (0..7F).  If to_ascci succeeds, the original sequence
//! and the resulting sequence are equivalent labels.
//!
//! @param s
//!   The sequence of Unicode code points to transform.
//! @param allow_unassigned
//!   Set this flag if the the string to transform is a "query string",
//!   and not a "stored string".  See RFC 3454.
//! @param use_std3_ascii_rules
//!   Set this flag to enforce the restrictions on ASCII characters in
//!   host names imposed by STD3.
//!
string to_ascii(string s, int(0..1)|void allow_unassigned,
		int(0..1)|void use_std3_ascii_rules)
{
  int is_ascii = max(@values(s)) < 128;
  if(!is_ascii) {
    s = nameprep(s, allow_unassigned);
    is_ascii = max(@values(s)) < 128;
  }
  if(use_std3_ascii_rules) {
    if(array_sscanf(s, "%*[^\0-,./:-@[-`{-\x7f]%n")[0] != sizeof(s))
      error("Label is not valid accoring to STD3: non-LDH codepoint\n");
    if(sizeof(s) && (s[0] == '-' || s[-1] == '-'))
      error("Label is not valid accoring to STD3: leading or trailing hyphen-minus\n");
  }
  if(!is_ascii) {
    if(lower_case(s[..3]) == "xn--")
      error("Label to encode begins with ACE prefix!\n");
    s = "xn--"+Punycode->encode(s);
  }
  if(sizeof(s)<1 || sizeof(s)>63)
    error("Label is not within the 1-63 character length range\n");
  return s;
}


//! The to_unicode operation takes a sequence of Unicode code points that
//! make up one label and returns a sequence of Unicode code points.  If
//! the input sequence is a label in ACE form, then the result is an
//! equivalent internationalized label that is not in ACE form, otherwise
//! the original sequence is returned unaltered.
//!
//! @param s
//!   The sequence of Unicode code points to transform.
string to_unicode(string s)
{
  string s0 = s;
  if(max(@values(s)) >= 128 &&
     catch(s = nameprep(s, 1)))
    return s0;
  if(lower_case(s[..3]) != "xn--")
    return s0;
  string ascii_s = s;
  catch {
    s = Punycode->decode(s[4..]);
    if(lower_case(to_ascii(s, 1)) == lower_case(ascii_s))
      return s;
  };
  return s0;
}


//! Takes a sequence of labels separated by '.' and applies
//! @[to_ascii] on each.
string zone_to_ascii(string s, int(0..1)|void allow_unassigned,
		     int(0..1)|void use_std3_ascii_rules)
{
  if(has_suffix(s, ".") && !has_suffix(s, ".."))
    return sizeof(s)>1?
      zone_to_ascii(s[..sizeof(s)-2], allow_unassigned,
		    use_std3_ascii_rules)+"."
      : s;
  else
    return to_ascii((s/".")[*], allow_unassigned, use_std3_ascii_rules)*".";
}


//! Takes a sequence of labels separated by '.' and applies
//! @[to_unicode] on each.
string zone_to_unicode(string s)
{
  return to_unicode((s/".")[*])*".";
}
