/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Florian Boucault <florian@fluendo.com>
 *
 * The inverse 4x4 matrix function comes from the Mesa 3D Graphics Library
 * and is a contribution by Jacques Leroy <jle@star.be>.
 */

/**
 * SECTION:pgmlinearalgebra
 * @short_description: Various macros and functions used for linear algebra.
 *
 * <refsect2>
 * <para>
 * Various macros and functions used by Pigment for linear algebra.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-08-27 (0.3.1)
 */

/* FIXME: Convert PgmVec, PgmVec4 and PgmMat4x4 to GBoxed */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gst/gst.h>
#include "pgmlinearalgebra.h"
#include "pgmcommon.h"

GST_DEBUG_CATEGORY (pgm_linearalgebra_debug);
#define GST_CAT_DEFAULT pgm_linearalgebra_debug

/**
 * pgm_vec3_length:
 * @v: the 3 components vector.
 *
 * Retrieves the length of the 3 components vector @v.
 *
 * MT safe.
 *
 * Returns: the length of @v.
 */
gfloat
pgm_vec3_length (const PgmVec3 *v)
{
  return sqrt (v->v[0] * v->v[0] + v->v[1] * v->v[1] + v->v[2] * v->v[2]);
}

/**
 * pgm_vec3_normalize:
 * @v: the 3 components vector.
 *
 * Performs an in place normalization of the @v 3 components vector.
 *
 * MT safe.
 */
void
pgm_vec3_normalize (PgmVec3 *v)
{
  gfloat length = 1.0f / pgm_vec3_length (v);

  v->v[0] *= length;
  v->v[1] *= length;
  v->v[2] *= length;
}

/**
 * pgm_vec3_scale:
 * @v: the resulting 3 components vector.
 * @v1: the 3 components vector to scale.
 * @k: the scalar to scale with.
 *
 * Computes the multiplication of the 3 components vector @v1 by the
 * scalar @k in the 3 components vector @v.
 *
 * MT safe.
 */
void
pgm_vec3_scale (PgmVec3 *v,
                const PgmVec3 *v1,
                const gfloat k)
{
  v->v[0] = k*v1->v[0];
  v->v[1] = k*v1->v[1];
  v->v[2] = k*v1->v[2];
}

/**
 * pgm_vec3_sum:
 * @v: the resulting 3 components vector.
 * @v1: a 3 components vector to add.
 * @v2: a 3 components vector to add.
 *
 * Computes the multiplication of the 3 components vector @v1 by the
 * scalar @k in the 3 components vector @v.
 *
 * MT safe.
 */
void
pgm_vec3_sum (PgmVec3 *v,
              const PgmVec3 *v1,
              const PgmVec3 *v2)
{
  v->v[0] = v1->v[0] + v2->v[0];
  v->v[1] = v1->v[1] + v2->v[1];
  v->v[2] = v1->v[2] + v2->v[2];
}

/**
 * pgm_vec3_cross_product:
 * @v: the resulting 3 components vector.
 * @v1: a 3 components vector to multiply.
 * @v2: a 3 components vector to multiply.
 *
 * Computes the cross product of the 3 components vector @v1 by the
 * the 3 components vector @v2 in the 3 components vector @v.
 *
 * MT safe.
 */
void
pgm_vec3_cross_product (PgmVec3 *v,
                        const PgmVec3 *v1,
                        const PgmVec3 *v2)
{
  v->v[0] = v1->v[1]*v2->v[2] - v1->v[2]*v2->v[1];
  v->v[1] = v1->v[2]*v2->v[0] - v1->v[0]*v2->v[2];
  v->v[2] = v1->v[0]*v2->v[1] - v1->v[1]*v2->v[0];
}

/**
 * pgm_vec3_dot_product:
 * @v1: a 3 components vector to multiply.
 * @v2: a 3 components vector to multiply.
 *
 * Computes the dot product of the 3 components vector @v1 by the
 * the 3 components vector @v2.
 *
 * MT safe.
 *
 * Returns: the dot product of @v1 by @v2.
 */
gfloat
pgm_vec3_dot_product (const PgmVec3 *v1,
                      const PgmVec3 *v2)
{
  return (v1->v[0]*v2->v[0] + v1->v[1]*v2->v[1] + v1->v[2]*v2->v[2]);
}


/**
 * pgm_vec4_mult_mat4x4_vec4:
 * @v: the resulting 4 components vector.
 * @m1: the 4x4 matrix.
 * @v1: the 4 components vector.
 *
 * Computes the @m 4x4 matrix product of the @m1 4x4 matrix by the @v1
 * 4 components vector.
 *
 * MT safe.
 */
void
pgm_vec4_mult_mat4x4_vec4 (PgmVec4 *v,
                           const PgmMat4x4 *m1,
                           const PgmVec4 *v1)
{
  v->v[0] = m1->m[0][0] * v1->v[0] + m1->m[1][0] * v1->v[1] +
            m1->m[2][0] * v1->v[2] + m1->m[3][0] * v1->v[3];
  v->v[1] = m1->m[0][1] * v1->v[0] + m1->m[1][1] * v1->v[1] +
            m1->m[2][1] * v1->v[2] + m1->m[3][1] * v1->v[3];
  v->v[2] = m1->m[0][2] * v1->v[0] + m1->m[1][2] * v1->v[1] +
            m1->m[2][2] * v1->v[2] + m1->m[3][2] * v1->v[3];
  v->v[3] = m1->m[0][3] * v1->v[0] + m1->m[1][3] * v1->v[1] +
            m1->m[2][3] * v1->v[2] + m1->m[3][3] * v1->v[3];
}

/**
 * pgm_mat4x4_transpose:
 * @m: the transpose 4x4 matrix.
 * @m1: the 4x4 matrix.
 *
 * Computes transpose matrix of the @m1 4x4 matrix in the @m 4x4 matrix.
 *
 * MT safe.
 */
void
pgm_mat4x4_transpose (PgmMat4x4 *m,
                      const PgmMat4x4 *m1)
{
  m->m[0][0]=m1->m[0][0]; m->m[0][1]=m1->m[1][0];
  m->m[0][2]=m1->m[2][0]; m->m[0][3]=m1->m[3][0];
  m->m[1][0]=m1->m[0][1]; m->m[1][1]=m1->m[1][1];
  m->m[1][2]=m1->m[2][1]; m->m[1][3]=m1->m[3][1];
  m->m[2][0]=m1->m[0][2]; m->m[2][1]=m1->m[1][2];
  m->m[2][2]=m1->m[2][2]; m->m[2][3]=m1->m[3][2];
  m->m[3][0]=m1->m[0][3]; m->m[3][1]=m1->m[1][3];
  m->m[3][2]=m1->m[2][3]; m->m[3][3]=m1->m[3][3];
}

/**
 * pgm_mat4x4_inverse:
 * @m: the inverse 4x4 matrix.
 * @m1: the 4x4 matrix.
 *
 * Computes inverse matrix of the @m1 4x4 matrix in the @m 4x4 matrix.
 *
 * MT safe.
 *
 * Returns: %TRUE for success, #FALSE otherwise (singular matrix).
 */
gboolean
pgm_mat4x4_inverse (PgmMat4x4 *out,
                    const PgmMat4x4 *m)
{
  gfloat m0, m1, m2, m3, s;
  gfloat *r0, *r1, *r2, *r3;
  gfloat wtmp[4][8];
  gfloat *tmp;

  r0 = wtmp[0], r1 = wtmp[1], r2 = wtmp[2], r3 = wtmp[3];

  r0[0] = m->m[0][0], r0[1] = m->m[0][1],
  r0[2] = m->m[0][2], r0[3] = m->m[0][3],
  r0[4] = 1.0, r0[5] = r0[6] = r0[7] = 0.0,

  r1[0] = m->m[1][0], r1[1] = m->m[1][1],
  r1[2] = m->m[1][2], r1[3] = m->m[1][3],
  r1[5] = 1.0, r1[4] = r1[6] = r1[7] = 0.0,

  r2[0] = m->m[2][0], r2[1] = m->m[2][1],
  r2[2] = m->m[2][2], r2[3] = m->m[2][3],
  r2[6] = 1.0, r2[4] = r2[5] = r2[7] = 0.0,

  r3[0] = m->m[3][0], r3[1] = m->m[3][1],
  r3[2] = m->m[3][2], r3[3] = m->m[3][3],
  r3[7] = 1.0, r3[4] = r3[5] = r3[6] = 0.0;

  /* choose pivot, or die */
  if (PGM_FABSF (r3[0]) > PGM_FABSF (r2[0])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (PGM_FABSF (r2[0]) > PGM_FABSF (r1[0])) { tmp = r2; r2 = r1; r1 = tmp; }
  if (PGM_FABSF (r1[0]) > PGM_FABSF (r0[0])) { tmp = r1; r1 = r0; r0 = tmp; }
  if (0.0 == r0[0]) return FALSE;

  /* eliminate first variable */
  m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
  s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
  s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
  s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;
  s = r0[4]; if (s != 0.0) { r1[4] -= m1 * s; r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r0[5]; if (s != 0.0) { r1[5] -= m1 * s; r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r0[6]; if (s != 0.0) { r1[6] -= m1 * s; r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r0[7]; if (s != 0.0) { r1[7] -= m1 * s; r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot, or die */
  if (PGM_FABSF (r3[1]) > PGM_FABSF(r2[1])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (PGM_FABSF (r2[1]) > PGM_FABSF(r1[1])) { tmp = r2; r2 = r1; r1 = tmp; }
  if (0.0 == r1[1]) return FALSE;

  /* eliminate second variable */
  m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
  r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
  r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
  s = r1[4]; if (0.0 != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r1[5]; if (0.0 != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r1[6]; if (0.0 != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r1[7]; if (0.0 != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot, or die */
  if (PGM_FABSF (r3[2]) > PGM_FABSF (r2[2])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (0.0 == r2[2]) return FALSE;

  /* eliminate third variable */
  m3 = r3[2]/r2[2];
  r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4],
  r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6],
  r3[7] -= m3 * r2[7];

  /* last check */
  if (0.0 == r3[3]) return FALSE;

  /* back substitute row 3 */
  s = 1.0f/r3[3];
  r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;

  /* back substitute row 2 */
  m2 = r2[3];
  s = 1.0F/r2[2];
  r2[4] = s * (r2[4] - r3[4] * m2), r2[5] = s * (r2[5] - r3[5] * m2),
  r2[6] = s * (r2[6] - r3[6] * m2), r2[7] = s * (r2[7] - r3[7] * m2);
  m1 = r1[3];
  r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1,
  r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
  m0 = r0[3];
  r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0,
  r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;

  /* back substitute row 1 */
  m1 = r1[2];
  s = 1.0F/r1[1];
  r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1),
  r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
  m0 = r0[2];
  r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0,
  r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;

  /* back substitute row 0 */
  m0 = r0[1];
  s = 1.0F/r0[0];
  r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0),
  r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);

  out->m[0][0] = r0[4]; out->m[0][1] = r0[5],
  out->m[0][2] = r0[6]; out->m[0][3] = r0[7],
  out->m[1][0] = r1[4]; out->m[1][1] = r1[5],
  out->m[1][2] = r1[6]; out->m[1][3] = r1[7],
  out->m[2][0] = r2[4]; out->m[2][1] = r2[5],
  out->m[2][2] = r2[6]; out->m[2][3] = r2[7],
  out->m[3][0] = r3[4]; out->m[3][1] = r3[5],
  out->m[3][2] = r3[6]; out->m[3][3] = r3[7];

  return TRUE;
}

/**
 * pgm_mat4x4_rotate_x:
 * @m: the 4x4 matrix.
 * @theta: the amount of rotation, in radians.
 *
 * Performs an in place rotation of the @m 4x4 matrix about the x cardinal axis.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_x (PgmMat4x4 *m,
                     gfloat theta)
{
  gfloat s, c;

  PGM_SINCOS (theta, &s, &c);

  m->m[0][0]=1.0f; m->m[0][1]=0.0f; m->m[0][2]=0.0f; m->m[0][3]=0.0f;
  m->m[1][0]=0.0f; m->m[1][1]=c;    m->m[1][2]=s;    m->m[1][3]=0.0f;
  m->m[2][0]=0.0f; m->m[2][1]=-s;   m->m[2][2]=c;    m->m[2][3]=0.0f;
  m->m[3][0]=0.0f; m->m[3][1]=0.0f; m->m[3][2]=0.0f; m->m[3][3]=1.0f;
}

/**
 * pgm_mat4x4_rotate_y:
 * @m: the 4x4 matrix.
 * @theta: the amount of rotation, in radians.
 *
 * Performs an in place rotation of the @m 4x4 matrix about the y cardinal axis.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_y (PgmMat4x4 *m,
                     gfloat theta)
{
  gfloat s, c;

  PGM_SINCOS (theta, &s, &c);

  m->m[0][0]=c;    m->m[0][1]=0.0f; m->m[0][2]=-s;   m->m[0][3]=0.0f;
  m->m[1][0]=0.0f; m->m[1][1]=1.0f; m->m[1][2]=0.0f; m->m[1][3]=0.0f;
  m->m[2][0]=s;    m->m[2][1]=0.0f; m->m[2][2]=c;    m->m[2][3]=0.0f;
  m->m[3][0]=0.0f; m->m[3][1]=0.0f; m->m[3][2]=0.0f; m->m[3][3]=1.0f;
}

/**
 * pgm_mat4x4_rotate_z:
 * @m: the 4x4 matrix.
 * @theta: the amount of rotation, in radians.
 *
 * Performs an in place rotation of the @m 4x4 matrix about the z cardinal axis.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_z (PgmMat4x4 *m,
                     gfloat theta)
{
  gfloat s, c;

  PGM_SINCOS (theta, &s, &c);

  m->m[0][0]=c;    m->m[0][1]=s;    m->m[0][2]=0.0f; m->m[0][3]=0.0f;
  m->m[1][0]=-s;   m->m[1][1]=c;    m->m[1][2]=0.0f; m->m[1][3]=0.0f;
  m->m[2][0]=0.0f; m->m[2][1]=0.0f; m->m[2][2]=1.0f; m->m[2][3]=0.0f;
  m->m[3][0]=0.0f; m->m[3][1]=0.0f; m->m[3][2]=0.0f; m->m[3][3]=1.0f;
}

/**
 * pgm_mat4x4_rotate:
 * @m: the 4x4 matrix.
 * @axis: the axis of rotation, must be a unit vector.
 * @theta: the amount of rotation, in radians.
 *
 * Performs an in place rotation of the @m 4x4 matrix about an arbitrary axis.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate (PgmMat4x4 *m,
                   const PgmVec3 *axis,
                   gfloat theta)
{
  gfloat s, c, a, ax, ay, az;

  PGM_SINCOS (theta, &s, &c);

  a = 1.0f - c;
  ax = a * axis->v[0];
  ay = a * axis->v[1];
  az = a * axis->v[2];

  m->m[0][0] = ax * axis->v[0] + c;
  m->m[0][1] = ax * axis->v[1] + axis->v[2] * s;
  m->m[0][2] = ax * axis->v[2] - axis->v[1] * s;
  m->m[0][3] = 0.0f;

  m->m[1][0] = ay * axis->v[0] - axis->v[2] * s;
  m->m[1][1] = ay * axis->v[1] + c;
  m->m[1][2] = ay * axis->v[2] + axis->v[0] * s;
  m->m[1][3] = 0.0f;

  m->m[2][0] = az * axis->v[0] + axis->v[1] * s;
  m->m[2][1] = az * axis->v[1] - axis->v[0] * s;
  m->m[2][2] = az * axis->v[2] + c;
  m->m[2][3] = 0.0f;

  m->m[3][0] = 0.0f;
  m->m[3][1] = 0.0f;
  m->m[3][2] = 0.0f;
  m->m[3][3] = 1.0f;
}

/**
 * pgm_mat4x4_translate:
 * @m: the 4x4 matrix.
 * @t: the translation to apply.
 *
 * Performs an in place translation of the @m 4x4 matrix.
 *
 * MT safe.
 */
void
pgm_mat4x4_translate (PgmMat4x4 *m,
                      const PgmVec3 *t)
{
  m->m[0][3] = m->m[0][0] * t->v[0] + m->m[0][1] * t->v[1] + m->m[0][2] * t->v[2] + m->m[0][3];
  m->m[1][3] = m->m[1][0] * t->v[0] + m->m[1][1] * t->v[1] + m->m[1][2] * t->v[2] + m->m[1][3];
  m->m[2][3] = m->m[2][0] * t->v[0] + m->m[2][1] * t->v[1] + m->m[2][2] * t->v[2] + m->m[2][3];
  m->m[3][3] = m->m[3][0] * t->v[0] + m->m[3][1] * t->v[1] + m->m[3][2] * t->v[2] + m->m[3][3];
}

/**
 * pgm_mat4x4_scale:
 * @m: the 4x4 matrix.
 * @s: the scale to apply.
 *
 * Performs an in place scaling of the @m 4x4 matrix.
 *
 * MT safe.
 */
void
pgm_mat4x4_scale (PgmMat4x4 *m,
                  const PgmVec3 *s)
{
  m->m[0][0] *= s->v[0]; m->m[0][1] *= s->v[1]; m->m[0][2] *= s->v[2];
  m->m[1][0] *= s->v[0]; m->m[1][1] *= s->v[1]; m->m[1][2] *= s->v[2];
  m->m[2][0] *= s->v[0]; m->m[2][1] *= s->v[1]; m->m[2][2] *= s->v[2];
  m->m[3][0] *= s->v[0]; m->m[3][1] *= s->v[1]; m->m[3][2] *= s->v[2];
}

/**
 * pgm_mat4x4_scale_along_axis:
 * @m: the 4x4 matrix.
 * @axis: the axis of scale, must be a unit vector.
 * @k: the scale to apply.
 *
 * Performs an in place scaling of the @m 4x4 matrix along an arbitrary axis.
 *
 * MT safe.
 */
void
pgm_mat4x4_scale_along_axis (PgmMat4x4 *m,
                             const PgmVec3 *axis,
                             gfloat k)
{
  gfloat a = k - 1.0f;
  gfloat ax = a * axis->v[0];
  gfloat ay = a * axis->v[1];
  gfloat az = a * axis->v[2];

  m->m[0][0] = ax * axis->v[0] + 1.0f;
  m->m[1][1] = ay * axis->v[1] + 1.0f;
  m->m[2][1] = az * axis->v[2] + 1.0f;

  m->m[0][1] = m->m[1][0] = ax * axis->v[1];
  m->m[0][2] = m->m[2][0] = ax * axis->v[2];
  m->m[1][2] = m->m[2][1] = ay * axis->v[2];

  m->m[0][3] = m->m[1][3] = m->m[2][3] = 0.0f;
  m->m[3][0] = m->m[3][1] = m->m[3][2] = 0.0f;
  m->m[3][3] = 1.0f;
}

/**
 * pgm_mat4x4_mult_mat4x4_mat4x4:
 * @m: the resulting 4x4 matrix.
 * @m1: the 1st 4x4 matrix.
 * @m2: the 2nd 4x4 matrix.
 *
 * Computes the @m 4x4 matrix product of the @m1 4x4 matrix by the @m2 4x4
 * matrix.
 *
 * MT safe.
 */
void
pgm_mat4x4_mult_mat4x4_mat4x4 (PgmMat4x4 *m,
                               const PgmMat4x4 *m1,
                               const PgmMat4x4 *m2)
{
  guint i;

  for (i = 0; i < 4; i++)
    {
      m->m[i][0] = m1->m[i][0] * m2->m[0][0] + m1->m[i][1] * m2->m[1][0] +
                   m1->m[i][2] * m2->m[2][0] + m1->m[i][3] * m2->m[3][0];
      m->m[i][1] = m1->m[i][0] * m2->m[0][1] + m1->m[i][1] * m2->m[1][1] +
                   m1->m[i][2] * m2->m[2][1] + m1->m[i][3] * m2->m[3][1];
      m->m[i][2] = m1->m[i][0] * m2->m[0][2] + m1->m[i][1] * m2->m[1][2] +
                   m1->m[i][2] * m2->m[2][2] + m1->m[i][3] * m2->m[3][2];
      m->m[i][3] = m1->m[i][0] * m2->m[0][3] + m1->m[i][1] * m2->m[1][3] +
                   m1->m[i][2] * m2->m[2][3] + m1->m[i][3] * m2->m[3][3];
    }
}


/**
 * pgm_intersection_line_plane:
 * @v: the intersection point of the line and the plane if any.
 * @l1: a point of the line.
 * @l2: a point of the line.
 * @p: a point of the plane.
 * @pu: a vector colinear to the plane.
 * @pv: a vector colinear to the plane.
 *
 * Computes the intersection of the line defined going through the points @l1
 * and @l2 with the plane (@p, @pu, @pv), with @p a point of the plane, @pu
 * and @pv two vectors colinear to the plane non colinear to each other.
 * The result is then put in @v if there is an intersection.
 *
 * Warning: if the line (@l1, l2) belongs to the plane (@p, @pu, @pv) then
 * this function considers that there is no intersection instead.
 *
 * MT safe.
 *
 * Returns: TRUE if the line (@l1, @l2) intersects the plane (@p, @pu, @pv),
 * FALSE otherwise.
 */
gboolean
pgm_intersection_line_plane (PgmVec3 *v,
                             const PgmVec3 *l1,
                             const PgmVec3 *l2,
                             const PgmVec3 *p,
                             const PgmVec3 *pu,
                             const PgmVec3 *pv)
{
  gboolean intersect = FALSE;
  PgmVec3 MA, MN, MP;
  PgmVec3 n;
  gfloat k;
  gfloat dot;

  pgm_vec3_scale (&MA, l1, -1);
  pgm_vec3_sum (&MA, p, &MA);

  pgm_vec3_scale (&MN, l1, -1);
  pgm_vec3_sum (&MN, l2, &MN);

  pgm_vec3_cross_product (&n, pu, pv);
  pgm_vec3_normalize (&n);

  dot = pgm_vec3_dot_product (&n, &MN);

  if (dot != 0.0)
    {
      k = pgm_vec3_dot_product (&n, &MA) / dot;
      pgm_vec3_scale (&MP, &MN, k);
      pgm_vec3_sum (v, l1, &MP);
      intersect = TRUE;
    }

  return intersect;
}

/**
 * pgm_vec3_belongs_rectangle:
 * @p: the 3 components point.
 * @r: a 3 components vector defining the position top-left corner of the
 * rectangle.
 * @ru: a 3 components vector defining the top edge of the rectangle.
 * @rv: a 3 components vector defining the left edge of the rectangle.
 *
 * Decides if @p belongs to the rectangle defined by the point @r and the
 * vectors @ru and @rv.
 *
 * MT safe.
 *
 * Returns: TRUE if the point @p is inside the rectangle (@r, @ru, @rv), FALSE
 * otherwise.
 */
gboolean
pgm_vec3_belongs_rectangle (const PgmVec3 *p,
                            const PgmVec3 *r,
                            const PgmVec3 *ru,
                            const PgmVec3 *rv)
{
  PgmVec3 AM;
  gfloat a, b;
  gfloat norm_ru_2, norm_rv_2;

  pgm_vec3_scale (&AM, r, -1);
  pgm_vec3_sum (&AM, p, &AM);

  a = pgm_vec3_dot_product (ru, &AM);
  b = pgm_vec3_dot_product (rv, &AM);
  norm_ru_2 = ru->v[0] * ru->v[0] + ru->v[1] * ru->v[1] + ru->v[2] * ru->v[2];
  norm_rv_2 = rv->v[0] * rv->v[0] + rv->v[1] * rv->v[1] + rv->v[2] * rv->v[2];

  return (a>=0 && a<=norm_ru_2 && b>=0 && b<=norm_rv_2);
}
