/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmimage
 * @short_description: An image drawable supporting various media.
 * @see_also: #PgmDrawable, #PgmText, #PgmCanvas.
 *
 * <refsect2>
 * <para>
 * #PgmImage is a drawable displaying media. It supports various ways of loading
 * images through buffers, file descriptors or videos through GStreamer.
 * </para>
 * <title>Loading image data</title>
 * <para>
 * Image data loading can happen with three different functions:
 * <itemizedlist>
 * <listitem>
 * <para>
 * pgm_image_set_from_buffer() takes a pre-loaded data buffer and sets it as
 * the currently displayed image. This is useful when you want to use an image
 * loading library (GdkPixbuf, FreeImage, etc) in your application and just
 * provide the pixels to #PgmImage for display. The data buffer containing the
 * pixels is copied internally, you can free the data buffer from the
 * application side as soon as the function returns.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_gst_buffer() takes a GStreamer #GstBuffer and sets it as
 * the currently displayed image. This is mostly used to do video rendering.
 * There's no copying of the buffer data to optimize performances, indeed the
 * reference count of the buffer is going to be increased to keep the buffer
 * around while it's needed for rendering. When you call pgm_image_clear() the
 * reference to the buffer will be decreased and the buffer can get freed. Note
 * that this method is used by #PgmSink to render video frames directly in a
 * #PgmImage when the pipeline is playing.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_fd() takes a pre-opened file descriptor to an image file
 * delegating image loading to Pigment. Thus the loading is asynchronous and
 * won't block the Pigment main loop.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * <title>Sharing image data between #PgmImage objects
 * </title>
 * <para>
 * pgm_image_set_from_image() is a convenient system to slave an image to
 * another one. Indeed you might want to load an image data once and then use
 * it in multiple image objects. In that case this image becomes a slave to the
 * one that has the image data loaded internally and each time it needs to draw
 * it will use that data.
 * </para>
 * <para>
 * Layout settings of the drawable are independent from one image to another.
 * That means that even if two image objects are using the same image, they can
 * have different colors, different #PgmDrawableLayoutType or different
 * #PgmDrawableAlignment.
 * </para>
 * <para>
 * Each time a new image data buffer is loaded in the master image object, all
 * the slave image objects are automatically updated. That means you can render
 * a video clip in ten different drawables without doing anything else than
 * slaving nine image objects to the one that's receiving the image data.
 * </para>
 * <para>
 * Note that an image that has been set as a master to an other image gets its
 * reference count increased for each slave it has, which means it won't get
 * freed until all the references are gone. This can happen either because the
 * slave drawable is removed from the canvas or if pgm_image_clear() is called
 * on the slave drawable.
 * </para>
 * <title>Image data aspect ratio</title>
 * <para>
 * This rarely happens with normal images but video rendering often has non
 * square pixels video frames coming out of the video decoders (DVD, DV cameras,
 * etc). In that case a calculation has to be done when projecting to the
 * viewport so that we put in adequation both the pixel aspect ratio and the
 * source video aspect ratio. You can set the image aspect ratio using
 * pgm_image_set_aspect_ratio() and be assured that Pigment is going to render
 * that image correctly on the viewport.
 * </para>
 * <title>Benefitting from hardware acceleration</title>
 * <para>
 * Depending on the viewport implementation, some #PgmImagePixelFormat (color
 * space) can be supported or not. When it comes to video rendering, hardware
 * acceleration is very important and you need to know what kind of pixel
 * formats are convenient for the rendering backend. you can get a list of
 * supported (accelerated) pixel formats using
 * pgm_viewport_get_pixel_formats().
 * </para>
 * <title>Rendering video frames directly in video memory</title>
 * <para>
 * On small machines, memory copies are very expensive for the hardware and they
 * should be avoided at all cost. GStreamer supports a nice system to render
 * video frames directly in hardware allocated memory buffers through the
 * gst_pad_alloc_buffer() function calls. This is why we have a method called
 * pgm_image_alloc_gst_buffer() which returns a pre-allocated #GstBuffer with
 * requested pixel format and size, and preferably allocated in video memory.
 * This #GstBuffer will come back through pgm_image_set_from_gst_buffer() and it
 * will be rendered directly without any copy of the raw video frame to system
 * memory. Some rendering plugins might not support that feature and will return
 * a system allocated memory buffer instead.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-04-12 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmimage.h"
#include "pgmmarshal.h"

/* Number of bytes read from the fd at each callback */
#define FD_LOADER_CHUNK_SIZE 1024

GST_DEBUG_CATEGORY (pgm_image_debug);
#define GST_CAT_DEFAULT pgm_image_debug

/* Image signals */
enum {
  FD_LOADED,
  LAST_SIGNAL
};

/* File descriptor loading */
typedef struct {
  PgmImage        *image;    /* Image object */
  GIOChannel      *channel;  /* File descriptor wrapper */
  GdkPixbufLoader *loader;   /* Pixbuf loader */
  GMutex          *lock;     /* Access lock */
  gboolean         loaded;   /* Is the loading finished? */
  guint            max_size; /* Scaling threshold */
  guint            source;   /* Pigment main loop source id */
} FdLoader;

static PgmDrawableClass *parent_class = NULL;
static guint pgm_image_signals[LAST_SIGNAL] = { 0 };

/* Private methods */

/* Update the ratio of the slaves.
 * Called with Image LOCK. */
static void
update_slaves_ratio (PgmImage *image)
{
  PgmImage *slave;
  GList *walk;

  walk = image->slaves;
  while (walk)
    {
      slave = walk->data;
      slave->par_n = image->par_n;
      slave->par_d = image->par_d;
      walk = walk->next;
    }
}

/* File descriptor loader reading chunk callback */
static gint
fd_loader_read_chunk_cb (gpointer data)
{
  FdLoader *fd_loader = (FdLoader*) data;
  PgmImage *image = fd_loader->image;
  gchar buffer[FD_LOADER_CHUNK_SIZE];
  gint ret = FALSE;
  gsize bytes_read;
  GIOStatus status;

  g_mutex_lock (fd_loader->lock);

  /* Let's read the chunk of data */
  status = g_io_channel_read_chars (fd_loader->channel, buffer,
                                    FD_LOADER_CHUNK_SIZE, &bytes_read, NULL);

  switch (status)
    {
      /* Image chunk loaded successfully */
    case G_IO_STATUS_NORMAL:
      {
        gboolean success;
        /* Write the chunk */
        success = gdk_pixbuf_loader_write (fd_loader->loader,
                                           (const guchar *) buffer,
                                           bytes_read, NULL);
        if (!success)
          {
            GST_DEBUG_OBJECT (image, "GdkPixbufLoader cannot parse the buffer");
            ret = FALSE;
            goto error;
          }

        ret = TRUE;
        goto incomplete;
      }
      break;

      /* Image data fully loaded */
    case G_IO_STATUS_EOF:
      {
        GdkPixbuf *pixbuf;
        gboolean success;

        /* Write the last chunk */
        success = gdk_pixbuf_loader_write (fd_loader->loader,
                                           (const guchar *) buffer,
                                           bytes_read, NULL);
        if (!success)
          {
            GST_DEBUG_OBJECT (image, "GdkPixbufLoader cannot parse the buffer");
            ret = FALSE;
            goto error;
          }

        /* Get a ref on the pixbuf */
        pixbuf = gdk_pixbuf_loader_get_pixbuf (fd_loader->loader);
        if (!pixbuf)
          {
            GST_DEBUG_OBJECT (image, "GdkPixbufLoader cannot get the pixbuf");
            ret = FALSE;
            goto error;
          }

        /* Clean up the previous data */
        pgm_image_clear (image);

        /* And fill the new storage informations */
        GST_OBJECT_LOCK (image);
        image->storage_type = PGM_IMAGE_FD;
        image->par_n = gdk_pixbuf_get_width (pixbuf);
        image->par_d = gdk_pixbuf_get_height (pixbuf);
        image->data.fd.pixbuf = gdk_pixbuf_ref (pixbuf);
        update_slaves_ratio (image);
        GST_OBJECT_UNLOCK (image);
        ret = FALSE;
      }
      break;

      /* An error occurred while reading */
    case G_IO_STATUS_ERROR:
      GST_DEBUG_OBJECT (image, "Cannot read from the GIOChannel: "
                        "an error occurred");
      ret = FALSE;
      goto error;

      /* The resource is busy */
    case G_IO_STATUS_AGAIN:
      GST_DEBUG_OBJECT (image, "Cannot read from the GIOChannel: "
                        "resource temporarily unavailable");
      ret = TRUE;
      goto incomplete;

    default:
      break;
    }

  /* FIXME: this might be the source of a deadlock */
  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_FD);
  g_signal_emit (G_OBJECT (image), pgm_image_signals[FD_LOADED], 0);

 error:
  g_io_channel_shutdown (fd_loader->channel, TRUE, NULL);
  g_io_channel_unref (fd_loader->channel);
  gdk_pixbuf_loader_close (fd_loader->loader, NULL);
  g_object_unref (fd_loader->loader);
  g_source_remove (fd_loader->source);
  fd_loader->loaded = TRUE;

 incomplete:
  g_mutex_unlock (fd_loader->lock);

  return ret;
}

/* GdkPixbufLoader handler called whenever loader's figured out the size */
static void
fd_loader_size_prepared_cb (GdkPixbufLoader *loader,
                            gint width,
                            gint height,
                            gpointer data)
{
  FdLoader *fd_loader = (FdLoader *) data;
  guint max_size = fd_loader->max_size;

  /* Request a scaling if the width or the height exceeds the maximum size */
  if (width > max_size || height > max_size)
    {
      if (width >= height)
        {
          height = height * max_size / width;
          width = max_size;
        }
      else
        {
          width = width * max_size / height;
          height = max_size;
        }

      gdk_pixbuf_loader_set_size (loader, width, height);
    }
}

/* File descriptor loader constructor */
static FdLoader *
fd_loader_new (PgmImage *image,
               guint fd,
               guint max_size)
{
  FdLoader *fd_loader;
  GIOStatus status;

  fd_loader = g_slice_new0 (FdLoader);

  fd_loader->channel = g_io_channel_unix_new (fd);

  /* Set the encoding to NULL to read raw binary data without interpretation */
  status = g_io_channel_set_encoding (fd_loader->channel, NULL, NULL);
  if (status != G_IO_STATUS_NORMAL)
    {
      GST_DEBUG_OBJECT (image, "Can't set encoding of the GIOChannel to RAW");
      g_io_channel_shutdown (fd_loader->channel, TRUE, NULL);
      g_io_channel_unref (fd_loader->channel);
      return NULL;
    }

  fd_loader->lock = g_mutex_new ();
  fd_loader->image = image;
  fd_loader->max_size = max_size;
  fd_loader->loaded = FALSE;

  fd_loader->loader = gdk_pixbuf_loader_new ();

  /* Connect to 'size_prepared' signal to adapt image size as requested */
  if (max_size != 0)
    g_signal_connect (fd_loader->loader, "size_prepared",
                      G_CALLBACK (fd_loader_size_prepared_cb), fd_loader);

  /* Add the progressive loading source to the main Pigment loop */
  fd_loader->source = g_idle_add (fd_loader_read_chunk_cb, fd_loader);

  return fd_loader;
}

/* File descriptor loader destructor */
static void
fd_loader_free (FdLoader *fd_loader)
{
  if (fd_loader == NULL)
    return;

  g_mutex_lock (fd_loader->lock);
  if (!fd_loader->loaded)
    {
      g_io_channel_shutdown (fd_loader->channel, TRUE, NULL);
      g_io_channel_unref (fd_loader->channel);
      gdk_pixbuf_loader_close (fd_loader->loader, NULL);
      g_object_unref (fd_loader->loader);
      g_source_remove (fd_loader->source);
    }
  g_mutex_unlock (fd_loader->lock);
  g_mutex_free (fd_loader->lock);

  g_slice_free (FdLoader, fd_loader);
}

/* GObject stuff */

G_DEFINE_TYPE (PgmImage, pgm_image, PGM_TYPE_DRAWABLE);

static void
pgm_image_dispose (GObject *object)
{
  PgmImage *image = PGM_IMAGE (object);

  fd_loader_free (image->fd_loader);

  /* Clean up storage data */
  switch (image->storage_type)
    {
    case PGM_IMAGE_BUFFER:
      g_free (image->data.buffer.buffer);
      break;

    case PGM_IMAGE_GST_BUFFER:
      gst_buffer_unref (image->data.gst_buffer.gst_buffer);
      break;

    case PGM_IMAGE_FD:
      gdk_pixbuf_unref ((GdkPixbuf*) (image->data.fd.pixbuf));
      break;

      /* A master Image object keeps a reference on its slave, so if the
       * storage type is PGM_IMAGE_IMAGE, this destructor is not called. The
       * user needs to explicitely call a clear() on a slave Image if he
       * actually wants to see it disposed. So we don't need to do anything
       * there. */
    case PGM_IMAGE_IMAGE:
      break;

    default:
      break;
    }

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_image_class_init (PgmImageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmImage::fd-loaded:
   * @image: the #PgmImage
   *
   * Will be emitted after @image has finished to load its data from the file
   * descriptor given in methods like pgm_image_set_from_fd() or
   * pgm_image_new_from_fd().
   */
  pgm_image_signals[FD_LOADED] =
    g_signal_new ("fd-loaded", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET (PgmImageClass, fd_loaded),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_image_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_image_debug, "pgm_image", 0,
                           "Pigment Image object");
}

static void
pgm_image_init (PgmImage *image)
{
  image->storage_type = PGM_IMAGE_EMPTY;
  image->par_n = 0;
  image->par_d = 1;
  image->align = PGM_IMAGE_CENTER;
  image->layout = PGM_IMAGE_SCALED;
  image->interp = PGM_IMAGE_BILINEAR;
  image->master = NULL;
  image->slaves = NULL;
  image->fd_loader = NULL;
}

/* public methods */

/**
 * pgm_image_new:
 *
 * Creates a new #PgmImage instance.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new (void)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_buffer:
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the image rowstride in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Creates a new #PgmImage instance with the image from the given buffer.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_buffer (PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_buffer (image, format, width, height, stride, size, data);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_fd:
 * @fd: the file descriptor from which data will be read to load image.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Creates a new #PgmImage instance loading progressively an image from the
 * pre-opened file descriptor @fd. The file descriptor is then closed by
 * Pigment. It optionally pre-scales the image so that it has a maximum width
 * and height of @max_size.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_fd (guint fd,
                       guint max_size)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_fd (image, fd, max_size);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_image:
 * @src_image: a #PgmImage which will be used as the master image.
 *
 * Creates a new #PgmImage instance with an image slaved from the
 * image of @src_image.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_image (PgmImage *src_image)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_image (image, src_image);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_set_from_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Loads an image in @image from an existing buffer using pixel format @format.
 * If you don't know the rowstride of the image you can set stride to 0. @data
 * is copied internally you can free it right after the function call returns.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_buffer (PgmImage *image,
                           PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  gpointer _data = NULL;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (data != NULL, PGM_ERROR_X);

  /* Let's set the storage data */
  GST_OBJECT_LOCK (image);

  /* The buffer sent is not the first of this size, it's not needed to call
   * a clear, just free the buffer. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_GST_BUFFER
                && image->data.buffer.width == width
                && image->data.buffer.height == height
                && image->data.buffer.format == format
                && image->data.buffer.stride == stride))
    {
      if (G_LIKELY (image->data.buffer.buffer))
        g_free (image->data.buffer.buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          GST_OBJECT_UNLOCK (image);
          pgm_image_clear (image);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_BUFFER;
      image->data.gst_buffer.format = format;
      image->data.gst_buffer.width = width;
      image->data.gst_buffer.height = height;
      image->data.gst_buffer.stride = stride;
      image->data.buffer.size = size;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Try to copy the given buffer */
  _data = g_memdup (data, size);

  image->data.buffer.buffer = (guint8 *) _data;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_gst_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @buffer: A #GstBuffer reference containing the video frame.
 *
 * Loads an image in @image from an existing #GstBuffer using the pixel format
 * @format. If you don't know the rowstride of the image you can set stride
 * to 0. @buffer will have its reference count increased by 1 and will not get
 * freed until the drawable gets cleaned up or that a new buffer is loaded.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_gst_buffer (PgmImage *image,
                               PgmImagePixelFormat format,
                               guint width,
                               guint height,
                               guint stride,
                               GstBuffer *buffer)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (GST_IS_BUFFER (buffer), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  /* The GstBuffer sent is not the first, it's not needed to call a clear,
   * just unref the GstBuffer. I mean that the implementation should not be
   * signaled to clear its allocated size/format for the stream of buffers
   * since it's really heavy to clear for each new one. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_GST_BUFFER
                && image->data.gst_buffer.width == width
                && image->data.gst_buffer.height == height
                && image->data.gst_buffer.format == format
                && image->data.gst_buffer.stride == stride))
    {
      if (G_LIKELY (image->data.gst_buffer.gst_buffer))
        gst_buffer_unref (image->data.gst_buffer.gst_buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          GST_OBJECT_UNLOCK (image);
          pgm_image_clear (image);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_GST_BUFFER;
      image->data.gst_buffer.format = format;
      image->data.gst_buffer.width = width;
      image->data.gst_buffer.height = height;
      image->data.gst_buffer.stride = stride;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Take a ref on the GstBuffer */
  image->data.gst_buffer.gst_buffer = gst_buffer_ref (buffer);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_GST_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_fd:
 * @image: a #PgmImage object.
 * @fd: the file descriptor from which data will be read to load image.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Progressively loads an image from the pre-opened file descriptor @fd. The
 * file descriptor is then closed by Pigment. It optionally pre-scales the
 * image so it has a maximum width and height of @max_size.
 *
 * This function is meant to be asynchronous, it loads the image by small chunks
 * of 1024 bytes using an idle source in the Pigment mainloop. The consequence
 * being that the storage type of @image doesn't change immediately, but only
 * once the whole image is loaded. You can connect a callback to the
 * <link linkend="PgmImage-fd-loaded">fd-loaded</link> signal to know when the
 * loading is done.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_fd (PgmImage *image,
                       guint fd,
                       guint max_size)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (fd > 0, PGM_ERROR_X);

  /* Free the current file descriptor loader and create the new one */
  fd_loader_free ((FdLoader*) image->fd_loader);
  image->fd_loader = (gpointer) fd_loader_new (image, fd, max_size);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_image:
 * @image: a #PgmImage object.
 * @src_image: the source #PgmImage object to use as a master.
 *
 * Slaves @image to @src_image. Every change to @src_image is reflected on
 * @image until you remove @image from the canvas or you call pgm_image_clear()
 * on @image. @src_image and @image see their reference count increased by 1
 * during that method as they need each other.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_image (PgmImage *image,
                          PgmImage *src_image)
{
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_IMAGE (src_image), PGM_ERROR_X);

  GST_DEBUG_OBJECT (image, "using image from %s", GST_OBJECT_NAME (src_image));

  pgm_image_clear (image);

  GST_OBJECT_LOCK (image);

  /* Trying to deadlock us? :) */
  if (G_UNLIKELY (image == src_image))
    {
      GST_WARNING_OBJECT (image, "trying to do a master/slave loop!");
      GST_OBJECT_UNLOCK (image);
      ret = PGM_ERROR_X;
      goto beach;
    }

  image->storage_type = PGM_IMAGE_IMAGE;

  GST_OBJECT_LOCK (src_image);

  /* The master requested is already a slave */
  if (G_UNLIKELY (src_image->master))
    {
      GST_DEBUG_OBJECT (image, "%s is already a slave to %s, using its master",
                        GST_OBJECT_NAME (src_image),
                        GST_OBJECT_NAME (src_image->master));

      GST_OBJECT_LOCK (src_image->master);
      /* Add ourself to the slave list with an increased reference */
      src_image->master->slaves = g_list_append (src_image->master->slaves,
                                                 gst_object_ref (image));
      /* Keep a reference to the real master */
      image->master = gst_object_ref (src_image->master);
      GST_OBJECT_UNLOCK (src_image->master);
    }

  /* The master requested is not a slave */
  else
    {
      /* Add ourself to the slave list with an increased reference */
      src_image->slaves = g_list_append (src_image->slaves,
                                         gst_object_ref (image));
      /* Keep a reference to our master */
      image->master = gst_object_ref (src_image);
    }

  image->par_n = src_image->par_n;
  image->par_d = src_image->par_d;

  GST_OBJECT_UNLOCK (src_image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_IMAGE);

beach:
  return ret;
}

/**
 * pgm_image_clear:
 * @image: a #PgmImage object.
 *
 * Removes any image from @image. If @image had some image data loaded, it's
 * cleared, if there was a #GstBuffer used, it's unreffed and if the @image was
 * a slave to another it is not anymore. If @image has slave images they all
 * get cleared but they still are slaves to @image. So if you load a new image
 * to @image, all the slaves will load it too.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_clear (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  switch (image->storage_type)
    {
    case PGM_IMAGE_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing buffer image");
      g_free (image->data.buffer.buffer);
      break;

    case PGM_IMAGE_GST_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing GstBuffer image");
      gst_buffer_unref (image->data.gst_buffer.gst_buffer);
      break;

    case PGM_IMAGE_FD:
      GST_DEBUG_OBJECT (image, "Clearing fd image");
      gdk_pixbuf_unref ((GdkPixbuf *) (image->data.fd.pixbuf));
      break;

    case PGM_IMAGE_IMAGE:
      GST_DEBUG_OBJECT (image, "Clearing slaved image");

      /* Remove ourself from the master's slave list dropping the ref */
      GST_OBJECT_LOCK (image->master);
      image->master->slaves = g_list_remove (image->master->slaves, image);
      gst_object_unref (image);
      GST_OBJECT_UNLOCK (image->master);

      /* Drop the reference we took from the master */
      gst_object_unref (image->master);
      image->master = NULL;
      break;

    default:
      break;
    }

  image->storage_type = PGM_IMAGE_EMPTY;
  image->par_n = 0;
  image->par_d = 1;
  update_slaves_ratio (image);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_storage_type:
 * @image: a #PgmImage object.
 * @storage: a #PgmImageStorageType where the storage type is going to be
 * stored.
 *
 * Retrieves the type of representation being used by @image to store image
 * data. If @image has no image data, the return value will be
 * #PGM_IMAGE_EMPTY.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_storage_type (PgmImage *image,
                            PgmImageStorageType *storage)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (storage != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *storage = image->storage_type;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_alignment:
 * @image: a #PgmImage object.
 * @align: a #PgmImageAlignment combination of flags.
 *
 * Defines the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_alignment (PgmImage *image,
                         PgmImageAlignment align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->align = align;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ALIGNMENT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_alignment:
 * @image: a #PgmImage object.
 * @align: a pointer to a #PgmImageAlignment where alignment flags are going
 * to be stored.
 *
 * Retrieves in @align the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_alignment (PgmImage *image,
                         PgmImageAlignment *align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (align != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *align = image->align;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_layout:
 * @image: a #PgmImage object.
 * @layout: a #PgmImageLayoutType layout type.
 *
 * Defines the way @image layouts its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_layout (PgmImage *image,
                      PgmImageLayoutType layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->layout = layout;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_LAYOUT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_layout:
 * @image: a #PgmImage object.
 * @layout: a pointer to a #PgmImageLayoutType where the layout type is going
 * to be stored.
 *
 * Retrieves in @layout the way @image layouts its its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */

PgmError
pgm_image_get_layout (PgmImage *image,
                      PgmImageLayoutType *layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (layout != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *layout = image->layout;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_interp:
 * @image: a #PgmImage object.
 * @interp: the interpolation type.
 *
 * Defines that @image will be rendered using @interp as its interpolation
 * type.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_interp (PgmImage *image,
                      PgmImageInterpType interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->interp = interp;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_INTERP);

  return  PGM_ERROR_OK;
}

/**
 * pgm_image_get_interp:
 * @image: a #PgmImage object.
 * @interp: a pointer to a #PgmImageInterpType where the interpolation type
 * is going to be stored.
 *
 * Retrieves in @interp the current interpolation type of @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_interp (PgmImage *image,
                      PgmImageInterpType *interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (interp != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *interp = image->interp;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: the numerator of the aspect ratio fraction.
 * @denominator: the denominator of the aspect ratio fraction.
 *
 * Customizes the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_aspect_ratio (PgmImage *image,
                            guint numerator,
                            guint denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  if (numerator == image->par_n && denominator == image->par_d)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_OK;
    }

  image->par_n = numerator;
  image->par_d = MAX (denominator, 1);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ASPECT_RATIO);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: a pointer to a #guint where the numerator of the aspect ratio
 * fraction will be stored.
 * @denominator: a pointer to a #guint where the denominator of the aspect
 * ratio fraction will be stored.
 *
 * Retrieves the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_aspect_ratio (PgmImage *image,
                            guint *numerator,
                            guint *denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (numerator != NULL, PGM_ERROR_X);
  g_return_val_if_fail (denominator != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *numerator = image->par_n;
  *denominator = image->par_d;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_alloc_gst_buffer:
 * @image: a #PgmImage object.
 * @buffer: a pointer to a #GstBuffer which will contain the newly allocated
 * buffer.
 * @format: the format of the image that will be written in the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @size: the required buffer size.
 *
 * Gets @image to allocate a video memory buffer of @width x @height with
 * @size bytes for GStreamer.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_alloc_gst_buffer (PgmImage *image,
                            GstBuffer **buffer,
                            PgmImagePixelFormat format,
                            guint width,
                            guint height,
                            guint size)
{
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (buffer != NULL, PGM_ERROR_X);

  /* FIXME */

  return ret;
}
