# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Mirco Müller <macslow@bangang.de>

from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.graph.text import Text
from pypgmtools.timing import implicit
from pypgmtools.utils import classinit
import gobject
import gst
import os
import pgm

# orientation
LEFT_RIGHT   = 0 # implies column-major layout
TOP_BOTTOM   = 1 # implies row-major layout

# scroll-mode
PAGE_WISE    = 0
ELEMENT_WISE = 1

# convenience constants
X          = 0
Y          = 1
Z          = 2
WIDTH      = 3
HEIGHT     = 4
OPACITY    = 5
PIPELINE   = 6 # this is just temporary to allow videos to be used in the demo

class GridView(Group):
    """
    This is a widget to display images, videos or composed widgets (groups, e.g.
    an image and a text) in a planar two-dimensional fashion. It allows dynamic
    rearrangement of its layout (top-to-bottom or left-to-right), changes of the
    grid-resolution (columns and rows). Furthermore it provides means to
    navigate in the set of displayed widgets using methods like prev(), next(),
    up(), down(), left() or right().

    @ivar _canvas:               The canvas the widget appears on
    @type _canvas:               pgm.Canvas
    @ivar _layer:                The layer to which the widget will be added
    @type _layer:                pgm.DrawableLayer
    @ivar _width:                The width of the widget in canvas-coordinates
    @type _width:                float
    @ivar _height:               The height of the widget in canvas-coordinates
    @type _height:               float
    @ivar _columns:              The number of columns to divide the grid into
    @type _columns:              int
    @ivar _rows:                 The number of rows to divide the grid into
    @type _rows:                 int
    @ivar _grid_width:           The widgets width - 2 * mask_size or just
                                 width, depending on the widgets orientation
                                 (in canvas-coordinates)
    @type _grid_width:           float
    @ivar _grid_height:          The widgets height - 2 * mask_size or just
                                 height, depending on the widgets orientation
                                 (in canvas-coordinates)
    @type _grid_height:          float
    @ivar _widget_width:          The width occupied by a cell- or element
                                 (in canvas-coordinates)
    @type _widget_width:          float
    @ivar _widget_height:         The height occupied by a cell- or element
                                 (in canvas-coordinates)
    @type _widget_height:         float
    @ivar _scroll_mode:          The current scroll-mode used (either PAGE_WISE
                                 or ELEMENT_WISE)
    @type _scroll_mode:          int
    @ivar _orientation:          The current orientation used (either TOP_BOTTOM
                                 or LEFT_RIGHT)
    @type _orientation:          int
    @ivar _duration:             The duration of animations in milliseconds
    @type _duration:             int
    @ivar _transformation:       The animation acceleration behaviour
    @type _transformation:       int
    @ivar _background_masks:     The two filenames of the images to use as the
                                 fade-out masks for the edges (depending on the
                                 set orientation different images have to be
                                 passed)
    @type _background_masks:     list of 2 str
    @ivar _background_mask_size: The "size" of the mask images, depending on the
                                 set orientation it is interpreted as width (in
                                 case of orientation LEFT_RIGHT) or as height
                                 (in case of orientation TOP_BOTTOM),
                                 interpreted to be in canvas-coordinates
    @type _background_mask_size: float
    @ivar _local_selection:      grid-coordinates X and Y of the currently
                                 selected widget on the page
    @type _local_selection:      list of 2 ints
    @ivar _global_selection:     grid-coordinates X and Y of the currently
                                 selected widget in the whole list of widgets
    @type _global_selection:     list of 2 ints
    @ivar _last_selection_index: list index (0..n-1) of previously selected
                                 widget
    @type _last_selection_index: int
    @ivar _selection_index:      list index (0..n-1) of currently selected widget
    @type _selection_index:      int
    @ivar _object_attribs:       list of computed target animation values for
                                 all widgets
    @type _object_attribs:       list of (float pos-x,
                                          float pos-y,
                                          float pos-z,
                                          float widget-width,
                                          float widget-height,
                                          int opacity,
                                          pipeline)
    @ivar _widgets:              list of non-animated widgets
    @type _widgets:              list of wrapped pgm.Drawables (Text, Image) or
                                 Group
    @ivar _animated_widgets:     list of implicitly animated widgets
    @type _animated_widgets:     list of AnimatedObjects
    @ivar _selector:
    @type _selector:
    @ivar _offset:               The offset by which to move on a "page-turn"
                                 (in canvas-coordinates)
    @type _offset:               float
    @ivar _last_offset_factor:   The previous "page" (see _offset_factor)
    @type _last_offset_factor:   int
    @ivar _offset_factor:        The "page" (not really a page if scroll-mode is
                                 set to ELEMENT_WISE)
    @type _offset_factor:        int
    """

    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    def __init__(self,
                 canvas,
                 layer,
                 width                 = 1.0,
                 height                = 1.0,
                 columns               = 4,
                 rows                  = 3,
                 scroll_mode           = PAGE_WISE,
                 orientation           = LEFT_RIGHT,
                 duration              = 250,
                 transformation        = implicit.DECELERATE,
                 background_masks      = None,
                 background_mask_size  = None):
        """
        Initialize a widget arranging elements in a NxM fashion.

        @param canvas:               The pigment canvas the grid-view widget
                                     should be added to.
        @type canvas:                pgm.Canvas
        @param layer:                The layer in the canvas to use for adding
                                     the grid-view widget to
        @type layer:                 pgm.DrawableLayer
        @param width                 The width of the widget
        @type width                  float
        @param height                The height of the widget
        @type height                 float
        @param columns               The number of columns for the grid
        @type columns                int
        @param rows                  The number of rows for the grid
        @type rows                   int
        @param scroll_mode           The method to "scroll" by (either PAGE_WISE
                                     or ELEMENT_WISE)
        @type scroll_mode            int
        @param orientation           The way to layout the elements in the
                                     widget (either TOP_BOTTOM or LEFT_RIGHT)
        @type orientation            int
        @param duration              Duration of animations in milliseconds
        @type duration               int
        @param transformation:       Transformation behaviour (e.g.
                                     implicit.DECELERATE)
        @type transformation:        int
        @param background_masks:     Filename of
        @type background_masks:      Two filenames of images to use as fade-out
                                     masks
        @param background_mask_size: list of str
        @type background_mask_size:  float
        """

        Group.__init__(self, canvas, layer)

        # private copies of initial parameters
        self._canvas               = canvas
        self._layer                = layer
        self._width                = width
        self._height               = height
        self._columns              = columns
        self._rows                 = rows
        self._grid_width           = self._width
        self._grid_height          = self._height
        self._widget_width         = self._grid_width / columns
        self._widget_height        = self._grid_height / rows
        self._scroll_mode          = scroll_mode
        self._orientation          = orientation
        self._duration             = duration
        self._transformation       = transformation
        self._background_masks     = background_masks
        self._background_mask_size = background_mask_size

        # just for internal use
        self._local_selection       = [0, 0]
        self._global_selection      = [0, 0]
        self._last_selection_index  = 0
        self._selection_index       = 0
        self._object_attribs        = []
        self.widgets                = []
        self.animated_widgets       = []
        self._selector              = None
        self._offset                = 0.0
        self._last_offset_factor    = -1
        self._offset_factor         = 0

        # initialize the size of the actual grid-area
        if self._orientation == TOP_BOTTOM:
            self._grid_width  = self._width
            self._grid_height = self._height - 2 * self._background_mask_size
        elif self._orientation == LEFT_RIGHT:
            self._grid_width  = self._width  - 2 * self._background_mask_size
            self._grid_height = self._height

        self._widget_width = self._grid_width / columns
        self._widget_height = self._grid_height / rows

        # set offset used for page- or element-wise scrolling
        self._update_offset()

        # user-stupidity check
        if self._columns < 1:
            self._columns = 1
        if self._rows < 1:
            self._rows = 1

        # setup the "fade out"-masks for the edges, since there is no drawable-
        # -compositing available in pigment yet I need to do some tricks in
        # order to get "edge fade out"-effects going
        if self._background_masks != None:
            # left edge
            image = Image()
            image.set_from_fd(os.open(self._background_masks[0], os.O_RDONLY))
            image.opacity  = 255
            image.visible  = True
            image.fg_color = (255, 255, 255, 255)
            image.bg_color = (0, 0, 0, 0)
            image.width    = self._width
            image.height   = self._height
            if self._orientation == LEFT_RIGHT:
                image.width = self._background_mask_size
            elif self._orientation == TOP_BOTTOM:
                image.height = self._background_mask_size
            image.position = (self.position[0],
                              self.position[1],
                              self.position[2] + 1.0)
            image.layout   = pgm.IMAGE_FILLED
            self.add(image)

            # right edge
            image = Image()
            image.set_from_fd(os.open(self._background_masks[1], os.O_RDONLY))
            image.opacity  = 255
            image.visible  = True
            image.fg_color = (255, 255, 255, 255)
            image.bg_color = (0, 0, 0, 0)
            image.width    = self._width
            image.height   = self._height
            if self._orientation == LEFT_RIGHT:
                image.width = self._background_mask_size
                image.position = (self.position[0] +
                                  self._width -
                                  self._background_mask_size,
                                  self.position[1],
                                  self.position[2] + 1.0)
            elif self._orientation == TOP_BOTTOM:
                image.height = self._background_mask_size
                image.position = (self.position[0],
                                  self.position[1] +
                                  self._height -
                                  self._background_mask_size,
                                  self.position[2] + 1.0)
            image.layout   = pgm.IMAGE_FILLED
            self.add(image)

        # do the inital arrangement and update for the widget
        self.layout()
        self._map_attribs_to_widgets()
        self.update()

    def _update_offset(self):
        if self._scroll_mode == PAGE_WISE:
            if self._orientation == LEFT_RIGHT:
                self._offset = self._grid_width
            elif self._orientation == TOP_BOTTOM:
                self._offset = self._grid_height
        elif self._scroll_mode == ELEMENT_WISE:
            if self._orientation == LEFT_RIGHT:
                self._offset = self._widget_width
            elif self._orientation == TOP_BOTTOM:
                self._offset = self._widget_height

    def _update_offset_factor(self):
        if self._scroll_mode == PAGE_WISE:
            self._last_offset_factor = self._offset_factor
            self._offset_factor = self._selection_index / \
                                  (self._columns * \
                                   self._rows)
        elif self._scroll_mode == ELEMENT_WISE:
            if self._orientation == LEFT_RIGHT:
                self._last_offset_factor = self._offset_factor
                self._offset_factor = self._selection_index / self._rows
            elif self._orientation == TOP_BOTTOM:
                self._last_offset_factor = self._offset_factor
                self._offset_factor = self._selection_index / self._columns

    # update the animated objects to the current set of attribute-values
    def _map_attribs_to_widgets(self):
        self._last_offset_factor = self._offset_factor
        selected_index = self._l2i(self._local_selection)
        scroll_offset = self._offset * self._offset_factor
        for index in xrange(len(self._object_attribs)):
            self.animated_widgets[index].width = self._object_attribs[index][WIDTH]
            self.animated_widgets[index].height = self._object_attribs[index][HEIGHT]
            self.animated_widgets[index].opacity = self._object_attribs[index][OPACITY]
            self.animated_widgets[index].fg_color = (128, 128, 128, 255)
            if self._orientation == LEFT_RIGHT:
                self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                                  scroll_offset
                self.animated_widgets[index].y = self._object_attribs[index][Y]
            elif self._orientation == TOP_BOTTOM:
                self.animated_widgets[index].x = self._object_attribs[index][X]
                self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                                  scroll_offset
            self.animated_widgets[index].z = self._object_attribs[index][Z]
            if self._object_attribs[index][PIPELINE]:
                self._object_attribs[index][PIPELINE].set_state(gst.STATE_PAUSED)

    # map the local selection X/Y-coordinates to global ones
    def _l2g(self, local_coord):
        global_coord = [-1, -1]

        if self._orientation == LEFT_RIGHT:
            if self._scroll_mode == PAGE_WISE:
                global_coord[X] = local_coord[X] + \
                                  self._offset_factor * \
                                  self._columns
            elif self._scroll_mode == ELEMENT_WISE:
                global_coord[X] = local_coord[X] + self._offset_factor
            global_coord[Y] = local_coord[Y]
        elif self._orientation == TOP_BOTTOM:
            global_coord[X] = local_coord[X]
            if self._scroll_mode == PAGE_WISE:
                global_coord[Y] = local_coord[Y] + \
                                  self._offset_factor * \
                                  self._rows
            elif self._scroll_mode == ELEMENT_WISE:
                global_coord[Y] = local_coord[Y] + self._offset_factor
        return global_coord

    # map the global selection X/Y-coordinates to local ones
    def _g2l(self, global_coord):
        local_coord = [-1, -1]

        if self._orientation == LEFT_RIGHT:
            if self._scroll_mode == PAGE_WISE:
                local_coord[X] = global_coord[X] - \
                                 self._offset_factor * \
                                 self._columns
            elif self._scroll_mode == ELEMENT_WISE:
                local_coord[X] = global_coord[X] - self._offset_factor
            local_coord[Y] = global_coord[Y]
        elif self._orientation == TOP_BOTTOM:
            local_coord[X] = global_coord[X]
            if self._scroll_mode == PAGE_WISE:
                local_coord[Y] = global_coord[Y] - \
                                 self._offset_factor * \
                                 self._rows
            elif self._scroll_mode == ELEMENT_WISE:
                local_coord[Y] = global_coord[Y] - self._offset_factor
        return local_coord

    # map the global selection X/Y-coordinates to the linear index
    def _g2i(self, global_coord):
        if self._orientation == LEFT_RIGHT:
            return global_coord[X] * self._rows + global_coord[Y]
        elif self._orientation == TOP_BOTTOM:
            return global_coord[Y] * self._columns + global_coord[X]

    # map the local selection X/Y-coordinates to the linear index
    def _l2i(self, local_coord):
        global_coord = self._l2g(local_coord)
        return self._g2i(global_coord)

    # map a linear index to a global selection X/Y-coordinate
    def _i2g(self, index):
        global_coord = [-1, -1]

        if self._orientation == LEFT_RIGHT:
            global_coord[X] = index / self._rows
            global_coord[Y] = index - global_coord[X] * self._rows
        elif self._orientation == TOP_BOTTOM:
            global_coord[Y] = index / self._columns
            global_coord[X] = index - global_coord[Y] * self._columns
        return global_coord

    # map a linear index to a local selection X/Y-coordinate
    def _i2l(self, index):
        return self._g2l(self._i2g(index))

    # check if the global coordinate is still within the actual list of widgets
    def _within_index(self, global_coord):
        index = self._g2i(global_coord)
        if index >= 0 and index <= len(self._object_attribs) - 1:
            return True
        else:
            return False

    # check if the global coordinate is still within the currently visible page
    def _witin_page(self, global_coord):
        page = self._rows * self._columns
        if self._scroll_mode == PAGE_WISE:
            offset = self._offset_factor * page
        elif self._scroll_mode == ELEMENT_WISE:
            if self._orientation == TOP_BOTTOM:
                offset = self._offset_factor * self._columns
            elif self._orientation == LEFT_RIGHT:
                offset = self._offset_factor * self._rows
        index = self._g2i(global_coord)
        if index - offset >= 0 and index - offset < page:
            if self._orientation == LEFT_RIGHT and \
               global_coord[Y] >= 0 and \
               global_coord[Y] < self._rows:
                return True
            elif self._orientation == TOP_BOTTOM and \
                 global_coord[X] >= 0 and \
                 global_coord[X] < self._columns:
                return True
            else:
                return False
        else:
            return False

    # get the currently selected number of columns displayed per page
    def columns__get(self):
        return self._columns

    # set the number of columns per page to display, if you pass a value equal
    # to current one or a value smaller than 1 this call has no effect
    # otherwise the made setting will affect the GridView-widget right away
    # don't call this method in a fequent fashion (e.g. multiple times a second)
    # remember that you can set the number of columns to use at widget creation
    # time also
    def columns__set(self, columns):
        if columns == self._columns or columns < 1:
            return

        self._columns = columns
        self._widget_width = self._grid_width / self._columns

        # set offset used for page- or element-wise scrolling
        if self._scroll_mode == ELEMENT_WISE and self._orientation == LEFT_RIGHT:
            self._offset = self._widget_width

        if self._scroll_mode == PAGE_WISE:
            self._last_offset_factor = self._offset_factor
            self._offset_factor = self._selection_index / \
                                  (self._columns * \
                                   self._rows)
        elif self._scroll_mode == ELEMENT_WISE and \
             self._orientation == TOP_BOTTOM:
            self._last_offset_factor = self._offset_factor
            self._offset_factor = self._selection_index / self._columns

        self._local_selection = self._i2l(self._selection_index)
        self._global_selection = self._i2g(self._selection_index)
        self._i2l(self._selection_index)

        self.layout()
        self._map_attribs_to_widgets()
        self.update()

    # get the currently selected number of rows displayed per page
    def rows__get(self):
        return self._rows

    # set the number of rows per page to display, if you pass a value equal to
    # current one or a value smaller than 1 this call has no effect
    # otherwise the made setting will affect the GridView-widget right away
    # don't call this method in a fequent fashion (e.g. multiple times a second)
    # remember that you can set the number of rows to use at widget creation
    # time also
    def rows__set(self, rows):
        if rows == self._rows or rows < 1:
            return

        self._rows = rows
        self._widget_height = self._grid_height / self._rows

        # set offset used for page- or element-wise scrolling
        if self._scroll_mode == ELEMENT_WISE and \
           self._orientation == TOP_BOTTOM:
            self._offset = self._widget_height

        if self._scroll_mode == PAGE_WISE:
            self._last_offset_factor = self._offset_factor
            self._offset_factor = self._selection_index / \
                                  (self._columns * \
                                   self._rows)
        elif self._scroll_mode == ELEMENT_WISE and \
             self._orientation == LEFT_RIGHT:
            self._last_offset_factor = self._offset_factor
            self._offset_factor = self._selection_index / self._rows

        self._local_selection = self._i2l(self._selection_index)
        self._global_selection = self._i2g(self._selection_index)
        self._i2l(self._selection_index)

        self.layout()
        self._map_attribs_to_widgets()
        self.update()

    # get the currently selected scroll-mode of the GridView-widget
    # it either return PAGE_WISE or ELEMENT_WISE
    def scroll_mode__get(self):
        return self._scroll_mode

    # set the scroll-mode of the GridView-widget to either PAGE_WISE or
    # ELEMENT_WISE
    # trying to set the scroll-mode to the same/currently set value has no
    # effect
    # the widget will automatically change its appearance after this call
    # don't call this frequently (e.g. multiple times per second)
    def scroll_mode__set(self, scroll_mode):
        if scroll_mode == self._scroll_mode:
            return

        self._scroll_mode = scroll_mode
        self._update_offset()
        self._update_offset_factor()
        self._map_attribs_to_widgets()

        self._local_selection = self._i2l(self._selection_index)
        self._global_selection = self._i2g(self._selection_index)
        self._i2l(self._selection_index)

        self.layout()
        self._map_attribs_to_widgets()
        self.update()

    # get the currently selected orientation of the GridView-widget
    # it either return TOP_BOTTOM or LEFT_RIGHT
    def orientation__get (self):
        return self._orientation

    # change the orientation of the GridView-widget to TOP_BOTTOM or LEFT_RIGHT
    # trying to set the orientation to the same/currently set value has no
    # effect
    # the widget will automatically change its appearance after this call
    # don't call this frequently (e.g. multiple times per second)
    def orientation__set(self, orientation):
        if orientation == self._orientation:
            return

        if orientation == TOP_BOTTOM:
            self._grid_width  = self._width
            self._grid_height = self._height - 2 * self._background_mask_size
        elif orientation == LEFT_RIGHT:
            self._grid_width  = self._width  - 2 * self._background_mask_size
            self._grid_height = self._height

        self._widget_width = self._grid_width / self._columns
        self._widget_height = self._grid_height / self._rows

        self._orientation = orientation
        self._update_offset()
        self._update_offset_factor()
        self._map_attribs_to_widgets()

        self._local_selection = self._i2l(self._selection_index)
        self._global_selection = self._i2g(self._selection_index)
        self._i2l(self._selection_index)

        self.layout()
        self._map_attribs_to_widgets()
        self.update()

    # insert an element or group (composed widget) in the GridView
    # for performance-considerations the GridView methods layout(), refresh()
    # and update() are meant to be called by the API-user and not implicitly
    # every time a new widget is inserted into the widget
    def insert(self, index, widget, pipeline = None):
        self.add(widget)
        animated_widget = implicit.AnimatedObject(widget)
        animated_widget.setup_next_animations(duration = self._duration,
                                             transformation = self._transformation)
        animated_widget.opacity = 255
        animated_widget.visible = True
        animated_widget.fg_color = (255, 255, 255, 255)
        animated_widget.bg_color = (0, 0, 0, 0)
        self.animated_widgets.insert(index, animated_widget)
        self.widgets.insert(index, widget)
        attribs = [self.position[X] + 0.2,
                   self.position[Y] + 0.2,
                   self.position[Z] + 0.1,
                   self._widget_width,
                   self._widget_height,
                   255,
                   pipeline]
        self._object_attribs.insert(index, attribs)

    # insert an element or group (composed widget) at the and of the list
    # displayed in the GridView
    # for performance-considerations the GridView methods layout(), refresh()
    # and update() are meant to be called by the API-user and not implicitly
    # every time a new widget is appended to the widget
    def append(self, widget, pipeline = None):
        self.insert(len(self._object_attribs), widget, pipeline)

    # remove an widget by index from the GridView
    # for performance-considerations the GridView methods layout(), refresh()
    # and update() are meant to be called by the API-user and not implicitly
    # every time an widget is removed from the widget
    def pop(self, index):
        self.animated_widgets.pop(index)
        self._object_attribs.pop(index)
        drawable = self.widgets.pop(index)
        Group.remove(drawable)

    def remove(self, widget):
        try:
            # if the widget is part of the List
            self.pop(self.widgets.index(widget))
        except:
            # if the widget is just part of the Group
            Group.remove(self, widget)

    # the bread and butter method of the grid-view widget, it causes to layout
    # the positions, sizes and opacities of all currently contained widgets
    # taking into account the set grid-size, orientation and scroll-mode
    # under normal conditions this does not need to be called directly by the
    # API user, only if you want to append/insert/remove any number of widgets
    # this needs to be called, it is not called implicitly on purpose because it
    # would render the adding or removing of many elements to become a very
    # costly operation runtime-wise
    def layout(self):
        current_row = 0
        current_column = 0
        page_offset_x = 0
        page_offset_y = 0
        if len(self._object_attribs) > 0:
            for index in xrange(len(self._object_attribs)):
                if self._orientation == TOP_BOTTOM:
                    self._object_attribs[index][X] = self.position[X] + \
                                                     page_offset_x * \
                                                     self._grid_width + \
                                                     current_column * \
                                                     self._widget_width
                    self._object_attribs[index][Y] = self.position[Y] + \
                                                     self._background_mask_size + \
                                                     page_offset_y * \
                                                     self._grid_height + \
                                                     current_row * \
                                                     self._widget_height
                elif self._orientation == LEFT_RIGHT:
                    self._object_attribs[index][X] = self.position[X] + \
                                                     self._background_mask_size + \
                                                     page_offset_x * \
                                                     self._grid_width + \
                                                     current_column * \
                                                     self._widget_width
                    self._object_attribs[index][Y] = self.position[Y] + \
                                                     page_offset_y * \
                                                     self._grid_height + \
                                                     current_row * \
                                                     self._widget_height
                self._object_attribs[index][Z] = self.position[Z] + 0.1
                self._object_attribs[index][OPACITY] = 255
                self._object_attribs[index][WIDTH] = self._widget_width
                self._object_attribs[index][HEIGHT] = self._widget_height

                if self._orientation == LEFT_RIGHT:
                    if current_row < self._rows - 1:
                        current_row += 1
                    else:
                        current_row = 0
                        if current_column < self._columns - 1:
                            current_column += 1
                        else:
                            current_column = 0
                            page_offset_x += 1
                elif self._orientation == TOP_BOTTOM:
                    if current_column < self._columns - 1:
                        current_column += 1
                    else:
                        current_column = 0
                        if current_row < self._rows - 1:
                            current_row += 1
                        else:
                            current_row = 0
                            page_offset_y += 1

    # causes the last and newly selected widgets to change their states, this
    # is hardly needed to be called by the widgets API-user
    def update(self):
        if self._last_offset_factor != self._offset_factor:
            self._map_attribs_to_widgets()

        scroll_offset = self._offset * self._offset_factor
        if self._last_selection_index != self._selection_index:
            width_offset = 0.0
            height_offset = 0.0
            scale_factor = 1.0
            index = self._selection_index
            if self._object_attribs[index][PIPELINE]:
                self._object_attribs[index][PIPELINE].set_state(gst.STATE_PLAYING)
                scale_factor = 2.0
                width_offset = self._widget_width / 2.0
                height_offset = self._widget_height / 2.0
            else:
                scale_factor = 1.2
                width_offset = 0.2 * self._widget_width / 2.0
                height_offset = 0.2 * self._widget_height / 2.0
            self.animated_widgets[index].width = scale_factor * \
                                                  self._widget_width
            self.animated_widgets[index].height = scale_factor * \
                                                   self._widget_height
            self.animated_widgets[index].opacity = 255
            self.animated_widgets[index].fg_color = (255, 255, 255, 255)
            if self._orientation == LEFT_RIGHT:
                self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                                  width_offset - scroll_offset
                self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                                  height_offset
            elif self._orientation == TOP_BOTTOM:
                self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                                  width_offset
                self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                                  height_offset - scroll_offset
            self.animated_widgets[index].z = 10.0

            index = self._last_selection_index
            self.animated_widgets[index].width = self._object_attribs[index][WIDTH]
            self.animated_widgets[index].height = self._object_attribs[index][HEIGHT]
            self.animated_widgets[index].opacity = self._object_attribs[index][OPACITY]
            self.animated_widgets[index].fg_color = (128, 128, 128, 255)
            if self._orientation == LEFT_RIGHT:
                self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                                  scroll_offset
                self.animated_widgets[index].y = self._object_attribs[index][Y]
            elif self._orientation == TOP_BOTTOM:
                self.animated_widgets[index].x = self._object_attribs[index][X]
                self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                                  scroll_offset
            self.animated_widgets[index].z = self._object_attribs[index][Z]
            if self._object_attribs[index][PIPELINE]:
                self._object_attribs[index][PIPELINE].set_state(gst.STATE_PAUSED)

    # causes all widgets to refresh their new animation-target values, this is
    # usually only needed if you want to append/insert/remove a large number
    # of widgets and only update the grid-view-widget state afterwards
    def refresh(self):
        self._map_attribs_to_widgets()

    # select the next widget relative to the currently selected widget if
    # there is one (obeying the linear sorting order of all widgets), if there
    # is no next widget (the currently selected widget is the last one) this
    # method does nothing
    def next(self):
        if self._selection_index < len(self._object_attribs) - 1:
            self._last_selection_index = self._selection_index
            self._selection_index += 1
            self._global_selection = self._i2g(self._selection_index)
            self._local_selection = self._i2l(self._selection_index)
            self._witin_page(self._i2g(self._selection_index))
            if not self._witin_page(self._i2g(self._selection_index)):
                self._last_offset_factor = self._offset_factor
                self._offset_factor += 1
                self._local_selection = self._i2l(self._selection_index)
            self.update()

    # select the previous widget relative to the currently selected widget if
    # there is one (obeying the linear sorting order of all widgets), if there
    # is no previous widget (the currently selected widget is the first one) this
    # method does nothing
    def prev(self):
        if self._selection_index > 0:
            self._last_selection_index = self._selection_index
            self._selection_index -= 1
            self._global_selection = self._i2g(self._selection_index)
            self._local_selection = self._i2l(self._selection_index)
            if not self._witin_page(self._i2g(self._selection_index)):
                self._last_offset_factor = self._offset_factor
                self._offset_factor -= 1
                self._local_selection = self._i2l(self._selection_index)
            self.update()

    # select the widget above the currently selected widget if there is one, if
    # there is nothing this method does nothing
    def up(self):
        if self._global_selection[Y] > 0:
            self._last_selection_index = self._selection_index
            self._global_selection[Y] -= 1

        if not self._witin_page(self._global_selection):
            self._last_offset_factor = self._offset_factor
            self._offset_factor -= 1

        self._local_selection = self._g2l(self._global_selection)
        self._selection_index = self._g2i(self._global_selection)

        self.update()

    # select the widget below the currently selected widget if there is one, if
    # there is nothing this method does nothing
    def down(self):
        if not self._within_index([self._global_selection[X],
                                   self._global_selection[Y] + 1]):
            return

        if self._orientation == LEFT_RIGHT and not \
           self._witin_page([self._global_selection[X],
                             self._global_selection[Y] + 1]):
            return

        self._last_selection_index = self._selection_index
        self._global_selection[Y] += 1

        if not self._witin_page(self._global_selection):
            self._last_offset_factor = self._offset_factor
            self._offset_factor += 1

        self._local_selection = self._g2l(self._global_selection)
        self._selection_index = self._g2i(self._global_selection)

        self.update()

    # select the widget to the left of the currently selected widget if there is
    # one, if there is nothing this method does nothing
    def left(self):
        if not self._within_index([self._global_selection[X] - 1,
                                   self._global_selection[Y]]):
            return

        if self._orientation == TOP_BOTTOM and not \
           self._witin_page([self._global_selection[X] - 1,
                             self._global_selection[Y]]):
            return

        self._last_selection_index = self._selection_index
        self._global_selection[X] -= 1

        if not self._witin_page(self._global_selection):
            self._last_offset_factor = self._offset_factor
            self._offset_factor -= 1

        self._local_selection = self._g2l(self._global_selection)
        self._selection_index = self._g2i(self._global_selection)

        self.update()

    # select the widget to the right of the currently selected widget if there is
    # one, if there is nothing this method does nothing
    def right(self):
        if not self._within_index([self._global_selection[X] + 1,
                                   self._global_selection[Y]]):
            return

        if self._orientation == TOP_BOTTOM and not \
           self._witin_page([self._global_selection[X] + 1,
                             self._global_selection[Y]]):
            return

        self._last_selection_index = self._selection_index
        self._global_selection[X] += 1

        if not self._witin_page(self._global_selection):
            self._last_offset_factor = self._offset_factor
            self._offset_factor += 1

        self._local_selection = self._g2l(self._global_selection)
        self._selection_index = self._g2i(self._global_selection)

        self.update()

    # this is used to trigger some "selection-animation"
    def select(self):
        index = self._l2i(self._local_selection)

        offset = self._offset * self._offset_factor
        new_width = 2.0 * self._widget_width
        new_height = 2.0 * self._widget_height
        offset_width = (new_width - self._widget_width) / 2.0
        offset_height = (new_height - self._widget_height) / 2.0

        self.animated_widgets[index].width = new_width
        self.animated_widgets[index].height = new_height
        self.animated_widgets[index].opacity = 0
        if self._orientation == LEFT_RIGHT:
            self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                              offset_width - \
                                              offset
            self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                              offset_height
        elif self._orientation == TOP_BOTTOM:
            self.animated_widgets[index].x = self._object_attribs[index][X] - \
                                              offset_width
            self.animated_widgets[index].y = self._object_attribs[index][Y] - \
                                              offset_height - \
                                              offset
        self.animated_widgets[index].z = 11.0

    def selected_item__get(self):
        return self._selection_index
        
    def selected_item__set(self, index):
        if index >= 0 and index < len(self._object_attribs):
            self._last_selection_index = self._selection_index
            self._selection_index = index
            self._global_selection = self._i2g(self._selection_index)
            self._local_selection = self._i2l(self._selection_index)
            self._witin_page(self._i2g(self._selection_index))
            if not self._witin_page(self._i2g(self._selection_index)):
                self._last_offset_factor = self._offset_factor
                if self._last_selection_index < self._selection_index:
                    self._offset_factor += 1
                else:
                    self._offset_factor -= 1
                self._local_selection = self._i2l(self._selection_index)
            self.update()


if __name__ == "__main__":

    import os, os.path
    import sys
    import random
    import pgm
    import gobject
    import gst
    from pypgmtools.graph.group import Group
    from pypgmtools.graph.text import Text
    from pypgmtools.graph.image import Image
    from pypgmtools.timing.implicit import AnimatedObject

    def create_video(uri, image):
        sink = gst.element_factory_make('pgmimagesink')
        pipeline = gst.element_factory_make('playbin')
        pipeline.set_property('queue-size', 120000000)
        pipeline.set_property('uri', uri)
        pipeline.set_property('video-sink', sink)
        sink.set_property('image', image)
        pipeline.set_state(gst.STATE_READY)
        pipeline.set_state(gst.STATE_PAUSED)
        return pipeline

    def on_delete(viewport, event, shade):
        shade.position = (0.0, 0.0, 0.0)
        gobject.timeout_add(1600, pgm.main_quit)

    def on_key_press(viewport, event, cover, shade):
        if event.type == pgm.KEY_PRESS:
            if event.keyval == pgm.keysyms.q or \
               event.keyval == pgm.keysyms.Escape:
                shade.position = (0.0, 0.0, 0.0)
                gobject.timeout_add(1600, pgm.main_quit)
            elif event.keyval == pgm.keysyms.space or \
                 event.keyval == pgm.keysyms.Return:
                cover.select()
            elif event.keyval == pgm.keysyms.Up:
                cover.up()
            elif event.keyval == pgm.keysyms.Down:
                cover.down()
            elif event.keyval == pgm.keysyms.Left:
                cover.left()
            elif event.keyval == pgm.keysyms.Right:
                cover.right()
            elif event.keyval == pgm.keysyms.n:
                cover.next()
            elif event.keyval == pgm.keysyms.p:
                cover.prev()
            elif event.keyval == pgm.keysyms.a:
                image = Image()
                images_filenames = os.listdir(sys.argv[1])
                i = random.randint(0, len(images_filenames)-1)
                image.set_from_fd(os.open(os.path.join(sys.argv[1],
                                                       images_filenames[i]),
                                  os.O_RDONLY))
                image.opacity  = 255
                image.visible  = True
                image.fg_color = (255, 255, 255, 255)
                image.bg_color = (0, 0, 0, 0)
                image.width    = 1.0
                image.height   = 1.0
                image.position = (0.2, 0.2, 0.1)
                image.layout   = pgm.IMAGE_ZOOMED
                cover.insert(2, image)
                cover.layout()
                cover.refresh()
                cover.update()
            elif event.keyval == pgm.keysyms.r:
                cover.pop(2)
                cover.layout()
                cover.refresh()
                cover.update()
            elif event.keyval == pgm.keysyms.t:
                cover.rows += 1
            elif event.keyval == pgm.keysyms.g:
                cover.rows -= 1
            elif event.keyval == pgm.keysyms.h:
                cover.columns += 1
            elif event.keyval == pgm.keysyms.f:
                cover.columns -= 1
            elif event.keyval == pgm.keysyms.o:
                if cover.orientation == TOP_BOTTOM:
                    cover.orientation = LEFT_RIGHT
                elif cover.orientation == LEFT_RIGHT:
                    cover.orientation = TOP_BOTTOM
            elif event.keyval == pgm.keysyms.s:
                if cover.scroll_mode == PAGE_WISE:
                    cover.scroll_mode = ELEMENT_WISE
                elif cover.scroll_mode == ELEMENT_WISE:
                    cover.scroll_mode = PAGE_WISE

    image_path = ""
    video_path = ""

    if len(sys.argv) < 2:
        print "Usage:"
        print "    python grid_view.py /path/to/images /path/to/videos"
        print "or like:"
        print "    python grid_view.py /path/to/images"
        sys.exit()
    elif len(sys.argv) == 2:
        image_path = sys.argv[1]
    elif len(sys.argv) == 3:
        image_path = sys.argv[1]
        video_path = sys.argv[2]

    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'GridView widget-test'

    # Canvas and image drawable creation
    canvas = pgm.Canvas()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    gl.show()

    bg_image          = Image()
    bg_image.set_from_fd(os.open("examples/pictures/bg.png",
                                 os.O_RDONLY))
    bg_image.opacity  = 255
    bg_image.visible  = True
    bg_image.fg_color = (255, 255, 255, 255)
    bg_image.bg_color = (0, 0, 0, 0)
    bg_image.position = (0.0, 0.0, 0.0)
    bg_image.width    = 4.0
    bg_image.height   = 3.0
    bg_image.layout   = pgm.IMAGE_FILLED

    bg_group = Group(canvas,
                     pgm.DRAWABLE_FAR)
    bg_group.add(bg_image)
    bg_group.opacity = 255
    bg_group.visible = True

    shade_image          = Image()
    shade_image.set_from_fd(os.open("examples/pictures/shade.png",
                                    os.O_RDONLY))
    shade_image.opacity  = 255
    shade_image.visible  = True
    shade_image.fg_color = (255, 255, 255, 255)
    shade_image.bg_color = (0, 0, 0, 0)
    shade_image.position = (0.0, .0, 0.0)
    shade_image.width    = 4.0
    shade_image.height   = 6.0
    shade_image.layout   = pgm.IMAGE_FILLED
    animated_shade_image = AnimatedObject(shade_image)
    animated_shade_image.setup_next_animations(duration = 1500)
    animated_shade_image.position = (0.0, -6.0, 0.0)

    shade_group = Group(canvas, pgm.DRAWABLE_NEAR)
    shade_group.add(shade_image)
    shade_group.opacity = 255
    shade_group.visible = True

    widget = GridView(canvas,
                      pgm.DRAWABLE_MIDDLE,
                      width                 = 4.0,
                      height                = 3.0,
                      columns               = 7,
                      rows                  = 6,
                      duration              = 250,
                      transformation        = implicit.DECELERATE,
                      scroll_mode           = PAGE_WISE,
                      orientation           = LEFT_RIGHT,
                      background_masks      = ["examples/pictures/bg-mask-left.png",
                                               "examples/pictures/bg-mask-right.png"],
                      background_mask_size  = 64.0 / 512.0)

    # insert your own path to images here
    if len(image_path) > 0:
        for image_file in os.listdir(image_path):
            cover = Image()
            cover.set_from_fd(os.open(os.path.join(image_path,
                                                   image_file),
                                      os.O_RDONLY))
            cover.opacity  = 255
            cover.visible  = True
            cover.fg_color = (255, 255, 255, 255)
            cover.bg_color = (0, 0, 0, 0)
            cover.width    = 1.0
            cover.height   = 1.0
            cover.position = (0.2, 0.2, 0.1)
            cover.layout   = pgm.IMAGE_ZOOMED
            widget.insert(-1, cover)

    # insert your own path to videos here
    if len(video_path) > 0:
        for video_file in os.listdir(video_path):
            video_image = Image()
            video_image.opacity  = 255
            video_image.visible  = True
            video_image.fg_color = (255, 255, 255, 255)
            video_image.bg_color = (0, 0, 0, 0)
            video_image.width    = 1.0
            video_image.height   = 1.0
            video_image.position = (0.2, 0.2, 0.1)
            video_image.layout   = pgm.IMAGE_ZOOMED
            video_uri            = "file://" + video_path + "/" + video_file
            pipeline             = create_video(video_uri, video_image)
            widget.insert(-1, video_image, pipeline)

    widget.layout()
    widget.refresh()
    widget.update()
    widget.visible = True

    gl.connect('key-press-event', on_key_press, widget, animated_shade_image)
    gl.connect('delete-event', on_delete, animated_shade_image)
    gobject.timeout_add(15, gl.update)
    pgm.main()
