# Copyright (c) 2014 Intel Corporation

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""Helper functions for test generators."""

from __future__ import print_function, absolute_import
import os
import errno
import functools


def safe_makedirs(dirs):
    """A safe function for creating a directory tree.

    This function wraps os.makedirs, and provides a couple of sanity checks,
    first, if the directory already exists it doesn't try to create it, and
    second if the directory comes into existence between the check and creation
    time (like if another generator creates it) then the exception will be
    caught.

    """
    if not os.path.exists(dirs):
        try:
            os.makedirs(dirs)
        except OSError as e:
            if e.errno == errno.EEXIST:
                pass
            else:
                raise


class lazy_property(object):
    """Decorator for lazy property loading.

    A property decorated with this class will execute it's code the first time
    it's run, but will store the value after that. This is useful for values
    that are either 1) expensive to compute, or 2) need to be computed only
    once and then read multiple times

    """
    def __init__(self, func):
        functools.wraps(func)
        self.__func = func

    def __get__(self, obj, cls):
        if not obj:
            return self
        value = self.__func(obj)
        setattr(obj, self.__func.__name__, value)
        return value
