# -*- coding: UTF-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

"""Todo 
* create more actions
* backgroundcolor SetItemBackgroundColor(item, col)
* console option (make execution gui independent)
* automatic updating of properties
* save file lists
"""
#---Global import

#import
import new, sys

#check wx
from wxcheck import ensure
wx = ensure('2.8','2.6')

#force wxpython2.6 for testing
##import wxversion
##wxversion.select('2.6')
##import wx

#check with other encoding
##wx.SetDefaultPyEncoding('ISO8859-15')

#standard library
import cPickle, webbrowser, os
    
#---Local import

#gui-independent
from core import api, ct
from core.message import FrameReceiver

#gui-dependent
from pyWx import images
import droplet, graphics, imageInspector, paint
from compatible import GCDC, FONT_SIZE
from pyWx.wxGlade import frame
from pyWx import dialogs
from pyWx import plugin
from clipboard import copy_text
from unicoding import exception_to_unicode

WX_ENCODING         = wx.GetDefaultPyEncoding()
COMMAND_PASTE       = \
    _('You can paste it as text into the properties of a new launcher.')

#---theme
def _theme():
    if wx.Platform != '__WXGTK__':
        set_theme('nuovext')
    
def set_theme(name='default'):
    if name == 'nuovext':
        from pyWx.nuovext import Provider
        wx.ArtProvider.Push(Provider())

def _icon(self):
    self.icon = wx.EmptyIcon()
    if sys.platform.startswith('win'):
        image   = images.ICON32GIF
    else:
        image   = images.ICON64
    self.icon.CopyFromBitmap(graphics.bitmap(image))
    self.SetIcon(self.icon)
    
#---Functions
def findWindowById(id):
    return wx.GetApp().GetTopWindow().FindWindowById(id)

class DialogsMixin:
    #---dialogs
    def show_error(self,message):
        return self.show_message(message,style=wx.OK|wx.ICON_ERROR)
            
    def show_execute_dialog(self,result,settings,files=None):
        dlg = dialogs.ExecuteDialog(self,drop=files)
        if settings['overwrite_existing_images_forced']:
            dlg.overwrite_existing_images.Disable()
        if files:
            #store in settings,not result as it will be saved
            settings['paths']   = files
        dlg.import_settings(settings)
        result['cancel'] = dlg.ShowModal() == wx.ID_CANCEL
        if result['cancel']:
            dlg.Destroy()
            return
        #Retrieve settings from dialog
        dlg.export_settings(settings)
        dlg.Destroy()
        
    def show_files_message(self,result,message,title,files):
        dlg     = dialogs.FilesDialog(self,message,title,files)
        x0, y0  = self.GetSize()
        x1, y1  = dlg.GetSize()
        x       = max(x0,x1)
        y       = max(y1,200)
        dlg.SetSize((x,y))
        result['cancel'] = dlg.ShowModal() == wx.ID_CANCEL
    
    def show_message(self,message,title='',style=wx.OK | wx.ICON_EXCLAMATION):
        dlg = wx.MessageDialog(self, 
                message,
                '%(name)s '%ct.INFO +title,
                style,
        )
        answer  = dlg.ShowModal()
        dlg.Destroy()
        return answer
    
    def show_info(self,message,title=''):
        return self.show_message(message,title,style=wx.OK | wx.ICON_INFORMATION)
    
    def show_progress(self,title,parent_max,child_max=1,message=''):
        dlg = dialogs.ProgressDialog(self,title,parent_max,child_max,message)
    
    def show_progress_error(self,result,message,ignore=True):
        message     += '\n\n'+_('See Tools>Show Log for more details.')
        errorDlg    = dialogs.ErrorDialog(self,message,ignore)
        answer      = errorDlg.ShowModal()
        result['stop_for_errors']   = not errorDlg.future_errors.GetValue()
        errorDlg.Destroy()
        if answer == wx.ID_ABORT:
            result['answer']    = _('abort')
        elif answer == wx.ID_FORWARD:
            result['answer']    = _('skip')
        else:
            result['answer']    = _('ignore')

    def show_scrolled_message(self,message,title,**keyw):
        import wx.lib.dialogs
        dlg = wx.lib.dialogs.ScrolledMessageDialog(self, message, title, 
                **keyw) 
        dlg.ShowModal()
        
    #---settings
    def get_setting(self,name):
        return wx.GetApp().settings[name]
    
    def set_setting(self,name,value):
        wx.GetApp().settings[name] = value
        
    #---data
    def load_actionlist_data(self,filename):
        if os.path.exists(filename):
            try:
                return api.open_actionlist(filename)
            except KeyError, details:
                self.show_error(
                    _('Sorry, you need to install the %s action'
                    ' for this action list.')\
                    %exception_to_unicode(details,WX_ENCODING))
                return
            except:
                self.show_error((
                    _('Sorry, the action list seems incompatible with\n')+\
                    _('the current version (%(version)s) of %(name)s.'))%\
                    ct.INFO)
                return
            if data['invalid labels']:
                self.show_message('\n'.join([(
                    _('This action list was made by an older %(name)s version.\n\n')+\
                    _('Sorry, the values of these options will be lost in\n')+\
                    'the current version (%(version)s) of %(name)s.')%ct.INFO,
                    '\n', 
                    '\n'.join(data['invalid labels'])
                    ]))


class Frame(DialogsMixin,dialogs.BrowseMixin,droplet.Mixin,paint.Mixin,
        frame.Frame,FrameReceiver):
    paint_message   = _("click '+' to add actions")
    paint_colour    = images.LOGO_COLOUR
    paint_logo      = images.LOGO
    
    def __init__(self,actionlist,*args,**keyw):
        frame.Frame.__init__(self,*args,**keyw)
        _theme()
        self.EnableBackgroundPainting(self.empty)
        self._menu()
        self._toolBar() 
        self._plugin()
        self.on_menu_file_new()
        _icon(self)
        self._title()
        self._description()
        self._drop()
        self._events()
        self._pubsub()
        if actionlist.endswith(ct.EXTENSION):
            self._open(actionlist)
        
    def _plugin(self):
        plugin.install_frame(self)
        
    def _description(self):
        self.show_description(False)
        
    def _drop(self):
        self.SetAsFileDropTarget(self.tree,self.on_drop)
        

    def _menu(self):
        #export menu
        self.menu_file_export   = self.menu_file_export_actionlist_to_clipboard.GetMenu()
        #file history
        self.menu_file.AppendSeparator()
        self.menu_file_recent   = wx.Menu()
        self.filehistory = wx.FileHistory()
        self.filehistory.UseMenu(self.menu_file_recent)
        self._set_file_history(self.get_setting('file_history'))
        self.Bind(wx.EVT_MENU_RANGE, self.on_menu_file_history, 
            id=wx.ID_FILE1, id2=wx.ID_FILE9)
        self.menu_file.AppendMenu(wx.ID_REFRESH, _("&Recent"), 
            self.menu_file_recent, "")
        #shell
        self.shell = None
        #menu_item (for enabling/disabling)
        edit    = [getattr(self,attr) for attr in dir(self) 
            if attr[:10] == 'menu_edit_']
        edit.remove(self.menu_edit_add)
        self.menu_item  = [
            (self.menu_file,[self.menu_file_new.GetId(),
            self.menu_file_save.GetId(),
                self.menu_file_save_as.GetId(),
                ]),
            (self.menu_edit,[item.GetId() for item in edit]),
            (self.menu_view,(self.menu_view_droplet.GetId(),
                self.menu_view_description.GetId())),
            (self.menu_tools,(self.menu_tools_execute.GetId(),)),
            (self.menu_file_export,
                (self.menu_file_export_actionlist_to_clipboard.GetId(),))
        ]
        self.on_show_droplet(False)
        # Mac  tweaks
        if wx.Platform == "__WXMAC__":
            #todo: about doesn't seem to work: why?!
            app = wx.GetApp()
            #app.SetMacHelpMenuTitleName(_('&Help'))
            app.SetMacAboutMenuItemId(wx.ID_ABOUT)
            app.SetMacExitMenuItemId(wx.ID_EXIT)
            #app.SetMacPreferencesMenuItemId(wx.ID_PREFERENCES)
        
    def _title(self):
        path, filename = os.path.split(self.filename)
        self.SetTitle(ct.FRAME_TITLE%(self.dirty,os.path.splitext(filename)[0]))
        self.frame_statusbar.SetStatusText(path)
        self.SetStatusText(ct.COPYRIGHT)
        
    #---toolBar
    def add_tool(self,bitmap,label,tooltip,method,item=wx.ITEM_NORMAL):
        id      = wx.NewId()
        bitmap  = graphics.bitmap(bitmap,self.tool_bitmap_size,
                    client=wx.ART_TOOLBAR)
        args    = (id, label, bitmap, wx.NullBitmap,item, 
                    tooltip, "")
        tool    = self.frame_toolbar.AddLabelTool(*args)
        self.Bind(wx.EVT_TOOL, method, id=id)
        return tool
        
    def _toolBar(self):
        self.tool_bitmap_size   = (32,32)
        self.frame_toolbar = wx.ToolBar(self, -1,style=wx.TB_FLAT)
        self.SetToolBar(self.frame_toolbar)
        self.frame_toolbar.SetToolBitmapSize(self.tool_bitmap_size)
        self.tools_item = []
        self.add_tool('ART_FILE_OPEN',_("Open"),
            _("Open an action list"),self.on_menu_file_open)
        tools_item  = [
            self.add_tool('ART_EXECUTABLE_FILE',_("Execute"),
                _("Execute the action"),self.on_menu_tools_execute)
        ]
        self.frame_toolbar.AddSeparator()
        self.add_tool('ART_ADD_BOOKMARK',_("Add"),
            _("Add an action"),self.on_menu_edit_add),
        tools_item.extend([
            self.add_tool('ART_DEL_BOOKMARK',_("Remove"),
                _("Remove the selected action"),self.on_menu_edit_remove),
            self.add_tool('ART_GO_UP',_("Up"),
                _("Move the selected action up"),self.on_menu_edit_up),
            self.add_tool('ART_GO_DOWN',_("Down"),
                _("Move the selected action down"),self.on_menu_edit_down),
        ])
        self.frame_toolbar.AddSeparator()
        self.add_tool('ART_FIND',_("Image Inspector"),
            _("Look up exif and iptc tags"),
            self.on_menu_tools_image_inspector)
        self.frame_toolbar.AddSeparator()
        self.toolbar_description = self.add_tool('ART_TIP',_("Description"),
            _("Show description of the action list"),
            self.on_menu_view_description,item=wx.ITEM_CHECK)
        tools_item.extend([self.toolbar_description])
        self.tools_item = [tool.GetId() for tool in tools_item]
        self._menu_toolbar_state = True
        self.frame_toolbar.Realize()
            
    def enable_actions(self, state=True):
        if state != self._menu_toolbar_state:
            for tool in self.tools_item:
                self.frame_toolbar.EnableTool(tool, state)
            for menu, items in self.menu_item:
                for item in items:
                    menu.Enable(item,state) 
            self._menu_toolbar_state = state
            self.tree.Show(state)
            self.empty.Show(not state)
            self.show_description(state and self.get_setting('description'))

#---menu events
    def on_menu_file_new(self, event=None):
        self._set_filename(ct.UNKNOWN)
        self.description.SetValue(ct.ACTION_LIST_DESCRIPTION)
        self._new()
        self.enable_actions(False)
        
    def on_menu_file_open(self, event):
        dlg = wx.FileDialog(self,
            message     = _('Choose an Action List File...'),
            defaultDir  = os.path.dirname(self.filename),
            wildcard    = ct.WILDCARD,
            style       = wx.OPEN,
        )
        if dlg.ShowModal() == wx.ID_OK:
            filename    = dlg.GetPath()
            self._open(filename)
        dlg.Destroy()
        
    def on_menu_file_save(self, event):
        if self.filename == ct.UNKNOWN:
            self.on_menu_file_save_as()
        else:
            self._save()
        
    def on_menu_file_save_as(self, event=None):
        dlg = wx.FileDialog(self, 
            message     = _('Save Action List As...'),
            defaultDir  = os.path.dirname(self.filename),
            wildcard    = ct.WILDCARD,
            style       = wx.SAVE,
        )
        if dlg.ShowModal() == wx.ID_OK:
            path        = dlg.GetPath()
            if dlg.GetFilterIndex() == 0 and not os.path.splitext(path)[1]:
                path    += ct.EXTENSION
            self._save(path)
        dlg.Destroy()
        
    def on_menu_file_export_actionlist_to_clipboard(self, event):
        if self.is_save_not_ok(): return
        copy_text(ct.COMMAND['DROP']%self.filename)
        self.show_info(' '.join([
         _('The droplet command for this actionlist was copied to the clipboard.'),
         COMMAND_PASTE]))

    def on_menu_file_export_recent_to_clipboard(self, event):
        if self.is_save_not_ok(): return
        copy_text(ct.COMMAND['RECENT'])
        self.show_info(' '.join([
         _('The droplet command for this actionlist was copied to the clipboard.'),
         COMMAND_PASTE]))

    def on_menu_file_export_inspector_to_clipboard(self, event):
        if self.is_save_not_ok(): return
        copy_text(ct.COMMAND['INSPECTOR'])
        self.show_info(' '.join([
         _('The droplet command for the image inspector was copied to the clipboard.'),
         COMMAND_PASTE]))

    def on_menu_file_quit(self, event):
        self.on_close()
        
    def on_menu_file_history(self,event):
        if self.is_save_not_ok(): return
        # get the file based on the menu ID
        filenum     = event.GetId() - wx.ID_FILE1
        filename    = self.filehistory.GetHistoryFile(filenum)
        self._open(filename)
        
    def on_menu_edit_add(self, event): 
        dlg = dialogs.ActionDialog(self,api.ACTIONS,-1,size=(400, 400),
            title="%(name)s "%ct.INFO+_("actions"), )
        if dlg.ShowModal() == wx.ID_OK:
            self.set_dirty(True)
            label   = dlg.GetStringSelection()
            self.tree.append_form_by_label_to_selected(label)
            self.enable_actions(True)
        dlg.Destroy()
        
    def on_menu_edit_remove(self, event): 
        if self.tree.remove_selected_form():
            if self.IsEmpty():
                self.enable_actions(False)
                self.set_dirty(False)
            else:
                self.set_dirty(True)
        
    def on_menu_edit_up(self, event):
        self.set_dirty(True)
        self.tree.move_selected_form_up()
        
    def on_menu_edit_down(self, event):
        self.set_dirty(True)
        self.tree.move_selected_form_down()
        
    def on_menu_edit_enable(self,event):
        self.tree.enable_selected_form(True)
        
    def on_menu_edit_disable(self,event):
        self.tree.enable_selected_form(False)
        
    def on_menu_view_droplet(self,event):
        self.show_droplet(event.IsChecked())
        
    def droplet_label_format(self,x):
        return x[:11].replace('_',' ')

    def on_menu_view_description(self, event):
        self.show_description(event.IsChecked())
            
    def on_menu_view_expand_all(self, event): 
        self.tree.expand_forms()

    def on_menu_view_collapse_all(self, event): 
        self.tree.collapse_forms()

    def on_menu_tools_execute(self, event):
        action_list = self.tree.export_forms()
        api.apply_actions(action_list,wx.GetApp().settings)
        
    def on_menu_tools_image_inspector(self, event):
        frame   = imageInspector.SingleFrame(self,
            size    = imageInspector.SIZE,
            icon    = self.icon)
        frame.Show()
        
    def on_menu_tools_show_log(self,event):
        if os.path.exists(ct.ERROR_LOG_PATH):
            log_file    = open(ct.ERROR_LOG_PATH)
            msg         = log_file.read()
            log_file.close()
        else:
            msg         = _ ('Nothing has been logged yet.')
        self.show_scrolled_message(msg,'%s - %s'%(_('Log'),ct.ERROR_LOG_PATH))
        
    def on_menu_tools_python_shell(self,event):
        import shell
        self.tree.close_popup()
        if self.shell is None:
            self.shell = shell.Frame(self,
                title   = '%(name)s '%ct.INFO+_('Shell'),
                intro   = '%(name)s '%ct.INFO,
                values  = {
                    '%s_%s'%(ct.TITLE,_('application')) : wx.GetApp(),
                    '%s_%s'%(ct.TITLE,_('frame'))       : self,
                    '%s_%s'%(ct.TITLE,_('actions'))     : self.tree.export_forms,
                    'unlock'                            : _,
                    },
                icon    = self.icon,
            )
        self.shell.Show(event.IsChecked())
        
    def on_menu_help_website(self, event): # wxGlade: Frame.<event_handler>
        webbrowser.open('http://photobatch.stani.be')

    def on_menu_help_documentation(self, event): 
        webbrowser.open('http://photobatch.wikidot.com/')
        
    def on_menu_help_forum(self, event):
        webbrowser.open('http://photobatch.wikidot.com/forum:start')

    def on_menu_help_translate(self, event): 
        webbrowser.open('https://translations.launchpad.net/phatch/trunk/+pots/phatch')
        
    def on_menu_help_bug(self, event):
        webbrowser.open('https://bugs.launchpad.net/phatch')
        
    def on_menu_help_plugin(self, event): 
        help_path   = self.get_setting("PHATCH_DOCS_PATH")
        webbrowser.open(os.path.join(help_path,'phatch_dev','index.html'))
        dlg = dialogs.WritePluginDialog(self,'\n'.join([
            _('A html tutorial will open in your internet browser.'),
            '',
            _('You only need to know PIL to write a plugin for Phatch,')+' ',
            _('as Phatch will generate the user interface automatically.'),
            _('Study the action plugins in:')+' '+ct.ACTIONS_PATH,
            '',
            _('If you want to contribute a plugin for Phatch,')+' '+\
            "%s: %s."%(_('please email'),ct.CONTACT)
        ]))
        dlg.ShowModal()
        #settings in mixin because now app bas, also config_path
    def on_menu_help_about(self, event): 
        import about
        from data.info import all_credits
        dlg = about.Dialog(self,
            title                   = '%(version)s'%ct.INFO,
            logo                    = graphics.bitmap(images.LOGO),
            description             = _('PHoto bATCH Processor & Renamer'),
            website                 = ct.INFO['url'],
            credits                 = all_credits(),
            license                 = ct.LICENSE,
        )
        dlg.ShowModal()
        dlg.Destroy()
        
    #---helper
    def _new(self):
        self.set_dirty(False)
        self.tree.delete_all_forms()
        
    def _open(self, filename):
        self._new()
        data = self.load_actionlist_data(filename)
        if data:
            self.description.SetValue(data['description'])
            if self.tree.append_forms(data['actions']):
                self.enable_actions(True)
            # add it to the history
            self.filehistory.AddFileToHistory(filename)
            self._set_filename(filename)
        
    def _save(self, filename=None):
        self.set_dirty(False)
        if filename:
            self._set_filename(filename)
        data    = {
            'description'   : self.description.GetValue(),
            'actions'       : self.tree.export_forms(),
        }
        api.save_actionlist(self.filename,data)
        if filename:
            # add it to the history
            self.filehistory.AddFileToHistory(filename)
        
    def _set_filename(self,filename):
        self.filename = filename
        self.set_dirty(False)
        
    #---view show/hide
    def show_description(self,checked):
        self.set_setting('description',checked)
        self.description.Show(checked)
        self.menu_view.Check(self.menu_view_description.GetId(),checked)
        self.frame_toolbar.ToggleTool(self.toolbar_description.GetId(),checked)
        self.Layout()
        
    def show_droplet(self,checked):
        if checked:
            if api.check_actionlist(self.tree.export_forms(),
                    wx.GetApp().settings):
                self.droplet = droplet.Frame(self,
                    title       = _("Drag & Drop")+' - '+ct.TITLE,
                    bitmap      = graphics.bitmap(images.DROPLET),
                    method      = self.on_drop,
                    label       = self.droplet_label_format(os.path.splitext(
                                    os.path.basename(self.filename))[0]),
                    label_colour= wx.Colour(156,166,165),
                    label_pos   = (73, 40),
                    label_angle = 360-20,
                    pos         = self.GetPosition(),
                    auto        = True,
                    OnShow      = self.on_show_droplet,
                    tooltip     = _(
                    "Drop any files and/or folders on this Phatch droplet\n"
                    "to batch process them.\n"
                    "Right-click or double-click to switch to normal view.")
                )
            else:
                wx.CallAfter(self.on_show_droplet,False)
        else:
            if self.droplet:
                self.droplet.show(False)
            
    def on_show_droplet(self,bool):
        self.menu_view.Check(self.menu_view_droplet.GetId(),bool)
        if not bool: self.droplet = None #don't keep a reference
        
    #---checks
    def is_last_action_valid(self,last_action):
        if last_action.is_valid_last_action():
            return True
        self.append_save_action()
        return False
        
    def append_save_action(self):
        self.show_message(
            ct.SAVE_ACTION_NEEDED+" "+
            _("Phatch will add one for you, please check its settings."))
        self.tree.append_form_by_label_to_last('Save')

    #---other events
    def _events(self):
        #wxPython events
        self.Bind(wx.EVT_CLOSE,self.on_close,self)
        self.Bind(wx.EVT_SIZE,self.on_size,self)
        self.Bind(wx.EVT_MENU_HIGHLIGHT_ALL,self.on_menu_tool_enter)
        self.Bind(wx.EVT_TOOL_ENTER,self.on_menu_tool_enter)
        self.tree.Bind(wx.EVT_TREE_END_DRAG,self.on_tree_end_drag,self.tree)
        self.tree.Bind(wx.EVT_CONTEXT_MENU, self.on_context_menu,self.tree)
        
    def on_drop(self,filenames,x,y):
        api.apply_actions(self.tree.export_forms(),wx.GetApp().settings,
            paths=filenames,drop=True)
        
    def on_menu_tool_enter(self,event):
        self.tree.close_popup()
        event.Skip()
        
    def on_tree_end_drag(self,event):
        event.Skip()
        wx.CallAfter(self.set_dirty,True)
        
    def on_context_menu(self,event):
        if self.tree.is_form_selected():
            self.tree.PopupMenu(self.menu_edit)
        
    def is_save_not_ok(self):
        if self.dirty:
            answer = self.show_message(
                _('Save last changes to')+
                    '\n"%s"?'%self.filename,
                style   = wx.YES_NO|wx.CANCEL|wx.ICON_EXCLAMATION,
            )
            if answer == wx.ID_CANCEL:
                return True
            if answer == wx.ID_YES:
                self.on_menu_file_save(None)
        return False
                
    def on_close(self,event=None):
        if self.is_save_not_ok(): return
        self.Hide() 
        wx.GetApp()._saveSettings()
        #Destroy everything
        self.Destroy()
        
    def on_size(self,event):
        event.Skip()
        if self.IsEmpty():
            self.empty.Refresh() 
        else:
            self.tree.resize_popup() 
            
    def IsEmpty(self):
        return not self.tree.has_forms()
        
    #---file history
    def _get_file_history(self):
        result  = [] 
        for index in range(self.filehistory.GetCount()):
            filename = self.filehistory.GetHistoryFile(index)
            if filename.strip():
                result.append(filename)
        return result
    
    def _set_file_history(self,files):
        for filename in files:
            if os.path.exists(filename):
                self.filehistory.AddFileToHistory(filename)
 
    #---settings
    def set_dirty(self,value):
        self.dirty  = ('','*')[value]
        self._title()
        
    #---droplet
    def get_droplet_folder(self):
        folder = self.show_dir_dialog(
            defaultPath = self.get_setting('droplet_path'),
            message=_('Choose the folder for the droplet'))
        if folder:
            self.set_setting('droplet_path',folder)
        return folder
    
    def menu_file_export_droplet(self,method,*args,**keyw):
        folder      = self.get_droplet_folder()
        if folder is None:
            return
        try:
            create_phatch_droplet(self.filename,folder)
            self.show_info(_('Phatch successfully created the droplet.'))
        except Exception, details:
            reason  = exception_to_unicode(details,WX_ENCODING)
            self.show_error(_('Phatch could not create the droplet:')+'\n\n'\
                +reason)
        
    def install_menu_item(self,menu,name,label,method,tooltip="",
            style=wx.ITEM_NORMAL):
        #item
        item        = wx.MenuItem(menu, -1,label, tooltip, style)
        setattr(self,name,item)
        menu.InsertItem(0,item) 
        #method
        method_name = 'on_'+name
        method      = new.instancemethod(method,self, self.__class__)
        setattr(self, method_name, method)
        #bind item & method
        self.Bind(wx.EVT_MENU, method,item)
        #return id
        return item.GetId()

#---Image Inspector
class ImageInspectorApp(wx.App):
    def __init__(self,paths,*args,**keyw):
        self.paths   = paths
        super(ImageInspectorApp,self).__init__(*args,**keyw)
        
    def OnInit(self):
        wx.InitAllImageHandlers()
        _theme()
        if not self.paths: self.paths = ['']
        for path in self.paths:
            frame   = imageInspector.SingleFrame(None,filename=path,
                    size=imageInspector.SIZE)
            _icon(frame)
            frame.Show()
        self.SetTopWindow(frame)
        return 1
    
def inspect(paths):
    app = ImageInspectorApp(paths,0)
    app.MainLoop()
    
#---Droplet
class DropletFrame(DialogsMixin,wx.Frame,FrameReceiver):
    def __init__(self,actionlist,paths,*args,**keyw):
        wx.Frame.__init__(self,*args,**keyw)
        self._pubsub()
        data  = self.load_actionlist_data(actionlist)
        if data:
            wx.CallAfter(self.execute,data['actions'],paths=paths)
        else:
            sys.exit()
    
    def execute(self,actionlist,paths):
        api.apply_actions(actionlist,wx.GetApp().settings,paths=paths,
            drop=True)
        self.Destroy()

class DropletApp(wx.App):
    def __init__(self,actionlist,paths,settings,*args,**keyw):
        self.actionlist = actionlist
        self.paths      = paths
        self._loadSettings(settings)
        super(DropletApp,self).__init__(*args,**keyw)
        
    def OnInit(self):
        wx.InitAllImageHandlers()
        #do all application initialisation
        self.init()
        api.init()
        #check for action list
        if self.actionlist == 'recent':
            self.actionlist = self.get_action_list(
                self.settings['file_history'])
        if self.actionlist is None:
            return 0
        #create frame
        frame = DropletFrame(self.actionlist,self.paths,None, -1, ct.TITLE)
        frame.Hide()
        self.SetTopWindow(frame)
        return 1
    
    def get_action_list_files(self):
        global_library  = glob.glob(os.path.join(
            self.settings["PHATCH_ACTIONLISTS_PATH"],'*'+ct.EXTENSION))
        local_library   = glob.glob(os.path.join(
            ct.USER_ACTIONLISTS_PATH,'*'+ct.EXTENSION))
        return global_library+local_library
    
    def get_action_list(self,file_list):
        if not file_list:
            file_list   = self.get_action_list_files()
        d   = {}
        for f in file_list:
            d[api.title(f)] = f
        actionlists = d.keys()
        actionlists.sort() 
        dlg = wx.SingleChoiceDialog(
                None, _('Select action list'), ct.TITLE,
                actionlists,
                wx.CHOICEDLG_STYLE
                )
        if dlg.ShowModal() == wx.ID_OK:
            actionlist = d[dlg.GetStringSelection()]
        else:
            actionlist = None
        dlg.Destroy()
        return actionlist
        

    def init(self):
        global cPickle, gettext, glob, os
        import cPickle, gettext, glob, os
        global api
        from core import api

        
    def _loadSettings(self,settings):
        self.settings   = settings
        if os.path.exists(ct.SETTINGS_PATH):
            f               = open(ct.SETTINGS_PATH,'rb')
            self.settings.update(cPickle.load(f))
            f.close()
            
    def _saveSettings(self):
        f   = open(ct.SETTINGS_PATH,'wb')
        cPickle.dump(self.settings,f)
        f.close()
        


def drop(actionlist,paths,settings):
    app         = DropletApp(actionlist,paths,settings,0)
    app.MainLoop()
    

#---Application
class App(DropletApp):
    def __init__(self,settings,actionlist,*args,**keyw):
        self._loadSettings(settings)
        self.filename = actionlist
        super(DropletApp,self).__init__(*args,**keyw)

    def OnInit(self):
        wx.InitAllImageHandlers()
        #frame
        self.splash = self._splash() 
        self.splash.CentreOnScreen()
        self.SetTopWindow(self.splash)
        self.splash.Show()
        wx.CallAfter(self.show_frame)
        return 1
    
    def _splash(self):
        return droplet.Frame(None,
            title       = ct.TITLE,
            bitmap      = graphics.bitmap(images.DROPLET),
            splash      = True,
        )

    def show_frame(self):
        #do all application initialisation
        self.init()
        api.init()
        #create frame
        frame = Frame(self.filename,None, -1, ct.TITLE)
        frame.CentreOnScreen()
        frame.Show()
        self.SetTopWindow(frame)
        #delete splash
        self.splash.Hide()
        self.splash.Destroy()
        del self.splash
        
    def init(self):
        super(App,self).init()
        global ICON_SIZE
        from treeEdit import ICON_SIZE
        
    def _saveSettings(self):
        self.settings['file_history'] = self.GetTopWindow()._get_file_history()
        super(App,self)._saveSettings()

def main(settings,actionlist):
    app         = App(settings,actionlist,0)
    app.MainLoop()
    


if __name__ == "__main__":
    main()