#==========================================================
# Database --
#
#   provides a layer of abstraction for common DB operations
#
#==========================================================
#
namespace eval Database {}


#----------------------------------------------------------
# getPrefObjList --
#
#   returns objects with respect to user preferences
#
# Arguments:
#   obj_    an object, either from PG or PGA
#   dbh_    an optional database handle
#   full_   optional, 1 to return all columns not just name
#   schema_ optional, 1 to return dot separating schema name
#
# Returns:
#   olist   either a list of names, or a list of lists
#----------------------------------------------------------
#
proc ::Database::getPrefObjList {obj_ {dbh_ ""} {full_ 0} {schema_ 0}} {

    global PgAcVar
    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set pg $PgAcVar(pref,systemtables)
    set pga $PgAcVar(pref,pgaccesstables)

    return [::Database::getObjectsList $obj_ $dbh_ $full_ $schema_ $pg $pga]

}; # end proc ::Database::getPrefObjList


#----------------------------------------------------------
# getObjectsList --
#
#   returns a list of names of items of a particular object
#   type, or a list of all the columns of each item
#
# Arguments:
#   obj_    an object, either from PG or PGA
#   dbh_    an optional database handle
#   full_   optional, 1 to return all columns not just name
#   schema_ optional, 1 to return dot separating schema name
#   pg_     optional, 1 to show PG system objects (tables,views)
#   pga_    optional, 1 to show PGA system objects (tables)
#
# Returns:
#   olist   either a list of names, or a list of lists
#----------------------------------------------------------
#
proc ::Database::getObjectsList {obj_ {dbh_ ""} {full_ 0} {schema_ 0} {pg_ 0} {pga_ 0}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set olist [list]

    if {$obj_ == "Tables"} {
        set olist [::Database::getTablesList $CurrentDB $pg_ $pga_]
    } elseif {$obj_ == "Views"} {
        set olist [::Database::getViewsList $CurrentDB $pg_]
    } elseif {$obj_ == "Functions"} {
        set olist [::Database::getFunctionsList]
    } elseif {$obj_ == "Sequences"} {
        set olist [::Database::getSequencesList]
    } else {
        set sql "
            SELECT *
              FROM pga_$obj_"
        wpg_select $CurrentDB $sql rec {
            if {$full_} {
                set clist [list]
                foreach col $rec(.headers) {
                    lappend clist $rec($col)
                }
                lappend olist $clist
            } else {
                lappend olist $rec([lindex $rec(.headers) 0])
            }
        }
    }

    set nlist [list]

    if {$schema_} {
        set nlist $olist
    } else {
        foreach o $olist {
            set splito [split $o .]
            if {[llength $splito] > 1} {
                set n [lindex [lrange [split $o .] 1 end] 0]
                regsub -all {\"} $n {} on
                lappend nlist $on
            } else {
                lappend nlist $o
            }
        }
    }

    return $nlist

}; # end proc ::Database::getObjectsList


#----------------------------------------------------------
# ::Database::getPermissions --
#
#   retrieve the acl list on a PG object
#
# Arguments:
#   obj_    a PG object to return perms for
#   dbh_    an optional database handle
#
# Returns:
#   acl     a string of permissions
#----------------------------------------------------------
#
proc ::Database::getPermissions {obj_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}

    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    set acl ""
    set sql ""

    if {$V < 7.3} {
        set sql "
            SELECT relacl
              FROM pg_class
             WHERE relname='[string trim $obj_ \"]'"
    } else {
        set sql "
            SELECT relacl
              FROM pg_catalog.pg_attribute A, pg_catalog.pg_class C
             WHERE A.attrelid='$obj_'::regclass
               AND A.attrelid=C.oid"
    }

    set res [pg_exec $CurrentDB $sql]
    if {[pg_result $res -numTuples] > 0} {
        set acl [pg_result $res -getTuple 0]
    } else {
        # default the acl to nothing if there isnt one, just in case
        set acl "=,"
    }
    pg_result $res -clear

    return $acl

}; # ::Database::getPermissions


#----------------------------------------------------------
# getTableIndexes --
#
#   returns a list index names in a table
#
# Arguments:
#   table_  name of a view or table (required)
#   dbh_    the db handle (optional)
#
# Results:
#   list of names of the indexes on the table
#----------------------------------------------------------
#
proc ::Database::getTableIndexes {table_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}


    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    if {$V < 7.3} {

        set sql "
            SELECT relname
              FROM pg_class
             WHERE oid IN (
            SELECT indexrelid
              FROM pg_index I, pg_class C
             WHERE (C.relname='$table_')
               AND (C.oid=I.indrelid))"

    } else {

        set sql "
            SELECT relname
              FROM pg_catalog.pg_class
             WHERE oid IN (
            SELECT indexrelid
              FROM pg_catalog.pg_attribute A, pg_catalog.pg_index I
             WHERE (A.attrelid='$table_'::regclass)
               AND (A.attrelid=I.indrelid))"

    }

    set tilist {}

    if {[catch {
        wpg_select $dbh_ $sql rec {
            lappend tilist [::Database::quoteObject $rec(relname)]
        }
    } gterrmsg]} {
        showError $gterrmsg
    }

    return $tilist

}; # end proc ::Database::getTableIndexes


#----------------------------------------------------------
# getTableInfo --
#
#   returns a list (from an array) of info on a table
#
# Arguments:
#   table_  name of a view or table (required)
#   dbh_    the db handle (optional)
#
# Results:
#   a list of name-value array pairs of table info columns
#----------------------------------------------------------
#
proc ::Database::getTableInfo {table_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}


    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    if {$V < 7.3} {

        set sql "
            SELECT attnum,attname,typname,attlen,attnotnull,atttypmod,
                   usename,usesysid,C.oid,relpages,reltuples,
                   relhaspkey,relhasrules,relacl
              FROM pg_user U, pg_attribute A,
                   pg_type T, pg_class C
             WHERE (C.relname='$table_')
               AND (C.oid=A.attrelid)
               AND (C.relowner=U.usesysid)
               AND (A.atttypid=T.oid)
          ORDER BY A.attnum"

    } else {

        set sql "
            SELECT attnum,attname,typname,attlen,attnotnull,atttypmod,
                   usename,usesysid,C.oid,relpages,reltuples,
                   relhaspkey,relhasrules,relacl
              FROM pg_catalog.pg_user U, pg_catalog.pg_attribute A,
                   pg_catalog.pg_type T, pg_catalog.pg_class C
             WHERE (A.attrelid='$table_'::regclass)
               AND (A.atttypid=T.oid)
               AND (A.attrelid=C.oid)
               AND (C.relowner=U.usesysid)
          ORDER BY A.attnum"

    }

    set tlist {}

    if {[catch {
        wpg_select $dbh_ $sql rec {
            lappend tlist [array get rec]
        }
    } gterrmsg]} {
        showError $gterrmsg
    }

    return $tlist

}; # end proc ::Database::getTableInfo


#----------------------------------------------------------
# getColumnsTypesList --
#
#   returns a list of names of columns and their types
#   in a given view or table
#
# Arguments:
#   table_   name of a view or table (required)
#   dbh_    the db handle (optional)
#
# Results:
#   a list of pairs of column names and types
#----------------------------------------------------------
#
proc ::Database::getColumnsTypesList {table_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}


    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    if {$V < 7.3} {

        set sql "
            SELECT A.attname, count(A.attname), T.typname
              FROM pg_class C, pg_attribute A, pg_type T
             WHERE (C.relname='[string trim $table_ \"]')
               AND (C.oid=A.attrelid)
               AND (A.attnum>0)
               AND (A.atttypid=T.oid)
          GROUP BY A.attname, A.attnum, T.typname
          ORDER BY A.attnum"

    } else {

        set sql "
            SELECT A.attname, count(A.attname), T.typname
              FROM pg_catalog.pg_attribute A, pg_catalog.pg_type T
             WHERE (A.attrelid='[string trim $table_ \"]'::regclass)
               AND (A.attnum>0)
               AND (A.atttypid=T.oid)
          GROUP BY A.attname, A.attnum, T.typname
          ORDER BY A.attnum"

    }

    set ctlist {}

    if {[catch {
        wpg_select $dbh_ $sql rec {
            if {$rec(count)!=0} {
                lappend ctlist [list $rec(attname) $rec(typname)]
            }
        }
    } gterrmsg]} {
        showError $gterrmsg
    }

    return $ctlist

}; # end proc ::Database::getColumnsTypesList


#----------------------------------------------------------
# getColumnsList --
#
#   returns a list of names of columns in a given view or table
#
# Arguments:
#   table_   name of a view or table (required)
#   dbh_    the db handle (optional)
#
# Results:
#   a list of column names
#----------------------------------------------------------
#
proc ::Database::getColumnsList {table_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}


    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    if {$V < 7.3} {

        set sql "
            SELECT A.attname, count(A.attname)
              FROM pg_class C, pg_attribute A
             WHERE (C.relname='[string trim $table_ \"]')
               AND (C.oid=A.attrelid)
               AND (A.attnum>0)
          GROUP BY A.attname, A.attnum
          ORDER BY A.attnum"

    } else {

        set sql "
            SELECT A.attname, count(A.attname)
              FROM pg_catalog.pg_attribute A
             WHERE (A.attrelid='$table_'::regclass)
               AND (A.attnum>0)
          GROUP BY A.attname, A.attnum
          ORDER BY A.attnum"

    }

    set clist {}

    if {[catch {
        wpg_select $dbh_ $sql rec {
            if {$rec(count)!=0} {
                lappend clist $rec(attname)
            }
        }
    } gterrmsg]} {
        showError $gterrmsg
    }

    return $clist

}; # end proc ::Database::getColumnsList


#----------------------------------------------------------
# getViewsList --
#
#   returns a list of views in the currentdb
#
# Arguments:
#    dbh_    optionally supply the db handle
#    pg_     whether or not to show the PG internal views
#
# Results:
#    a list of view names
#----------------------------------------------------------
#
proc ::Database::getViewsList {{dbh_ ""} {pg_ 0}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}

    set sql "
        SELECT C.relname, count(C.relname)
          FROM pg_class C, pg_rewrite R
         WHERE (R.ev_class = C.oid)
           AND (R.ev_type = '1')"

    if {!$pg_} {
        append sql " AND relname !~ '^pg_'"
    }

    append sql " GROUP BY relname"

    set vlist {}

    if {[catch {
        wpg_select $dbh_ $sql rec {
            if {$rec(count)!=0} {
                lappend vlist [::Database::quoteObject $rec(relname)]
            }
        }
    } gterrmsg]} {
        showError $gterrmsg
    }

    return $vlist

}; # end proc ::Database::getViewsList


#----------------------------------------------------------
# getSequencesList --
#
#   returns a list of sequences in the currentdb
#
# Argumens:
#   dbh_    optionally supply the db handle
#
# Returns:
#   a list of sequence names
#----------------------------------------------------------
#
proc ::Database::getSequencesList {{dbh_ ""}} {

    global PgAcVar CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}

    set sql "
        SELECT c.relname
          FROM [::Database::qualifySysTable pg_class] C LEFT JOIN pg_user u ON c.relowner = u.usesysid
         WHERE (relname NOT LIKE 'pg_%') 
           AND (relkind ='S') 
      ORDER BY relname"

    setCursor CLOCK

    set tlist [list]
    if {[catch {wpg_select $dbh_ "$sql" rec {
                lappend tlist [::Database::quoteObject $rec(relname)]
            }

    } err]} {
        showError $err
    }

    setCursor DEFAULT

    return $tlist

}; # end proc ::Database::getSequencesList


#----------------------------------------------------------
# getFunctionsList --
#
#   returns a list of functions in the currentdb
#
# Argumens:
#   dbh_    optionally supply the db handle
#
# Returns:
#   a list of function names
#----------------------------------------------------------
#
proc ::Database::getFunctionsList {{dbh_ ""}} {

    global PgAcVar CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}

    set maxim 16384
    setCursor CLOCK
    set dbname $PgAcVar(opendb,dbname)

    set sql "
        SELECT datlastsysoid 
          FROM [::Database::qualifySysTable pg_database]
         WHERE datname='$dbname'"

    set sql2 "
        SELECT oid 
          FROM [::Database::qualifySysTable pg_database]
         WHERE datname='template1'"

    set sql3 "
        SELECT (proname || '(' || oidvectortypes(proargtypes) || ')') AS proname
         FROM [::Database::qualifySysTable pg_proc] P
    LEFT JOIN [::Database::qualifySysTable pg_user] u
           ON p.proowner = u.usesysid, [::Database::qualifySysTable pg_language] L
        WHERE P.oid>$maxim
          AND P.prolang = L.oid
     ORDER BY proname"

    if [catch {wpg_select $dbh_ "$sql" rec {
        set maxim $rec(datlastsysoid)
    }
    }] {
    catch {
        wpg_select $dbh_ "$sql2" rec {
            set maxim $rec(oid)
        }
    }
    }

    set tlist [list]
    if {[catch {wpg_select $dbh_ "$sql3" rec {
                lappend tlist [::Database::quoteObject $rec(proname)]
            }

    } err]} {
        showError $err
    }

    setCursor DEFAULT

    return $tlist

}; # end proc ::Database::getFunctionsList


#------------------------------------------------------------
# getTablesList --
#
#    returns a list of tables in the currentdb
#
# Arguments:
#    dbh_    optionally supply the db handle
#    pg_     whether to show the PG system objects
#    pga_    whether to show the PGA system objects
#
# Results:
#    a list of table names
#------------------------------------------------------------
#
proc ::Database::getTablesList {{dbh_ ""} {pg_ 0} {pga_ 0}} {

    global CurrentDB PgAcVar

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    if {[string match "" $CurrentDB]} {return [list]}


    set id [::Connections::getIdFromHandle $dbh_]
    set V  $::Connections::Conn(pgversion,$id)

    if {![info exists ::Connections::Conn(viewsystem,$id)]} {
        set ::Connections::Conn(viewsystem,$id) $PgAcVar(pref,systemtables)
    }

    if {![info exists ::Connections::Conn(viewpgaccess,$id)]} {
        set ::Connections::Conn(viewpgaccess,$id) $PgAcVar(pref,pgaccesstables)
    }

    if {$V < 7.3} {

        set sql "
            SELECT c.relname AS table
              FROM pg_class c
             WHERE c.relkind = 'r'"

        if {!$pg_} {
            append sql " AND c.relname !~ '^pg_'"
        }

        if {!$pga_} {
           append sql " AND c.relname !~ '^pga_'"
        }

    } else {

        set sql "
            SELECT n.nspname || '.' || c.relname AS table
              FROM pg_catalog.pg_class c
                    LEFT JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace
             WHERE c.relkind = 'r'"

        if {!$pg_} {
            append sql " AND c.relname !~ '^pg_'"
        }

        if {!$pga_} {
            append sql " AND c.relname !~ '^pga_'"
        }

    }

    # lets order the results by table name
    append sql " ORDER BY c.relname"

    set tlist [list]
    if {[catch {wpg_select $dbh_ "$sql" rec {
                lappend tlist [::Database::quoteObject $rec(table)]
            }

    } err]} {
        showError $err
    }

    return $tlist

}; # end proc getTablesList



# TO BE DELETED ?
proc ::Database::getTablesList-old {} {
global CurrentDB PgAcVar

    set sql(1) "
        SELECT c.relname,count(c.relname)
          FROM [::Database::qualifySysTable pg_class] C, [::Database::qualifySysTable pg_rewrite] R
         WHERE (r.ev_class = C.oid) 
           AND (r.ev_type = '1') 
      GROUP BY relname"

    set sql(2) "
        SELECT relname 
          FROM [::Database::qualifySysTable pg_class]
         WHERE (relname !~ '^pg_') 
           AND (relkind='r') 
      ORDER BY relname"

    set sql(3) "
        SELECT relname 
          FROM [::Database::qualifySysTable pg_class]
         WHERE (relkind='r') 
      ORDER BY relname"

    set tlist {}
    if {[catch {
        wpg_select $CurrentDB "$sql(1)" rec {
            if {$rec(count)!=0} {
                set itsaview($rec(relname)) 1
            }
        }
        if {! $PgAcVar(pref,systemtables)} {
            wpg_select $CurrentDB "$sql(2)" rec {
                if {![regexp "^pga_" $rec(relname)]} then {
                    if {![info exists itsaview($rec(relname))]} {
                        lappend tlist $rec(relname)
                    }
                }
            }
        } else {
            wpg_select $CurrentDB "$sql(3)" rec {
                if {![info exists itsaview($rec(relname))]} {
                    lappend tlist $rec(relname)
                }
            }
        }
    } gterrmsg]} {
        showError $gterrmsg
    }
    return $tlist
}; # end proc ::Database::getTablesList-old



#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Database::vacuum {} {
global PgAcVar CurrentDB
    if {$CurrentDB==""} return;
    set PgAcVar(statusline,dbname) [format [intlmsg "vacuuming database %s ..."] $PgAcVar(currentdb,dbname)]
    setCursor CLOCK
    set pgres [wpg_exec $CurrentDB "vacuum;"]
    catch {pg_result $pgres -clear}
    setCursor DEFAULT
    set PgAcVar(statusline,dbname) $PgAcVar(currentdb,dbname)
}; # end proc ::Database::vacuum


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Database::getPgType {oid {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set ret "unknown"

    set sql "SELECT typname
               FROM pg_type
              WHERE oid=$oid"

    wpg_select $dbh_ $sql rec {
        set ret $rec(typname)
    }

    return $ret

}; # end proc ::Database::getPgType


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Database::executeUpdate {sql_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    return [sql_exec noquiet $sql_ $dbh_]

}; # end proc ::Database::executeUpdate


#----------------------------------------------------------
# ::Database::getPgVersion --
#
#    Gets the version of the PG database
#
# Arguments:
#    db_    This is the db handle of the DB. If it is
#           not supplied, then CurrentDB is assumed
#
# Results:
#    pgversion
#----------------------------------------------------------
#
proc ::Database::getPgVersion {{db_ ""}} {

    if {(![info exists ::CurrentDB]) || ([string match "" $::CurrentDB])} {
	showError [intlmsg "Could not find a good db handle"]
	return
    }

    if {[string match "" $db_]} {set db_ $::CurrentDB}

    if {[catch {wpg_select $db_ "
        SELECT version()" rec {
        set res $rec(version)
    }} err]} {

        return ""
    }

    regexp {PostgreSQL ([.\w]+)} $res m ver

    return $ver

}; # end proc ::Database::getPgVersion


#------------------------------------------------------------
# ::Database::qualifySysTable --
#
#    This just qualifies a system table; checking the PG
#    version number, and it >= 7.3, it will prepend
#    the Pg_catalog schema that is used for the 
#    system tables
#
# Arguments:
#    table_   the table name that needs qualified
#    dbh_    the db handle of the database to use. It defaults
#             to the current db handle (CurrentDB)
#
# Results:
#    none returned
#------------------------------------------------------------
#
proc ::Database::qualifySysTable {table_ {dbh_ ""}} {

    if {[string match "" $dbh_]} {
        set dbh_ $::CurrentDB
    }

    set V [string range [getPgVersion $dbh_] 0 2]

    if {$V >= 7.3} {
	set table_ "pg_catalog.${table_}"
    }

    return [quoteObject $table_]

}; # end proc ::Database::qualifySysTable

#------------------------------------------------------------
# ::Database::quoteObject --
#
#    This makes sure that an object is quoted properly,
#    especially if it is schema qualified.
#
# Arguments:
#    obj_     name of the object to quote
#
# Results:
#    returns the properly quoted object
#------------------------------------------------------------
#
proc ::Database::quoteObject {obj_} {
    return \"[string map [list \" "" . \".\"] $obj_]\"
}; # end proc ::Database::quoteTable


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Database::quoteSQL {sql_} {

    set retval ""

    regsub -all {\\} $sql_ {\\\\} retval
    set sql_ $retval
    regsub -all {\"} $sql_ {\\"} retval
    set sql_ $retval
    regsub -all {\'} $sql_ {\\'} retval
#    set sql_ $retval
#    regsub -all {\$} $sql_ {\\$} retval
#    set sql_ $retval
#    regsub -all {\*} $sql_ {\\*} retval

    return $retval

}; # end proc ::Database::quoteSQL


#----------------------------------------------------------
# convenience proc to return available column types
#----------------------------------------------------------
#
proc ::Database::getAllColumnTypes {} {

    return [list char varchar text int2 int4 serial float4 float8 money abstime date datetime interval reltime time timespan timestamp boolean box circle line lseg path point polygon]

}; # end proc ::Database::getAllColumnTypes


#------------------------------------------------------------
#
#   Returns the results as ones list
#------------------------------------------------------------
#
proc ::Database::getList {sql_ {dbh_ ""}} {

    if {[string match "" $dbh_]} {
	set dbh_ $::CurrentDB
    }

    set res [list]
    wpg_select $dbh_ "$sql_" tuple {
	foreach A $tuple(.headers) {
	    lappend res $tuple($A)
	}
    }

    return $res
};  # end proc ::Database::getList


#------------------------------------------------------------
#
#   Returns the results as a list of lists, where each
#   embedded list is a tuple (row) in the results
#------------------------------------------------------------
#
proc ::Database::getListOfList {sql_ {dbh_ ""}} {

    if {[string match "" $dbh_]} {
	set dbh_ $::CurrentDB
    }

    set res [list]
    wpg_select $dbh_ "$sql_" tuple {
	set subl [list]
	foreach A $tuple(.headers) {
	    lappend subl $tuple($A)
	}

	lappend res $subl
    }

    return $res
};  # end proc ::Database::getList
