/* $Id$ */

/* forward declarations*/
%wrapper %{

typedef struct PyPCShellCtx {
  PetscCookie cookie; /* safeguard      */
  PC          pc_obj; /* PC handle      */
  PyObject*   py_obj; /* Python context */
} PyPCShellCtx;

#define _PyPCShellCtx_Check(shctx) \
  (((shctx) != NULL) && ((shctx)->cookie == PC_COOKIE))

#undef __FUNCT__
#define __FUNCT__ "_PyPCShellCtx_New"
static PetscErrorCode 
_PyPCShellCtx_New(PC pc, PyObject* obj, PyPCShellCtx** shctx)
{
  PyPCShellCtx   *newctx=NULL;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  *shctx = newctx;
  if (obj != NULL && obj != Py_None) {
    ierr = PetscNew(PyPCShellCtx,&newctx);CHKERRQ(ierr);
    if (newctx != NULL) {
      newctx->cookie = PC_COOKIE;
      newctx->pc_obj = pc;
      newctx->py_obj = obj; Py_INCREF(obj);
    }
  }
  *shctx = newctx;
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "_PyPCShellCtx_Del"
static PetscErrorCode
_PyPCShellCtx_Del(PyPCShellCtx *shctx)
{
  PetscErrorCode  ierr;
  PetscFunctionBegin;
  if (_PyPCShellCtx_Check(shctx)) {
    Py_XDECREF(shctx->py_obj);
    shctx->cookie = PETSCFREEDHEADER;
    shctx->pc_obj = PETSC_NULL;
    shctx->py_obj = NULL;
    ierr = PetscFree(shctx);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

static PetscErrorCode _Py_PCShellGetContext(PC, PyObject**);
static PetscErrorCode _Py_PCShellSetContext(PC, PyObject*);
static PetscErrorCode _Py_PCShellDestroy(void* ctx);

#undef PYPCSH_GETCTX
#define PYPCSH_GETCTX(pc, CTX) \
  { ierr = _Py_PCShellGetContext((pc),(CTX)); CHKERRQ(ierr); }
#undef PYPCSH_SETCTX
#define PYPCSH_SETCTX(pc, CTX) \
  { ierr = _Py_PCShellSetContext((pc),(CTX)); CHKERRQ(ierr); }
%}

%apply PC* CREATE { PC* shell };
PETSC_OVERRIDE(
PetscErrorCode,
PCCreateShell,(MPI_Comm comm, PyObject* ctx, PC* shell), {
  PetscErrorCode ierr;
  PetscFunctionBegin;
  PetscValidPointer(shell,3);
  ierr = PCCreate(comm,shell); CHKERRQ(ierr);
  ierr = PCSetType(*shell,PCSHELL); CHKERRQ(ierr);
  PYPCSH_SETCTX(*shell, ctx);
  PetscFunctionReturn(0);
})
%clear PC* shell;

PETSC_OVERRIDE(
PetscErrorCode,
PCShellGetName,(PC pc, const char* name[]), {
  char* cname;
  PetscTruth flg;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject)pc,PCSHELL,&flg);CHKERRQ(ierr);
  if (!flg) SETERRQ(PETSC_ERR_ARG_WRONG, "not a shell preconditioner");
  ierr = PCShellGetName(pc, &cname); CHKERRQ(ierr);
  *name = cname;
  PetscFunctionReturn(0);
})

PETSC_OVERRIDE(
PetscErrorCode,
PCShellSetName,(PC pc, const char name[]), {
  PetscTruth flg;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject)pc,PCSHELL,&flg);CHKERRQ(ierr);
  if (!flg) SETERRQ(PETSC_ERR_ARG_WRONG, "not a shell preconditioner");
  ierr = PCShellSetName(pc, name); CHKERRQ(ierr);
  PetscFunctionReturn(0);
})

%apply PyObject** OUTPUT { PyObject** ctx }
PETSC_OVERRIDE(
PetscErrorCode,
PCShellGetContext,(PC pc, PyObject** ctx), {
  PetscTruth flg;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject)pc,PCSHELL,&flg);CHKERRQ(ierr);
  if (!flg) SETERRQ(PETSC_ERR_ARG_WRONG, "not a shell preconditioner");
  PYPCSH_GETCTX(pc, ctx);
  PetscFunctionReturn(0);
})
%clear PyObject** ctx;


PETSC_OVERRIDE(
PetscErrorCode,
PCShellSetContext,(PC pc, PyObject* ctx), {
  PetscTruth flg;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject)pc,PCSHELL,&flg);CHKERRQ(ierr);
  if (!flg) SETERRQ(PETSC_ERR_ARG_WRONG, "not a shell preconditioner");
  PYPCSH_SETCTX(pc, ctx);
  PetscFunctionReturn(0);
})


/* ---------------------------------------------------------------- */

%wrapper %{

#undef __FUNCT__
#define __FUNCT__ "PyPCShell_Error"
static PetscErrorCode
_PyPCShell_Error(const char* method) {
  PyObject *ptype, *pvalue, *ptrbck;
  PyObject *stype = NULL, *svalue = NULL;
  char *typestr = NULL, *valuestr = NULL;
  char errmesg[256];
  PetscFunctionBegin;
  PyErr_Fetch(&ptype, &pvalue, &ptrbck);
  if (ptype && (stype = PyObject_Str(ptype)))
    typestr = PyString_AS_STRING(stype);
  if (pvalue && (svalue = PyObject_Str(pvalue)))
    valuestr = PyString_AS_STRING(svalue);
  Py_XDECREF(stype);
  Py_XDECREF(svalue);
  PyErr_Restore(ptype, pvalue, ptrbck);
  PyOS_snprintf(errmesg, sizeof(errmesg),
		"in method '%s' of context object\n%s: %s",
		method ? method : "<unknown>",
		typestr ? typestr : "",
		valuestr? valuestr: "");
  SETERRQ(1, errmesg);
  PetscFunctionReturn(0);
}

#define PYPCSH_GETOBJS(SHCTX, PCOBJ, PYOBJ)		  \
do {							  \
  PyPCShellCtx *shctx = (PyPCShellCtx*)(SHCTX);		  \
  if (!shctx) SETERRQ(1,"context object not set");	  \
  if (!_PyPCShellCtx_Check(shctx))			  \
    SETERRQ(1,"invalid context for Python");		  \
  (PCOBJ) = shctx->pc_obj; (PYOBJ) = shctx->py_obj;	  \
  if (!(PCOBJ)) SETERRQ(1,"null pointer for PC handle");  \
  if (!(PYOBJ)) SETERRQ(1,"null pointer for Py context"); \
  Py_INCREF((PYOBJ));					  \
} while(0)

#define PYPCSH_CALLMETH PyObject_CallMethod

#define PYPCSH_CHKCALL(CTX, RETVAL, METHODNAME)				\
do {									\
  PetscErrorCode __ierr;						\
  Py_XDECREF((CTX));							\
  if ((RETVAL)) {							\
    Py_DECREF((RETVAL));						\
  } else {								\
    __ierr = _PyPCShell_Error((METHODNAME)); CHKERRQ(__ierr);		\
  }									\
} while(0)

#undef __FUNCT__  
#define __FUNCT__ "PyPCShell_view"
static PetscErrorCode 
_PyPCShell_view(void* ctx, PetscViewer viewer) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "view", "O&",
			PyPetscViewer_Ref, viewer);
  PYPCSH_CHKCALL(obj, ret, "view");
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PyPCShell_setUp"
static PetscErrorCode 
_PyPCShell_setUp(void* ctx) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  {
    Mat            amat=PETSC_NULL;
    Mat            pmat=PETSC_NULL;
    MatStructure   flag=DIFFERENT_NONZERO_PATTERN;
    PetscErrorCode ierr;
    ierr = PCGetOperators(pc,&amat,&pmat,&flag);CHKERRQ(ierr);
    ret = PYPCSH_CALLMETH(obj, "setUp", "O&O&l",
			  PyMat_Ref, amat,
			  PyMat_Ref, pmat,
			  (long)     flag);
  }
  PYPCSH_CHKCALL(obj, ret, "setUp");
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PyPCShell_preSolve"
static PetscErrorCode 
_PyPCShell_preSolve(void* ctx, KSP ksp, Vec b, Vec x) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "preSolve", "O&O&O&",
			PyKSP_Ref, ksp,
			PyVec_Ref, b,
			PyVec_Ref, x);
  PYPCSH_CHKCALL(obj, ret, "preSolve");
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PyPCShell_postSolve"
static PetscErrorCode 
_PyPCShell_postSolve(void* ctx, KSP ksp, Vec b, Vec x) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "postSolve", "O&O&O&",
			PyKSP_Ref, ksp,
			PyVec_Ref, b,
			PyVec_Ref, x);
  PYPCSH_CHKCALL(obj, ret, "postSolve");
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PyPCShell_apply"
static PetscErrorCode 
_PyPCShell_apply(void* ctx, Vec x, Vec y) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "apply", "O&O&",
			PyVec_Ref, x,
			PyVec_Ref, y);
  PYPCSH_CHKCALL(obj, ret, "apply");
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PyPCShell_applyTranspose"
static PetscErrorCode 
_PyPCShell_applyTranspose(void* ctx, Vec x, Vec y) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "applyTranspose", "O&O&",
			PyVec_Ref, x,
			PyVec_Ref, y);
  PYPCSH_CHKCALL(obj, ret, "applyTranspose");
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PyPCShell_applyRichardson"
static PetscErrorCode 
  _PyPCShell_applyRichardson(void* ctx, Vec x, Vec y, Vec w,
			     PetscReal rt, PetscReal at, PetscReal dt,PetscInt it) {
  PC        pc  = PETSC_NULL;
  PyObject* obj = NULL;
  PyObject* ret = NULL;
  PetscFunctionBegin;
  PYPCSH_GETOBJS(ctx, pc, obj);
  ret = PYPCSH_CALLMETH(obj, "applyRichardson", "O&O&0&dddl",
			PyVec_Ref, x,
			PyVec_Ref, y,
			PyVec_Ref, w,
			(double)   rt,
			(double)   at,
			(double)   dt,
			(long)     it);
  PYPCSH_CHKCALL(obj, ret, "applyRichardson");
  PetscFunctionReturn(0);
}


#undef PYPCSH_GETOBJS
#undef PYPCSH_CALLMETH
#undef PYPCSH_CHKCALL
%}


%wrapper %{

#undef PYPCSH_SETFUNC
#define PYPCSH_SETFUNC(pc, CTX,  PYNAME, CNAME, ARGS) \
{ \
  PetscErrorCode (*func)ARGS; \
  if (ctx == NULL || ctx == Py_None || !PyObject_HasAttrString(ctx, #PYNAME)) \
    { func = (PetscErrorCode(*)ARGS)NULL; } \
  else  \
    { func = (PetscErrorCode(*)ARGS)_PyPCShell_##PYNAME; } \
  ierr = PCShellSet##CNAME(pc, func);CHKERRQ(ierr); \
}

#undef __FUNCT__
#define __FUNCT__ "Py_PCShellSetContext"
static PetscErrorCode
_Py_PCShellSetContext(PC pc, PyObject* ctx)
{
  PyPCShellCtx   *oldshctx;
  PyPCShellCtx   *newshctx;
  PetscErrorCode ierr;
  
  PetscFunctionBegin;
  ierr = PCShellGetContext(pc,(void**)&oldshctx);CHKERRQ(ierr);
  ierr = _PyPCShellCtx_New(pc,ctx,&newshctx);CHKERRQ(ierr);
  ierr = _PyPCShellCtx_Del(oldshctx);CHKERRQ(ierr);
  ierr = PCShellSetContext(pc,(void*)newshctx);CHKERRQ(ierr);
  ierr = PCShellSetDestroy(pc,newshctx?_Py_PCShellDestroy:NULL);CHKERRQ(ierr);
  PYPCSH_SETFUNC(pc, ctx, view,            View,            (void*,PetscViewer));
  PYPCSH_SETFUNC(pc, ctx, setUp,           SetUp,           (void*)            );
  PYPCSH_SETFUNC(pc, ctx, preSolve,        PreSolve,        (void*,KSP,Vec,Vec));
  PYPCSH_SETFUNC(pc, ctx, postSolve,       PostSolve,       (void*,KSP,Vec,Vec));
  PYPCSH_SETFUNC(pc, ctx, apply,           Apply,           (void*,Vec,Vec)    );
  PYPCSH_SETFUNC(pc, ctx, applyTranspose,  ApplyTranspose,  (void*,Vec,Vec)    );
  PYPCSH_SETFUNC(pc, ctx, applyRichardson, ApplyRichardson, (void*,Vec,Vec,Vec,
							     PetscReal,PetscReal,PetscReal,PetscInt));

  PetscFunctionReturn(0);
} 

#undef PYPCSH_SETFUNC

#undef __FUNCT__
#define __FUNCT__ "Py_PCShellGetContext"
static PetscErrorCode
_Py_PCShellGetContext(PC pc, PyObject** ctx) 
{
  PyPCShellCtx   *shctx;
  PetscTruth     isshell;
  PetscErrorCode ierr;
  
  PetscFunctionBegin;
  PetscValidPointer(ctx,2);
  *ctx = NULL;
  ierr = PCShellGetContext(pc,(void**)&shctx);CHKERRQ(ierr);
  if (shctx) {
    if (_PyPCShellCtx_Check(shctx)) {
      if (shctx->py_obj) { 
	Py_INCREF(shctx->py_obj);
	*ctx = shctx->py_obj;
      } else { 
	SETERRQ(1,"null pointer for context object in shell preconditioner"); 
      }
    } else {
      SETERRQ(1,"context in shell preconditioner is not a Python object");
    }
  } else {
    ierr = PetscTypeCompare((PetscObject)pc,PCSHELL,&isshell);CHKERRQ(ierr);
    if (!isshell) { 
      SETERRQ(PETSC_ERR_ARG_WRONG,
	      "input preconditioner is not a shell preconditioner"); 
    } else {
      SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	      "context object not set in shell preconditioner"); 
    }
  }
  
  PetscFunctionReturn(0);
} 

#undef __FUNCT__  
#define __FUNCT__ "Py_PCShellDestroy"
static 
PetscErrorCode 
_Py_PCShellDestroy(void* ctx) 
{
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = _PyPCShellCtx_Del((PyPCShellCtx*)ctx);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

%}


/*
 * Local Variables:
 * mode: C
 * End:
 */
