# Author:    Lisandro Dalcin
# Contact:   dalcinl@users.sourceforge.net
# Copyright: This module has been placed in the public domain.
# Id: $Id$

"""
Random Numbers
==============

Random numbers generation.
"""

# --------------------------------------------------------------------

__date__     = '$Date$'
__version__  = '$Version$'
__revision__ = '$Revision$'

__docformat__ = 'reStructuredText'

# --------------------------------------------------------------------

__all__ = ['Random',
           'RandomRAND',
           'RandomRAND48',
           'RandomSPRNG',]

# --------------------------------------------------------------------

from petsc4py.lib import _petsc

from petsc4py.Object import Object

# --------------------------------------------------------------------


class Random(Object):

    """
    Abstract PETSc object that manages generating random numbers.

    .. note:: By default, random numbers are generated via
       srand48()/drand48() that are uniformly distributed over [0,1).
       The user can shift and stretch this interval by calling
       `Random.setInterval()`.

    .. tip:: Use `Vec.setRandom()` to set the elements of a vector to
       random numbers.
    """

    class Type:
        """
        Random types.
        """
        # native
        PETSCRAND   = _petsc.PETSCRAND
        PETSCRAND48 = _petsc.PETSCRAND48
        SPRNG       = _petsc.SPRNG
        # aliases
        RAND   = PETSCRAND
        RAND48 = PETSCRAND48
        
    def __init__(self, *targs, **kwargs):
        super(Random, self).__init__(*targs, **kwargs)

    def __call__(self):
        """Same as `Random.getValue()`"""
        return _petsc.PetscRandomGetValue(self)

    def create(self, comm=None):
        """
        Creates a context for generating random numbers, and
        initializes the random-number generator.
        """
        return _petsc.PetscRandomCreate(comm, self)

    def getValue(self):
        """
        Generates and returns a random number.
        """
        return _petsc.PetscRandomGetValue(self)

    def getValueReal(self):
        """
        Generates and returns a random number.
        """
        return _petsc.PetscRandomGetValueReal(self)

    def getValueImaginary(self):
        """
        Generates and returns a random number.
        """
        return _petsc.PetscRandomGetValueImaginary(self)

    def getSeed(self):
        """
        Gets the random seed.
        """
        return _petsc.PetscRandomGetSeed(self)
    
    def setSeed(self, seed=None):
        """
        Sets the random seed and seeds the generator.
        """
        if seed is not None:
            _petsc.PetscRandomSetSeed(self, seed)
        _petsc.PetscRandomSeed(self)

    def getInterval(self):
        """
        Gets the interval over which the random numbers will be
        randomly distributed.  By default, this interval is [0,1).

        :Returns:
          - the interval '[low,high)' (as a tuple) over which the
            random numbers will be randomly distributed.  By default,
            this interval is [0,1).
        """
        return tuple(_petsc.PetscRandomGetInterval(self))

    def setInterval(self, interval):
        """
        Sets the interval '[low,high)' over which the random numbers
        will be randomly distributed. By default, this interval is
        [0,1).

        :Parameters:
          - `interval`: 2-sequence with the interval over which the
            random numbers will be randomly distributed.
        """
        low, high = interval
        _petsc.PetscRandomSetInterval(self, low, high)

    seed     = property(getSeed, setSeed)
    interval = property(getInterval, setInterval)


# --------------------------------------------------------------------


class RandomType(object):
    def __init__(self, rnd_type):
        self._type = rnd_type
    def __call__(self, name, bases, dct):
        klass = type(Random)(name, bases, dct)
        setattr(klass, 'TYPE', self._type)
        if '__init__' in dct: return klass
        def __init__(self, rnd=None, *targs, **kwargs):
            """See `Random.create()` and `Random.setType()`."""
            super(Random, self).__init__(*targs, **kwargs)
            _type = getattr(type(self), 'TYPE')
            if isinstance(rnd, Random):
                _petsc.Random_init(self, rnd, _type)
            else: # assume arg is a comm
                _petsc.PetscRandomCreate(rnd, self)
                _petsc.PetscRandomSetType(self, _type)
        setattr(klass, '__init__', __init__)
        return klass

class RandomRAND(Random):
    """
    Random implementation based on ``rand()``.
    """
    __metaclass__ = RandomType(Random.Type.RAND)

class RandomRAND48(Random):
    """
    Random implementation based on ``rand48()``.
    """
    __metaclass__ = RandomType(Random.Type.RAND48)

class RandomSPRNG(Random):
    """
    Random implementation based on SPRNG library.
    """
    __metaclass__ = RandomType(Random.Type.SPRNG)


# --------------------------------------------------------------------
