/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "src/util.h"
#include "src/types.h"
#include "src/list.h"
#include "src/parser.h"
#include "src/macro.h"
#include "src/schedule.h"
#include "src/compile.h"

static const char **filenames = NULL;
static size_t filenames_count = 0;
static list_t block_systems;
static list_t scheduled_systems;
static char dry_run = 0;

static void
parse_cmd(int argc, char *argv[])
{
	int i;

	for (i = 1; i < argc; i++)
	  {
		if (!strcmp(argv[i], "--help"))
		  {
			printf("Usage: %s [options] file...\n", argv[0]);
			printf("Options:\n");
			printf("  --help                    Display this "
			       "information\n");
			printf("  --version                 Display the "
			       "compiler version information\n");
			printf("  --output-dir=<directory>  Directory where to "
			       "put generated files\n");
			printf("  --dry-run                 Do not generate "
			       "output files (implies all\n"
			       "                              --no-<option> "
			       "options)\n");
			printf("  --no-code                 Do not generate "
			       "source files (implies\n"
			       "                             --no-descriptor "
			       "and --no-makefile)\n");
			printf("  --no-descriptor           Do not generate "
			       "lv2_descriptor.c (implies\n"
			       "                              --no-makefile)"
			       "\n");
			printf("  --no-makefile             Do not generate "
			       "the Makefile\n");
			printf("  --no-ttl                  Do not generate "
			       ".ttl files (implies --no-manifest,\n"
			       "                              --no-plugins-ttl "
			       "and --no-extra-ttl)\n");
			printf("  --no-manifest             Do not generate "
			       "manifest.ttl\n");
			printf("  --no-plugin-ttl           Do not generate "
			       "plugin-specific .ttl files\n");
			printf("  --no-extra-ttl            Do not generate "
			       "plugin-specific -extra.ttl files\n");
			printf("  --uri-prefix=<prefix>     Use <prefix> as "
			       "the initial part of URIs\n");
			printf("  --license-uri=<uri>       Use <uri> as the "
			       "DOAP license URI\n");
			exit(EXIT_SUCCESS);
		  }
	  }

	for (i = 1; i < argc; i++)
	  {
		if (!strcmp(argv[i], "--version"))
		  {
			printf("permafrost (Permafrost) 0.2.0\n");
			printf("Copyright (C) 2009, 2010 Stefano D'Angelo "
			       "<zanga.mail@gmail.com>\n");
			exit(EXIT_SUCCESS);
		  }
	  }

	for (i = 1; i < argc; i++)
	  {
		if (argv[i][0] == '-')
		  {
			if (!strncmp(argv[i], "--output-dir=",
				     sizeof("--output-dir=") - 1))
			  {
				compile_output_dir = argv[i] - 1
						     + sizeof("--output-dir=");
				continue;
			  }
			if (!strcmp(argv[i], "--dry-run"))
			  {
				dry_run = 1;
				continue;
			  }
			if (!strcmp(argv[i], "--no-code"))
			  {
				compile_gen_code = 0;
				compile_gen_descriptor = 0;
				compile_gen_makefile = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-descriptor"))
			  {
				compile_gen_descriptor = 0;
				compile_gen_makefile = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-makefile"))
			  {
				compile_gen_makefile = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-ttl"))
			  {
				compile_gen_manifest = 0;
				compile_gen_plugin_ttl = 0;
				compile_gen_extra_ttl = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-manifest"))
			  {
				compile_gen_manifest = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-plugin-ttl"))
			  {
				compile_gen_plugin_ttl = 0;
				continue;
			  }
			if (!strcmp(argv[i], "--no-extra-ttl"))
			  {
				compile_gen_extra_ttl = 0;
				continue;
			  }
			if (!strncmp(argv[i], "--uri-prefix=",
				     sizeof("--uri-prefix=") - 1))
			  {
				compile_uri_prefix = argv[i] - 1
						     + sizeof("--uri-prefix=");
				continue;
			  }
			if (!strncmp(argv[i], "--license-uri=",
				     sizeof("--license-uri=") - 1))
			  {
				compile_license_uri = argv[i] - 1
						     + sizeof("--license-uri=");
				continue;
			  }

			fprintf(stderr, "%s: unrecognized option `%s'\n",
				argv[0], argv[i]);
			exit(EXIT_FAILURE);
		  }

		filenames_count++;
		filenames = xrealloc(filenames,
				     filenames_count * sizeof(const char *));
		filenames[filenames_count - 1] = argv[i];
	  }

	if (filenames == NULL)
	  {
		fprintf(stderr, "%s: no input files\n", argv[0]);
		exit(EXIT_FAILURE);
	  }
}

static void
do_macro_to_blocks(void *data, void *context)
{
	struct system *s, *ns;

	s = (struct system *)data;

	ns = macro_to_blocks(s);

	list_append(block_systems, ns);
}

static void
do_schedule(void *data, void *context)
{
	struct system *s;
	struct scheduled_system *ss;

	s = (struct system *)data;

	ss = schedule(s);

	list_append(scheduled_systems, ss);
}

static void
free_scheduled_system(void *data, void *context)
{
	struct scheduled_system *ss;

	ss = (struct scheduled_system *)data;

	scheduled_system_free(ss);
}

int
main(int argc, char *argv[])
{
	size_t i;

	parse_cmd(argc, argv);

	for (i = 0; i < filenames_count; i++)
		parser_parse(filenames[i]);

	block_systems = list_new();
	list_for_each(parser_systems, do_macro_to_blocks, NULL);

	scheduled_systems = list_new();
	list_for_each(block_systems, do_schedule, NULL);

	if (!dry_run)
		compile(scheduled_systems);

	list_for_each(scheduled_systems, free_scheduled_system, NULL);
	list_free(scheduled_systems);

	list_for_each(block_systems, free_system, NULL);
	list_free(block_systems);

	parser_free();

	free(filenames);

	return EXIT_SUCCESS;
}
